/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "utilities.h"
#include "undo.h"


Undo::Undo (int dumy)
{
  clear_internal ();
}


Undo::~Undo ()
{
}


// Value that defines how soon after a change in the buffer a snapshot is taken.
// Setting it too soon gives many snapshots for little changes.
// Setting it too late gives now enough snapshots.
#define UNDO_TIME 7


void Undo::clear (Editor& editor)
{
  clear_internal ();
  previous_snapshot = editor.text_get_all ();
  store_cursorpositions (editor);
}


void Undo::clear_internal ()
{
  // Clear the whole undo stack.
  cursorpositions_text.clear ();
  cursorpositions_notes.clear ();
  snapshots.clear ();
  pointer = 0; // This is normally equal to the size of the stack.
  timer = UNDO_TIME;
}


bool Undo::tick ()
/*
  Is called at regular intervals from the main window's timer.
  If it is time to store an image of the buffer, that is, if a change needs
  to be recorded, it returns true, else false.
*/
{
  timer++;
  if (timer == UNDO_TIME) {
    return true;
  }
  return false;
}


void Undo::store_snapshot (Editor& editor)
{
  // If the pointer does not point to the end of the buffer, erase from there to the end.
  if (pointer != snapshots.size ()) {
    int n = snapshots.size () - pointer;
    for (int i = 0; i < n; i++) {
      cursorpositions_text.pop_back ();
      cursorpositions_notes.pop_back ();
      snapshots.pop_back ();
    }
  }
  // Store the new data in the stack.
  cursorpositions_text.push_back (previous_cursorposition_text);
  cursorpositions_notes.push_back (previous_cursorposition_notes);
  snapshots.push_back (previous_snapshot);
  // Retrieve the raw usfm code which is now in the buffer and store for next time.
  previous_snapshot = editor.text_get_all ();
  // Retrieve the cursor positions, store for next time.
  store_cursorpositions (editor);
  // Trim buffer it if it grows too large.
  if (snapshots.size () > 30) {
    cursorpositions_text.erase (cursorpositions_text.begin ());
    cursorpositions_notes.erase (cursorpositions_notes.begin ());
    snapshots.erase (snapshots.begin ());
  }
  // Update the pointer.
  pointer = snapshots.size ();
}


void Undo::text_changed (Editor& editor)
{
  timer = 0;
  store_cursorpositions (editor);
}


void Undo::undo (Editor& editor, Configuration * configuration)
{
  if (can_undo ()) {
    // Update pointer.
    pointer--;
    // Restore text.
    ParseLine parseline (snapshots[pointer]);
    editor.chapter_load (configuration, NULL, 0, &parseline.lines);
    editor.format (configuration);
    cursorpositions_set (editor, cursorpositions_text[pointer], cursorpositions_notes[pointer]);
    // Update shadow data.
    previous_snapshot = snapshots[pointer];
    previous_cursorposition_text = cursorpositions_text[pointer];
    previous_cursorposition_notes = cursorpositions_notes[pointer];
  }
  /*
  Prevent taking a snapshots of this change again, which would lead to an 
  infinite loop. Undoing an image triggers a reformat, which triggers plenty of 
  insert and delete signals on the textbuffers, because reformatting involves 
  inserting and deleting in the buffer.
  */
  timer = UNDO_TIME;
}


void Undo::redo (Editor& editor, Configuration * configuration)
{
  if (can_redo ()) {
    // Restore text.
    ParseLine parseline (snapshots[pointer]);
    editor.chapter_load (configuration, NULL, 0, &parseline.lines);
    editor.format (configuration);
    cursorpositions_set (editor, cursorpositions_text[pointer], cursorpositions_notes[pointer]);
    // Update shadow data.
    previous_snapshot = snapshots[pointer];
    previous_cursorposition_text = cursorpositions_text[pointer];
    previous_cursorposition_notes = cursorpositions_notes[pointer];
    // Update pointer.
    pointer++;
  }
  // Prevent taking a snapshots of this change.
  timer = UNDO_TIME;
}


bool Undo::can_undo ()
{
  return (pointer > 0);
}


bool Undo::can_redo ()
{
  return (pointer < snapshots.size ());
}


void Undo::cursorpositions_set (Editor& editor, int textposition, int notesposition)
{
  GtkTextIter iter;
  gtk_text_buffer_get_iter_at_offset (editor.textbuffer_text, &iter, textposition);
  gtk_text_buffer_place_cursor (editor.textbuffer_text, &iter);
  gtk_text_buffer_get_iter_at_offset (editor.textbuffer_notes, &iter, notesposition);
  gtk_text_buffer_place_cursor (editor.textbuffer_notes, &iter);
}


void Undo::store_cursorpositions (Editor& editor)
{
  GtkTextIter iter;
  gtk_text_buffer_get_iter_at_mark (editor.textbuffer_text, &iter, gtk_text_buffer_get_insert (editor.textbuffer_text));
  previous_cursorposition_text = gtk_text_iter_get_offset (&iter);
  gtk_text_buffer_get_iter_at_mark (editor.textbuffer_notes, &iter, gtk_text_buffer_get_insert (editor.textbuffer_notes));
  previous_cursorposition_notes = gtk_text_iter_get_offset (&iter);
}
