/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <errno.h>

#include <tdb.h>
#include <glib/gthread.h>

#include "kozo-db.h"
#include "kozo-db-private.h"
#include "kozo-entry-private.h"
#include "kozo-private.h"
#include "kozo-pack.h"
#include "kozo-util.h"
#include "kozo-id-cache.h"

struct _KozoDB {
        TDB_CONTEXT *db;
        char *name;
        char *filename;

        KozoIDCache *fields;

        GStaticRWLock lock;

        int ref_count;
};

static GHashTable *dbs;
static GMutex *dbs_lock;

#define KOZO_KEY_PREFIX "kozo:"
#define KOZO_KEY_VERSION "kozo:version"
#define KOZO_KEY_FIELDS "kozo:fields"
#define KOZO_KEY_NAME "kozo:name"

#define KOZO_INDEX_KEY_PREFIX "kozo:index:"

#define SEPARATOR 0x01
#define STRING_SEPARATOR "\x01"

GQuark
kozo_db_error_quark (void)
{
        static GQuark quark = 0;
        if (quark == 0)
                quark = g_quark_from_static_string ("kozo-db-error-quark");
        return quark;
}

static void
handle_lookup_error (KozoDB  *db,
                     GError **error)
{
        if (G_LIKELY (tdb_error (db->db) == TDB_ERR_NOEXIST)) {
                g_set_error (error,
                             KOZO_DB_ERROR,
                             KOZO_DB_ERROR_KEY_NOT_FOUND,
                             tdb_errorstr (db->db));
        } else {
                g_set_error (error,
                             KOZO_DB_ERROR,
                             KOZO_DB_ERROR_BACKEND,
                             tdb_errorstr (db->db));
        }
}

static void
handle_put_error (KozoDB  *db,
                  GError **error)
{
        if (G_LIKELY (tdb_error (db->db) == TDB_ERR_EXISTS)) {
                g_set_error (error, KOZO_DB_ERROR,
                             KOZO_DB_ERROR_KEY_EXISTS,
                             tdb_errorstr (db->db));
        } else {
                g_set_error (error, KOZO_DB_ERROR,
                             KOZO_DB_ERROR_BACKEND,
                             tdb_errorstr (db->db));
        }
}

static TDB_CONTEXT *
load_db (const char *filename,
         GError    **error)
{
        TDB_CONTEXT *db;

        db = tdb_open (filename, 0, 0, O_RDWR | O_CREAT, 0644);

#if 0
        /* FIXME: Errors? */
        if (G_UNLIKELY (db == NULL)) {
                if (G_UNLIKELY (g_mkdir_with_parents
                                ("/home/iain/.kozo/databases", 0755) < 0)) {
                        g_set_error (error, KOZO_DB_ERROR,
                                     KOZO_DB_ERROR_IO,
                                     g_strerror (errno));
                        g_free (fn);
                        return NULL;
                }

                db = tdb_open (filename, 0, 0, O_RDWR | O_CREAT, 0644);
        }
#endif

        /* g_free (fn); */

        if (G_UNLIKELY (db == NULL)) {
                g_set_error (error, KOZO_DB_ERROR,
                             KOZO_DB_ERROR_BACKEND,
                             "Error opening %s", filename);

                return NULL;
        }

        return db;
}

static guint
get_version (TDB_CONTEXT *db)
{
        TDB_DATA key, data;
        guint db_version;

        key.dptr = (guchar *) KOZO_KEY_VERSION;
        key.dsize = strlen (KOZO_KEY_VERSION);

        tdb_lockall_read (db);
        data = tdb_fetch (db, key);
        tdb_unlockall_read (db);

        if (data.dptr == NULL) {
                return 0;
        }

        db_version = * (guint *) data.dptr;
        g_free (data.dptr);

        return db_version;
}

static gboolean
check_version (TDB_CONTEXT *db,
               guint        version,
               GError     **error)
{
        guint db_version;

        db_version = get_version (db);
        if (db_version == 0) {
                TDB_DATA key, data;
                int err;

                /* Don't have the key, this must be a new db so we add it */
                tdb_transaction_start (db);

                key.dptr = (guchar *) KOZO_KEY_VERSION;
                key.dsize = strlen (KOZO_KEY_VERSION);

                data.dptr = (guchar *) &version;
                data.dsize = sizeof (guint);

                err = tdb_store (db, key, data, TDB_INSERT);

                tdb_transaction_commit (db);

                /* The version obviously matches */
                return TRUE;
        }

        if (db_version != version) {
                g_set_error (error, KOZO_DB_ERROR,
                             KOZO_DB_ERROR_VERSION_MISMATCH,
                             "Database version (%u) does not match requested version (%u)",
                             db_version, version);
                return FALSE;
        }

        return TRUE;
}

static KozoIDCache *
get_fields (TDB_CONTEXT *db)
{
        TDB_DATA key, data;
        KozoIDCache *cache;

        key.dptr = (guchar *) KOZO_KEY_FIELDS;
        key.dsize = strlen (KOZO_KEY_FIELDS);

        tdb_lockall_read (db);
        data = tdb_fetch (db, key);
        tdb_unlockall_read (db);

        cache = kozo_id_cache_new ((char *) data.dptr, data.dsize);
        g_free (data.dptr);

        return cache;
}

static char *
get_name (TDB_CONTEXT *db)
{
        TDB_DATA key, data;
        char *name;

        key.dptr = (guchar *) KOZO_KEY_NAME;
        key.dsize = strlen (KOZO_KEY_NAME);

        tdb_lockall_read (db);
        data = tdb_fetch (db, key);
        tdb_unlockall_read (db);

        name = g_strndup ((char *) data.dptr, data.dsize);
        free (data.dptr);

        return name;
}

static KozoDB *
kozo_db_new (const char *filename,
             const char *db_name,
             guint       version,
             GError    **error)
{
        KozoDB *ret;

        /* Initialize structure */
        ret = g_new0 (KozoDB, 1);

        /* Load database */
        ret->db = load_db (filename, error);
        if (G_UNLIKELY (!ret->db)) {
                g_free (ret);
                return NULL;
        }

        /* Check version */
        if (G_UNLIKELY (check_version (ret->db, version, error)) == FALSE) {
                tdb_close (ret->db);
                g_free (ret);
                return NULL;
        }

        /* Read fields */
        ret->fields = get_fields (ret->db);
        if (G_UNLIKELY (!ret->fields)) {
                tdb_close (ret->db);
                g_free (ret);
                return NULL;
        }

        ret->name = get_name (ret->db);
        if (ret->name == NULL) {
                kozo_db_set_name (ret, db_name);
        }
        ret->filename = g_strdup (filename);
        ret->ref_count = 1;

        g_static_rw_lock_init (&ret->lock);

        return ret;
}

KozoDB *
kozo_db_get_for_path (const char *path,
                      const char *db_name,
                      guint       version,
                      GError    **error)
{
        KozoDB *ret;

        ret = kozo_db_new (path, db_name, version, error);
        if (ret == NULL) {
                return NULL;
        }

        return ret;
}

KozoDB *
kozo_db_get (const char *db_name,
             guint       version,
             GError    **error)
{
        KozoDB *ret;
        char *filename;

        g_mutex_lock (dbs_lock);

        ret = g_hash_table_lookup (dbs, db_name);
        if (ret) {
                ret = kozo_db_ref (ret);

                g_mutex_unlock (dbs_lock);

                return ret;
        }

        filename = g_build_filename (g_get_home_dir (),
                                     ".kozo", "databases", db_name, NULL);

        ret = kozo_db_new (filename, db_name, version, error);
        if (G_LIKELY (ret))
                g_hash_table_insert (dbs, ret->name, ret);
        g_free (filename);

        g_mutex_unlock (dbs_lock);

        return ret;
}

KozoDB *
kozo_db_get_for_name (const char *filename,
                      const char *db_name,
                      guint       version,
                      GError    **error)
{
        KozoDB *ret;
        char *fullname;

        g_mutex_lock (dbs_lock);

        ret = g_hash_table_lookup (dbs, db_name);
        if (ret) {
                ret = kozo_db_ref (ret);

                g_mutex_unlock (dbs_lock);
                return ret;
        }

        fullname = g_build_filename (g_get_home_dir (),
                                     ".kozo", "databases", filename, NULL);

        ret = kozo_db_new (fullname, db_name, version, error);
        if (G_LIKELY (ret)) {
                g_hash_table_insert (dbs, ret->name, ret);
        }
        g_free (fullname);

        g_mutex_unlock (dbs_lock);

        return ret;
}

static void
kozo_db_free (KozoDB *db)
{
        g_hash_table_remove (dbs, db->name);

        tdb_close (db->db);

        g_static_rw_lock_free (&db->lock);

        kozo_id_cache_free (db->fields);
        g_free (db->name);
        g_free (db->filename);

        g_free (db);
}

KozoDB *
kozo_db_ref (KozoDB *db)
{
        g_atomic_int_inc (&db->ref_count);

        return db;
}

void
kozo_db_unref (KozoDB *db)
{
        g_mutex_lock (dbs_lock);

        if (g_atomic_int_dec_and_test (&db->ref_count))
                kozo_db_free (db);

        g_mutex_unlock (dbs_lock);
}

const char *
kozo_db_get_filename (KozoDB *db)
{
        return db->filename;
}

const char *
kozo_db_get_name (KozoDB *db)
{
        return db->name;
}

void
kozo_db_set_name (KozoDB     *db,
                  const char *db_name)
{
        TDB_DATA key, data;

        key.dptr = (guchar *) KOZO_KEY_NAME;
        key.dsize = strlen (KOZO_KEY_NAME);

        data.dptr = (guchar *) db_name;
        data.dsize = strlen (db_name);

        tdb_lockall (db->db);
        tdb_transaction_start (db->db);

        tdb_store (db->db, key, data, TDB_REPLACE);

        tdb_transaction_commit (db->db);
        tdb_unlockall (db->db);

        g_free (db->name);
        db->name = g_strdup (db_name);
}

guint
kozo_db_get_version (KozoDB *db)
{
        return get_version (db->db);
}

void
kozo_db_reader_lock (KozoDB *db)
{
        g_static_rw_lock_reader_lock (&db->lock);
        tdb_lockall_read (db->db);
}

void
kozo_db_reader_unlock (KozoDB *db)
{
        tdb_unlockall_read (db->db);
        g_static_rw_lock_reader_unlock (&db->lock);
}

void
kozo_db_writer_lock (KozoDB *db)
{
        g_static_rw_lock_writer_lock (&db->lock);
        tdb_lockall (db->db);
}

void
kozo_db_writer_unlock (KozoDB *db)
{
        tdb_unlockall (db->db);
        g_static_rw_lock_writer_unlock (&db->lock);
}

void
kozo_db_associate_lock (KozoDB *db)
{
}

void
kozo_db_associate_unlock (KozoDB *db)
{
}

KozoEntry *
kozo_db_lookup (KozoDB     *db,
                const char *key_s,
                GSList     *field_ids,
                GError    **error)
{
        KozoPackContext context;
        TDB_DATA key, data;

	memset (&key, 0, sizeof (TDB_DATA));
        key.dptr = (gpointer) key_s;
        key.dsize = strlen (key_s);

	memset (&data, 0, sizeof (TDB_DATA));

        kozo_db_reader_lock (db);

        data = tdb_fetch (db->db, key);
        if (data.dptr == NULL) {
                handle_lookup_error (db, error);

                kozo_db_reader_unlock (db);
                return NULL;
        }

        if (field_ids == NULL) {
                kozo_db_reader_unlock (db);

                return kozo_entry_new (data.dptr, data.dsize);
        } else {
                kozo_pack_context_init (&context, FALSE,
                                        g_slist_length (field_ids),
                                        sizeof (guint32));
        }

        kozo_pack_start (&context);
        kozo_db_harvest_entry (key, data,
                               &context, field_ids);
        context.dest = g_malloc (context.length);
        kozo_pack_finish (&context);

        kozo_db_reader_unlock (db);

        g_free (data.dptr);

        return kozo_entry_new (context.dest, context.length);
}

TDB_CONTEXT *
kozo_db_get_db (KozoDB *db)
{
        return db->db;
}

static gboolean
finish_put (KozoDB          *db,
            KozoPackContext *context,
            TDB_DATA         key,
            TDB_DATA         data,
            int              flag,
            GError         **error)
{
        int err;

        context->dest = g_alloca (context->length);
        kozo_pack_finish (context);

        data.dptr = context->dest;
        data.dsize = context->length;

#if 0
        {
                KozoEntry entry;

                entry.data = context->dest;
                entry.length = context->length;

                kozo_entry_dump (&entry, context->n_fields);
        }
#endif

        err = tdb_store (db->db, key, data, flag);
        if (G_UNLIKELY (err)) {
                handle_put_error (db, error);

                tdb_transaction_cancel (db->db);
                kozo_db_writer_unlock (db);

                return FALSE;
        }

        tdb_transaction_commit (db->db);
        kozo_db_writer_unlock (db);

        return TRUE;
}

gboolean
kozo_db_set (KozoDB     *db,
             const char *key_s,
             GSList     *fields,
             GError    **error)
{
        TDB_DATA key, data;
        KozoPackContext context;
        KozoField **field_cache;
        KozoField *field;
        GSList *l;
        KozoEntry entry;
        gboolean ret;

        memset (&key, 0, sizeof (TDB_DATA));
        key.dptr = (gpointer) key_s;
        key.dsize = strlen (key_s);

        kozo_db_writer_lock (db);

        /* Get original */
        data = tdb_fetch (db->db, key);

        tdb_transaction_start (db->db);

        entry.data = data.dptr;
        entry.length = data.dsize;

        /* Construct and put new entry */
        kozo_pack_context_init (&context, FALSE,
                                kozo_id_cache_n_ids (db->fields),
                                sizeof (guint32));
        kozo_pack_start (&context);

        field_cache = g_newa (KozoField *, context.n_fields);
        memset (field_cache, 0, sizeof (KozoField *) * context.n_fields);

        for (l = fields; l; l = l->next) {
                field = l->data;
                field_cache[kozo_field_get_id (field)] = field;
        }

        while (context.cur_idx < context.n_fields) {
                if (field_cache[context.cur_idx]) {
                        kozo_pack_field (&context,
                                         field_cache[context.cur_idx]);
                } else {
                        field = kozo_entry_get_field (&entry, context.cur_idx);
                        kozo_pack_field (&context, field);
                        kozo_field_free (field);
                }
        }

        ret = finish_put (db, &context, key, data, TDB_MODIFY, error);
        g_free (data.dptr);

        return ret;
}

gboolean
kozo_db_add (KozoDB     *db,
             const char *key_s,
             GSList     *fields,
             GError    **error)
{
        TDB_DATA key, data;
        KozoPackContext context;
        KozoField **field_cache;
        KozoField *field;
        GSList *l;

        memset (&key, 0, sizeof (TDB_DATA));
        key.dptr = (gpointer) key_s;
        key.dsize = strlen (key_s);

        memset (&data, 0, sizeof (TDB_DATA));

        kozo_db_writer_lock (db);

        tdb_transaction_start (db->db);

        /* Construct and put new entry */
        kozo_pack_context_init (&context, FALSE,
                                kozo_id_cache_n_ids (db->fields),
                                sizeof (guint32));
        kozo_pack_start (&context);

        field_cache = g_newa (KozoField *, context.n_fields);
        memset (field_cache, 0, sizeof (KozoField *) * context.n_fields);

        for (l = fields; l; l = l->next) {
                field = l->data;
                field_cache[kozo_field_get_id (field)] = field;
        }

        while (context.cur_idx < context.n_fields) {
                if (field_cache[context.cur_idx]) {
                        kozo_pack_field (&context,
                                         field_cache[context.cur_idx]);
                } else
                        kozo_pack_empty_field (&context);
        }

        return finish_put (db, &context, key, data, TDB_INSERT, error);
}

gboolean
kozo_db_remove (KozoDB     *db,
                const char *key_s,
                GError    **error)
{
        TDB_DATA key;
        int err;

        memset (&key, 0, sizeof (TDB_DATA));
        key.dptr = (gpointer) key_s;
        key.dsize = strlen (key_s);

        kozo_db_writer_lock (db);
        tdb_transaction_start (db->db);

        if (G_LIKELY (tdb_exists (db->db, key) == 1)) {
        /* Delete it */
                err = tdb_delete (db->db, key);
                if (G_UNLIKELY (err)) {
                        handle_lookup_error (db, error);

                        tdb_transaction_cancel (db->db);
                        kozo_db_writer_unlock (db);
                        return FALSE;
                }
        }

        tdb_transaction_commit (db->db);
        kozo_db_writer_unlock (db);

        return TRUE;
}

void
kozo_db_flush (KozoDB *db)
{
#if 0
        kozo_db_writer_lock (db);

        /* Sync fields file */
        kozo_id_cache_flush (db->fields);

        kozo_db_writer_unlock (db);
#endif
}

/* This could be sped up surely by calculating all the necessary extra spaces */
static gboolean
add_empty_field_to_db (KozoDB     *db,
                       const char *desc,
                       GError    **error)
{
        KozoPackContext context;
        TDB_DATA key, data;
        KozoEntry entry;
        KozoField *field;

        memset (&key, 0, sizeof (TDB_DATA));
	memset (&data, 0, sizeof (TDB_DATA));

        kozo_pack_context_init (&context, FALSE,
                                kozo_id_cache_n_ids (db->fields) + 1,
                                sizeof (guint32));

        key = tdb_firstkey (db->db);
        while (key.dptr) {
                TDB_DATA newkey;

                if (G_UNLIKELY (strncmp ((char *) key.dptr, KOZO_KEY_PREFIX,
                                         strlen (KOZO_KEY_PREFIX)) == 0)) {
                        newkey = tdb_nextkey (db->db, key);
                        g_free (key.dptr);
                        key = newkey;
                        continue;
                }

                data = tdb_fetch (db->db, key);

                entry.data = data.dptr;
                entry.length = data.dsize;

                /* Append field to entry */
                kozo_pack_start (&context);
                while (context.cur_idx < (context.n_fields - 1)) {
                        field = kozo_entry_get_field (&entry, context.cur_idx);
                        kozo_pack_field (&context, field);
                        kozo_field_free (field);
                }

                kozo_pack_empty_field (&context);

                context.dest = g_alloca (context.length);
                kozo_pack_finish (&context);

                data.dptr = context.dest;
                data.dsize = context.length;

                tdb_store (db->db, key, data, TDB_MODIFY);
                newkey = tdb_nextkey (db->db, key);
                free (key.dptr);
                key = newkey;
        }

        return TRUE;
}

static void
update_fields (TDB_CONTEXT *db,
               const char  *fields)
{
        TDB_DATA key, data;

        key.dptr = (guchar *) KOZO_KEY_FIELDS;
        key.dsize = strlen (KOZO_KEY_FIELDS);

        data.dptr = (guchar *) fields;
        data.dsize = strlen (fields);

        tdb_store (db, key, data, TDB_REPLACE);
}

int
kozo_db_register_field (KozoDB     *db,
                        const char *desc,
                        GError    **error)
{
        char *str;
        int id;

        /* Check existence */
        id = kozo_id_cache_get (db->fields, desc);
        if (id >= 0) {
                return id;
        }

        kozo_db_writer_lock (db);

        /* Add field to database */
        if (G_UNLIKELY (!add_empty_field_to_db (db, desc, error))) {
                kozo_db_writer_unlock (db);

                return -1;
        }

        /* Update id cache */
        id = kozo_id_cache_add (db->fields, desc, error);

        str = kozo_id_cache_get_string (db->fields);
        update_fields (db->db, str);
        g_free (str);

        kozo_db_writer_unlock (db);

        return id;
}

void
kozo_db_init (void)
{
        dbs = g_hash_table_new (g_str_hash, g_str_equal);

        dbs_lock = g_mutex_new ();
}

void
kozo_db_shutdown (void)
{
        g_mutex_free (dbs_lock);

        /* FIXME: Should we unref all open databases? */
        g_hash_table_destroy (dbs);
}

void
kozo_db_harvest_entry (const TDB_DATA   key,
                       const TDB_DATA   data,
                       KozoPackContext *c,
                       GSList          *field_ids)
{
        GSList *l;
        int field_id;
        KozoEntry entry;
        KozoField *field;

        entry.data = data.dptr;
        entry.length = data.dsize;

        if (field_ids == NULL) {
                kozo_pack_data (c, data.dptr, data.dsize);
                return;
        }

        for (l = field_ids; l; l = l->next) {
                field_id = GPOINTER_TO_INT (l->data);

                if (field_id >= 0) {
                        field = kozo_entry_get_field (&entry, field_id);
                        kozo_pack_field (c, field);
                        kozo_field_free (field);
                } else
                        kozo_pack_data (c, key.dptr, key.dsize);
        }
}

static void
dump_data (TDB_DATA *data)
{
        int i;

        for (i = 0; i < data->dsize; i++) {
                if (g_ascii_isprint (data->dptr[i])) {
                        g_print ("%c", data->dptr[i]);
                } else {
                        g_print (".");
                }
        }
        g_print ("\n");
}

struct _ForeachData {
        KozoDB *db;
        KozoDBForeachFunc func;
        gpointer userdata;
};

static int
traverse_read_func (TDB_CONTEXT *db,
                    TDB_DATA     key,
                    TDB_DATA     data,
                    gpointer     userdata)
{
        KozoEntry entry;
        char *tmp_key;
        struct _ForeachData *fd = userdata;

        /* The public does not care about any internal Kozo keys */
        if (strncmp ((char *) key.dptr, KOZO_KEY_PREFIX,
                     strlen (KOZO_KEY_PREFIX)) == 0) {
                return 0;
        }

        entry.data = data.dptr;
        entry.length = data.dsize;

        tmp_key = g_newa (char, key.dsize + 1);
        memcpy (tmp_key, key.dptr, key.dsize);
        tmp_key[key.dsize] = 0;

        /* Traverse read wants 0 to continue, anything else to stop
           which is the reverse of sane logic: FALSE -> stop, TRUE -> continue
           so we negate the response from the foreach function */
        return !fd->func (fd->db, (const char *) tmp_key, &entry, fd->userdata);
}

void
kozo_db_foreach (KozoDB           *db,
                 KozoDBForeachFunc func,
                 gpointer          userdata)
{
        struct _ForeachData data;

        data.db = db;
        data.func = func;
        data.userdata = userdata;

        tdb_traverse_read (db->db, traverse_read_func, &data);
}

gboolean
kozo_db_index_add_word (KozoDB     *db,
                        const char *word,
                        const char *uri,
                        gboolean   *new_word,
                        GError    **error)
{
        TDB_DATA key, data;
        char *uris;
        char **uri_vector;
        int err, i;

        *new_word = FALSE;

        key.dptr = (guchar *) g_strdup_printf ("%s%s",
                                               KOZO_INDEX_KEY_PREFIX, word);
        key.dsize = strlen ((char *) key.dptr);

        memset (&data, 0, sizeof (TDB_DATA));

        kozo_db_writer_lock (db);

        data = tdb_fetch (db->db, key);
        if (data.dptr == NULL) {
                if (tdb_error (db->db) != TDB_ERR_NOEXIST) {
                        handle_lookup_error (db, error);
                        kozo_db_writer_unlock (db);
                        g_free (key.dptr);

                        return FALSE;
                }

                /* This is the first occurance of the word */
                *new_word = TRUE;

                data.dptr = (guchar *) uri;
                data.dsize = strlen (uri);

                tdb_transaction_start (db->db);

                err = tdb_store (db->db, key, data, TDB_INSERT);
                if (G_UNLIKELY (err)) {
                        handle_put_error (db, error);

                        tdb_transaction_cancel (db->db);
                        kozo_db_writer_unlock (db);

                        g_free (key.dptr);
                        return FALSE;
                }

                tdb_transaction_commit (db->db);
                kozo_db_writer_unlock (db);

                g_free (key.dptr);
                return TRUE;
        }

        /* We have other occurances of this word so we append the uri to
           the end of the data */
        uris = g_strndup ((char *) data.dptr, data.dsize);
        g_free (data.dptr);

        /* We split the uris into a vector to see if we
           already have this uri */
        uri_vector = g_strsplit (uris, STRING_SEPARATOR, 0);
        for (i = 0; uri_vector[i]; i++) {
                if (g_str_equal (uri_vector[i], uri)) {
                        g_strfreev (uri_vector);

                        g_free (uris);
                        g_free (key.dptr);
                        kozo_db_writer_unlock (db);
                        return TRUE;
                }
        }

        g_strfreev (uri_vector);

        data.dptr = (guchar *) g_strdup_printf ("%s%c%s", uris, SEPARATOR, uri);
        data.dsize = strlen ((char *) data.dptr);

        g_free (uris);

        tdb_transaction_start (db->db);

        err = tdb_store (db->db, key, data, TDB_REPLACE);
        if (G_UNLIKELY (err)) {
                handle_put_error (db, error);

                tdb_transaction_cancel (db->db);
                kozo_db_writer_unlock (db);

                g_free (key.dptr);
                g_free (data.dptr);
                return FALSE;
        }

        tdb_transaction_commit (db->db);
        kozo_db_writer_unlock (db);

        g_free (key.dptr);
        g_free (data.dptr);

        return TRUE;
}

GList *
kozo_db_index_lookup (KozoDB     *db,
                      const char *word,
                      GError    **error)
{
        TDB_DATA key, data;
        GList *results = NULL;
        char *uris, **uri_vector;
        int i;

        key.dptr = (guchar *) g_strdup_printf ("%s%s",
                                               KOZO_INDEX_KEY_PREFIX, word);
        key.dsize = strlen ((char *) key.dptr);

        kozo_db_reader_lock (db);
        data = tdb_fetch (db->db, key);

        if (data.dptr == NULL) {
                g_free (key.dptr);

                kozo_db_reader_unlock (db);
                return NULL;
        }

        g_free (key.dptr);

        uris = g_strndup ((char *) data.dptr, data.dsize);
        g_free (data.dptr);

        uri_vector = g_strsplit (uris, STRING_SEPARATOR, 0);
        g_free (uris);

        for (i = 0; uri_vector[i]; i++) {
                results = g_list_prepend (results, uri_vector[i]);
        }
        /* Don't need to free the strings, just the array */
        g_free (uri_vector);

        kozo_db_reader_unlock (db);
        return results;
}

gboolean
kozo_db_index_remove_word (KozoDB     *db,
                           const char *word,
                           const char *uri,
                           gboolean   *deleted,
                           GError    **error)
{
        TDB_DATA key, data;
        GString *builder;
        char *uris;
        char **uri_vector;
        int err, i;
        gboolean empty = TRUE;

        *deleted = FALSE;

        key.dptr = (guchar *) g_strdup_printf ("%s%s",
                                               KOZO_INDEX_KEY_PREFIX, word);
        key.dsize = strlen ((char *) key.dptr);

        memset (&data, 0, sizeof (TDB_DATA));

        kozo_db_writer_lock (db);

        data = tdb_fetch (db->db, key);
        if (data.dptr == NULL) {
                /* Word wasn't in the index anyway */
                g_free (key.dptr);

                kozo_db_writer_unlock (db);
                return TRUE;
        }

        uris = g_strndup ((char *) data.dptr, data.dsize);
        g_free (data.dptr);

        /* We split the uris into a vector to find the uri */
        uri_vector = g_strsplit (uris, STRING_SEPARATOR, 0);
        g_free (uris);

        builder = g_string_new ("");
        for (i = 0; uri_vector[i]; i++) {
                if (g_str_equal (uri_vector[i], uri)) {
                        continue;
                }

                if (i != 0) {
                        g_string_append_c (builder, SEPARATOR);
                }
                g_string_append (builder, uri_vector[i]);
                empty = FALSE;
        }

        g_strfreev (uri_vector);

        if (empty) {
                /* We don't have anything that contains this word anymore
                   so remove it */
                tdb_transaction_start (db->db);
                err = tdb_delete (db->db, key);
                if (G_UNLIKELY (err)) {
                        handle_lookup_error (db, error);

                        g_free (key.dptr);
                        g_string_free (builder, TRUE);

                        tdb_transaction_cancel (db->db);
                        kozo_db_writer_unlock (db);
                        return FALSE;
                }

                *deleted = TRUE;
                tdb_transaction_commit (db->db);

                g_free (key.dptr);
                g_string_free (builder, TRUE);

                kozo_db_writer_unlock (db);
                return TRUE;
        }

        data.dptr = (guchar *) builder->str;
        data.dsize = strlen (builder->str);
        g_string_free (builder, FALSE);

        tdb_transaction_start (db->db);
        err = tdb_store (db->db, key, data, TDB_REPLACE);
        if (G_UNLIKELY (err)) {
                handle_put_error (db, error);

                tdb_transaction_cancel (db->db);
                kozo_db_writer_unlock (db);

                g_free (key.dptr);
                g_free (data.dptr);
                return FALSE;
        }

        tdb_transaction_commit (db->db);
        kozo_db_writer_unlock (db);

        g_free (key.dptr);
        g_free (data.dptr);

        return TRUE;
}

static int
get_words_func (TDB_CONTEXT *db,
                TDB_DATA     key,
                TDB_DATA     data,
                gpointer     userdata)
{
        char *word;
        GSequence *words = userdata;

        if (strncmp ((char *) key.dptr, KOZO_INDEX_KEY_PREFIX,
                     strlen (KOZO_INDEX_KEY_PREFIX)) != 0) {
                return 0;
        }

        word = g_strndup ((char *) key.dptr + strlen (KOZO_INDEX_KEY_PREFIX),
                          key.dsize - strlen (KOZO_INDEX_KEY_PREFIX));
        g_sequence_prepend (words, word);

        return 0;
}

static int
compare_words (gconstpointer a,
               gconstpointer b,
               gpointer      data)
{
        return strcmp (a, b);
}

GSequence *
kozo_db_get_index_words (KozoDB *db)
{
        GSequence *words;

        words = g_sequence_new (g_free);
        tdb_traverse_read (db->db, get_words_func, words);

        g_sequence_sort (words, (GCompareDataFunc) compare_words, NULL);
        return words;
}
