/*
 * Copyright 2009-2011 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

const EXPORTED_SYMBOLS = ["Synchroniser", "BookmarksObserver"];

const Cc = Components.classes;
const Ci = Components.interfaces;
const Cr = Components.results;
const Cu = Components.utils;

Cu.import("resource://bindwood/logging.jsm");
Cu.import("resource://bindwood/migration.jsm");

const GUID_ANNOTATION = "bindwood/uuid";
const PARENT_ANNOTATION = "bindwood/parent";

const RECORD_PREFIX = (
    "http://www.freedesktop.org/wiki/Specifications/desktopcouch/");
const TYPE_BOOKMARK = RECORD_PREFIX + "bookmark";
const TYPE_FOLDER = RECORD_PREFIX + "folder";
const TYPE_FEED = RECORD_PREFIX + "feed";
const TYPE_SEPARATOR = RECORD_PREFIX + "separator";

const RECORD_TYPE_VERSION = 2;

var bookmarksService = Cc["@mozilla.org/browser/nav-bookmarks-service;1"]
    .getService(Ci.nsINavBookmarksService);
var livemarkService = Cc["@mozilla.org/browser/livemark-service;2"]
    .getService(Ci.nsILivemarkService);
var annotationService = Cc["@mozilla.org/browser/annotation-service;1"]
    .getService(Ci.nsIAnnotationService);
var historyService = Cc["@mozilla.org/browser/nav-history-service;1"]
    .getService(Ci.nsINavHistoryService);
var uuidService = Cc["@mozilla.org/uuid-generator;1"]
    .getService(Ci.nsIUUIDGenerator);
var ioService = Cc["@mozilla.org/network/io-service;1"]
    .getService(Ci.nsIIOService);


function Synchroniser(couch, profile) {
    this.couch = couch;
    this.profile = profile;
    this.last_seq = 0;
    this.guid_item_map = {};
    this.folders_to_reorder = {};
    this.first_push = true;
    this.observer = null;
}

Synchroniser.prototype = {
    init: function() {
        Log.debug("Initialising synchroniser.");
        this.ensureDatabase();
        this.ensureCouchViews();
        this.maybeMigrateData();
        this.first_push = true;
        this.observer = new BookmarksObserver(this);
        bookmarksService.addObserver(this.observer, false);
    },

    uninit: function() {
        Log.debug("Uninitialising synchroniser.");
        if (this.observer) {
            bookmarksService.removeObserver(this.observer);
            this.observer = null;
        }
    },

    ensureDatabase: function() {
        // This function will create the database if it does not
        // exist, but we return a boolean representing whether or not
        // the database existed prior.
        try {
            this.couch.createDb();
        } catch (e) {
            if (e.error != 'file_exists') {
                Log.exception("Error creating database: ", e);
                throw(e);
            }
        }
    },

    ensureCouchViews: function() {
        var design_doc_id = "_design/bindwood", doc;

        try {
            doc = this.couch.open(design_doc_id);
        } catch (e) {
            Log.exception("Problem retrieving view view", e);
            return;
        }
        if (!doc) {
            doc = { _id: design_doc_id };
        }
        if (!doc.views)
            doc.views = {};
        doc.views.bookmarks = {
            map: ("function(doc) {\n" +
                  "  try {\n" +
                  "    var annot = doc.application_annotations.Firefox;\n" +
                  "    if (annot.profile)\n" +
                  "      emit(annot.profile, doc._id);\n" +
                  "  } catch (e) { /* ignore error */ }\n" +
                  "}")
        };
        if (!doc.filters)
            doc.filters = {};
        doc.filters.by_profile = (
            "function (doc, req) {\n" +
            "  if (doc._deleted)\n" +
            "    return true;\n" +
            "  try {\n" +
            "    return (doc.application_annotations.Firefox.profile == \n" +
            "            req.query.profile);\n" +
            "  } catch (e) { /* ignore error */ }\n" +
            "}")
        // XXX: should we check to see whether we've changed the
        // document before saving it?
        try {
            this.couch.save(doc);
        } catch (e) {
            Log.exception("Problem saving view", e);
        }
    },

    maybeMigrateData: function() {
        migration = new SchemaMigration(this.couch, this.profile);

        if (migration.upgrade()) {
            Log.debug("Schema was upgraded, so resetting last known " +
                      "sequence number.");
            this.last_seq = 0;
        }
    },

    set_guid: function(item_id, guid) {
        // If the item has already been annotated with a GUID, clear
        // the old GUID mapping from our cache.
        try {
            var old_guid = annotationService.getItemAnnotation(
                item_id, GUID_ANNOTATION);
            delete this.guid_item_map[old_guid];
        } catch (e) {
            // Ignore errors.
        }

        if (!guid) {
            guid = uuidService.generateUUID().toString();
        }
        annotationService.setItemAnnotation(
            item_id, GUID_ANNOTATION, guid,
            0, annotationService.EXPIRE_NEVER);
        // Cache the GUID->Item mapping for later calls.
        this.guid_item_map[guid] = item_id;
        Log.debug("Set GUID on item " + item_id + " to " + guid);
        return guid;
    },

    guid_from_id: function(item_id) {
        // Handle special items.
        if (item_id == bookmarksService.placesRoot)
            return "root_" + this.profile;
        else if (item_id == bookmarksService.toolbarFolder)
            return "toolbar_" + this.profile;
        else if (item_id == bookmarksService.bookmarksMenuFolder)
            return "menu_" + this.profile;
        else if (item_id == bookmarksService.unfiledBookmarksFolder)
            return "unfiled_" + this.profile;

        // Try to look up the uuid, and failing that, assign a new one
        // and return it.
        var guid;
        try {
            guid = annotationService.getItemAnnotation(
                item_id, GUID_ANNOTATION);
            this.guid_item_map[guid] = item_id;
        } catch(e) {
            //Bindwood.writeError(
            //    "Couldn't find a UUID for itemId: " + itemId, e);
            guid = this.set_guid(item_id, null);
        }
        return guid;
    },

    guid_to_id: function(guid) {
        // Handle special items.
        if (guid == "root_" + this.profile)
            return bookmarksService.placesRoot;
        else if (guid == "toolbar_" + this.profile)
            return  bookmarksService.toolbarFolder;
        else if (guid == "menu_" + this.profile)
            return  bookmarksService.bookmarksMenuFolder;
        else if (guid == "unfiled_" + this.profile)
            return  bookmarksService.unfiledBookmarksFolder;

        // First, try to look it up in our local cache, barring that
        // (which shouldn't happen), look it up slowly.
        var item_id = this.guid_item_map[guid];
        if (item_id)
            return item_id;

        var items = annotationService.getItemsWithAnnotation(
                GUID_ANNOTATION, {});
        for each (item_id in items) {
            var item_guid = annotationService.getItemAnnotation(
                item_id, GUID_ANNOTATION);
            // Save the GUIDs in the cache as we go.
            this.guid_item_map[item_guid] = item_id;
            if (item_guid == guid) {
                return item_id;
            }
        }
        return null;
    },

    _get_folder_root: function(folder_id) {
        var query = historyService.getNewQuery();
        var options = historyService.getNewQueryOptions();
        query.setFolders([folder_id], 1);
        return historyService.executeQuery(query, options).root;
    },

    get_folder_children: function(parent_id) {
        if (parent_id == bookmarksService.placesRoot) {
            // Special case the root folder, returning the three trees
            // we're interested in.
            return [
                bookmarksService.toolbarFolder,
                bookmarksService.bookmarksMenuFolder,
                bookmarksService.unfiledBookmarksFolder];
        }
        var query = historyService.getNewQuery();
        var options = historyService.getNewQueryOptions();
        query.setFolders([parent_id], 1);
        var root = this._get_folder_root(parent_id);

        var children = [];
        root.containerOpen = true;
        for (var i = 0; i < root.childCount; i++) {
            var child = root.getChild(i);
            children.push(child.itemId);
        }
        return children;
    },

    get_all_bookmarks: function() {
        var bookmarks = {}
        bookmarks[this.guid_from_id(bookmarksService.placesRoot)] = true;
        var sync = this;
        var add_bookmarks = function(node) {
            bookmarks[sync.guid_from_id(node.itemId)] = true;

            if (node.type != node.RESULT_TYPE_FOLDER ||
                livemarkService.isLivemark(node.itemId))
                return;
            node.QueryInterface(Ci.nsINavHistoryQueryResultNode);
            node.containerOpen = true;
            for (var i = 0; i < node.childCount; i++) {
                add_bookmarks(node.getChild(i));
            }
        };

        add_bookmarks(this._get_folder_root(
            bookmarksService.toolbarFolder));
        add_bookmarks(this._get_folder_root(
            bookmarksService.bookmarksMenuFolder));
        add_bookmarks(this._get_folder_root(
            bookmarksService.unfiledBookmarksFolder));

        return bookmarks;
    },

    sync: function() {
        Log.debug("Starting two-way synchronisation.");
        this.pullChanges();
        this.pushChanges();
        //this.cleanup();
    },

    pullChanges: function() {
        var info = this.couch.info();
        if (this.last_seq <= info.purge_seq) {
            Log.debug("Retrieving all bookmarks for profile " + this.profile);
            // We haven't pulled any records before, or the database
            // has been compacted past the recorded last_seq.
            this.last_seq = info.update_seq;
            var result = this.couch.view("bindwood/bookmarks", {
                    key: this.profile,
                    include_docs: true,
                });
            for each (var row in result.rows) {
                this.processRecord(row.doc);
            }
        } else {
            Log.debug("Retrieving bookmarks for profile " + this.profile +
                      " since sequence number " + this.last_seq);
            var changes = this.couch.changes({
                    since: this.last_seq,
                    filter: "bindwood/by_profile",
                    profile: this.profile,
                    include_docs: true,
                });
            this.last_seq = changes.last_seq;
            // Iterate backwards through the changes list, keeping
            // only the last change for each record ID.
            var seen_ids = {};
            for (var i = changes.results.length-1; i >= 0; i--) {
                var rev_id = changes.results[i].changes[0].rev;
                if (seen_ids[rev_id]) {
                    changes.results.splice(i, 1);
                }
                seen_ids[rev_id] = true;
            }
            for each (var row in changes.results) {
                this.processRecord(row.doc);
            }
        }
        this.reorder_children();
    },

    processRecord: function (doc) {
        var item_id = this.guid_to_id(doc._id);
        if (item_id == bookmarksService.placesRoot) {
            Log.debug("Ignoring change to places root.\n");
            return;
        }
        // XXX: We should perform duplicate detection here if item_id
        // is null.
        if (item_id != null) {
            this.updateItem(item_id, doc);
        } else {
            this.createItem(doc);
        }
    },

    createItem: function(doc) {
        if (doc._deleted) {
            /* Bookmark must have been created and deleted within the
             * period of one sync cycle, so ignore. */
            Log.debug("Not creating item, since document " + doc._id +
                      " is marked deleted");
            return;
        }
        Log.debug("Creating local item for document " + doc._id);

        // Determine the parent folder.  If it doesn't exist,
        // temporarily add it as an unfiled bookmark to handle later.
        var parent_id = this.guid_to_id(doc.parent_guid);
        var has_parent = (parent_id != null);
        if (!has_parent) {
            parent_id = bookmarksService.unfiledBookmarksFolder;
        }

        var item_id;
        switch (doc.record_type) {
        case TYPE_BOOKMARK:
            var uri = ioService.newURI(doc.uri, null, null);
            item_id = bookmarksService.insertBookmark(
                parent_id, uri, bookmarksService.DEFAULT_INDEX, doc.title);
            break;
        case TYPE_FOLDER:
            item_id = bookmarksService.createFolder(
                parent_id, doc.title, bookmarksService.DEFAULT_INDEX);
            this.reparent_orphans(item_id, doc._id);
            this.folders_to_reorder[doc._id] = doc.children;
            break;
        case TYPE_FEED:
            var site_uri = doc.site_uri ?
                ioService.newURI(doc.site_uri, null, null) : null;
            var feed_uri = ioService.newURI(doc.feed_uri, null, null);
            item_id = livemarkService.createLivemark(
                parent_id, doc.title, site_uri, feed_uri,
                bookmarksService.DEFAULT_INDEX);
            break;
        case TYPE_SEPARATOR:
            item_id = bookmarksService.insertSeparator(
                parent_id, bookmarksService.DEFAULT_INDEX);
            break;
        default:
            item_id = null;
        }
        if (item_id) {
            Log.debug("Created item " + item_id + " for document " + doc._id);
            this.set_guid(item_id, doc._id);
            if (!has_parent) {
                this.set_orphan(item_id, doc.parent_guid);
            }
        }
    },

    updateItem: function(item_id, doc) {
        Log.debug("Updating local item " + item_id + " from document " +
                  doc._id);
        // If we've got this far, then the item ID existed and was
        // mapped to the document's GUID.  But it is possible that the
        // item has been removed locally and we only found the item ID
        // from the cache.  In this case, we ignore the changes and
        // assume the deletion will be pushed back later in this sync
        // cycle.
        try {
            var item_type = bookmarksService.getItemType(item_id)
        } catch (e) {
            Log.debug("Item has been removed locally: ignoring changes.");
            return;
        }

        if (doc._deleted) {
            // Document has been deleted in Couch: propagate to local.
            Log.debug("Removing local item.");
            try {
                bookmarksService.removeItem(item_id);
            } catch (e) {
                Log.exception("Remove failed", e);
                throw e;
            }
            return;
        }

        // If the local item is newer than the version in Couch,
        // ignore.
        if (doc.application_annotations &&
            doc.application_annotations.Firefox &&
            doc.application_annotations.Firefox.last_modified &&
            bookmarksService.getItemLastModified(item_id) >=
            doc.application_annotations.Firefox.last_modified) {
            Log.debug("Ignoring changes, since local item is newer.");
            return;
        }

        var parent_id = this.guid_to_id(doc.parent_guid);
        if (parent_id == null) {
            this.set_orphan(item_id, doc.parent_guid);
        } else {
            if (bookmarksService.getFolderIdForItem(item_id) != parent_id) {
                Log.debug("Moving local item " + item_id + " to new parent " +
                          parent_id);
                bookmarksService.moveItem(
                    item_id, parent_id, bookmarksService.DEFAULT_INDEX);
            }
            annotationService.removeItemAnnotation(
                item_id, PARENT_ANNOTATION);
        }

        switch (doc.record_type) {
        case TYPE_BOOKMARK:
            var uri = ioService.newURI(doc.uri, null, null);
            bookmarksService.setItemTitle(item_id, doc.title);
            bookmarksService.changeBookmarkURI(item_id, uri);
            break;
        case TYPE_FOLDER:
            bookmarksService.setItemTitle(item_id, doc.title);
            this.folders_to_reorder[doc._id] = doc.children;
            break;
        case TYPE_FEED:
            var site_uri = doc.site_uri ?
                ioService.newURI(doc.site_uri, null, null) : null;
            var feed_uri = ioService.newURI(doc.feed_uri, null, null);
            bookmarksService.setItemTitle(item_id, doc.title);
            livemarkService.setSiteURI(item_id, site_uri);
            livemarkService.setFeedURI(item_id, feed_uri);
            break;
        case TYPE_SEPARATOR:
            item_id = bookmarksService.insertSeparator(
                parent_id, bookmarksService.DEFAULT_INDEX);
            break;
        default:
            item_id = null;
        }
    },

    set_orphan: function (item_id, parent_guid) {
        Log.debug("Setting local item " + item_id +
                  " as an orphan of parent " + parent_guid);
        annotationService.setItemAnnotation(
            item_id, PARENT_ANNOTATION, parent_guid,
            0, annotationService.EXPIRE_NEVER);
    },

    reparent_orphans: function (parent_id, parent_guid) {
        Log.debug("Reparenting orphans of " + parent_guid);
        var items = annotationService.getItemsWithAnnotation(
                PARENT_ANNOTATION, {});
        for each (orphan_id in items) {
            if (annotationService.getItemAnnotation(
                orphan_id, PARENT_ANNOTATION) != parent_guid) {
                continue;
            }
            Log.debug("Reclaiming orphaned item " + orphan_id);
            // XXX: check for errors
            bookmarksService.moveItem(
                orphan_id, parent_id, bookmarksService.DEFAULT_INDEX)
            annotationService.removeItemAnnotation(
                orphan_id, PARENT_ANNOTATION);
        }
    },

    reorder_children: function() {
        for (var parent_guid in this.folders_to_reorder) {
            var parent_id = this.guid_to_id(parent_guid);
            if (!parent_id)
                continue;

            Log.debug("Reordering children for local item " + parent_id);
            var children = this.get_folder_children(parent_id);
            var idx = 0;
            for each (child_guid in this.folders_to_reorder[parent_guid]) {
                var child_id = this.guid_to_id(child_guid);
                if (!child_id)
                    continue;

                // If the child exists in the folder, assign its new
                // index.
                var existing_idx = children.indexOf(child_id);
                if (existing_idx >= 0) {
                    bookmarksService.setItemIndex(child_id, idx++);
                    children.splice(existing_idx, 1);
                }
            }
            // Stick the remaining items at the end of the folder.
            for each (var child_id in children) {
                bookmarksService.setItemIndex(child_id, idx++);
            }
        }
        this.folders_to_reorder = {}
    },

    pushChanges: function() {
        Log.debug("Pushing changes.  first_push = " + this.first_push);
        var changed_guids = this.observer.clear_changes();
        if (this.first_push) {
            // First time: process all bookmarks.
            changed_guids = this.get_all_bookmarks();
            this.first_push = false;
        }
        for (var item_guid in changed_guids) {
            this.exportItem(item_guid);
        }
    },

    exportItem: function(item_guid) {
        var item_id = this.guid_to_id(item_guid);
        Log.debug("Exporting item " + item_guid +
                  " (local ID " + item_id + ")");
        var item_type = null, deleted = false;
        if (item_id) {
            // Get the item type, which also checks whether the item exists.
            try {
                item_type = bookmarksService.getItemType(item_id);
            } catch (e) {
                deleted = true;
            }
        } else {
            deleted = true;
        }

        var doc = this.couch.open(item_guid);
        if (deleted) {
            if (doc != null) {
                this.couch.deleteDoc(doc);
            }
            delete this.guid_item_map[item_guid];
            return;
        }

        var changed = false;
        if (doc == null) {
            doc = { _id: item_guid };
            changed = true;
        }

        var _setattr = function(doc, attr, value) {
            if (attr == 'children') {
                // Special handling, since 'children' is an array
                if (!doc[attr] || value.join('\n') != doc[attr].join('\n')) {
                    doc[attr] = value;
                    changed = true;
                }
            } else if (doc[attr] != value) {
                doc[attr] = value;
                changed = true;
            }
        }

        _setattr(doc, "record_type_version", RECORD_TYPE_VERSION);
        if (!doc.application_annotations)
            doc.application_annotations = {};
        if (!doc.application_annotations.Firefox)
            doc.application_annotations.Firefox = {};
        _setattr(doc.application_annotations.Firefox, "profile", this.profile);

        var parent_id = bookmarksService.getFolderIdForItem(item_id);
        if (parent_id > 0) {
            var parent_guid = this.guid_from_id(parent_id);
            _setattr(doc, "parent_guid", parent_guid);
            _setattr(
                doc, "parent_title", bookmarksService.getItemTitle(parent_id));
        }
        switch (item_type) {
        case bookmarksService.TYPE_BOOKMARK:
            _setattr(doc, "record_type", TYPE_BOOKMARK);
            _setattr(doc, "uri", bookmarksService.getBookmarkURI(item_id).spec);
            _setattr(doc, "title", bookmarksService.getItemTitle(item_id));
            break;
        case bookmarksService.TYPE_FOLDER:
            _setattr(doc, "title", bookmarksService.getItemTitle(item_id));
            if (livemarkService.isLivemark(item_id)) {
                _setattr(doc, "record_type", TYPE_FEED);
                var site_uri = livemarkService.getSiteURI(item_id);
                if (site_uri)
                    _setattr(doc, "site_uri", site_uri.spec);
                _setattr(
                    doc, "feed_uri", livemarkService.getFeedURI(item_id).spec);
            } else {
                _setattr(doc, "record_type", TYPE_FOLDER);
                var children = [
                    this.guid_from_id(child_id)
                    for each (child_id in this.get_folder_children(item_id))];
                _setattr(doc, "children", children);
            }
            break;
        case bookmarksService.TYPE_SEPARATOR:
            _setattr(doc, "record_type", TYPE_SEPARATOR);
            _setattr(doc, "position", bookmarksService.getItemIndex(item_id));
            break;
        default:
            Log.error("Can not handle item " + item_id + " of type " +
                      item_type);
            return;
        }
        if (changed) {
            doc.application_annotations.Firefox.last_modified = (
                bookmarksService.getItemLastModified(item_id));
            this.couch.save(doc);
        }
    },
};


function BookmarksObserver(sync) {
    this.sync = sync;
    this.changed_guids = {}
}

BookmarksObserver.prototype = {
    QueryInterface: function(iid) {
        if (iid.equals(Ci.nsINavBookmarkObserver) ||
            iid.equals(Ci.nsINavBookmarkObserver_MOZILLA_1_9_1_ADDITIONS) ||
            iid.equals(Ci.nsISupports)) {
            return this;
        }
        throw Cr.NS_ERROR_NO_INTERFACE;
    },

    should_ignore: function(item_id, folder_id) {
        if (arguments.length < 2) {
            folder_id = bookmarksService.getFolderIdForItem(item_id);
        }
        // Ignore children of Livemarks, since they are automatically
        // created.
        if (livemarkService.isLivemark(folder_id)) {
            return true;
        }
        // XXX: Should also verify that the item is a child of a
        // folder we care about.
        return false;
    },

    record_change: function(item_id) {
        var guid = this.sync.guid_from_id(item_id);
        this.changed_guids[guid] = true;
    },

    clear_changes: function() {
        var changes = this.changed_guids;
        this.changed_guids = {};
        return changes;
    },

    onItemAdded: function(item_id, folder_id, child_index) {
        if (this.should_ignore(item_id, folder_id))
            return;

        // The parent folder is modified, since its child list has
        // changed.
        this.record_change(item_id);
        this.record_change(folder_id);
    },

    onBeforeItemRemoved: function(item_id) {
        folder_id = bookmarksService.getFolderIdForItem(item_id);
        if (this.should_ignore(item_id, folder_id))
            return;
        this.record_change(item_id);
        this.record_change(folder_id);
    },

    onItemChanged: function(item_id, property, is_annotation, value) {
        if (this.should_ignore(item_id))
            return;

        // XXX: We could optimise the list of property changes we
        // respond to like Firefox Sync here.

        // Ignore icon changes.
        if (property == "favicon")
            return;

        this.record_change(item_id);
    },

    onItemMoved: function(item_id, old_parent, old_index,
                          new_parent, new_index) {
        if (this.should_ignore(item_id, new_parent))
            return;

        this.record_change(old_parent);
        if (old_parent != new_parent) {
            // The item has been moved to a different folder, rather
            // than just to a new position within the same folder.
            this.record_change(item_id);
            this.record_change(new_parent);
        }
    },

    // Currently unhandled
    onBeginUpdateBatch: function() {},
    onEndUpdateBatch: function() {},
    onItemRemoved: function(item_id, folder_id, index) {},
    onItemVisited: function(aBookmarkId, aVisitID, time) {},
};
