/*
    Copyright (C) 2012, 2014 Johan Mattsson

    This library is free software; you can redistribute it and/or modify 
    it under the terms of the GNU Lesser General Public License as 
    published by the Free Software Foundation; either version 3 of the 
    License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful, but 
    WITHOUT ANY WARRANTY; without even the implied warranty of 
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
    Lesser General Public License for more details.
*/

using Cairo;
using Math;

namespace BirdFont {

public class GlyphCollection : GLib.Object {
	VersionList versions;
	unichar unicode_character;
	string name;
	bool unassigned = false;

	public GlyphCollection (unichar unicode_character, string name) {
		this.unicode_character = unicode_character;
		this.name = name;
		versions = new VersionList (null, this);
	}

	public GlyphCollection.with_glyph (unichar unicode_character, string name) {
		Glyph g;
		
		this.unicode_character = unicode_character;
		this.name = name;
		
		g = new Glyph (name, unicode_character);
		versions = new VersionList (g, this);
	}

	public void set_unassigned (bool a) {
		unassigned = a;
	}

	public bool is_unassigned () {
		return unassigned;
	}
	
	public void add_glyph (Glyph g) {
		get_version_list ().add_glyph (g);
	}

	public VersionList get_version_list () {
		return versions;
	}
	
	public Glyph get_current () {
		return versions.get_current ();
	}
	
	public void insert_glyph (Glyph g, bool selected) {
		versions.add_glyph (g, selected);		
		assert (versions.glyphs.size > 0);
	}
	
	public uint length () {
		return versions.glyphs.size;
	}
	
	public string get_unicode () {
		StringBuilder unicode = new StringBuilder ();
		unicode.append_unichar (unicode_character);
		return unicode.str;
	}

	public void set_unicode_character (unichar c) {
		unicode_character = c;
	}

	public unichar get_unicode_character () {
		return unicode_character;
	}
		
	public string get_name () {
		return name;
	}
	
	public void set_name (string n) {
		name = n;
	}
	
	public int get_selected_id () {
		return versions.get_current ().version_id;	
	}
	
	public int get_last_id () {
		return versions.get_last_id ();
	}
	
	public void set_selected_version (int version_id) {
		versions.set_selected_version (version_id);
	}
	
	/** Create a copy of this list. This method will copy the list data but 
	 * keep pointers to the original glyphs.
	 * @return a new list with copies of pointers to the glyphs
	 */
	public GlyphCollection copy () {
		GlyphCollection n = new GlyphCollection (unicode_character, name);
		
		foreach (Glyph g in versions.glyphs) {
			n.insert_glyph (g, false);
		}
		
		n.versions.set_selected_version (versions.current_version_id);
		n.unassigned = unassigned;
		
		return n;
	}
}
	
}
