# -*- coding: utf-8 -*-

#  Bluemindo (Base modules)
#  lyrics.py

#    Bluemindo: A really simple but powerful audio player in Python/PyGTK.
#    Copyright (C) 2007-2008  Erwan Briand

#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation version 3 of the License.

#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.

#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.

from gettext import gettext as _
from threading import Thread as pythread
from gobject import idle_add
from os.path import join, isdir, isfile, expanduser
from gtk import WRAP_WORD, STOCK_FILE, WIN_POS_CENTER_ALWAYS
from gtk.glade import XML as glade_XML
from os import makedirs, remove
from urllib import urlopen as urllib_urlopen, quote as urllib_quote
from re import findall as re_findall
import ConfigParser

from common.config import ConfigLoader
from common.functions import Functions

class Lyrics:
    def __init__(self, module):   
        self.module = {'name': 'Lyrics',
                       'logo': STOCK_FILE,
                       'configurable': True}

        module.connect('OnBluemindoStarted', self.loadmodule)
        module.connect('OnModuleConfiguration', self.loadconfig)
        module.connect('OnModuleConfigurationSave', self.saveconfig)
        module.connect('OnPlayNewSong', self.handler_play_new_song)
        module.connect('OnToolLyricsPressed', self.hideorshow)
        module.connect('OnToolReloadLyricsPressed', self.handler_reload_lyrics)
        module.connect('OnToolSaveLyricsPressed', self.handler_save_lyrics)
        
        config = ConfigLoader()
        self.module_conf = join(config.confdir, 'modules', 'lyrics')
        if not isdir(self.module_conf):
            makedirs(self.module_conf)

        self.configfile = ConfigParser.ConfigParser()
        self.configfile_ = self.module['name'] + '.cfg'

        self.default_server = 'lyricwiki.org'
        self.default_active = 'True'

        if not isfile(join(self.module_conf, self.configfile_)):
            self.write_default_config()

        self.module_data = join(config.datadir, 'modules', 'lyrics')
        if not isdir(self.module_data):
            makedirs(self.module_data)

        self.functions = Functions()

    def loadmodule(self, glade):
        self.glade = glade
        self.lyrics_text = glade.get_widget('hbox3')
        self.lyrics_tool = glade.get_widget('toolbar3')
        
        self.lyrics_text.hide()
        self.lyrics_tool.hide()
        
        self.bluemindo = glade.get_widget('vpaned1')
        self.toolbarplaylist = glade.get_widget('toolbar4')
        self.state = 0

    def hideorshow(self):
        if self.state == 0:
            self.state = 1
            self.lyrics_text.show()
            self.lyrics_tool.show()
            self.bluemindo.hide()
            self.toolbarplaylist.hide()

        elif self.state == 1:
            self.state = 0
            self.lyrics_text.hide()
            self.lyrics_tool.hide()
            self.bluemindo.show()
            self.toolbarplaylist.show()

    def handler_play_new_song(self, song):
        title = song[1]
        artist = song[2]
        pythread(None, self.download_lyrics, None, (title, artist), {}).start()

    def download_lyrics(self, title, artist):
        self.configfile.read(join(self.module_conf, self.configfile_))
        lyricsactive = self.configfile.get(self.module['name'], 'active')
        lyricsserver = self.configfile.get(self.module['name'], 'server')
        lyricsfile = join(self.module_data,
                          self.functions.get_hash(title, artist) + '.lyrics')

        lyrics = None

        if not isfile(lyricsfile) and lyricsactive == self.default_active:
            if lyricsserver == 'lyricwiki.org':
                artist.replace(' ', '_')
                title.replace(' ', '_')
                url = ("http://www.lyricwiki.org/%s:%s" %
                       (urllib_quote(artist.encode('utf-8')), 
                        urllib_quote(title.encode('utf-8'))))

                page = urllib_urlopen(url).read(200000)
                for lyrics in re_findall('(?s)<div id="lyric">(.*?)</div>',
                                         page):
                    lyrics = lyrics.replace('<br/>', '\n')

            elif lyricsserver == 'lyrc.com.ar':
                url = (
                    'http://lyrc.com.ar/en/tema1en.php?artist=%s&songname=%s' %
                    (urllib_quote(artist.encode('utf-8')),
                     urllib_quote(title.encode('utf-8'))))
                page = urllib_urlopen(url).read(200000)
                for lyrics in re_findall(
                    '(?s)</script></td></tr></table>(.*?)<br><br><a href="',
                    page):
                    lyrics = lyrics.replace('<br />', '')

            elif lyricsserver == 'lyriki.com':
                artist.replace(' ', '_')
                title.replace(' ', '_')
                url = ('http://www.lyriki.com/%s:%s' %
                       (urllib_quote(artist.encode('utf-8')), 
                        urllib_quote(title.encode('utf-8'))))

                page = urllib_urlopen(url).read(200000)
                for lyrics in re_findall('(?s)<p>(.*?)</p>', page):
                    lyrics = lyrics.replace('<br />', '\n')

            if not lyrics == None:
                file_ = open(lyricsfile, 'w')
                file_.write(lyrics)
                file_.close()

            else:
                lyrics = _('Lyrics not found (for %(title)s - %(artist)s).' % {'title': title, 'artist': artist})

        elif isfile(lyricsfile) and lyricsactive == self.default_active:
            file_ = open(lyricsfile)
            lyrics = file_.read()
            file_.close()

        elif not lyricsactive == self.default_active:
            lyrics = _('Lyrics retrieve disabled.')

        idle_add(self.show_lyrics, lyrics)

    def show_lyrics(self, lyrics):
        text = self.glade.get_widget('textview1')
        text.set_wrap_mode(WRAP_WORD)
        buffer_ = text.get_buffer()
        buffer_.set_text(lyrics)

    def handler_reload_lyrics(self):
        title = unicode(self.glade.get_widget('label1').get_text())
        artist = unicode(self.glade.get_widget('label2').get_text())
        
        if not title == '' and not artist == '':
            lyricsfile = join(self.module_data,
                              self.functions.get_hash(title, artist) + '.lyrics')
        
            if isfile(lyricsfile):
                remove(lyricsfile)

            pythread(None, self.download_lyrics, None,
                     (title, artist), {}).start()
    
    def handler_save_lyrics(self):
        title = unicode(self.glade.get_widget('label1').get_text())
        artist = unicode(self.glade.get_widget('label2').get_text())
        
        if not title == '' and not artist == '':
            lyricsfile = join(self.module_data,
                              self.functions.get_hash(title, artist) + '.lyrics')
            text = self.glade.get_widget('textview1')
            buffer_ = text.get_buffer()
            lyrics = buffer_.get_text(buffer_.get_start_iter(),
                                      buffer_.get_end_iter())
            file_ = open(lyricsfile, 'w')
            file_.write(lyrics)
            file_.close()

    # This function shows the configuration and interface
    def loadconfig(self, (module, confglade)):
        if module == self.module['name']:
            # Load the glade and put the vertical box in the module's
            # configuration one
            self.conf_widgets = glade_XML('modules/lyrics/configuration.glade',
                                          'vbox1', domain='bluemindo')

            hbox = confglade.get_widget('hbox1')

            try:
                kids = hbox.get_children()
                hbox.remove(kids[2])
            except:
                pass

            hbox.add(self.conf_widgets.get_widget('vbox1'))

            try:
                self.configfile.read(join(self.module_conf, self.configfile_))
                form0 = self.configfile.get(self.module['name'], 'active')
                form1 = self.configfile.get(self.module['name'], 'server')

                file_exist = True
            except:
                file_exist = False

            if file_exist:
                combo = self.conf_widgets.get_widget('combobox1')
                chkbox = self.conf_widgets.get_widget('checkbutton1')

                if form1 == self.default_server:
                    combo.set_active(0)
                elif form1 == 'lyrc.com.ar':
                    combo.set_active(1)
                elif form1 == 'lyriki.com':
                    combo.set_active(2)

                if form0 == self.default_active:
                    chkbox.set_active(True)

    # This function saves the configuration
    def saveconfig(self, extension):
        if extension == self.module['name']:
            try:
                field0 = self.conf_widgets.get_widget('combobox1')
                field1 = self.conf_widgets.get_widget('checkbutton1')
                widgets_retrieved = True
            except:
                widgets_retrieved = False

            if widgets_retrieved:
                try:
                    self.configfile.add_section(self.module['name'])
                except:
                    pass

                form0 = field1.get_active()
                form1 = field0.get_model()[field0.get_active()][0]

                self.configfile.set(self.module['name'], 'active', form0)
                self.configfile.set(self.module['name'], 'server', form1)

                self.configfile.write(open(join(self.module_conf,
                                                self.configfile_), 'w'))

    # This function writes the default configuration
    def write_default_config(self):
        self.configfile.add_section(self.module['name'])
        self.configfile.set(self.module['name'], 'active', self.default_active)
        self.configfile.set(self.module['name'], 'server', self.default_server)

        self.configfile.write(open(join(self.module_conf,
                                        self.configfile_), 'w'))
