# -*- coding: utf-8 -*-

#  Bluemindo (Base modules)
#  player.py

#    Bluemindo: A really simple but powerful audio player in Python/PyGTK.
#    Copyright (C) 2007-2008  Erwan Briand

#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation version 3 of the License.

#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.

#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.

from gettext import gettext as _
from xml.parsers.expat import ExpatError
from threading import Thread as pythread
from gobject import idle_add, timeout_add, markup_escape_text
from os.path import join, isdir, isfile, expanduser, exists
from os import makedirs, remove
from gtk import (STOCK_MEDIA_PLAY, STOCK_MEDIA_PAUSE, JUSTIFY_LEFT,
                 FileChooserDialog, FILE_CHOOSER_ACTION_OPEN, STOCK_CANCEL,
                 RESPONSE_CANCEL, STOCK_OPEN, RESPONSE_OK, FileFilter,
                 WIN_POS_CENTER_ALWAYS)
from gtk.gdk import INTERP_BILINEAR, pixbuf_new_from_file, display_get_default
from gtk.glade import XML as glade_XML
from xml.sax.saxutils import unescape, escape
from urllib import urlretrieve
from shutil import copyfile
import ConfigParser
import gst

from libs.amazon import setLicense as amazonkey, searchByKeyword as amazonsearch
from common.config import ConfigLoader
from common.functions import Functions
from media.gstreamer import GStreamer
amazonkey('0T1CC8DE1HQGRWEMBG82')

class Player:
    def __init__(self, module):   
        self.module = {'name': 'Player',
                       'logo': STOCK_MEDIA_PLAY,
                       'configurable': True}

        module.connect('OnBluemindoStarted', self.loadmodule)
        module.connect('OnModuleConfiguration', self.loadconfig)
        module.connect('OnModuleConfigurationSave', self.saveconfig)
        module.connect('OnPreviousPressed', self.handler_previous)
        module.connect('OnStopPressed', self.handler_stop)
        module.connect('OnPlayPressed', self.handler_play)
        module.connect('OnNextPressed', self.handler_next)
        module.connect('OnPlayNewSong', self.handler_play_new_song)
        
        self.instance_module = module
        
        config = ConfigLoader()
        self.module_conf = join(config.confdir, 'modules', 'player')
        if not isdir(self.module_conf):
            makedirs(self.module_conf)

        self.configfile = ConfigParser.ConfigParser()
        self.configfile_ = self.module['name'] + '.cfg'

        self.default_artwork = 'True'
        self.default_title = 'True'
        self.default_popup = 'True'

        if not isfile(join(self.module_conf, self.configfile_)):
            self.write_default_config()

        self.module_data = join(config.datadir, 'modules', 'player')
        if not isdir(self.module_data):
            makedirs(self.module_data)

        self.gst = GStreamer()

        self.functions = Functions()
        self.config = config

    def loadmodule(self, glade):
        self.glade = glade
        self.new_album_state = None

        self.repeat = glade.get_widget('tool-repeat')
        self.tool = glade.get_widget('toolbar2')
        self.tool.hide()
    
        album_cover = glade.get_widget('image1')
        self.image_set(album_cover,
                       self.functions.datadir + '/image/logo_head_big.png', 85)

        alb_btn = glade.get_widget('button1')
        alb_btn.connect('clicked', self.on_album_clicked)
        alb_btn.connect('enter', self.on_album_enter)
        alb_btn.connect('leave', self.on_album_leave)

        glade.get_widget('label2').set_alignment(0, 0.5)
        glade.get_widget('label3').set_alignment(0, 0.5)
        
        glade.get_widget('hscale1').connect('change-value',
                                            self.on_change_value)

    def on_change_value(self, widget, scroll, value):
        # Jump to another place in the song
        seconds = int(value)
        self.gst.seek(seconds)

    def on_album_enter(self, widget):
        self.configfile.read(join(self.module_conf, self.configfile_))
        can_show_popup = self.configfile.get(self.module['name'], 'popup')

        if can_show_popup == 'True':
            # Retrieve album and artist
            artist = self.glade.get_widget('label2').get_text()
            album = self.glade.get_widget('label3').get_text()

            # Test if we have succeeded
            if artist and album:
                # Create a popup window with the album cover
                widg = glade_XML(
                       self.functions.datadir + '/glade/albumpreview.glade')
                main_window = widg.get_widget('window1')
                self.new_album_state = main_window

                # Position
                dsp = display_get_default()
                scr = dsp.get_default_screen()
                tmp1, x0, y0, tmp2 = dsp.get_pointer()
                main_window.move((x0 + 5), (y0 + 5))

                main_window.show()

                # Add the album cover
                pimg = widg.get_widget('image1')
                _file = join(self.module_data,
                             self.functions.get_hash(album, artist) + '.covers')
                if exists(_file):
                    image = [_file, 290]
                else:
                    image = [self.functions.datadir + '/image/logo_head_big.png',
                             285]

                # Change image from pixbuf
                pixbuf = pixbuf_new_from_file(image[0])
                pixbuf = pixbuf.scale_simple(image[1], image[1], INTERP_BILINEAR)
                pimg.set_from_pixbuf(pixbuf)

    def on_album_leave(self, widget):
        # Delete the popup window if exists
        if self.new_album_state:
            self.new_album_state.destroy()

    def on_album_clicked(self, widget):
        # Change album artwork
        album = unicode(self.glade.get_widget('label3').get_text())
        artist = unicode(self.glade.get_widget('label2').get_text())

        if not album == '' and not artist == '':
            file_selector = FileChooserDialog(_('Change album artwork'), None,
                            FILE_CHOOSER_ACTION_OPEN, (STOCK_CANCEL,
                            RESPONSE_CANCEL, STOCK_OPEN, RESPONSE_OK))

            filter_ = FileFilter()
            filter_.add_pattern('*.gif')
            filter_.add_pattern('*.png')
            filter_.add_pattern('*.jpg')
            filter_.add_pattern('*.jpeg')
            file_selector.set_filter(filter_)

            fld = file_selector.run()

            # Select a file, copy it in the covers directory
            file_ = None
            if fld == RESPONSE_OK:
                file_ = file_selector.get_filename()
                if not file_ == None:

                    _file = join(self.module_data,
                                 self.functions.get_hash(album,
                                                         artist) + '.covers')
                    if isfile(_file):
                        remove(_file)

                    copyfile(file_, _file)
                    self.image_set(self.glade.get_widget('image1'), _file, 90)

            file_selector.destroy()

    def handler_previous(self):
        # Handle previous button
        cur =  self.gst.getnow()
        self.gst.stop()

        # If the playlist is not empty
        playlist = self.glade.get_widget('treeview2').get_model()
        
        if len(playlist) > 0:
            n = 0
            id_in_playlist = 0
            for song in playlist:
                if unescape(song[6]) == cur:
                    id_in_playlist = n
                n = n + 1

            _p = 0
            _n = id_in_playlist - 1

            # Load the previous song
            if _n < _p and self.repeat.get_active():
                id_ = len(playlist) - 1
                track = self.functions.clear_bold(unescape(playlist[id_][0]))
                title = self.functions.clear_bold(unescape(playlist[id_][1]))
                artist = self.functions.clear_bold(unescape(playlist[id_][2]))
                album = self.functions.clear_bold(unescape(playlist[id_][3]))
                genre = self.functions.clear_bold(unescape(playlist[id_][4]))
                length = self.functions.clear_bold(unescape(playlist[id_][5]))
                filename = unescape(playlist[id_][6])
                reallength = playlist[id_][7]

                self.instance_module.load_event('OnPlayNewSong', (track, title,
                                                                  artist, album,
                                                                  genre, length,
                                                                  filename,
                                                                  reallength))
            # Don't do anything
            elif _n < _p and not self.repeat.get_active():
                pass
            # Load the previous song
            else:
                track = self.functions.clear_bold(unescape(playlist[_n][0]))
                title = self.functions.clear_bold(unescape(playlist[_n][1]))
                artist = self.functions.clear_bold(unescape(playlist[_n][2]))
                album = self.functions.clear_bold(unescape(playlist[_n][3]))
                genre = self.functions.clear_bold(unescape(playlist[_n][4]))
                length = self.functions.clear_bold(unescape(playlist[_n][5]))
                filename = unescape(playlist[_n][6])
                reallength = playlist[_n][7]

                self.instance_module.load_event('OnPlayNewSong', (track, title,
                                                                  artist, album,
                                                                  genre, length,
                                                                  filename,
                                                                  reallength))
        else:
            # The playlist is empty but we want another song
            self.instance_module.load_event('OnPlayPressedWithoutQueue')

    def handler_next(self):
        # Handle next button
        cur = self.gst.getnow()
        self.gst.stop()

        # If the playlist is not empty
        playlist = self.glade.get_widget('treeview2').get_model()

        if len(playlist) > 0:
            n = 0
            id_in_playlist = -1
            for song in playlist:
                if unescape(song[6]) == cur:
                    id_in_playlist = n
                n = n + 1

            _p = len(playlist) - 1
            _n = id_in_playlist + 1

            # Load the next song
            if _n > _p and self.repeat.get_active():
                track = self.functions.clear_bold(unescape(playlist[0][0]))
                title = self.functions.clear_bold(unescape(playlist[0][1]))
                artist = self.functions.clear_bold(unescape(playlist[0][2]))
                album = self.functions.clear_bold(unescape(playlist[0][3]))
                genre = self.functions.clear_bold(unescape(playlist[0][4]))
                length = self.functions.clear_bold(unescape(playlist[0][5]))
                filename = unescape(playlist[0][6])
                reallength = playlist[0][7]

                self.instance_module.load_event('OnPlayNewSong', (track, title,
                                                                  artist, album,
                                                                  genre, length,
                                                                  filename,
                                                                  reallength))
            # Don't do anything
            elif _n > _p and not self.repeat.get_active():
                pass
            # Load the next song
            else:
                track = self.functions.clear_bold(unescape(playlist[_n][0]))
                title = self.functions.clear_bold(unescape(playlist[_n][1]))
                artist = self.functions.clear_bold(unescape(playlist[_n][2]))
                album = self.functions.clear_bold(unescape(playlist[_n][3]))
                genre = self.functions.clear_bold(unescape(playlist[_n][4]))
                length = self.functions.clear_bold(unescape(playlist[_n][5]))
                filename = unescape(playlist[_n][6])
                reallength = playlist[_n][7]

                self.instance_module.load_event('OnPlayNewSong', (track, title,
                                                                  artist, album,
                                                                  genre, length,
                                                                  filename,
                                                                  reallength))
        else:
            # The playlist is empty but we want another song
            self.instance_module.load_event('OnPlayPressedWithoutQueue')

    def handler_stop(self):
        # Handle stop button
        self.glade.get_widget('tool-play').set_stock_id(STOCK_MEDIA_PLAY)
        self.glade.get_widget('hscale1').set_value(0)
        self.image_set(self.glade.get_widget('image1'),
                       self.functions.datadir + '/image/logo_head_big.png', 85)
        self.glade.get_widget('label1').set_text('')
        self.glade.get_widget('label2').set_text('')
        self.glade.get_widget('label3').set_text('')
        self.glade.get_widget('label5').set_text('')
        self.glade.get_widget('label6').set_text('')

        # If the current playing song is in the playlist, remove emphasis
        playlist = self.glade.get_widget('treeview2').get_model()
        u = -1
        for sng in playlist:
            u += 1
            if unescape(sng[6]) == self.gst.getnow():
                sng = playlist[u]
                playlist[u] = [self.functions.clear_bold(sng[0]),
                               self.functions.clear_bold(sng[1]),
                               self.functions.clear_bold(sng[2]),
                               self.functions.clear_bold(sng[3]),
                               self.functions.clear_bold(sng[4]),
                               self.functions.clear_bold(sng[5]),
                               sng[6], sng[7]
                              ]
        # Stop Gstreamer
        self.gst.stop()
        self.current_playing_file('')

    def handler_play(self):
        """Handle play button"""
        # Get status
        rtn = self.gst.playpause(None)
        sta = self.gst.getstatus()

        playlist = self.glade.get_widget('treeview2').get_model()
        # If the playlist is not empty, load the first in
        if len(playlist) > 0 and (sta == 'NULL' or sta == 'STOP'):
            track = self.functions.clear_bold(unescape(playlist[0][0]))
            title = self.functions.clear_bold(unescape(playlist[0][1]))
            artist = self.functions.clear_bold(unescape(playlist[0][2]))
            album = self.functions.clear_bold(unescape(playlist[0][3]))
            genre = self.functions.clear_bold(unescape(playlist[0][4]))
            length = self.functions.clear_bold(unescape(playlist[0][5]))
            filename = unescape(playlist[0][6])
            reallength = playlist[0][7]

            self.instance_module.load_event('OnPlayNewSong', (track, title,
                                                              artist, album,
                                                              genre, length,
                                                              filename,
                                                              reallength))
        # We don't listen anything but we want to listen music
        elif len(playlist) == 0 and (sta == 'NULL' or sta == 'STOP'):
            self.instance_module.load_event('OnPlayPressedWithoutQueue')

        # Change button to PAUSE if we are playing and vice-versa
        if rtn == 42 or rtn == 'STOPPED' or rtn == 'PAUSED':
            self.glade.get_widget('tool-play').set_stock_id(STOCK_MEDIA_PLAY)
        elif rtn == 'PLAYING':
            self.glade.get_widget('tool-play').set_stock_id(STOCK_MEDIA_PAUSE)

    def handler_play_new_song(self, song):
        # Load a new song
        self.tool.show()
        self.glade.get_widget('toolbutton1').set_expand(True)
        self.glade.get_widget('tool-play').set_stock_id(STOCK_MEDIA_PAUSE)
        self.glade.get_widget('label1').set_markup(
                              '<b>%s</b>' % markup_escape_text(unicode(song[1])))
        self.glade.get_widget('label2').set_markup(
                              '<i>%s</i>' % markup_escape_text(unicode(song[2])))
        self.glade.get_widget('label3').set_text(song[3])
        self.glade.get_widget('label5').set_text('00:00')
        self.glade.get_widget('label6').set_text(' - ' + song[5])

        # Start the timer
        timeout_add(500, self.scale_timer)

        # Download the cover
        pythread(None, self.download_cover, None,
                 (song[3], song[1], song[2]), {}).start()

        # Start Gstreamer
        self.gst.playpause(song[6])
        self.current_playing_file(song[2] + ' - ' + song[1] +
                                            ' (from: ' + song[3] + ')')

        # Change emphasis of current playing song
        playlist = self.glade.get_widget('treeview2').get_model()

        if len(playlist) > 0:
            # Fetching the playlist and get the current playing song id
            i = 0
            for sng in playlist:
                if unescape(sng[6]) == song[6]:
                    break
                i += 1

            sng = playlist[i]
            if (not playlist[i][0].startswith('<b>')
                and not playlist[i][0].endswith('</b>')):
                playlist[i] = ['<b>%s</b>' % sng[0], '<b>%s</b>' % sng[1],
                               '<b>%s</b>' % sng[2], '<b>%s</b>' % sng[3],
                               '<b>%s</b>' % sng[4], '<b>%s</b>' % sng[5],
                               sng[6], sng[7]
                              ]

            # Get the normal text, without bold emphasis
            u = 0
            for sng in playlist:
                if (sng[1].startswith('<b>') and sng[1].endswith('</b>')
                    and  i != u):
                    sng = playlist[u]
                    playlist[u] = [self.functions.clear_bold(sng[0]),
                                   self.functions.clear_bold(sng[1]),
                                   self.functions.clear_bold(sng[2]),
                                   self.functions.clear_bold(sng[3]),
                                   self.functions.clear_bold(sng[4]),
                                   self.functions.clear_bold(sng[5]),
                                   sng[6], sng[7]
                                  ]

                u += 1

    def scale_timer(self):
        # This function updates the seek bar
        try:
            self.glade.get_widget(
                 'label5').set_text(self.functions.great_length(int(
                                    self.gst.getposition() / 1000000000)))
            self.glade.get_widget(
                'hscale1').set_value(int(self.gst.getposition() / 1000000000))
            return True
        except gst.QueryError:
            pass

        state = self.gst.getstatus()
        if state == 'NULL':
            self.handler_next()

    def download_cover(self, album, title, artist):
        # Download an album cover via Amazon and store it into `datadir`
        # If we cannot find any cover, we show the beautiful Bluemindo's
        # logo made by terr1en
        _file = join(self.module_data, self.functions.get_hash(album, artist)
                                                                    + '.covers')
        pimg = self.glade.get_widget('image1')
        self.image_set(pimg, 
                       self.functions.datadir + '/image/logo_head_big.png', 85)

        self.configfile.read(join(self.module_conf, self.configfile_))
        _artwork = self.configfile.get(self.module['name'], 'artwork')
        _title = self.configfile.get(self.module['name'], 'title')

        if _title == self.default_title:
            self.glade.get_widget('window1').set_title(
                'Bluemindo [ ' + title + ' - ' + artist + ' ]')
        else:
            self.glade.get_widget('window1').set_title('Bluemindo')

        if _artwork == self.default_artwork and not isfile(_file):
            try:
                amazonresult = amazonsearch(album.encode('latin1') + ' ' +
                                            artist.encode('latin1'),
                                            product_line='music')
            except (ExpatError, UnicodeEncodeError):
                # This is an ExpatError in PyAmazon, not in Bluemindo O:-)
                amazonresult = 'noresult'

            if not amazonresult == 'noresult':
                url = amazonresult[0].URL
                urlretrieve(amazonresult[0].ImageUrlLarge, _file)

                if exists(_file):
                    self.image_set(pimg, _file, 90)
                else:
                    self.image_set(pimg, self.functions.datadir + 
                                         '/image/logo_head_big.png', 85)
            else:
                self.image_set(pimg, self.functions.datadir +
                                     '/image/logo_head_big.png', 85)

        elif _artwork == self.default_artwork and isfile(_file):
            self.image_set(pimg, _file, 90)

    def image_set(self, pimg, pfile, pscale):
        # Change image from pixbuf
        pixbuf = pixbuf_new_from_file(pfile)
        pixbuf = pixbuf.scale_simple(pscale, pscale, INTERP_BILINEAR)
        idle_add(pimg.set_from_pixbuf, pixbuf)

    def current_playing_file(self, data):
        # Update the current-playing file in `datadir`
        file_ = open(join(self.config.datadir, 'current-playing'), 'wb')
        file_.write(data)
        file_.close()

    # This function shows the configuration and interface
    def loadconfig(self, (module, confglade)):
        if module == self.module['name']:
            # Load the glade and put the vertical box in the module's
            # configuration one
            self.conf_widgets = glade_XML('modules/player/configuration.glade',
                                          'vbox1', domain='bluemindo')

            hbox = confglade.get_widget('hbox1')

            try:
                kids = hbox.get_children()
                hbox.remove(kids[2])
            except:
                pass

            hbox.add(self.conf_widgets.get_widget('vbox1'))

            self.configfile.read(join(self.module_conf, self.configfile_))
            form0 = self.configfile.get(self.module['name'], 'artwork')
            form1 = self.configfile.get(self.module['name'], 'title')
            form2 = self.configfile.get(self.module['name'], 'popup')

            if form0 == self.default_artwork:
                self.conf_widgets.get_widget('checkbutton1').set_active(True)

            if form1 == self.default_title:
                self.conf_widgets.get_widget('checkbutton3').set_active(True)

            if form2 == self.default_popup:
                self.conf_widgets.get_widget('checkbutton4').set_active(True)

    # This function saves the configuration
    def saveconfig(self, extension):
        if extension == self.module['name']:
            try:
                chkbox0 = self.conf_widgets.get_widget('checkbutton1')
                chkbox1 = self.conf_widgets.get_widget('checkbutton3')
                chkbox2 = self.conf_widgets.get_widget('checkbutton4')
                widgets_retrieved = True
            except:
                widgets_retrieved = False

            if widgets_retrieved:
                try:
                    self.configfile.add_section(self.module['name'])
                except:
                    pass

                form0 = chkbox0.get_active()
                form1 = chkbox1.get_active()
                form2 = chkbox2.get_active()

                self.configfile.set(self.module['name'], 'artwork', form0)
                self.configfile.set(self.module['name'], 'title', form1)
                self.configfile.set(self.module['name'], 'popup', form2)

                self.configfile.write(open(join(self.module_conf,
                                                self.configfile_), 'w'))

    # This function writes a default configuration
    def write_default_config(self):
        self.configfile.add_section(self.module['name'])
        self.configfile.set(self.module['name'],
                            'artwork', self.default_artwork)
        self.configfile.set(self.module['name'], 'title', self.default_title)
        self.configfile.set(self.module['name'], 'popup', self.default_popup)

        self.configfile.write(open(join(self.module_conf, self.configfile_),
                                   'w'))
