# -*- coding: utf-8 -*-

#  Bluemindo
#  jabber.py

#    Bluemindo: A really simple but powerful audio player in Python/PyGTK.
#    Copyright (C) 2007-2008  Erwan Briand

#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation version 3 of the License.

#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.

#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.

try:
    from xmpp import NS_PUBSUB, Node, Iq, Client, Presence
    xmpp_imported = True
except ImportError:
    xmpp_imported = False

from gettext import gettext as _
from gtk import WIN_POS_CENTER_ALWAYS
from gtk.glade import XML as glade_XML
from thread import start_new_thread
from base64 import b64decode, b64encode
from os.path import isdir, join
from os import makedirs, chmod
from sys import exit
import ConfigParser

from common.config import ConfigLoader
from common.functions import Functions

class Jabber:
    def __init__(self, plugin):   
        self.plugin = {'name': 'Jabber',
                       'version': 0.1,
                       'logo': 'plugins/jabber/jabber_small.png',
                       'configurable': True,
                       'authors': 'Erwan Briand <erwan@codingteam.net>',
                       'license': 'GNU General Public License 3',
                       'description': _('Send PEP notifications via your '
                                        'Jabber account.')}

        plugin.connect('OnModuleConfiguration', self.loadconfig)
        plugin.connect('OnModuleConfigurationSave', self.saveconfig)
        plugin.connect('OnPlayNewSong', self.handler_play_new_song)
        plugin.connect('OnBluemindoQuitted', self.close_jabber_connection)
        
        config = ConfigLoader()
        self.plugin_conf = join(config.confdir, 'plugins', 'jabber')
        if not isdir(self.plugin_conf):
            makedirs(self.plugin_conf)

        self.configfile = ConfigParser.ConfigParser()
        self.configfile_ = self.plugin['name'] + '.cfg'
        
        self.functions = Functions()

        # Get config
        try:
            self.configfile.read(join(self.plugin_conf, self.configfile_))
            form0 = self.configfile.get(self.plugin['name'], 'username')
            form1 = self.configfile.get(self.plugin['name'], 'password')
            file_exist = True
        except:
            file_exist = False

        # Get Jabber account
        if file_exist and not form0 == '' and not form1 == '':
            self.is_jabber_account = False
            self.jid = form0

            pwd = b64decode(form1)

            try:
                user = self.jid.split('@')[0]
                server = self.jid.split('@')[1]
            except IndexError:
                return

            # Start the connection to the Jabber server
            self.connection = Client(server)

            def connect_to_account(server, user, pwd):
                if not self.connection.connect():
                    raise Exception('Unable to connect to %s' % server)
                if not self.connection.auth(user=user,
                                            password=pwd,
                                            resource='Bluemindo'):
                    raise Exception('Unable to authenticate as %s' % (self.jid))

                self.is_jabber_account = True

                # We _really_ don't care about the roster
                self.connection.sendInitPresence(requestRoster=0)

                # And we don't give a **** with messages too :)
                self.connection.send(Presence(frm=self.jid, priority=-42))

                while 1:
                    try:
                        self.connection.Process(5)
                    except:
                        exit(0)

            # Thread the conection
            start_new_thread(connect_to_account, (server, user, pwd))
        else:
            self.is_jabber_account = False

    def handler_play_new_song(self, song):
        title = song[1]
        artist = song[2]
        album = song[3]
        track = song[0]
        length = song[7]

        if self.is_jabber_account:
            # Test if connection is active, if not, reconnect and reauth
            if not self.connection.isConnected():
                self.connection.reconnectAndReauth()

            # Create the PEP stanza
            item = Node('tune', {'xmlns': 'http://jabber.org/protocol/tune'})

            itemchild = item.addChild('artist')
            itemchild.addData(artist)

            itemchild = item.addChild('source')
            itemchild.addData(album)

            itemchild = item.addChild('title')
            itemchild.addData(title)

            itemchild = item.addChild('track')
            itemchild.addData(track)

            itemchild = item.addChild('length')
            itemchild.addData(length)

            query = Iq('set', to=self.jid)
            querychild = query.addChild('pubsub', namespace=NS_PUBSUB)
            querychild = querychild.addChild('publish',
                         {'node': 'http://jabber.org/protocol/tune'})
            querychild = querychild.addChild('item', {}, [item])

            # Send the PEP notification
            self.connection.send(query)

    # This function disconnects from the Jabber account
    def close_jabber_connection(self):
        if self.is_jabber_account:
            self.is_jabber_account = False
            self.connection.send(Presence(frm=self.jid, typ='unavailable'))
            del self.connection

    # This function shows the configuration and interface
    def loadconfig(self, (module, confglade)):
        if module == self.plugin['name']:
            # Load the glade and put the vertical box in the plugin's
            # configuration one
            self.conf_widgets = glade_XML('plugins/jabber/configuration.glade',
                                          'vbox1', domain='bluemindo')

            hbox = confglade.get_widget('hbox2')

            try:
                kids = hbox.get_children()
                hbox.remove(kids[2])
            except:
                pass

            hbox.add(self.conf_widgets.get_widget('vbox1'))

            try:
                self.configfile.read(join(self.plugin_conf, self.configfile_))
                form0 = self.configfile.get(self.plugin['name'], 'username')
                form1 = self.configfile.get(self.plugin['name'], 'password')

                file_exist = True
            except:
                file_exist = False

            if file_exist:
                self.conf_widgets.get_widget('entry1').set_text(form0)
                self.conf_widgets.get_widget('entry2').set_text(b64decode(form1))

    # This function saves the configuration
    def saveconfig(self, extension):
        if extension == self.plugin['name']:
            try:
                field0 = self.conf_widgets.get_widget('entry1')
                field1 = self.conf_widgets.get_widget('entry2')
                widgets_retrieved = True
            except:
                widgets_retrieved = False

            if widgets_retrieved:
                try:
                    self.configfile.add_section(self.plugin['name'])
                except:
                    pass

                form0 = field0.get_text()
                form1 = field1.get_text()

                self.configfile.set(self.plugin['name'], 'username', form0)
                self.configfile.set(self.plugin['name'],
                                    'password', b64encode(form1))

                self.configfile.write(open(join(self.plugin_conf,
                                                self.configfile_), 'w'))
                chmod(join(self.plugin_conf, self.configfile_), 0600)
