/*
 *
 *  Headset Profile support for Linux
 *
 *  Copyright (C) 2006  Fabien Chevalier
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <syslog.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/un.h>
#include <unistd.h>
#include <stdlib.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/sdp.h>
#include <bluetooth/l2cap.h>

#include "sdp.h"

sdp_session_t *sdp_headset_register(uint8_t channel)
{
	sdp_list_t *svclass, *pfseq, *apseq, *root, *aproto;
	uuid_t root_uuid, l2cap, rfcomm, hag, ga;
	sdp_profile_desc_t profile[1];
	sdp_list_t *proto[2];
	int status;
	sdp_session_t *sdp_session;
	sdp_record_t *sdp_record;

	sdp_session = sdp_connect(BDADDR_ANY, BDADDR_LOCAL, 0);
	if (!sdp_session) {
		syslog(LOG_ERR, "Failed to connect to the local SDP server. %s(%d)", 
				strerror(errno), errno);
		return 0;
	}

	sdp_record = sdp_record_alloc();
	if (!sdp_record) {
		syslog(LOG_ERR, "Failed to allocate service record");
		sdp_close(sdp_session);
		return 0;
	}

	sdp_uuid16_create(&root_uuid, PUBLIC_BROWSE_GROUP);
	root = sdp_list_append(NULL, &root_uuid);
	sdp_set_browse_groups(sdp_record, root);

	sdp_uuid16_create(&l2cap, L2CAP_UUID);
	proto[0] = sdp_list_append(NULL, &l2cap);
	apseq    = sdp_list_append(NULL, proto[0]);

	sdp_uuid16_create(&rfcomm, RFCOMM_UUID);
	proto[1] = sdp_list_append(NULL, &rfcomm);
	proto[1] = sdp_list_append(proto[1], sdp_data_alloc(SDP_UINT8, &channel));
	apseq    = sdp_list_append(apseq, proto[1]);

	aproto   = sdp_list_append(NULL, apseq);
	sdp_set_access_protos(sdp_record, aproto);

	sdp_uuid16_create(&hag, HEADSET_AGW_PROFILE_ID);    /* Headset Audio gateway */
	sdp_uuid16_create(&ga,  GENERIC_AUDIO_SVCLASS_ID);  /* Generic Audio */

	svclass = sdp_list_append(NULL, &hag);
	svclass = sdp_list_append(svclass, &ga);
	sdp_set_service_classes(sdp_record, svclass);

	sdp_uuid16_create(&profile[0].uuid, HEADSET_AGW_SVCLASS_ID); /* Audio Gateway */
	profile[0].version = 0x0100;
	pfseq = sdp_list_append(NULL, &profile[0]);
	sdp_set_profile_descs(sdp_record, pfseq);

	sdp_set_info_attr(sdp_record, "Audio Gateway", NULL, NULL);

	status = sdp_device_record_register(sdp_session, BDADDR_ANY, sdp_record, 0);
	sdp_record_free(sdp_record);
	if (status) {
		sdp_close(sdp_session);
		syslog(LOG_ERR, "SDP registration failed.");
		return 0;
	}
	return sdp_session;
}
