/*
*
*  BlueZ - Bluetooth protocol stack for Linux
*
*  Copyright (C) 2004-2005  Marcel Holtmann <marcel@holtmann.org>
*
*
*  This library is free software; you can redistribute it and/or
*  modify it under the terms of the GNU Lesser General Public
*  License as published by the Free Software Foundation; either
*  version 2.1 of the License, or (at your option) any later version.
*
*  This library is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*  Lesser General Public License for more details.
*
*  You should have received a copy of the GNU Lesser General Public
*  License along with this library; if not, write to the Free Software
*  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <errno.h>
#include <malloc.h>
#include <signal.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <syslog.h>

#include <alsa/asoundlib.h>
#include <alsa/pcm_external.h>
#include <alsa/timer.h>

#include "../a2dp.h"
#include "a2dpd_protocol.h"
#include "a2dpd_protocol.h"
#include "a2dpd_timer.h"
#include "a2dpd_ipc.h"

char* g_prefix="PCM.A2DPD";
FILE* g_fdout = DEFAULTFDOUT;
int g_bdebug = DEFAULTDEBUG;

typedef struct snd_pcm_a2dp {
	snd_pcm_ioplug_t io;
	int opened_for;
	int streamid;
	int rateconstraint;
	int rate;
	int sk;
	snd_pcm_sframes_t last_pointer;
	int channels;
	snd_pcm_sframes_t num;
	unsigned int frame_bytes;
	TIMERINFO TimerInfos;
} snd_pcm_a2dp_t;

static int a2dp_disconnect(snd_pcm_a2dp_t * a2dp)
{
	close_socket(&a2dp->sk);
	return 0;
}

static int a2dp_connect(snd_pcm_a2dp_t * a2dp)
{
	if (a2dp->sk < 0) {
		a2dp->sk = make_client_socket();
	}
	return a2dp->sk;
}

static int a2dp_send_constraint(snd_pcm_a2dp_t * a2dp)
{
	int result = -1;
	int32_t client_type = a2dp->opened_for;
	if (a2dp->sk >= 0) {
		if (send_socket(a2dp->sk, &client_type, sizeof(client_type)) == sizeof(client_type)) {
			// Fill stream informations
			AUDIOSTREAMINFOS StreamInfos = INVALIDAUDIOSTREAMINFOS;
			StreamInfos.rate = a2dp->rate;
			StreamInfos.channels = a2dp->channels;
			StreamInfos.bitspersample = a2dp->frame_bytes/a2dp->channels;
			StreamInfos.streamid = a2dp->streamid;
			StreamInfos.buffer_size = a2dp->io.buffer_size;
			StreamInfos.start_threshold = a2dp->io.buffer_size;
			switch(a2dp->io.format) {
			case SND_PCM_FORMAT_S8:     StreamInfos.format = A2DPD_PCM_FORMAT_S8; break;
			case SND_PCM_FORMAT_U8:     StreamInfos.format = A2DPD_PCM_FORMAT_U8; break;
			case SND_PCM_FORMAT_S16_LE: StreamInfos.format = A2DPD_PCM_FORMAT_S16_LE; break;
			default: StreamInfos.format = A2DPD_PCM_FORMAT_UNKNOWN; break;
			}
			if (send_socket(a2dp->sk, &StreamInfos, sizeof(StreamInfos)) == sizeof(StreamInfos)) {
				DBG2("Connected a2dp %p, sk %d, fps %ld", a2dp, a2dp->sk, (long)(a2dp->TimerInfos.fpsX/TIMERFACTOR));
				result = 0;
			} else {
				a2dp_disconnect(a2dp);
			}
		} else {
			a2dp_disconnect(a2dp);
		}
	}
	return result;
}

static snd_pcm_a2dp_t *a2dp_alloc(void)
{
	snd_pcm_a2dp_t *a2dp;
	a2dp = malloc(sizeof(*a2dp));
	if (a2dp) {
		memset(a2dp, 0, sizeof(*a2dp));
		a2dp->sk = -1;
	}
	return a2dp;
}

static void a2dp_free(snd_pcm_a2dp_t * a2dp)
{
	a2dp_disconnect(a2dp);
	free(a2dp);
}

static int a2dp_start(snd_pcm_ioplug_t * io)
{
	int transfer = 0;
	snd_pcm_a2dp_t *a2dp = io->private_data;
	GETDELAYREQ  StreamReq = { A2DPD_PLUGIN_START };
	DBG2("");
	transfer = send_socket(a2dp->sk, &StreamReq, sizeof(StreamReq));
	if(transfer < 0)
		DBG("Error during request");
	return (transfer>=0)?0:transfer;
}

static int a2dp_stop(snd_pcm_ioplug_t * io)
{
	int transfer = 0;
	snd_pcm_a2dp_t *a2dp = io->private_data;
	GETDELAYREQ  StreamReq = { A2DPD_PLUGIN_STOP };
	DBG2("");
	transfer = send_socket(a2dp->sk, &StreamReq, sizeof(StreamReq));
	if(transfer < 0)
		DBG("Error during request");
	return (transfer>=0)?0:transfer;
}

static snd_pcm_sframes_t a2dp_pointer(snd_pcm_ioplug_t * io)
{
	snd_pcm_a2dp_t *a2dp = io->private_data;
	int transfer = 0;
	snd_pcm_sframes_t result = -EPIPE;
	GETPOINTERRESP GetPointerResp = { 0 };
	int accepted = 0;

	if(poll_error(a2dp->sk, 0)) {
		return a2dp->io.appl_ptr;
	}

	// Purge incoming datas to get latest delay sent by daemon
	if(!poll_accept(a2dp->sk, 0)) {
		// Sleep if nothing is to be received now
		DBG4("Sleeping");

		//FIXME  Mmmm love these bits of code
		// xmms loops calling get_avail then usleep, amarok loops calling get_avail without sleeping
		// If we don't wait, amarok will make cpu spin
		// If we wait, xmms will wait too much and this will cause underruns
		// Before waiting, make sure buffer is contains enough data
		if(((long)a2dp->io.appl_ptr - (long)a2dp->last_pointer) > (long)a2dp->io.buffer_size/2)
			usleep(1000);
	}

	// Purge incoming datas to get latest delay sent by daemon
	while(poll_accept(a2dp->sk, 0)) {
		transfer = recv_socket(a2dp->sk, &GetPointerResp, sizeof(GetPointerResp));
		if(transfer != sizeof(GetPointerResp)) {
			break;
		}
		accepted = 1;
	}

	switch(a2dp->io.state) {
	case SND_PCM_STATE_RUNNING:
		if(accepted) {
			 if(transfer == sizeof(GetPointerResp)) {

				if(GetPointerResp.pcm_state != A2DPD_PCM_STATE_XRUN) {
					DBG3("Received");
					result = GetPointerResp.hw_ptr;
					a2dp->last_pointer = result;
				} else {
					DBG("Underrun returned");
					a2dp->io.state = SND_PCM_STATE_XRUN;
					result = a2dp->num;
				}
/*
				if(a2dp->io.appl_ptr != 0 && (long int)a2dp->io.appl_ptr == (long int)a2dp->io.hw_ptr) {
					DBG("Underrun detected");
					result = -EPIPE;
				} else {
					DBG3("Received");
					result = GetPointerResp.hw_ptr;
	
					// FIXME looks like a2dpd send big bug values
					if(result > a2dp->io.appl_ptr) {
						DBG("(result:%ld, num:%ld) filtering result to %ld ", a2dp->num, result, a2dp->num);
						result = a2dp->io.appl_ptr;
					}
	
					a2dp->last_pointer = result;
				}*/
			} else {
				DBG3("Error while receiving");
				result = -1;
			}
		} else {
			DBG3("Not accepted");
			result = a2dp->last_pointer;
		}
		break;
	case SND_PCM_STATE_PREPARED:
		result = a2dp->io.hw_ptr;
		break;
	case SND_PCM_STATE_SETUP:
	case SND_PCM_STATE_XRUN:
	default:
		result = a2dp->io.appl_ptr;
		break;
	}

	DBG3("result:%ld alsa_appl:%ld, a2dpd_hw:%ld, state:%s/a2dpd_state:%s", 
		(long)result,
		(long)a2dp->io.appl_ptr,
		(long)GetPointerResp.hw_ptr,
		(a2dp->io.state == SND_PCM_STATE_OPEN)?"open":
		(a2dp->io.state == SND_PCM_STATE_SETUP)?"setup":
		(a2dp->io.state == SND_PCM_STATE_PREPARED)?"prepared":
		(a2dp->io.state == SND_PCM_STATE_RUNNING)?"running":
		(a2dp->io.state == SND_PCM_STATE_XRUN)?"xrun":
		(a2dp->io.state == SND_PCM_STATE_DRAINING)?"draining":
		(a2dp->io.state == SND_PCM_STATE_PAUSED)?"paused":
		(a2dp->io.state == SND_PCM_STATE_SUSPENDED)?"suspended":
		(a2dp->io.state == SND_PCM_STATE_DISCONNECTED)?"disconnected":
		"*other*",
		(GetPointerResp.pcm_state == A2DPD_PCM_STATE_OPEN)?"open":
		(GetPointerResp.pcm_state == A2DPD_PCM_STATE_SETUP)?"setup":
		(GetPointerResp.pcm_state == A2DPD_PCM_STATE_PREPARED)?"prepared":
		(GetPointerResp.pcm_state == A2DPD_PCM_STATE_RUNNING)?"running":
		(GetPointerResp.pcm_state == A2DPD_PCM_STATE_XRUN)?"xrun":
		(GetPointerResp.pcm_state == A2DPD_PCM_STATE_DRAINING)?"draining":
		(GetPointerResp.pcm_state == A2DPD_PCM_STATE_PAUSED)?"paused":
		(GetPointerResp.pcm_state == A2DPD_PCM_STATE_SUSPENDED)?"suspended":
		(GetPointerResp.pcm_state == A2DPD_PCM_STATE_DISCONNECTED)?"disconnected":
		"*other*");

	DBG8("pointer=%ld app=%ld queue_len=%ld free=%ld", (long)result, (long)a2dp->io.appl_ptr, (long)a2dp->io.appl_ptr - (long)result, (long)a2dp->io.buffer_size-((long)a2dp->io.appl_ptr - (long)result));

	return result;
}

// This is the main transfer func which does the transfer and sleep job
static snd_pcm_sframes_t a2dp_transfer2(snd_pcm_ioplug_t * io, char *buf, int32_t datatoread)
{
	snd_pcm_a2dp_t *a2dp = io->private_data;
	int transfer = 0;
	GETDELAYREQ  StreamReq = { A2DPD_PLUGIN_PCM_WRITE_3 };
	AUDIOPACKETHEADER hdr;
	gettimeofday(&hdr.packet_date, NULL);
	hdr.pcm_buffer_size = datatoread;

#if 1
	static struct timeval send_date = { 0,0 };
	static struct timeval prev_date = { 0,0 };
	struct timeval send_delay = { 0,0 };

	gettimeofday(&send_date, NULL);
	if(prev_date.tv_sec != 0) {
		int idelay;
		timersub(&send_date, &prev_date, &send_delay);
		idelay = send_delay.tv_sec*1000 + send_delay.tv_usec/1000;
		if(idelay>0)
			DBG8("send %d (delay=%3d ms)", datatoread/4, idelay);
	}
	prev_date = send_date;
#endif

	// Connect if needed and send
	if (transfer >= 0)
		transfer = send_socket(a2dp->sk, &StreamReq, sizeof(StreamReq));
	if (transfer >= 0)
		transfer = send_socket(a2dp->sk, &hdr, sizeof(hdr));
	if (transfer >= 0)
		transfer = send_socket(a2dp->sk, buf, hdr.pcm_buffer_size);

	// Disconnect if error detected
	if (transfer < 0) {
		DBG("Error while sending");
		a2dp_disconnect(a2dp);
	}

	// update pointer, tell alsa we're done
	a2dp->num += datatoread / a2dp->frame_bytes;

	return datatoread / a2dp->frame_bytes;
}

// also works but sleeps between transfers
// This is the main transfer func which does the transfer and sleep job
static snd_pcm_sframes_t a2dp_transfer(snd_pcm_ioplug_t * io, const snd_pcm_channel_area_t * areas, snd_pcm_uframes_t offset, snd_pcm_uframes_t nframes)
{
	snd_pcm_a2dp_t *a2dp = io->private_data;
	snd_pcm_sframes_t totaltransfered = 0;

	if(poll_error(a2dp->sk, 0)) {
		return nframes;
	}

	if(a2dp->io.state == SND_PCM_STATE_XRUN) {
		DBG("Got an underrun");
		return -EPIPE;
	}

#define ENABLE_HACK_XMMS
#ifdef ENABLE_HACK_XMMS
	if(a2dp->io.state != SND_PCM_STATE_RUNNING && a2dp->io.state != SND_PCM_STATE_PREPARED) {
		DBG("Transfer while not running and not prepared");
		//return -EBADFD;
		//FIXME alsa says transfer should return BADFD if state is not running and not prepared
		// xmms only checks the XRUN state before calling prepare() else a warning is displayed
		// When calling snd_pcm_drop() then snd_pcm_write(), the result is not EPIPE so xrun_recover() is not done
		a2dp->io.state = SND_PCM_STATE_XRUN;
		return -EPIPE;
	}
#endif

	DBG3("To transfer : %ld", nframes);

	while (totaltransfered < nframes) {
		char *buf = (char *) areas->addr + (areas->first + areas->step * offset) / 8;
		// Do not send more than io.period_size
		int datatoread = min(a2dp->io.period_size, (nframes-totaltransfered)) * a2dp->frame_bytes;
		//int datatoread = nframes * a2dp->frame_bytes;
		snd_pcm_sframes_t transfered = a2dp_transfer2(io, buf, datatoread);
		if (transfered > 0) {
			offset += transfered;
			totaltransfered += transfered;
			DBG3("totaltransfered+=%ld", transfered);
		} else {
			break;
		}
	}

	DBG3("totaltransfered=%ld", totaltransfered);
	return totaltransfered;
}

// This is the main transfer func which does the transfer and sleep job
static snd_pcm_sframes_t a2dp_read2(snd_pcm_ioplug_t * io, char *buf, int32_t datatoread)
{
	snd_pcm_a2dp_t *a2dp = io->private_data;
	int transfer = 0;
	uint32_t pkt_size = 0;
	int dummy = 0;
	int delay = 0;

	DBG3("writing 2");
	if(transfer>=0)
		transfer = send_socket(a2dp->sk, &dummy, sizeof(dummy));

	DBG3("reading 2 = %d", transfer);
	if(transfer>=0)
		transfer = recv_socket(a2dp->sk, &pkt_size, sizeof(pkt_size));
	DBG3("got size %d:%d", transfer, pkt_size);
	if(transfer>=0) {
		if(pkt_size<datatoread) {
			transfer = recv_socket(a2dp->sk, buf, min(datatoread, pkt_size));
			DBG3("got data %d/%d", transfer, pkt_size);
		} else {
			DBG("invalid pkt size");
		}
	}

	// update pointer, tell alsa we're done
	if(transfer>=0)
		a2dp->num += transfer / a2dp->frame_bytes;

	// Disconnect if error detected
	if (transfer < 0) {
		a2dp_disconnect(a2dp);
		transfer = 0;
	}

	a2dp_timer_notifyframe(&a2dp->TimerInfos);
	delay = a2dp_timer_sleep(&a2dp->TimerInfos, 0);
	//usleep(delay);

	return transfer / a2dp->frame_bytes;
}

static snd_pcm_sframes_t a2dp_read(snd_pcm_ioplug_t *io,
				  const snd_pcm_channel_area_t *areas,
				  snd_pcm_uframes_t offset,
				  snd_pcm_uframes_t nframes)
{
	snd_pcm_a2dp_t *a2dp = io->private_data;

	DBG3("reading");

	char *buf = (char *) areas->addr + (areas->first + areas->step * offset) / 8;
	int datatoread = min(A2DPD_BLOCK_SIZE, nframes * a2dp->frame_bytes);
	snd_pcm_uframes_t totaltransfered = a2dp_read2(io, buf, datatoread);
	return totaltransfered;
}


static int a2dp_close(snd_pcm_ioplug_t * io)
{
	snd_pcm_a2dp_t *a2dp = io->private_data;

	DBG2("");

	a2dp_disconnect(a2dp);
	a2dp_free(a2dp);
	DBG2("OK");
	return 0;
}

static int a2dp_params(snd_pcm_ioplug_t * io, snd_pcm_hw_params_t * params)
{
	long block = ((long)A2DPD_BLOCK_SIZE);
	snd_pcm_a2dp_t *a2dp = io->private_data;
	unsigned int period_bytes;

	a2dp->frame_bytes = (snd_pcm_format_physical_width(io->format) * io->channels) / 8;

	period_bytes = io->period_size * a2dp->frame_bytes;

	DBG2("format %s rate %d channels %d", snd_pcm_format_name(io->format), io->rate, io->channels);
	DBG2("period_size %ld buffer_size %ld", io->period_size, io->buffer_size);

	a2dp->rate = io->rate;
	a2dp->channels = io->channels;

	a2dp->TimerInfos.fpsX = a2dp->rate * a2dp->frame_bytes * TIMERFACTOR / block;

	if(a2dp->opened_for == A2DPD_PLUGIN_PCM_READ) {
		block = 48;
	}

	DBG2("block %ld, %ld fps", block, (long)(a2dp->TimerInfos.fpsX / TIMERFACTOR));

	return a2dp_send_constraint(a2dp);
}

static int a2dp_prepare(snd_pcm_ioplug_t * io)
{
	int transfer = 0;
	snd_pcm_a2dp_t *a2dp = io->private_data;
	GETDELAYREQ  StreamReq = { A2DPD_PLUGIN_PREPARE, a2dp->io.buffer_size };
	GETPOINTERRESP GetPointerResp = { 0 };

	a2dp->last_pointer = 0;
	a2dp->num = 0;

	DBG2("");

	if(a2dp->opened_for == A2DPD_PLUGIN_PCM_READ) {
		// ALSA library is really picky on the fact num is not null. If it is, capture won't start 
		a2dp->num = io->period_size;
	}

	transfer = send_socket(a2dp->sk, &StreamReq, sizeof(StreamReq));
	if(transfer < 0)
		DBG("Error during transfer");

	// Purge incoming datas to get alls delays sent by daemon
	while(poll_accept(a2dp->sk, 0)) {
		transfer = recv_socket(a2dp->sk, &GetPointerResp, sizeof(GetPointerResp));
	}

	return 0;
}

/*
static int a2dp_drain(snd_pcm_ioplug_t * io)
{
	int transfer = 0;
	snd_pcm_a2dp_t *a2dp = io->private_data;
	GETDELAYREQ  StreamReq = { A2DPD_PLUGIN_DRAIN };
	DBG2("");
	transfer = send_socket(a2dp->sk, &StreamReq, sizeof(StreamReq));
	if(transfer < 0)
		DBG("Error during request");
	return (transfer>=0)?0:transfer;
}

static int a2dp_pause(snd_pcm_ioplug_t * io, int enabled)
{
	int transfer = 0;
	snd_pcm_a2dp_t *a2dp = io->private_data;
	GETDELAYREQ  StreamReq = { enabled?A2DPD_PLUGIN_PAUSE_1:A2DPD_PLUGIN_PAUSE_0 };
	DBG2("");
	transfer = send_socket(a2dp->sk, &StreamReq, sizeof(StreamReq));
	if(transfer < 0)
		DBG("Error during request");
	return (transfer>=0)?0:transfer;
}

static int a2dp_resume(snd_pcm_ioplug_t * io)
{
	int transfer = 0;
	snd_pcm_a2dp_t *a2dp = io->private_data;
	GETDELAYREQ  StreamReq = { A2DPD_PLUGIN_RESUME };
	DBG2("");
	transfer = send_socket(a2dp->sk, &StreamReq, sizeof(StreamReq));
	if(transfer < 0)
		DBG("Error during request");
	return (transfer>=0)?0:transfer;
}

static int a2dp_descriptors_count(snd_pcm_ioplug_t * io)
{
	snd_pcm_a2dp_t *a2dp = io->private_data;
	return a2dp->sk > 0 ? 1 : 0;
}

static int a2dp_descriptors(snd_pcm_ioplug_t * io, struct pollfd *pfds, unsigned int space)
{
	snd_pcm_a2dp_t *a2dp = io->private_data;
	if (space < 1) {
		DBG("Can't fill in descriptors");
		SNDERR("Can't fill in descriptors");
		return 0;
	}
	if (a2dp->sk <= 0) {
		DBG("Not connected");
		SNDERR("Not connected");
		return 0;
	}

	// Alsa does make sure writing now will not block
	pfds[0].fd = a2dp->sk;
	pfds[0].events = POLLOUT;
	pfds[0].revents = 0;
	return 1;
}

static int a2dp_poll_revents(snd_pcm_ioplug_t *io, struct pollfd *pfds, unsigned int nfds, unsigned short *revents)
{
	snd_pcm_a2dp_t *a2dp = io->private_data;
	*revents = 0;

	if (pfds[0].revents & POLLOUT) {
		//DBG("POLLOUT");

		// If underrun state, then transfer is no longer called
		// So alsa keeps polling forever
		if(a2dp->io.state == SND_PCM_STATE_XRUN) {
			DBG("Underrun detected while polling");
			return -EPIPE;
		}
	} else if (pfds[0].revents & POLLHUP) {
		DBG("POLLHUP");

		a2dp_disconnect(a2dp);
	} else {
		DBG("Non prevu");
	}

	return 0;
}

static int a2dp_delay(snd_pcm_ioplug_t *io, snd_pcm_sframes_t *delayp)
{
	snd_pcm_a2dp_t *a2dp = io->private_data;
	int result = -EPIPE;
	snd_pcm_sframes_t delay = a2dp->num % a2dp->io.buffer_size ;

	a2dp_pointer(io);

	if(a2dp->io.state != SND_PCM_STATE_XRUN) {
		if(delay >= a2dp->last_pointer) {
			*delayp = delay - a2dp->last_pointer;
			result = 0 ;
		} else {
			*delayp = a2dp->io.buffer_size - a2dp->last_pointer + delay ;
			result = 0 ;
		}
	} else {
		*delayp = 0;
		result = -1;
	}
	DBG("num=%04ld ptr=%04ld delay==%04ld, result=%d", delay, a2dp->last_pointer, *delayp, result);
	return result;
}
*/
static snd_pcm_ioplug_callback_t a2dp_write_callback = {
	.close = a2dp_close,
	.start = a2dp_start,
	.stop = a2dp_stop,
	.prepare = a2dp_prepare,
	.pointer = a2dp_pointer,
	.transfer = a2dp_transfer,
	.hw_params = a2dp_params,
	//.drain = a2dp_drain,
	//.pause = a2dp_pause,
	//.resume = a2dp_resume
	//.poll_descriptors_count = a2dp_descriptors_count,
	//.poll_descriptors = a2dp_descriptors,
	//.poll_revents = a2dp_poll_revents,
	//.delay = a2dp_delay,
};

static snd_pcm_ioplug_callback_t a2dp_read_callback = {
	.close = a2dp_close,
	.start = a2dp_start,
	.stop = a2dp_stop,
	.prepare = a2dp_prepare,
	.pointer = a2dp_pointer,
	.transfer = a2dp_read,
	.hw_params = a2dp_params,
};

// Alsa can convert about any format/channels/rate to any other rate
// However, since we added some code in the daemon to convert, why not do it ourselves!!!
// Moreover some player like aplay won't play a wav file if the device that do not natively support the requested format
// If you want alsa to do the conversion, just remove the value you want to see converted
static int a2dp_constraint(snd_pcm_a2dp_t * a2dp)
{
	snd_pcm_ioplug_t *io = &a2dp->io;
	snd_pcm_access_t access_list[] = { SND_PCM_ACCESS_RW_INTERLEAVED, SND_PCM_ACCESS_MMAP_INTERLEAVED };
	unsigned int formats[] = { SND_PCM_FORMAT_U8, SND_PCM_FORMAT_S8, SND_PCM_FORMAT_S16_LE };
	unsigned int channels[] = { 2 };
	unsigned int rates[] = { 8000, 11025, 22050, 32000, 44100, 48000 };
	int formats_nb = ARRAY_SIZE(formats);
	int channels_nb = ARRAY_SIZE(channels);
	int rates_nb = ARRAY_SIZE(rates);
	int rate_daemon = 0;
	int rate_prefered = 0;
	int periods = 2;
	int period_bytes = 32 * 1024 ;
	char srcfilename[512];
	int err;

	if(a2dp->opened_for == A2DPD_PLUGIN_PCM_WRITE_3) {
		get_config_filename(srcfilename, sizeof(srcfilename));
		// Default is same as the daemon
		rate_daemon = read_config_int(srcfilename, "a2dpd", "rate", A2DPD_FRAME_RATE);
		// If a value is specified, use it
		rate_prefered = read_config_int(srcfilename, "a2dpd", "plugin-rate", rate_daemon);

		// If a constraint is defined for the plugin use it
		if(a2dp->rateconstraint != -1) {
			rate_prefered = a2dp->rateconstraint;
		}

		// If this value is not 0, alsa will convert to plugin-rate
		if(rate_prefered != 0) {
			// use defaults settings the rate specified + 16 bits stereo
			rates[0] = rate_prefered;
			rates_nb = 1;
			formats[0] = SND_PCM_FORMAT_S16_LE;
			formats_nb = 1;
			channels[0] = 2;
			channels_nb = 1;
			DBG2("%d", rate_prefered);
		} else {
			// If this value is 0, the daemon will do most conversions
		}
		DBG2("A2DPD_PLUGIN_PCM_WRITE_3 %d", rate_prefered);
	} else if(a2dp->opened_for == A2DPD_PLUGIN_PCM_READ) {
		// Capture is only available using SCO
		// 8000 hz, 16bits, Mono
		rates[0] = 8000;
		rates_nb = 1;
		formats[0] = SND_PCM_FORMAT_S16_LE;
		formats_nb = 1;
		channels[0] = 1;
		channels_nb = 1;
		DBG2("A2DPD_PLUGIN_PCM_READ %d", 8000);
	}

	err = snd_pcm_ioplug_set_param_list(io, SND_PCM_IOPLUG_HW_ACCESS, ARRAY_SIZE(access_list), access_list);
	if (err < 0)
		return err;

	err = snd_pcm_ioplug_set_param_list(io, SND_PCM_IOPLUG_HW_FORMAT, formats_nb, formats);
	if (err < 0)
		return err;

	err = snd_pcm_ioplug_set_param_list(io, SND_PCM_IOPLUG_HW_CHANNELS, channels_nb, channels);
	if (err < 0)
		return err;

	err = snd_pcm_ioplug_set_param_list(io, SND_PCM_IOPLUG_HW_RATE, rates_nb, rates);
	if (err < 0)
		return err;

	err = snd_pcm_ioplug_set_param_minmax(io, SND_PCM_IOPLUG_HW_PERIOD_BYTES, period_bytes, period_bytes);
	if (err < 0)
		return err;

	err = snd_pcm_ioplug_set_param_minmax(io, SND_PCM_IOPLUG_HW_PERIODS, periods, periods);
	if (err < 0)
		return err;

	return 0;
}

SND_PCM_PLUGIN_DEFINE_FUNC(a2dpd)
{
	snd_pcm_a2dp_t *a2dp = NULL;
	snd_config_iterator_t i, next;
	int err = 0;
	long rateconstraint = -1;
	long streamid = 0;
	long debug = 0;

	DBG("AlsaVersion=%d.%d.%d, Opened for %s", SND_LIB_MAJOR, SND_LIB_MINOR, SND_LIB_SUBMINOR, stream == SND_PCM_STREAM_PLAYBACK ? "Playback" : "Capture");

	snd_config_for_each(i, next, conf) {
		snd_config_t *n = snd_config_iterator_entry(i);
		const char *id;

		if (snd_config_get_id(n, &id) < 0)
			continue;

		// Alsa specific options
		if (!strcmp(id, "comment") || !strcmp(id, "type"))
			continue;

		// Ignore old options
		if (strstr("ipaddr bdaddr port src dst use_rfcomm", id))
			continue;

		// rate of the plugin (overwrite plugin-rate in .a2dprc)
		if (!strcmp(id, "rateconstraint")) {
			if (snd_config_get_integer(n, &rateconstraint) < 0) {
				SNDERR("Invalid type for %s", id);
				return -EINVAL;
			}
			continue;
		}

		// streamid (to be used later)
		if (!strcmp(id, "streamid")) {
			if (snd_config_get_integer(n, &streamid) < 0) {
				SNDERR("Invalid type for %s", id);
				return -EINVAL;
			}
			continue;
		}

		// debug level
		if (!strcmp(id, "debug")) {
			if (snd_config_get_integer(n, &debug) < 0) {
				SNDERR("Invalid type for %s", id);
				return -EINVAL;
			}
			continue;
		}

		SNDERR("Unknown field %s", id);
		return -EINVAL;
	}

	g_bdebug = debug;

	a2dp = a2dp_alloc();
	if (!a2dp) {
		SNDERR("Can't allocate plugin data");
		return -ENOMEM;
	}

	// Notify plugin
	a2dp->rateconstraint = rateconstraint;
	a2dp->streamid = streamid;
	a2dp->io.version = SND_PCM_IOPLUG_VERSION;
	a2dp->io.name = "Bluetooth Advanced Audio Distribution";
	a2dp->io.mmap_rw = 0;
	a2dp->io.poll_fd     =  1; /* Do not use poll !! */
	a2dp->io.poll_events =  POLLOUT; /* Do not use poll !! */
	a2dp->io.callback = (stream == SND_PCM_STREAM_PLAYBACK)?(&a2dp_write_callback):(&a2dp_read_callback);
	a2dp->io.private_data = a2dp;
	a2dp->opened_for = (stream == SND_PCM_STREAM_PLAYBACK)?(A2DPD_PLUGIN_PCM_WRITE_3):(A2DPD_PLUGIN_PCM_READ);

	if ((err = snd_pcm_ioplug_create(&a2dp->io, name, stream, mode)) < 0)
		goto error;

	if((err = a2dp_connect(a2dp))<0)
		goto error;

	if ((err = a2dp_constraint(a2dp)) < 0)
		goto error;

	*pcmp = a2dp->io.pcm;
	return 0;

      error:

	// Calls snd_pcm_close
	if(a2dp->io.pcm != NULL)
		snd_pcm_ioplug_delete(&a2dp->io);

	return err;
}

SND_PCM_PLUGIN_SYMBOL(a2dpd);
