#ifndef __SBC_MATH_H
#define __SBC_MATH_H

#include <stdint.h>

#define fabs(x) ((x) < 0 ?-(x) : (x))

#ifdef USE_FIXED

#ifdef USE_FIXED32

#define SCALE_PROTO4_TBL 18
#define SCALE_ANA4_TBL 20
#define SCALE_PROTO8_TBL 18
#define SCALE_ANA8_TBL 20
#define SCALE4_STAGE1_BITS 14
#define SCALE4_STAGE2_BITS 18
#define SCALE4_STAGED1_BITS 18
#define SCALE4_STAGED2_BITS 18
#define SCALE8_STAGE1_BITS 14
#define SCALE8_STAGE2_BITS 18
#define SCALE8_STAGED1_BITS 18
#define SCALE8_STAGED2_BITS 18

typedef int32_t sbc_fixed_t;

#define SBC_FIXED_0(val) {val = 0;}
#define DIV2(dst, src) {dst = src >> 1;}

#define ADD(dst, src)  {dst += src;}
#define SUB(dst, src)  {dst -= src;}
#define MUL(dst, a, b) {dst = a*b;}
#define MULA(dst, a, b)  {dst += a*b;}
#define SCALE4_STAGE1(src) (src >> SCALE4_STAGE1_BITS)
#define SCALE4_STAGE2(src) (src >> SCALE4_STAGE2_BITS)
#define SCALE4_STAGED1(src) (src >> SCALE4_STAGED1_BITS)
#define SCALE4_STAGED2(src) (src >> SCALE4_STAGED2_BITS)
#define SCALE8_STAGE1(src) (src >> SCALE8_STAGE1_BITS)
#define SCALE8_STAGE2(src) (src >> SCALE8_STAGE2_BITS)
#define SCALE8_STAGED1(src) (src >> SCALE8_STAGED1_BITS)
#define SCALE8_STAGED2(src) (src >> SCALE8_STAGED2_BITS)

#else // USE_FIXED32

#define SCALE4_STAGE1_BITS 24
#define SCALE4_STAGE2_BITS 6
#define SCALE8_STAGE1_BITS 24
#define SCALE8_STAGE2_BITS 7

struct sbc_fixed_struct{
	int32_t hi;
	uint32_t lo;
};
union sbc_fixed_union {
	int64_t whole;
	struct sbc_fixed_struct p;
};
typedef struct sbc_fixed_struct sbc_fixed_t;

#define SBC_FIXED_0(val) {val.hi = 0;val.lo = 0;}
#define DIV2(dst, src) {dst.hi = src.hi >> 1; dst.lo = (src.lo >> 1); if (src.hi & 1) dst.lo |= 0x80000000; else dst.lo &= 0x7FFFFFFF;}

#ifdef __arm__

#define ADD(dst, src)  \
    asm ("adds	%0, %0, %2\n\t"  \
	 "adc	%1, %1, %3"	\
	 : "+r" (dst.lo), "+r" (dst.hi)  \
	 : "%r" (src.lo), "r" (src.hi))

#define SUB(dst, src)  \
    asm ("subs	%0, %0, %2\n\t"  \
	 "sbc	%1, %1, %3"	\
	 : "+r" (dst.lo), "+r" (dst.hi)  \
	 : "%r" (src.lo), "r" (src.hi))

#define MUL(dst, a, b) \
    asm ("smull	%0, %1, %2, %3"  \
	 : "=&r" (dst.lo), "=&r" (dst.hi)  \
	 : "%r" (a), "r" (b))

#define MULA(dst, a, b)  \
    asm ("smlal	%0, %1, %2, %3"  \
	 : "+r" (dst.lo), "+r" (dst.hi)  \
	 : "%r" (a), "r" (b))

#define SCALE4_STAGE1(src) \
    ({ int32_t __result;  \
    asm ("movs	%0, %1, lsr %3\n\t"  \
	 "adc	%0, %0, %2, lsl %4"  \
	 : "=&r" (__result)  \
	 : "r" (src.lo), "r" (src.hi), \
	 "M" (SCALE4_STAGE1_BITS), "M" (32 - SCALE4_STAGE1_BITS)  \
	 : "cc");  \
    __result;  \
    })

#define SCALE4_STAGE2(src) (src.hi >> SCALE4_STAGE2_BITS)

#define SCALE8_STAGE1(src) \
    ({ int32_t __result;  \
    asm ("movs	%0, %1, lsr %3\n\t"  \
	 "adc	%0, %0, %2, lsl %4"  \
	 : "=&r" (__result)  \
	 : "r" (src.lo), "r" (src.hi), \
	 "M" (SCALE8_STAGE1_BITS), "M" (32 - SCALE8_STAGE1_BITS)  \
	 : "cc");  \
    __result;  \
    })

#define SCALE8_STAGE2(src) (src.hi >> SCALE8_STAGE2_BITS)
#define SCALE4_STAGED1(src) (src.hi)
#define SCALE4_STAGED2(src) (src.hi)
#define SCALE8_STAGED1(src) (src.hi)
#define SCALE8_STAGED2(src) (src.hi)
#else // _arm_

#define ADD(dst, src)  {dst.lo += src.lo; dst.hi += src.hi; if ( dst.lo < src.lo ) dst.hi++;}
#define SUB(dst, src)  {dst.hi -= src.hi; if ( dst.lo < src.lo ) dst.hi--; dst.lo -= src.lo;}
#define MUL(dst, a, b) {int64_t x; x = (int64_t)a * b; dst.hi = x >> 32; dst.lo = x;}
#define MULA(dst, a, b)  {int64_t x; x = (int64_t)a * b; dst.hi += (x >> 32); dst.lo += x; if (dst.lo < (uint32_t)x) dst.hi++;}
#define SCALE4_STAGE1(src) ((int32_t)((src.hi << (32 - SCALE4_STAGE1_BITS)) | ((src.lo >> SCALE4_STAGE1_BITS) & (0xFFFFFFFF >> SCALE4_STAGE1_BITS))))
#define SCALE4_STAGE2(src) (src.hi >> SCALE4_STAGE2_BITS)
#define SCALE4_STAGED1(src) (src.hi)
#define SCALE4_STAGED2(src) (src.hi)
#define SCALE8_STAGE1(src) ((int32_t)((src.hi << (32 - SCALE8_STAGE1_BITS)) | ((src.lo >> SCALE8_STAGE1_BITS) & (0xFFFFFFFF >> SCALE8_STAGE1_BITS))))
#define SCALE8_STAGE2(src) (src.hi >> SCALE8_STAGE2_BITS)
#define SCALE8_STAGED1(src) (src.hi)
#define SCALE8_STAGED2(src) (src.hi)

#endif // __arm__

#endif // USE_FIXED32

#else // USE_FIXED

typedef double sbc_fixed_t;

#define SBC_FIXED_0(val) {val = 0;}
#define DIV2(dst, src) {dst = src / 2;}
#define ADD(dst, src)  {dst += src;}
#define SUB(dst, src)  {dst -= src;}
#define MUL(dst, a, b) {dst = a*b;}
#define MULA(dst, a, b)  {dst += a*b;}
#define SCALE4_STAGE1(src) (src)
#define SCALE4_STAGE2(src) (src)
#define SCALE4_STAGED1(src) (src)
#define SCALE4_STAGED2(src) (src)
#define SCALE8_STAGE1(src) (src)
#define SCALE8_STAGE2(src) (src)
#define SCALE8_STAGED1(src) (src)
#define SCALE8_STAGED2(src) (src)

#endif // USE_FIXED

#endif // __SBC_MATH_H
