/*  BMPx - The Dumb Music Player
 *  Copyright (C) 2005 BMPx development team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *

 * The BMPx project hereby grant permission for non-gpl compatible GStreamer
 * plugins to be used and distributed together with GStreamer and BMPx. This
 * permission are above and beyond the permissions granted by the GPL license
 * BMPx is covered by.
 */

/*
 *
 * BMPx Bookmarks XBEL based backend code
 *
 * (C) 2005-2006 M. Derezynski
 *
 * XBEL: http://pyxml.sourceforge.net/topics/xbel/docs/html/xbel.html
 *
 * We use only the following structure:
 *
 * <xbel version="1.0">
 *    <folder>
 *  <title></title>
 *
 *  <bookmark href="">
 *    <title></title>
 *  </bookmark>
 *
 *      ...
 *
 *   </folder>
 *
 */

#include <cstring>
#include <iostream>

#include <glib/gprintf.h>
#include <glibmm.h>
#include <gtkmm.h>

#include "main.hh"
#include "paths.hh"
#include "bookmarks.hh"
#include "xml.hh"

#include <dbus/dbus-glib.h>
#include <dbus/dbus-gtype-specialized.h>

#include <bmp/dbus.hh>

/* XBEL 1.0 standard entities */
#define XBEL_VERSION        "1.0"
#define XBEL_DTD_NICK       "xbel"
#define XBEL_DTD_SYSTEM     "+//IDN python.org//DTD XML Bookmark " \
                          "Exchange Language 1.0//EN//XML"

#define XBEL_DTD_URI        "http://www.python.org/topics/xml/dtds/xbel-1.0.dtd"

#define XBEL_ELEMENT_ROOT       "xbel"
#define XBEL_ELEMENT_FOLDER     "folder"
#define XBEL_ELEMENT_BOOKMARK   "bookmark"
#define XBEL_ELEMENT_ALIAS      "alias"
#define XBEL_ELEMENT_SEPARATOR  "separator"
#define XBEL_ELEMENT_TITLE      "title"
#define XBEL_ELEMENT_DESC       "desc"
#define XBEL_ELEMENT_INFO       "info"
#define XBEL_ELEMENT_METADATA   "metadata"

#define XBEL_ATTRIBUTE_VERSION  "version"
#define XBEL_ATTRIBUTE_FOLDED   "folded"
#define XBEL_ATTRIBUTE_OWNER    "owner"
#define XBEL_ATTRIBUTE_ADDED    "added"
#define XBEL_ATTRIBUTE_VISITED  "visited"
#define XBEL_ATTRIBUTE_MODIFIED "modified"
#define XBEL_ATTRIBUTE_ID       "id"
#define XBEL_ATTRIBUTE_HREF     "href"
#define XBEL_ATTRIBUTE_REF      "ref"

#define XBEL_YES_VALUE          "yes"
#define XBEL_NO_VALUE           "no"

/* date and times inside XBEL entities should be expressed using ISO 8601
 * formats; we use the extended date and time format, relative to UTC
 */
#define ISO_8601_FORMAT "%Y-%m-%dT%H:%M:%SZ" /* strftime(3) format */
#define ISO_8601_LEN    21

namespace Bmp
{
  namespace Streams
  {
    GArray*
    Bookmarks::get_array ()
    {
      using namespace Glib;
      using namespace Gtk;

      GArray *array = g_array_new (TRUE, TRUE, sizeof(G_TYPE_VALUE_ARRAY)); 
      typedef TreeModel::Children Nodes;
      Nodes nodes = bookmarks->children();

      for (Nodes::const_iterator    node  = nodes.begin (), end = nodes.end ();
                                    node != end; ++node)
        {
          Gtk::TreeModel::const_iterator const& cur = (*node);

          const ustring title ((*cur)[columns.title]);
          const ustring descr ((*cur)[columns.desc]);
          const ustring href  ((*cur)[columns.href]);

          GValueArray *varray = g_value_array_new (3);
          GValue value = {0, };

          g_value_init (&value, G_TYPE_STRING);
          g_value_take_string (&value, strdup(title.c_str()));
          g_value_array_append (varray, &value);
    
          g_value_take_string (&value, strdup(descr.c_str()));
          g_value_array_append (varray, &value);

          g_value_take_string (&value, strdup(href.c_str()));
          g_value_array_append (varray, &value);

          g_array_append_val (array, varray);
        }

        return array;
    }

    bool
    Bookmarks::append_node (Gtk::TreeModel::iterator const& iter, xmlNodePtr root)
    {
      Gtk::TreeRow  row;
      xmlNodePtr    bookmark;

      row = *iter;
      bookmark = xmlNewNode (NULL, BAD_CAST XBEL_ELEMENT_BOOKMARK);
      xmlAddChild (root, bookmark);

      const Glib::ustring title(row[columns.title]);
      const Glib::ustring descr(row[columns.desc]);
      const Glib::ustring href(row[columns.href]);

      xmlNewTextChild (bookmark, NULL, BAD_CAST XBEL_ELEMENT_TITLE, BAD_CAST title.c_str());
      xmlNewTextChild (bookmark, NULL, BAD_CAST XBEL_ELEMENT_DESC,  BAD_CAST descr.c_str());

      xmlSetProp (bookmark, BAD_CAST XBEL_ATTRIBUTE_HREF, BAD_CAST href.c_str());

      return false;
    }

    Bookmarks::Bookmarks ()
    {
      xmlDocPtr         doc;
      xmlXPathObjectPtr xpathobj;

      bookmarks = Gtk::ListStore::create (columns);

      std::string filename = Glib::build_filename (BMP_PATH_USER_DIR, "bookmarks.xbel");
      if (!Glib::file_test (filename.c_str(), Glib::FILE_TEST_EXISTS)) return;

      doc = xmlParseFile (filename.c_str());
      if  (!doc) return;

      const char *xpath = "/" XBEL_ELEMENT_ROOT "//" XBEL_ELEMENT_BOOKMARK;

      xpathobj = xml_execute_xpath_expression (doc, BAD_CAST xpath, NULL);

      if (!xpathobj) return;
      if (!xpathobj->nodesetval) return;
      if (!xpathobj->nodesetval->nodeNr) return;

      for (int n = 0; n < xpathobj->nodesetval->nodeNr; n++)
      {
        Gtk::TreeModel::iterator model_iter;

        enum
        {
                STR_TITLE,
                STR_HREF,
                STR_DESC,
                N_STRINGS
        };

        char            *STR[N_STRINGS];
        xmlNodePtr       node;
        xmlNodePtr       sibling;

        STR[STR_HREF]    = NULL;
        STR[STR_TITLE]   = NULL;
        STR[STR_DESC]    = NULL;

        node = xpathobj->nodesetval->nodeTab[n];
        sibling = node->children;

        STR[STR_HREF] = reinterpret_cast<char *> (xmlGetProp (node, BAD_CAST XBEL_ATTRIBUTE_HREF));

        while (sibling)
            {
              if ((sibling->type == XML_ELEMENT_NODE) && (sibling->children))
                  {
                      if (!std::strcmp (reinterpret_cast<const char *> (sibling->name),
                                        reinterpret_cast<const char *> (XBEL_ELEMENT_TITLE)))
                      {
                          STR[STR_TITLE]  = reinterpret_cast<char *> (XML_GET_CONTENT (sibling->children));
                      }

                      if (!std::strcmp (reinterpret_cast<const char *> (sibling->name),
                                        reinterpret_cast<const char *> (XBEL_ELEMENT_DESC)))
                      {
            
                          STR[STR_DESC]   = reinterpret_cast<char *> (XML_GET_CONTENT (sibling->children));
                      }
                  }

              sibling = sibling->next;
            }

        if (STR[STR_HREF])
          {
            model_iter = bookmarks->append ();

            if (STR[STR_TITLE])
              {
                (*model_iter)[columns.title] = STR[STR_TITLE];
              }

            if (STR[STR_DESC])
              {
                (*model_iter)[columns.desc]  = STR[STR_DESC];
              }

            (*model_iter)[columns.href]  = STR[STR_HREF];

            g_free (STR[STR_TITLE]);
            g_free (STR[STR_HREF]);
            g_free (STR[STR_DESC]);
          }
      }
    }
  
    void
    Bookmarks::save ()
    {
      xmlDocPtr doc = xmlNewDoc (BAD_CAST "1.0");
      xmlNodePtr root = xmlNewNode (NULL, BAD_CAST XBEL_ELEMENT_ROOT);
      xmlDocSetRootElement (doc, root);
      if (!bookmarks->children (). empty ())
        {
          bookmarks->foreach_iter (sigc::bind (sigc::mem_fun (this, &Bmp::Streams::Bookmarks::append_node), root));
        }
      xmlChar *data;
      int size;
      xmlDocDumpFormatMemoryEnc (doc, &data, &size, "UTF-8", 1);
      std::string filename = Glib::build_filename (BMP_PATH_USER_DIR, "bookmarks.xbel");
      g_file_set_contents (filename.c_str (), reinterpret_cast<char *>(data), size, NULL);
      xmlFreeDoc (doc);
    }

    Bookmarks::~Bookmarks ()
    {
      save ();
    }
  }//!Streams:: 
}//!Bmp::
