//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifndef BMP_LAST_FM_HH
#define BMP_LAST_FM_HH

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <vector>
#include <queue>

#include <glibmm.h>
#include <glibmm/markup.h>
#include <gtkmm.h>
#include <libglademm.h>
#include <mcs/mcs.h>
#include <exception>

// XML Queue Format
namespace Bmp
{
  namespace LastFM
  {

/*
    <track mbid="" date="" length="" artist="" track="" album=""/>
*/

    struct LQMItem
    { 
      Glib::ustring   mbid;
      time_t          date;
      unsigned int    length;
      Glib::ustring   artist;
      Glib::ustring   track;
      Glib::ustring   album;
    };
  }
}

namespace std
{
    template <>
    struct less<Bmp::LastFM::LQMItem>
    {
      bool operator() (Bmp::LastFM::LQMItem const& x, Bmp::LastFM::LQMItem const& y)
      {
        return (x.date > y.date);
      }
    };
}

namespace Bmp
{
  namespace LastFM
  {
    typedef std::vector< LQMItem > VLQM; 
    typedef std::priority_queue< LQMItem, VLQM > LQMQueueBase;
    
    struct LQMQueue : public LQMQueueBase
    {
      public:
        LQMItem const& get_nth (LQMQueueBase::size_type item) const { return c[item]; }
    };

    class LQMQueueParser : public Glib::Markup::Parser
    {
        public:

          LQMQueueParser (LQMQueue & q) : m_queue (q), state (0) {}
          virtual ~LQMQueueParser () {}

          void check_sanity ();

        protected:

          virtual void
          on_start_element  (Glib::Markup::ParseContext & context,
                             Glib::ustring const&         name,
                             AttributeMap  const&         props);
          virtual void
          on_end_element    (Glib::Markup::ParseContext & context,
                             Glib::ustring const&         name);

        private:

          enum Element
          {
              E_NONE        = 0, 
              E_LQM         = 1 << 0,
              E_QUEUE       = 1 << 1,
              E_TRACK       = 1 << 2,
          };

          LQMQueue & m_queue; 
          int state;
    };

    void load_lqm (std::string const& filename, LQMQueue & queue);
    void save_lqm (std::string const& filename, LQMQueue const& queue);

  } // namespace LastFM
} // namespace Bmp

namespace Bmp
{
  namespace LastFM
  {

    void neon_status_check (int result, int status, std::string const& errorMessage, std::string const& response = std::string());

#include "exception.hh"

    EXCEPTION(NetworkError)
    EXCEPTION(ConnectionError)
    EXCEPTION(TimeOutError)
    EXCEPTION(ResourceNotFoundError)
    EXCEPTION(AuthenticationError)
    EXCEPTION(ResponseError)
    EXCEPTION(RequestError)

    EXCEPTION(LastFMBadauthError)
    EXCEPTION(LastFMNotConnectedError)
    EXCEPTION(LastFMStreamTuningError)

    struct Friend
    { 
      Glib::ustring	username;
      Glib::ustring	url;
      Glib::ustring	image;
    };
    typedef std::vector< Friend > Friends;

    class FriendParser  : public Glib::Markup::Parser
    {
        public:

          FriendParser (Friends & r) : friends (r), state (0) {}
          virtual ~FriendParser () {}

          Friends & friends; 

        protected:

          virtual void
          on_start_element  (Glib::Markup::ParseContext   &context,
                             Glib::ustring const          &element_name,
                             const AttributeMap           &attributes);
          virtual void
          on_end_element    (Glib::Markup::ParseContext   &context,
                             Glib::ustring const          &element_name);
          virtual void
          on_text	          (Glib::Markup::ParseContext   &context,
                             Glib::ustring const          &text);
        private:

            enum Element
            {
              E_NONE	= 0, 
              E_USER  = 1 << 0,
              E_URL 	= 1 << 1,
              E_IMAGE = 1 << 2,
            };

            Friend  current;
            int     state;
    };

    void get_friends (Glib::ustring const& username, Friends& friends);
    void get_neighbours (Glib::ustring const& username, Friends& friends);
  
    struct Tag
    {
      Glib::ustring name;
      unsigned int  count;
      Glib::ustring url;
    };
    typedef std::vector< Tag > Tags;

    enum TagOrigin
    {
      TAG_ORIGIN_USER,
      TAG_ORIGIN_GLOBAL,
    };

    enum TagType
    {
      TAGS_TOPTAGS,
      TAGS_ARTIST,
      TAGS_ALBUM,
      TAGS_TRACK, 
    };

    class TagParser  : public Glib::Markup::Parser
    {
        public:
  
          TagParser (Tags& t, TagType type, TagOrigin origin) : tags (t), m_type (type), m_origin (origin), state (0) {}
          virtual ~TagParser () {}

          Tags & tags; 

          TagType m_type;
          TagOrigin m_origin;

        protected:

          virtual void
          on_start_element  (Glib::Markup::ParseContext   &context,
                             Glib::ustring const          &element_name,
                             const AttributeMap           &attributes);
          virtual void
          on_end_element    (Glib::Markup::ParseContext   &context,
                             Glib::ustring const          &element_name);
          virtual void
          on_text	          (Glib::Markup::ParseContext   &context,
                             Glib::ustring const          &text);
        private:

            enum Element
            {
              E_NONE	= 0, 
              E_TAG   = 1 << 0,
              E_NAME  = 1 << 1,
              E_COUNT = 1 << 2,
              E_URL   = 1 << 3,
            };

            Tag     current;
            int     state;
    };
    void get_tags (TagType type,
                   TagOrigin origin,
                   Tags & tags,
                   Glib::ustring const& username = Glib::ustring(),
                   Glib::ustring const& id1 = Glib::ustring(),
                   Glib::ustring const& id2 = Glib::ustring());


    /** Client code for Last.FM Radio
     *
     */
    class Radio
    {
      public:

        struct Session
        {
            Glib::ustring session;
            Glib::ustring streamurl;
            bool          subscriber;
            bool          framehack;
            Glib::ustring base_url;
            Glib::ustring base_path;

            Session () : subscriber (false), framehack (false) {}
        };

        struct Metadata
        {
            Glib::ustring station;
            Glib::ustring station_url;
            Glib::ustring artist;
            Glib::ustring artist_url;
            Glib::ustring track;
            Glib::ustring track_url;
            Glib::ustring album;
            Glib::ustring album_url;
            Glib::ustring albumcover_large;
            unsigned int  duration;
            bool          radiomode;
            bool          streaming;
            bool          rtp;

            Metadata () : duration (0), radiomode (false), streaming (false), rtp (false) {}
        };

        Radio ();
        ~Radio ();

        enum Command
        {
          LOVE,
          SKIP,
          BAN,
          RTP,
          NORTP
        };

        void  connect   ();
        void  tune      (std::string& url);
        void  metadata  (Metadata& metadata) const;
        void  command   (Command command); 

        void  session   (Session& session) const;
        bool  connected () const;
        
      private:

        Session m_session;
        bool    m_connected;

    };

    /** Client code for Last.FM
     *
     * Bmp::LastFM::Scrobbler is a client implementation of Audioscrobbler Protocol 1.1,
     * (http://www.audioscrobbler.com), used by Last.FM (http://last.fm), a public,
     * free service that tracks your listening habits and provides you with statistics
     * and friends. 
     *
     */
    class Scrobbler
    {
      public:

        Scrobbler ();
        ~Scrobbler ();

        // Signals
        typedef sigc::signal<void, unsigned int> SignalQueueSize;
        typedef sigc::signal<void> SignalSubmitStart;
        typedef sigc::signal<void> SignalSubmitEnd;
        typedef sigc::signal<void, Glib::ustring> SignalSubmitError;

        SignalQueueSize&
        signal_queue_size   ()  { return signal_queue_size_; }

        SignalSubmitStart&
        signal_submit_start ()  { return signal_submit_start_; }

        SignalSubmitEnd&
        signal_submit_end   ()  { return signal_submit_end_; }

        SignalSubmitError&
        signal_submit_error ()  { return signal_submit_error_; }

        void emit_queue_size ();
        void send_song_information (Glib::ustring const& uri);

      private:

        SignalQueueSize     signal_queue_size_; 
        SignalSubmitStart   signal_submit_start_; 
        SignalSubmitEnd     signal_submit_end_;
        SignalSubmitError   signal_submit_error_;

        struct HandshakeInfo
        {
            /** Whether the server signalized that we're up to date 
             */
            bool uptodate;

            /** URI for track submissions 
             */
            std::string uri;

            /** MD5 challenge string from the server 
             */
            std::string md5;

            /** Initial interval received from the server 
             */
            int interval;

            /** Used for FAILED reason and UPDATE url
             */
            std::string info;
        };

        enum HandshakeStatus
        {
          LASTFM_HANDSHAKE_UPTODATE,
          LASTFM_HANDSHAKE_UPDATE,
          LASTFM_HANDSHAKE_BADUSER,
          LASTFM_HANDSHAKE_FAILED,
          LASTFM_HANDSHAKE_STATE_UNDEFINED
        };

        enum SignalHandlers
        {
          H_LASTFM_ENABLE,
          H_LASTFM_ENABLE_GENERAL,
          N_HANDLERS
        };
           
        Glib::Mutex	          lock_m_interval_reached;
        Glib::Mutex	          lock_queue;
        Glib::Mutex	          lock_conn_items;
         
        bool		              m_interval_reached;
        sigc::connection      conn_m_interval_reached;

        HandshakeInfo	        info;
        std::string	          md5_response;
        bool		              handshaked;
        bool		              enabled;
        bool		              sent;
        
        LQMQueue              items;   
        sigc::connection      conn_items;

        gulong		            handler[N_HANDLERS];

        HandshakeStatus handshake ();
        void run_handshake ();

        void interval_reached_timeout (int seconds);
        bool interval_reached ();

        bool process_queue ();
        void process_queue_real ();
        
        void queue_enable (MCS_CB_DEFAULT_SIGNATURE);
        void submit_enable (MCS_CB_DEFAULT_SIGNATURE);
    };
  } // namespace LastFM
} // namespace Bmp


namespace Bmp
{
  namespace LastFM
  {
    class RecommendDialog : public Gtk::Dialog
    {
      public:

        RecommendDialog (BaseObjectType                       *cobject,
                         const Glib::RefPtr<Gnome::Glade::Xml> &xml);
        ~RecommendDialog ();
        static RecommendDialog* create ();

        void run (Glib::ustring const& artist, Glib::ustring const& album, Glib::ustring const& title);

      private:

        void update_view ();

        int sort_func (Gtk::TreeModel::iterator const& m_iter1,
                       Gtk::TreeModel::iterator const& m_iter2);

        class FriendsCR : public Gtk::TreeModel::ColumnRecord
        {
          public:

            Gtk::TreeModelColumn<Glib::RefPtr<Gdk::Pixbuf> > avatar;
            Gtk::TreeModelColumn<Glib::ustring>              name;
            Gtk::TreeModelColumn<Glib::ustring>              username;

            FriendsCR()
            {
              add (avatar);
              add (name);
              add (username);
            }
        };
        FriendsCR  friends_cr;

        Glib::RefPtr<Gnome::Glade::Xml>     m_ref_xml;

        Glib::RefPtr<Gtk::ListStore>        liststore;

        Gtk::TreeView                     * treeview;
        Gtk::Notebook                     * notebook;
        Gtk::Image                        * image;
        Gtk::Image                        * image_lastfm;

        Gtk::Entry                        * e_artist, *e_album, *e_title;
        Gtk::Label                        * l_artist, *l_album, *l_title;
        Gtk::ComboBox                     * cbox_recommend;
        Gtk::TextView                     * view_notes;

        void recommend_changed ();
        void recommend ();

    };
  } // ns::LastFM
} // ns::Bmp

#endif // !BMP_LAST_FM_HH

