//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2006 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non GPL-compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <iostream>
#include <fstream>

#include <libxml/tree.h>
#include <libxml/parser.h>
#include <glibmm.h>
#include "neon++/request.hh"

#include "paths.hh"
#include "util_string.hh"

#include "lyrics.hh"

using namespace std;

namespace
{
  enum ElementLyrics
  {
    E_LYRICS_NONE                = 0,
    E_LYRICS_LYRICS              = 1 << 0,
  };
  struct LyricsParserContext
  {
    Glib::ustring & m_lyrics;
    int             m_state;
    bool            m_element;

    LyricsParserContext (Glib::ustring & lyrics) : m_lyrics(lyrics), m_state(0), m_element(false) {}
  };

  struct CheckParserContext
  {
    bool            m_exists;
    int             m_state;
    bool            m_element;

    CheckParserContext () : m_exists(false), m_state(0), m_element(false) {}
  };
  enum ElementCheck
  {
    E_CHECK_NONE                        = 0,
    E_CHECK_RETURN                      = 1 << 0,
  };

  typedef std::map < Glib::ustring, Glib::ustring > ElementAttributes; 

#define SET_STATE(e)    ((context->m_state |= e))
#define CLEAR_STATE(e)  ((context->m_state &= ~e))
#define STATE(e)        ((context->m_state & e) != 0)

  //////////////////////////////////////////////////////////////////////////////

  void
  check_start_element   (void * ctxptr, const xmlChar * _name, const xmlChar ** _attributes)
  {
    CheckParserContext * context = reinterpret_cast<CheckParserContext *>(ctxptr);

    Glib::ustring name ((const char*)_name); 
    context->m_element = true;

    if (name == "return") 
    {
      SET_STATE(E_CHECK_RETURN);
      return;
    }
  }

  void
  check_end_element     (void * ctxptr, const xmlChar * _name)
  {
    CheckParserContext * context = reinterpret_cast<CheckParserContext *>(ctxptr);

    Glib::ustring name ((const char*)(_name));
    context->m_element = false;

    if (name == "return") 
    {
      CLEAR_STATE(E_CHECK_RETURN);
      return;
    }
  }

  void
  check_pcdata         (void * ctxptr, const xmlChar * _text, int length)
  {
    CheckParserContext * context = reinterpret_cast<CheckParserContext *>(ctxptr);

    if (!context->m_element)
      return;

    if (!STATE(E_CHECK_RETURN))
      return;

    Glib::ustring text (std::string((const char*)_text).substr(0, length)); 
    context->m_exists = bool(strtol(text.c_str(), NULL, 10)); 
  }

  //////////////////////////////////////////////////////////////////////////////

  void
  lyrics_start_element   (void * ctxptr, const xmlChar * _name, const xmlChar ** _attributes)
  {
    LyricsParserContext * context = reinterpret_cast<LyricsParserContext *>(ctxptr);

    Glib::ustring name ((const char*)_name); 
    context->m_element = true;

    if (name == "lyrics") 
    {
      SET_STATE(E_LYRICS_LYRICS);
      return;
    }
  }

  void
  lyrics_end_element     (void * ctxptr, const xmlChar * _name)
  {
    LyricsParserContext * context = reinterpret_cast<LyricsParserContext *>(ctxptr);

    Glib::ustring name ((const char*)(_name));
    context->m_element = false;

    if (name == "lyrics") 
    {
      CLEAR_STATE(E_LYRICS_LYRICS);
      return;
    }
  }

  void
  lyrics_pcdata         (void * ctxptr, const xmlChar * _text, int length)
  {
    LyricsParserContext * context = reinterpret_cast<LyricsParserContext *>(ctxptr);

    if (!context->m_element)
      return;

    if (!STATE(E_LYRICS_LYRICS))
        return;

    Glib::ustring text (std::string((const char*)_text).substr(0, length)); 
    context->m_lyrics.append(text);
  }

  ////// Common /////////////////////////////////////////////////////////////////

  xmlEntityPtr
  get_entity      (void * ctxptr, const xmlChar *name)
  {   
    return xmlGetPredefinedEntity (name);
  } 

  void
  whitespace      (void * ctxptr, const xmlChar * _text, int length)
  {
  }

  void
  warning         (void * ctxptr, const char * message, ...)
  {
    LyricsParserContext * context = reinterpret_cast<LyricsParserContext *>(ctxptr);
    va_list args;
    va_start(args, message);
    g_logv("LyricsWiki", G_LOG_LEVEL_WARNING, message, args);
    va_end(args);
  }

  void
  error           (void * ctxptr, const char * message, ...)  
  {
    va_list args;
    va_start(args, message);
    g_logv("LyricsWiki", G_LOG_LEVEL_CRITICAL, message, args);
    va_end(args);
  }

  void
  fatal_error     (void * ctxptr, const char * message, ...)
  {
    va_list args;
    va_start(args, message);
    g_logv("LyricsWiki", G_LOG_LEVEL_CRITICAL, message, args);
    va_end(args);
  }
    
  xmlSAXHandler LyricsParser =
  {
      (internalSubsetSAXFunc)0,                           // internalSubset 
      (isStandaloneSAXFunc)0,                             // isStandalone 
      (hasInternalSubsetSAXFunc)0,                        // hasInternalSubset 
      (hasExternalSubsetSAXFunc)0,                        // hasExternalSubset 
      (resolveEntitySAXFunc)0,                            // resolveEntity 
      (getEntitySAXFunc)get_entity,                       // getEntity 
      (entityDeclSAXFunc)0,                               // entityDecl 
      (notationDeclSAXFunc)0,                             // notationDecl 
      (attributeDeclSAXFunc)0,                            // attributeDecl 
      (elementDeclSAXFunc)0,                              // elementDecl 
      (unparsedEntityDeclSAXFunc)0,                       // unparsedEntityDecl 
      (setDocumentLocatorSAXFunc)0,                       // setDocumentLocator 
      (startDocumentSAXFunc)0,                            // startDocument 
      (endDocumentSAXFunc)0,                              // endDocument 
      (startElementSAXFunc)lyrics_start_element,          // startElement 
      (endElementSAXFunc)lyrics_end_element,              // endElement 
      (referenceSAXFunc)0,                                // reference 
      (charactersSAXFunc)lyrics_pcdata,                   // characters 
      (ignorableWhitespaceSAXFunc)whitespace,             // ignorableWhitespace 
      (processingInstructionSAXFunc)0,                    // processingInstruction 
      (commentSAXFunc)0,                                  // comment 
      (warningSAXFunc)warning,                            // warning 
      (errorSAXFunc)error,                                // error 
      (fatalErrorSAXFunc)fatal_error,                     // fatalError 
  };

  xmlSAXHandler CheckParser =
  {
      (internalSubsetSAXFunc)0,                           // internalSubset 
      (isStandaloneSAXFunc)0,                             // isStandalone 
      (hasInternalSubsetSAXFunc)0,                        // hasInternalSubset 
      (hasExternalSubsetSAXFunc)0,                        // hasExternalSubset 
      (resolveEntitySAXFunc)0,                            // resolveEntity 
      (getEntitySAXFunc)get_entity,                       // getEntity 
      (entityDeclSAXFunc)0,                               // entityDecl 
      (notationDeclSAXFunc)0,                             // notationDecl 
      (attributeDeclSAXFunc)0,                            // attributeDecl 
      (elementDeclSAXFunc)0,                              // elementDecl 
      (unparsedEntityDeclSAXFunc)0,                       // unparsedEntityDecl 
      (setDocumentLocatorSAXFunc)0,                       // setDocumentLocator 
      (startDocumentSAXFunc)0,                            // startDocument 
      (endDocumentSAXFunc)0,                              // endDocument 
      (startElementSAXFunc)check_start_element,           // startElement 
      (endElementSAXFunc)check_end_element,               // endElement 
      (referenceSAXFunc)0,                                // reference 
      (charactersSAXFunc)check_pcdata,                    // characters 
      (ignorableWhitespaceSAXFunc)0,                      // ignorableWhitespace 
      (processingInstructionSAXFunc)0,                    // processingInstruction 
      (commentSAXFunc)0,                                  // comment 
      (warningSAXFunc)warning,                            // warning 
      (errorSAXFunc)error,                                // error 
      (fatalErrorSAXFunc)fatal_error,                     // fatalError 
  };

  bool
  check_lyrics (Glib::ustring const& artist, Glib::ustring const& title)
  {
    typedef std::map  < Glib::ustring, xmlNsPtr > NamespaceMap;
    typedef std::pair < Glib::ustring, xmlNsPtr > NamespacePair;
    NamespaceMap namespaces;
  
    xmlDocPtr doc = xmlNewDoc (BAD_CAST "1.0");  
    xmlNodePtr node_envelope = xmlNewNode (NULL, BAD_CAST "Envelope");
    xmlNodePtr node_body = xmlNewChild (node_envelope, NULL, BAD_CAST "Body", NULL);  

    xmlDocSetRootElement (doc, node_envelope);

    xmlNsPtr ns_soap_env = xmlNewNs (node_envelope, BAD_CAST "http://schema.xmlsoap.org/soap/envelope/", BAD_CAST "soap-env");
    xmlNsPtr ns_soap_enc = xmlNewNs (node_envelope, BAD_CAST "http://schemas.xmlsoap.org/soap/encoding", BAD_CAST "soap-enc");
    xmlNsPtr ns_xsd = xmlNewNs (node_envelope, BAD_CAST "http://www.w3.org/2001/XMLSchema", BAD_CAST "xsd");
    xmlNsPtr ns_xsi = xmlNewNs (node_envelope, BAD_CAST "http://www.w3.org/2001/XMLSchema-instance", BAD_CAST "xsi");
    xmlNsPtr ns_si = xmlNewNs (node_envelope, BAD_CAST "http://soapinterop.org/xsd", BAD_CAST "si");
    xmlNsPtr ns_soap = xmlNewNs (node_envelope, BAD_CAST "http://schemas.xmlsoap.org/wsdl/soap", BAD_CAST "soap");
    xmlNsPtr ns_wsdl = xmlNewNs (node_envelope, BAD_CAST "http://schemas.xmlsoap.org/wsdl", BAD_CAST "wsdl");
    xmlNsPtr ns_tns = xmlNewNs (node_envelope, BAD_CAST "urn:LyricWiki", BAD_CAST "tns");
    xmlNsPtr ns_mns = xmlNewNs (node_envelope, BAD_CAST "urn:LyricWiki", BAD_CAST "mns");

    xmlSetNs (node_envelope, ns_soap_env); 
    xmlSetNs (node_body, ns_soap_env); 

    xmlNodePtr getSongResult = xmlNewChild (node_body, ns_mns, BAD_CAST "checkSongExists", NULL);
    xmlSetNsProp (getSongResult, ns_soap_env, 
                    BAD_CAST "encodingStyle",
                    BAD_CAST "http://schemas.xmlsoap.org/soap/encoding/");

    xmlNodePtr node_artist = xmlNewTextChild (getSongResult, NULL, BAD_CAST "artist", BAD_CAST artist.c_str()); 
    xmlSetNsProp (node_artist, ns_xsi, BAD_CAST "type", BAD_CAST "xsd:string"); 

    xmlNodePtr song = xmlNewTextChild (getSongResult, NULL, BAD_CAST "song", BAD_CAST title.c_str()); 
    xmlSetNsProp (song, ns_xsi, BAD_CAST "type", BAD_CAST "xsd:string"); 

    xmlChar * mem = 0;
    int size = 0;
    xmlDocDumpMemory (doc, &mem, &size);
    xmlFreeDoc (doc);
  
    Neon::Request r ("lyricwiki.org", "/server.php", 80, Neon::Request::RESPONSE_READ,
                                                         Neon::Request::METHOD_POST);
    r.add_request_header (Neon::Request::HEADER_XMLCONTENT);
    r.set_request_data ((const char*)mem, size);

    int result = r.dispatch (); 
    int status = r.get_status ()->code;

    if (status != 200)
      return false;

    std::string response;
    r >> response;
    r.clear ();

    CheckParserContext context;
    int rc = xmlSAXUserParseMemory (&CheckParser, reinterpret_cast<void *>(&context),
                                    response.c_str(), strlen(response.c_str()));

    if (rc < 0)
      return false;

    return context.m_exists;
  }
}

namespace Bmp
{
  namespace LyricWiki
  {
    bool
    get_lyrics (Glib::ustring const& artist, Glib::ustring const& title, Glib::ustring & lyrics, bool force)
    {
      std::string collation = artist.lowercase() + std::string("_") + title.lowercase();
      std::string md5hex = Util::md5_hex ((char* const)collation.c_str(), strlen(collation.c_str()));
      std::string lyrics_cache_path = Glib::build_filename (BMP_PATH_LYRIC_CACHE_DIR, md5hex + std::string (".lyrics"));

      if (!force && Glib::file_test (lyrics_cache_path, Glib::FILE_TEST_EXISTS))
      {
        try{
          lyrics = Glib::file_get_contents (lyrics_cache_path);
          return true;
          }
        catch (...) { return false; }
      }

      if (!check_lyrics (artist, title))
        return false;
 
      typedef std::map  < Glib::ustring, xmlNsPtr > NamespaceMap;
      typedef std::pair < Glib::ustring, xmlNsPtr > NamespacePair;
      NamespaceMap namespaces;
    
      xmlDocPtr doc = xmlNewDoc (BAD_CAST "1.0");  

      xmlNodePtr node_envelope = xmlNewNode (NULL, BAD_CAST "Envelope");
      xmlNodePtr node_body = xmlNewChild (node_envelope, NULL, BAD_CAST "Body", NULL);  

      xmlDocSetRootElement (doc, node_envelope);

      xmlNsPtr ns_soap_env = xmlNewNs (node_envelope, BAD_CAST "http://schema.xmlsoap.org/soap/envelope/", BAD_CAST "soap-env");
      xmlNsPtr ns_soap_enc = xmlNewNs (node_envelope, BAD_CAST "http://schemas.xmlsoap.org/soap/encoding", BAD_CAST "soap-enc");
      xmlNsPtr ns_xsd = xmlNewNs (node_envelope, BAD_CAST "http://www.w3.org/2001/XMLSchema", BAD_CAST "xsd");
      xmlNsPtr ns_xsi = xmlNewNs (node_envelope, BAD_CAST "http://www.w3.org/2001/XMLSchema-instance", BAD_CAST "xsi");
      xmlNsPtr ns_si = xmlNewNs (node_envelope, BAD_CAST "http://soapinterop.org/xsd", BAD_CAST "si");
      xmlNsPtr ns_soap = xmlNewNs (node_envelope, BAD_CAST "http://schemas.xmlsoap.org/wsdl/soap", BAD_CAST "soap");
      xmlNsPtr ns_wsdl = xmlNewNs (node_envelope, BAD_CAST "http://schemas.xmlsoap.org/wsdl", BAD_CAST "wsdl");
      xmlNsPtr ns_tns = xmlNewNs (node_envelope, BAD_CAST "urn:LyricWiki", BAD_CAST "tns");
      xmlNsPtr ns_mns = xmlNewNs (node_envelope, BAD_CAST "urn:LyricWiki", BAD_CAST "mns");

      xmlSetNs (node_envelope, ns_soap_env); 
      xmlSetNs (node_body, ns_soap_env); 

      xmlNodePtr getSongResult = xmlNewChild (node_body, ns_mns, BAD_CAST "getSongResult", NULL);
      xmlSetNsProp (getSongResult, ns_soap_env, 
                      BAD_CAST "encodingStyle",
                      BAD_CAST "http://schemas.xmlsoap.org/soap/encoding/");
  
      xmlNodePtr node_artist = xmlNewTextChild (getSongResult, NULL, BAD_CAST "artist", BAD_CAST artist.c_str()); 
      xmlSetNsProp (node_artist, ns_xsi, BAD_CAST "type", BAD_CAST "xsd:string"); 

      xmlNodePtr song = xmlNewTextChild (getSongResult, NULL, BAD_CAST "song", BAD_CAST title.c_str()); 
      xmlSetNsProp (song, ns_xsi, BAD_CAST "type", BAD_CAST "xsd:string"); 

      xmlChar * mem = 0;
      int size = 0;
      xmlDocDumpMemory (doc, &mem, &size);
      xmlFreeDoc (doc);

      Neon::Request r ("lyricwiki.org", "/server.php", 80, Neon::Request::RESPONSE_READ,
                                                           Neon::Request::METHOD_POST);
      r.add_request_header (Neon::Request::HEADER_XMLCONTENT);
      r.set_request_data ((const char*)mem, size);

      int result = r.dispatch (); 
      int status = r.get_status ()->code;

      if (status != 200)
        return false;

      std::string response;
      r >> response;
      r.clear ();

      LyricsParserContext context (lyrics); 
      int rc = xmlSAXUserParseMemory (&LyricsParser, reinterpret_cast<void *>(&context),
                                      response.c_str(), strlen(response.c_str()));

      if (rc < 0)
        return false;

      try{
        std::ofstream o;
        o.open (lyrics_cache_path.c_str());
        o << lyrics; 
        o.close ();
        }
      catch (...) {}
      return true;
    }
  }
}
