//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2006 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non GPL-compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/fcntl.h>
#include <errno.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <stdio.h>
#include <unistd.h>
#include <cstring>

#include <iostream>
#include <sstream>

#include <ne_basic.h>
#include <ne_compress.h>
#include <ne_socket.h>
#include <ne_utils.h>
#include <ne_props.h>
#include <ne_session.h>
#include <ne_request.h>
#include <ne_uri.h>

#include "util.hh"
#include "uri++.hh"

#include <build.h>
#include <revision.h>

#define SERVER_TEST_NAME  "google.com" 
#define SERVER_PORT        80

#define SERVER_NAME       "beep-media-player.org" 

#include "network.hh"

namespace
{
  bool connected = false;
  bool attempted_connect = false;
}

namespace Bmp
{
  namespace Network
  {
    void
    disable ()
    {
      attempted_connect = true; 
      connected = false;
    }

    void
    send_statistic (StatisticAction action)
    {
      std::stringstream          post;
      ne_session                *sess;
      ne_request                *reqs;

      if (!check_host (SERVER_NAME, 80)) return;
     
      const char *action_str = (action == ACTION_QUIT) ? "quit":"start"; 
      char *aux = ne_path_escape (BUILD_ARCH);

      post  << "http://" 
            << SERVER_NAME
            << "/stats.php"
            << "?version="
            << VERSION 
            << "&svnrev="
            << RV_REVISION
            << "&system="
            << aux
            << "&action="
            << action_str;

      free (aux);

      ne_uri u;
      ne_uri_parse (post.str().c_str(), &u);
  
      sess = ne_session_create (u.scheme, u.host, 80);	
			

      reqs = ne_request_create (sess, "GET", u.path); 
      ne_add_request_header (reqs, "Content-Type", "application/x-www-form-urlencoded");
      ne_begin_request (reqs);
      if (ne_discard_response (reqs))
      {
      	g_warning (G_STRLOC ": Couldn't send statistic. libneon error: %s", ne_get_error (sess));
      }
      ne_end_request (reqs);

      ne_uri_free (&u);
      ne_request_destroy (reqs);
      ne_session_destroy (sess);
    }

    bool
    check_host (std::string const& hostname, int port, bool write)
    {
      if (attempted_connect && !connected) return false;

      int    sd, rc;
      struct sockaddr_in localAddr, servAddr;
      struct hostent *h;
      fd_set rfds;
      struct timeval tv;
      int retval;

      h = gethostbyname (hostname.c_str());

      if ( h == NULL )
      {
        return false;
      }

      servAddr.sin_family = h->h_addrtype;
      std::memcpy((char *) &servAddr.sin_addr.s_addr, h->h_addr_list[0], h->h_length);
      servAddr.sin_port = htons (port);

      // Create socket
      sd = socket (AF_INET, SOCK_STREAM, 0);
      if (sd < 0)
      {
        return false; 
      }

      // Bind Any Port Number
      localAddr.sin_family = AF_INET;
      localAddr.sin_addr.s_addr = htonl (INADDR_ANY);
      localAddr.sin_port = htons(0);
      
      rc = bind (sd, (struct sockaddr *) &localAddr, sizeof(localAddr));
      if ( rc < 0 )
      {
      	return false;
      }

      int flags = fcntl (rc, F_GETFL, 0);
      fcntl (rc, F_SETFL, flags | O_NONBLOCK);

      // Connect To Server
      rc = connect (sd, (struct sockaddr *) &servAddr, sizeof(servAddr));
      int errsv = errno;
      if (rc < 0)
      {
        if (!(rc == (-1) && errsv == EINPROGRESS))
          {
      	    return false;
          }
      }

      const unsigned int tries = 10;
      double val = .5; 

      for (unsigned int n = 0; n < tries; ++n)
      {
        FD_ZERO(&rfds); 
        FD_SET(rc, &rfds);
        tv.tv_sec = int(val) / 1; 
        tv.tv_usec = (int(val)*1000000) - tv.tv_sec; 

        if (write)
          retval = select (rc+1, &rfds, NULL, NULL, &tv);
        else
          retval = select (rc+1, NULL, &rfds, NULL, &tv);
  
        if (retval >= 0)
          {
            connect (sd, (struct sockaddr *) &servAddr, sizeof(servAddr));
            shutdown (sd, SHUT_RDWR);
            return true;
          }
        val += .5;
      }

      shutdown (sd, SHUT_RDWR);
      return false; 
    }

    bool
    check_connected (bool force)
    {
      if (attempted_connect && !force)
          return connected;
      else if (force)
          connected = false;

      connected = check_host (SERVER_TEST_NAME, 80);
      attempted_connect = true;
      return connected;
    }
  }
}

