//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif // HAVE_CONFIG_H

#include <glibmm/i18n.h>
#include <glib/gstdio.h>
#include <gtkmm.h>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <sstream>
#include <fstream>

#include <boost/format.hpp>
#include <boost/optional.hpp>
#include <mcs/mcs.h>

#include "x_lastfm.hh"
#include "x_library.hh"
#include "x_play.hh"
#include "x_service_core.hh"

#include "main.hh"
#include "paths.hh"
#include "debug.hh"
#include "stock.hh"
#include "network.hh"

#include "ui_toolbox.hh"
#include "button.hh"
#include "ui-banner-image.hh"
#include "util.hh"
#include "uri++.hh"

#include "ui-part-albums.hh"
#include "ui-part-cdda.hh"
#include "ui-part-lastfm.hh"
#include "ui-part-playlist.hh"
#include "ui-part-podcasts.hh"

#include "ui-part-shoutcast-streams.hh"
#include "ui-part-icecast-streams.hh"

#include "popup.hh"

#ifdef HAVE_MOOD
#  include "ui-part-downloads.hh"
#endif // HAVE_MOOD

#ifdef HAVE_GAIM
#  include "gaim-dbus.hh"
#  include "dialog-gaim-send-file.hh"
#endif // HAVE_GAIM

#include <widgets/bmp_tray_icon.h>
#include <widgets/bmp_status_icon.h>

#include "preferences-ui.hh"

#include "dialog-about.hh"
#include "dialog-lyrics.hh"

#ifdef HAVE_VISUALIZATIONS
#  include "video-widget.hh"
#  define MAIN_TOGGLE_ACTION_VIS "main-toggle-action-vis"
#endif //HAVE_VISUALIZATIONS

#include "playershell.hh"

namespace
{
  static boost::format f_time ("<b>%02d:%02d</b>");
  static boost::format f_button_name ("tb-main-%d");
  static boost::format f_image_name  ("i-toolbar-%d");

  const char *source_icons_stock [] =
  {
    BMP_STOCK_LIBRARY,
    BMP_STOCK_SHOUTCAST,
    BMP_STOCK_XIPH,
    BMP_STOCK_CDDA,
    BMP_STOCK_LASTFM,
    BMP_STOCK_FEED,
    BMP_STOCK_PLAYLIST,
#ifdef HAVE_MOOD
    BMP_STOCK_MUSEEK,
#endif
  };

#define PLAYER_SHELL_ACTION_ABOUT     "player-shell-action-about"
#define PLAYER_SHELL_ACTION_PLAY      "player-shell-action-play"
#define PLAYER_SHELL_ACTION_PAUSE     "player-shell-action-pause"
#define PLAYER_SHELL_ACTION_PREV      "player-shell-action-prev"
#define PLAYER_SHELL_ACTION_NEXT      "player-shell-action-next"
#define PLAYER_SHELL_ACTION_STOP      "player-shell-action-stop"
#define PLAYER_SHELL_ACTION_QUIT      "player-shell-action-quit"
#define PLAYER_SHELL_ACTION_PREFS     "player-shell-action-prefs"

#define PLAYER_SHELL_ACTION_RECOMMEND_WITH_LAST_FM  "player-shell-action-recommend-with-lastfm"
#define PLAYER_SHELL_ACTION_LYRICWIKI_LYRICS        "player-shell-action-lyricwiki-lyrics"

#ifdef HAVE_GAIM
#  define PLAYER_SHELL_ACTION_GAIM_SEND             "player-shell-action-gaim-send"
#endif

  const char *ui_main_menubar =
  "<ui>"
  ""
  "<menubar name='MenuBarMain'>"
  ""
  "   <menu action='MenuBmp'>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_PREFS "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_ABOUT "'/>"
#ifdef HAVE_VISUALIZATIONS
  "     <separator/>"
  "     <menuitem action='" MAIN_TOGGLE_ACTION_VIS "'/>"
#endif //HAVE_VISUALIZATIONS
  "     <separator/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_QUIT "'/>"
  "   </menu>"
  ""
  "   <menu action='MenuTrack'>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_RECOMMEND_WITH_LAST_FM "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_LYRICWIKI_LYRICS "'/>"
#ifdef HAVE_GAIM
  "     <menuitem action='" PLAYER_SHELL_ACTION_GAIM_SEND "'/>"
#endif //HAVE_GAIM
  "   </menu>"
  ""
  "   <menu action='MenuPlayback'>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_PLAY "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_PAUSE "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_PREV "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_NEXT "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_STOP "'/>"
  "   </menu>"
  ""
  "   <placeholder name='PlaceholderSource'/>"
  ""
  "</menubar>"
  ""
  "</ui>";

  const char *ui_tray_icon = 
  "<ui>"
  " <menubar name='popup-tray'>"
  " <menu action='dummy' name='menu-tray'>"
  "   <menuitem action='" PLAYER_SHELL_ACTION_ABOUT  "'/>"
  "   <separator/>"
  "   <menuitem action='" PLAYER_SHELL_ACTION_PLAY "'/>"
  "   <menuitem action='" PLAYER_SHELL_ACTION_STOP "'/>"
  "   <menuitem action='" PLAYER_SHELL_ACTION_PAUSE "'/>"
  "   <menuitem action='" PLAYER_SHELL_ACTION_PREV "'/>"
  "   <menuitem action='" PLAYER_SHELL_ACTION_NEXT "'/>"
  "   <separator/>"
  "   <menuitem action='" PLAYER_SHELL_ACTION_QUIT "'/>"
  " </menu>"
  " </menubar>"
  "</ui>";

  void mcs_volume_changed (MCS_CB_DEFAULT_SIGNATURE, Gtk::Range *range)
  {
    int volume = boost::get<int>(value); 
    play->property_volume() = volume;
    if (int(range->get_value()) != volume)
      {
        range->set_value (volume);
      }
  }

  void show_prefs (boost::shared_ptr<Bmp::Preferences>& preferences)
  {  
    preferences->present ();
  }

  bool do_shutdown (GdkEventAny *event)
  {
    core->stop ();
    return true;
  }

  const int M_SEEK_REQUEST_NONE = -1;
}

namespace Bmp
{
  Mcs::Bind *mcs_bind = 0;
}

namespace Bmp
{
  class Scroll : public Gtk::HScale
  {
    public :

      Scroll (double min, double max, double step) : Gtk::HScale (min, max, step), pass (false) {}
      virtual ~Scroll () {}
             
    protected :

      virtual bool on_event (GdkEvent *event)
      {
        if (event->type == GDK_SCROLL)
          {
            if (pass)
              {
                pass = false;
                return false;
              }

            GdkEventScroll *scroll = (GdkEventScroll*)(event);
            GdkEvent *copy = gdk_event_copy (event);

            if (scroll->direction == GDK_SCROLL_UP)
              {
                ((GdkEventScroll*)copy)->direction = GDK_SCROLL_DOWN;
              }
            else if (scroll->direction == GDK_SCROLL_DOWN)
              {
                ((GdkEventScroll*)copy)->direction = GDK_SCROLL_UP;
              }
            
            pass = true;
            gdk_event_put (copy);
            gdk_event_free (copy);
            return true;
          }

        return false;
      }

    private :

      bool pass;
  };
}

namespace Bmp
{
      bool
      PlayerShell::display_popup_delayed ()
      {
        m_popup->tooltip_mode ();
        return false;
      }

      gboolean
      PlayerShell::status_icon_enter (BmpStatusIcon *icon, GdkEventCrossing *event, gpointer data)
      {
        Bmp::PlayerShell *main = reinterpret_cast<Bmp::PlayerShell *>(data);
        if (main->m_active_source != SOURCE_NONE)
          {
            main->delay_popup = Glib::signal_timeout().connect
              (sigc::mem_fun (main, &Bmp::PlayerShell::display_popup_delayed), 400);
          }
        return FALSE;
      }

      gboolean
      PlayerShell::status_icon_leave (BmpStatusIcon *icon, GdkEventCrossing *event, gpointer data) 
      {
        Bmp::PlayerShell *main = reinterpret_cast<Bmp::PlayerShell *>(data);
        main->delay_popup.disconnect ();
        main->m_popup->tooltip_mode (false);
        return FALSE;
      }

      void
      PlayerShell::status_icon_popup_menu (BmpStatusIcon *icon, guint arg1, guint arg2, gpointer data)
      {
        Bmp::PlayerShell *main = reinterpret_cast<Bmp::PlayerShell *>(data);

        main->delay_popup.disconnect ();
        if (main->m_popup) main->m_popup->tooltip_mode (false, true);

        GtkWidget *menu = Util::ui_manager_get_popup (main->m_ui_manager->gobj(), "/popup-tray/menu-tray");
        gtk_widget_realize (menu);
        gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL, arg1, arg2);
      }

      void
      PlayerShell::status_icon_click (BmpStatusIcon *icon, gpointer data)
      {
        Bmp::PlayerShell *main = reinterpret_cast<Bmp::PlayerShell *>(data);

        main->delay_popup.disconnect ();
        if (main->m_popup) main->m_popup->tooltip_mode (false, true);

        if (main->m_visible)
          {
            main->iconify ();
          }
        else
          {
            main->deiconify ();
          }
      }

      void
      PlayerShell::status_icon_scroll_up (BmpStatusIcon *icon, gpointer data)
      {
        int volume;
        volume = mcs->key_get<int>("bmp", "volume"); 
        volume += 4;
        volume = (volume > 100) ? 100 : volume;
        mcs->key_set<int>("bmp", "volume", volume);
      }

      void
      PlayerShell::status_icon_scroll_down (BmpStatusIcon *icon, gpointer data)
      {
        int volume;
        volume = mcs->key_get<int>("bmp", "volume"); 
        volume -= 4;
        volume = (volume < 0) ? 0 : volume;
        mcs->key_set<int>("bmp", "volume", volume);
      }

      void
      PlayerShell::create_popup (GtkPlug *plug, gpointer data)
      {
        Bmp::PlayerShell *main = reinterpret_cast<Bmp::PlayerShell *>(data);

        GtkWidget *widget = GTK_WIDGET (plug);

        Glib::ustring text ("");
        Gdk::Color outline, inlay;
        outline.set_rgb (0, 0, 0);
        inlay.set_rgb (65535, 65535, 65535);
        main->m_popup = new Bmp::Popup (widget, outline, inlay, Glib::RefPtr<Gdk::Pixbuf>(0), text);
        g_object_connect (G_OBJECT (plug),
                          "signal::enter-notify-event",
                          G_CALLBACK(PlayerShell::status_icon_enter),
                          main,
                          "signal::leave-notify-event",
                          G_CALLBACK(PlayerShell::status_icon_leave),
                          main,
                          NULL);
      }
      
      void
      PlayerShell::display_popup (SimpleTrackInfo& sti)
      {
        if (m_popup)
        {
            m_popup->set_info (sti);
            if (mcs->key_get<bool>("bmp","display-notifications"))
              {
                m_popup->show ();
              }
        }
      }

      PlayerShell::PlayerShell (BaseObjectType                       *  obj,
                                Glib::RefPtr<Gnome::Glade::Xml> const&  xml)

          : Gtk::Window       (obj),
            m_ref_xml         (xml),
            m_active_source   (SOURCE_NONE),
            m_popup           (0),
            m_visible         (true),
            m_seeking         (false),
            m_seek_request    (-1),
            m_sent            (false),
            m_error_condition (false),
            m_monitor         (false),
            m_about_dialog    (new Bmp::AboutDialog()),
            m_ui_merge_id     (0)

      {
          bool network_connected = Network::check_connected();

          mcs_bind = new Mcs::Bind(mcs);
          Bmp::register_stock_icons ();

          // Connect to Shutdown
          signal_delete_event().connect (sigc::ptr_fun (&do_shutdown));

          // Additional UIs
          m_preferences = boost::shared_ptr<Bmp::Preferences>(Bmp::Preferences::create());

#ifdef HAVE_GAIM
          gaim_dbus = boost::shared_ptr<Bmp::Gaim::DBUS>(new Bmp::Gaim::DBUS());
          gaim_dbus->signal_gaim_lost().connect(sigc::mem_fun (this, &Bmp::PlayerShell::gaim_lost));
          gaim_dbus->signal_gaim_present().connect(sigc::mem_fun (this, &Bmp::PlayerShell::gaim_present));
#endif //HAVE_GAIM

          // Header Banner Image
          {
            Glib::RefPtr<Gdk::Pixbuf> left, right, slice;
            left  = Gdk::Pixbuf::create_from_file (BMP_IMAGE_DIR_MAIN G_DIR_SEPARATOR_S "header-logo-left.png");
            right = Gdk::Pixbuf::create_from_file (BMP_IMAGE_DIR_MAIN G_DIR_SEPARATOR_S "header-logo-right.png");
            slice = Gdk::Pixbuf::create_from_file (BMP_IMAGE_DIR_MAIN G_DIR_SEPARATOR_S "header-logo-slice.png");
            Bmp::UI::BannerImage *banner = Gtk::manage (new Bmp::UI::BannerImage (left, right, slice));
            Gtk::Alignment *a;
            m_ref_xml->get_widget ("header_alignment", a);
            a->add (*banner);
            a->show_all ();
            banner->show_all ();
            banner->set_size_request (-1, slice->get_height());
          }

          Util::window_set_icon_list (GTK_WIDGET (gobj ()), "player");

          // Bottom status row
          Gdk::Color fg, bg;
          fg.set_rgb_p (1., 1., 1.);
          bg.set_rgb_p (0., 0., 0.);

          m_ref_xml->get_widget ("status_image", status_image);
          m_ref_xml->get_widget ("status_label", status_label);
          m_ref_xml->get_widget ("time_label"  , time_label);
          m_ref_xml->get_widget ("status_evbox", status_evbox);

          status_evbox->add_events (Gdk::BUTTON_PRESS_MASK);
          status_evbox->signal_button_press_event().connect
                  (sigc::mem_fun (this, &Bmp::PlayerShell::error_clear));
          status_evbox->modify_bg (Gtk::STATE_NORMAL, bg);

          m_warning = Gdk::Pixbuf::create_from_file (Glib::build_filename (BMP_IMAGE_DIR_MAIN, "warning.png"));

          m_ref_xml->get_widget ("table_metadata")->modify_bg (Gtk::STATE_NORMAL, bg);
          m_ref_xml->get_widget ("metadata_evbox")->modify_bg (Gtk::STATE_NORMAL, bg);
          m_ref_xml->get_widget ("time_evbox")->modify_bg (Gtk::STATE_NORMAL, bg);
          status_label->modify_fg (Gtk::STATE_NORMAL, fg);
          time_label->modify_fg (Gtk::STATE_NORMAL, fg);

          yy_playing  = Gdk::PixbufAnimation::create_from_file
                (Glib::build_filename (BMP_IMAGE_DIR_MAIN, "yingyang.gif"));
          yy_paused   = Gdk::PixbufAnimation::create_from_file
                (Glib::build_filename (BMP_IMAGE_DIR_MAIN, "yingyang_paused.gif"));
          yy_seeking  = Gdk::Pixbuf::create_from_file
                (Glib::build_filename (BMP_IMAGE_DIR_MAIN, "yingyang_seeking.png"));
          yy_stopped  = yy_playing->get_static_image (); 

          status_image->set (yy_stopped);

          // Volume
          volume = new Bmp::Scroll (0., 110., 5);
          volume->set_draw_value (0);
          volume->set_size_request (100, -1);
          volume->show_all ();
          dynamic_cast<Gtk::HBox *>(m_ref_xml->get_widget ("hbox-volume"))->pack_end (*volume, true, true);

          mcs_bind->bind_range (reinterpret_cast<Gtk::Range *>(volume), "bmp", "volume");
          mcs->subscribe ("PlayerShell", "bmp", "volume", sigc::bind (sigc::ptr_fun (&mcs_volume_changed), volume)); 

          // Tray icon
          m_status_icon_image_default =
            Gdk::Pixbuf::create_from_file (Glib::build_filename (BMP_TRAY_ICON_DIR, "tray-icon-default.png"));
          m_status_icon_image_paused =
            Gdk::Pixbuf::create_from_file (Glib::build_filename (BMP_TRAY_ICON_DIR, "tray-icon-paused.png"));
          m_status_icon_image_playing =
            Gdk::Pixbuf::create_from_file (Glib::build_filename (BMP_TRAY_ICON_DIR, "tray-icon-playing.png"));

          m_status_icon = bmp_status_icon_new_from_pixbuf (m_status_icon_image_default->gobj());
          bmp_status_icon_set_visible (m_status_icon, TRUE);

          g_object_connect (G_OBJECT (m_status_icon),
                            "signal::click",
                            G_CALLBACK(PlayerShell::status_icon_click),
                            this,
                            "signal::popup-menu",
                            G_CALLBACK(PlayerShell::status_icon_popup_menu),
                            this,
                            "signal::scroll-up",
                            G_CALLBACK(PlayerShell::status_icon_scroll_up),
                            this,
                            "signal::scroll-down",
                            G_CALLBACK(PlayerShell::status_icon_scroll_down),
                            this,
                            NULL);

          GtkWidget *tray_icon = bmp_status_icon_get_tray_icon (m_status_icon);
          gtk_widget_realize (GTK_WIDGET (tray_icon));
          g_object_connect (G_OBJECT (tray_icon),
                            "signal::embedded",
                            G_CALLBACK (Bmp::PlayerShell::create_popup),
                            this,
                            NULL);

          // Navigation
          m_ref_xml->get_widget ("notebook-main", notebook_main);

#ifdef HAVE_VISUALIZATIONS
          m_ref_xml->get_widget ("notebook-over", notebook_over);
#endif //HAVE_VISUALIZATIONS

          m_ref_xml->get_widget_derived ("vbox-sources", m_source_selection);

          ::play->property_status().signal_changed().connect
                (sigc::mem_fun (this, &Bmp::PlayerShell::playstatus_changed));
          ::play->signal_eos().connect
                (sigc::mem_fun (this, &Bmp::PlayerShell::next));
          ::play->signal_position().connect
                (sigc::mem_fun (this, &Bmp::PlayerShell::play_position));
          ::play->signal_seek().connect
                (sigc::mem_fun (this, &Bmp::PlayerShell::play_seek));
          ::play->signal_buffering().connect
                (sigc::mem_fun (this, &Bmp::PlayerShell::buffering));
          ::play->signal_title().connect
                (sigc::mem_fun (this, &Bmp::PlayerShell::play_set_title));

          // Seek Range
          seekrange = dynamic_cast<Gtk::HScale *>(m_ref_xml->get_widget ("scale-seek"));
          seekrange->set_events (Gdk::BUTTON_PRESS_MASK | Gdk::BUTTON_RELEASE_MASK);
          seekrange->signal_value_changed().connect
                (sigc::mem_fun (this, &Bmp::PlayerShell::seek_display));
          seekrange->signal_event().connect
                (sigc::mem_fun (this, &Bmp::PlayerShell::seek_event));
          seekrange->set_sensitive (false);

          m_ref_xml->get_widget ("eventbox-bottom", evb);
          evb->add_events (Gdk::BUTTON_PRESS_MASK);
          evb->signal_button_press_event ().connect (sigc::mem_fun (this, &Bmp::PlayerShell::source_restore_context));

          // UIManager + Menus + Proxy Widgets 
          m_ui_manager = Gtk::UIManager::create ();

          dynamic_cast<Gtk::Window *>(m_ref_xml->get_widget("main-ui"))->add_accel_group(m_ui_manager->get_accel_group());
                
          m_actions_main = Gtk::ActionGroup::create ("ActionsMain"); 

          // Menus
          m_actions_main->add (Gtk::Action::create ("dummy", "dummy")); //dummy action for popups

          m_actions_main->add (Gtk::Action::create ("MenuBmp", _("_BMP")));
          m_actions_main->add (Gtk::Action::create ("MenuTrack", _("_Track")));
          m_actions_main->add (Gtk::Action::create ("MenuPlayback", _("_Playback")));
          m_actions_main->add (Gtk::Action::create ("MenuHelp", _("_Help")));

          // Playback Actions
          m_actions_main->add ( Gtk::Action::create (PLAYER_SHELL_ACTION_PLAY,
                                    Gtk::StockID (GTK_STOCK_MEDIA_PLAY),
                                    _("_Play")),
                                    Gtk::AccelKey ("<alt>z"),
                                    sigc::mem_fun (this, &Bmp::PlayerShell::play));

          m_actions_main->add ( Gtk::Action::create (PLAYER_SHELL_ACTION_PAUSE,
                                    Gtk::StockID (GTK_STOCK_MEDIA_PAUSE),
                                    _("P_ause")),
                                    Gtk::AccelKey ("<alt>x"),
                                    sigc::mem_fun (this, &Bmp::PlayerShell::pause));

          m_actions_main->add ( Gtk::Action::create (PLAYER_SHELL_ACTION_PREV,
                                    Gtk::StockID (GTK_STOCK_MEDIA_PREVIOUS),
                                    _("P_rev")),
                                    Gtk::AccelKey ("<alt>c"),
                                    sigc::mem_fun (this, &Bmp::PlayerShell::prev));

          m_actions_main->add ( Gtk::Action::create (PLAYER_SHELL_ACTION_NEXT,
                                    Gtk::StockID (GTK_STOCK_MEDIA_NEXT),
                                    _("_Next")),
                                    Gtk::AccelKey ("<alt>v"),
                                    sigc::mem_fun (this, &Bmp::PlayerShell::next));

          m_actions_main->add ( Gtk::Action::create (PLAYER_SHELL_ACTION_STOP,
                                    Gtk::StockID (GTK_STOCK_MEDIA_STOP),
                                    _("_Stop")),
                                    Gtk::AccelKey ("<alt>b"),
                                    sigc::mem_fun (this, &Bmp::PlayerShell::stop));


          // Menu Actions
          m_actions_main->add ( Gtk::Action::create (PLAYER_SHELL_ACTION_ABOUT,
                                    Gtk::StockID (BMP_STOCK_ABOUT),
                                    _("_About BMP")),
                                    sigc::mem_fun (this, &Bmp::PlayerShell::display_about_dialog));

          m_actions_main->add ( Gtk::Action::create (PLAYER_SHELL_ACTION_PREFS,
                                    Gtk::StockID (GTK_STOCK_PREFERENCES),
                                    _("_Preferences")),
                                    (sigc::bind (sigc::ptr_fun (&show_prefs), m_preferences)));

          m_actions_main->add ( Gtk::Action::create (PLAYER_SHELL_ACTION_QUIT,
                                    Gtk::StockID (GTK_STOCK_QUIT),
                                    _("_Quit")),
                                    Gtk::AccelKey ("<control>q"),
                                    sigc::mem_fun (core, &Bmp::ServiceCore::stop));

          m_actions_main->add ( Gtk::Action::create (PLAYER_SHELL_ACTION_RECOMMEND_WITH_LAST_FM,
                                    Gtk::StockID (BMP_STOCK_LASTFM),
                                    _("Last.FM _Recommend")),
                                    Gtk::AccelKey ("<alt>r"),
                                    sigc::mem_fun (this, &Bmp::PlayerShell::recommend));

          m_actions_main->add ( Gtk::Action::create (PLAYER_SHELL_ACTION_LYRICWIKI_LYRICS,
                                    Gtk::StockID (GTK_STOCK_EDIT),
                                    _("Show Track Lyrics")),
                                    Gtk::AccelKey ("<alt>l"),
                                    sigc::mem_fun (this, &Bmp::PlayerShell::lyrics));

#ifdef HAVE_GAIM
          m_actions_main->add ( Gtk::Action::create (PLAYER_SHELL_ACTION_GAIM_SEND,
                                    Gtk::StockID (BMP_STOCK_GAIM),
                                    _("Send to _Gaim Buddy")),
                                    Gtk::AccelKey ("<alt>g"),
                                    sigc::mem_fun (this, &Bmp::PlayerShell::gaim_send));
#endif //HAVE_GAIM

#ifdef HAVE_VISUALIZATIONS

          m_actions_main->add ( Gtk::ToggleAction::create (MAIN_TOGGLE_ACTION_VIS,
                                    Gtk::StockID (GTK_STOCK_FULLSCREEN),
                                    _("Visualization")),
                                    sigc::mem_fun (this, &Bmp::PlayerShell::toggle_vis));
#endif //HAVE_VISUALIZATIONS

          m_ui_manager->insert_action_group (m_actions_main);

          m_ui_manager->add_ui_from_string (ui_tray_icon);
          m_ui_manager->add_ui_from_string (ui_main_menubar);

          dynamic_cast<Gtk::Alignment *>(m_ref_xml->get_widget("alignment-menu"))->add (*(m_ui_manager->get_widget ("/MenuBarMain")));

          // Playback Proxies 
          m_actions_main->get_action (PLAYER_SHELL_ACTION_PLAY)->connect_proxy
                (*(dynamic_cast<Gtk::Button *>(m_ref_xml->get_widget ("b_play"))));
          m_actions_main->get_action (PLAYER_SHELL_ACTION_PAUSE)->connect_proxy
                (*(dynamic_cast<Gtk::Button *>(m_ref_xml->get_widget ("b_pause"))));
          m_actions_main->get_action (PLAYER_SHELL_ACTION_PREV)->connect_proxy
                (*(dynamic_cast<Gtk::Button *>(m_ref_xml->get_widget ("b_prev"))));
          m_actions_main->get_action (PLAYER_SHELL_ACTION_NEXT)->connect_proxy
                (*(dynamic_cast<Gtk::Button *>(m_ref_xml->get_widget ("b_next"))));
          m_actions_main->get_action (PLAYER_SHELL_ACTION_STOP)->connect_proxy
                (*(dynamic_cast<Gtk::Button *>(m_ref_xml->get_widget ("b_stop"))));

          m_actions_main->get_action (PLAYER_SHELL_ACTION_PLAY)->set_sensitive (false);
          m_actions_main->get_action (PLAYER_SHELL_ACTION_PAUSE)->set_sensitive (false);
          m_actions_main->get_action (PLAYER_SHELL_ACTION_PREV)->set_sensitive (false);
          m_actions_main->get_action (PLAYER_SHELL_ACTION_NEXT)->set_sensitive (false);
          m_actions_main->get_action (PLAYER_SHELL_ACTION_STOP)->set_sensitive (false);
          m_actions_main->get_action (PLAYER_SHELL_ACTION_RECOMMEND_WITH_LAST_FM)->set_sensitive (false);

#ifdef HAVE_GAIM
          m_actions_main->get_action (PLAYER_SHELL_ACTION_GAIM_SEND)->set_sensitive (false);
#endif //HAVE_GAIM

          // Playlist
          m_sources[SOURCE_PLAYLIST] = boost::shared_ptr<UiPart::Playlist>
              (new UiPart::Playlist (m_ref_xml, m_ui_manager));
          connect_source_signals (SOURCE_PLAYLIST);

          // Shoutcast
          if (network_connected)
          {
            m_sources[SOURCE_SHOUTCAST] = boost::shared_ptr<UiPart::ShoutcastStreams>
                (new UiPart::ShoutcastStreams (m_ref_xml, m_ui_manager));
            connect_source_signals (SOURCE_SHOUTCAST);
          }


          // Icecast
          if (network_connected)
          {
            m_sources[SOURCE_ICECAST] = boost::shared_ptr<UiPart::IcecastStreams>
              (new UiPart::IcecastStreams (m_ref_xml, m_ui_manager));
            connect_source_signals (SOURCE_ICECAST);
          }


          // LastFM
          if (network_connected)
          {
            m_sources[SOURCE_LASTFM] = boost::shared_ptr<UiPart::LASTFM>
              (new UiPart::LASTFM (m_ref_xml, m_ui_manager));
            connect_source_signals (SOURCE_LASTFM);
          }

          // CDDA
          m_sources[SOURCE_CDDA] = boost::shared_ptr<UiPart::CDDA>
            (new UiPart::CDDA (m_ref_xml, m_ui_manager));
          connect_source_signals (SOURCE_CDDA);

          boost::dynamic_pointer_cast<Bmp::UiPart::CDDA>(m_sources[SOURCE_CDDA])
                ->signal_request_import().connect
                (sigc::mem_fun (this, &Bmp::PlayerShell::import_request));

          // PodCasts
          if (network_connected)
          {
            m_sources[SOURCE_PODCASTS] = boost::shared_ptr<UiPart::Podcasts>
              (new UiPart::Podcasts (m_ref_xml, m_ui_manager));
            connect_source_signals (SOURCE_PODCASTS);
          }

#ifdef HAVE_MOOD

          // Downloads
          if (network_connected)
          {
            m_sources[SOURCE_DOWNLOADS] = boost::shared_ptr<UiPart::Downloads>
              (new UiPart::Downloads (m_ref_xml, m_ui_manager));
            connect_source_signals (SOURCE_DOWNLOADS);

            boost::dynamic_pointer_cast<Bmp::UiPart::Downloads>(m_sources[SOURCE_DOWNLOADS])
                  ->signal_request_import().connect
                  (sigc::mem_fun (this, &Bmp::PlayerShell::import_request));
          }

#endif //HAVE_MOOD
 
          show (); 
          set_sensitive (false);
          while (gtk_events_pending()) gtk_main_iteration ();  

          // Albums
          m_sources[SOURCE_ALBUMS] = boost::shared_ptr<UiPart::Albums>
            (new UiPart::Albums (m_ref_xml, m_ui_manager));
          connect_source_signals (SOURCE_ALBUMS);

          boost::dynamic_pointer_cast<Bmp::UiPart::Albums>(m_sources[SOURCE_ALBUMS])
                ->signal_uri_enqueue_request().connect
                (sigc::bind (sigc::mem_fun (this, &Bmp::PlayerShell::play_uris), false));

#ifdef HAVE_HAL

          boost::dynamic_pointer_cast<Bmp::UiPart::Albums>(m_sources[SOURCE_ALBUMS])
                ->signal_rescan_devices().connect
                (sigc::mem_fun (m_preferences.get(), &Bmp::Preferences::rescan_devices));

#endif //HAVE_HAL

          // Connect Selection
          m_source_selection->source_selected().connect
                (sigc::mem_fun (this, &Bmp::PlayerShell::source_changed));

          for (int n = 0; n < N_SOURCES; ++n)
          {
            if (m_sources.find (PlaybackSourceID (n)) == m_sources.end())
              continue;

            boost::shared_ptr<Bmp::PlaybackSource> const& source (m_sources.find (PlaybackSourceID (n))->second);

            if (!source.get())
              continue;

            source->send_caps ();
            source->send_flags ();
          }

          // Start Up with Albums
          source_changed (SOURCE_ALBUMS);

#ifdef HAVE_VISUALIZATIONS

          vis = new Bmp::VideoWidget();

          dynamic_cast<Gtk::EventBox *>(m_ref_xml->get_widget ("evbox-vis"))->add (*vis);
          dynamic_cast<Gtk::EventBox *>(m_ref_xml->get_widget ("evbox-vis"))->show_all ();

          ::play->signal_xoverlay_setup().connect
                (sigc::mem_fun (this, &Bmp::PlayerShell::xoverlay_setup));

          vis->show_all ();

#endif //HAVE_VISUALIZATIONS

          set_sensitive (true);
          while (gtk_events_pending()) gtk_main_iteration ();  
      }

#ifdef HAVE_VISUALIZATIONS

      void
      PlayerShell::xoverlay_setup (GstElement *element)
      {
        vis->set_xoverlay (element);
      }

      void
      PlayerShell::toggle_vis ()
      {
        bool active = Glib::RefPtr<Gtk::ToggleAction>::cast_static(m_actions_main->get_action (MAIN_TOGGLE_ACTION_VIS))->get_active();
        if (active)
          {
            ::play->replug_vis ();
            notebook_over->set_current_page (1);
          }
        else
          {
            notebook_over->set_current_page (0);
            ::play->unplug_vis ();
          }
      }

#endif //HAVE_VISUALIZATIONS

      void
      PlayerShell::connect_source_signals (PlaybackSourceID source)
      {
        m_sources[source]->signal_caps().connect
              (sigc::bind (sigc::mem_fun (this, &Bmp::PlayerShell::source_caps),
                source));

        m_sources[source]->signal_flags().connect
              (sigc::bind (sigc::mem_fun (this, &Bmp::PlayerShell::source_flags),
                source));

        m_sources[source]->signal_track_info().connect
              (sigc::bind (sigc::mem_fun (this, &Bmp::PlayerShell::source_track_info),
                source));

        m_sources[source]->signal_error_message().connect
              (sigc::bind (sigc::mem_fun (this, &Bmp::PlayerShell::error_set),
                source));

        m_sources[source]->signal_stop_request().connect
              (sigc::mem_fun (this, &Bmp::PlayerShell::stop));

        m_sources[source]->signal_playback_request().connect
              (sigc::bind (sigc::mem_fun (this, &Bmp::PlayerShell::source_playback_request),
                source));
      }

      bool
      PlayerShell::error_clear (GdkEventButton*)
      {
        if (!m_error_condition) return false;

        status_label->set_text("");
        status_image->set (yy_stopped);
        m_error_condition = false;

        return false;
      }

      void
      PlayerShell::error_set (Glib::ustring message, PlaybackSourceID source)
      //FIXME: We should keep a queue and clicking the icon should skip to the next message
      {
        m_error_condition = true;
        status_label->set_markup (message);
        status_image->set (m_warning);
      }

#ifdef HAVE_GAIM
      void
      PlayerShell::gaim_present ()
      {
        Playstatus status = Playstatus (::play->property_status().get_value());
        try {
            Bmp::URI u (::play->property_stream().get_value());
            if (status != PLAYSTATUS_STOPPED)
            {
              m_actions_main->get_action (PLAYER_SHELL_ACTION_GAIM_SEND)->set_sensitive (u.get_protocol() == Bmp::URI::PROTOCOL_FILE);
              return;
            }
          }
        catch (...)
          {
              m_actions_main->get_action (PLAYER_SHELL_ACTION_GAIM_SEND)->set_sensitive (false);
          }
      }

      void
      PlayerShell::gaim_lost ()
      {
        m_actions_main->get_action (PLAYER_SHELL_ACTION_GAIM_SEND)->set_sensitive (false);
      }

      void
      PlayerShell::gaim_send ()
      {
        Glib::ustring uri (::play->property_stream().get_value());

        // Make sure
        try {
          Bmp::URI u (uri);
          if (u.get_protocol () != Bmp::URI::PROTOCOL_FILE)
            {
              Gtk::MessageDialog dialog (_("You can only send locally accessible files trough Gaim!"), false,
                                          Gtk::MESSAGE_ERROR,
                                          Gtk::BUTTONS_OK,
                                          true);

              dialog.set_title (_("Send File: Error - BMP"));
              dialog.run ();
              return;
            }
          }
        catch (...) {} // FIXME: 

        std::string filename = Glib::filename_from_uri (uri);

        try {
          boost::shared_ptr<Bmp::Gaim::SendFile> gaim_dialog (Bmp::Gaim::SendFile::create());
          Bmp::Gaim::DBUS::OnlineBuddies buddies;
          gaim_dbus->get_online_buddies (buddies);
          if ( gaim_dialog->run( buddies , uri ) == GTK_RESPONSE_OK )
          {
            gaim_dbus->send_file( gaim_dialog->get_current_buddy () , filename );
          }
          gaim_dialog->hide ();
        }
        catch (Bmp::Gaim::DBUS::Exception& cxe)
          {
              Glib::ustring message (_("An Error occured while communicating with Gaim: "));
              message.append (cxe.what());
              Gtk::MessageDialog dialog (message, false, Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK, true);
              dialog.set_title (_("Send File: Error - BMP"));
              dialog.run ();
          }
        catch (...) {} //FIXME: What else could happen from our side..?!
      }
#endif

      void
      PlayerShell::recommend ()
      {
        boost::shared_ptr<Bmp::LastFM::RecommendDialog> dialog (Bmp::LastFM::RecommendDialog::create());
        dialog->run (m_sti.get().artist.get(), m_sti.get().album.get(), m_sti.get().title.get());
      }

      void
      PlayerShell::lyrics ()
      {
        if (!Network::check_connected())
          return;
  
        if (!m_sti)
          return;

        boost::shared_ptr<Bmp::Lyrics> dialog (Bmp::Lyrics::create());
        dialog->run (m_sti.get()); 
      }

      void
      PlayerShell::display_about_dialog ()
      {
        m_about_dialog->present ();
      }

      void
      PlayerShell::seek_display ()
      {
        double value = seekrange->get_value();
        int position = int (value);
        int minutes = position / 60;
        int seconds = position % 60;
        time_label->set_markup ((f_time % minutes % seconds).str());
      }

      bool
      PlayerShell::seek_event (GdkEvent *event)
      {
        if (event->type == GDK_BUTTON_PRESS)
          {
            m_seeking = true;
            status_image->set (yy_seeking);
          }
        else
        if (event->type == GDK_BUTTON_RELEASE)
          {
            int pos_seek = int(seekrange->get_value());
            int pos_cur = int(::play->property_position().get_value());

            m_seek_request = pos_seek; 

            if (pos_cur > pos_seek)
              m_seek_direction = M_SEEK_BACKWARD; 
            else
              m_seek_direction = M_SEEK_FORWARD; 

            ::play->seek (pos_seek);
            status_image->set (yy_playing);
            m_seeking = false;
          }
        return false;
      }

      void
      PlayerShell::buffering (double buffered)
      {
        if (buffered == 0.)
          {
            status_image->set (yy_seeking);
          }
        else
        if (buffered == 1.)
          {
            status_image->set (yy_playing);
          }

        int buffered_i (int(buffered * 100.));
        
        if (buffered_i < 100)
          status_label->set_markup ((boost::format (_("Prebuffering: %d%%")) % buffered_i).str());
        else
          status_label->set_markup (_("Buffering Complete."));
      }

      void
      PlayerShell::play_set_title (const char *title)
      {
        if (m_active_source == SOURCE_NONE)
          return; //XXX: Should not happen really, as what would be sending the title then?

        PlaybackSource::Flags flags = m_source_flags.find (m_active_source)->second;

        if (flags & PlaybackSource::F_HANDLE_STREAMINFO)
        {
          SimpleTrackInfo sti;
          sti.title = title;
          source_track_info (title, sti, m_active_source); 
        }
      }

      void
      PlayerShell::play_seek      (int position)
      {
        int length  = ::play->property_length().get_value();
        if (m_monitor && ((position > 240) || (position >= (length/2)))) 
          {
            m_sent = true;
          }
      }

      void
      PlayerShell::play_position  (int position)
      {
        using namespace Glib;

        if (m_seeking)
          return;

        int minutes = position / 60;
        int seconds = position % 60;
        int length  = ::play->property_length().get_value();

        if (G_LIKELY(m_seek_request == M_SEEK_REQUEST_NONE))
          { 
            if (m_popup && m_popup->is_visible()) m_popup->queue_draw ();
            time_label->set_markup ((f_time % minutes % seconds).str());
          }
  
        if (seekrange->is_sensitive()) 
          {
            if (G_UNLIKELY (m_seek_request != M_SEEK_REQUEST_NONE))
              {
                switch (m_seek_direction)
                  {
                    case M_SEEK_FORWARD:
                      if (position < m_seek_request) return;

                    case M_SEEK_BACKWARD:
                      if (position > m_seek_request) return;
                  }
                m_seek_request = M_SEEK_REQUEST_NONE;
              }
  
            seekrange->set_range (0., (length > 0) ? double(length) : 1.); 
            seekrange->set_value ((position > 0) ? double(position): 0.);
          }
        else return;

        if (m_monitor && ((position > 240) || (position >= (length/2))) && (!m_sent) && (length > 30))
          {
            if (mcs->key_get<bool>("lastfm", "queue-enable") &&
                  ((m_active_source == SOURCE_ALBUMS) || (m_active_source == SOURCE_PLAYLIST)))
            {
              Glib::ustring uri (::play->property_stream().get_value());
              try {
                ::lastfm_scrobbler->send_song_information (uri);
                }
              catch (...) {}
              m_sent = true;
            }
          }
      }

      bool
      PlayerShell::source_restore_context (GdkEventButton *event)
      {
        if (m_active_source == SOURCE_NONE)
          return false;

        m_source_selection->select_source (m_active_source);
        source_changed (m_active_source);
        boost::shared_ptr<Bmp::PlaybackSource> & source = m_sources.find (m_active_source)->second;
        source->restore_context ();

        return false;
      }

      void
      PlayerShell::source_track_info (Glib::ustring title, SimpleTrackInfo sti, PlaybackSourceID source)
      {
        status_label->set_markup (title);
        display_popup (sti);
        sti.image.clear ();
        m_sti = sti;
        m_actions_main->get_action (PLAYER_SHELL_ACTION_RECOMMEND_WITH_LAST_FM)->set_sensitive
          ((m_sti->artist && m_sti->album && m_sti->title));
      }

      void
      PlayerShell::source_caps (Bmp::PlaybackSource::Caps caps, PlaybackSourceID source)
      {
        m_source_caps[source] = caps;

        if ((m_active_source == SOURCE_NONE) || (int(m_active_source) == int(notebook_main->get_current_page())))
          {
            m_actions_main->get_action (PLAYER_SHELL_ACTION_PAUSE)->set_sensitive (caps & PlaybackSource::CAN_PAUSE);
          }

        m_actions_main->get_action (PLAYER_SHELL_ACTION_PREV)->set_sensitive (caps & PlaybackSource::CAN_GO_PREV);
        m_actions_main->get_action (PLAYER_SHELL_ACTION_NEXT)->set_sensitive (caps & PlaybackSource::CAN_GO_NEXT);
        m_actions_main->get_action (PLAYER_SHELL_ACTION_PLAY)->set_sensitive (caps & PlaybackSource::CAN_PLAY);
      }

      void
      PlayerShell::source_flags (Bmp::PlaybackSource::Flags flags, PlaybackSourceID source)
      {
        m_source_flags[source] = flags;
      }

      void
      PlayerShell::source_playback_request (PlaybackSourceID source)
      {
        if (m_active_source != SOURCE_NONE)
          {
            m_sources.find (m_active_source)->second->stop();
          }

        m_active_source = source; 
        m_source_selection->set_active_source (source);
        ::play->switch_stream (m_sources.find (source)->second->get_uri(), m_sources.find (source)->second->get_type());
        m_actions_main->get_action (PLAYER_SHELL_ACTION_STOP)->set_sensitive (true);
      }

      void
      PlayerShell::source_changed (int n)
      {
        notebook_main->set_current_page (n);
        Sources::iterator i = m_sources.find (PlaybackSourceID (n));

        if (i != m_sources.end())
          {
            i->second->send_caps ();
            i->second->send_flags ();

            if (m_ui_merge_id)
              m_ui_manager->remove_ui (m_ui_merge_id);

            m_ui_merge_id = boost::dynamic_pointer_cast <Bmp::UiPart::Base> (i->second)->add_ui();

#ifdef HAVE_VISUALIZATIONS

            Glib::RefPtr<Gtk::ToggleAction>::cast_static 
                  (m_actions_main->get_action (MAIN_TOGGLE_ACTION_VIS))->set_active(false);

#endif //HAVE_VISUALIZATIONS

          }
        else
          {
            m_actions_main->get_action (PLAYER_SHELL_ACTION_PLAY)->set_sensitive (false);
            m_actions_main->get_action (PLAYER_SHELL_ACTION_PREV)->set_sensitive (false);
            m_actions_main->get_action (PLAYER_SHELL_ACTION_NEXT)->set_sensitive (false);
          }
      }

      //////////////////////////////// External Methods for Playback Control

      void
      PlayerShell::play_ext ()
      {
        play ();
      }

      void
      PlayerShell::pause_ext ()
      {
        pause ();
      }

      void
      PlayerShell::prev_ext ()
      {
        PlaybackSource::Caps caps = m_source_caps.find (m_active_source)->second;
        if (caps & PlaybackSource::CAN_GO_PREV) prev();
      }
  
      void
      PlayerShell::next_ext ()
      {
        PlaybackSource::Caps caps = m_source_caps.find (m_active_source)->second;
        if (caps & PlaybackSource::CAN_GO_NEXT) next ();
      }
  
      void
      PlayerShell::stop_ext ()
      {
        stop ();
      }
  
      //////////////////////////////// Internal Playback Control
      void
      PlayerShell::play ()
      {
        PlaybackSourceID new_source = PlaybackSourceID (notebook_main->get_current_page());

        if (m_active_source != SOURCE_NONE)
          {
            m_sources.find (m_active_source)->second->stop();
            if (new_source != m_active_source)
              {
                status_image->set (yy_stopped);
                status_label->set_text (_("Changing Source.."));
                ::play->request_status (PLAYSTATUS_STOPPED);
              }
          }
        else
          {
            status_image->set (yy_stopped);
            status_label->set_text (_("Changing Source.."));
          }

        m_active_source = new_source;
        m_source_selection->set_active_source (m_active_source);
        m_source_selection->select_source (m_active_source);

        try {
            m_sources.find (m_active_source)->second->play();

            ::play->switch_stream ( m_sources.find (m_active_source)->second->get_uri(),
                                    m_sources.find (m_active_source)->second->get_type());

            m_actions_main->get_action (PLAYER_SHELL_ACTION_STOP)->set_sensitive (true);
          }
        catch (Bmp::PlaybackSource::UnableToInitiatePlaybackError& cxe)
          {
            error_set (cxe.what(), m_active_source);
            m_active_source = SOURCE_NONE;
            m_source_selection->set_active_source (m_active_source);
            ::play->request_status (PLAYSTATUS_STOPPED);
          }
      }

      void
      PlayerShell::pause ()
      {
        ::play->request_status (PLAYSTATUS_PAUSED);
      }

      void
      PlayerShell::prev ()
      {
        m_actions_main->get_action (PLAYER_SHELL_ACTION_RECOMMEND_WITH_LAST_FM)->set_sensitive (false);

        boost::shared_ptr<Bmp::PlaybackSource> &source = m_sources.find (m_active_source)->second;
        PlaybackSource::Caps caps = m_source_caps.find (m_active_source)->second;

        if (caps & PlaybackSource::CAN_GO_PREV)
        {
            if (source->go_prev())
            {
              ::play->switch_stream (source->get_uri(), source->get_type()); 
              m_sent = false;
              return;
            }
        }
        stop ();
      }
  
      void
      PlayerShell::next ()
      {
        m_seeking = false;

        m_actions_main->get_action (PLAYER_SHELL_ACTION_RECOMMEND_WITH_LAST_FM)->set_sensitive (false);

        boost::shared_ptr<Bmp::PlaybackSource> &source = m_sources.find (m_active_source)->second;
        PlaybackSource::Caps caps = m_source_caps.find (m_active_source)->second;
        PlaybackSource::Flags flags = m_source_flags.find (m_active_source)->second;

        if (caps & PlaybackSource::CAN_GO_NEXT)
        {
            if (source->go_next())
            {
              if (!(flags & PlaybackSource::F_PHONY_NEXT))
                {
                  ::play->switch_stream (source->get_uri(), source->get_type()); 
                }
              m_sent = false;
              return;
            }
        }
        stop ();
      }

      void
      PlayerShell::stop ()
      {
        using namespace Glib;
        if (m_active_source != SOURCE_NONE)
          {
            m_sources.find (m_active_source)->second->stop();
            ::play->request_status (PLAYSTATUS_STOPPED);

            Sources::iterator s_iter = m_sources.find (PlaybackSourceID (notebook_main->get_current_page()));
            if (s_iter != m_sources.end())
              {
                s_iter->second->send_caps ();
              }
            else
              {
                m_actions_main->get_action (PLAYER_SHELL_ACTION_PLAY)->set_sensitive (false);
                m_actions_main->get_action (PLAYER_SHELL_ACTION_PAUSE)->set_sensitive (false);
                m_actions_main->get_action (PLAYER_SHELL_ACTION_PREV)->set_sensitive (false);
                m_actions_main->get_action (PLAYER_SHELL_ACTION_NEXT)->set_sensitive (false);
              }

            m_actions_main->get_action (PLAYER_SHELL_ACTION_RECOMMEND_WITH_LAST_FM)->set_sensitive (false);
            m_sti.reset ();

#ifdef HAVE_VISUALIZATIONS
            RefPtr<Gtk::ToggleAction>::cast_static(m_actions_main->get_action
              (MAIN_TOGGLE_ACTION_VIS))->set_active(false);
#endif //HAVE_VISUALIZATIONS
          }
      }

      void
      PlayerShell::playstatus_changed ()
      {
        Playstatus status = Playstatus (::play->property_status().get_value());
        PlaybackSource::Caps caps = m_source_caps.find (m_active_source)->second;

        Bmp::URI u;

        switch (status)
        {
            case PLAYSTATUS_STOPPED:

              if (!m_error_condition)
                {
                  status_label->set_text ("");
                  status_image->set (yy_stopped);
                }

              if (m_popup)
                {
                  m_popup->hide ();
                  m_popup->set_source_icon (Glib::RefPtr<Gdk::Pixbuf>(0));
                }

              if (m_active_source != SOURCE_NONE)
                {
                  m_sources.find (m_active_source)->second->stop();
                }

              m_source_selection->set_active_source (SOURCE_NONE);
              m_active_source = SOURCE_NONE;

              bmp_status_icon_set_from_pixbuf (m_status_icon, m_status_icon_image_default->gobj());

              seekrange->set_sensitive (false);
              seekrange->set_range (0., 1.); 
              seekrange->set_value (double(0.));

              time_label->set_markup ("<b>--:--</b>");

              m_actions_main->get_action (PLAYER_SHELL_ACTION_RECOMMEND_WITH_LAST_FM)->set_sensitive (false);
              m_actions_main->get_action (PLAYER_SHELL_ACTION_STOP)->set_sensitive (false);

              m_sti.reset ();
              m_sent = false;

#ifdef HAVE_GAIM
              m_actions_main->get_action (PLAYER_SHELL_ACTION_GAIM_SEND)->set_sensitive (false);
#endif //HAVE_GAIM

#ifdef HAVE_VISUALIZATIONS
              vis->unset_xoverlay ();
#endif //HAVE_VISUALIZATIONS

              break;

            case PLAYSTATUS_WAITING:
              bmp_status_icon_set_from_pixbuf (m_status_icon, m_status_icon_image_default->gobj());
              status_image->set (yy_stopped);
              break;

            case PLAYSTATUS_PLAYING:

              m_error_condition = false;

              if (caps & PlaybackSource::CAN_SEEK)
                {
                  seekrange->set_sensitive (true);
                }
              else
                {
                  seekrange->set_sensitive (false);
                  seekrange->set_range (0., 1.); 
                  seekrange->set_value (double(0.));
                }

              bmp_status_icon_set_from_pixbuf (m_status_icon, m_status_icon_image_playing->gobj());
              status_image->set (yy_playing);

              if (m_popup && (m_active_source != SOURCE_NONE))
                {
                  m_popup->set_source_icon 
                          (m_popup->render_icon (Gtk::StockID (source_icons_stock[m_active_source]),
                                                 Gtk::ICON_SIZE_LARGE_TOOLBAR));
                }

#ifdef HAVE_GAIM
              try {
                  u = Bmp::URI (::play->property_stream().get_value());
                  m_actions_main->get_action (PLAYER_SHELL_ACTION_GAIM_SEND)->set_sensitive
                          ((u.get_protocol() == Bmp::URI::PROTOCOL_FILE) && gaim_dbus->gaim_is_present()); 
                }
              catch (...) 
                {
                  m_actions_main->get_action (PLAYER_SHELL_ACTION_GAIM_SEND)->set_sensitive
                          (false);
                }
#endif //HAVE_GAIM

              u  = Bmp::URI (::play->property_stream().get_value());
              m_monitor = (u.get_protocol() == Bmp::URI::PROTOCOL_FILE);
              break;

            case PLAYSTATUS_PAUSED:
              if (caps & PlaybackSource::CAN_SEEK)
                {
                  seekrange->set_sensitive (false);
                }
              bmp_status_icon_set_from_pixbuf (m_status_icon, m_status_icon_image_paused->gobj());
              status_image->set (yy_paused);
              break;

            default: ;
        }
      }

      bool
      PlayerShell::on_window_state_event (GdkEventWindowState *event)
      {
        if (!event->changed_mask & GDK_WINDOW_STATE_ICONIFIED)
          return false;

        bool iconified = (event->new_window_state & GDK_WINDOW_STATE_ICONIFIED);

        if (iconified)
          {
            if (mcs->key_get<bool>("bmp", "ui-esc-trayconify"))
            {
              set_skip_taskbar_hint (true);
              set_skip_pager_hint (true);
              m_visible = false;
            }
        }
        else
        {
            if (mcs->key_get<bool>("bmp", "ui-esc-trayconify"))
            {
              set_skip_taskbar_hint (false);
              set_skip_pager_hint (false);
              present ();
              m_visible = true;
            }
        }

        return false;
      }

      bool
      PlayerShell::on_key_press_event (GdkEventKey *event)
      {
        if (event->keyval == GDK_Escape)
          {
            iconify ();
            return true;
          }

        return Gtk::Widget::on_key_press_event (event); 
      }

      PlayerShell*
      PlayerShell::create ()
      {
#ifdef HAVE_VISUALIZATIONS
        const std::string path (BMP_GLADE_DIR G_DIR_SEPARATOR_S "main-ui-vis.glade");
#else
        const std::string path (BMP_GLADE_DIR G_DIR_SEPARATOR_S "main-ui.glade");
#endif //HAVE_VISUALIZATIONS
        Glib::RefPtr<Gnome::Glade::Xml> glade_xml = Gnome::Glade::Xml::create (path);
        PlayerShell *p = 0;
        glade_xml->get_widget_derived ("main-ui", p);
        return p; 
      }

      PlayerShell::~PlayerShell ()
      {
        g_object_unref (m_status_icon);
#ifdef HAVE_VISUALIZATIONS
        delete vis;
#endif //HAVE_VISUALIZATIONS
        delete mcs_bind;
      }

      GHashTable*
      PlayerShell::get_metadata_from_source (PlaybackSourceID source)
      {
        Sources::iterator s_iter = m_sources.find (source);
        if (s_iter == m_sources.end()) throw InvalidSourceError();

        SourceCaps::iterator c_iter = m_source_caps.find (source);
        if (c_iter == m_source_caps.end()) throw InvalidSourceError();


        if (c_iter->second & PlaybackSource::CAN_PROVIDE_METADATA)
        {
          GHashTable *m = s_iter->second->get_metadata ();
          if (!m) throw NoMetadataError();
          return m;
        }
        else
        {
          throw NoMetadataError();
        }
      }

      PlaybackSource::Caps
      PlayerShell::get_source_caps (PlaybackSourceID source)
      {
        SourceCaps::iterator i = m_source_caps.find (source);

        if (i == m_source_caps.end())
          throw InvalidSourceError();
        else
          return i->second; 
      }

      PlaybackSource::Flags
      PlayerShell::get_source_flags (PlaybackSourceID source)
      {
        SourceFlags::iterator i = m_source_flags.find (source);

        if (i == m_source_flags.end())
          throw InvalidSourceError();
        else
          return i->second; 
      }

      PlaybackSourceID
      PlayerShell::get_current_source ()
      {
        return m_active_source;
      }

      void
      PlayerShell::play_uris (VUri& list, bool playback)
      {
        if (playback)
          {
            stop ();
            m_active_source = SOURCE_PLAYLIST;
            m_source_selection->set_active_source (SOURCE_PLAYLIST);
            m_source_selection->select_source (SOURCE_PLAYLIST);
            notebook_main->set_current_page (int(SOURCE_PLAYLIST));
          }

        reinterpret_cast<Bmp::UiPart::Playlist*>(m_sources[SOURCE_PLAYLIST].get())->add_uris (list, playback);

        if (playback) 
          {
            m_sources[SOURCE_PLAYLIST]->play_requested();
            ::play->switch_stream (m_sources[SOURCE_PLAYLIST]->get_uri(),
              m_sources[SOURCE_PLAYLIST]->get_type());
            m_actions_main->get_action (PLAYER_SHELL_ACTION_STOP)->set_sensitive (true);
          } 
      }

      void
      PlayerShell::play_lastfm_uri (Glib::ustring const& lastfm_uri)
      {
        if (!Network::check_connected())
          return;

        reinterpret_cast<Bmp::UiPart::LASTFM*>(m_sources[SOURCE_LASTFM].get())->set_uri (lastfm_uri);
        m_source_selection->select_source (SOURCE_LASTFM);
        notebook_main->set_current_page (int(SOURCE_LASTFM));
      }

      void
      PlayerShell::import_request (Bmp::VUri& list)
      {
        notebook_main->set_current_page (int(SOURCE_ALBUMS));
        reinterpret_cast<Bmp::UiPart::Albums*>(m_sources[SOURCE_ALBUMS].get())->import_external (list);
      }

      void
      PlayerShell::raise_ui ()
      {
        if (!m_visible)
        {
          deiconify ();
        }
      }

      /////////////////// Sources TreeView

      PlayerShell::SourceSelection::SourceSelection

            (BaseObjectType                       * obj,
             Glib::RefPtr<Gnome::Glade::Xml> const& xml)

               :    Gtk::VBox         (obj),
                    m_active_source   (SOURCE_NONE),
                    m_selected_source (SOURCE_NONE)

      {
         struct { 
              char *  name;
              bool    network_dependent;
          } sources[] = {
            { N_("_Albums"),     false,  },
            { N_("_Shoutcast"),  true,   },
            { N_("_Icecast"),    true,   },
            { N_("Audio _CD"),   false,  },
            { N_("Last._FM"),    true,   },
            { N_("P_odcast"),    true,   },
            { N_("Pla_ylist"),   false,  },
#ifdef HAVE_MOOD
            { N_("_Downloads"),  true    },
#endif //HAVE_MOOD
          };

        bool network_present = Network::check_connected();

        if (!network_present)
          set_homogeneous (false);

        for (unsigned int n = 0; n < G_N_ELEMENTS(sources); ++n)
        {
          using namespace Gtk;
          using namespace Glib;

          if (!network_present && sources[n].network_dependent)
            continue;

          Bmp::SourceButton * button = manage (new Bmp::SourceButton());

          ustring text; 
          text .append ("<b>") .append (_(sources[n].name)) .append ("</b>");
          button->set_label (text, true);
          button->set_image (render_icon (Gtk::StockID (source_icons_stock[n]),
                                          Gtk::ICON_SIZE_DND)); 

          button->signal_clicked().connect
              (sigc::bind (sigc::mem_fun (this, &Bmp::PlayerShell::SourceSelection::source_activated),
               PlaybackSourceID(n)));

          pack_start(*button, network_present, network_present);
          button->show_all ();
          b_map.insert (std::make_pair (PlaybackSourceID (n), button));
          st_map.insert (std::make_pair (PlaybackSourceID (n), n));
        }
      }

      PlayerShell::SourceSelection::~SourceSelection ()
      {
      }

      bool
      PlayerShell::SourceSelection::on_expose_event (GdkEventExpose * event)
      {
        Cairo::RefPtr<Cairo::Context> cr = Util::create_gdk_cairo_context (get_window ());

        int xoff = get_allocation().get_x();
        int yoff = get_allocation().get_y();

        cr->rectangle (xoff,
                       yoff,
                       get_allocation().get_width(),
                       get_allocation().get_height());

        cr->set_source_rgba (1., 1., 1., 1.);
        cr->fill ();

        gtk_paint_shadow (GTK_WIDGET (gobj())->style,
                          GTK_WIDGET (gobj())->window,
                          GTK_STATE_NORMAL,
                          GTK_SHADOW_IN,
                          NULL,
                          GTK_WIDGET (gobj()),
                          "scrolled_window",
                          get_allocation().get_x(),
                          get_allocation().get_y(),
                          get_allocation().get_width(),
                          get_allocation().get_height());

        Gtk::Widget::on_expose_event (event);

        return false;
      }

      void
      PlayerShell::SourceSelection::set_active_source (PlaybackSourceID source)
      {
        if (m_active_source != SOURCE_NONE)
          b_map.find (m_active_source)->second->set_playing (false);

        m_active_source = source;

        if (m_active_source != SOURCE_NONE)
          b_map.find (m_active_source)->second->set_playing (true);

        queue_draw ();
      }

      void
      PlayerShell::SourceSelection::select_source (PlaybackSourceID source)
      {
        if (m_selected_source != SOURCE_NONE)
          b_map.find (m_selected_source)->second->set_active (false);

        m_selected_source = source;

        if (m_selected_source != SOURCE_NONE)
          b_map.find (m_selected_source)->second->set_active (true);

        queue_draw ();

      }

      PlaybackSourceID 
      PlayerShell::SourceSelection::get_active_source ()
      {
        return m_active_source; 
      }


      void
      PlayerShell::SourceSelection::source_activated (PlaybackSourceID source)
      {
        using namespace Gtk;
        select_source (source);
        source_selected_.emit (source);
      }
}
