//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifndef BMP_UI_PART_ALBUMS_HH 
#define BMP_UI_PART_ALBUMS_HH

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <boost/tuple/tuple.hpp>
#include <boost/tuple/tuple_comparison.hpp>

#include <glibmm.h>
#include <glibmm/markup.h>
#include <gtkmm.h>
#include <libglademm.h>

#include <mcs/mcs.h>
#include <bmp/base_types.hh>

#ifdef HAVE_HAL
#  include "hal.hh"
#endif //HAVE_HAL

#include "amazon.hh"
#include "library.hh"
#include "vfs.hh"

#include "util.hh"
#include "util_file.hh"

#include "xds.hh"

#include "playbacksource.hh"
#include "ui-part-base.hh"

using namespace Bmp::Library;

namespace Bmp
{
  class PlayerShell;

  class AlbumView : public Gtk::TreeView
  {
    public:

      enum ViewSize { NORMAL, SMALL };

      class AlbumCR : public Gtk::TreeModel::ColumnRecord
      {
        public:

          Gtk::TreeModelColumn<Glib::RefPtr<Gdk::Pixbuf> >  cover;
          Gtk::TreeModelColumn<Library::Album>              album;
          Gtk::TreeModelColumn<guint64>                     guid;

          AlbumCR ()
          {
            add (cover);
            add (album);
            add (guid);
          }
      };
      AlbumCR m_album; 

    private:

      // Data Store Handling
      int  default_sort_func (Gtk::TreeModel::iterator const& iter_a, Gtk::TreeModel::iterator const& iter_b);
      bool visible_func (Gtk::TreeModel::iterator const& iter);
      void album_title_cell_data_func (Gtk::CellRenderer* cell_, Gtk::TreeModel::iterator const& m_iter);
      void album_new_cell_data_func (Gtk::CellRenderer* cell_, Gtk::TreeModel::iterator const& m_iter);
      void album_cover_cell_data_func (Gtk::CellRenderer* cell_, Gtk::TreeModel::iterator const& m_iter);

      Gtk::CellRendererPixbuf   *m_cell_album_new;
      Gtk::CellRendererText     *m_cell_album_text;


      Glib::RefPtr<Gtk::ListStore>        m_store_base;
      Glib::RefPtr<Gtk::TreeModelFilter>  m_store_filter;
      Glib::RefPtr<Gdk::Pixbuf>           m_emblem_new;
      Glib::RefPtr<Gdk::Pixbuf>           m_disc_normal;
      Glib::RefPtr<Gdk::Pixbuf>           m_disc_small;
      Glib::ustring                       m_sort_string;

      // Viewsize
      ViewSize m_viewsize; 

      // Selection Handling
      typedef std::pair < unsigned int, Gtk::TreeModel::RowReference > SelectionPair; 
      typedef std::list < SelectionPair > SelectionList; 

      bool          m_selection_list_modified;
      bool          m_selection_change_block;

      SelectionList m_selection_list_albums,
                    m_selection_list_albums_saved;

      bool record_selection_change (Glib::RefPtr<Gtk::TreeModel> const&, Gtk::TreeModel::Path const&, bool);

      // Misc
      bool m_show_only_new;
      bool m_show_only_sel;
      bool m_filter_artist_only;
      guint64 m_guid;

      typedef std::map< guint64 , Gtk::TreeModel::RowReference > NewItemsMap;
      NewItemsMap m_new_items_map;

      typedef std::map< ::Bmp::Library::Album::TR , Gtk::TreeModel::iterator > AlbumMap;
      AlbumMap m_album_map;

    public:

      bool has_new_items ();

      void new_remove_one (guint64 uid);
      void new_approve_one (guint64 uid);
      void new_approve_all ();
      void new_drop_all ();

      void save_context ();
      void restore_context ();

      void restore_selection ();

      bool matching_album (Library::Album const& album, Gtk::TreeModel::iterator & m_iter);
      bool matching_album (Library::Album const& album);

      void set_viewsize (ViewSize viewsize);
      void set_filtering (bool show_only_new, bool show_only_sel);
/*
      void set_sorting (Glib::ustring const& sort_text, bool artist_only = false);
*/
      void set_sorting (Glib::ustring const& sort_text);


#ifdef HAVE_HAL
      void modify (Bmp::VUriVrpPair const& pairs);
#else
      void modify (Bmp::VUri const& locations);
#endif //HAVE_HAL

      bool relocate     ();
      void remove       (bool unattended);
      void insert       (Bmp::Library::Album const& album, bool new_album, Gtk::TreeModel::iterator & m_iter);
      void insert       (Bmp::Library::Album const& album, bool new_album);
      void update_all   (bool only_cached = true);
      void update_new   ();
      void approve_sel  ();

      void current_selection (Library::AlbumV & list);
      void get_cover (Gtk::TreeModel::iterator const& m_iter, oustring const& asin, bool only_cached = true);

      AlbumView (BaseObjectType                       * cobject,
                 Glib::RefPtr<Gnome::Glade::Xml> const& xml);
      ~AlbumView () {}

  };

  class TrackView : public Gtk::TreeView
  {
    public:

      typedef sigc::signal<void> SignalColumnToggled;

    private:

      SignalColumnToggled signal_column_toggled_;

      typedef std::map <Glib::ustring, Gtk::TreeModel::RowReference> RowMapTracks;

      void cell_data_func_tracks (Gtk::CellRenderer*, Gtk::TreeModel::iterator const&);
      bool select_func_tracks (Glib::RefPtr<Gtk::TreeModel> const&, Gtk::TreeModel::Path const&, bool);

      RowMapTracks m_row_map_tracks;
      void notify_remove (Glib::ustring const&);

      Gtk::CellRenderer *cell_playing,
                        *cell_rating,
                        *cell_new;

      Glib::RefPtr<Gdk::Pixbuf>     m_rating[6];
      Glib::RefPtr<Gdk::Pixbuf>     m_pixbuf_playing;

      Glib::RefPtr<Gtk::ListStore>  m_store;

      void column_toggled ();

      int           m_sort_string_column_saved;
      Gtk::SortType m_sort_string_order_saved;

      boost::optional<Gtk::TreeModel::iterator> playing_row;
      boost::optional<Glib::ustring>            playing_uri;

      // External Callbacks
      void check_update (Glib::ustring const& location, Bmp::Library::UTrack track);

    protected:

      virtual bool on_event (GdkEvent * ev);

    public:

      SignalColumnToggled&
      signal_column_toggled () { return signal_column_toggled_; }

      class TrackCR : public Gtk::TreeModel::ColumnRecord
      {
        public:

          Gtk::TreeModelColumn<int>           track;
          Gtk::TreeModelColumn<Glib::ustring> title;
          Gtk::TreeModelColumn<int>           rating;
          Gtk::TreeModelColumn<Glib::ustring> artist;
          Gtk::TreeModelColumn<Glib::ustring> album;
          Gtk::TreeModelColumn<Glib::ustring> genre;
          Gtk::TreeModelColumn<int>           count;
          Gtk::TreeModelColumn<int>           bitrate;
          Gtk::TreeModelColumn<int>           samplerate;
          Gtk::TreeModelColumn<Glib::ustring> location;
          Gtk::TreeModelColumn<bool>          present;
#ifdef HAVE_HAL
          Gtk::TreeModelColumn<Glib::ustring> volume_udi;
          Gtk::TreeModelColumn<Glib::ustring> volume_relative_path;
#endif //HAVE_HAL
          Gtk::TreeModelColumn<bool>          new_item;
          Gtk::TreeModelColumn<guint64>       guid;

          TrackCR ()
          {
            add (track);
            add (title);
            add (rating);
            add (artist);
            add (album);
            add (genre);
            add (count);
            add (bitrate);
            add (samplerate);
            add (location);
            add (present);
#ifdef HAVE_HAL
            add (volume_udi);
            add (volume_relative_path);
#endif //HAVE_HAL
            add (new_item);
            add (guid);
          }
      };
      TrackCR m_track;

      void clear  ();
      void unsort ();
      bool remove ();

      void add_tracks (Bmp::DB::VRows const& vector);
      void get_files  (Util::FileList & list);

      bool all_tracks_present ();

#ifdef HAVE_HAL
      void get_tracks (Bmp::VUriVrpPair & pairs);
#else //!HAVE_HAL
      void get_tracks (Bmp::VUri & locations);
#endif //HAVE_HAL

      void save_context  ()
      {
        m_store->get_sort_column_id (m_sort_string_column_saved, m_sort_string_order_saved);
      }

      void restore_context ()
      {
        m_store->clear();
        m_store->set_sort_column (m_sort_string_column_saved, m_sort_string_order_saved);
      }
  
      // PlaybackSource
      Glib::ustring playback_uri ();
      bool notify_play (unsigned int& position, unsigned int& size);
      bool notify_next (unsigned int& position, unsigned int& size);
      bool notify_prev (unsigned int& position, unsigned int& size);
      void notify_stop ();
      void has_next_prev (bool &next, bool &prev);

      guint64 m_guid;

#include "exception.hh"

      EXCEPTION(NoCurrentUriError)


      TrackView (BaseObjectType                        *cobject,
                 Glib::RefPtr<Gnome::Glade::Xml> const& xml);
      ~TrackView () {}
  };

  namespace UiPart
  {
    class Albums : public Bmp::PlaybackSource, public Bmp::UiPart::Base
    {
      public:

        typedef sigc::signal<void, Bmp::VUri&> SignalURIEnqueueRequest;
#ifdef HAVE_HAL
        typedef sigc::signal<void> SignalRescanDevices;
#endif //HAVE_HAL
    
        SignalURIEnqueueRequest&
        signal_uri_enqueue_request  () { return s_uri_enqueue_request_; }

#ifdef HAVE_HAL
        SignalRescanDevices&
        signal_rescan_devices       () { return s_rescan_devices_; }
#endif //HAVE_HAL

      private:

        SignalURIEnqueueRequest s_uri_enqueue_request_;

#ifdef HAVE_HAL
        SignalRescanDevices     s_rescan_devices_;
#endif //HAVE_HAL

        typedef std::map < Glib::ustring, Gtk::TreeModel::iterator> IterMap;

#ifdef HAVE_HAL
        void hal_volume_removed  (Library::HAL::Volume volume);
        void hal_volume_added    (Library::HAL::Volume volume);
#endif //HAVE_HAL

        void remove (bool unattended = false);
        void update (bool newonly = true);

        void on_sort_entry_changed ();

        void on_new_approve_all  ();
        void on_new_approve_sel  ();
        void on_new_drop_all ();
				void on_get_covers ();

        void on_tracks_selection_changed ();
        void on_albums_selection_changed ();

        void on_relocate ();
        void on_modify ();
        void on_remove ();

        void on_tracks_remove ();
        void on_tracks_enqueue ();

        void on_tracks_import ();
        void on_tracks_import_mbtag ();

        void on_view_size_large_activated ();
        void on_view_size_small_activated ();
        void on_show_album_list_toggled ();
#if 0
        void on_show_artist_list_toggled ();
#endif

        void on_continuous_play_toggled ();

        void activate_default (Gtk::TreeModel::Path const& path, Gtk::TreeViewColumn* column);

        Bmp::AlbumView        *m_view_albums;
        Bmp::TrackView        *m_view_tracks;
        Gtk::Entry            *m_albums_search;

#if 0
        Gtk::TreeView         *m_view_artists;
        Bmp::Resizable        *m_alignment_artists;
        void artist_selected ();
#endif

        class CompletionModel : public Gtk::TreeModel::ColumnRecord
        {
          public: 
            Gtk::TreeModelColumn<Glib::ustring> name;
            Gtk::TreeModelColumn<std::string>   key; 
          CompletionModel () { add (name); add (key); }
        };
        CompletionModel completion_model;

        Glib::RefPtr<Gtk::EntryCompletion>  m_albums_search_completion;
        Glib::RefPtr<Gtk::ListStore>        m_albums_search_completion_model;

        int   completion_sort_func  (Gtk::TreeModel::iterator const& iter_a, Gtk::TreeModel::iterator const& iter_b);
        void  update_completion     ();       

        bool on_sort_entry_changed_timeout_func ();
        Glib::Timer entry_changed_timer;
        sigc::connection sort_entry_changed_timeout_conn;

        Gtk::ProgressBar * m_progress;

        void progress_start (int n_mod_tracks);
        void progress_step  ();
        void progress_stop  ();
        int  m_n_mod_tracks;
        int  m_n_mod_current;

        Gtk::ComboBox * albums_cbox_viewmode;
        void on_viewmode_changed ();
        enum AlbumsViewMode
        {
          M_EVERYTHING,
          M_SELECTED,
          M_NEW,
        };

        Gtk::Label  * metadata_label_current_artist;
        Gtk::Label  * metadata_label_current_album;
        Gtk::Label  * metadata_label_current_date;
        Gtk::Image  * metadata_cover_current_album;

        Glib::RefPtr<Gtk::ActionGroup>  m_actions;
        Glib::RefPtr<Gtk::ToggleAction> m_action_show_only_new;

        bool popup_menu_tracks (GdkEvent *ev);
        bool popup_menu_albums (GdkEvent *ev);

        void send_title ();

        Glib::Mutex m_modify_lock;
        Glib::Cond  m_modify_cond;
        bool        m_busy;
        bool        m_init;

        boost::optional<Glib::ustring>  m_saved_context_filter;

        Glib::RefPtr<Gdk::Pixbuf> m_audio_empty;
        Glib::RefPtr<Gdk::Pixbuf> m_audio_multiple;
        Glib::RefPtr<Gdk::Pixbuf> m_audio_default;

        Gtk::Notebook * notebook_albums;

        friend class Bmp::PlayerShell;

      protected:

        virtual guint
        add_ui ();

        virtual Glib::ustring
        get_uri (); 

        virtual Glib::ustring
        get_type () { return Glib::ustring(); }

        virtual bool
        go_next ();

        virtual bool
        go_prev ();

        virtual void
        stop ();

        virtual void
        play ();

        virtual void
        play_requested () {}

        virtual void
        restore_context ();

        virtual GHashTable*
        get_metadata ();

      public:

        void import_external (VUri & uri_list);

        Albums (Glib::RefPtr<Gnome::Glade::Xml> const& xml, Glib::RefPtr<Gtk::UIManager> ui_manager);
        ~Albums ();

    };
  }
}
#endif
