//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2007 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <glibmm.h>
#include <glibmm/i18n.h>
#include <gtkmm.h>
#include <libglademm.h>
#include <glade/glade.h>
#include <glib/gstdio.h>

#include <cstring>
#include <iostream>
#include <sstream>
#include <string>

#include <boost/format.hpp>
#include <boost/optional.hpp>

#include "dialog-track-details.hh"

#include "database.hh"
#include "lastfm.hh"
#include "main.hh"
#include "network.hh"
#include "paths.hh"
#include "playbacksource.hh"
#include "stock.hh"
#include "ui-tools.hh"
#include "util.hh"
#include "uri++.hh"

#include "x_amazon.hh"
#include "x_library.hh"
#include "x_mcsbind.hh"

#include "musicbrainz/mb-utils.hh"

using namespace Glib;
using namespace Gtk;
using namespace Bmp;
using namespace DB;
using namespace std; 

namespace Bmp
{
  class TrackInfoView
    : public TreeView
  {
      class Columns
        : public TreeModel::ColumnRecord
      {
        public:

          TreeModelColumn <ustring>  value; 
          TreeModelColumn <ustring>  title; 
          TreeModelColumn <string>   attribute; 

          Columns ()
          {
            add (value);
            add (title);
            add (attribute); 
          }
      };

      Columns            m_columns;
      RefPtr <ListStore> m_store;
      Label              m_label1, m_label2;
      Bmp::DB::Row       m_row;

    public:

      TrackInfoView (BaseObjectType                 * obj,
                     RefPtr<Gnome::Glade::Xml> const& xml)
      : TreeView (obj)
      {
        {
          CellRendererText * cell = manage (new CellRendererText());
          TreeViewColumn * column = manage (new TreeViewColumn ("", *cell));
          column->add_attribute (*cell, "text", m_columns.title);
          column->set_resizable (false);
          column->set_expand (false);
          append_column (*column);
        }

        {
          CellRendererText * cell = manage (new CellRendererText());
          TreeViewColumn * column = manage (new TreeViewColumn ("", *cell));
          cell->property_editable() = true; // This way people can copy out text, but we don't actually edit it here
          column->set_cell_data_func (*cell, sigc::mem_fun (this, &Bmp::TrackInfoView::cell_data_func));
          column->set_resizable (false);
          column->set_expand (false);
          append_column (*column);
        }

        m_label1.property_xalign() = 0.0;
        m_label2.property_xalign() = 0.0;
          
        m_label1.property_label() = "<small><b>Attribute</b></small>";
        m_label2.property_label() = "<small><b>Value</b></small>";

        m_label1.property_use_markup() = true;
        m_label2.property_use_markup() = true; 

        m_label1.show_all ();
        m_label2.show_all ();

        get_column(0)->set_widget (m_label1);
        get_column(1)->set_widget (m_label2);

        m_store = ListStore::create (m_columns);  
        m_store->set_default_sort_func (sigc::mem_fun (this, &Bmp::TrackInfoView::default_sort_func));
        m_store->set_sort_column (TreeSortable::DEFAULT_SORT_COLUMN_ID, SORT_ASCENDING);

        set_grid_lines (TREE_VIEW_GRID_LINES_BOTH);
        set_model (m_store);
      }

      ~TrackInfoView()
      {}

      int
      default_sort_func (TreeModel::iterator const& iter_a,
                         TreeModel::iterator const& iter_b)
      {
        ustring str_a = (*iter_a)[m_columns.title];
        ustring str_b = (*iter_b)[m_columns.title];
        return str_a.lowercase().compare (str_b.lowercase());
      }

      void
      cell_data_func (CellRenderer * basecell, TreeModel::iterator const& iter)
      {
        CellRendererText * cell = dynamic_cast <CellRendererText*> (basecell);

        Row::const_iterator i = m_row.find ((*iter)[m_columns.attribute]);
        if (i == m_row.end())
        {
          cell->property_text() = "";
          return;
        }
      
        switch (i->second.which()) 
        {
          case VALUE_TYPE_INT:
            cell->property_text() = ((boost::format ("%llu") % boost::get <uint64_t> (i->second)).str());
            break;

          case VALUE_TYPE_REAL:
            cell->property_text() = ((boost::format ("%f")   % boost::get <double> (i->second)).str());
            break;

          case VALUE_TYPE_STRING:
            cell->property_text() = boost::get <string> (i->second);
            break;
        }
      }

      bool
      set_track (string const& location)
      {
        m_row = DB::Row();
        library->get_metadata (location, m_row);  
        unsigned int last = ATTRIBUTE_TYPE;
        for (unsigned int n = 0; n <= last; ++n)
        {
          TreeIter iter (m_store->append());
          (*iter)[m_columns.attribute] = get_attribute_info (AttributeId (n)).id;
          (*iter)[m_columns.title] = get_attribute_info (AttributeId (n)).title;
        }
        return false;
      }

      bool
      set_metadata (TrackMetadata const& metadata)
      {
        m_row = DB::Row();
        TreeIter iter;

#define ADD_ATTR(member,attr) \
        if (metadata. member) \
        { \
          iter = m_store->append(); \
          (*iter)[m_columns.attribute] = get_attribute_info (AttributeId (attr)).id; \
          (*iter)[m_columns.title] = get_attribute_info (AttributeId (attr)).title; \
          m_row.insert (make_pair (get_attribute_info (AttributeId (attr)).id, ::Bmp::DB::Variant (metadata. member .get()))); \
        } \


#define ADD_ATTR_NAME(member,attr,name) \
        if (metadata. member) \
        { \
          iter = m_store->append(); \
          (*iter)[m_columns.attribute] = get_attribute_info (AttributeId (attr)).id; \
          (*iter)[m_columns.title] = name; \
          m_row.insert (make_pair (get_attribute_info (AttributeId (attr)).id, ::Bmp::DB::Variant (metadata. member .get()))); \
        } \


        ADD_ATTR(artist, ATTRIBUTE_ARTIST);
        ADD_ATTR(title, ATTRIBUTE_TITLE);
        ADD_ATTR_NAME(album, ATTRIBUTE_ALBUM, _("Album/Source"));
        ADD_ATTR(genre, ATTRIBUTE_GENRE);

        return false;
      }

      void
      clear ()
      {
        m_store->clear ();
      }
  };
}

namespace Bmp
{
  TrackDetails::~TrackDetails ()
  {
    m_lyrics_request.clear ();
    m_artist_request.clear ();
  }

  TrackDetails*
  TrackDetails::create ()
  {
      const string path (BMP_GLADE_DIR G_DIR_SEPARATOR_S "dialog-track-details.glade");
      RefPtr<Gnome::Glade::Xml> glade_xml = Gnome::Glade::Xml::create (path);
      TrackDetails* dialog = 0;
      glade_xml->get_widget_derived ("dialog-track-details", dialog);
      glade_xml_signal_autoconnect (glade_xml->gobj());
      return dialog;
  }

  TrackDetails::TrackDetails (BaseObjectType                 * obj,
                              RefPtr<Gnome::Glade::Xml> const& xml)
  : Window            (obj)
  , m_ref_xml         (xml)
  , m_artist_request  (LastFM::XMLRPC::ArtistMetadataRequestRefPtr (0))
  , m_lyrics_request  (LyricWiki::TextRequestRefP (0))
  {
      Util::window_set_icon_list (*this, "player");

      m_ref_xml->get_widget ("image-cover",     m_widget_cover);
      m_ref_xml->get_widget ("textview-lyrics", m_widget_lyrics);
      m_ref_xml->get_widget ("textview-artist", m_widget_artist);

      m_ref_xml->get_widget ("notebook-artist", m_notebook_artist);
      m_ref_xml->get_widget ("notebook-lyrics", m_notebook_lyrics);

      dynamic_cast<Image*>(m_ref_xml->get_widget ("throbber-artist"))->set (build_filename (BMP_IMAGE_DIR, BMP_THROBBER));
      dynamic_cast<Image*>(m_ref_xml->get_widget ("throbber-lyrics"))->set (build_filename (BMP_IMAGE_DIR, BMP_THROBBER));

      m_ref_xml->get_widget_derived ("treeview-trackinfo", m_widget_trackinfo);
      m_widget_trackinfo->show_all ();

      dynamic_cast<Button*>(m_ref_xml->get_widget ("button-refresh-lyrics"))->signal_clicked().connect
        (sigc::bind (sigc::mem_fun (*this, &Bmp::TrackDetails::get_lyrics), true));
      dynamic_cast<Button*>(m_ref_xml->get_widget ("button-refresh-artist"))->signal_clicked().connect
        (sigc::mem_fun (*this, &Bmp::TrackDetails::get_artist));
  }

  void
  TrackDetails::got_artist (std::string const& metadata, guint code)
  {
    m_widget_artist->get_buffer()->set_text (Util::sanitize_lastfm (metadata));
    m_ref_xml->get_widget ("button-refresh-artist")->set_sensitive (1);
    m_notebook_artist->set_current_page (0);
  }

  void
  TrackDetails::get_artist ()
  {
    if (m_artist_request)
    {
      m_artist_request->cancel ();
    }

    if (m_metadata.mb_artist_sort_name)
    {
      std::string a = m_metadata.mb_artist_sort_name.get();      
      if (MusicBrainzUtil::reverse_sortname (a))
      {
        m_ref_xml->get_widget ("button-refresh-artist")->set_sensitive (0);
        m_notebook_artist->set_current_page (1);
        m_artist_request = LastFM::XMLRPC::ArtistMetadataRequest::create (a);
        m_artist_request->reply().connect (sigc::mem_fun (*this, &TrackDetails::got_artist));
        m_artist_request->run ();
        return;
      }
    }

    if (m_metadata.artist)
    {
      m_ref_xml->get_widget ("button-refresh-artist")->set_sensitive (0);
      m_notebook_artist->set_current_page (1);
      m_artist_request = LastFM::XMLRPC::ArtistMetadataRequest::create (m_metadata.artist.get());
      m_artist_request->reply().connect (sigc::mem_fun (*this, &TrackDetails::got_artist));
      m_artist_request->run ();
      return;
    }
  }

  void
  TrackDetails::got_lyrics (std::string const& lyrics, bool have_em)
  {
    if (have_em)
    {
      m_widget_lyrics->get_buffer()->set_text (lyrics);
    }

    m_notebook_lyrics->set_current_page (0);
    m_ref_xml->get_widget ("button-refresh-lyrics")->set_sensitive (1);
  }

  void
  TrackDetails::get_lyrics (bool forced)
  {
    if (m_lyrics_request)
    {
      m_lyrics_request->cancel ();
    }

    if (m_metadata.artist && m_metadata.title)
    {
      m_notebook_lyrics->set_current_page (1);
      m_ref_xml->get_widget ("button-refresh-lyrics")->set_sensitive (0);
      m_lyrics_request = LyricWiki::TextRequest::create (m_metadata.artist.get(), m_metadata.title.get(), forced);
      m_lyrics_request->lyrics().connect (sigc::mem_fun (*this, &TrackDetails::got_lyrics));
      m_lyrics_request->run ();
    }
  }

  void
  TrackDetails::set_metadata (TrackMetadata const& metadata)
  {
    m_metadata = metadata;

    m_widget_artist->get_buffer()->set_text ("");
    m_widget_lyrics->get_buffer()->set_text ("");
    m_widget_trackinfo->clear ();

    if (m_metadata.location)
    { 
      signal_idle().connect (sigc::bind (sigc::mem_fun (*m_widget_trackinfo, &Bmp::TrackInfoView::set_track), m_metadata.location.get()));
    }
    else
    {
      signal_idle().connect (sigc::bind (sigc::mem_fun (*m_widget_trackinfo, &Bmp::TrackInfoView::set_metadata), m_metadata));
    }

    if (m_metadata.image)
    {
      m_widget_cover->set (m_metadata.image->scale_simple (256, 256, Gdk::INTERP_BILINEAR));
    }
    else if (m_metadata.asin)
    {
      try
      {
        RefPtr<Gdk::Pixbuf> cover;
        amazon->fetch (m_metadata.asin.get(), cover, true);
        m_widget_cover->set (cover->scale_simple (256, 256, Gdk::INTERP_BILINEAR));
      }
      catch (...)
      {
        m_widget_cover->set (Gdk::Pixbuf::create_from_file (build_filename
          (BMP_IMAGE_DIR, BMP_COVER_IMAGE_DEFAULT))->scale_simple (256, 256, Gdk::INTERP_BILINEAR));
      }
    }
    else
    {
      m_widget_cover->set (Gdk::Pixbuf::create_from_file (build_filename
        (BMP_IMAGE_DIR, BMP_COVER_IMAGE_DEFAULT))->scale_simple (256, 256, Gdk::INTERP_BILINEAR));
    }

    get_lyrics (0);
    get_artist ();
    
  }
} // namespace Bmp
