//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2007 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifndef BMP_LAST_FM_HH
#define BMP_LAST_FM_HH

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <vector>
#include <queue>
#include <exception>

#include <glibmm/markup.h>
#include <glibmm/ustring.h>
#include <gtkmm.h>
#include <libglademm.h>

#include <boost/variant.hpp>

#include "mcs/mcs.h"
#include "bmp/base-types.hh"
#include "bmp/library-types.hh"

#include "xspf-types.hh"
#include "lastfm-types.hh"
#include "xts.hh"

#include "minisoup.hh"

using namespace Glib;

namespace Bmp
{
  namespace LastFM
  {

    typedef sigc::signal <void> Signal;

#include "exception.hh"

    EXCEPTION(LastFMNotConnectedError)
    EXCEPTION(LastFMStreamTuningError)

    namespace XMLRPC
    {
      /////////////////////////////////////////////////
      ///// Requests
      /////////////////////////////////////////////////

      ustring
      formatXmlRpc    (ustring const& method, xmlRpcVariantV const& argv);

      class XmlRpcCall; 
      typedef Glib::RefPtr<XmlRpcCall> XmlRpcCallRefPtr;

      class XmlRpcCall
        : public Glib::Object
      {
        public:
          typedef sigc::signal<void, std::string const&, guint> SignalReply;
        public:
          SignalReply & reply() { return s_; }
        protected:
          xmlRpcVariantV m_v;
          Bmp::Soup::RequestRefP m_soup_request;
          SignalReply s_;
          XmlRpcCall ();
        public:
          virtual ~XmlRpcCall ();
          void cancel ();
          void setMethod (const ustring& method);
      };

      class XmlRpcCallSync
        : public Glib::Object
      {
        protected:
          xmlRpcVariantV m_v;
          Bmp::Soup::RequestSyncRefP m_soup_request;
          XmlRpcCallSync ();
        public:
          virtual ~XmlRpcCallSync ();
          void setMethod (const ustring& method);
      };

      class ArtistMetadataRequest;
      typedef Glib::RefPtr<ArtistMetadataRequest> ArtistMetadataRequestRefPtr;

      class ArtistMetadataRequest
        : public XmlRpcCall
      {
          ustring m_artist;
          void reply_cb (char const* data, guint size, guint status_code);
        private: 
          ArtistMetadataRequest (ustring const& artist); 
        public:
          static ArtistMetadataRequestRefPtr create (ustring const& artist); 
          virtual ~ArtistMetadataRequest () {};
          void run ();
      };

      class RequestBase
        : public XmlRpcCallSync
      {
        public:
  
          RequestBase (ustring const& method);
          virtual ~RequestBase () {};

          void run ();

        protected:

          std::string   m_pass;
          std::string   m_user;
          std::string   m_time;
          std::string   m_key;
          std::string   m_kmd5;
          std::string   m_name;
      };

      class TrackAction
        : public RequestBase
      {
        public:

          TrackAction (ustring const& method, XSPF::Item const& item);
          virtual ~TrackAction () {}
      };

      class TagAction 
        : public RequestBase
      {
        public:

          TagAction (ustring const& method, Bmp::XSPF::Item const& item, Bmp::UStrV const& tags);
          virtual ~TagAction () {}
      };

      class RecommendAction
        : public RequestBase
      {
        public:

          enum RecommendItem
          {
            RECOMMEND_ARTIST,
            RECOMMEND_ALBUM,
            RECOMMEND_TRACK,
          };

          RecommendAction (RecommendItem item,
                      ustring const& artist, ustring const& album, ustring const& title, ustring const& user, ustring const& message); 
          virtual ~RecommendAction () {}
      };
    } // namespace:XMLRPC

    namespace WS
    {
      class TagsGlobReq;
      typedef Glib::RefPtr<TagsGlobReq> TagsGlobReqRefP;

      class TagsGlobReq
        : public Glib::Object
      {
          TagsGlobReq (TagGlobalType type, std::string const& id1, std::string const& id2 = std::string() );
          void reply_cb (char const* data, guint size, guint status_code);
        public:
          ~TagsGlobReq ();
          static TagsGlobReqRefP create (TagGlobalType type, std::string const& id1, std::string const& id2 = std::string() );
          void run ();
          void cancel ();
        private:
          Soup::RequestRefP m_soup_request;
          TagV m_tags;
        public:
          typedef sigc::signal<void, TagV const&> SignalTags;
        private:
          SignalTags s_tags_;
        public:
          SignalTags & tags () { return s_tags_; }
      };

      class TagsUserReq;
      typedef Glib::RefPtr<TagsUserReq> TagsUserReqRefP;

      class TagsUserReq
        : public Glib::Object
      {
          TagsUserReq (TagUserType type, std::string const& id1, std::string const& id2 = std::string() );
          void reply_cb (char const* data, guint size, guint status_code);
        public:
          ~TagsUserReq ();
          static TagsUserReqRefP create (TagUserType type, std::string const& id1, std::string const& id2 = std::string() );
          void run ();
        private:
          Soup::RequestRefP m_soup_request;
          TagV m_tags;
        public:
          typedef sigc::signal<void, TagV const&> SignalTags;
        private:
          SignalTags s_tags_;
        public:
          SignalTags & tags () { return s_tags_; }
      };


      // md XXX: Sync API (try to deprecate it)

      void tags_glob        ( TagGlobalType type,
                              TagV & tags,
                              std::string const& id1,
                              std::string const& id2 = std::string());

      void tags_user        ( TagUserType type,
                              TagV & tags,
                              std::string const& id1 = std::string(),
                              std::string const& id2 = std::string());

      void artists          ( ArtistType type, std::string const& id, LastFMArtistV & x );
      void users            ( UserType type, std::string const& username, UserV & x );
      void matchtags        ( std::string const& username, RankedTagV & ranked );

      std::string
      avatar_url            ( std::string const& username );
                              
    } // namespace:WS

  } // namespace LastFM
} // namespace Bmp

namespace Bmp
{
  namespace LastFM
  {
    enum RadioErrorCode
    {
        RADIO_ERROR_NOT_ENOUGH_CONTENT = 1, // There is not enough content to play this station.
        RADIO_ERROR_FEWGROUPMEMBERS,        // This group does not have enough members for radio.
        RADIO_ERROR_FEWFANS,                // This artist does not have enough fans for radio.
        RADIO_ERROR_UNAVAILABLE,            // This item is not available for streaming.
        RADIO_ERROR_SUBSCRIBE,              // This feature is only available to subscribers.
        RADIO_ERROR_FEWNEIGHBOURS,          // There are not enough neighbours for this radio.
        RADIO_ERROR_OFFLINE                 // The streaming system is offline for maintenance, please try again later
    };

    /** Client code for Last.FM Radio
     *
     */
    class Radio
    {
      public:

        struct Session
        {
            ustring session;
            bool    subscriber;
            ustring base_url;
            ustring base_path;

            Session ()
            : subscriber (0) {}
        };

      public:

        Radio ();
        ~Radio ();

        void
        handshake ();

        void 
        playurl (std::string const& url);
  
        XSPF::Playlist
        get_xspf_playlist ();

        const Session&
        session () const;

        bool
        connected () const;

      private:

        Signal signal_disconnected_;
        Signal signal_connected_;
    
      public:

        Signal&
        signal_disconnected()
        {
          return signal_disconnected_;
        }

        Signal&
        signal_connected()
        {
          return signal_connected_;
        }

      private:

        void handshake_cb (char const * data, guint size, guint code);

        Soup::RequestRefP m_handshake_request;
        Session           m_session;
        bool              m_connected;
    };
  }
}

namespace Bmp
{
  namespace LastFM
  {
    /** Client code for Scrobbler
     *
     * Bmp::LastFM::Scrobbler is a client implementation of Audioscrobbler Protocol 1.2,
     * (http://www.audioscrobbler.com), used by Last.FM (http://last.fm), a public,
     * free service that tracks your listening habits and provides you with statistics
     * and friends.
     *
     */
    class Scrobbler
    {
      private:

        Signal signal_disconnected_;
        Signal signal_connected_;
    
      public:

        Signal&
        signal_disconnected()
        {
          return signal_disconnected_;
        }

        Signal&
        signal_connected()
        {
          return signal_connected_;
        }

      public:

        Scrobbler ();
        ~Scrobbler ();

        RecMutex  m_live_mutex;
        Thread  * m_thread;
        LQM       m_queue;

        Mutex     m_lqm_process_lock;
        bool      m_lqm_process;

        Mutex     m_lqm_lock;
        Cond      m_lqm_cond;
        bool      m_lqm_push;
        bool      m_lqm_eof;

        void
        scrobble    (XSPF::Item     const& item);
        bool
        scrobble_idle (XSPF::Item     const& item);

        void
        scrobble    (TrackQueueItem const& item);
        bool
        scrobble_idle_tqi (TrackQueueItem const& item);

        void
        now_playing (TrackQueueItem const& item);

        void
        handshake   ();

        bool
        connected   () const;

      private:

        void
        now_playing_thread (TrackQueueItem const& item);

        void
        scrobble_thread ();

        void
        scrobble_thread_start ();

        void
        scrobble_thread_stop ();

        void
        mcs_queue_enable (MCS_CB_DEFAULT_SIGNATURE);

        enum LastFmHandshakeState
        {
          AS_HANDSHAKE_STATE_UNDEFINED  = 0,
          AS_HANDSHAKE_BANNED           = 1,
          AS_HANDSHAKE_BADAUTH          = 2,
          AS_HANDSHAKE_BADTIME          = 3,
          AS_HANDSHAKE_FAILED           = 4,
          AS_HANDSHAKE_STATE_OK         = 5,
        };

        struct HandshakeData
        {
          /** Session ID 
           */
          std::string id;

          /** URI for track submissions
           */
          std::string uri;

          /** Now-Playing URI 
           */
          std::string np;

          /** Used for FAILED reason and UPDATE url
           */
          std::string info;

          /** Password as calculated during handshake
           */
          std::string pwd;

          /** The handshake state 
           */
          int state;

          HandshakeData ()
          : state (AS_HANDSHAKE_STATE_UNDEFINED) {}
        };

        void load_lqm (std::string const& filename);
        void save_lqm (std::string const& filename, bool lock = true);

        void handshake_cb (char const * data, guint size, guint code);

        Soup::RequestRefP m_handshake_request;
        HandshakeData     m_handshake_data;
    };
  } // namespace LastFM
} // namespace Bmp

#endif // !BMP_LAST_FM_HH

