//  BMP
//  Copyright (C) 2003-2007 BMP <http://beep-media-player.org> 
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <gtkmm.h>
#include <glibmm/i18n.h>
#include <cstdlib>
#include <cstring>
#include <set>

#include <boost/format.hpp>
#include <boost/optional.hpp>

// BMP Audio
#include "audio/audio.hh"

// BMP Musicbrainz
#include "musicbrainz/mbxml-v2.hh"
#include "mb-tagger.hh"
#include "ld.hh"

#include "debug.hh"
#include "main.hh"
#include "paths.hh"
#include "stock.hh"
#include "util-file.hh"
#include "util.hh"
#include "ui-tools.hh"

#include "x_amazon.hh"
#include "x_library.hh"
#include "x_vfs.hh"

using namespace Glib;
using namespace Gtk;
using namespace std;
using namespace Bmp::MusicBrainzXml;

#define ACTION_SEARCH                 "action-update"
#define ACTION_MOVE_UP                "action-move-up"
#define ACTION_MOVE_DOWN              "action-move-down"
#define ACTION_REMOVE                 "action-remove"
#define ACTION_RELAYOUT_NONE          "action-relayout-none"
#define ACTION_RELAYOUT_TRACKNUMBERS  "action-relayout-tracknumber"
#define ACTION_RELAYOUT_MATCH         "action-relayout-levenshtein"
#define ACTION_USE_FILENAMES          "action-use-filenames"
#define ACTION_RESTORE                "action-restore"

#ifdef HAVE_OFA
#  define ACTION_SCAN_RELEASE         "action-scan-release"
#endif

namespace
{
  void
  cbe_set_sensitive (CheckButton *cb, ComboBoxEntry *cbe)
  {
      cbe->set_sensitive (cb->get_active());
  }

  void
  fcb_set_sensitive (CheckButton *cb, FileChooserButton *fcb)
  {
      fcb->set_sensitive (cb->get_active());
  }

  void
  set_cbox_sensitive (CheckButton *cb, ComboBoxEntry *cb_entry)
  {
    cb_entry->set_sensitive (cb->get_active());
  }

#ifdef HAVE_OFA
  const char *modify_menu =
  "<ui>"
  "   <menubar name='popup-modify'>"
  "   <menu action='dummy' name='menu-modify'>"
  "     <menuitem action='" ACTION_SCAN_RELEASE "'/>"
  "     <separator name='sep1'/>"
  "     <menuitem action='" ACTION_REMOVE "'/>"
  "     <separator name='sep2'/>"
  "     <menuitem action='" ACTION_RELAYOUT_NONE "'/>"
  "     <menuitem action='" ACTION_RELAYOUT_TRACKNUMBERS "'/>"
  "     <menuitem action='" ACTION_RELAYOUT_MATCH "'/>"
  "     <separator name='sep3'/>"
  "     <menuitem action='" ACTION_USE_FILENAMES "'/>"
  "     <menuitem action='" ACTION_RESTORE "'/>"
  "   </menu>"
  "   </menubar>"
  "</ui>";
#else
  const char *modify_menu =
  "<ui>"
  "   <menubar name='popup-modify'>"
  "   <menu action='dummy' name='menu-modify'>"
  "     <menuitem action='" ACTION_REMOVE "'/>"
  "     <separator name='sep1'/>"
  "     <menuitem action='" ACTION_RELAYOUT_NONE "'/>"
  "     <menuitem action='" ACTION_RELAYOUT_TRACKNUMBERS "'/>"
  "     <menuitem action='" ACTION_RELAYOUT_MATCH "'/>"
  "     <separator name='sep2'/>"
  "     <menuitem action='" ACTION_USE_FILENAMES "'/>"
  "     <menuitem action='" ACTION_RESTORE "'/>"
  "   </menu>"
  "   </menubar>"
  "</ui>";
#endif //HAVE_OFA

  static boost::format uint64_t_f ("%lld");
}

using namespace std;
using namespace Gdk;
using namespace Gtk;
using namespace Glib;
using namespace Bmp::DB;
using namespace Bmp::MusicBrainzXml;

namespace Bmp
{
    MusicBrainzTagger::MusicBrainzTagger (BaseObjectType                 * obj,
                                          RefPtr<Gnome::Glade::Xml> const& xml)

    : Dialog              (obj)
    , m_ref_xml           (xml)
    , m_current_ordering  (ORDERING_AUTO)
#ifdef HAVE_OFA
    , m_processed         (0)
#endif //HAVE_OFA
    {
        set_title (_("Edit Metadata - BMP"));

        m_ref_xml->get_widget ("button-search", b_search);
        m_ref_xml->get_widget ("button-up", b_up);
        m_ref_xml->get_widget ("button-down", b_down);

        m_ref_xml->get_widget ("combo-box-artist", m_cb_artist);
        m_ref_xml->get_widget ("combo-box-album", m_cb_album);
        m_ref_xml->get_widget ("combo-box-genre", m_cb_genre);
        m_ref_xml->get_widget ("combo-box-release-date", m_cb_release_date);

        m_cb_artist_store = ListStore::create (m_cr_artists);
        m_cb_album_store = ListStore::create (m_cr_albums);

        m_cb_release_event_store = ListStore::create (m_cr_events);
        CellRendererText * cell = manage (new CellRendererText());
        m_cb_release_date->pack_start (*cell, false);
        m_cb_release_date->add_attribute (*cell, "text", m_cr_events.event_string);

        m_cb_artist->set_model (m_cb_artist_store);
        m_cb_album->set_model (m_cb_album_store);
        m_cb_release_date->set_model (m_cb_release_event_store);

        m_cb_artist->signal_changed().connect
          (sigc::mem_fun (*this, &Bmp::MusicBrainzTagger::on_artist_changed));
        m_cb_artist->set_text_column (0);

        m_cb_album->signal_changed().connect
          (sigc::mem_fun (*this, &Bmp::MusicBrainzTagger::on_album_changed));
        m_cb_album->set_text_column (0);

        m_ref_xml->get_widget ("entry-mb-album-id", m_entry_album_id);
        m_ref_xml->get_widget ("image-coverart", m_image);

        m_ref_xml->get_widget ("check-button-modify-genre", m_cb_modify_genre);
        m_cb_genre_entry = m_cb_genre->get_entry ();

        m_cb_modify_genre->signal_toggled().connect
          (sigc::bind (sigc::ptr_fun (&set_cbox_sensitive), m_cb_modify_genre, m_cb_genre));

        m_ref_xml->get_widget ("release-view", m_view_release);
        m_ref_xml->get_widget ("local-view", m_view_local);

        m_ui_manager = UIManager::create ();
        m_actions = ActionGroup::create ("Actions_LibraryModify");

        m_actions->add (Action::create ("dummy", "dummy"));
        m_actions->add (Action::create (ACTION_SEARCH,
                                             Stock::NETWORK,
                                             _("Search")),
                                             sigc::mem_fun (*this, &MusicBrainzTagger::on_action_search));
        m_actions->add (Action::create (ACTION_MOVE_UP,
                                             Stock::GO_UP,
                                             _("Move Up")),
                                             sigc::mem_fun (*this, &MusicBrainzTagger::on_action_up));
        m_actions->add (Action::create (ACTION_MOVE_DOWN,
                                             Stock::GO_DOWN,
                                             _("Move Down")),
                                             sigc::mem_fun (*this, &MusicBrainzTagger::on_action_down));

        m_actions->get_action (ACTION_SEARCH)->connect_proxy (*b_search);
        m_actions->get_action (ACTION_MOVE_UP)->connect_proxy (*b_up);
        m_actions->get_action (ACTION_MOVE_DOWN)->connect_proxy (*b_down);
        m_actions->get_action (ACTION_MOVE_UP)->set_sensitive (0);
        m_actions->get_action (ACTION_MOVE_DOWN)->set_sensitive (0);

        m_actions->add (Action::create (ACTION_REMOVE,
                                             StockID (BMP_STOCK_DELETE),
                                             _("Remove Track From List")),
                                             sigc::mem_fun (*this, &MusicBrainzTagger::on_action_remove));

        m_actions->add (Action::create (ACTION_RELAYOUT_NONE,
                                             StockID (BMP_STOCK_SCAN),
                                             _("Re-Layout Tracks Using No Ordering")),
                                             sigc::bind (sigc::mem_fun (*this, &MusicBrainzTagger::on_action_relayout),
                                             ORDERING_NONE));

        m_actions->add (Action::create (ACTION_RELAYOUT_TRACKNUMBERS,
                                             StockID (BMP_STOCK_SCAN),
                                             _("Re-Layout Tracks Using Tracknumbers")),
                                             sigc::bind (sigc::mem_fun (*this, &MusicBrainzTagger::on_action_relayout),
                                             ORDERING_TRACKNUMBERS));

        m_actions->add (Action::create (ACTION_RELAYOUT_MATCH,
                                             StockID (BMP_STOCK_SCAN),
                                             _("Re-Layout Tracks Using Name Matching")),
                                             sigc::bind (sigc::mem_fun (*this, &MusicBrainzTagger::on_action_relayout),
                                             ORDERING_MATCH));

        m_actions->add (Action::create (ACTION_RESTORE,
                                             Stock::REFRESH,
                                             _("Restore Original Track List")),
                                             sigc::mem_fun (*this, &MusicBrainzTagger::on_action_restore));

#ifdef HAVE_OFA
        m_actions->add (Action::create (ACTION_SCAN_RELEASE,
                                             Stock::FIND,
                                             _("Find Release based on Selected File")),
                                             AccelKey (""),
                                             sigc::mem_fun (*this, &MusicBrainzTagger::on_action_find));

        m_actions->get_action (ACTION_SCAN_RELEASE)->set_sensitive (false);
#endif //HAVE_OFA

        m_actions->add (ToggleAction::create (ACTION_USE_FILENAMES,
                                                   Stock::FILE,
                                                   _("Display/Use Filenames Rather than Track Names")),
                                                   sigc::mem_fun (*this, &MusicBrainzTagger::on_action_use_filenames_toggled));

        m_action_use_filenames = RefPtr<ToggleAction>::cast_static (m_actions->get_action (ACTION_USE_FILENAMES));

        m_ui_manager->insert_action_group (m_actions);
        m_ui_manager->add_ui_from_string (modify_menu);

        m_view_local->signal_event().connect
          (sigc::mem_fun (*this, &Bmp::MusicBrainzTagger::popup_rhs_menu));

        m_cb_artist_entry = m_cb_artist->get_entry ();
        m_cb_album_entry  = m_cb_album->get_entry ();

        m_cb_artist_entry->set_text ("");
        m_cb_album_entry->set_text ("");

        m_image->clear ();

        struct {
            char const* title;
            double      xalign;
            int         width;
        } const columns[] = {
            { N_("#"),     0.5, 0,  },
            { N_("Title"), 0.0, 100 },
        };

        // Release
        {
          for (unsigned int n = 0; n < G_N_ELEMENTS (columns); ++n)
          {
            CellRendererText * cell = manage (new CellRendererText());
            cell->property_xalign() = columns[n].xalign;
            m_view_release->append_column (_(columns[n].title), *cell);
            TreeViewColumn * column = m_view_release->get_column (n);
            column->add_attribute (*cell, "text", n);
            column->set_resizable (true);
          }
          m_store_release = ListStore::create (m_cr_tracks_release);
          m_view_release->set_model (m_store_release);
          m_view_release->get_selection()->set_mode (SELECTION_NONE);
        }

        // Local
        {
          {
            CellRendererToggle * cell = manage (new CellRendererToggle());
            m_view_local->append_column ("", *cell);
            TreeViewColumn *column = m_view_local->get_column (0);
            column->add_attribute (*cell, "active", 0);
            column->set_resizable (false);
            cell->signal_toggled().connect
              (sigc::mem_fun (*this, &Bmp::MusicBrainzTagger::on_modify_toggled));
          }

          {
            CellRendererText * cell = manage (new CellRendererText());
            cell->property_xalign() = 1.0;
            m_view_local->append_column ("#", *cell);
            TreeViewColumn * column = m_view_local->get_column (1);

            column->set_cell_data_func (*cell,
              (sigc::bind (sigc::mem_fun (*this, &Bmp::MusicBrainzTagger::cell_data_func), COLUMN_NUMBER)));

            cell->property_editable () = true;
            cell->signal_edited().connect
              (sigc::mem_fun (*this, &Bmp::MusicBrainzTagger::on_track_cell_edited));
          }

          {
            CellRendererText *cell = manage (new CellRendererText());
            cell->property_xalign() = 0.0;
            m_view_local->append_column (_("Title"), *cell);
            TreeViewColumn * column = m_view_local->get_column (2);
            column->add_attribute (*cell, "text", 2);
          }

  #ifdef HAVE_OFA
          {
            CellRendererProgress * cell = manage (new CellRendererProgress());
            m_view_local->append_column ("", *cell);
            TreeViewColumn * column = m_view_local->get_column (3);
            column->add_attribute (*cell, "value", m_cr_tracks_local.puid_progress);
            column->set_expand (false);
            column->set_visible (false);
          }
  #endif //HAVE_OFA

          m_store_local = ListStore::create (m_cr_tracks_local);
          m_view_local->set_model (m_store_local);
          m_view_local->get_selection()->set_mode (SELECTION_SINGLE);
          m_view_local->get_selection()->signal_changed().connect
            (sigc::mem_fun (*this, &Bmp::MusicBrainzTagger::on_local_selection_changed));
        }

        set_response_sensitive (RESPONSE_OK, false);
    }

    MusicBrainzTagger*
    MusicBrainzTagger::Create ()
    {
      std::string const path (BMP_GLADE_DIR G_DIR_SEPARATOR_S "library-ui-modify-import.glade");
      RefPtr<Gnome::Glade::Xml> xml = Gnome::Glade::Xml::create (path);
      MusicBrainzTagger * p = 0;
      xml->get_widget_derived ("dialog-window", p);
      return p;
    }

#ifdef HAVE_OFA
    void
    MusicBrainzTagger::set_processed ()
    {
      m_processed = true;
    }
#endif

    void
    MusicBrainzTagger::on_track_cell_edited (ustring const& path_1, ustring const& new_text)
    {
      TreeNodeChildren::size_type n_target (g_ascii_strtoll (new_text.c_str(), NULL, 10));
      TreeNodeChildren::size_type n_children (m_store_local->children().size());
      TreeNodeChildren::size_type n_children_release (m_store_local->children().size());

      if( (n_children < n_target) && (n_target <= n_children_release) )
      {
        for (TreeNodeChildren::size_type i = 0; i < ((n_target) - n_children); ++i)
        {
          m_store_local->append ();
        }
      }

      TreeIter iter1 (m_store_local->get_iter (path_1));
      TreeIter iter2 (m_store_local->get_iter (TreeModel::Path (1, n_target-1)));

      if( ! bool ((*iter2)[m_cr_tracks_local.has_track]) )
      {
        (*iter1)[m_cr_tracks_local.tracknumber] = n_target;
        m_store_local->iter_swap (iter1, iter2);
      }
    }

    void
    MusicBrainzTagger::cell_data_func (CellRenderer * basecell, TreeIter const& i, ColumnId column)
    {
      CellRendererText * cell_t = 0; 

      uint64_t n ((*i)[m_cr_tracks_local.tracknumber]);

      switch (column)
      {
        case COLUMN_NUMBER:
          cell_t = reinterpret_cast<CellRendererText *>(basecell);
          cell_t->property_text() = ((n == 0) ? "" : (uint64_t_f % n).str().c_str());
          break;
      }
    }

    void
    MusicBrainzTagger::on_modify_toggled (ustring const& path)
    {
      if( m_store_release->get_iter (path) )
      { 
        TreeIter i (m_store_local->get_iter (path));
        (*i)[m_cr_tracks_local.modify] = !(*i)[m_cr_tracks_local.modify];
      }
    }

    void
    MusicBrainzTagger::on_local_selection_changed ()
    {
      bool selected (m_view_local->get_selection()->count_selected_rows () != 0);

      m_actions->get_action (ACTION_MOVE_UP)->set_sensitive (selected);
      m_actions->get_action (ACTION_MOVE_DOWN)->set_sensitive (selected);
#ifdef HAVE_OFA
      m_actions->get_action (ACTION_SCAN_RELEASE)->set_sensitive (selected);
#endif //HAVE_OFA
    }

    void
    MusicBrainzTagger::on_action_remove ()
    {
      TreeIter iter = m_view_local->get_selection()->get_selected ();
      TreePath path = m_store_local->get_path (iter);
      std::string location = Track ((*iter)[m_cr_tracks_local.local_track]).location.get();

      m_store_local->erase (iter);
      m_store_local->insert (m_store_local->get_iter (path));

      for (TrackV::iterator i = m_tracks.begin(); i != m_tracks.end(); ++i)
      {
        if (i->location.get() == location) 
        {
          m_tracks.erase (i);
          break;
        }
      }
    }

    void
    MusicBrainzTagger::on_action_relayout (OrderingType ordering_type)
    {
      populate_local_store (ordering_type);
    }

    void
    MusicBrainzTagger::on_action_restore ()
    {
      m_tracks = m_tracks_original;
      populate_local_store (m_current_ordering);
    }

    void
    MusicBrainzTagger::on_action_up ()
    {
      TreeIter i (m_view_local->get_selection()->get_selected ());
      if( i == m_store_local->children().begin() )
        return;

      (*i)[m_cr_tracks_local.tracknumber] = 0; 
      TreeIter d_iter = i;
      d_iter--; // yes, this is slow, but it doesn't matter much as a release doesn't have that many tracks
      m_store_local->iter_swap (i, d_iter);
      
    }

    void
    MusicBrainzTagger::on_action_down ()
    {
      TreeIter i (m_view_local->get_selection()->get_selected ());
      if( i == m_store_local->children().end() )
        return;

      (*i)[m_cr_tracks_local.tracknumber] = 0; 
      TreeIter d_iter = i;
      d_iter++;
      m_store_local->iter_swap (i, d_iter);
    }

    void
    MusicBrainzTagger::populate_matches (MusicBrainzReleaseV const& x)
    {
      m_cb_artist_entry->set_text ("");
      m_cb_album_entry->set_text ("");

      m_image->clear ();
      m_artist_to_releases_map.clear ();

      m_store_release->clear ();
      m_store_local->clear ();
      m_cb_artist_store->clear (); 
      m_cb_album_store->clear (); 

      for (MusicBrainzReleaseV::const_iterator i = x.begin (); i != x.end(); ++i)
      {
        MusicBrainzRelease const& release = (*i);
        m_releases.insert (make_pair (release.releaseId, release));
        
        ArtistIdToReleasesMap::iterator map_iter (m_artist_to_releases_map.find (release.mArtist.artistId));
        if( map_iter != m_artist_to_releases_map.end () )
        {
          map_iter->second.push_back (release.releaseId);
        }
        else
        {
          TreeIter iter = m_cb_artist_store->append ();
          (*iter)[m_cr_artists.name] = release.mArtist.artistName;
          (*iter)[m_cr_artists.id]   = release.mArtist.artistId;

          UStrV id_v;
          id_v.push_back (release.releaseId);
          m_artist_to_releases_map.insert (make_pair (release.mArtist.artistId, id_v));
        }
      }

      m_cb_artist->set_model (m_cb_artist_store);
      m_cb_artist->set_active (0);

      m_cb_album->set_model (m_cb_album_store);
      m_cb_album->set_active (0);

      set_response_sensitive (RESPONSE_OK, m_store_release->children().size());
    }

    bool
    MusicBrainzTagger::popup_rhs_menu (GdkEvent * e)
    {
      GdkEventButton * event = 0;
      if( e->type == GDK_BUTTON_PRESS )
      {
        event = reinterpret_cast<GdkEventButton *>(e);
        if( event->button == 3 )
        {
          m_actions->get_action (ACTION_REMOVE)->set_sensitive (m_view_local->get_selection()->count_selected_rows());
          m_actions->get_action (ACTION_RELAYOUT_MATCH)->set_sensitive (m_store_release->children().size());

          Menu * menu = dynamic_cast<Menu *>(Util::get_popup (m_ui_manager, "/popup-modify/menu-modify"));
          if (menu) // better safe than screwed
          {
            menu->popup (event->button, event->time);
          }
          return true;
        }
      }
      return false;
    }

#ifdef HAVE_OFA
    void
    MusicBrainzTagger::on_action_find ()
    {
      scan_action_begin ();

      TreeIter i (m_view_local->get_selection()->get_selected ());
      boost::optional <ustring> puid (scan_file (i));

      if( puid )
      {
        MusicBrainzTracklistTrackV tracks;
        mb_tracks_by_puid (puid.get(), tracks);
        MusicBrainzReleaseV releases;
        for (MusicBrainzTracklistTrackV::const_iterator i = tracks.begin(); i != tracks.end(); ++i)
        {
          if( !i->mReleaseV.empty() )
          {
            for (MusicBrainzReleaseV::const_iterator n = i->mReleaseV.begin(); n != i->mReleaseV.end(); ++n)
            {
              mb_releases_by_id (n->releaseId, releases);
            }
          }
        }
        populate_matches (releases);
      }
      scan_action_end ();
    }

    void
    MusicBrainzTagger::scan_action_begin ()
    {
      set_sensitive (false);
      m_view_local->get_column (3)->set_visible (true);
    }
    
    void
    MusicBrainzTagger::scan_action_end ()
    {
      set_sensitive (true);
      m_view_local->get_column (3)->set_visible (false);
    }
    
    boost::optional <ustring>
    MusicBrainzTagger::scan_file (TreeIter & i)
    {
      Bmp::Audio::ProcessorPUID * p = new Bmp::Audio::ProcessorPUID;

      m_processed = false;

      p->signal_eos().connect
        (sigc::mem_fun (*this, &Bmp::MusicBrainzTagger::set_processed));
      p->signal_position().connect
        (sigc::bind (sigc::mem_fun (*this, &Bmp::MusicBrainzTagger::set_iter_puid_progress), i));

      Track const& local_track = (*i)[m_cr_tracks_local.local_track];
      p->set_uri (local_track.location.get());
      p->run ();

      while (!m_processed)
        while (Main::events_pending()) Main::iteration ();

      boost::optional <ustring> puid (p->get_puid ());
      delete p;
      return puid;
    }

    void
    MusicBrainzTagger::set_iter_puid_progress (int position, TreeIter & i)
    {
      (*i)[m_cr_tracks_local.puid_progress] = int (((1.*position) / 135.) * 100);
      m_store_local->row_changed (TreePath (i), i);
    }
#endif //HAVE_OFA

    void
    MusicBrainzTagger::on_action_search ()
    {
      MusicBrainzReleaseV releases;

      if( !m_entry_album_id->get_text().empty() && m_cb_artist_entry->get_text().empty() &&  m_cb_album_entry->get_text ().empty() )
      {
        mb_releases_by_id (m_entry_album_id->get_text(), releases);
      }
      else
      {
        mb_releases_query (m_cb_artist_entry->get_text(), m_cb_album_entry->get_text(), releases);
      }

      populate_matches (releases);
      populate_local_store (ORDERING_AUTO);
    }

    void
    MusicBrainzTagger::on_action_use_filenames_toggled ()
    {
      if( m_action_use_filenames->get_active() )
      {
        m_actions->get_action (ACTION_RELAYOUT_TRACKNUMBERS)->set_sensitive (false);
        m_current_ordering = ORDERING_MATCH;
        populate_local_store (m_current_ordering);
      }
      else
      {
        m_actions->get_action (ACTION_RELAYOUT_TRACKNUMBERS)->set_sensitive (true);
        m_current_ordering = ORDERING_AUTO;
        populate_local_store (m_current_ordering);
      }
    }

    void
    MusicBrainzTagger::populate_local_store (OrderingType requested_ordering_type)
    {
      TreeModel::Children const&  nodes_r (m_store_release->children());
      std::set <unsigned int>     tracknumbers;
      OrderingType                ordering_type;
      unsigned int                n_rows;

      m_current_ordering = requested_ordering_type;

      if( m_release.mTrackV.size() > m_tracks.size() )
        n_rows = m_release.mTrackV.size();
      else
        n_rows = m_tracks.size();

      if( !nodes_r )
      {
        // don't order by track number or levenshtein matching when there is no release selected
        requested_ordering_type = ORDERING_NONE;
      }

      m_store_local->clear();

      for (unsigned int n = 0 ; n < n_rows; ++n)
      {
        TreeIter iter = m_store_local->append ();

        (*iter)[m_cr_tracks_local.tracknumber] = 0;
        (*iter)[m_cr_tracks_local.has_track] = 0;
        (*iter)[m_cr_tracks_local.modify] = 0;
        (*iter)[m_cr_tracks_local.title] = ustring (); 
#ifdef HAVE_OFA
        (*iter)[m_cr_tracks_local.puid_progress] = 0; 
#endif //HAVE_OFA
      }

      if( requested_ordering_type == ORDERING_AUTO )
      {

        if( m_action_use_filenames->get_active() )

            ordering_type = ORDERING_MATCH;

        else
        {
            ordering_type = ORDERING_TRACKNUMBERS;

          for (TrackV::const_iterator i = m_tracks.begin (); i != m_tracks.end (); ++i)
          {
                Track const& track = *i;
                if( !track.tracknumber || !track.tracknumber.get() || (tracknumbers.find (track.tracknumber.get()) != tracknumbers.end()) )
                {
                      ordering_type = ORDERING_MATCH;
                      break;
                }
                tracknumbers.insert (track.tracknumber.get());
          }
        }

      }
      else
      {
            ordering_type = requested_ordering_type;
      }

      TreePath::value_type path_index = 0;
      for (TrackV::const_iterator i = m_tracks.begin (); i != m_tracks.end(); ++i)
      {
        Track const& track = *i;
        int index = 0; 

        switch (ordering_type)
        {
          case ORDERING_MATCH:
          {
                ustring::size_type d = ustring::npos;
                for (TreeNodeChildren::const_iterator n = nodes_r.begin(); n != nodes_r.end(); ++n)
                {
                      ustring::size_type d_calc = ld_distance<ustring> (track.title.get (), (*n)[m_cr_tracks_release.title]);
                      if( d_calc < d )
                      {
                          d = d_calc;
                          index = m_store_release->get_path (n).get_indices().data()[0];
                      }
                }
                break;
              }

          case ORDERING_TRACKNUMBERS:
          {
                index = track.tracknumber.get() - 1;
                break;
          }

          case ORDERING_NONE:
          {
                index = path_index++;
                break;
          }

          default: break;
        }

        TreePath path (TreePath::size_type (1), TreePath::value_type (index));

        TreeIter iter_l (m_store_local->get_iter (path));
        (*iter_l)[m_cr_tracks_local.local_track] = track;
        (*iter_l)[m_cr_tracks_local.has_track] = 1;
        (*iter_l)[m_cr_tracks_local.modify] = bool (m_store_release->get_iter (path));

        if( m_action_use_filenames->get_active() )
        {
          (*iter_l)[m_cr_tracks_local.tracknumber] = 0;
          (*iter_l)[m_cr_tracks_local.title] = path_get_basename (filename_from_uri (track.location.get()));
        }
        else
        {
          (*iter_l)[m_cr_tracks_local.tracknumber] = track.tracknumber ? track.tracknumber.get() : 0;
          (*iter_l)[m_cr_tracks_local.title] = track.title ? track.title.get() : path_get_basename (filename_from_uri (track.location.get()));
        }
      }
      m_view_local->columns_autosize ();
    }

    void
    MusicBrainzTagger::on_album_changed ()
    {
      if( m_cb_album->get_active_row_number () >= 0 )
      {
        m_store_release->clear ();

        MusicBrainzReleaseV v;
        mb_releases_by_id (m_releases.find ((*m_cb_album->get_active())[m_cr_albums.id])->second.releaseId, v);
        m_release = v[0];
        m_entry_album_id->set_text (m_release.releaseId);

        m_image->clear ();
        try{
            if( !m_release.releaseASIN.empty() )
            {
              RefPtr <Pixbuf> cover;
              amazon->fetch (m_release.releaseASIN, cover);
              m_image->set (cover->scale_simple (128, 128, INTERP_BILINEAR));
            }
          }
        catch (...)
          {
          }

        MusicBrainzTrackV const& x (m_release.mTrackV);
        for (MusicBrainzTrackV::const_iterator i = x.begin(); i != x.end() ; ++i)
        {
              TreeIter iter = m_store_release->append();
              (*iter)[m_cr_tracks_release.tracknumber] = i->trackTrackNumber;
              (*iter)[m_cr_tracks_release.title]       = i->trackTitle;
              (*iter)[m_cr_tracks_release.track]       = (*i);
        }

        m_cb_release_date->set_sensitive (0);
        m_cb_release_event_store->clear ();

        if( m_release.mReleaseEventV.size() )
        {
          for (MusicBrainzReleaseEventV::const_iterator i = m_release.mReleaseEventV.begin(); i != m_release.mReleaseEventV.end(); ++i)
          {
            g_message ("%s: Appending date...", G_STRLOC);

            TreeIter iter = m_cb_release_event_store->append ();
            if (! i->releaseEventCountry.empty())
              (*iter)[m_cr_events.event_string] = (boost::format ("%s (%s)") % i->releaseEventDate % i->releaseEventCountry).str();
            else
              (*iter)[m_cr_events.event_string] = i->releaseEventDate; 
          }
          m_cb_release_date->set_active (0);
          m_cb_release_date->set_sensitive (1);
        }
        else
            g_message ("%s: No dates", G_STRLOC);

        m_view_release->columns_autosize ();
        populate_local_store (ORDERING_AUTO);
        set_response_sensitive (RESPONSE_OK, true);
      }
    }

    void
    MusicBrainzTagger::on_artist_changed ()
    {
      if( m_cb_artist->get_active_row_number () >= 0 )
      {
        m_cb_album_store->clear ();
        m_cb_album_entry->set_text ("");

        m_store_release->clear ();

        TreeIter const& i = m_cb_artist->get_active ();
        UStrV const& id_v = m_artist_to_releases_map.find ((*i)[m_cr_artists.id])->second;

        std::set<ustring> ReleaseSet;

        for (UStrV::const_iterator i  = id_v.begin(); i != id_v.end(); ++i)
        {
          MusicBrainzRelease const& release = m_releases.find (*i)->second;
          if( ReleaseSet.find (release.releaseId) == ReleaseSet.end() )
          {
            ReleaseSet.insert (release.releaseId);

            TreeIter iter = m_cb_album_store->append ();
            (*iter)[m_cr_albums.name] = release.releaseTitle;
            (*iter)[m_cr_albums.id] = release.releaseId;
          }
        }

        populate_local_store (ORDERING_AUTO);
        m_cb_album->set_active (0);
        set_response_sensitive (RESPONSE_OK, m_store_release->children().size());
      }
    }

    int
    MusicBrainzTagger::run (TrackV const& tracks, TrackV & modify_tracks)
    {
      m_tracks          = tracks;
      m_tracks_original = tracks;

      populate_local_store (m_current_ordering);

      int response = Dialog::run ();

      if( response == RESPONSE_OK )
      {
        TreeNodeChildren const& nodes_r (m_store_release->children());
        TreeNodeChildren const& nodes_l (m_store_local->children());

        TreeNodeChildren::const_iterator iter_r (nodes_r.begin());
        TreeNodeChildren::const_iterator iter_l (nodes_l.begin());

        int release_event_index = -1;
        if( m_release.mReleaseEventV.size() )
        {
           release_event_index = m_cb_release_date->get_active_row_number ();
        }

        for (; iter_r != nodes_r.end(), iter_l != nodes_l.end(); ++iter_r, ++iter_l)
        {
              if( (*iter_l)[m_cr_tracks_local.modify] )
              {
                    Track track ((*iter_l)[m_cr_tracks_local.local_track]);
                    MusicBrainzTrack track_mb ((*iter_r)[m_cr_tracks_release.track]);

                    imbue_album (m_release, track, release_event_index);
                    imbue_track (track_mb, track);

                    if( m_cb_modify_genre->get_active() )
                    {
                      track.genre = m_cb_genre_entry->get_text ();
                    }

                    modify_tracks.push_back (track);
              }

              if( (iter_l == nodes_l.end()) || (iter_r == nodes_r.end()) )
              {
                    break;
              }
        }
      }
      return response;
    }
}
