//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2007 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non GPL-compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <glibmm.h>

#include "mb-libxml2-sax-release.hh"

using namespace Glib;
using namespace std;
using namespace Bmp::MusicBrainzXml;

namespace
{
  enum ElementId
  {
    E_RELEASE,
    E_TITLE,
    E_TEXT_REPR,
    E_ARTIST,
    E_NAME,
    E_SORT_NAME,
    E_DISAMBIGUATION,
    E_LIFE_SPAN,
    E_DISC_LIST,
    E_DISC,
    E_RELATION_LIST,
    E_RELATION,
    E_TRACK_LIST,
    E_TRACK,
    E_ASIN,
    E_DURATION,
    E_RELEASE_EVENT_LIST,
    E_EVENT,
  };
}

#include "parser/libxml2-sax-base.hh"
using Bmp::XPath;

namespace
{
  Element_tag elements[] =
  {
    { "release"             },
    { "title"               },
    { "text-representation" },
    { "artist"              }, 
    { "name"                },
    { "sort-name"           },
    { "disambiguation"      },
    { "life-span"           },
    { "disc-list"           },
    { "disc"                },
    { "relation-list"       },
    { "relation"            },
    { "track-list"          },
    { "track"               },
    { "asin"                },
    { "duration"            },
    { "release-event-list"  },
    { "event"               },
  };

  struct MbReleaseParseContext : public ParseContextBase
  {
    MusicBrainzReleaseV &         mReleaseV;
    unsigned int                  tracknum;
    
    MbReleaseParseContext (MusicBrainzReleaseV & v) : ParseContextBase(), mReleaseV (v), tracknum (1)  {}
  };

#define DEFAULT_REFS \
  MbReleaseParseContext & context (static_cast<MbReleaseParseContext&>(_context)); \
  MusicBrainzRelease & r (*context.mReleaseV.rbegin()); \
  MusicBrainzArtist & a (r.mArtist);

  namespace Handlers
  {
    HANDLER(release)
    {
      static_cast<MbReleaseParseContext&>(_context).mReleaseV.push_back(MusicBrainzRelease());

      // NOTE we get DEFAULT_REFS only after adding the release
      DEFAULT_REFS

      r.releaseId = props["id"];
    }

    HANDLER(artist)
    {
      DEFAULT_REFS

      a.artistId = props["id"];
    }

    HANDLER(track_artist)
    {
      DEFAULT_REFS

      MusicBrainzTrack & t (*r.mTrackV.rbegin());
      t.artistId = props["id"];
    }

    HANDLER(track_list)
    {
      DEFAULT_REFS

      if (props.count ("count") != 0)
      {
        r.mTrackListCount = g_ascii_strtoull (props["count"].c_str(), NULL, 10);
      }

      if (props.count ("offset") != 0)
      {
        r.mTrackListOffset = g_ascii_strtoull (props["offset"].c_str(), NULL, 10);
      }
    }

    HANDLER(track)
    {
      DEFAULT_REFS

      r.mTrackV.push_back (MusicBrainzTrack());
      MusicBrainzTrack & t (*r.mTrackV.rbegin());
      t.trackId = props["id"];
    }

    HANDLER(disc_list)
    {
      DEFAULT_REFS

      if (props.count ("count") != 0)
      {
        r.mDiscListCount = g_ascii_strtoull (props["count"].c_str(), NULL, 10);
      }
    }

    HANDLER(disc)
    {
      DEFAULT_REFS

      r.mDiscV.push_back (MusicBrainzDisc (uint64_t (g_ascii_strtoull (props["sectors"].c_str(), NULL, 10)), props["id"]));
    }

    HANDLER(event)
    {
      DEFAULT_REFS

      r.mReleaseEventV.push_back (MusicBrainzReleaseEvent (props["date"], props["country"]));
    }
  }

  Handler_tag handlers[] = 
  {
    { XPath(elements[E_RELEASE]), 
      &Handlers::release      },
    { XPath(elements[E_RELEASE]) / XPath(elements[E_ARTIST]), 
      &Handlers::artist       },
    
    { XPath(elements[E_RELEASE]) / XPath(elements[E_DISC_LIST]), 
      &Handlers::disc_list    },
    { XPath(elements[E_RELEASE]) / XPath(elements[E_DISC_LIST]) / XPath(elements[E_DISC]), 
      &Handlers::disc         },
    
    { XPath(elements[E_RELEASE]) / XPath(elements[E_TRACK_LIST]), 
      &Handlers::track_list   },
    { XPath(elements[E_RELEASE]) / XPath(elements[E_TRACK_LIST]) / XPath(elements[E_TRACK]), 
      &Handlers::track        },
    { XPath(elements[E_RELEASE]) / XPath(elements[E_TRACK_LIST]) / XPath(elements[E_TRACK]) / XPath(elements[E_ARTIST]), 
      &Handlers::track_artist },

    { XPath(elements[E_RELEASE]) / XPath(elements[E_RELEASE_EVENT_LIST]) / XPath(elements[E_EVENT]), 
      &Handlers::event        },
  };
  //---------
  
  namespace HandlersEnd
  {
    HANDLER_END(track)
    {
      DEFAULT_REFS

      MusicBrainzTrack & t (*r.mTrackV.rbegin());

      if (t.artistId.empty())
        t.artistId = r.mArtist.artistId;

      if (t.artistName.empty())
        t.artistName = r.mArtist.artistName;

      if (t.artistSortName.empty())
        t.artistSortName = r.mArtist.artistSortName;

      t.trackTrackNumber = context.tracknum + r.mTrackListOffset;
      context.tracknum++;
    }
  };
  
  HandlerEnd_tag handlers_end[] =
  {
    { XPath(elements[E_RELEASE]) / XPath(elements[E_TRACK_LIST]) / XPath(elements[E_TRACK]), 
      &HandlersEnd::track  },
  };
  //---------

  namespace HandlersPCDATA
  {
    HANDLER_PCDATA(title)
    {
      DEFAULT_REFS

      r.releaseTitle += text;
    }

    HANDLER_PCDATA(track_title)
    {
      DEFAULT_REFS

      MusicBrainzTrack & t (*r.mTrackV.rbegin());
      t.trackTitle += text;
    }

    HANDLER_PCDATA(name)
    {
      DEFAULT_REFS

      a.artistName += text;
    }

    HANDLER_PCDATA(track_name)
    {
      DEFAULT_REFS

      MusicBrainzTrack & t (*r.mTrackV.rbegin());
      t.artistName += text;
    }

    HANDLER_PCDATA(sortname)
    {
      DEFAULT_REFS

      a.artistSortName += text;
    }

    HANDLER_PCDATA(track_sortname)
    {
      DEFAULT_REFS

      MusicBrainzTrack & t (*r.mTrackV.rbegin());
      t.artistSortName += text;
    }

    HANDLER_PCDATA(asin)
    {
      DEFAULT_REFS

      r.releaseASIN += text;
    }

    HANDLER_PCDATA(duration)
    {
      DEFAULT_REFS

      MusicBrainzTrack & t (*r.mTrackV.rbegin());
      t.trackDuration = uint64_t (g_ascii_strtoull (text.c_str(), NULL, 10));
    }

  }

  HandlerPCDATA_tag handlers_pcdata[] = 
  {
    { XPath(elements[E_RELEASE]) / XPath(elements[E_TITLE]), 
      &HandlersPCDATA::title          },
    { XPath(elements[E_RELEASE]) / XPath(elements[E_ARTIST]) / XPath(elements[E_NAME]), 
      &HandlersPCDATA::name           },
    { XPath(elements[E_RELEASE]) / XPath(elements[E_ARTIST]) / XPath(elements[E_SORT_NAME]), 
      &HandlersPCDATA::sortname       },
    { XPath(elements[E_RELEASE]) / XPath(elements[E_ASIN]), 
      &HandlersPCDATA::asin           },
    
    { XPath(elements[E_RELEASE]) / XPath(elements[E_TRACK_LIST]) / XPath(elements[E_TRACK]) / XPath(elements[E_TITLE]), 
      &HandlersPCDATA::track_title  },
    { XPath(elements[E_RELEASE]) / XPath(elements[E_TRACK_LIST]) / XPath(elements[E_TRACK]) / XPath(elements[E_ARTIST]) / XPath(elements[E_NAME]), 
      &HandlersPCDATA::track_name   },
    { XPath(elements[E_RELEASE]) / XPath(elements[E_TRACK_LIST]) / XPath(elements[E_TRACK]) / XPath(elements[E_ARTIST]) / XPath(elements[E_SORT_NAME]),
      &HandlersPCDATA::track_sortname },
    { XPath(elements[E_RELEASE]) / XPath(elements[E_TRACK_LIST]) / XPath(elements[E_TRACK]) / XPath(elements[E_DURATION]), 
      &HandlersPCDATA::duration       },
  };
  //---------

}

namespace Bmp
{
  namespace MusicBrainzXml
  {
    int mb_xml_release_parse (std::string const &data, MusicBrainzReleaseV & releases)
    {
      MbReleaseParseContext context (releases); 

      for (unsigned int n = 0; n < G_N_ELEMENTS(handlers); ++n)
       {
         context.mHandlers.insert (std::make_pair (handlers[n].elementId, handlers[n].handler));
       }
       
       // handler/end 
       for (unsigned int n = 0; n < G_N_ELEMENTS(handlers_end); ++n)
       {
         context.mHandlersEnd.insert (std::make_pair (handlers_end[n].elementId, handlers_end[n].handler));
       }
       
       // handler/pcdata
       for (unsigned int n = 0; n < G_N_ELEMENTS(handlers_pcdata); ++n)
       {
         context.mHandlersPCDATA.insert (std::make_pair (handlers_pcdata[n].elementId, handlers_pcdata[n].handler));
       }
       
       // name <-> id map 
       for (unsigned int n = 0; n < G_N_ELEMENTS(elements); ++n)
       {
         context.mNameIdMap.insert (std::make_pair (elements[n], ElementId (n)));
       }
       
       return SaxParserBase::xml_base_parse (data, context);
    }
  }
}
