//  BMPx - The Dumb Music Player
//  Copyright (C) 2007 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif // HAVE_CONFIG_H

#include <signal.h>

#include <cstdlib>
#include <cstring>
#include <cmath>
#include <iostream>
#include <sstream>
#include <fstream>
#include <map>

#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#include <glibmm/i18n.h>
#include <glib/gstdio.h>
#include <gtkmm.h>
#include <cairomm/cairomm.h>

#include <gst/gst.h>
#include <gst/gstelement.h>
#include "dialog-gsterror.hh"

#include <boost/algorithm/string.hpp>
#include <boost/format.hpp>
#include <boost/optional.hpp>
#include <mcs/mcs.h>

#undef BMP_PLUGIN_BUILD
#include "x_play.hh"

#include "x_amazon.hh"
#define USING_DBUS 1
#include "x_core.hh"
#undef USING_DBUS
#ifdef HAVE_HAL
#include "x_hal.hh"
#endif //HAVE_HAL
#include "x_lastfm.hh"
#include "x_library.hh"

#include "debug.hh"
#include "main.hh"
#include "network.hh"
#include "paths.hh"
#include "stock.hh"
#include "ui-tools.hh"
#include "uri++.hh"
#include "util.hh"
#include "util-string.hh"

#include "ui-part-cdda.hh"
#include "ui-part-jamendo.hh"
#include "ui-part-lastfm.hh"
#include "ui-part-library.hh"
#include "ui-part-playlist.hh"
#include "ui-part-podcasts.hh"
#include "ui-part-radio.hh"

#include "bmp/base-types.hh"
#include "dbus-marshalers.h"

#include "widgets/ccwidgets.hh"
#include "widgets/bmp_tray_icon.h"
#include "widgets/bmp_status_icon.h"
#include "widgets/button.hh"
#include "widgets/popup.hh"
#include "widgets/taskdialog.hh"

#include "musicbrainz/mb-utils.hh"

#include "dialog-about.hh"
#include "dialog-equalizer.hh"
#include "dialog-progress.hh"
#include "lastfm-recommend-dialog.hh"

#include "preferences.hh"
#include "shell.hh"

using namespace Glib;
using namespace Gtk;
using namespace std;
using namespace Bmp::Util;

#define PLAYER_SHELL_ACTION_PLAY                    "player-shell-action-play"
#define PLAYER_SHELL_ACTION_PAUSE                   "player-shell-action-pause"
#define PLAYER_SHELL_ACTION_PREV                    "player-shell-action-prev"
#define PLAYER_SHELL_ACTION_NEXT                    "player-shell-action-next"
#define PLAYER_SHELL_ACTION_STOP                    "player-shell-action-stop"

#define PLAYER_SHELL_ACTION_REPEAT                  "player-shell-action-repeat"
#define PLAYER_SHELL_ACTION_SHUFFLE                 "player-shell-action-shuffle"

#define PLAYER_SHELL_ACTION_TRACK_DETAILS           "player-shell-action-track-info"
#define PLAYER_SHELL_ACTION_EQ                      "player-shell-action-eq"

#define PLAYER_SHELL_ACTION_QUIT                    "player-shell-action-quit"
#define PLAYER_SHELL_ACTION_PREFS                   "player-shell-action-prefs"
#define PLAYER_SHELL_ACTION_ABOUT                   "player-shell-action-about"
#define PLAYER_SHELL_ACTION_STATUSBAR               "player-shell-action-show-statusbar"
#define PLAYER_SHELL_ACTION_COMPACT                 "player-shell-action-compact"
#define PLAYER_SHELL_ACTION_MENUBAR                 "player-shell-action-menubar"

#define PLAYER_SHELL_ACTION_LASTFM                  "player-shell-action-lastfm"
#define PLAYER_SHELL_ACTION_RECOMMEND_LAST_FM       "player-shell-action-recommend-lastfm"

#define PLAYER_SHELL_ACTION_BOOKMARKS               "player-shell-action-bookmarks"
#define PLAYER_SHELL_ACTION_CREATE_BOOKMARK         "player-shell-action-create-bookmark"

namespace Bmp
{
  Mcs::Bind * mcs_bind = 0;
}

namespace
{
  struct SourceInfo
  {
      const std::string name;
      const std::string icon;
      bool              netOnly;
      const std::string id;
      const std::string stock;
  } sources[] = {
    { N_("Library"),  "library.png",  false,  "library",  BMP_STOCK_LIBRARY   },
    { N_("Playlist"), "playlist.png", false,  "playlist", BMP_STOCK_PLAYLIST  },
    { N_("Radio"),    "radio.png",    true,   "radio",    BMP_STOCK_XIPH      },
    { N_("Last.fm"),  "lastfm.png",   true,   "lastfm",   BMP_STOCK_LASTFM    },
    { N_("Podcast"),  "podcasts.png", true,   "podcast",  BMP_STOCK_FEED      },
    { N_("Audio CD"), "audiocd.png",  false,  "audiocd",  BMP_STOCK_CDDA      },
    { N_("Jamendo"),  "jamendo.png",  true,   "jamendo",  BMP_STOCK_JAMENDO   }, 
  };
}

namespace
{
  static  boost::format time_f ("%02d:%02d");
  static  boost::format format_int ("%llu");

  char const * ui_main_menubar =
  "<ui>"
  ""
  "<menubar name='MenuBarMain'>"
  ""
  "   <menu action='MenuBmp'>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_PREFS "'/>"
  "     <separator/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_ABOUT "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_QUIT "'/>"
  "   </menu>"
  ""
  "   <menu action='MenuView'>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_STATUSBAR "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_MENUBAR "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_COMPACT "'/>"
  "   </menu>"
  ""
  "   <menu action='MenuTrack'>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_TRACK_DETAILS "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_RECOMMEND_LAST_FM "'/>"
  "   </menu>"
  ""
  "   <menu action='MenuPlayback'>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_EQ "'/>"
  "     <separator/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_REPEAT "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_SHUFFLE "'/>"
  "     <separator/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_PLAY "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_PAUSE "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_PREV "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_NEXT "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_STOP "'/>"
  "     <separator/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_LASTFM "'/>"
  "   </menu>"
  ""
  "   <menu action='MenuBookmarks'>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_CREATE_BOOKMARK "'/>"
  "     <separator/>"
  "     <menu action='MenuBookmarks-radio'/>"
  "     <menu action='MenuBookmarks-jamendo'/>"
  "   </menu>"
  ""
  "   <placeholder name='PlaceholderSource'/>"
  ""
  "</menubar>"
  ""
  "</ui>";

  char const * ui_main_menubar_offline =
  "<ui>"
  ""
  "<menubar name='MenuBarMain'>"
  ""
  "   <menu action='MenuBmp'>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_PREFS "'/>"
  "     <separator/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_ABOUT "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_QUIT "'/>"
  "   </menu>"
  ""
  "   <menu action='MenuView'>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_STATUSBAR "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_COMPACT "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_MENUBAR "'/>"
  "   </menu>"
  ""
  "   <menu action='MenuTrack'>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_TRACK_DETAILS "'/>"
  "   </menu>"
  ""
  "   <menu action='MenuPlayback'>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_EQ "'/>"
  "     <separator/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_REPEAT "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_SHUFFLE "'/>"
  "     <separator/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_PLAY "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_PAUSE "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_PREV "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_NEXT "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_STOP "'/>"
  "   </menu>"
  ""
  "   <menu action='MenuBookmarks'>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_CREATE_BOOKMARK "'/>"
  "     <separator/>"
  "     <menu action='MenuBookmarks-radio'/>"
  "     <menu action='MenuBookmarks-jamendo'/>"
  "   </menu>"
  ""
  "   <placeholder name='PlaceholderSource'/>"
  ""
  "</menubar>"
  ""
  "</ui>";


  char const * ui_tray_icon =
  "<ui>"
  " <menubar name='popup-tray'>"
  " <menu action='dummy' name='menu-tray'>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_PAUSE "'/>"
  "   <separator/>"
  "   <placeholder name='PlaceholderSourceContext'/>"
  "   <separator/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_TRACK_DETAILS "'/>"
  "   <separator/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_STOP "'/>"
  "   <separator/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_PREV "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_NEXT "'/>"
  "   <separator/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_QUIT "'/>"
  " </menu>"
  " </menubar>"
  "</ui>";

  const char * ui_black_bar_popup =
  "<ui>"
  "<menubar name='popup-shell'>"
  "   <menu action='dummy' name='menu-shell-context'>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_TRACK_DETAILS "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_RECOMMEND_LAST_FM "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_LASTFM "'/>"
  "   <separator/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_MENUBAR "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_COMPACT "'/>"
  "   <separator/>"
  "   <placeholder name='PlaceholderSourceContext'/>"
  "   </menu>"
  "</menubar>"
  "</ui>";

  const char * ui_black_bar_popup_offline =
  "<ui>"
  "<menubar name='popup-shell'>"
  "   <menu action='dummy' name='menu-shell-context'>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_TRACK_DETAILS "'/>"
  "   <separator/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_MENUBAR "'/>"
  "     <menuitem action='" PLAYER_SHELL_ACTION_COMPACT "'/>"
  "   <separator/>"
  "   <placeholder name='PlaceholderSourceContext'/>"
  "   </menu>"
  "</menubar>"
  "</ui>";

}

namespace Bmp
{
  enum LayoutID
  {
    L_TITLE,
    L_ARTIST,
    L_ALBUM,
    N_LAYOUTS
  };

  struct LayoutData 
  {
    double  alpha;
    int     x;
    int     y;

  //FIXME: Encode size here as well
  } layout_data[] = { 
    {-0.0, 108, 10},
    {-0.4, 108, 42},
    {-0.4, 108, 60}
  };

  class BlackBar
    : public EventBox
  {
    private:

      typedef std::vector<int> SpectrumHold;

      Spectrum      m_spectrum_data;
      Spectrum      m_spectrum_peak;
      SpectrumHold  m_spectrum_hold;

      struct Text
      {
        Text (Gtk::Widget & w, ustring const& text, LayoutID id)
          : m_alpha   (layout_data[id].alpha)
          , m_x       (layout_data[id].x)
          , m_y       (layout_data[id].y)
        {
          m_layout = w.create_pango_layout ("");
          m_layout->set_markup (text);
        }

        ~Text () {}

        RefPtr<Pango::Layout> m_layout;
        double                m_alpha;
        int                   m_x, m_y;
        sigc::connection      m_conn;
      };
  
      typedef boost::shared_ptr <Text>    TextP;
      typedef boost::shared_ptr <Mutex>   MutexP;

      typedef std::map <LayoutID, TextP>    Layouts;
      typedef std::map <LayoutID, MutexP>   LayoutsLocks;
    
      Layouts         m_layouts;
      LayoutsLocks    m_locks;
      Mutex     m_layouts_lock;
      ustring   m_text[N_LAYOUTS];

      RefPtr<Pango::Layout> m_layout_message;
      bool m_showing_message;
      
      Cairo::RefPtr<Cairo::ImageSurface>  m_surface;
      Mutex                               m_surface_lock;
      double                              m_surface_alpha;
      sigc::connection                    m_surface_conn;

      Cairo::RefPtr<Cairo::ImageSurface>  m_surface_frame;
      bool m_frame;

      RefPixbuf                           m_source_icon;
      bool m_compact;

    public:

      BlackBar (BaseObjectType                 * obj,
                RefPtr<Gnome::Glade::Xml> const& xml)
      : EventBox (obj)
      , m_showing_message (false)
      , m_frame (false)
      , m_source_icon (RefPixbuf(0))
      , m_compact (false)
      {
        modify_bg (Gtk::STATE_NORMAL, Gdk::Color ("#000000"));
        modify_base (Gtk::STATE_NORMAL, Gdk::Color ("#000000"));

        for (unsigned int n = 0; n < SPECT_BANDS; ++n)
        {
          m_spectrum_data.push_back (0);
          m_spectrum_hold.push_back (0);
          m_spectrum_peak.push_back (0);
        }

        ::play->signal_spectrum().connect (sigc::mem_fun (*this, &Bmp::BlackBar::play_update_spectrum));

        m_locks.insert (make_pair (L_TITLE, MutexP (new Mutex())));
        m_locks.insert (make_pair (L_ARTIST, MutexP (new Mutex())));
        m_locks.insert (make_pair (L_ALBUM, MutexP (new Mutex())));

        m_surface_frame = Util::cairo_image_surface_from_pixbuf (Gdk::Pixbuf::create_from_file (build_filename (BMP_IMAGE_DIR, "cover-frame.png")));
      }

      ~BlackBar ()
      {}

      void
      set_source (Glib::RefPtr<Gdk::Pixbuf> source_icon)
      {
        m_source_icon = source_icon;
        queue_draw ();
      }

      void
      set_compact (bool compact)
      {
        m_compact = compact;
      }

      void
      reset ()
      {
        Mutex::Lock L1 (m_layouts_lock);
        Mutex::Lock L2 (m_surface_lock);
        for (unsigned int n = 0; n < SPECT_BANDS; m_spectrum_data[n++] = 0); 
        for (unsigned int n = 0; n < SPECT_BANDS; m_spectrum_peak[n++] = 0); 
        for (unsigned int n = 0; n < SPECT_BANDS; m_spectrum_hold[n++] = 0); 
        remove_layout_if_exists (L_ARTIST);
        remove_layout_if_exists (L_ALBUM);
        remove_layout_if_exists (L_TITLE);
        m_surface_conn.disconnect ();
        m_surface = Cairo::RefPtr<Cairo::ImageSurface>(0);

        m_text[0] = ustring ();
        m_text[1] = ustring ();
        m_text[2] = ustring ();

        // Not calling clear_message as it locks
        m_showing_message = false;
        m_layout_message = RefPtr<Pango::Layout>(0);

        queue_draw ();
      }

      void
      set_message (ustring const& text)
      {
        Mutex::Lock L (m_layouts_lock);
        m_showing_message = true;
        m_layout_message = create_pango_layout ("");
        m_layout_message->set_markup ((boost::format ("<span size='xx-large'>%s</span>") % text.c_str()).str());
        queue_draw ();
      }

      void
      clear_message ()
      {
        Mutex::Lock L (m_layouts_lock);
        m_showing_message = false;
        m_layout_message = RefPtr<Pango::Layout>(0);
        queue_draw ();
      }

      void
      set_text (LayoutID id, ustring const& text)
      {
        Mutex::Lock L (m_layouts_lock);
        if (text != m_text[id])
        {
          m_text[id] = text;
          TextP p = TextP (new Text (*this, text, id));

          remove_layout_if_exists (id);
          insert_layout_and_connect (id, p);
        } 
      }

      void
      set_frame (bool frame)
      {
        Mutex::Lock L (m_surface_lock); 
        m_frame = frame;
        queue_draw ();
      }

      void
      set_image (RefPtr<Gdk::Pixbuf> pixbuf, bool draw_frame)
      {
        Mutex::Lock L (m_surface_lock); 
        m_frame = draw_frame;
        m_surface = Util::cairo_image_surface_from_pixbuf (pixbuf);
        m_surface_alpha = -0.5;
        m_surface_conn.disconnect ();
        m_surface_conn = signal_timeout().connect (sigc::mem_fun (*this, &Bmp::BlackBar::fade_in_surface), 20);
      }

      void
      set_image (Cairo::RefPtr<Cairo::ImageSurface> surface, bool draw_frame)
      {
        Mutex::Lock L (m_surface_lock); 
        m_frame = draw_frame;
        m_surface = surface; 
        m_surface_alpha = -0.5;
        m_surface_conn.disconnect ();
        m_surface_conn = signal_timeout().connect (sigc::mem_fun (*this, &Bmp::BlackBar::fade_in_surface), 20);
      }

    private:

      void
      insert_layout_and_connect (LayoutID id, TextP & p)
      {
        m_layouts.insert (make_pair (id, p)); 
        p->m_conn = signal_timeout().connect (sigc::bind (sigc::mem_fun (*this, &Bmp::BlackBar::fade_in), id), 20);
      }

      void
      remove_layout_if_exists (LayoutID id)
      {
        Layouts::iterator i = m_layouts.find (id);
        if (i != m_layouts.end())
        {
          TextP & p = i->second;
          m_locks.find (id)->second->lock ();
          p->m_conn.disconnect ();
          m_layouts.erase (i);
          m_locks.find (id)->second->unlock ();
        }
      }

      bool 
      fade_in (LayoutID id)
      {
        Mutex::Lock L (*(m_locks.find (id)->second.get()));
        TextP & p (m_layouts.find (id)->second);
        p->m_alpha += 0.1;
        bool r = p->m_alpha < 1.;
        queue_draw ();
        return r;
      }

      bool 
      fade_in_surface ()
      {
        Mutex::Lock L (m_surface_lock);
        m_surface_alpha += 0.1;
        bool r = m_surface_alpha < 1.;
        queue_draw ();
        return r;
      }

      void
      play_update_spectrum (Spectrum const& spectrum)
      {
        for (unsigned int n = 0; n < SPECT_BANDS; ++n) 
        {
          if (spectrum[n] < m_spectrum_data[n])
          {
            m_spectrum_data[n] = (((m_spectrum_data[n] - 6) < 0) ? 0 : (m_spectrum_data[n] - 6));
            ++(m_spectrum_hold[n]);
          }
          else
          {
            m_spectrum_data[n] = spectrum[n]; 
            m_spectrum_peak[n] = spectrum[n];
            m_spectrum_hold[n] = 0;
          }
        } 

        queue_draw ();
      }

    protected:

      virtual bool
      on_expose_event (GdkEventExpose * event)
      {
        Widget::on_expose_event (event);

        Cairo::RefPtr<Cairo::Context> cr = get_window ()->create_cairo_context ();
    
        m_layouts_lock.lock ();  
        if (!m_showing_message)
        {
          for (Layouts::const_iterator i = m_layouts.begin(); i != m_layouts.end(); ++i)
          {
            Mutex::Lock L (*(m_locks.find (i->first)->second.get()));
            TextP const& p (i->second);
            if (p->m_alpha < 0)
              continue;
            cr->set_source_rgba (1., 1., 1., p->m_alpha);
            cr->set_operator (Cairo::OPERATOR_ATOP);
            cr->move_to (p->m_x, p->m_y);
            p->m_layout->set_single_paragraph_mode (true);
            p->m_layout->set_ellipsize (Pango::ELLIPSIZE_END);
            p->m_layout->set_width ((get_allocation().get_width() - p->m_x - 210) * PANGO_SCALE);
            p->m_layout->set_wrap (Pango::WRAP_CHAR);
            pango_cairo_show_layout (cr->cobj(), p->m_layout->gobj());
          }
        }
        else
        {
          Pango::Rectangle r = m_layout_message->get_ink_extents();
          int w = r.get_width()/PANGO_SCALE, h = r.get_height()/PANGO_SCALE;
          cr->set_source_rgba (1., 1., 1., .9); 
          cr->set_operator (Cairo::OPERATOR_ATOP);
          cr->move_to ( ((get_allocation().get_width() - w)/2),
                        ((get_allocation().get_height() - h)/2));
          pango_cairo_show_layout (cr->cobj(), m_layout_message->gobj());
        }
        m_layouts_lock.unlock ();  

        m_surface_lock.lock ();
        if (m_surface && (m_surface_alpha >= 0))
        {
          cr->set_operator (Cairo::OPERATOR_SOURCE);
          cr->set_source (m_surface, 30, 12); 
          cr->rectangle (30, 12, 64, 64); 
          cr->save (); 
          cr->clip ();
          cr->paint_with_alpha (m_surface_alpha);
          cr->restore ();

          if (m_frame)
          {
            cr->set_operator (Cairo::OPERATOR_ATOP);
            cr->set_source (m_surface_frame, 26, 8);
            cr->rectangle (26, 8, 72, 72); 
            cr->save (); 
            cr->clip ();
            cr->paint_with_alpha (m_surface_alpha);
            cr->restore ();
          }
        }
        m_surface_lock.unlock ();

        for (unsigned int n = 0; n < SPECT_BANDS; ++n)
        {
          int x = 0, y = 0, w = 0, h = 0;

          // Bar
          x = (get_allocation().get_width ()) - 200 + (n*12);
          y = 10 + (64 - m_spectrum_data[n]); 
          w = 10;
          h = m_spectrum_data[n];

          cr->set_source_rgba (1., 1., 1., 0.3);
          cr->rectangle (x,y,w,h);
          cr->fill ();

          // Peak
          if ((m_spectrum_hold[n] < 5) && (m_spectrum_peak[n] > 0))
          {
            y = 10 + (64 - m_spectrum_peak[n]); 
            h = 2;
            cr->set_source_rgba (1., 1., 1., 0.55);
            cr->rectangle (x, y-2, w, h);
            cr->fill ();
          }
        }

        if (m_source_icon && m_compact)
        {
          cr->set_operator (Cairo::OPERATOR_ATOP);
          Gdk::Cairo::set_source_pixbuf (cr, m_source_icon, get_allocation().get_width() - 28, 12); 
          cr->rectangle (get_allocation().get_width() - 28, 12, 20, 20);
          cr->fill ();
        }

        return true;
      }
  };
}

namespace
{
  void
  parse_metadata (Bmp::TrackMetadata const &metadata,
                  ustring             &artist,
                  ustring             &album,
                  ustring             &title)
  {
    using namespace Bmp;

    static char const * 
      text_b_f ("<b>%s</b>");

    static char const * 
      text_b_f2 ("<b>%s</b> (%s)");

    static char const * 
      text_big_f ("<big>%s</big>");

    static char const * 
      text_album_artist_date_f ("%s (%s, <small>%s</small>)");

    static char const * 
      text_album_artist_f ("%s (%s)");

    static char const * 
      text_album_date_f ("%s (<small>%s</small>)");

    static char const * 
      text_album_f ("%s");

    static char const * 
      text_artist_date_f ("(%s, <small>%s</small>)");

    static char const * 
      text_artist_f ("(%s)");

    static char const *
      text_date_f ("(<small>%s</small>)");

    opt_stdstring date;

    if (metadata.mb_release_date) 
      date = Util::get_date_string (metadata.mb_release_date.get());
    else if (metadata.date && (metadata.date.get() != 0))
      date = (format_int % metadata.date.get()).str();

    if ((metadata.mb_album_artist_id != metadata.mb_artist_id) && metadata.mb_album_artist)
    {
      if ((date) && (date.get().size()))
      {
        if (metadata.album)
        {
          album = sql_uprintf (text_album_artist_date_f,
                               Markup::escape_text (metadata.album.get()).c_str(),
                               Markup::escape_text (metadata.mb_album_artist.get()).c_str(),
                               date.get().c_str());
        }
        else
        {
          album = sql_uprintf (text_artist_date_f,
                               Markup::escape_text (metadata.mb_album_artist.get()).c_str(),
                               date.get().c_str());
        }
      }
      else
      {
        if (metadata.album)
        {
          album = sql_uprintf (text_album_artist_f,
                               Markup::escape_text (metadata.album.get()).c_str(),
                               Markup::escape_text (metadata.mb_album_artist.get()).c_str());
        }
        else
        {
          album = sql_uprintf (text_artist_f,
                               Markup::escape_text (metadata.mb_album_artist.get()).c_str());
        }
      }
    }
    else
    {
      if ((date) && (date.get().size()))
      {
        if (metadata.album)
        {
          album = sql_uprintf (text_album_date_f,
                               Markup::escape_text (metadata.album.get()).c_str(), 
                               date.get().c_str());
        }
        else
        {
          album = sql_uprintf (text_date_f,
                               date.get().c_str());
        }
      }
      else
      {
        if (metadata.album)
        {
          album = sql_uprintf (text_album_f,
                               Markup::escape_text (metadata.album.get()).c_str());
        }
      }
    }

    if ((metadata.mb_artist_sort_name && metadata.artist) &&
        (metadata.mb_artist_sort_name != metadata.artist))
    {
      std::string a = metadata.mb_artist_sort_name.get(); 

      if (MusicBrainzUtil::reverse_sortname (a))
      {
        std::string b = metadata.artist.get(); 

        if (a == b)
        {
          artist = sql_uprintf (text_b_f, Markup::escape_text (metadata.artist.get()).c_str()); 
          goto artist_out;
        }
      }

      artist = sql_uprintf (text_b_f2,
                            Markup::escape_text (metadata.mb_artist_sort_name.get()).c_str(),
                            Markup::escape_text (metadata.artist.get()).c_str());
    }
    else
    if (metadata.artist)
    {
      artist = sql_uprintf (text_b_f, Markup::escape_text (metadata.artist.get()).c_str()); 
    }

    artist_out:

    if (metadata.title)
    {
      title = sql_uprintf (text_big_f, Markup::escape_text (metadata.title.get()).c_str());
    }
  }
}

namespace Bmp
{

#define TYPE_DBUS_OBJ_PLAYER (Player::get_type ())
#define DBUS_OBJ_PLAYER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), TYPE_DBUS_OBJ_PLAYER, Player))

  enum MPRISPlayerSignals
  {
    SIGNAL_TRACK_CHANGE,
    SIGNAL_STATUS_CHANGE,
    SIGNAL_CAPS_CHANGE,
  
    LAST_SIGNAL
  };

  namespace MPRISPlayer
  {
    static guint mpris_player_signals[LAST_SIGNAL] = { 0 };
  }

  class PlayerShell::Player;
  struct PlayerClass
  {
    GObjectClass parent;

    void  (* track_change  )  ( PlayerShell::Player * player, GHashTable*);
    void  (* status_change )  ( PlayerShell::Player * player, int status);
    void  (* caps_change   )  ( PlayerShell::Player * player, int caps);
  };

  struct PlayerShell::Player
  {
    GObject parent;
    PlayerShell * m_shell;

    static gpointer parent_class;

    static GType
    get_type ();

    static Player *
    create (PlayerShell & shell);

    static void
    class_init (gpointer klass,
                gpointer class_data);

    static GObject *
    constructor (GType                   type,
                 guint                   n_construct_properties,
                 GObjectConstructParam * construct_properties);

    //// Remote signal invocation
    
    static bool
    emit_track_change (Player * self);
    
    static void
    emit_status_change (Player * self, int status);

    static void
    emit_caps_change (Player * self, int caps);

    //// Remote methods

    static gboolean
    volume_set (Player * self,
                int            volume,
                GError **      error);

    static gboolean
    volume_get (Player * self,
                int *          volume,
                GError **      error);

    static gboolean
    next (Player * self,
          GError ** error);

    static gboolean
    prev (Player * self,
          GError ** error);

    static gboolean
    pause  (Player * self,
            GError ** error);

    static gboolean
    play   (Player * self,
            GError ** error);

    static gboolean
    stop   (Player * self,
            GError ** error);

    static gboolean
    get_metadata (Player *       self,
                  GHashTable**   metadata,
                  GError**       error);

    static gboolean
    play_last_fm (Player *       self,
                  char*          uri,
                  GError**       error);

    /// SessionPresence
    static void
    on_spm_status (DBusGProxy * proxy,
                   GValue * p1,
                   gpointer data);
  };

  gpointer PlayerShell::Player::parent_class = 0;

// HACK: Hackery to rename functions in glue
#define player_next                 next
#define player_prev                 prev
#define player_pause                pause
#define player_stop                 stop
#define player_play                 play
#define player_volume_set           volume_set 
#define player_volume_get           volume_get
#define player_get_metadata         get_metadata
#define player_play_last_fm         play_last_fm

#include "dbus-obj-player-glue.h"

  void
  PlayerShell::Player::class_init (gpointer klass,
                                   gpointer class_data)
  {
    parent_class = g_type_class_peek_parent (klass);

    GObjectClass *gobject_class = reinterpret_cast<GObjectClass*>(klass);
    gobject_class->constructor  = &Player::constructor;

    ::Bmp::MPRISPlayer::mpris_player_signals[SIGNAL_TRACK_CHANGE] =
      g_signal_new ("track-change",
                    G_OBJECT_CLASS_TYPE (klass),
                    G_SIGNAL_RUN_LAST,
                    G_STRUCT_OFFSET (PlayerClass, track_change),
                    NULL, NULL,
                    g_cclosure_marshal_VOID__BOXED,
                    G_TYPE_NONE, 1, dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, G_TYPE_VALUE)); 

    ::Bmp::MPRISPlayer::mpris_player_signals[SIGNAL_STATUS_CHANGE] =
      g_signal_new ("status-change",
                    G_OBJECT_CLASS_TYPE (klass),
                    G_SIGNAL_RUN_LAST,
                    G_STRUCT_OFFSET (PlayerClass, status_change),
                    NULL, NULL,
                    g_cclosure_marshal_VOID__INT,
                    G_TYPE_NONE, 1, G_TYPE_INT);

    ::Bmp::MPRISPlayer::mpris_player_signals[SIGNAL_CAPS_CHANGE] =
      g_signal_new ("caps-change",
                    G_OBJECT_CLASS_TYPE (klass),
                    G_SIGNAL_RUN_LAST,
                    G_STRUCT_OFFSET (PlayerClass, caps_change),
                    NULL, NULL,
                    g_cclosure_marshal_VOID__INT,
                    G_TYPE_NONE, 1, G_TYPE_INT);
  }

  GObject *
  PlayerShell::Player::constructor (GType                   type,
                                    guint                   n_construct_properties,
                                    GObjectConstructParam*  construct_properties)
  {
    GObject *object = G_OBJECT_CLASS (parent_class)->constructor (type, n_construct_properties, construct_properties);
    return object;
  }

  void
  PlayerShell::Player::on_spm_status (DBusGProxy * proxy,
                                      GValue * p1,
                                      gpointer data) 
  {
    if (::mcs->key_get <bool> ("bmp", "spm-listen"))
    {
      PlayerShell::Player & player = *(reinterpret_cast<PlayerShell::Player*>(data));

      GType target_type = dbus_g_type_get_struct ("GValueArray", G_TYPE_STRV, G_TYPE_STRING, G_TYPE_INVALID);
      g_return_if_fail (p1->g_type == target_type);

      GValue * v1 = g_value_array_get_nth ((GValueArray*)(g_value_get_boxed (p1)), 0);
      //GValue * v2 = g_value_array_get_nth ((GValueArray*)(g_value_get_boxed (p1)), 1);

      char ** strv = (char**)(g_value_get_boxed (v1));

      if (strv && strv[0])
      {
        if ((std::string (strv[0]) == "absent") && ::play->property_status() == PLAYSTATUS_PLAYING)
        {
          ::play->request_status (PLAYSTATUS_PAUSED);
          player.m_shell->m_spm_paused = true;
        }
        else if ((std::string (strv[0]) == "present") && ::play->property_status() == PLAYSTATUS_PAUSED && player.m_shell->m_spm_paused)
        {
          ::play->request_status (PLAYSTATUS_PLAYING);
        }
      }
    }

    g_value_unset (p1);
  }

  PlayerShell::Player *
  PlayerShell::Player::create (PlayerShell & shell)
  {
    dbus_g_object_type_install_info (TYPE_DBUS_OBJ_PLAYER, &dbus_glib_player_object_info);

    Player * self = DBUS_OBJ_PLAYER (g_object_new (TYPE_DBUS_OBJ_PLAYER, NULL));
    self->m_shell = &shell;

    if (::Bmp::DBus::dbus_connected)
    {
      dbus_connection_setup_with_g_main (dbus_g_connection_get_connection (::Bmp::DBus::dbus_session_bus), g_main_context_default());

      dbus_g_connection_register_g_object (::Bmp::DBus::dbus_session_bus, BMP_DBUS_PATH__MPRIS_PLAYER, G_OBJECT(self));
      g_message ("object: '/%s', interface '%s', service '%s'", G_OBJECT_TYPE_NAME(self), BMP_DBUS_INTERFACE__MPRIS, BMP_DBUS_SERVICE);

      // Connect SessionPresence stuff
      DBusGProxy * spm_proxy = dbus_g_proxy_new_for_name (::Bmp::DBus::dbus_session_bus, "com.weej.SessionPresenceManager", 
            "/com/weej/SessionPresenceManager", "com.weej.SessionPresenceManager");

      dbus_g_object_register_marshaller (bmpdbus_VOID__BOXED, G_TYPE_NONE, G_TYPE_VALUE, G_TYPE_INVALID);
      dbus_g_proxy_add_signal (spm_proxy, "StatusChanged", G_TYPE_VALUE, G_TYPE_INVALID);
      dbus_g_proxy_connect_signal (spm_proxy, "StatusChanged", G_CALLBACK (PlayerShell::Player::on_spm_status), self, NULL);
    }

    return self;
  }

  GType
  PlayerShell::Player::get_type ()
  {
    static GType type = 0;

    if (G_UNLIKELY (type == 0))
      {
        static GTypeInfo const type_info =
          {
            sizeof (PlayerClass),
            NULL,
            NULL,
            &class_init,
            NULL,
            NULL,
            sizeof (Player),
            0,
            NULL
          };

        type = g_type_register_static (G_TYPE_OBJECT, "Player", &type_info, GTypeFlags (0));
      }

    return type;
  }

  gboolean
  PlayerShell::Player::get_metadata (Player *       self,
                                     GHashTable**   metadata,
                                     GError**       error)
  {
    if (self->m_shell->m_selection->m_active_source == SOURCE_NONE)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "GetMetadata RMI: No Active Source"); 
      *error = g_error_new (g_quark_from_string ("org.mpris.bmp.DBusError"), 0, "No Active Source");
      return FALSE;
    }

    *metadata = self->m_shell->get_metadata_hash_table ();
    if (!*metadata)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "Invalid source specified: %d", self->m_shell->m_selection->m_active_source);
      *error = g_error_new (g_quark_from_string ("org.mpris.bmp.DBusError"), 0, "No Metadata Available");
      return FALSE;
    }

    return TRUE;
  }

  gboolean
  PlayerShell::Player::play_last_fm  (Player*   self,
                                      char*     uri, 
                                      GError**  error)
  {
    SourceSP & source (self->m_shell->m_sources[SOURCE_LASTFM]);
    if (source)
    {
      dynamic_cast <Bmp::UiPart::LASTFM*>(source.get())->set_uri (uri);
    }
    return TRUE;
  }

  gboolean
  PlayerShell::Player::volume_set (Player*   self,
                                   int      volume,
                                   GError** error)
  {
    mcs->key_set<int> ("bmp", "volume", volume);
    return TRUE;
  }

  gboolean
  PlayerShell::Player::volume_get (Player*   self,
                                   int*     volume,
                                   GError** error)
  {
    *volume = mcs->key_get<int> ("bmp", "volume");
    return TRUE;
  }

  gboolean
  PlayerShell::Player::next  (Player*     self,
                              GError**   error)
  {
    if (self->m_shell->m_selection->m_active_source == SOURCE_NONE)
      return TRUE;

    if (self->m_shell->m_source_caps[self->m_shell->m_selection->m_active_source] & PlaybackSource::CAN_GO_NEXT)
      self->m_shell->next ();

    return TRUE;
  }

  gboolean
  PlayerShell::Player::prev  (Player*     self,
                              GError**   error)
  {
    if (self->m_shell->m_selection->m_active_source == SOURCE_NONE)
      return TRUE;

    if (self->m_shell->m_source_caps[self->m_shell->m_selection->m_active_source] & PlaybackSource::CAN_GO_PREV)
      self->m_shell->prev ();

    return TRUE;
  }

  gboolean
  PlayerShell::Player::pause   (Player*      self,
                                GError**    error)
  {
    self->m_shell->pause ();
    return TRUE;
  }

  gboolean
  PlayerShell::Player::stop   (Player*       self,
                               GError**     error)
  {
    self->m_shell->stop ();
    return TRUE;
  }

  gboolean
  PlayerShell::Player::play   (Player*       self,
                               GError**     error)
  {
    self->m_shell->play (); 
    return TRUE;
  }

  bool
  PlayerShell::Player::emit_track_change (Player* self)
  {
    g_signal_emit (self, ::Bmp::MPRISPlayer::mpris_player_signals[SIGNAL_TRACK_CHANGE], 0, self->m_shell->get_metadata_hash_table ());
    return false;
  }

  void
  PlayerShell::Player::emit_caps_change (Player* self, int caps)
  {
    g_signal_emit (self, ::Bmp::MPRISPlayer::mpris_player_signals[SIGNAL_CAPS_CHANGE], 0, caps);
  }

  void
  PlayerShell::Player::emit_status_change (Player* self, int status)
  {
    using namespace Bmp;

    int mpris_playstatus = -1;

    switch (status)
    {
      case PLAYSTATUS_PLAYING:
        mpris_playstatus = 0;
        break;

      case PLAYSTATUS_PAUSED:
        mpris_playstatus = 1;
        break;

      case PLAYSTATUS_STOPPED:
        mpris_playstatus = 2;
        break;
    };

    g_signal_emit (self, ::Bmp::MPRISPlayer::mpris_player_signals[SIGNAL_STATUS_CHANGE], 0, mpris_playstatus);
  }

  //// SourceSelection

  SourceSelection::SourceSelection (BaseObjectType                 * obj,
                                    RefPtr<Gnome::Glade::Xml> const& xml)
  : VBox              (obj),
    m_active_source   (SOURCE_NONE),
    m_selected_source (SOURCE_NONE)
  {
    bool network_present = Network::check_connected();
    for (unsigned int n = 0; n < N_SOURCES; ++n)
    {
      if (sources[n].netOnly && !network_present)
      {
        continue;
      }
      else
      {
        SourceButton * b = manage (new SourceButton());
        b->set_label ((boost::format ("<span weight='600'>%s</span> <small>(_%u)</small>") % _(sources[n].name.c_str()) % (n+1)).str(), true);
        b->set_image (Gdk::Pixbuf::create_from_file (build_filename (BMP_IMAGE_DIR_SOURCES, sources[n].icon))->scale_simple (32, 32, Gdk::INTERP_BILINEAR));
        b->show_all ();

        b->signal_clicked().connect (sigc::bind (sigc::mem_fun (*this, &SourceSelection::source_button_clicked), PlaybackSourceID (n)));

        pack_start (*b, false, false);

        m_source_selection_buttons_map.insert (make_pair (PlaybackSourceID (n), b));

        m_notebook_tab_map.insert (make_pair (PlaybackSourceID (n), n));
      }
    }
    set_homogeneous (false);
  }
  
  bool
  SourceSelection::on_expose_event (GdkEventExpose * event)
  {
    Cairo::RefPtr<Cairo::Context> cr = get_window ()->create_cairo_context ();

    int xoff = get_allocation().get_x();
    int yoff = get_allocation().get_y();

    cr->rectangle (xoff,
                   yoff,
                   get_allocation().get_width(),
                   get_allocation().get_height());

    Gdk::Color c (get_style()->get_base (Gtk::STATE_NORMAL));
    cr->set_source_rgba (c.get_red_p(), c.get_green_p(), c.get_blue_p(), 1.);
    cr->fill ();

    Widget::on_expose_event (event);

    gtk_paint_shadow (GTK_WIDGET (gobj())->style,
                      GTK_WIDGET (gobj())->window,
                      GTK_STATE_NORMAL,
                      GTK_SHADOW_IN,
                      NULL,
                      GTK_WIDGET (gobj()),
                      "scrolled_window",
                      get_allocation().get_x(),
                      get_allocation().get_y(),
                      get_allocation().get_width(),
                      get_allocation().get_height());
    return true;
  }

  void
  SourceSelection::source_activate (PlaybackSourceID source)
  {
    if (m_active_source != SOURCE_NONE)
      m_source_selection_buttons_map.find (m_active_source)->second->set_playing (0);

    m_active_source = source;

    if (m_active_source != SOURCE_NONE)
      m_source_selection_buttons_map.find (m_active_source)->second->set_playing (1);

    queue_draw ();
  }

  void
  SourceSelection::source_select (PlaybackSourceID source)
  {
    if (m_selected_source != SOURCE_NONE)
      m_source_selection_buttons_map.find (m_selected_source)->second->set_active (0);

    m_selected_source = source;

    if (m_selected_source != SOURCE_NONE)
      m_source_selection_buttons_map.find (m_selected_source)->second->set_active (1);

    source_selected_.emit (source);

    queue_draw ();
  }

  void
  SourceSelection::source_button_clicked (PlaybackSourceID source)
  {
    source_select (source);
  }

  //// C++ PlayerShell

  PlayerShell::PlayerShell (BaseObjectType                 *  obj,
                            RefPtr<Gnome::Glade::Xml> const&  xml)
  : Window                  (obj)
  , m_player_dbus_obj       (PlayerShell::Player::create (*this))
  , m_n_status_messages     (0)
  , m_ref_xml               (xml)
  , m_ui_merge_id           (0)
  , m_ui_merge_id_context   (0)
  , m_ui_merge_id_tray      (0)
  , m_popup                 (0)
  , m_visible               (true)
  , m_seeking               (false)
  , m_spm_paused            (0)
  {
    int network_present = Network::check_connected();

    register_stock_icons ();
    window_set_icon_list (*this, "player");

    mcs_bind = new Mcs::Bind (mcs);
    m_about = new Bmp::AboutDialog;
    m_prefs = Preferences::create ();
    m_track_details = TrackDetails::create ();
    m_simple_progress = SimpleProgress::create ();

    m_ref_xml->get_widget ("statusbar", m_statusbar);
    dynamic_cast <Button *>(m_ref_xml->get_widget ("statusbar-clear"))->signal_clicked().connect
      (sigc::mem_fun (*this, &Bmp::PlayerShell::status_clear_message));

    // Source Images
    {
      for (unsigned int n = 0; n < N_SOURCES; ++n)
      {
        m_source_icons[n] = Gdk::Pixbuf::create_from_file (build_filename (BMP_IMAGE_DIR_SOURCES, sources[n].icon));
        m_source_icons_med[n] = m_source_icons[n]->scale_simple (64, 64, Gdk::INTERP_HYPER);
        m_source_icons_small[n] = m_source_icons[n]->scale_simple (32, 32, Gdk::INTERP_HYPER);
      }
    }

    // Metadata Bottom Area
    {
      m_ref_xml->get_widget ("label-time", m_time_label);
      m_ref_xml->get_widget_derived ("eventbox33", m_blackBar);

      PangoFontDescription * desc = 0; 

      desc = pango_font_description_new ();
      pango_font_description_set_absolute_size (desc, 12 * PANGO_SCALE);
      pango_font_description_set_family (desc, "Monospace");
      pango_font_description_set_weight (desc, PANGO_WEIGHT_BOLD);
      gtk_widget_modify_font (GTK_WIDGET (m_time_label->gobj()), desc);
      pango_font_description_free (desc);

      yy_playing  = Gdk::PixbufAnimation::create_from_file (BMP_IMAGE_DIR_MAIN G_DIR_SEPARATOR_S "yingyang.gif");
      yy_paused   = Gdk::PixbufAnimation::create_from_file (BMP_IMAGE_DIR_MAIN G_DIR_SEPARATOR_S "yingyang_paused.gif");
      yy_seeking  = Gdk::Pixbuf::create_from_file (BMP_IMAGE_DIR_MAIN G_DIR_SEPARATOR_S "yingyang_seeking.png");
      yy_stopped  = yy_playing->get_static_image ();

      m_ref_xml->get_widget ("image-yy", m_yy_image);
      m_yy_image->modify_bg (Gtk::STATE_NORMAL, Gdk::Color ("#000000"));
      m_yy_image->modify_base (Gtk::STATE_NORMAL, Gdk::Color ("#000000"));
      m_yy_image->set (yy_stopped);
      m_yy_image->hide ();
    }

    // Volume
    {
      m_ref_xml->get_widget_derived ("scale-volume", m_volume);
      mcs_bind->bind_range (*dynamic_cast<Range*>(m_volume), "bmp", "volume");
      mcs->subscribe ("PlayerShell", "bmp", "volume", sigc::mem_fun (*this, &Bmp::PlayerShell::on_volume_changed));
    }

    // Navigation
    {
      m_ref_xml->get_widget ("notebook-main", m_notebook_main);
      m_ref_xml->get_widget_derived ("vbox-sources", m_selection);
    }

    // Playback Engine Signals
    {
      ::play->property_status().signal_changed().connect
            (sigc::mem_fun (*this, &Bmp::PlayerShell::on_play_status_changed));
      ::play->signal_position().connect
            (sigc::mem_fun (*this, &Bmp::PlayerShell::on_play_position));
      ::play->signal_buffering().connect
            (sigc::mem_fun (*this, &Bmp::PlayerShell::on_play_buffering));
      ::play->signal_error().connect
            (sigc::mem_fun (*this, &Bmp::PlayerShell::on_play_error));
      ::play->signal_eos().connect
            (sigc::mem_fun (*this, &Bmp::PlayerShell::on_play_eos));
      ::play->signal_metadata().connect
            (sigc::mem_fun (*this, &Bmp::PlayerShell::on_gst_metadata));
    }

    // Seek Range
    {
      m_ref_xml->get_widget ("scale-seek", m_seek_range);
      m_seek_range->set_events (Gdk::BUTTON_PRESS_MASK | Gdk::BUTTON_RELEASE_MASK);
      m_seek_range->signal_value_changed().connect
            (sigc::mem_fun (*this, &Bmp::PlayerShell::on_seek_range_changed));
      m_seek_range->signal_event().connect
            (sigc::mem_fun (*this, &Bmp::PlayerShell::on_seek_event));
      m_seek_range->set_sensitive (false);
    }

    // UIManager + Menus + Proxy Widgets
    {
      m_ui_manager = UIManager::create ();
      m_actions = ActionGroup::create ("ActionsMain");

      m_actions->add (Action::create ("dummy", "dummy")); 

      m_actions->add (Action::create ("MenuBmp", "_BMP"));
      m_actions->add (Action::create ("MenuView", _("_View")));
      m_actions->add (Action::create ("MenuTrack", _("_Track")));
      m_actions->add (Action::create ("MenuPlayback", _("_Playback")));

      ///////////////////////////////
      // BOOKMARKS
      ///////////////////////////////
  
      m_actions->add (Action::create ("MenuBookmarks",
                                      _("Boo_kmarks")));
      
      m_actions->add (Action::create ("MenuBookmarks-radio",
                                      Gtk::StockID (BMP_STOCK_XIPH),
                                      "Radio"));

      m_actions->add (Action::create ("MenuBookmarks-jamendo",
                                      Gtk::StockID (BMP_STOCK_JAMENDO),
                                      "Jamendo"));

      m_actions->add (Action::create (PLAYER_SHELL_ACTION_CREATE_BOOKMARK,
                                      Gtk::Stock::ADD,
                                      _("_Create Bookmark")),
                                      AccelKey ("<control>d"),
                                      sigc::mem_fun (*this, &PlayerShell::on_bookmark_create));

      ////////////////////////////////
      // PLAYBACK ACTIONS
      ////////////////////////////////

      m_actions->add (Action::create (PLAYER_SHELL_ACTION_PLAY,
                                      Gtk::Stock::MEDIA_PLAY,
                                      _("_Play")),
                                      AccelKey ("<control><alt>z"),
                                      sigc::mem_fun (*this, &PlayerShell::play));

      m_actions->add (Action::create (PLAYER_SHELL_ACTION_PAUSE,
                                      Gtk::Stock::MEDIA_PAUSE,
                                      _("P_ause")),
                                      AccelKey ("<control><alt>x"),
                                      sigc::mem_fun (*this, &PlayerShell::pause));

      m_actions->add (Action::create (PLAYER_SHELL_ACTION_PREV,
                                      Gtk::Stock::MEDIA_PREVIOUS,
                                      _("P_rev")),
                                      AccelKey ("<control><alt>c"),
                                      sigc::mem_fun (*this, &PlayerShell::prev));

      m_actions->add (Action::create (PLAYER_SHELL_ACTION_NEXT,
                                      Gtk::Stock::MEDIA_NEXT,
                                      _("_Next")),
                                      AccelKey ("<control><alt>v"),
                                      sigc::mem_fun (*this, &PlayerShell::next));

      m_actions->add (Action::create (PLAYER_SHELL_ACTION_STOP,
                                      Gtk::Stock::MEDIA_STOP,
                                      _("_Stop")),
                                      AccelKey ("<control>x"),
                                     sigc::mem_fun (*this, &PlayerShell::stop));

      m_actions->add (Action::create (PLAYER_SHELL_ACTION_EQ,
                                      Gtk::StockID (BMP_STOCK_EQ),
                                      _("_Equalizer")),
                                      AccelKey ("<control>e"),
                                      sigc::mem_fun (*this, &PlayerShell::display_eq));

      m_actions->add (ToggleAction::create (PLAYER_SHELL_ACTION_REPEAT,
                                      Gtk::StockID (BMP_STOCK_REPEAT),
                                      _("_Repeat")),
                                      AccelKey ("<control>r"));
                                      
      m_actions->add (ToggleAction::create (PLAYER_SHELL_ACTION_SHUFFLE,
                                      Gtk::StockID (BMP_STOCK_SHUFFLE),
                                      _("_Shuffle")),
                                      AccelKey ("<control>s"));

      //////////////////////////////
      // MISC ACTIONS
      //////////////////////////////

      m_actions->add (Action::create (PLAYER_SHELL_ACTION_ABOUT,
                                      StockID (BMP_STOCK_ABOUT),
                                      _("_About BMP")),
                                      sigc::mem_fun (*m_about, &Bmp::AboutDialog::present));

      m_actions->add (Action::create (PLAYER_SHELL_ACTION_PREFS,
                                      Stock::PREFERENCES,
                                      _("_Preferences...")),
                                      sigc::mem_fun (*this, &PlayerShell::display_preferences));

      m_actions->add (Action::create (PLAYER_SHELL_ACTION_QUIT,
                                      Stock::QUIT,
                                      _("_Quit")),
                                      AccelKey ("<control>q"),
                                      sigc::mem_fun (core, &Core::stop));

      m_actions->add (ToggleAction::create (PLAYER_SHELL_ACTION_STATUSBAR,
                                      _("Show Statusbar")),
                                      sigc::mem_fun (*this, &PlayerShell::on_toggle_statusbar));
                                    
      m_actions->add (ToggleAction::create (PLAYER_SHELL_ACTION_COMPACT,
                                      _("Compact Mode")),
                                      AccelKey ("<control>h"),
                                      sigc::mem_fun (*this, &PlayerShell::on_toggle_compact));

      m_actions->add (ToggleAction::create (PLAYER_SHELL_ACTION_MENUBAR,
                                      _("Hide Menu Bar")),
                                      AccelKey ("<control>m"),
                                      sigc::mem_fun (*this, &PlayerShell::on_toggle_menubar));

      m_actions->add (Action::create (PLAYER_SHELL_ACTION_TRACK_DETAILS,
                                      Stock::INFO,
                                      _("Track Details")),
                                      AccelKey ("<control>t"),
                                      sigc::mem_fun (*this, &PlayerShell::display_track_details));

      m_actions->add (Action::create (PLAYER_SHELL_ACTION_RECOMMEND_LAST_FM,
                                      StockID (BMP_STOCK_LASTFM),
                                      _("Last.fm _Recommend")),
                                      AccelKey ("<control>f"),
                                      sigc::mem_fun (*this, &PlayerShell::display_lastfm_recommend));

      m_actions->add (Action::create (PLAYER_SHELL_ACTION_LASTFM,
                                      Gtk::StockID (BMP_STOCK_LASTFM),
                                      _("Play Current Artist on Last.fm")),
                                      AccelKey ("<control><alt>l"),
                                      sigc::mem_fun (*this, &PlayerShell::play_artist_on_lastfm));

      RefPtr<ToggleAction>::cast_static (m_actions->get_action (PLAYER_SHELL_ACTION_STATUSBAR))->set_active (false);

      m_actions->get_action (PLAYER_SHELL_ACTION_TRACK_DETAILS)->set_sensitive (false);
      m_actions->get_action (PLAYER_SHELL_ACTION_RECOMMEND_LAST_FM)->set_sensitive (false);
      m_actions->get_action (PLAYER_SHELL_ACTION_LASTFM)->set_sensitive (false);
      m_actions->get_action (PLAYER_SHELL_ACTION_CREATE_BOOKMARK)->set_sensitive (false);

      m_ui_manager->insert_action_group (m_actions);

      m_ui_manager->add_ui_from_string (network_present ? ui_main_menubar : ui_main_menubar_offline);
      m_ui_manager->add_ui_from_string (network_present ? ui_black_bar_popup : ui_black_bar_popup_offline);
      m_ui_manager->add_ui_from_string (ui_tray_icon);

      if (!network_present)
      {
        m_actions->get_action ("MenuBookmarks-radio")->set_visible (0);
        m_actions->get_action ("MenuBookmarks-jamendo")->set_visible (0);
      }

      m_menu_bar = m_ui_manager->get_widget ("/MenuBarMain");
      dynamic_cast<Alignment *>(m_ref_xml->get_widget("alignment-menu-default"))->add (*(m_menu_bar));
    }

    // Playback Proxies
    {
      m_actions->get_action (PLAYER_SHELL_ACTION_PLAY)->connect_proxy
            (*(dynamic_cast<Button *>(m_ref_xml->get_widget ("b_play"))));
      m_actions->get_action (PLAYER_SHELL_ACTION_PAUSE)->connect_proxy
            (*(dynamic_cast<Button *>(m_ref_xml->get_widget ("b_pause"))));
      m_actions->get_action (PLAYER_SHELL_ACTION_PREV)->connect_proxy
            (*(dynamic_cast<Button *>(m_ref_xml->get_widget ("b_prev"))));
      m_actions->get_action (PLAYER_SHELL_ACTION_NEXT)->connect_proxy
            (*(dynamic_cast<Button *>(m_ref_xml->get_widget ("b_next"))));
      m_actions->get_action (PLAYER_SHELL_ACTION_STOP)->connect_proxy
            (*(dynamic_cast<Button *>(m_ref_xml->get_widget ("b_stop"))));

      dynamic_cast<Gtk::Image *>(m_ref_xml->get_widget ("image-button-repeat"))
        ->set (render_icon (Gtk::StockID (BMP_STOCK_REPEAT), Gtk::ICON_SIZE_BUTTON));
      dynamic_cast<Gtk::Image *>(m_ref_xml->get_widget ("image-button-shuffle"))
        ->set (render_icon (Gtk::StockID (BMP_STOCK_SHUFFLE), Gtk::ICON_SIZE_BUTTON));

      m_actions->get_action (PLAYER_SHELL_ACTION_REPEAT)->connect_proxy
            (*(dynamic_cast<ToggleButton *>(m_ref_xml->get_widget ("button-repeat"))));
      m_actions->get_action (PLAYER_SHELL_ACTION_SHUFFLE)->connect_proxy
            (*(dynamic_cast<ToggleButton *>(m_ref_xml->get_widget ("button-shuffle"))));

      mcs_bind->bind_toggle_action
        (RefPtr<ToggleAction>::cast_static (m_actions->get_action (PLAYER_SHELL_ACTION_SHUFFLE)), "bmp", "shuffle");
      mcs_bind->bind_toggle_action
        (RefPtr<ToggleAction>::cast_static (m_actions->get_action (PLAYER_SHELL_ACTION_REPEAT)), "bmp", "repeat");

      m_actions->get_action (PLAYER_SHELL_ACTION_PLAY)->set_sensitive (false);
      m_actions->get_action (PLAYER_SHELL_ACTION_PAUSE)->set_sensitive (false);
      m_actions->get_action (PLAYER_SHELL_ACTION_PREV)->set_sensitive (false);
      m_actions->get_action (PLAYER_SHELL_ACTION_NEXT)->set_sensitive (false);
      m_actions->get_action (PLAYER_SHELL_ACTION_STOP)->set_sensitive (false);
    }

    add_accel_group (m_ui_manager->get_accel_group());

    ////////////////////////////////// Sources

    if ((sources[SOURCE_RADIO].netOnly && network_present) || (!sources[SOURCE_RADIO].netOnly))
    {
      m_sources[SOURCE_RADIO] = boost::shared_ptr<UiPart::Radio> (new UiPart::Radio (m_ref_xml, m_ui_manager));
      connect_source_signals (SOURCE_RADIO);
    }

    if ((sources[SOURCE_CDDA].netOnly && network_present) || (!sources[SOURCE_CDDA].netOnly))
    {
      m_sources[SOURCE_CDDA] = boost::shared_ptr<UiPart::CDDA> (new UiPart::CDDA (m_ref_xml, m_ui_manager));
      connect_source_signals (SOURCE_CDDA);
    }

    if ((sources[SOURCE_PODCASTS].netOnly && network_present) || (!sources[SOURCE_PODCASTS].netOnly))
    {
      m_sources[SOURCE_PODCASTS] = boost::shared_ptr<UiPart::Podcasts> (new UiPart::Podcasts (m_ref_xml, m_ui_manager));
      connect_source_signals (SOURCE_PODCASTS);
    }

    if ((sources[SOURCE_LIBRARY].netOnly && network_present) || (!sources[SOURCE_LIBRARY].netOnly))
    {
      m_sources[SOURCE_LIBRARY] = boost::shared_ptr<UiPart::Library> (new UiPart::Library (m_ref_xml, m_ui_manager));
      connect_source_signals (SOURCE_LIBRARY);
    }

    if ((sources[SOURCE_PLAYLIST].netOnly && network_present) || (!sources[SOURCE_PLAYLIST].netOnly))
    {
      m_sources[SOURCE_PLAYLIST] = boost::shared_ptr<UiPart::Playlist> (new UiPart::Playlist (m_ref_xml, m_ui_manager));
      connect_source_signals (SOURCE_PLAYLIST);
    }

    if ((sources[SOURCE_LASTFM].netOnly && network_present) || (!sources[SOURCE_LASTFM].netOnly))
    {
      m_sources[SOURCE_LASTFM] = boost::shared_ptr<UiPart::LASTFM> (new UiPart::LASTFM (m_ref_xml, m_ui_manager));
      connect_source_signals (SOURCE_LASTFM);
    }

    if ((sources[SOURCE_JAMENDO].netOnly && network_present) || (!sources[SOURCE_JAMENDO].netOnly))
    {
      m_sources[SOURCE_JAMENDO] = boost::shared_ptr<UiPart::Jamendo> (new UiPart::Jamendo (m_ref_xml, m_ui_manager));
      connect_source_signals (SOURCE_JAMENDO);
    }

    m_selection->source_selected().connect
      (sigc::mem_fun (*this, &Bmp::PlayerShell::source_changed));

    for (int n = 0; n < N_SOURCES; ++n)
    {
      SourceSP & source (m_sources[n]);
      if (source)
      {
        source->send_caps ();
        source->send_flags ();
      }
    }

    show ();
    while (gtk_events_pending()) gtk_main_iteration();

    // Setup EQ
    if (Audio::test_element ("equalizer-10bands"))
    {
      m_equalizer = Bmp::Equalizer::create ();
      m_actions->get_action (PLAYER_SHELL_ACTION_EQ)->set_sensitive (true);
    }
    else
    {
      m_actions->get_action (PLAYER_SHELL_ACTION_EQ)->set_sensitive (false);
    }

    // Tray icon
    {
      m_status_icon_image_default =
        Gdk::Pixbuf::create_from_file (BMP_TRAY_ICON_DIR G_DIR_SEPARATOR_S "tray-icon-default.png");
      m_status_icon_image_paused =
        Gdk::Pixbuf::create_from_file (BMP_TRAY_ICON_DIR G_DIR_SEPARATOR_S "tray-icon-paused.png");
      m_status_icon_image_playing =
        Gdk::Pixbuf::create_from_file (BMP_TRAY_ICON_DIR G_DIR_SEPARATOR_S "tray-icon-playing.png");

      m_status_icon = bmp_status_icon_new_from_pixbuf (m_status_icon_image_default->gobj());
      bmp_status_icon_set_visible (m_status_icon, TRUE);

      g_object_connect (G_OBJECT (m_status_icon),
                        "signal::click",
                        G_CALLBACK(PlayerShell::status_icon_click),
                        this,
                        "signal::popup-menu",
                        G_CALLBACK(PlayerShell::status_icon_popup_menu),
                        this,
                        "signal::scroll-up",
                        G_CALLBACK(PlayerShell::status_icon_scroll_up),
                        this,
                        "signal::scroll-down",
                        G_CALLBACK(PlayerShell::status_icon_scroll_down),
                        this,
                        NULL);

      GtkWidget *tray_icon = bmp_status_icon_get_tray_icon (m_status_icon);
      gtk_widget_realize (GTK_WIDGET (tray_icon));
      g_object_connect (G_OBJECT (tray_icon),
                        "signal::embedded",
                        G_CALLBACK (Bmp::PlayerShell::create_popup),
                        this,
                        NULL);
    }

    library->signal_modify_start().connect
      (sigc::bind (sigc::mem_fun (*this, &Bmp::PlayerShell::progress_start), _("Modifying Tracks")));
    library->signal_modify_end().connect
      (sigc::bind (sigc::mem_fun (*this, &Bmp::PlayerShell::progress_end), false));
    library->signal_modify_step().connect
      (sigc::mem_fun (*this, &Bmp::PlayerShell::progress_step));

    library->signal_vacuum_begin().connect
      (sigc::bind (sigc::mem_fun (*this, &Bmp::PlayerShell::progress_start), "Reticulating Splines"));
    library->signal_vacuum_end().connect
      (sigc::bind (sigc::mem_fun (*this, &Bmp::PlayerShell::progress_end), false));
    library->signal_vacuum_step().connect
      (sigc::mem_fun (*this, &Bmp::PlayerShell::progress_step));

    UiPart::Library * p = dynamic_cast<UiPart::Library*>(m_sources[SOURCE_LIBRARY].get());
    m_prefs->signal_library_update_request().connect
      (sigc::mem_fun (*p, &Bmp::UiPart::Library::update));

    m_bookmark_manager.signal_bookmark_loaded().connect
      (sigc::mem_fun (*this, &Bmp::PlayerShell::on_bookmark_loaded));
    m_bookmark_manager.load_bookmarks ();
  
    m_blackBar->signal_event().connect
      (sigc::mem_fun (*this, &Bmp::PlayerShell::on_black_bar_event));

    lastfm_radio->signal_disconnected().connect
      (sigc::mem_fun (*this, &Bmp::PlayerShell::on_last_fm_radio_disconnected));
    lastfm_radio->signal_connected().connect
      (sigc::mem_fun (*this, &Bmp::PlayerShell::on_last_fm_radio_connected));

    lastfm_scrobbler->signal_disconnected().connect
      (sigc::mem_fun (*this, &Bmp::PlayerShell::on_scrobbler_disconnected));
    lastfm_scrobbler->signal_connected().connect
      (sigc::mem_fun (*this, &Bmp::PlayerShell::on_scrobbler_connected));
  }

  void
  PlayerShell::init ()
  {
    m_prefs->setup_lastfm ();
  }

  void
  PlayerShell::connect_source_signals (PlaybackSourceID source)
  {
    m_sources[source]->signal_caps().connect
      (sigc::bind (sigc::mem_fun (*this, &Bmp::PlayerShell::on_source_caps), source));

    m_sources[source]->signal_flags().connect
      (sigc::bind (sigc::mem_fun (*this, &Bmp::PlayerShell::on_source_flags), source));

    m_sources[source]->signal_track_metadata().connect
      (sigc::bind (sigc::mem_fun (*this, &Bmp::PlayerShell::on_source_track_metadata), source, PARSE_FLAGS_ALL));

    m_sources[source]->signal_playback_request().connect
      (sigc::bind (sigc::mem_fun (*this, &Bmp::PlayerShell::on_source_play_request), source));

    m_sources[source]->signal_segment().connect
      (sigc::bind (sigc::mem_fun (*this, &Bmp::PlayerShell::on_source_segment), source));

    m_sources[source]->signal_stop_request().connect
      (sigc::bind (sigc::mem_fun (*this, &Bmp::PlayerShell::on_source_stop), source));

    m_sources[source]->signal_next_request().connect
      (sigc::bind (sigc::mem_fun (*this, &Bmp::PlayerShell::on_source_next), source));

    m_sources[source]->signal_message().connect
      (sigc::mem_fun (*this, &Bmp::PlayerShell::on_source_message_set));

    m_sources[source]->signal_message_clear().connect
      (sigc::mem_fun (*this, &Bmp::PlayerShell::on_source_message_clear));
  }

  bool
  PlayerShell::display_popup_delayed ()
  {
    m_popup->tooltip_mode ();
    return false;
  }

  gboolean
  PlayerShell::status_icon_enter (BmpStatusIcon *icon, GdkEventCrossing *event, gpointer data)
  {
    PlayerShell & shell = (*(reinterpret_cast<PlayerShell*>(data)));

    if (shell.m_selection->m_active_source != SOURCE_NONE)
    {
      shell.delay_popup = signal_timeout().connect (sigc::mem_fun (&shell, &Bmp::PlayerShell::display_popup_delayed), 400);
    }

    return FALSE;
  }

  gboolean
  PlayerShell::status_icon_leave (BmpStatusIcon *icon, GdkEventCrossing *event, gpointer data)
  {
    PlayerShell & shell = (*(reinterpret_cast<PlayerShell*>(data)));

    shell.delay_popup.disconnect ();

    shell.m_popup->tooltip_mode (false);

    return FALSE;
  }

  void
  PlayerShell::status_icon_popup_menu (BmpStatusIcon *icon, guint arg1, guint arg2, gpointer data)
  {
    PlayerShell & shell = (*(reinterpret_cast<PlayerShell*>(data)));

    shell.delay_popup.disconnect ();

    if (shell.m_popup)
    {
      shell.m_popup->tooltip_mode (false, true);
    }

    GtkWidget* menu = Util::ui_manager_get_popup (shell.m_ui_manager->gobj(), "/popup-tray/menu-tray");
    gtk_widget_realize (menu);
    gtk_menu_popup (GTK_MENU (menu), /* parent shell */ NULL,
                                     /* parent item  */ NULL,
                                     GtkMenuPositionFunc (bmp_status_icon_position_menu), icon, arg1, arg2);
  }

  void
  PlayerShell::status_icon_click (BmpStatusIcon *icon, gpointer data)
  {
    PlayerShell & shell = (*(reinterpret_cast<Bmp::PlayerShell*>(data)));

    shell.delay_popup.disconnect ();

    if (shell.m_popup)
    {
        shell.m_popup->tooltip_mode (false, true);
    }

    if (shell.m_visible)
    {
        shell.iconify ();
    }
    else
    {
        shell.deiconify ();
    }
  }

  void
  PlayerShell::status_icon_scroll_up (BmpStatusIcon *icon, gpointer data)
  {
    int volume;
    volume = mcs->key_get<int>("bmp", "volume");
    volume += 4;
    volume = (volume > 100) ? 100 : volume;
    mcs->key_set<int>("bmp", "volume", volume);
  }

  void
  PlayerShell::status_icon_scroll_down (BmpStatusIcon *icon, gpointer data)
  {
    int volume;
    volume = mcs->key_get<int>("bmp", "volume");
    volume -= 4;
    volume = (volume < 0) ? 0 : volume;
    mcs->key_set<int>("bmp", "volume", volume);
  }

  void
  PlayerShell::create_popup (GtkPlug *plug, gpointer data)
  {
    Bmp::PlayerShell & shell = (*(reinterpret_cast<Bmp::PlayerShell*>(data)));

    GtkWidget *widget = GTK_WIDGET (plug);

    ustring text ("");
    Gdk::Color outline, inlay;
    outline.set_rgb (0, 0, 0);
    inlay.set_rgb (65535, 65535, 65535);
    shell.m_popup = new Bmp::Popup (widget, outline, inlay, shell.m_metadata);
    g_object_connect (G_OBJECT (plug),
                      "signal::enter-notify-event",
                      G_CALLBACK(PlayerShell::status_icon_enter),
                      &shell,
                      "signal::leave-notify-event",
                      G_CALLBACK(PlayerShell::status_icon_leave),
                      &shell,
                      NULL);
  }

  void
  PlayerShell::on_volume_changed (MCS_CB_DEFAULT_SIGNATURE)
  {
    ::play->property_volume() = boost::get <int> (value);
  }

  bool
  PlayerShell::on_delete_event (GdkEventAny *event)
  {
    core->stop ();
    return true;
  }

  bool
  PlayerShell::on_black_bar_event (GdkEvent * ev)
  {
    if( ev->type == GDK_BUTTON_PRESS )
    {
      GdkEventButton * event = reinterpret_cast <GdkEventButton *> (ev);
      if( event->button == 3 )
      {
        Gtk::Menu * menu = dynamic_cast < Gtk::Menu* >
                              (Util::get_popup (m_ui_manager, "/popup-shell/menu-shell-context"));

        if (menu) // better safe than screwed
        {
          menu->popup (event->button, event->time);
        }
        return true;
      }
    }
    return false;
  }

  void
  PlayerShell::progress_start (int n_items, const char * title)
  {
    m_n_progress_items = n_items;
    m_nth_progress_item = 0;

    m_simple_progress->step  (n_items, 0);
    m_simple_progress->set_label (title);

    m_simple_progress->show ();
  }

  void
  PlayerShell::progress_step ()
  {
    ++m_nth_progress_item;
    m_simple_progress->step  (m_n_progress_items, m_nth_progress_item);
  }

  void
  PlayerShell::progress_end (bool wait_for_close)
  {
    m_simple_progress->done (wait_for_close);
    m_simple_progress->hide ();
  }

  bool
  PlayerShell::source_restore_context (GdkEventButton *event)
  {
    if (m_selection->m_active_source != SOURCE_NONE)
    {
      m_selection->source_select (m_selection->m_active_source);
      source_changed (m_selection->m_active_source);
      m_sources[m_selection->m_active_source]->restore_context();
    }
    return false;
  }

  // EQ
  void
  PlayerShell::display_eq ()
  {
    m_equalizer->present ();
  }

  // Preferences
  void
  PlayerShell::display_preferences ()
  {
    m_prefs->present ();
  }

  // Statusbar API
  void
  PlayerShell::status_clear_message ()
  {
    m_statusbar->pop (0);
    m_n_status_messages--;
    if (m_n_status_messages == 0)
    {
      dynamic_cast <Button *>(m_ref_xml->get_widget ("statusbar-clear"))->set_sensitive (false);
      RefPtr<ToggleAction>::cast_static (m_actions->get_action (PLAYER_SHELL_ACTION_STATUSBAR))->set_active (false);
    }
  }

  guint
  PlayerShell::status_push_message (ustring const& message)
  {
    dynamic_cast <Button *>(m_ref_xml->get_widget ("statusbar-clear"))->set_sensitive ();
    m_n_status_messages++;
    RefPtr<ToggleAction>::cast_static (m_actions->get_action (PLAYER_SHELL_ACTION_STATUSBAR))->set_active();
    return m_statusbar->push (message, 0);
  }

  void
  PlayerShell::display_lastfm_recommend ()
  {
    boost::shared_ptr<Bmp::LastFM::RecommendDialog> dialog (Bmp::LastFM::RecommendDialog::create());
    dialog->run (m_metadata.artist.get().c_str(), m_metadata.album.get().c_str(), m_metadata.title.get().c_str());
  }

  void
  PlayerShell::play_artist_on_lastfm ()
  {
    static boost::format
      f_artist ("lastfm://artist/%s/similarartists");

    if (m_selection->m_active_source != SOURCE_NONE)
    {
      m_sources[m_selection->m_active_source]->stop ();
      m_blackBar->reset ();
    }

    SourceSP & source (m_sources[SOURCE_LASTFM]);
    if (source && m_metadata.artist)
    {
      m_selection->source_activate (SOURCE_LASTFM);
      m_selection->source_select (SOURCE_LASTFM);
      dynamic_cast <Bmp::UiPart::LASTFM*>(source.get())->set_uri ((f_artist % m_metadata.artist.get().c_str()).str());
    }
  }

  void
  PlayerShell::display_track_details ()
  {
    m_track_details->show ();
  }

  void
  PlayerShell::reset_seek_range ()
  {
    m_seek_range->set_sensitive (false);
    m_seek_range->set_range (0., 1.);
    m_seek_range->set_value (double(0.));
    m_time_label->set_text ("");
  }

  void
  PlayerShell::on_seek_range_changed ()
  {
    double value = m_seek_range->get_value();
    uint64_t position = uint64_t (value);
    uint64_t minutes = position / 60;
    uint64_t seconds = position % 60;
    m_time_label->set_text ((time_f % minutes % seconds).str());
  }

  bool
  PlayerShell::on_seek_event (GdkEvent *event)
  {
    if (event->type == GDK_BUTTON_PRESS)
    {
      m_seeking = true;
      m_yy_image->set (yy_seeking);
    }
    else if (event->type == GDK_BUTTON_RELEASE)
    {
      ::play->seek (uint64_t (m_seek_range->get_value()));
      m_yy_image->clear ();
      m_seeking = false;
    }
    return false;
  }

  void
  PlayerShell::on_play_error (ustring const& element, ustring const& uri, ustring const& debug, GError * error, GstElement const* src)
  {
    ustring text, help;

    if (error->domain == GST_CORE_ERROR)
    {
      switch (error->code)
      {
        case GST_CORE_ERROR_MISSING_PLUGIN:
        {
          text = _("No plugin available to play stream");
          break;
        }

        case GST_CORE_ERROR_SEEK:
        {
          text = _("Error during Seek");
          break;
        }

        case GST_CORE_ERROR_STATE_CHANGE:
        {
          struct StringPairs
          {
            ustring state_pre;
            ustring state_post;
          };

          static StringPairs const string_pairs[] =
          {
            {"0",       "READY"},
            {"READY",   "PAUSED"},
            {"PAUSED",  "PLAYING"},
            {"PLAYING", "PAUSED"},
            {"PAUSED",  "READY"},
            {"READY",   "0"},
          };

          text = _("Error occured during state change from ");
          text += string_pairs[GST_STATE_PENDING(src)].state_pre  + " -> ";
          text += string_pairs[GST_STATE_PENDING(src)].state_post;
          break;
        }

        case GST_CORE_ERROR_PAD:
        {
          text = _("Error while performing pad linking");
          break;
        }

        case GST_CORE_ERROR_NEGOTIATION:
        {
          text = _("Error occured during element(s) negotiation");
          break;
        }

        default:
        {
          text = (boost::format (_("Unhandled error of domain GST_CORE_ERROR with code %d")) % error->code).str();
          break;
        }

      }
    }
    else if (error->domain == GST_RESOURCE_ERROR)
    {
      switch (error->code)
      {
        case GST_RESOURCE_ERROR_SEEK:
        {
          text = _("Error during Seek (resource; end of stream?)");
          break;
        }

        case GST_RESOURCE_ERROR_NOT_FOUND:
        case GST_RESOURCE_ERROR_OPEN_READ:
        case GST_RESOURCE_ERROR_OPEN_READ_WRITE:
        case GST_RESOURCE_ERROR_READ:
        {
          text = _("Unable to Open Resource (stream)");
          help = _("The requested resource was not found (The file/stream does not exist/is invalid?)");
          break;
        }

        case GST_RESOURCE_ERROR_SYNC:
        {
          text = _("Synchronization Error");
          break;
        }

        case GST_RESOURCE_ERROR_BUSY:
        {
          text = _("Device is in use");
          if (element == "sink")
          {
            help = _("The audio device seems to be in use. (Maybe another audio/media app is running?)");
          }
          break;
        }

        case GST_RESOURCE_ERROR_FAILED:
        {
          text = _("Operation Failed");
          break;
        }

        default:
        {
          text = (boost::format (_("Unhandled error of domain GST_RESOURCE_ERROR with code %d")) % error->code).str();
        }
      }
    }
    else if (error->domain == GST_STREAM_ERROR)
    {
        switch (error->code)
        {
          case GST_STREAM_ERROR_FAILED:
          {
            text = _("Unable to play/continue playing stream (generic error)");
            break;
          }

          case GST_STREAM_ERROR_TYPE_NOT_FOUND:
          {
            text = _("Unable to Determine Stream Data");
            help = _("GStreamer was unable to determine the stream/file data\nand was unable to play it back");
            break;
          }

          case GST_STREAM_ERROR_WRONG_TYPE:
          {
            text = _("Element is unable to handle this stream type");
            break;
          }

          case GST_STREAM_ERROR_CODEC_NOT_FOUND:
          {
            text = _("No Codec installed to Handle this Stream");
            help = _("GStreamer is not able to play this kind of stream (Maybe missing plugin)");
            break;
          }

          case GST_STREAM_ERROR_DECODE:
          {
            text = _("Error Decoding the Stream");
            break;
          }

          case GST_STREAM_ERROR_DEMUX:
          {
            text = _("Error Demultiplexing the Stream");
            break;
          }

          case GST_STREAM_ERROR_FORMAT:
          {
            text = _("Error Parsing the Stream Format");
            break;
          }

          default:
          {
            text = (boost::format (_("Unhandled error of domain GST_STREAM_ERROR with code %d")) % error->code).str();
            break;
          }
        }
    }

    DialogGstError * d = DialogGstError::create ();
    d->run (text, element, uri, debug, help, error->domain, error->code);
    g_error_free (error);
    delete d;
  }

  void
  PlayerShell::on_play_buffering (double buffered)
  {
    if (buffered == 0.)
    {
      m_yy_image->set (yy_seeking);
    }
    else if (buffered == 1.)
    {
      if (m_selection->m_active_source != SOURCE_NONE)
      {
        m_sources[m_selection->m_active_source]->buffering_done();
      }
      m_yy_image->clear ();
    }

    int buffered_i (int(buffered * 100.));

    if (buffered_i < 100)
    {
      if (buffered_i == 0)
        m_blackBar->set_message (_("Connecting..."));
      else
        m_blackBar->set_message ((boost::format (_("Prebuffering: %d%%")) % buffered_i).str());
    }
    else
    {
      m_blackBar->set_message (_("Buffering Complete.")); 
      m_blackBar->clear_message ();
    }

  }

  void
  PlayerShell::on_gst_metadata (BmpGstMetadataField field)
  {
    BmpGstMetadata const& m = ::play->get_metadata();
    BmpMetadataParseFlags parse_flags = BmpMetadataParseFlags (0); 

    switch (field)
    {
      case FIELD_IMAGE:
        parse_flags = PARSE_FLAGS_IMAGE;
        m_metadata.image = m.m_image.get()->copy();
        reparse_metadata (parse_flags);
        return;

      case FIELD_TITLE:
        if (m_source_flags[m_selection->m_active_source] & PlaybackSource::F_HANDLE_STREAMINFO) 
        {
          parse_flags = PARSE_FLAGS_TEXT;
          m_metadata.title = m.m_title.get();
          reparse_metadata (parse_flags);
          set_current_source_image ();
        }
        return;
      
      case FIELD_ALBUM:
        if (m_source_flags[m_selection->m_active_source] & PlaybackSource::F_HANDLE_STREAMINFO) 
        {
          parse_flags = PARSE_FLAGS_TEXT;
          m_metadata.album = m.m_album.get();
          reparse_metadata (parse_flags);
          set_current_source_image ();
        }
        return;

      case FIELD_BITRATE: 
        m_metadata.bitrate = m.m_bitrate.get();
        return; //XXX! Currently updating the bitrate doesn't change anything visually -- md
                //FIXME: Actually it does, the track info window

      case FIELD_CODEC:
        // not handled in TrackMetadata (FIXME or separate handling?)
        break;
    } 
  }

  void
  PlayerShell::on_play_position (int position)
  {
    PlaybackSource::Caps caps (m_source_caps[m_selection->m_active_source]);

    if (m_seeking) //FIXME: Provides timing excludes seek (?)
      return;

    if (caps & PlaybackSource::PROVIDES_TIMING)
      return;

    int64_t minutes  (position / 60);
    int64_t seconds  (position % 60);
    int64_t duration (::play->property_duration().get_value());

    m_time_label->set_text ((time_f % minutes % seconds).str());
    if ((duration >= 0) && (position <= duration) && (position >= 0))
    {
      m_seek_range->set_range (0., duration); 
      m_seek_range->set_value (position);
      if (m_popup)
      {
        m_popup->set_position (position, duration);
      }
    }
  }

  void
  PlayerShell::reparse_metadata (BmpMetadataParseFlags parse_flags)
  {
    struct NoCurrentImage {};

    try{
        if (parse_flags & PARSE_FLAGS_IMAGE)
        {
          if (m_metadata.image)
          {
            m_blackBar->set_image (m_metadata.image->scale_simple (64, 64, Gdk::INTERP_HYPER),
                                    m_source_flags[m_selection->m_active_source] & PlaybackSource::F_ALWAYS_IMAGE_FRAME);
            m_asin.reset ();
          }
          else
          if (m_metadata.asin)
          {
            if (m_metadata.asin != m_asin)
            {
              try{
                  m_blackBar->set_image (amazon->fetch (m_metadata.asin.get(), COVER_SIZE_INFO_AREA, true), true);
                  amazon->fetch (m_metadata.asin.get(), m_metadata.image, true); //FIXME: Redundancy here ^
                  m_asin = m_metadata.asin;
                }
              catch (Bmp::Amazon::NoCoverError & cxe)
                {
                  throw NoCurrentImage ();
                }
            }
            else
            {
              // Same ASIN, so same cover, save the reparsing
              parse_flags = BmpMetadataParseFlags (parse_flags & ~PARSE_FLAGS_IMAGE);
            }
          }
          else
          {
            throw NoCurrentImage();        
          }
        }
      }
    catch (NoCurrentImage & cxe)
      {
        m_asin.reset ();
        set_current_source_image ();
      }

    if (parse_flags & PARSE_FLAGS_TEXT)
    {
      ustring artist, album, title;
      parse_metadata (m_metadata, artist, album, title);

      m_blackBar->set_text (L_TITLE, title);
      m_blackBar->set_text (L_ARTIST, artist);
      m_blackBar->set_text (L_ALBUM, album);

      if (Network::check_connected() && (m_metadata.artist && m_metadata.album && m_metadata.title))
        m_actions->get_action (PLAYER_SHELL_ACTION_RECOMMEND_LAST_FM)->set_sensitive (1 && lastfm_scrobbler->connected());
      else
        m_actions->get_action (PLAYER_SHELL_ACTION_RECOMMEND_LAST_FM)->set_sensitive (0 && lastfm_scrobbler->connected());

      if (m_metadata.artist && m_metadata.album && m_metadata.title)
        m_actions->get_action (PLAYER_SHELL_ACTION_TRACK_DETAILS)->set_sensitive (1);
      else
        m_actions->get_action (PLAYER_SHELL_ACTION_TRACK_DETAILS)->set_sensitive (0);

      m_actions->get_action (PLAYER_SHELL_ACTION_LASTFM)->set_sensitive (bool (m_metadata.artist) && lastfm_radio->connected());
    }

    signal_idle().connect (sigc::bind (sigc::mem_fun (*this, &Bmp::PlayerShell::metadata_update_when_idle), parse_flags));
  }

  void
  PlayerShell::on_last_fm_radio_connected ()
  {
    if (::play->property_status().get_value() != PLAYSTATUS_STOPPED)
      m_actions->get_action (PLAYER_SHELL_ACTION_LASTFM)->set_sensitive (bool (m_metadata.artist));
  }

  void
  PlayerShell::on_last_fm_radio_disconnected ()
  {
      m_actions->get_action (PLAYER_SHELL_ACTION_LASTFM)->set_sensitive (false);
  }

  void
  PlayerShell::on_scrobbler_connected ()
  {
    if ((::play->property_status().get_value() != PLAYSTATUS_STOPPED) &&
          (m_metadata.artist && m_metadata.album && m_metadata.title))
    {
        m_actions->get_action (PLAYER_SHELL_ACTION_RECOMMEND_LAST_FM)->set_sensitive (true);
    }
  }

  void
  PlayerShell::on_scrobbler_disconnected ()
  {
      m_actions->get_action (PLAYER_SHELL_ACTION_RECOMMEND_LAST_FM)->set_sensitive (false);
  }

  bool
  PlayerShell::metadata_update_when_idle (BmpMetadataParseFlags parse_flags)
  {
    if (m_popup)
    {
      m_popup->metadata_updated (parse_flags);
      if (mcs->key_get<bool>("bmp", "display-notifications"))
      {
        m_popup->show ();
      }
    }

    m_track_details->set_metadata (m_metadata);
    return false;
  }

  void
  PlayerShell::set_current_source_image ()
  {
    m_metadata.image = m_source_icons[m_selection->m_active_source];
    m_blackBar->set_image (m_source_icons_med[m_selection->m_active_source], false); 
  }

  void
  PlayerShell::on_source_caps (Bmp::PlaybackSource::Caps caps, PlaybackSourceID source)
  {
    Mutex::Lock L (m_source_state_lock);
  
    m_source_caps[source] = caps;
    m_player_dbus_obj->emit_caps_change (m_player_dbus_obj, int (caps));

    if ((source == m_selection->m_active_source) || (m_notebook_main->get_current_page() == source))
    {
      m_actions->get_action (PLAYER_SHELL_ACTION_PREV)->set_sensitive (caps & PlaybackSource::CAN_GO_PREV);
      m_actions->get_action (PLAYER_SHELL_ACTION_NEXT)->set_sensitive (caps & PlaybackSource::CAN_GO_NEXT);
      m_actions->get_action (PLAYER_SHELL_ACTION_PLAY)->set_sensitive (caps & PlaybackSource::CAN_PLAY);
      m_actions->get_action (PLAYER_SHELL_ACTION_CREATE_BOOKMARK)->set_sensitive (caps & PlaybackSource::CAN_BOOKMARK);
    }

    if (source == m_selection->m_active_source)
    {
      if (::play->property_status().get_value() == PLAYSTATUS_PLAYING)
      {
        m_actions->get_action (PLAYER_SHELL_ACTION_PAUSE)->set_sensitive (caps & PlaybackSource::CAN_PAUSE);

        if (caps & PlaybackSource::CAN_SEEK)
        {
          m_seek_range->set_sensitive (true);
        }
        else
        {
          reset_seek_range ();
        }
      }
    }
  }

  void
  PlayerShell::on_source_track_metadata (TrackMetadata const& metadata, PlaybackSourceID source, BmpMetadataParseFlags parse_flags)
  {
    Mutex::Lock L (m_metadata_mutex);
    m_metadata = metadata;
    reparse_metadata ();
  }

  void
  PlayerShell::on_source_flags (Bmp::PlaybackSource::Flags flags, PlaybackSourceID source)
  {
    Mutex::Lock L (m_source_state_lock);
    m_source_flags[source] = flags;
    m_actions->get_action (PLAYER_SHELL_ACTION_REPEAT)->set_sensitive (flags & PlaybackSource::F_USES_REPEAT);
    m_actions->get_action (PLAYER_SHELL_ACTION_SHUFFLE)->set_sensitive (flags & PlaybackSource::F_USES_SHUFFLE);
  }

  void
  PlayerShell::on_source_play_request (PlaybackSourceID source_id)
  {
    if (!m_playback_mutex.trylock())
      return;

    if ((m_selection->m_active_source != SOURCE_NONE) &&
        (m_selection->m_active_source != source_id))
    {
      if (m_source_flags[m_selection->m_active_source] & PlaybackSource::F_HANDLE_LASTFM)
      {
            shell_lastfm_scrobble_current (); // scrobble the current item before switching
      }

      m_sources[m_selection->m_active_source]->stop ();
      m_blackBar->reset ();
    }

    SourceSP & source (m_sources[source_id]);

    try{
            source->play ();
            ::play->switch_stream (source->get_uri(), source->get_type());
    }
    catch (PlaybackSource::UnableToInitiatePlaybackError& cxe)
    {
            ::play->request_status (PLAYSTATUS_STOPPED);
            status_push_message ((boost::format ("BMP: Couldn't initiate playback: %s") % cxe.what()).str());
            m_selection->source_activate (SOURCE_NONE);
            m_playback_mutex.unlock ();
            return;
    }

    if (::play->property_sane() == true)
    {
          m_selection->source_select (source_id);
          m_selection->source_activate (source_id);

          source->play_post ();
          source->send_caps ();
          source->send_flags ();

          m_popup->set_source_icon (m_source_icons_small[source_id]);

          if (m_source_flags[source_id] & PlaybackSource::F_HANDLE_LASTFM)
          {
                shell_lastfm_now_playing ();
          }

          m_player_dbus_obj->emit_track_change (m_player_dbus_obj);

          if (m_ui_merge_id_context)
            m_ui_manager->remove_ui (m_ui_merge_id_context);

          if (m_ui_merge_id_tray)
            m_ui_manager->remove_ui (m_ui_merge_id_tray);

          m_ui_merge_id_context = boost::dynamic_pointer_cast <Bmp::UiPart::Base> (source)->add_context_ui();
          m_ui_merge_id_tray = boost::dynamic_pointer_cast <Bmp::UiPart::Base> (source)->add_tray_ui();
          m_blackBar->set_source (m_source_icons_small[source_id]->scale_simple (20, 20, Gdk::INTERP_BILINEAR));
    }

    m_playback_mutex.unlock ();
  }

  void
  PlayerShell::on_source_message_set (Glib::ustring const& message)
  {
    m_blackBar->set_message (message);
    while (gtk_events_pending()) gtk_main_iteration ();
  }

  void
  PlayerShell::on_source_message_clear ()
  {
    m_blackBar->clear_message ();
    while (gtk_events_pending()) gtk_main_iteration ();
  }

  void
  PlayerShell::on_source_segment (PlaybackSourceID source)
  {
    if (source == m_selection->m_active_source)
    {
      m_blackBar->reset ();
      m_sources[source]->segment ();
    }
    else
      g_warning ("%s: Source '%d' requested segment, but is not the active source", G_STRLOC, source);
  }

  void
  PlayerShell::on_source_stop (PlaybackSourceID source)
  {
    if (source == m_selection->m_active_source)
      stop ();
    else
      g_warning ("%s: Source '%d' requested stop, but is not the active source", G_STRLOC, source);
  }

  void
  PlayerShell::on_source_next (PlaybackSourceID source)
  {
    if (source == m_selection->m_active_source)
      next ();
    else
      g_warning ("%s: Source '%d' requested next, but is not the active source", G_STRLOC, source);
  }

  void
  PlayerShell::source_changed (int n)
  {
    m_notebook_main->set_current_page (n);

    SourceSP & source (m_sources[n]);
    if (source)
    {
      source->send_caps ();
      source->send_flags ();

      if (m_ui_merge_id)
      {
        m_ui_manager->remove_ui (m_ui_merge_id);
      }

      m_ui_merge_id = boost::dynamic_pointer_cast <Bmp::UiPart::Base> (source)->add_ui();
    }
    else
    {
      m_actions->get_action (PLAYER_SHELL_ACTION_PLAY)->set_sensitive (false);
      m_actions->get_action (PLAYER_SHELL_ACTION_PREV)->set_sensitive (false);
      m_actions->get_action (PLAYER_SHELL_ACTION_NEXT)->set_sensitive (false);
    }
  }

  //////////////////////////////// Internal Playback Control

  void
  PlayerShell::play ()
  {
    if (!m_playback_mutex.trylock())
      return;

    PlaybackSourceID source_id = PlaybackSourceID (m_notebook_main->get_current_page());

    if (m_selection->m_active_source != SOURCE_NONE)
    {
          m_sources[m_selection->m_active_source]->stop ();
          m_blackBar->reset ();

          if (source_id != m_selection->m_active_source)
          {
                m_yy_image->set (yy_stopped);
                ::play->request_status (PLAYSTATUS_STOPPED);
          }
    }

    SourceSP & source = m_sources[source_id];

    try
    {
          source->play ();
          ::play->switch_stream (source->get_uri(), source->get_type());
    }
    catch (PlaybackSource::UnableToInitiatePlaybackError& cxe)
    {
          ::play->request_status (PLAYSTATUS_STOPPED);
          status_push_message ((boost::format ("Couldn't initiate playback: %s") % cxe.what()).str());
          m_selection->source_activate (SOURCE_NONE);
          m_playback_mutex.unlock ();
          return;
    }

    if (::play->property_sane() == true)
    {
          m_selection->source_activate (source_id);

          source->play_post ();
          source->send_caps ();
          source->send_flags ();

          m_popup->set_source_icon (m_source_icons_small[source_id]);

          if (m_source_flags[source_id] & PlaybackSource::F_HANDLE_LASTFM)
          {
                shell_lastfm_now_playing ();
          }

          m_player_dbus_obj->emit_track_change (m_player_dbus_obj);

          if (m_ui_merge_id_context)
            m_ui_manager->remove_ui (m_ui_merge_id_context);

          if (m_ui_merge_id_tray)
            m_ui_manager->remove_ui (m_ui_merge_id_tray);

          m_ui_merge_id_context = boost::dynamic_pointer_cast <Bmp::UiPart::Base> (source)->add_context_ui();
          m_ui_merge_id_tray = boost::dynamic_pointer_cast <Bmp::UiPart::Base> (source)->add_tray_ui();
          m_blackBar->set_source (m_source_icons_small[source_id]->scale_simple (20, 20, Gdk::INTERP_BILINEAR));
    }

    m_playback_mutex.unlock ();
  }

  void
  PlayerShell::pause ()
  {
    ::play->request_status (PLAYSTATUS_PAUSED);
  }

  void
  PlayerShell::prev ()
  {
    SourceSP & source (m_sources[m_selection->m_active_source]);
    PlaybackSource::Flags f = m_source_flags[m_selection->m_active_source];
    PlaybackSource::Caps c = m_source_caps[m_selection->m_active_source];

    if (f & PlaybackSource::F_HANDLE_LASTFM)
    {
          shell_lastfm_scrobble_current ();
    }

    if ((c & PlaybackSource::CAN_GO_PREV) && source->go_prev())
    {
          if ((f & PlaybackSource::F_PHONY_PREV) == 0)
          {
                ::play->switch_stream (source->get_uri(), source->get_type());
                if (::play->property_sane() == true)
                {
                      source->prev_post ();
                      source->send_caps ();
                      source->send_flags ();

                      if (f & PlaybackSource::F_HANDLE_LASTFM)
                      {
                            shell_lastfm_now_playing ();
                      }

                      m_player_dbus_obj->emit_track_change (m_player_dbus_obj);
                }
          }
    }
    else
    {
          stop ();
    }
  }

  void
  PlayerShell::on_play_eos ()
  {
    SourceSP & source (m_sources[m_selection->m_active_source]);
    PlaybackSource::Flags f = m_source_flags[m_selection->m_active_source];
    PlaybackSource::Caps c = m_source_caps[m_selection->m_active_source];

    if (f & PlaybackSource::F_HANDLE_LASTFM)
    {
          shell_lastfm_scrobble_current ();
    }

    if ((c & PlaybackSource::CAN_GO_NEXT) && source->go_next())
    {
          if ((f & PlaybackSource::F_PHONY_NEXT) == 0)
          {
                ::play->switch_stream (source->get_uri(), source->get_type());
                if (::play->property_sane() == true)
                {
                      source->next_post ();
                      source->send_caps ();
                      source->send_flags ();

                      if (f & PlaybackSource::F_HANDLE_LASTFM)
                      {
                            shell_lastfm_now_playing ();
                      }

                      m_player_dbus_obj->emit_track_change (m_player_dbus_obj);
                }
          }
    }
    else
    {
          stop ();
    }
  }

  void
  PlayerShell::next ()
  {
    if (m_selection->m_active_source != SOURCE_NONE)
    {
      SourceSP & source (m_sources[m_selection->m_active_source]);
      source->skipped ();
    }

    on_play_eos ();
  }

  void
  PlayerShell::shell_lastfm_scrobble_current ()
  {
    if (mcs->key_get<bool>("lastfm","queue-enable") && ::play->lastfm_qualifies())
    {
      Mutex::Lock L (m_metadata_mutex);
      TrackQueueItem item (m_metadata);
      g_message ("%s: Scrobbling item, duration: %llu", G_STRLOC, item.length);
      lastfm_scrobbler->scrobble (item);
    } 
  }

  void
  PlayerShell::shell_lastfm_now_playing ()
  {
    Mutex::Lock L (m_metadata_mutex);
    TrackQueueItem item (m_metadata);
    g_message ("%s: Now-playing, duration: %llu", G_STRLOC, item.length);
    lastfm_scrobbler->now_playing (item);
  }

  void
  PlayerShell::stop ()
  {
    if (m_selection->m_active_source != SOURCE_NONE)
      ::play->request_status (PLAYSTATUS_STOPPED);
  }

  void
  PlayerShell::on_play_status_changed ()
  {
    BmpPlaystatus status = BmpPlaystatus (::play->property_status().get_value());
    PlaybackSource::Caps caps = m_source_caps[m_selection->m_active_source];

    Bmp::URI u;

    switch (status)
    {
      case PLAYSTATUS_STOPPED:
        m_seeking = false;
        if (Network::check_connected())
        {
          m_actions->get_action (PLAYER_SHELL_ACTION_RECOMMEND_LAST_FM)->set_sensitive (false);
        }
        m_actions->get_action (PLAYER_SHELL_ACTION_LASTFM)->set_sensitive (false);

        if (m_ui_merge_id_context)
          m_ui_manager->remove_ui (m_ui_merge_id_context);

        if (m_ui_merge_id_tray)
          m_ui_manager->remove_ui (m_ui_merge_id_tray);

        break;

      case PLAYSTATUS_PLAYING:
        m_seeking = false;
        m_actions->get_action (PLAYER_SHELL_ACTION_STOP)->set_sensitive (true);
        break;

      default: ;
    }

    switch (status)
    {
        case PLAYSTATUS_NONE:
          /*  ... */
          break;

        case PLAYSTATUS_SEEKING:
          m_yy_image->set (yy_seeking);
          m_yy_image->show ();
          break;

        case PLAYSTATUS_STOPPED:
        {
          if (m_selection->m_active_source != SOURCE_NONE)
          {
            m_sources[m_selection->m_active_source]->stop ();
            m_sources[m_selection->m_active_source]->send_caps ();
          }

          m_blackBar->set_source (RefPixbuf(0));
          m_yy_image->set (yy_stopped);
          m_yy_image->hide ();
          m_blackBar->reset ();
          m_asin.reset ();
          m_metadata = TrackMetadata();
      
          if (m_popup)
          {
            m_popup->hide ();
          }

          m_selection->source_activate (SOURCE_NONE);
          bmp_status_icon_set_from_pixbuf (m_status_icon, m_status_icon_image_default->gobj());
          reset_seek_range ();

          m_actions->get_action (PLAYER_SHELL_ACTION_RECOMMEND_LAST_FM)->set_sensitive (false);
          m_actions->get_action (PLAYER_SHELL_ACTION_STOP)->set_sensitive (false);
          m_actions->get_action (PLAYER_SHELL_ACTION_PAUSE)->set_sensitive (false);
          m_actions->get_action (PLAYER_SHELL_ACTION_TRACK_DETAILS)->set_sensitive (false);
          m_actions->get_action (PLAYER_SHELL_ACTION_CREATE_BOOKMARK)->set_sensitive (false);
          break;
        }

        case PLAYSTATUS_WAITING:
        {
          reset_seek_range ();
          bmp_status_icon_set_from_pixbuf (m_status_icon, m_status_icon_image_default->gobj());
          m_yy_image->set (yy_paused);
          break;
        }

        case PLAYSTATUS_PLAYING:
        {
          bmp_status_icon_set_from_pixbuf (m_status_icon, m_status_icon_image_playing->gobj());
          m_yy_image->clear ();
          m_yy_image->show ();
          m_spm_paused = false;
          break;
        }

        case PLAYSTATUS_PAUSED:
        {
          bmp_status_icon_set_from_pixbuf (m_status_icon, m_status_icon_image_paused->gobj());
          m_yy_image->set (yy_paused);
          m_spm_paused = false;
          break;
        }
    }

    if (m_popup)
    {
      m_popup->set_playstatus (status);
    }

    m_player_dbus_obj->emit_status_change (m_player_dbus_obj, int (status));
  }

  bool
  PlayerShell::on_window_state_event (GdkEventWindowState *event)
  {
    if (!event->changed_mask & GDK_WINDOW_STATE_ICONIFIED)
      return false;

    if (m_popup && mcs->key_get<bool>("bmp", "ui-esc-trayconify"))
    {
      if (event->new_window_state & GDK_WINDOW_STATE_ICONIFIED)
      {
          set_skip_taskbar_hint (true);
          set_skip_pager_hint (true);
          hide ();
          m_visible = false;
      }
      else
      {
          set_skip_taskbar_hint (false);
          set_skip_pager_hint (false);
          show ();
          present ();
          m_visible = true;
      }
    }
    return false;
  }

  bool
  PlayerShell::on_key_press_event (GdkEventKey *event)
  {
    if (event->keyval == GDK_Escape)
    {
      iconify ();
      return true;
    }
    return Widget::on_key_press_event (event);
  }

  PlayerShell*
  PlayerShell::create ()
  {
    const string path (BMP_GLADE_DIR G_DIR_SEPARATOR_S "main.glade");
    RefPtr<Gnome::Glade::Xml> glade_xml = Gnome::Glade::Xml::create (path);
    PlayerShell *p = 0;
    glade_xml->get_widget_derived ("main-ui", p);
    return p;
  }

  PlayerShell::~PlayerShell ()
  {
    g_object_unref (m_status_icon);
    delete mcs_bind;
    delete m_track_details;
    delete m_about;
  }

  GHashTable*
  PlayerShell::get_metadata_hash_table ()
  {
    if (m_selection->m_active_source == SOURCE_NONE)
      return NULL;

    GHashTable * table = m_sources[m_selection->m_active_source]->get_metadata(); 
    if (table)
      return table;

    // Nope.

    table = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, GDestroyNotify (Util::free_gvalue_ptr));
    GValue* value = 0;

    if (m_metadata.bitrate)
    {
      value = g_new0 (GValue,1);
      g_value_init (value, G_TYPE_INT);
      g_value_set_int (value, int(m_metadata.bitrate.get()));
      g_hash_table_insert (table, g_strdup(get_attribute_info(ATTRIBUTE_BITRATE).id), value);
    }

    if (m_metadata.duration)
    {
      value = g_new0 (GValue,1);
      g_value_init (value, G_TYPE_INT);
      g_value_set_int (value, int(m_metadata.duration.get()/1000));
      g_hash_table_insert (table, g_strdup(get_attribute_info(ATTRIBUTE_TIME).id), value);
    }

    if (m_metadata.artist)
    {
      value = g_new0 (GValue,1);
      g_value_init (value, G_TYPE_STRING);
      g_value_set_string (value, m_metadata.artist.get().c_str());
      g_hash_table_insert (table, g_strdup(get_attribute_info(ATTRIBUTE_ARTIST).id), value);
    }

    if (m_metadata.mb_artist_id)
    {
      value = g_new0 (GValue,1);
      g_value_init (value, G_TYPE_STRING);
      g_value_set_string (value, m_metadata.mb_artist_id.get().c_str());
      g_hash_table_insert (table, g_strdup(get_attribute_info(ATTRIBUTE_MB_ARTIST_ID).id), value);
    }

    if (m_metadata.album)
    {
      value = g_new0 (GValue,1);
      g_value_init (value, G_TYPE_STRING);
      g_value_set_string (value, m_metadata.album.get().c_str());
      g_hash_table_insert (table, g_strdup(get_attribute_info(ATTRIBUTE_ALBUM).id), value);
    }

    if (m_metadata.mb_album_id)
    {
      value = g_new0 (GValue,1);
      g_value_init (value, G_TYPE_STRING);
      g_value_set_string (value, m_metadata.mb_album_id.get().c_str());
      g_hash_table_insert (table, g_strdup(get_attribute_info(ATTRIBUTE_MB_ALBUM_ID).id), value);
    }

    if (m_metadata.title)
    {
      value = g_new0 (GValue,1);
      g_value_init (value, G_TYPE_STRING);
      g_value_set_string (value, m_metadata.title.get().c_str());
      g_hash_table_insert (table, g_strdup(get_attribute_info(ATTRIBUTE_TITLE).id), value);
    }

    if (m_metadata.mb_track_id)
    {
      value = g_new0 (GValue,1);
      g_value_init (value, G_TYPE_STRING);
      g_value_set_string (value, m_metadata.mb_track_id.get().c_str());
      g_hash_table_insert (table, g_strdup(get_attribute_info(ATTRIBUTE_MB_TRACK_ID).id), value);
    }

    if (m_metadata.asin)
    {
      value = g_new0 (GValue,1);
      g_value_init (value, G_TYPE_STRING);
      g_value_set_string (value, m_metadata.asin.get().c_str());
      g_hash_table_insert (table, g_strdup(get_attribute_info(ATTRIBUTE_ASIN).id), value);
    }

    std::string genre;

    if (m_metadata.genre)
    {
      genre = m_metadata.genre.get();
    }
    else if (m_selection->m_active_source != SOURCE_NONE)
    {
      genre = m_sources[m_selection->m_active_source]->get_name();
    }

    value = g_new0 (GValue,1);
    g_value_init (value, G_TYPE_STRING);
    g_value_set_string (value, genre.c_str()); 
    g_hash_table_insert (table, g_strdup(get_attribute_info(ATTRIBUTE_GENRE).id), value);

    return table;
  }

  void
  PlayerShell::raise ()
  {
    if (!m_visible)
    {
      deiconify ();
      present ();
    }
  }

  void
  PlayerShell::on_toggle_menubar ()
  {
    Gtk::Window * window = dynamic_cast<Gtk::Window*>(m_ref_xml->get_widget ("main-ui"));
    int w, h;

    bool active (RefPtr<ToggleAction>::cast_static (m_actions->get_action (PLAYER_SHELL_ACTION_MENUBAR))->get_active());
    if (active)
    {
      Gtk::Allocation alloc = m_ref_xml->get_widget ("alignment-menu-default")->get_allocation ();
      window->get_size (w, h);

      m_ref_xml->get_widget ("alignment-menu-default")->hide ();
      window->resize (w, h - alloc.get_height());
    }
    else
    {
      m_ref_xml->get_widget ("alignment-menu-default")->show ();
    }
  }

  void
  PlayerShell::on_toggle_statusbar ()
  {
    Gtk::Window * window = dynamic_cast<Gtk::Window*>(m_ref_xml->get_widget ("main-ui"));
    int w, h;

    bool active (RefPtr<ToggleAction>::cast_static (m_actions->get_action (PLAYER_SHELL_ACTION_STATUSBAR))->get_active());
    if (active)
    {
      m_ref_xml->get_widget ("hbox-status")->show ();
    }
    else
    {
      Gtk::Allocation alloc = m_ref_xml->get_widget ("hbox-status")->get_allocation ();
      window->get_size (w, h);
      m_ref_xml->get_widget ("hbox-status")->hide ();
      window->resize (w, h - alloc.get_height());
    }
  }

  void
  PlayerShell::on_toggle_compact ()
  {
    Gtk::Window * window = dynamic_cast<Gtk::Window*>(m_ref_xml->get_widget ("main-ui"));
    int w, h;

    bool active (RefPtr<ToggleAction>::cast_static (m_actions->get_action (PLAYER_SHELL_ACTION_COMPACT))->get_active());

    if (active)
    {
      Gtk::Allocation alloc = m_ref_xml->get_widget ("vbox-sources")->get_allocation ();
      window->get_size (w, h);
      m_ref_xml->get_widget ("vbox-sources")->hide ();
      m_ref_xml->get_widget ("vbox-ui-bottom")->hide ();
      window->resize (w, h - alloc.get_height());
    }
    else
    {
      m_ref_xml->get_widget ("vbox-ui-bottom")->show ();
      m_ref_xml->get_widget ("vbox-sources")->show ();
    }

    m_blackBar->set_compact (active);
  }


  ////////////////////////////////////////
  // BOOKMARKS API
  ////////////////////////////////////////
 
  void  
  PlayerShell::bookmark_append (Bookmark const& bookmark, uint64_t b_id)
  {
    static boost::format name_f ("bookmark-%llu");  
    static boost::format path_f ("/MenuBarMain/MenuBookmarks/MenuBookmarks-%s");  
    ustring name = (name_f % b_id).str();
    m_actions->add (Action::create (name, Gtk::StockID (sources[bookmark.m_id].stock), bookmark.m_title), 
                                    sigc::bind (sigc::mem_fun (*this, &PlayerShell::on_bookmark_activate), bookmark.m_url));
    m_ui_manager->add_ui (bookmark.m_merge_id,
                          (path_f % sources[bookmark.m_id].id).str(), 
                          bookmark.m_title, 
                          name, 
                          Gtk::UI_MANAGER_AUTO,
                          false);
  }

  void
  PlayerShell::on_bookmark_loaded (Bookmark const& b1)
  {
    Bookmark b = b1; 
    b.m_merge_id = m_ui_manager->new_merge_id ();
    uint64_t b_id = m_bookmark_manager.add_bookmark (b);
    bookmark_append (b, b_id);
  }

  void 
  PlayerShell::on_bookmark_create ()
  {
    PlaybackSourceID id = PlaybackSourceID (m_selection->m_active_source);

    if (id != SOURCE_NONE)
    {
          PlaybackSource::Caps caps (m_source_caps[id]);
          if (caps & PlaybackSource::CAN_BOOKMARK)
          {
                BookmarkHandler * handler = dynamic_cast<BookmarkHandler*>(m_sources[id].get());

                ustring url, title;
                handler->get_bookmark (sources[id].id, url, title);

                UIManager::ui_merge_id merge_id = m_ui_manager->new_merge_id ();
                Bookmark b (title, id, url, merge_id);

                uint64_t b_id = m_bookmark_manager.add_bookmark (b);
                bookmark_append (b, b_id);
          }
    }
  }

  void
  PlayerShell::on_bookmark_activate (std::string const& url)
  {
    Bmp::URI u (url);
    BookmarkHandler * handler = 0; 

    if (u.hostname == "radio")
    {
          handler = dynamic_cast<BookmarkHandler*>(m_sources[SOURCE_RADIO].get());
    }

    if (u.hostname == "jamendo")
    {
          handler = dynamic_cast<BookmarkHandler*>(m_sources[SOURCE_JAMENDO].get());
    }

    if (handler)
    {
          handler->set_bookmark (url);
    }
  }
}
