//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2007 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <glibmm.h>
#include <glibmm/i18n.h>
#include <glib/gstdio.h>
#include <gtkmm.h>
#include <libglademm.h>

#include <cstring>
#include <iostream>
#include <sstream>
#include <string>

#include <boost/algorithm/string.hpp>
#include <boost/format.hpp>

#include "base64.h"
#include "sha1.h"

// BMP Audio
#include "audio/audio.hh"

// BMP Musicbrainz
#include "musicbrainz/mbxml-v2.hh"

// BMP Widgets
#include "widgets/taskdialog.hh"

// BMP Misc
#include "debug.hh"
#include "main.hh"
#include "network.hh"
#include "paths.hh"
#include "stock.hh"
#include "ui-tools.hh"
#include "uri++.hh"
#include "util.hh"
#include "util-file.hh"

#include "x_amazon.hh"
#include "x_core.hh"
#ifdef HAVE_HAL
#  include "x_hal.hh"
#endif //HAVE_HAL
#include "x_library.hh"

#include "ui-part-cdda.hh"

extern "C"
{
  #include <cdda_interface.h>
  namespace { const char * _dummy = strerror_tr[0]; }
}

using namespace Gtk;
using namespace Glib;
using namespace std;

using namespace Bmp::MusicBrainzXml;
using namespace Bmp::Audio;

namespace
{
  const char *unknown_artist = N_("Unknown Artist");
  const char *unknown_album  = N_("Unknown Album");
  const char *unknown_title  = N_("Unknown Title");

  int
  get_duration_from_sectors (int sectors)
  {
    const int bytes_per_sector  = 2352;
    const int bytes_per_seconds = (44100 / 8) / 16 / 2;
    return (sectors * bytes_per_sector / bytes_per_seconds)/1000;
  }

  void
  get_data_track_mapping (cdrom_drive * d, Bmp::BoolV & tracks, int & n_tracks)
  {
    for (int n = 0; n < d->tracks; ++n)
    {
      tracks.push_back (IS_AUDIO(d, n) ? true : false);
    }
    n_tracks = d->tracks;
  }

  void 
  get_discid (cdrom_drive * d, std::string & disc_id)
  {
    SHA_INFO        sha;
    unsigned char   digest[20], *base64;
    unsigned long   size;
    char            tmp[17]; /* for 8 hex digits (16 to avoid trouble) */
    char            buf[33];

    sha_init(&sha);
    sprintf(tmp, "%02X", d->disc_toc[0].bTrack);
    sha_update(&sha, (unsigned char *) tmp, strlen(tmp));

    sprintf(tmp, "%02X", d->disc_toc[d->tracks-1].bTrack);
    sha_update(&sha, (unsigned char *) tmp, strlen(tmp));

    int track_offsets[100];
    for (int n = 0; n < 100; track_offsets[n++] = 0);

    track_offsets[0] = d->disc_toc[d->tracks].dwStartSector + 150;

    for (int n = d->disc_toc[0].bTrack-1; n <= d->disc_toc[d->tracks-1].bTrack-1; ++n) 
    {
      track_offsets[n+1] = d->disc_toc[n].dwStartSector + 150;
    }

    for (int i = 0; i < 100; i++)
    {
      sprintf(tmp, "%08X", track_offsets[i]);
      sha_update(&sha, (unsigned char *) tmp, strlen(tmp));
    }

    sha_final(digest, &sha);
    base64 = rfc822_binary (digest, sizeof(digest), &size);
    memcpy(buf, base64, size);
    buf[size] = '\0';
    free(base64);
    disc_id = buf;
  }

  const char *processor_names[] =
  {
    "MP3 (MPEG 1 Layer III)",
    "Ogg Vorbis",
    "FLAC Lossless Encoding"
  };

  const char *suffixes[] =
  {
    ".mp3",
    ".ogg",
    ".flac"
  };

  static boost::format track_f ("%02d");
  static boost::format cdda_uri_f ("cdda:///%d");

}

namespace
{
  using namespace Bmp;
  const PlaybackSource::Flags flags =
    PlaybackSource::Flags (PlaybackSource::F_ALWAYS_IMAGE_FRAME | PlaybackSource::F_HANDLE_LASTFM |
                           PlaybackSource::F_USES_REPEAT | PlaybackSource::F_USES_SHUFFLE);
}

namespace Bmp
{
  namespace UiPart
  {
    CDDA::CDDA (RefPtr<Gnome::Glade::Xml> const& xml, RefPtr<UIManager> ui_manager)
    : PlaybackSource  (_("Audio CD"), CAN_SEEK, flags) 
    , Base            (xml, ui_manager)
    , m_cover_default (Gdk::Pixbuf::create_from_file (build_filename                                                                                          (BMP_IMAGE_DIR, BMP_COVER_IMAGE_DEFAULT))->scale_simple (128, 128, Gdk::INTERP_BILINEAR))
    , m_anonymous_cd  (false)
    , m_processing_cd (false)
    , m_stream_eos    (false)
    {
        m_ref_xml->get_widget ("notebook-cdda", m_notebook_cdda);
        m_ref_xml->get_widget ("cdda-label-current-artist", m_label_artist);
        m_ref_xml->get_widget ("cdda-label-current-album", m_label_album);
        m_ref_xml->get_widget ("cdda-label-current-date", m_label_date);
        m_ref_xml->get_widget ("cdda-image-cover", m_cover_image);

        Audio::Caps audiocaps (Bmp::Audio::get_caps());
        if( !(audiocaps & Audio::CAPS_CDDA) )
        {
          dynamic_cast<Image *>(m_ref_xml->get_widget ("i-audiocd-bad-2"))->set
                          (BMP_IMAGE_DIR_CDDA G_DIR_SEPARATOR_S "audiocd-unplayable.png");
          m_notebook_cdda->set_current_page (PAGE_NO_SUPPORT);
          m_ref_xml->get_widget ("hbox-cdda-controls")->hide ();
          return;
        }

        m_ref_xml->get_widget ("b-cdda-rip", m_b_cdda_rip);
        m_ref_xml->get_widget ("b-cdda-reload", m_b_cdda_reload);
        m_ref_xml->get_widget ("b-cdda-stop", m_b_cdda_stop);
        m_ref_xml->get_widget ("cdda-cbox-format", m_cdda_cbox_format);
        m_ref_xml->get_widget ("cdda-cbox-quality", m_cdda_cbox_quality);

        m_cdda_cbox_quality->set_active (1);
        m_cdda_cbox_format->clear ();
        CellRendererText *cell = manage (new CellRendererText());
        m_cdda_cbox_format->pack_start (*cell);
        m_cdda_cbox_format->add_attribute (*cell, "text", 0);
        cbox_processors_store = ListStore::create (processors);
        m_cdda_cbox_format->set_model (cbox_processors_store);

        for (int n = 0; n < N_PROCS; ++n)
        {
          if( Audio::ProcessorFactory::test_processor (ProcessorType (n)) )
          {
            TreeModel::iterator iter (cbox_processors_store->append ());
            (*iter)[processors.name] = processor_names[n];
            (*iter)[processors.type] = ProcessorType (n);
          }
        }

        dynamic_cast<Image *>(m_ref_xml->get_widget ("throbber-cdda"))->set
                        (BMP_IMAGE_DIR G_DIR_SEPARATOR_S BMP_THROBBER);

        dynamic_cast<Image *>(m_ref_xml->get_widget ("i-audiocd"))->set
                        (BMP_IMAGE_DIR_CDDA G_DIR_SEPARATOR_S "audiocd-big.png");

        dynamic_cast<Image *>(m_ref_xml->get_widget ("i-audiocd-bad"))->set
                        (BMP_IMAGE_DIR_CDDA G_DIR_SEPARATOR_S "audiocd-unplayable.png");

        const char *headers[] =
        {
           N_("Track"),
           N_("Artist"),
           N_("Title"),
           N_("Duration"),
           N_("Progress"),
           N_("State"),
        };

        // --- CDDA View+Store
        m_ref_xml->get_widget ("cdda-view", m_view);
        m_view->get_selection()->set_mode (SELECTION_SINGLE);

        TreeViewColumn *column = 0;

        m_cell_playing = manage (new CellRendererPixbuf());
        m_cell_playing->property_xalign() = 0.5;

        m_view->append_column ("", *m_cell_playing);

        column = m_view->get_column (0);
        column->set_resizable (false);
        column->set_expand (false);
        column->property_min_width() = 30;
        column->property_max_width() = 30;
        column->set_cell_data_func (*m_cell_playing, sigc::mem_fun (*this, &Bmp::UiPart::CDDA::cell_data_func_playing));

        for (unsigned int n = 0; n < G_N_ELEMENTS(headers); ++n)
        {
          if( n == 3 )
            {
              CellRendererText *cell = manage ( new CellRendererText() );

              m_view->append_column (_(headers[n]), *cell);

              column = m_view->get_column (n+1);
              column->set_resizable (false);
              column->set_expand (false);
              column->set_cell_data_func (*cell, sigc::mem_fun (*this, &Bmp::UiPart::CDDA::cell_data_func_numbers));
            }
          else
          if( n == 4 )
            {
              CellRendererProgress *cell = manage ( new CellRendererProgress() );

              m_view->append_column (_(headers[n]), *cell);

              column = m_view->get_column (n+1);
              column->add_attribute (*cell, "value", n);
              column->set_resizable (false);
              column->set_expand (false);
              column->set_fixed_width (100);
              column->set_sizing (TREE_VIEW_COLUMN_FIXED);
            }
          else
            {
              CellRendererText *cell = manage ( new CellRendererText() );
              if (n == 0) cell->property_xalign() = 1.0;

              m_view->append_column (_(headers[n]), *cell);
              if( n != 5 )
              {
                column = m_view->get_column (n+1);
                column->add_attribute (*cell, "markup", n);
                column->set_resizable (false);
                column->set_expand (false);
              }
              else
              {
                column = m_view->get_column (n+1);
                column->set_resizable (false);
                column->set_expand (true);
                column->set_cell_data_func (*cell, sigc::mem_fun (*this, &Bmp::UiPart::CDDA::cell_data_func_state));
              }
            }
        }

        m_view->get_column (5)->set_visible (false);
        m_view->get_column (6)->set_visible (false);

        m_store = ListStore::create (audiocd);
        m_view->set_model (m_store);
        m_view->get_selection()->signal_changed().connect
            (sigc::mem_fun (*this, &Bmp::UiPart::CDDA::selection_changed));

        if( cbox_processors_store->children().size() )
        {
          m_cdda_cbox_format-> set_active (0);
#ifdef HAVE_HAL
          m_b_cdda_reload->hide();
          hal->signal_cdda_inserted().connect
              (sigc::mem_fun (*this, &Bmp::UiPart::CDDA::on_hal_cdda_inserted));
          hal->signal_device_removed().connect
              (sigc::mem_fun (*this, &Bmp::UiPart::CDDA::on_hal_device_removed));
          hal->signal_ejected().connect
              (sigc::mem_fun (*this, &Bmp::UiPart::CDDA::on_hal_ejected));
#else
          m_b_cdda_reload->signal_clicked().connect
              (sigc::mem_fun (*this, &Bmp::UiPart::CDDA::refresh));
#endif //HAVE_HAL

          m_b_cdda_rip->signal_clicked().connect
              (sigc::mem_fun (*this, &Bmp::UiPart::CDDA::rip_start));
          m_b_cdda_stop->signal_clicked().connect
              (sigc::mem_fun (*this, &Bmp::UiPart::CDDA::rip_stop));

          m_b_cdda_rip->set_sensitive (false);
#ifdef HAVE_HAL
          m_notebook_cdda->set_current_page (PAGE_CD_INSERT);
#else
          m_notebook_cdda->set_current_page (PAGE_TOC);
#endif //HAVE_HAL
        }
      else
        {
          m_cdda_cbox_format->set_sensitive (false);
          m_cdda_cbox_quality->set_sensitive (false);

#ifdef HAVE_HAL
          m_b_cdda_reload->hide();
          m_b_cdda_reload->set_sensitive (false);
#endif //HAVE_HAL

          m_b_cdda_rip->set_sensitive (false);
          m_view->set_sensitive (false);
          m_notebook_cdda->set_current_page (PAGE_TOC);
        }

        m_view->signal_row_activated().connect
              (sigc::mem_fun (*this, &Bmp::UiPart::CDDA::activate_default));

        core->signal_shutdown_request().connect
              (sigc::mem_fun (*this, &Bmp::UiPart::CDDA::on_shutdown_request));
    }

    guint
    CDDA::add_ui ()
    {
      return 0;
    }

    CDDA::~CDDA () {}

    bool
    CDDA::on_shutdown_request ()
    {
      using namespace Gtk;

      if( processing_cd() )
      {
        MessageDialog dialog ((*(dynamic_cast<Window *>(m_ref_xml->get_widget ("main-ui")))),
                                _("<b>Ripping in Progress: Can't Shutdown!</b>\n\nBMP currently cannot shut down because a "
                                  "CD is being ripped. "
                                  "Please either abort the encoding process, or wait until it is finished to exit BMP."),
                                  true, MESSAGE_WARNING, BUTTONS_OK, true);
        dialog.set_title (_("Ripping CD: Can't Shutdown - BMP"));
        dialog.set_position (WIN_POS_CENTER);
        dialog.run();
        return false;
      }
      else
      {
        return true;
      }
    }

    void
    CDDA::cell_data_func_numbers (CellRenderer* cell, TreeModel::iterator const& iter)
    {
      static boost::format fduration ("%d:%02d");
      CellRendererText *cell_text = dynamic_cast<CellRendererText*>(cell);
      unsigned int duration = (*iter)[audiocd.duration];
      unsigned int minutes = duration / 60;
      unsigned int seconds = duration % 60;
      cell_text->property_text() = (fduration % minutes % seconds).str();
    }

    void
    CDDA::activate_default (TreeModel::Path const& path, TreeViewColumn* column)
    {
      assign_current_iter (m_store->get_iter (path));
      s_playback_request_.emit ();
    }

    void
    CDDA::populate_data (void * _d, MusicBrainzReleaseV const& releases, BoolV const& audiotracks)
    {
      cdrom_drive * d = static_cast<cdrom_drive*>(_d);
      if( releases.empty() )
      {
        m_label_artist->set_markup ((boost::format ("<b>%s</b>") % _(unknown_artist)).str());
        m_label_album->set_markup ((boost::format ("<i>%s</i>") % _(unknown_album)).str());
        m_label_date->set_text ("");
        m_cover_image->set (m_cover_default);

        for (int n = d->disc_toc[0].bTrack; n <= d->disc_toc[d->tracks-1].bTrack; ++n)
        {
          TreeModel::iterator iter = m_store->append ();

          (*iter)[audiocd.tracknumber]  = n;
          (*iter)[audiocd.artist]       = _(unknown_artist);
          (*iter)[audiocd.duration]     = get_duration_from_sectors (cdda_track_lastsector (d, n) - cdda_track_firstsector (d, n));
          (*iter)[audiocd.audio]        = audiotracks[n];

          if( audiotracks[n-1] )
          {
            (*iter)[audiocd.title] = (boost::format ("Track %02d") % (n+1)).str();
          }
          else
          {
            (*iter)[audiocd.title] = _("(Data Track)");
          }
        }

        m_anonymous_cd = true;
        m_b_cdda_rip->set_sensitive (false);
        m_cdda_cbox_format->set_sensitive (false);
        m_cdda_cbox_quality->set_sensitive (false);
      }
      else
      {
        m_release = releases[0];
        m_label_artist->set_markup ((boost::format ("<b>%s</b>") %
                        Markup::escape_text (!m_release.mArtist.artistSortName.empty()
                                             ? m_release.mArtist.artistSortName
                                             : m_release.mArtist.artistName).c_str()).str());

        m_label_album->set_markup ((boost::format ("<i>%s</i>") %
                        Markup::escape_text (m_release.releaseTitle).c_str()).str());

        if( !m_release.mReleaseEventV.empty() )
          m_label_date->set_text (Util::get_date_string (m_release.mReleaseEventV[0].releaseEventDate));
        else
          m_label_date->set_text ("");

        if( !m_release.releaseASIN.empty() )
        {
          RefPtr<Gdk::Pixbuf> cover;
          amazon->fetch (m_release.releaseASIN, cover, false);
          m_cover_image->set (cover->scale_simple (128, 128, Gdk::INTERP_BILINEAR));
        }
        else
        {
          m_cover_image->set (m_cover_default);
        }

        unsigned int i = d->disc_toc[0].bTrack;
        MusicBrainzTrackV const& mbtracks (m_release.mTrackV);
        for (MusicBrainzTrackV::const_iterator n = mbtracks.begin(); n != mbtracks.end(); ++n)
        {
          TreeModel::iterator iter = m_store->append ();

          if( n->trackTrackNumber != i )
          {
            g_message ("%s: CDParanoia says track is %d, but release has it as track %llu", G_STRFUNC, i, n->trackTrackNumber);
          }

          (*iter)[audiocd.tracknumber]  = n->trackTrackNumber;
          (*iter)[audiocd.artist]       = Markup::escape_text (n->artistName);
          (*iter)[audiocd.duration]     = get_duration_from_sectors (cdda_track_lastsector (d, i) - cdda_track_firstsector (d, i));
          (*iter)[audiocd.percentage]   = 0;
          (*iter)[audiocd.state]        = ES_UNPROCESSED;

          if( audiotracks[i-1] )
            {
              (*iter)[audiocd.title] = Markup::escape_text (n->trackTitle);
              (*iter)[audiocd.track] = *n;
            }
          else
            {
              (*iter)[audiocd.title] = _("(Data Track)");
            }

          (*iter)[audiocd.audio] = audiotracks[i-1];
          ++i;
        }

        m_anonymous_cd = false;
        m_b_cdda_rip->set_sensitive (true);
        m_cdda_cbox_format->set_sensitive (true);
        m_cdda_cbox_quality->set_sensitive (true);
      }
    }

    void
    CDDA::clear_display ()
    {
      m_store->clear();
      m_label_artist->set_text ("");
      m_label_album->set_text ("");
      m_label_date->set_text ("");
      m_cover_image->clear();
      m_b_cdda_rip->set_sensitive (false);
      m_b_cdda_stop->set_sensitive (false);
      m_cdda_cbox_format->set_sensitive (false);
      m_cdda_cbox_quality->set_sensitive (false);
    }

    void
    CDDA::set_page (int page)
    {
      m_notebook_cdda->set_current_page (page);
      while (gtk_events_pending()) gtk_main_iteration ();
    }

#ifdef HAVE_HAL
    void
    CDDA::on_hal_cdda_inserted (string udi, string devicefile)
    {
      if( !processing_cd() )
      {
        m_current_udi = udi;
        m_current_device = devicefile;
        mcs->key_set ("audio", "cdrom-device", devicefile);

        clear_display ();
        set_page (PAGE_LOADING);

        MusicBrainzReleaseV releases;
        BoolV               audiotracks;
        int                 n_tracks = 0;
        std::string         disc_id;

        cdrom_drive *d = 0;

        d = cdda_identify (devicefile.c_str(), 0, 0);

        if( !d )
        {
          g_warning ("%s: cdda_identify", G_STRLOC);
          set_page (PAGE_CD_BAD);
          return;
        }

        if( cdda_open (d) )
        {
          g_warning ("%s: cdda_open", G_STRLOC);
          set_page (PAGE_CD_BAD);
          return;
        }

        d->read_toc (d);

        get_data_track_mapping (d, audiotracks, n_tracks);
        get_discid (d, disc_id);

        if( disc_id.size() )
        {
          mb_releases_by_disc_id (disc_id, releases);
        }

        populate_data (d, releases, audiotracks);
        if (d) cdda_close (d);

        set_page (PAGE_TOC);
      }
    }

    void
    CDDA::on_hal_ejected (string udi)
    {
      if( udi == m_current_udi )
      {
        if( processing_cd() )
          rip_stop ();
        else
          s_stop_request_.emit ();
      }
    }

    void
    CDDA::on_hal_device_removed (string udi)
    {
      if( (!processing_cd()) && udi == m_current_udi )
      {
        m_store->clear ();
        m_label_artist->set_text ("");
        m_label_album->set_text ("");
        m_label_date->set_text ("");
        m_cover_image->clear();
        m_b_cdda_rip->set_sensitive (false);
        m_notebook_cdda->set_current_page (PAGE_CD_INSERT);
        m_cdda_cbox_format->set_sensitive (false);
        m_cdda_cbox_quality->set_sensitive (false);
        m_view->get_column (5)->set_visible (false);
        m_view->get_column (6)->set_visible (false);
      }
    }

#else //!HAVE_HAL

    void
    CDDA::refresh ()
    {
      if( !processing_cd() )
      {
        clear_display ();
        set_page (PAGE_LOADING);

        MusicBrainzReleaseV releases;
        BoolV               audiotracks;
        int                 n_tracks = 0;
        std::string         disc_id;

        cdrom_drive *d = 0;

        d = cdda_identify (mcs->key_get<string>("audio", "cdrom-device").c_str(), 0, 0);

        if( !d )
        {
          g_warning ("%s: cdda_identify", G_STRLOC);
          set_page (PAGE_CD_BAD);
          return;
        }

        if( cdda_open (d) )
        {
          g_warning ("%s: cdda_open", G_STRLOC);
          set_page (PAGE_CD_BAD);
          return;
        }

        d->read_toc (d);

        get_data_track_mapping (d, audiotracks, n_tracks);
        get_discid (d, disc_id);

        if( disc_id.size() )
        {
          mb_releases_by_disc_id (disc_id, releases);
        }

        populate_data (d, releases, audiotracks);
        if (d) cdda_close (d);

        set_page (PAGE_TOC);
      }
   }
#endif //HAVE_HAL

    void
    CDDA::rip_stop ()
    {
      m_mutex_rip.unlock ();
    }

    void
    CDDA::rip_start ()
    {
      if( processing_cd() )
        return;

      stop ();

      unsigned int current_rip_track;
      std::string  current_rip_basepath;

      // Disable Controls
      m_b_cdda_rip->set_sensitive (false);
      m_cdda_cbox_format->set_sensitive (false);
      m_cdda_cbox_quality->set_sensitive (false);
      set_processing_cd (true);

      // Select the Destination Folder
      FileChooserDialog file_chooser_dialog (_("Select Destination Folder - BMP"), FILE_CHOOSER_ACTION_SELECT_FOLDER);
      file_chooser_dialog.add_button (Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
      file_chooser_dialog.add_button (Gtk::Stock::OK, Gtk::RESPONSE_OK);
      file_chooser_dialog.set_default_response (Gtk::RESPONSE_OK);
      file_chooser_dialog.set_current_folder (mcs->key_get<std::string>("bmp", "file-chooser-path"));

      if( file_chooser_dialog.run () == Gtk::RESPONSE_CANCEL )
      {
        file_chooser_dialog.hide ();
        m_b_cdda_rip->set_sensitive (true);
        m_cdda_cbox_format->set_sensitive (true);
        m_cdda_cbox_quality->set_sensitive (true);
        set_processing_cd (false);
        return;
      }

      file_chooser_dialog.hide ();
      current_rip_basepath = filename_from_uri (file_chooser_dialog.get_current_folder_uri());

      for (TreeNodeChildren::iterator iter  = m_store->children().begin()  ;
                                      iter != m_store->children().end()    ; ++iter)
      {
        (*iter)[audiocd.state] = ES_WAITING;
      }

      m_mutex_rip.lock ();
      m_b_cdda_stop->set_sensitive (true);

#ifndef HAVE_HAL
      m_b_cdda_reload->set_sensitive (false);
#endif //HAVE_HAL

      m_view->get_column (5)->set_visible (true);
      m_view->get_column (6)->set_visible (true);

      m_view->get_selection()->unselect_all ();
      m_view->get_selection()->set_mode (SELECTION_NONE);

      m_caps = Caps (m_caps & ~PlaybackSource::CAN_PLAY);
      s_caps_.emit (m_caps);

      TreeModel::iterator c_iter = m_cdda_cbox_format->get_active ();
      ProcessorType processor_type = (*c_iter)[processors.type];

      for (unsigned int n = 0; n < m_release.mTrackV.size(); ++n)
      {
        current_rip_track = n;

        TreePath path;
        path.append_index (current_rip_track);

        m_current_rip_iter = m_store->get_iter (path);

        if( (*m_current_rip_iter)[audiocd.state] == ES_DONE )
          continue;

        MusicBrainzTrack const& track = (*m_current_rip_iter)[audiocd.track];

        ustring filename;
        filename  .append ((track_f % (n+1)).str())
                  .append (" - ")
                  .append (m_release.mArtist.artistName)
                  .append (" - ")
                  .append (m_release.releaseTitle)
                  .append (" - ")
                  .append (track.trackTitle);

        filename.append (suffixes[processor_type]);

        Audio::ProcessorBase *processor = 0;
        string rip_basename = filename_from_utf8 (filename);
        boost::algorithm::replace_all (rip_basename, "/" , "-");
        string rip_filename = build_filename (current_rip_basepath, rip_basename);
        ustring rip_uri = filename_to_uri (rip_filename);
        (*m_current_rip_iter)[audiocd.location] = rip_uri;

#ifdef HAVE_HAL
        std::string device (m_current_device);
#else
        std::string device (mcs->key_get<string>("audio", "cdrom-device"));
#endif //HAVE_HAL

        processor = Audio::ProcessorFactory::get_processor (processor_type, rip_filename, current_rip_track+1, device,
          m_cdda_cbox_quality->get_active_row_number());

        processor->signal_position().connect
          (sigc::mem_fun (*this, &Bmp::UiPart::CDDA::stream_position));
        processor->signal_eos().connect
          (sigc::mem_fun (*this, &Bmp::UiPart::CDDA::stream_eos));
        processor->signal_error().connect
          (sigc::mem_fun (*this, &Bmp::UiPart::CDDA::stream_error));

        set_eos (false);
        (*m_current_rip_iter)[audiocd.state] = ES_RIPPING;
        processor->run ();

        while (!is_eos())
        {
          while (gtk_events_pending()) gtk_main_iteration ();
          if( m_mutex_rip.trylock() )
          {
            processor->stop ();
            delete processor;

            (*m_current_rip_iter)[audiocd.state] = ES_WAITING;

            m_b_cdda_rip->set_sensitive (true);
            m_b_cdda_stop->set_sensitive (false);
#ifndef HAVE_HAL
            m_b_cdda_reload->set_sensitive (true);
#endif //!HAVE_HAL

            m_view->get_selection()->set_mode (SELECTION_SINGLE);

            set_processing_cd (false);
            m_mutex_rip.unlock ();
            return;
          }
        }

        delete processor;

        (*m_current_rip_iter)[audiocd.state] = ES_TAGGING;

        Track local_track;
        imbue_track ((*m_current_rip_iter)[audiocd.track], local_track);
        imbue_album (m_release, local_track);

        local_track.location = rip_uri;

        try {
            library->metadata_set_taglib (local_track);
            (*m_current_rip_iter)[audiocd.percentage] = 100;
            (*m_current_rip_iter)[audiocd.state] = ES_DONE;
          }
        catch (Bmp::MetadataWriteError & cxe)
          {
            g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL, "Unable to modify metadata for file %s, Error: %s", rip_filename.c_str(), cxe.what());
            (*m_current_rip_iter)[audiocd.percentage] = 0;
            (*m_current_rip_iter)[audiocd.state] = ES_ERROR;
            g_unlink (rip_filename.c_str());
          }
      }

      m_view->get_selection()->set_mode (SELECTION_SINGLE);
      m_b_cdda_rip->set_sensitive (true);
      m_b_cdda_stop->set_sensitive (false);

      m_label_artist->set_text ("");
      m_label_album->set_text ("");
      m_label_date->set_text ("");
      m_cover_image->clear();

      set_processing_cd (false);
      m_mutex_rip.unlock ();

      m_view->get_column (5)->set_visible (false);
      m_view->get_column (6)->set_visible (false);

      VUri uri_list;
      bool warn = false;
      for (TreeNodeChildren::iterator iter  = m_store->children().begin();
                                      iter != m_store->children().end(); ++iter)
      {
        if( (*iter)[audiocd.state] == ES_DONE )
        {
          uri_list.push_back ((*iter)[audiocd.location]);
        }
        else
        {
          warn = true;
        }
      }

      ustring message = _("You can now insert the new album into the Library if you wish to.");

      if( warn )
      {
        message += _("\n<b>NOTE:</b> Not all files were ripped successfully, the import will be incomplete!");
      }

      TaskDialog dialog ( m_ref_xml->get_widget ("main-ui"),
                          _("Audio CD Ripping - BMP"),
                          _("Ripping Complete."),
                          MESSAGE_INFO, message);

      dialog.add_button ( _("Import Album"),
                          _("This will import the album into your Music Library"),
                          Stock::GO_FORWARD,
                          GTK_RESPONSE_OK);

      dialog.add_button ( _("Don't Import"),
                          _("Do not import the Album at this time"),
                          Stock::CANCEL,
                          GTK_RESPONSE_CANCEL);

      dialog.set_default_response (GTK_RESPONSE_OK);

      int response = dialog.run ();
      dialog.hide ();

      if( response == GTK_RESPONSE_OK )
      {
        if( !m_release.releaseASIN.empty() )
        {
          try{
            amazon->cache (m_release.releaseASIN);
            }
          catch (...) {}
        }
        s_request_import_.emit (uri_list);
      }
    }

    void
    CDDA::stream_eos ()
    {
      set_eos (true);
    }

    void
    CDDA::stream_error (ustring error)
    {
      set_eos (true);
      MessageDialog dialog (error, true, MESSAGE_ERROR, BUTTONS_OK, true);
      dialog.run ();
    }

    void
    CDDA::stream_position (int position)
    {
      unsigned int duration = (*m_current_rip_iter)[audiocd.duration];
      int percentage = (int(((1.*position)/(1.*duration)) * 100));
      if( !duration || !percentage )
        {
          (*m_current_rip_iter)[audiocd.percentage] = 0;
        }
      else
        {
          (*m_current_rip_iter)[audiocd.percentage] = percentage;
        }

      while (gtk_events_pending()) gtk_main_iteration ();

    }

    void
    CDDA::cell_data_func_state (CellRenderer* cell, TreeModel::iterator const& iter)
    {
      const char *states[] =
      {
        N_("Unprocessed"),
        N_("Waiting"),
        N_("Encoding"),
        N_("Tagging"),
        N_("Done"),
        N_("Error!")
      };
      CellRendererText *cell_text = dynamic_cast<CellRendererText *>(cell);
      cell_text->property_text() = _(states[int ((*iter)[audiocd.state])]);
    }

    void
    CDDA::send_title ()
    {
      TreeModel::iterator const& iter = m_current_iter.get();

      if( !m_anonymous_cd )
      {
        MusicBrainzTrack const& track = (*iter)[audiocd.track];

        TrackMetadata metadata;

        //FIXME: Use MusicBrainzXml imbue_track() ?
        metadata.mb_track_id                = track.trackId;
        metadata.artist                     = track.artistName;
        metadata.mb_artist_id               = track.artistId;
        metadata.mb_artist_sort_name        = track.artistSortName;
        metadata.mb_album_artist            = m_release.mArtist.artistName;
        metadata.mb_album_artist_sort_name  = m_release.mArtist.artistSortName;
        metadata.mb_album_artist_id         = m_release.mArtist.artistId;
        metadata.title                      = track.trackTitle;
        metadata.album                      = m_release.releaseTitle;
        metadata.duration                   = track.trackDuration/1000;
        metadata.asin                       = m_release.releaseASIN;
        metadata.tracknumber                = track.trackTrackNumber;
        metadata.genre                      = _("Audio CD"); 

        if( !m_release.releaseASIN.empty() )
        {
          amazon->fetch (m_release.releaseASIN, metadata.image);
          metadata.asin = m_release.releaseASIN;
        }

        s_track_metadata_.emit (metadata);
      }
      else
      {
          ustring const& title = (*iter)[audiocd.title];

          TrackMetadata metadata;
          metadata.artist    = ustring (_(unknown_artist));
          metadata.album     = ustring (_(unknown_album));
          metadata.title     = title;
          metadata.duration  = int ((*iter)[audiocd.duration]/1000);

          s_track_metadata_.emit (metadata);
      }
    }

    void
    CDDA::assign_current_iter (TreeIter const& iter)
    {
      m_current_iter = iter;
      TreeModel::Path path1, path2, path3 (m_current_iter.get());
      if( m_view->get_visible_range (path1, path2) )
      {
        if( (path3 < path1) || (path3 > path2) )
        {
          m_view->scroll_to_row (path3, 0.5);
        }
      }
    }

    // Bmp::PlaybackSource
    void
    CDDA::cell_data_func_playing (CellRenderer* cell_, TreeModel::iterator const& iter)
    {
      CellRendererPixbuf *cell = dynamic_cast<CellRendererPixbuf *>(cell_);
      if( m_current_iter && (m_current_iter.get() == iter) )
      {
        cell->property_pixbuf() = m_view->render_icon (Gtk::StockID (BMP_STOCK_PLAY), Gtk::ICON_SIZE_MENU);
        return;
      }
      else
      {
        cell->property_pixbuf() = RefPtr<Gdk::Pixbuf>(0);
      }
    }

    ustring
    CDDA::get_uri ()
    {
      return ustring ((cdda_uri_f % (TreePath (m_current_iter.get()).get_indices().data()[0] + 1)).str());
    }

    bool
    CDDA::go_next ()
    {
      TreeModel::Path path (m_current_iter.get());
      TreeModel::Path path_old (path);
      TreeModel::iterator iter_old = m_view->get_model()->get_iter (path);

      TreePath::value_type size = m_view->get_model()->children().size()-1;
      TreePath::value_type position = path.get_indices().data()[0];

      go_next_loop:

      if( (position + 1) > size )
      {
        if( mcs->key_get <bool> ("bmp","repeat") )
        {
          position = -1;
        }
        else
        {
          return false;
        }
      }

      path = TreeModel::Path (TreePath::size_type (1), TreePath::value_type (position+1));
      TreeModel::iterator const& iter = m_view->get_model()->get_iter (path);
      if( !(*iter)[audiocd.audio] )
        {
          position++;
          goto go_next_loop;
        }

      assign_current_iter (iter);

      m_view->get_model()->row_changed (path_old, iter_old);
      m_view->get_model()->row_changed (path, iter);

      if( (position < size) || mcs->key_get <bool> ("bmp", "repeat") )
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_NEXT);
      else
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);

      if( position > 0 )
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_PREV);
      else
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);

      s_caps_.emit (m_caps);
      send_title ();
      return true;
    }

    bool
    CDDA::go_prev ()
    {
      TreeModel::Path path (m_current_iter.get());

      TreeModel::Path path_old (path);
      TreeModel::iterator iter_old = m_view->get_model()->get_iter (path);

      unsigned int size = m_view->get_model()->children().size()-1;
      unsigned int position = path.get_indices().data()[0];

      go_prev_loop:

      if ((position - 1) < 0) return false;

      path = TreeModel::Path (1, position-1);
      TreeModel::iterator const& iter = m_view->get_model()->get_iter (path);
      if( !(*iter)[audiocd.audio] )
        {
          position--;
          goto go_prev_loop;
        }

      assign_current_iter (iter);

      m_view->get_model()->row_changed (path_old, iter_old);
      m_view->get_model()->row_changed (path, iter);

      if( (position < size) || mcs->key_get <bool> ("bmp", "repeat") )
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_NEXT);
      else
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);

      if( position > 0 )
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_PREV);
      else
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);

      s_caps_.emit (m_caps);
      send_title ();
      return true;
    }

    void
    CDDA::stop ()
    {
      if( m_current_iter )
        {
          TreeModel::iterator iter = m_current_iter.get();
          TreeModel::Path path (iter);
          m_current_iter.reset ();
          m_view->get_model()->row_changed (path, iter);
        }

      m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);
      m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
      m_caps = Caps (m_caps & ~PlaybackSource::CAN_PAUSE);
      m_caps = Caps (m_caps & ~PlaybackSource::CAN_PROVIDE_METADATA);

      s_caps_.emit (m_caps);

      m_b_cdda_rip->set_sensitive (true);
      m_cdda_cbox_format->set_sensitive (true);
      m_cdda_cbox_quality->set_sensitive (true);
    }

    void
    CDDA::play ()
    {
      if( !m_current_iter )
      {
        if( m_view->get_selection()->count_selected_rows() )
          assign_current_iter (m_view->get_selection()->get_selected());
        else
          assign_current_iter (m_store->get_iter (TreePath (TreePath::size_type (1), TreePath::value_type (0))));
      }

 
      TreeModel::Path path (m_current_iter.get());

      m_b_cdda_rip->set_sensitive (false);
      m_cdda_cbox_format->set_sensitive (false);
      m_cdda_cbox_quality->set_sensitive (false);

      m_view->get_column (5)->set_visible (false);
      m_view->get_column (6)->set_visible (false);

      unsigned int position = path.get_indices().data()[0];
      unsigned int size     = m_view->get_model()->children().size()-1;

      if( (position < size) || mcs->key_get <bool> ("bmp", "repeat") )
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_NEXT);
      else
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);

      if( position > 0 )
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_PREV);
      else
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);

      m_caps = Caps (m_caps | PlaybackSource::CAN_PAUSE);

      if (!m_anonymous_cd) m_caps = Caps (m_caps | PlaybackSource::CAN_PROVIDE_METADATA);

      s_caps_.emit (m_caps);
      send_title ();
    }

    void
    CDDA::play_post ()
    {
      TreeModel::Path path (m_current_iter.get());

      m_b_cdda_rip->set_sensitive (false);
      m_cdda_cbox_format->set_sensitive (false);
      m_cdda_cbox_quality->set_sensitive (false);

      m_view->get_column (5)->set_visible (false);
      m_view->get_column (6)->set_visible (false);

      unsigned int position = path.get_indices().data()[0];
      unsigned int size     = m_view->get_model()->children().size()-1;

      if( (position < size) || mcs->key_get <bool> ("bmp", "repeat") )
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_NEXT);
      else
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);

      if( position > 0 )
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_PREV);
      else
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);


      if( !m_anonymous_cd )
      {
        m_caps = Caps (m_caps | PlaybackSource::CAN_PROVIDE_METADATA);
      }
      m_caps = Caps (m_caps | PlaybackSource::CAN_PAUSE);

      send_title ();
    }

    void
    CDDA::restore_context ()
    {
    }

    void
    CDDA::selection_changed ()
    {
      if( m_view->get_selection ()->count_selected_rows() == 1 )
      {
        TreeModel::iterator const& iter = m_view->get_selection()->get_selected ();
        if( (*iter)[audiocd.audio] )
        {
          m_caps = Caps (m_caps |  PlaybackSource::CAN_PLAY);
          s_caps_.emit (m_caps);
          return;
        }
      }

      m_caps = Caps (m_caps & ~PlaybackSource::CAN_PLAY);
      s_caps_.emit (m_caps);
    }

    bool
    CDDA::processing_cd ()
    {
      m_processing_check_lock.lock ();
      bool processing = m_processing_cd;
      m_processing_check_lock.unlock ();
      return processing;
    }

    void
    CDDA::set_processing_cd (bool set)
    {
      m_processing_check_lock.lock ();
      m_processing_cd = set;
      m_processing_check_lock.unlock ();
    }

    bool
    CDDA::is_eos ()
    {
      m_eos_check_lock.lock ();
      bool eos = m_stream_eos;
      m_eos_check_lock.unlock ();
      return eos;
    }

    void
    CDDA::set_eos (bool set)
    {
      m_eos_check_lock.lock ();
      m_stream_eos = set;
      m_eos_check_lock.unlock ();
    }

  } // namespace UiPart
} // namespace Bmp
