//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2007 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <glibmm.h>
#include <glibmm/i18n.h>
#include <gtkmm.h>
#include <libglademm.h>

#include <cstring>
#include <iostream>
#include <sstream>
#include <string>

#include <boost/algorithm/string.hpp>
#include <boost/format.hpp>

// BMP Widgets
#include "widgets/banner-image.hh"

// BMP Audio
#include "audio/audio.hh"

// BMP Misc
#include "x_core.hh"
#include "x_library.hh"
#include "x_play.hh"
#include "x_vfs.hh"
#include "x_lastfm.hh"
#include "x_mcsbind.hh"

#include "debug.hh"
#include "main.hh"
#include "network.hh"
#include "paths.hh"
#include "stock.hh"
#include "uri++.hh"
#include "util.hh"
#include "util-file.hh"
#include "util-string.hh"

#include "ui-tools.hh"
#include "ui-part-lastfm.hh"

#include "lastfm-tag-dialog.hh"

using namespace Glib;
using namespace Gtk;
using namespace std;

using namespace Bmp::Util;
using namespace Bmp::LastFM;
using namespace Bmp::LastFM::XMLRPC;
using namespace Bmp::LastFM::WS;

namespace
{
  static boost::format f_neighbours ("lastfm://user/%s/neighbours");
  static boost::format f_personal   ("lastfm://user/%s/personal");
  static boost::format f_loved      ("lastfm://user/%s/loved");
  static boost::format f_recommend  ("lastfm://user/%s/recommended/%i");
  static boost::format f_globaltag  ("lastfm://globaltags/%s");
  static boost::format f_artist     ("lastfm://artist/%s/similarartists");

  static Glib::ustring t_personal  (_("Personal"));
  static Glib::ustring t_loved     (_("Loved Tracks"));
  static Glib::ustring t_neighbour (_("Neighbourhood"));
  static Glib::ustring t_recommend (_("Recommended"));
}

#define LASTFM_ACTION_LOVE          "lastfm-action-love"
#define LASTFM_ACTION_BAN           "lastfm-action-ban"
#define LASTFM_ACTION_TAG           "lastfm-action-tag"

#define LASTFM_ACTION_UPDATE_LIST   "lastfm-action-update-list"
#define LASTFM_ACTION_MATCH_TAGS    "lastfm-action-match-tags"

namespace
{
  const char * ui_string_lastfm =
  "<ui>"
  ""
  "<menubar name='MenuBarMain'>"
  "   <menu action='MenuUiPartLastFm'>"
  "     <menuitem action='" LASTFM_ACTION_LOVE "'/>"
  "     <menuitem action='" LASTFM_ACTION_BAN "'/>"
  "     <separator/>"
  "     <menuitem action='" LASTFM_ACTION_TAG "'/>"
  "     <separator/>"
  "     <menuitem action='" LASTFM_ACTION_UPDATE_LIST "'/>"
  "     <menuitem action='" LASTFM_ACTION_MATCH_TAGS "'/>"
  "   </menu>"
  "</menubar>"
  ""
  "</ui>";

  const char * ui_string_lastfm_context =
  "<ui>"
  "<menubar name='popup-shell'>"
  " <menu action='dummy' name='menu-shell-context'>"
  "   <placeholder name='PlaceholderSourceContext'>"
  "     <menuitem action='" LASTFM_ACTION_LOVE "'/>"
  "     <menuitem action='" LASTFM_ACTION_BAN "'/>"
  "     <menuitem action='" LASTFM_ACTION_TAG "'/>"
  "   </placeholder>"
  " </menu>"
  "</menubar>"
  "</ui>";

  char const * ui_string_lastfm_tray =
  "<ui>"
  " <menubar name='popup-tray'>"
  " <menu action='dummy' name='menu-tray'>"
  "   <placeholder name='PlaceholderSourceContext'>"
  "     <menuitem action='" LASTFM_ACTION_LOVE "'/>"
  "     <menuitem action='" LASTFM_ACTION_BAN "'/>"
  "   </placeholder>"
  " </menu>"
  " </menubar>"
  "</ui>";

}

namespace Bmp
{
  namespace UiPart
  {
    guint
    LASTFM::add_ui ()
    {
      return m_ui_manager->add_ui_from_string  (ui_string_lastfm);
    };

    guint
    LASTFM::add_context_ui ()
    {
      return m_ui_manager->add_ui_from_string  (ui_string_lastfm_context);
    }

    guint
    LASTFM::add_tray_ui ()
    {
      return m_ui_manager->add_ui_from_string  (ui_string_lastfm_tray);
    }

    LASTFM::~LASTFM ()
    {
      m_tags_request.clear();
    }

    LASTFM::LASTFM (RefPtr <Gnome::Glade::Xml> const& xml, RefPtr <UIManager> ui_manager)
    : PlaybackSource  (_("Last.fm"), PROVIDES_TIMING, F_ALWAYS_IMAGE_FRAME)
    , Base (xml, ui_manager)
    {
      m_actions = Gtk::ActionGroup::create ("Actions_LastFm");
      m_actions->add (Gtk::Action::create ("MenuUiPartLastFm", _("_Last.fm")));
      m_actions->add  (Gtk::Action::create (LASTFM_ACTION_LOVE,
                                            Gtk::StockID (BMP_STOCK_LASTFM_LOVE),
                                            _("Love Track")),
                                            (sigc::mem_fun (*this, &UiPart::LASTFM::on_love)));
      m_actions->add  (Gtk::Action::create (LASTFM_ACTION_BAN,
                                            Gtk::StockID (BMP_STOCK_LASTFM_BAN),
                                            _("Ban Track")),
                                            (sigc::mem_fun (*this, &UiPart::LASTFM::on_ban)));
      m_actions->add  (Gtk::Action::create (LASTFM_ACTION_TAG,
                                            Gtk::StockID (BMP_STOCK_TAG),
                                            _("Tag Current Item")),
                                            (sigc::mem_fun (*this, &UiPart::LASTFM::on_tag)));
      m_actions->add  (Gtk::Action::create (LASTFM_ACTION_UPDATE_LIST,
                                            Gtk::Stock::GO_FORWARD,
                                            _("Reload Stations")),
                                            (sigc::bind (sigc::mem_fun (*this, &UiPart::LASTFM::on_update_list), false)));
      m_actions->add  (Gtk::Action::create (LASTFM_ACTION_MATCH_TAGS,
                                            Gtk::Stock::GO_FORWARD,
                                            _("Calculate Matching Tags")),
                                            (sigc::bind (sigc::mem_fun (*this, &UiPart::LASTFM::on_update_list), true)));
      m_actions->get_action (LASTFM_ACTION_LOVE)->set_sensitive (false);
      m_actions->get_action (LASTFM_ACTION_BAN)->set_sensitive (false);
      m_actions->get_action (LASTFM_ACTION_TAG)->set_sensitive (false);
      m_ui_manager->insert_action_group (m_actions);

      dynamic_cast<Gtk::Image *>(m_ref_xml->get_widget ("lastfm-i-station"))->set
        (Gdk::Pixbuf::create_from_file (build_filename (BMP_IMAGE_DIR_LASTFM, "lastfm-station.png")));
      dynamic_cast <Gtk::Image *>(m_ref_xml->get_widget ("lastfm-stations-throbber"))->set
        (Glib::build_filename (BMP_IMAGE_DIR, BMP_THROBBER));
      dynamic_cast <Gtk::Image *>(m_ref_xml->get_widget ("lastfm-tags-throbber"))->set
        (Glib::build_filename (BMP_IMAGE_DIR, BMP_THROBBER));
      dynamic_cast<Button*>(m_ref_xml->get_widget ("lastfm-station-play"))->signal_clicked().connect
        (sigc::mem_fun (*this, &LASTFM::playurl));

      m_ref_xml->get_widget ("lastfm-current-station", m_lastfm_radio_station);
      m_ref_xml->get_widget ("lastfm-station", m_station_url_entry);

      m_station_url_entry->set_text("");
      m_station_url_entry->signal_activate().connect
        (sigc::mem_fun (*this, &LASTFM::playurl));

      m_ref_xml->get_widget ("lastfm-cbox-stationchoice", m_station_choice);
      dynamic_cast <CellLayout*> (m_station_choice)->clear ();
      CellRendererText * cell = manage (new Gtk::CellRendererText());
      cell->property_xalign() = 1.0;
      m_station_choice->pack_start (*cell, false);
      m_station_choice->add_attribute (*cell, "text", 0);
      m_station_choice->set_active (0);

      m_ref_xml->get_widget_derived ("lastfm-stations", m_treeview_tags);
      m_treeview_tags->station_activated().connect
        (sigc::mem_fun (*this, &LASTFM::set_uri));

      m_ref_xml->get_widget_derived ("lastfm-textview-tags", m_tag_view);
      m_tag_buffer = m_tag_view->get_buffer (); 

      m_ref_xml->get_widget ("lastfm-limiter", m_tag_limiter);
      m_tag_limiter->signal_value_changed().connect
        (sigc::mem_fun (*this, &LASTFM::display_tags_limited));
      m_tag_limiter->set_value (100);
      m_tag_limiter->set_sensitive (0);

      m_ref_xml->get_widget ("vbox-lastfm")->set_sensitive (false);
      lastfm_radio->signal_connected().connect
        (sigc::mem_fun (*this, &UiPart::LASTFM::on_radio_connected));
      lastfm_radio->signal_disconnected().connect
        (sigc::mem_fun (*this, &UiPart::LASTFM::on_radio_disconnected));

      m_caps = Caps (m_caps & ~PlaybackSource::CAN_SEEK);
      send_caps ();
    }

#define CATCH_DEFAULT_ERRORS \
        catch (LastFMNotConnectedError& cxe)  \
          { \
            g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL, "ConnectionError: %s", cxe.what());       \
            core->status_push_message ((boost::format (_("Last.fm Radio: Not Handshaked/Session Expired: %s")) % cxe.what()).str());     \
            return; \
          }

    void
    LASTFM::on_radio_connected ()
    {
      m_treeview_tags->set_user (mcs->key_get<string>("lastfm","username"),
        lastfm_radio->session().subscriber, false); 
      m_ref_xml->get_widget ("vbox-lastfm")->set_sensitive (true);
      m_caps = Caps (m_caps | PlaybackSource::CAN_PLAY);
      send_caps ();
    }

    void
    LASTFM::on_radio_disconnected ()
    {
      m_ref_xml->get_widget ("vbox-lastfm")->set_sensitive (false);
      m_caps = Caps (m_caps & ~PlaybackSource::CAN_PLAY);
      send_caps ();
      m_treeview_tags->clear_user ();
      s_stop_request_.emit();
    }

    void
    LASTFM::clear_metadata_info ()
    {
      m_lastfm_radio_station->set_text ("");
      m_tag_buffer->set_text ("");
    }

    //////////////////////////////////////////////
    // PLAYBACKSOURCE                           //
    //////////////////////////////////////////////

    ustring
    LASTFM::get_uri ()
    {
      XSPF::Item const& item = (*m_playlist_iter);
      return item.location;
    }

    ustring
    LASTFM::get_type ()
    {
      return "audio/mpeg";
    }

    bool
    LASTFM::go_next ()
    {
      if( ::play->lastfm_qualifies_duration (m_playlist_iter->duration) )
      {
        ::lastfm_scrobbler->scrobble (*m_playlist_iter);
      }

      m_playlist_iter++;
      if( m_playlist_iter == m_playlist.items.end() )
      {
        try{
            m_playlist = lastfm_radio->get_xspf_playlist ();
            if (!m_playlist.items.empty())
            {
              m_playlist_iter = m_playlist.items.begin();
              return true;
            }
            core->status_push_message (_("Last.fm Radio: No Items Available for this Station"));
            return false;
          }
        catch (LastFMStreamTuningError & cxe)
          {
            core->status_push_message ((boost::format (_("Last.fm Radio: %s")) % cxe.what()).str());
            return false;
          }
      }
      return true;
    }

    bool
    LASTFM::go_prev ()
    {
      return false;
    }

    void
    LASTFM::stop ()
    {
      if (m_tags_request)
      {
        m_tags_request->cancel ();
      }

      m_tag_limiter->set_sensitive (0);

      m_actions->get_action (LASTFM_ACTION_TAG)->set_sensitive (0);
      m_actions->get_action (LASTFM_ACTION_LOVE)->set_sensitive (0);
      m_actions->get_action (LASTFM_ACTION_BAN)->set_sensitive (0);

      m_conn.disconnect ();

      m_playlist = XSPF::Playlist();
      m_playlist_iter = m_playlist.items.end();
      m_current_tags = TagV();

      m_caps = Caps (m_caps & ~PlaybackSource::CAN_PROVIDE_METADATA);
      m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
      send_caps ();

      clear_metadata_info ();
    }

    void
    LASTFM::play ()
    {
      m_conn = ::play->signal_http_status().connect
        (sigc::mem_fun (*this, &LASTFM::on_play_http_status));

      s_message_.emit (_("Getting Playlist"));
      m_playlist = lastfm_radio->get_xspf_playlist();

      if( !m_playlist.items.empty() )
      {
        m_playlist_iter = m_playlist.items.begin();
        s_message_clear_.emit ();
      }
      else
      {
        throw UnableToInitiatePlaybackError(_("Last.fm Radio: No Items available for current Radio"));
      }
    }

    void
    LASTFM::play_post ()
    {
      m_tag_limiter->set_sensitive (1);

      m_actions->get_action (LASTFM_ACTION_LOVE)->set_sensitive (1);
      m_actions->get_action (LASTFM_ACTION_BAN)->set_sensitive (1);
      m_actions->get_action (LASTFM_ACTION_TAG)->set_sensitive (1);

      XSPF::Item const& item = (*m_playlist_iter);

      m_lastfm_radio_station->set_text (m_playlist.title);
 
      TrackMetadata metadata;
      metadata.artist     = item.creator; 
      metadata.album      = item.album; 
      metadata.title      = item.title; 
      metadata.duration   = item.duration; 
      metadata.image      = Util::get_image_from_uri (item.image); 
      metadata.genre      = ustring ("Last.fm: " + m_playlist.title);

      s_track_metadata_.emit (metadata);

      m_metadata = metadata;

      get_track_tags ();

      if( m_playlist_iter != m_playlist.items.end() )
        m_caps = Caps (m_caps | PlaybackSource::CAN_GO_NEXT);
      else
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
    }

    void
    LASTFM::display_tags_limited ()
    {
      m_tag_buffer->set_text ("");
      TextBuffer::iterator iter = m_tag_buffer->begin();

      int value = (100 - int (m_tag_limiter->get_value()));
  
      unsigned int id = 0;
      for (TagV::const_iterator i = m_current_tags.begin(); i != m_current_tags.end(); ++i)
      {
        if (i->count >= value)
        {
          RefPtr<LastFmLinkTag> tag = LastFmLinkTag::create ((f_globaltag % (i->name)).str(), i->count);
          m_tag_buffer->get_tag_table()->add (tag);

          tag->signal_url_activated().connect
            (sigc::mem_fun (*this, &LASTFM::set_uri));

          iter = m_tag_buffer->insert_with_tag (iter, i->name, tag);
          iter = m_tag_buffer->insert (iter, "        ");
        }
      }
    }

    void
    LASTFM::got_track_tags (TagV const& tags)
    {
      m_current_tags = tags;

      if (m_current_tags.size() > 1)
      {
        std::random_shuffle( m_current_tags.begin(), m_current_tags.end()-1 );
        std::iter_swap( m_current_tags.begin()+m_current_tags.size()/2, m_current_tags.end()-1);
      }

      display_tags_limited ();

      dynamic_cast <Gtk::Notebook *>(m_ref_xml->get_widget ("lastfm-tags-notebook"))->set_current_page (0);
      while (gtk_events_pending()) gtk_main_iteration();
    }

    void
    LASTFM::get_track_tags ()
    {
      dynamic_cast <Gtk::Notebook *>(m_ref_xml->get_widget ("lastfm-tags-notebook"))->set_current_page (1);
      while (gtk_events_pending()) gtk_main_iteration();

      XSPF::Item const& item = (*m_playlist_iter);
      m_current_tags = TagV();
  
      m_tags_request = TagsGlobReq::create (TAGS_GLOBAL_TRACK, item.creator, item.title);     
      m_tags_request->tags().connect (sigc::mem_fun (*this, &LASTFM::got_track_tags));
      m_tags_request->run ();
    }

    void
    LASTFM::next_post ()
    {
      play_post ();
    }

    void
    LASTFM::prev_post ()
    {
      play_post ();
    }

    void
    LASTFM::restore_context ()
    {
    }

    void
    LASTFM::segment ()
    {
      clear_metadata_info ();
      m_actions->get_action (LASTFM_ACTION_LOVE)->set_sensitive (1);
      m_actions->get_action (LASTFM_ACTION_BAN)->set_sensitive (1);
    }

    void
    LASTFM::skipped ()
    {
      XSPF::Item & item = (*m_playlist_iter);
      item.rating = "S";
    }

    ////////////////////////////// !PLAYBACKSOURCE

    void
    LASTFM::on_play_http_status (int status)
    {
      switch (status)
      {
        case 200: break;
        case 401:
          {
            core->status_push_message (_("Last.fm Radio: Session expired. Please reconnect."));
            s_stop_request_.emit ();
            break;
          }

        case 503:
          {
            core->status_push_message (_("Last.fm Radio: The radio server is too busy at the moment, please try "
                                         "again in a few minutes."));
            s_stop_request_.emit ();
            break;
          }

        case 666:
          {
            core->status_push_message (_("Last.fm Radio: Server is down for maintenance, please try again in a few minutes."));
            s_stop_request_.emit ();
            break;
          }

        case 667:
          {
            core->status_push_message (_("Last.fm Radio: Not enough content (left) to play this station; please tune "
                                         "in to a different one."));
            s_stop_request_.emit ();
            break;
          }
      }
    }
  
    void
    LASTFM::playurl ()
    {
      ustring text = m_station_url_entry->get_text ();
      m_station_url_entry->set_text ("");
      if( !text.empty() )
      {
        ustring uri;
        switch (m_station_choice->get_active_row_number())
        {
          case 0: 
            uri = (f_globaltag % text.c_str()).str();
            break;

          case 1:
            uri = (f_artist % text.c_str()).str();
            break;

          case 2:
            uri = (f_neighbours % text.c_str()).str();
            break;

          case 3:
            if( text.substr (0, 9)  == "lastfm://" )
            {
              text = text.substr (10);
            }
            uri = ("lastfm://" + text);
            break;

        }

        try{
            lastfm_radio->handshake ();  
            lastfm_radio->playurl (uri);
            s_playback_request_.emit ();
          }
        catch (LastFMStreamTuningError & cxe)
          {
            core->status_push_message ((boost::format (_("Last.fm Radio: Couldn't tune into station: %s")) % cxe.what()).str());
          }
      }
      else
      {
        s_playback_request_.emit ();
      }
    }

    void
    LASTFM::set_uri (ustring const& uri)
    {
      s_stop_request_.emit ();

      m_station_url_entry->set_text ("");
  
      URI u (uri);
      u.unescape();
      m_current_uri = ustring (u);
      tune ();
    }

    void
    LASTFM::tune ()
    {
      try{
          s_message_.emit (_("Connecting to Last.fm"));
          lastfm_radio->handshake ();  
          lastfm_radio->playurl (m_current_uri);
          s_message_.emit (_("Starting Playback"));
          s_playback_request_.emit ();
        }
      catch (LastFMStreamTuningError & cxe)
        {
          core->status_push_message ((boost::format (_("Last.fm Radio Couldn't tune into station: %s")) % cxe.what()).str());
          s_stop_request_.emit();
        }
      CATCH_DEFAULT_ERRORS
    }

    void
    LASTFM::on_love ()
    {
      using namespace XMLRPC;
      try{
          XSPF::Item & item = (*m_playlist_iter);
          m_actions->get_action (LASTFM_ACTION_LOVE)->set_sensitive (0);
          m_actions->get_action (LASTFM_ACTION_BAN)->set_sensitive (0);
          item.rating = "L";
          TrackAction action ("loveTrack", item);
          action.run ();
        }
      CATCH_DEFAULT_ERRORS
    }

    void
    LASTFM::on_ban ()
    {
      using namespace XMLRPC;
      try{
          XSPF::Item & item = (*m_playlist_iter);
          m_actions->get_action (LASTFM_ACTION_BAN)->set_sensitive (0);
          m_actions->get_action (LASTFM_ACTION_LOVE)->set_sensitive (0);
          item.rating = "B";
          TrackAction action ("banTrack", item);
          s_next_request_.emit ();
        }
      CATCH_DEFAULT_ERRORS
    }

    void
    LASTFM::on_tag ()
    {
      using namespace XMLRPC;
      try{
          LastFM::TagDialog * d = LastFM::TagDialog::create ();
          XSPF::Item const& item = (*m_playlist_iter);
          d->run (item);
          delete d;
          get_track_tags ();
        }
      CATCH_DEFAULT_ERRORS
    }

    ////////////////////////////////////////////

    void
    LASTFM::on_update_list (bool match_tags)
    {
      m_treeview_tags->set_user (mcs->key_get <string> ("lastfm","username"), lastfm_radio->session().subscriber, match_tags); 
    }

    //// URLS TreeView

    LASTFM::TreeViewTags::TreeViewTags (BaseObjectType                 * obj,
                                        RefPtr<Gnome::Glade::Xml> const& xml)
    : TreeView (obj)
    {
      m_store = TreeStore::create (columns);
      xml->get_widget ("lastfm-stations-notebook", m_notebook);

      // md FIXME This is all somewhat strange here because we just assume the sizes of the images are as we need them,
      //          and while they currently in fact are it's not guaranteed for all times ever

      m_pb_artist = Gdk::Pixbuf::create_from_file (build_filename (BMP_IMAGE_DIR_LASTFM , "lastfm-artist.png"));
      m_pb_rec = Gdk::Pixbuf::create_from_file (build_filename (BMP_IMAGE_DIR_LASTFM , "lastfm-recommended-radio.png"));
      m_pb_url = Gdk::Pixbuf::create_from_file (build_filename (BMP_IMAGE_DIR_LASTFM , "lastfm-station-small.png"));
      m_pb_neighbour = Gdk::Pixbuf::create_from_file (build_filename (BMP_IMAGE_DIR_LASTFM , "lastfm-neighbour.png"));
      m_pb_friend = Gdk::Pixbuf::create_from_file (build_filename (BMP_IMAGE_DIR_LASTFM , "lastfm-friend.png"));
      m_pb_user = Gdk::Pixbuf::create_from_file (build_filename (BMP_IMAGE_DIR_LASTFM , "lastfm-user.png"));
      m_pb_mainuser = Gdk::Pixbuf::create_from_file (build_filename (BMP_IMAGE_DIR_LASTFM , "lastfm-mainuser.png"));
      m_pb_tag = Gdk::Pixbuf::create_from_file (build_filename (BMP_IMAGE_DIR_LASTFM , "lastfm-tag.png"));

      TreeViewColumn *column = 0; 
      CellRendererPixbuf *cell1 = 0; 
      CellRendererText *cell2 = 0; 

      column = manage (new TreeViewColumn ());
      column->set_resizable (false);
      column->set_expand (false);

      cell1 = manage (new CellRendererPixbuf ());
      column->pack_start (*cell1, false);
      column->set_cell_data_func (*cell1, sigc::mem_fun (*this, &LASTFM::TreeViewTags::cell_data_func));
       
      cell2 = manage (new CellRendererText ());
      column->pack_start (*cell2, false);
      column->add_attribute (*cell2, "markup", columns.name);

      append_column (*column);

      column = manage (new TreeViewColumn ());
      cell2 = manage (new CellRendererText ());
      column->pack_start (*cell2, false);
      column->add_attribute (*cell2, "markup", columns.score);
      append_column (*column);

      get_selection()->set_mode (SELECTION_SINGLE);
      set_model (m_store);
    }

    LASTFM::TreeViewTags::~TreeViewTags () {}

    void
    LASTFM::TreeViewTags::clear_user ()
    {
      m_store->clear ();
      set_sensitive (false);
      while (gtk_events_pending()) gtk_main_iteration ();
    }

    void
    LASTFM::TreeViewTags::set_user (ustring const& username, bool subscriber, bool match_tags)
    {
      using boost::algorithm::split;
      using boost::algorithm::is_any_of;

      clear_user ();  

      i_root_stations = m_store->append ();
      (*i_root_stations)[columns.name] = (boost::format (_("%s's stations")) % username.c_str()).str();
      (*i_root_stations)[columns.type] = TreeViewTags::ROW_MAINUSER;

      i_root_recommended = m_store->append ();
      (*i_root_recommended)[columns.name] = (boost::format (_("%s's recommendations")) % username.c_str()).str();
      (*i_root_recommended)[columns.type] = TreeViewTags::ROW_SYSREC_ROOT;

      i_root_neighbours = m_store->append ();
      (*i_root_neighbours)[columns.name] = (boost::format (_("%s's neighbours")) % username.c_str()).str();
      (*i_root_neighbours)[columns.type] = TreeViewTags::ROW_NEIGHBOUR_ROOT;

      i_root_friends = m_store->append ();
      (*i_root_friends)[columns.name] = (boost::format (_("%s's friends")) % username.c_str()).str();
      (*i_root_friends)[columns.type] = TreeViewTags::ROW_FRIEND_ROOT;

      i_root_usertags = m_store->append ();
      (*i_root_usertags)[columns.name] = (boost::format (_("%s's personal tags")) % username.c_str()).str();
      (*i_root_usertags)[columns.type] = TreeViewTags::ROW_TAG_ROOT;

      TreeIter iter = m_store->append (i_root_stations->children());
      (*iter)[columns.name] = Markup::escape_text (t_neighbour);
      (*iter)[columns.station]  = (f_neighbours % username.c_str()).str();
      (*iter)[columns.type]  = TreeViewTags::ROW_URL;

      iter = m_store->append (i_root_stations->children());
      (*iter)[columns.name] = Markup::escape_text (t_recommend);
      (*iter)[columns.station]  = (f_recommend % username.c_str() % 100).str();
      (*iter)[columns.type]  = TreeViewTags::ROW_URL;

      if( subscriber )
      {
            iter = m_store->append (i_root_stations->children());
            (*iter)[columns.name] = Markup::escape_text (t_personal);
            (*iter)[columns.station]  = (f_personal % username.c_str()).str();
            (*iter)[columns.type]  = TreeViewTags::ROW_URL;

            iter = m_store->append (i_root_stations->children());
            (*iter)[columns.name] = Markup::escape_text (t_loved);
            (*iter)[columns.station]  = (f_loved % username.c_str()).str();
            (*iter)[columns.type]  = TreeViewTags::ROW_URL;
      }

      // Append System Recommendations 
      LastFMArtistV recommended;
      WS::artists (AT_USER_SYSTEMREC, username, recommended);
      for (LastFMArtistV::const_iterator i = recommended.begin(); i != recommended.end(); ++i)
      {
            LastFMArtist const& x (*i);
            iter = m_store->append (i_root_recommended->children());
            (*iter)[columns.name] = Markup::escape_text (x.name);
            (*iter)[columns.type] = TreeViewTags::ROW_ARTIST;
            (*iter)[columns.station] = (f_artist % x.name.c_str()).str();

            while (gtk_events_pending()) gtk_main_iteration ();
      }

      // Append Neighbours
      UserV neighbours;
      WS::users (UT_N, username, neighbours);
      for (UserV::const_iterator i = neighbours.begin(); i != neighbours.end(); ++i)
      {
            User const& x (*i);
            iter = m_store->append (i_root_neighbours->children());
            (*iter)[columns.name] = Markup::escape_text (x.username);
            (*iter)[columns.type] = TreeViewTags::ROW_NEIGHBOUR;
            (*iter)[columns.score] = ((boost::format ("<small><b>%s: %.2f</b></small>") % _("Match") % x.match).str()); 

            TreeModel::iterator n, p, l, r;
            n = m_store->append (iter->children());
            r = m_store->append (iter->children());

            (*n)[columns.name] = Markup::escape_text (t_neighbour);
            (*n)[columns.station] = (f_neighbours % x.username.c_str()).str();
            (*n)[columns.type] = TreeViewTags::ROW_URL;

            (*r)[columns.name] = Markup::escape_text (t_recommend);
            (*r)[columns.station] = (f_recommend % x.username.c_str() % 100).str();
            (*r)[columns.type]  = TreeViewTags::ROW_URL;

            if( subscriber )
            {
                  p = m_store->append (iter->children());
                  l = m_store->append (iter->children());

                  (*p)[columns.name] = Markup::escape_text (t_personal);
                  (*p)[columns.station] = (f_personal % x.username.c_str()).str();
                  (*p)[columns.type] = TreeViewTags::ROW_URL;

                  (*l)[columns.name] = Markup::escape_text (t_loved);
                  (*l)[columns.station] = (f_loved % x.username.c_str()).str();
                  (*l)[columns.type] = TreeViewTags::ROW_URL;
            }

            while (gtk_events_pending()) gtk_main_iteration ();
      }

      // Append Friends 
      UserV friends;
      WS::users (UT_F, username, friends);
      for (UserV::const_iterator i = friends.begin(); i != friends.end(); ++i)
      {
            User const& x (*i);
            iter = m_store->append (i_root_friends->children());
            (*iter)[columns.name] = Markup::escape_text (x.username);
            (*iter)[columns.type]  = TreeViewTags::ROW_FRIEND;

            TreeModel::iterator n, r, p, l;
            n = m_store->append (iter->children());
            r = m_store->append (iter->children());

            (*n)[columns.name] = Markup::escape_text (t_neighbour);
            (*n)[columns.station] = (f_neighbours % x.username.c_str()).str();
            (*n)[columns.type] = TreeViewTags::ROW_URL;

            (*r)[columns.name] = Markup::escape_text (t_recommend);
            (*r)[columns.station] = (f_recommend % x.username.c_str() % 100).str();
            (*r)[columns.type] = TreeViewTags::ROW_URL;

            if( subscriber )
            {
                  p = m_store->append (iter->children());
                  l = m_store->append (iter->children());

                  (*p)[columns.name] = Markup::escape_text (t_personal);
                  (*p)[columns.station] = (f_personal % x.username.c_str()).str();
                  (*p)[columns.type] = TreeViewTags::ROW_URL;

                  (*l)[columns.name] = Markup::escape_text (t_loved);
                  (*l)[columns.station] = (f_loved % x.username.c_str()).str();
                  (*l)[columns.type] = TreeViewTags::ROW_URL;
            }

            while (gtk_events_pending()) gtk_main_iteration ();
      }

      // Append User's Tags 
      TagV user_tags;
      WS::tags_user (TAGS_USER_TOPTAGS, user_tags, username);
      for (TagV::const_iterator i = user_tags.begin(); i != user_tags.end(); ++i)
      {
            Tag const& x (*i);
            iter = m_store->append (i_root_usertags->children());
            (*iter)[columns.name] = Markup::escape_text (x.name);
            (*iter)[columns.type] = TreeViewTags::ROW_TAG;
            (*iter)[columns.station] = (f_globaltag % x.name.c_str()).str();

            while (gtk_events_pending()) gtk_main_iteration ();
      }


      m_match_tags.clear ();
      set_sensitive (true);
      if( match_tags )
      {
            m_notebook->set_current_page (1);
            while (gtk_events_pending()) gtk_main_iteration();

            i_root_matchtags = m_store->append ();
            (*i_root_matchtags)[columns.name] = _("Matching Tags");
            (*i_root_matchtags)[columns.type] = TreeViewTags::ROW_TAG_ROOT;

            Glib::Thread * t1 = Glib::Thread::create
              (sigc::bind (sigc::mem_fun (*this, &UiPart::LASTFM::TreeViewTags::get_user_match_tags), username), false);
            disp.connect (sigc::mem_fun (*this, &UiPart::LASTFM::TreeViewTags::set_user_match_tags));
      }
    }

    void
    LASTFM::TreeViewTags::get_user_match_tags (std::string const& username)
    {
      WS::matchtags (username, m_match_tags);
      disp.emit();
    }

    void
    LASTFM::TreeViewTags::set_user_match_tags ()
    {
      m_notebook->set_current_page (0);
      while (gtk_events_pending()) gtk_main_iteration();

      TreeIter iter;
      std::sort (m_match_tags.begin(), m_match_tags.end());

      for (RankedTagV::const_iterator i = m_match_tags.begin(); i != m_match_tags.end(); ++i)
      {
            Tag const& tag (i->first);

            iter = m_store->append (i_root_matchtags->children());

            (*iter)[columns.name]     = Markup::escape_text (tag.name); 
            (*iter)[columns.score]    = ((boost::format ("<small><b>%s: %.2f</b></small>")  % _("Score") % (double (i->second) / 100.)).str());
            (*iter)[columns.station]  = (f_globaltag % tag.name).str();
            (*iter)[columns.type]     = TreeViewTags::ROW_TAG;

            if( i != m_match_tags.begin() )
            {
                  expand_row (m_store->get_path (i_root_matchtags), false);
            }

            while (gtk_events_pending()) gtk_main_iteration ();
      }
    }

    void
    LASTFM::TreeViewTags::cell_data_func (CellRenderer * _cell, TreeModel::iterator const& iter)
    {
      CellRendererPixbuf * cell = dynamic_cast<CellRendererPixbuf *>(_cell);
      switch ((*iter)[columns.type])
      {
            case ROW_MAINUSER:
                cell->property_pixbuf() = m_pb_mainuser;
                return;

            case ROW_SYSREC_ROOT:
                cell->property_pixbuf() = m_pb_rec;
                return;

            case ROW_ARTIST:
                cell->property_pixbuf() = m_pb_artist;
                return;

            case ROW_URL:
                cell->property_pixbuf() = m_pb_url;
                return;

            case ROW_TAG_ROOT:
                cell->property_pixbuf() = m_pb_tag;
                return;
            case ROW_TAG:
                cell->property_pixbuf() = m_pb_url;
                return;

            case ROW_NEIGHBOUR_ROOT:
                cell->property_pixbuf() = m_pb_neighbour;
                return;
            case ROW_NEIGHBOUR:
                cell->property_pixbuf() = m_pb_user;
                return;

            case ROW_FRIEND_ROOT:
                cell->property_pixbuf() = m_pb_friend;
                return;
            case ROW_FRIEND:
                cell->property_pixbuf() = m_pb_user;
                return;

          default: return;
      }
    }

    void
    LASTFM::TreeViewTags::on_row_activated (TreeModel::Path const& path, TreeViewColumn * column)
    {
      RowType type ((m_store->get_iter(path).operator*().operator[](columns.type)));
      if( (type == ROW_URL) || (type == ROW_TAG) || (type == ROW_ARTIST) )
      {
        station_activated_.emit (m_store->get_iter (path).operator*().operator[](columns.station));
        get_selection()->unselect_all ();
      }
    }
  } // end namespace UiPart
} // end namespace Bmp
