//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2007 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non GPL-compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <glibmm/ustring.h>

using namespace std;
using namespace Glib;

namespace
{
  enum ElementId
  {
    E_PLAYLIST,

    E_TRACK_LIST,
    E_TRACK,

    E_LOCATION,
    E_TITLE,
    E_ID,
    E_ALBUM,
    E_CREATOR,
    E_DURATION,
    E_IMAGE,
    E_TRACKAUTH,
    E_ALBUM_ID,
    E_ARTIST_ID,
  };
}

#include "parser/libxml2-sax-base.hh"
using Bmp::XPath;

#include "xspf-libxml2-sax.hh"

namespace
{
  Element_tag elements[] =
  {
      { "playlist"    },

      { "trackList"   },
      { "track"       },

      { "location"    },
      { "title"       },
      { "id"          },
      { "album"       },
      { "creator"     },
      { "duration"    },
      { "image"       },
      { "trackauth"   },
      { "albumId"     },
      { "artistId"    },
  };
  
  struct XSPFParserContext : public ParseContextBase
  {
    Bmp::XSPF::Item       m_item;
    Bmp::XSPF::Playlist & m_playlist;

    XSPFParserContext (Bmp::XSPF::Playlist & playlist)
      : ParseContextBase()
      , m_playlist (playlist)
    {}
  };

  typedef std::map < ustring, ustring > ElementAttributes; 

#define DEFAULT_REFS \
  XSPFParserContext & context (static_cast<XSPFParserContext&>(_context));  \
  Bmp::XSPF::Playlist & playlist (context.m_playlist);                      \
  Bmp::XSPF::Item & item (context.m_item);  

  //////////////////////////////////////////////////////////////////////////////

  namespace Handlers
  {
    namespace Track
    {
      HANDLER(trackList_track)
      {
        DEFAULT_REFS
        context.m_item = Bmp::XSPF::Item();
      }
    }
  }

  Handler_tag handlers[] = 
  {
    { XPath(elements[E_PLAYLIST]) / XPath(elements[E_TRACK_LIST]) / XPath(elements[E_TRACK]), 
      &Handlers::Track::trackList_track },
  };
  //---------
  
  namespace HandlersEnd
  {
    namespace Track
    {
      HANDLER_END(trackList_track)
      {
        DEFAULT_REFS
        playlist.items.push_back (item);
      }
    }
  }
  
  HandlerEnd_tag handlers_end[] =
  {
    { XPath(elements[E_PLAYLIST]) / XPath(elements[E_TRACK_LIST]) / XPath(elements[E_TRACK]), 
        &HandlersEnd::Track::trackList_track },
  };
  //---------
  
  namespace HandlersPCDATA
  {
    namespace Playlist
    {
      HANDLER_PCDATA(title)
      {
        DEFAULT_REFS
        playlist.title += text;
      }
    }

    HANDLER_PCDATA(location)
    {
      DEFAULT_REFS
      item.location += text;
    }

    HANDLER_PCDATA(title)
    {
      DEFAULT_REFS
      item.title += text;
    }

    HANDLER_PCDATA(id)
    {
      DEFAULT_REFS
      item.id = g_ascii_strtoull (text.c_str(), NULL, 10); 
    }

    HANDLER_PCDATA(album)
    {
      DEFAULT_REFS
      item.album += text;
    }

    HANDLER_PCDATA(creator)
    {
      DEFAULT_REFS
      item.creator += text;
    }

    HANDLER_PCDATA(duration)
    {
      DEFAULT_REFS
      item.duration = g_ascii_strtoull (text.c_str(), NULL, 10); 
    }

    HANDLER_PCDATA(image)
    {
      DEFAULT_REFS
      item.image += text; 
    }

    HANDLER_PCDATA(trackauth)
    {
      DEFAULT_REFS
      item.trackauth += text; 
    }

    HANDLER_PCDATA(albumId)
    {
      DEFAULT_REFS
      item.albumId = g_ascii_strtoull (text.c_str(), NULL, 10); 
    }

    HANDLER_PCDATA(artistId)
    {
      DEFAULT_REFS
      item.artistId = g_ascii_strtoull (text.c_str(), NULL, 10); 
    }
  }

  static XPath PathTrackBase (XPath(elements[E_PLAYLIST]) / XPath(elements[E_TRACK_LIST]) / XPath(elements[E_TRACK]));

  HandlerPCDATA_tag handlers_pcdata[] = 
  {
      { XPath(elements[E_PLAYLIST]) / XPath(elements[E_TITLE]),
        &HandlersPCDATA::Playlist::title},

      { PathTrackBase / XPath(elements[E_LOCATION]),
        &HandlersPCDATA::location},

      { PathTrackBase / XPath(elements[E_TITLE]),
        &HandlersPCDATA::title},

      { PathTrackBase / XPath(elements[E_ID]),
        &HandlersPCDATA::id},

      { PathTrackBase / XPath(elements[E_ALBUM]),
        &HandlersPCDATA::album},

      { PathTrackBase / XPath(elements[E_CREATOR]),
        &HandlersPCDATA::creator},

      { PathTrackBase / XPath(elements[E_DURATION]),
        &HandlersPCDATA::duration},

      { PathTrackBase / XPath(elements[E_IMAGE]),
        &HandlersPCDATA::image},

      { PathTrackBase / XPath(elements[E_TRACKAUTH]),
        &HandlersPCDATA::trackauth},

      { PathTrackBase / XPath(elements[E_ALBUM_ID]),
        &HandlersPCDATA::albumId},

      { PathTrackBase / XPath(elements[E_ARTIST_ID]),
        &HandlersPCDATA::artistId},
  };
  //---------
}

namespace Bmp
{
  namespace XSPF
  {
    int XSPF_parse (Bmp::XSPF::Playlist & playlist, std::string const& data)
    {
      XSPFParserContext context (playlist);

      for (unsigned int n = 0; n < G_N_ELEMENTS(handlers); ++n)
       {
         context.mHandlers.insert (std::make_pair (handlers[n].elementId, handlers[n].handler));
       }
       
       // handler/end 
       for (unsigned int n = 0; n < G_N_ELEMENTS(handlers_end); ++n)
       {
         context.mHandlersEnd.insert (std::make_pair (handlers_end[n].elementId, handlers_end[n].handler));
       }
       
       // handler/pcdata
       for (unsigned int n = 0; n < G_N_ELEMENTS(handlers_pcdata); ++n)
       {
         context.mHandlersPCDATA.insert (std::make_pair (handlers_pcdata[n].elementId, handlers_pcdata[n].handler));
       }
       
       // name <-> id map 
       for (unsigned int n = 0; n < G_N_ELEMENTS(elements); ++n)
       {
         context.mNameIdMap.insert (std::make_pair (elements[n], ElementId (n)));
       }

      return SaxParserBase::xml_base_parse(data, context);
    }
  }
}
