/***************************************************************************
 *            bonfire-project-manager.c
 *
 *  mer mai 24 14:22:56 2006
 *  Copyright  2006  Rouquier Philippe
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */



#include <string.h>

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>

#include <gtk/gtkstock.h>
#include <gtk/gtkaction.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtkfilefilter.h>
#include <gtk/gtkfilechooser.h>
#include <gtk/gtkfilechooserwidget.h>
#include <gtk/gtkfilechooserdialog.h>
#include <gtk/gtkmessagedialog.h>
#include <gtk/gtkuimanager.h>
#include <gtk/gtkscrolledwindow.h>

#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-file-info.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include "project.h"
#include "bonfire-layout.h"
#include "bonfire-project-manager.h"
#include "bonfire-file-chooser.h"
#include "burn-options-dialog.h"
#include "project-type-chooser.h"
#include "disc-copy-dialog.h"

#ifdef BUILD_SEARCH
#include "search.h"
#endif

#ifdef BUILD_PLAYLIST
#include "play-list.h"
#endif

#ifdef BUILD_PREVIEW
#include "player.h"
#endif

static void bonfire_project_manager_class_init (BonfireProjectManagerClass *klass);
static void bonfire_project_manager_init (BonfireProjectManager *sp);
static void bonfire_project_manager_finalize (GObject *object);

static void
bonfire_project_manager_type_changed_cb (BonfireProjectTypeChooser *chooser,
					 BonfireProjectType type,
					 BonfireProjectManager *manager);

static void
bonfire_project_manager_new_empty_prj_cb (GtkAction *action, BonfireProjectManager *manager);
static void
bonfire_project_manager_new_audio_prj_cb (GtkAction *action, BonfireProjectManager *manager);
static void
bonfire_project_manager_new_data_prj_cb (GtkAction *action, BonfireProjectManager *manager);
static void
bonfire_project_manager_new_copy_prj_cb (GtkAction *action, BonfireProjectManager *manager);
static void
bonfire_project_manager_new_iso_prj_cb (GtkAction *action, BonfireProjectManager *manager);
static void
bonfire_project_manager_open_cb (GtkAction *action, BonfireProjectManager *manager);

/* menus */
static GtkActionEntry entries [] = {
	{"New", GTK_STOCK_NEW, N_("New project"), NULL,
	 N_("Create a new project"), NULL },
	{"NewChoose", GTK_STOCK_NEW, N_("Empty project"), NULL,
	 N_("Let you choose your new project"), G_CALLBACK (bonfire_project_manager_new_empty_prj_cb)},
	{"NewAudio", NULL, N_("New audio project"), NULL,
	 N_("Create a new audio project"), G_CALLBACK (bonfire_project_manager_new_audio_prj_cb)},
	{"NewData", NULL, N_("New data project"), NULL,
	 N_("Create a new data project"), G_CALLBACK (bonfire_project_manager_new_data_prj_cb)},
	{"NewCopy", NULL, N_("Copy disc"), NULL,
	 N_("Copy a disc"), G_CALLBACK (bonfire_project_manager_new_copy_prj_cb)},
	{"NewIso", NULL, N_("Burn an image"), NULL,
	 N_("Burn an image"), G_CALLBACK (bonfire_project_manager_new_iso_prj_cb)},

	{"Open", GTK_STOCK_OPEN, N_("_Open"), NULL,
	 N_("Open a project"), G_CALLBACK (bonfire_project_manager_open_cb)},
};

static const char *description = {
	"<ui>"
	    "<menubar name='menubar' >"
		"<menu action='ProjectMenu'>"
			"<placeholder name='ViewPlaceholder'/>"

			"<menu action='New'>"
				"<menuitem action='NewAudio'/>"
				"<menuitem action='NewData'/>"
				"<menuitem action='NewCopy'/>"	
				"<menuitem action='NewIso'/>"	
			"</menu>"

			"<separator/>"
			"<placeholder name='ProjectPlaceholder'/>"
			    "<menuitem action='Open'/>"
			    "<separator/>"
		"</menu>"
	    "</menubar>"

	    "<toolbar name='toolbar'>"
		"<toolitem action='NewChoose'/>"
		"<toolitem action='Open'/>"
		"<toolitem action='Save'/>"
		"<separator/>"
	    "</toolbar>"
	"</ui>"
};

struct BonfireProjectManagerPrivate {
	GtkWidget *project;
	GtkWidget *layout;

	GtkActionGroup *action_group;
};

static GObjectClass *parent_class = NULL;

GType
bonfire_project_manager_get_type ()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireProjectManagerClass),
			NULL,
			NULL,
			(GClassInitFunc)bonfire_project_manager_class_init,
			NULL,
			NULL,
			sizeof (BonfireProjectManager),
			0,
			(GInstanceInitFunc)bonfire_project_manager_init,
		};

		type = g_type_register_static (GTK_TYPE_NOTEBOOK,
					       "BonfireProjectManager",
					       &our_info,
					       0);
	}

	return type;
}

static void
bonfire_project_manager_class_init (BonfireProjectManagerClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent(klass);
	object_class->finalize = bonfire_project_manager_finalize;
}

static void
bonfire_project_manager_init (BonfireProjectManager *obj)
{
	GtkWidget *type;
	GtkWidget *scroll;
	GtkWidget *chooser;

	obj->priv = g_new0 (BonfireProjectManagerPrivate, 1);

	gtk_notebook_set_show_border (GTK_NOTEBOOK (obj), FALSE);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (obj), FALSE);

	obj->priv->action_group = gtk_action_group_new ("ProjectManagerAction");
	gtk_action_group_set_translation_domain (obj->priv->action_group, GETTEXT_PACKAGE);
	gtk_action_group_add_actions (obj->priv->action_group,
				      entries,
				      G_N_ELEMENTS (entries),
				      obj);

	/* add the project type chooser to the notebook */
	type = bonfire_project_type_chooser_new ();
	gtk_widget_show (type);
	scroll = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_show (scroll);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scroll), type);

	g_signal_connect (G_OBJECT (type),
			  "chosen",
			  G_CALLBACK (bonfire_project_manager_type_changed_cb),
			  obj);

	gtk_notebook_prepend_page (GTK_NOTEBOOK (obj), scroll, NULL);

	/* add the layout */
	obj->priv->layout = bonfire_layout_new ();
	gtk_widget_show (obj->priv->layout);
	gtk_notebook_append_page (GTK_NOTEBOOK (obj), obj->priv->layout, NULL);

	/* create the project for audio and data discs */
	obj->priv->project = bonfire_project_new ();
	gtk_widget_show_all (obj->priv->project);

#ifdef BUILD_PREVIEW
	GtkWidget *preview;

	preview = bonfire_player_new ();
	gtk_widget_show (preview);
	bonfire_player_add_source (BONFIRE_PLAYER (preview),
				   BONFIRE_URI_CONTAINER (obj->priv->project));

#endif /* BUILD_PREVIEW */

	chooser = bonfire_file_chooser_new ();
	gtk_widget_show_all (chooser);
	bonfire_layout_add_source (BONFIRE_LAYOUT (obj->priv->layout),
				   chooser,
				   "Chooser",
				   _("<big><b>Choose a file</b></big>"),
				   _("File chooser"),
				   _("Display file chooser"),
				   GTK_STOCK_DIRECTORY, 
				   BONFIRE_LAYOUT_AUDIO|BONFIRE_LAYOUT_DATA);
	bonfire_project_add_source (BONFIRE_PROJECT (obj->priv->project),
				    BONFIRE_URI_CONTAINER (chooser));

#ifdef BUILD_PREVIEW
	bonfire_player_add_source (BONFIRE_PLAYER (preview),
				   BONFIRE_URI_CONTAINER (chooser));
#endif

	bonfire_layout_add_project (BONFIRE_LAYOUT (obj->priv->layout),
				    obj->priv->project);
#ifdef BUILD_SEARCH
	GtkWidget *search;

	search = bonfire_search_new ();
	gtk_widget_show_all (search);
	bonfire_layout_add_source (BONFIRE_LAYOUT (obj->priv->layout),
				   search,
				   "Search",
				   _("<big><b>Search a file</b></big>"),
				   _("Search a file"),
				   _("Display search"),
				   GTK_STOCK_FIND, 
				   BONFIRE_LAYOUT_AUDIO|BONFIRE_LAYOUT_DATA);
	bonfire_project_add_source (BONFIRE_PROJECT (obj->priv->project),
				    BONFIRE_URI_CONTAINER (search));

#ifdef BUILD_PREVIEW
	bonfire_player_add_source (BONFIRE_PLAYER (preview),
				   BONFIRE_URI_CONTAINER (search));
#endif

#endif /* BUILD_SEARCH */

#ifdef BUILD_PLAYLIST
	GtkWidget *playlist;

	playlist = bonfire_playlist_new ();
	gtk_widget_show_all (playlist);
	bonfire_layout_add_source (BONFIRE_LAYOUT (obj->priv->layout),
				   playlist,
				   "Playlist",
				   _("<big><b>Playlists</b></big>"),
				   _("Playlist"),
				   _("Display playlist"),
				   GTK_STOCK_INDEX, 
				   BONFIRE_LAYOUT_AUDIO);
	bonfire_project_add_source (BONFIRE_PROJECT (obj->priv->project),
				    BONFIRE_URI_CONTAINER (playlist));

#ifdef BUILD_PREVIEW
	bonfire_player_add_source (BONFIRE_PLAYER (preview),
				   BONFIRE_URI_CONTAINER (playlist));

	bonfire_layout_add_preview (BONFIRE_LAYOUT (obj->priv->layout),
				    preview);
#endif

#endif /* BUILD_PLAYLIST */
}

static void
bonfire_project_manager_finalize (GObject *object)
{
	BonfireProjectManager *cobj;

	cobj = BONFIRE_PROJECT_MANAGER (object);
	g_free (cobj->priv);

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

GtkWidget *
bonfire_project_manager_new ()
{
	BonfireProjectManager *obj;
	
	obj = BONFIRE_PROJECT_MANAGER (g_object_new (BONFIRE_TYPE_PROJECT_MANAGER, NULL));
	
	return GTK_WIDGET (obj);
}

void
bonfire_project_manager_register_menu (BonfireProjectManager *manager,
				       GtkUIManager *ui_manager)
{
	GError *error = NULL;

	gtk_ui_manager_insert_action_group (ui_manager, manager->priv->action_group, 0);
	if (!gtk_ui_manager_add_ui_from_string (ui_manager, description, -1, &error)) {
		g_message ("building menus failed: %s", error->message);
		g_error_free (error);
	}

	bonfire_layout_register_menu (BONFIRE_LAYOUT (manager->priv->layout), ui_manager);
	bonfire_project_register_menu (BONFIRE_PROJECT (manager->priv->project), ui_manager);
}

static void
bonfire_project_manager_burn_iso (BonfireProjectManager *manager,
				  const char *image_uri)
{
	BonfireTrackSource *track = NULL;
	GnomeVFSFileInfo *info;
	GnomeVFSResult result;
	GtkWidget *toplevel;
	GtkWidget *dialog;
	char *escaped_uri;
	char *uri;

	if (!image_uri) {
		int answer;

		toplevel = gtk_widget_get_toplevel (GTK_WIDGET (manager));
		dialog = gtk_file_chooser_dialog_new (_("Choose a disc image or a cue file"),
						      GTK_WINDOW (toplevel),
						      GTK_FILE_CHOOSER_ACTION_OPEN,
						      GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
						      GTK_STOCK_OPEN, GTK_RESPONSE_OK,
						      NULL);

		gtk_file_chooser_set_local_only (GTK_FILE_CHOOSER (dialog), FALSE);
		gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog),
						     g_get_home_dir ());
	
		gtk_widget_show (dialog);
		answer = gtk_dialog_run (GTK_DIALOG (dialog));

		if (answer != GTK_RESPONSE_OK) {
			gtk_widget_destroy (dialog);
			return;
		}

		uri = gtk_file_chooser_get_uri (GTK_FILE_CHOOSER (dialog));
		gtk_widget_destroy (dialog);
	}
	else
		uri = g_strdup (image_uri);

	/* check if it is an iso and if it is not a remote file */
	info = gnome_vfs_file_info_new ();
	escaped_uri = gnome_vfs_escape_host_and_path_string (uri);
	result = gnome_vfs_get_file_info (escaped_uri,
					  info,
					  GNOME_VFS_FILE_INFO_GET_MIME_TYPE|
					  GNOME_VFS_FILE_INFO_FORCE_SLOW_MIME_TYPE);
	g_free (escaped_uri);

	if (result != GNOME_VFS_OK)
		goto error;

	if (!strcmp (info->mime_type, "application/x-cd-image")) {
		track = g_new0 (BonfireTrackSource, 1);
		track->type = BONFIRE_TRACK_SOURCE_ISO;
		track->contents.iso.image = gnome_vfs_get_local_path_from_uri (uri);
	}
	else if (!strcmp (info->mime_type, "text/plain")) {
		track = g_new0 (BonfireTrackSource, 1);
		track->type = BONFIRE_TRACK_SOURCE_CUE;
		track->contents.cue.toc = gnome_vfs_get_local_path_from_uri (uri);
	}
	else 
		goto error;

	g_free (uri);
	gnome_vfs_file_info_clear (info);
	gnome_vfs_file_info_unref (info);

	/* create the burn dialog */
	toplevel = gtk_widget_get_toplevel (GTK_WIDGET (manager));
	dialog = bonfire_burn_option_dialog_new (track, FALSE);

	gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (toplevel));
	gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
	gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER_ON_PARENT);
	gtk_widget_show_all (dialog);

	if (track)
		bonfire_track_source_free (track);

	return;


error:
	{
		GtkWidget *dialog;
		GtkWidget *toplevel;

		gnome_vfs_file_info_clear (info);
		gnome_vfs_file_info_unref (info);

		toplevel = gtk_widget_get_toplevel (GTK_WIDGET (manager));
		dialog = gtk_message_dialog_new (GTK_WINDOW (toplevel),
						 GTK_DIALOG_DESTROY_WITH_PARENT |
						 GTK_DIALOG_MODAL,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_CLOSE,
						 _("This image can't be burnt:"));

		gtk_window_set_title (GTK_WINDOW (dialog), _("Burning failure"));

		gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
							  _("it doesn't appear to be a valid image or cue file."));
	
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);

	}
}

static void
bonfire_project_manager_switch (BonfireProjectManager *manager,
				BonfireProjectType type,
				const char *uri)
{
	GtkWidget *toplevel;
	GtkAction *action;

	toplevel = gtk_widget_get_toplevel (GTK_WIDGET (manager));
	action = gtk_action_group_get_action (manager->priv->action_group, "NewChoose");

	if (type == BONFIRE_PROJECT_TYPE_INVALID) {
		bonfire_layout_load (BONFIRE_LAYOUT (manager->priv->layout), BONFIRE_LAYOUT_NONE);
		bonfire_project_set_none (BONFIRE_PROJECT (manager->priv->project));

		gtk_notebook_set_current_page (GTK_NOTEBOOK (manager), 0);
		gtk_action_set_sensitive (action, FALSE);

		if (toplevel)
			gtk_window_set_title (GTK_WINDOW (toplevel), "Bonfire");
	}
	else if (type == BONFIRE_PROJECT_TYPE_AUDIO) {
		bonfire_layout_load (BONFIRE_LAYOUT (manager->priv->layout), BONFIRE_LAYOUT_AUDIO);

		gtk_notebook_set_current_page (GTK_NOTEBOOK (manager), 1);
		gtk_action_set_sensitive (action, TRUE);

		/* tell the BonfireProject object that we want an audio selection */
		bonfire_project_set_audio (BONFIRE_PROJECT (manager->priv->project));

		if (toplevel)
			gtk_window_set_title (GTK_WINDOW (toplevel), _("Bonfire - New audio disc project"));
	}
	else if (type == BONFIRE_PROJECT_TYPE_DATA) {
		bonfire_layout_load (BONFIRE_LAYOUT (manager->priv->layout), BONFIRE_LAYOUT_DATA);

		gtk_notebook_set_current_page (GTK_NOTEBOOK (manager), 1);
		gtk_action_set_sensitive (action, TRUE);

		/* tell the BonfireProject object that we want a data selection */
		bonfire_project_set_data (BONFIRE_PROJECT (manager->priv->project));

		if (toplevel)
			gtk_window_set_title (GTK_WINDOW (toplevel), _("Bonfire - New data disc project"));
	}
	else if (type == BONFIRE_PROJECT_TYPE_ISO) {
		bonfire_layout_load (BONFIRE_LAYOUT (manager->priv->layout), BONFIRE_LAYOUT_NONE);
		bonfire_project_set_none (BONFIRE_PROJECT (manager->priv->project));

		gtk_notebook_set_current_page (GTK_NOTEBOOK (manager), 0);
		gtk_action_set_sensitive (action, FALSE);

		if (toplevel)
			gtk_window_set_title (GTK_WINDOW (toplevel), _("Bonfire - New image file"));
		bonfire_project_manager_burn_iso (manager, uri);
	}
	else if (type == BONFIRE_PROJECT_TYPE_COPY) {
		GtkWidget *dialog;

		bonfire_layout_load (BONFIRE_LAYOUT (manager->priv->layout), BONFIRE_LAYOUT_NONE);
		bonfire_project_set_none (BONFIRE_PROJECT (manager->priv->project));

		gtk_notebook_set_current_page (GTK_NOTEBOOK (manager), 0);
		gtk_action_set_sensitive (action, FALSE);

		if (toplevel)
			gtk_window_set_title (GTK_WINDOW (toplevel), _("Bonfire - Copy a disc"));

		dialog = bonfire_disc_copy_dialog_new ();
	
		gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (toplevel));
		gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
		gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER_ON_PARENT);

		gtk_widget_show_all (dialog);
	}
}

static void
bonfire_project_manager_type_changed_cb (BonfireProjectTypeChooser *chooser,
					 BonfireProjectType type,
					 BonfireProjectManager *manager)
{
	bonfire_project_manager_switch (manager, type, NULL);
}

static void
bonfire_project_manager_new_empty_prj_cb (GtkAction *action, BonfireProjectManager *manager)
{
	bonfire_project_manager_switch (manager, BONFIRE_PROJECT_TYPE_INVALID, NULL);
}

static void
bonfire_project_manager_new_audio_prj_cb (GtkAction *action, BonfireProjectManager *manager)
{
	bonfire_project_manager_switch (manager, BONFIRE_PROJECT_TYPE_AUDIO, NULL);
}

static void
bonfire_project_manager_new_data_prj_cb (GtkAction *action, BonfireProjectManager *manager)
{
	bonfire_project_manager_switch (manager, BONFIRE_PROJECT_TYPE_DATA, NULL);
}

static void
bonfire_project_manager_new_copy_prj_cb (GtkAction *action, BonfireProjectManager *manager)
{
	bonfire_project_manager_switch (manager, BONFIRE_PROJECT_TYPE_COPY, NULL);
}

static void
bonfire_project_manager_new_iso_prj_cb (GtkAction *action, BonfireProjectManager *manager)
{
	bonfire_project_manager_switch (manager, BONFIRE_PROJECT_TYPE_ISO, NULL);
}

static void
bonfire_project_manager_open_cb (GtkAction *action, BonfireProjectManager *manager)
{
	BonfireProjectType type;

	type = bonfire_project_open_project (BONFIRE_PROJECT (manager->priv->project), NULL);
	if (type == BONFIRE_PROJECT_TYPE_INVALID)
		bonfire_project_manager_switch (manager, BONFIRE_PROJECT_TYPE_INVALID, NULL);
	else if (type == BONFIRE_PROJECT_TYPE_DATA)
		bonfire_layout_load (BONFIRE_LAYOUT (manager->priv->layout), BONFIRE_LAYOUT_DATA);
	else
		bonfire_layout_load (BONFIRE_LAYOUT (manager->priv->layout), BONFIRE_LAYOUT_AUDIO);
	gtk_notebook_set_current_page (GTK_NOTEBOOK (manager), 1);

	action = gtk_action_group_get_action (manager->priv->action_group, "NewChoose");
	gtk_action_set_sensitive (action, TRUE);
}

void
bonfire_project_manager_audio (BonfireProjectManager *manager)
{
	bonfire_project_manager_switch (manager, BONFIRE_PROJECT_TYPE_AUDIO, NULL);
}

void
bonfire_project_manager_data (BonfireProjectManager *manager)
{
	bonfire_project_manager_switch (manager, BONFIRE_PROJECT_TYPE_DATA, NULL);
}

void
bonfire_project_manager_copy (BonfireProjectManager *manager)
{
	bonfire_project_manager_switch (manager, BONFIRE_PROJECT_TYPE_COPY, NULL);
}

void
bonfire_project_manager_iso (BonfireProjectManager *manager, const char *uri)
{
	bonfire_project_manager_switch (manager, BONFIRE_PROJECT_TYPE_ISO, NULL);
}

void
bonfire_project_manager_open (BonfireProjectManager *manager, const char *uri)
{
	BonfireProjectType type;

	gtk_notebook_set_current_page (GTK_NOTEBOOK (manager), 1);
	type = bonfire_project_open_project (BONFIRE_PROJECT (manager->priv->project), uri);

	if (type == BONFIRE_PROJECT_TYPE_INVALID)
		bonfire_project_manager_switch (manager, BONFIRE_PROJECT_TYPE_INVALID, NULL);
	else if (type == BONFIRE_PROJECT_TYPE_DATA)
		bonfire_layout_load (BONFIRE_LAYOUT (manager->priv->layout), BONFIRE_LAYOUT_DATA);
	else
		bonfire_layout_load (BONFIRE_LAYOUT (manager->priv->layout), BONFIRE_LAYOUT_AUDIO);
}

void
bonfire_project_manager_empty (BonfireProjectManager *manager)
{
	bonfire_project_manager_switch (manager, BONFIRE_PROJECT_TYPE_INVALID, NULL);
}
