/***************************************************************************
 *            burn-caps.c
 *
 *  mar avr 18 20:58:42 2006
 *  Copyright  2006  Rouquier Philippe
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <glib.h>
#include <glib-object.h>

#include <nautilus-burn-drive.h>

/* NOTE: the day we want to make a library out of this we might want to avoid
 * the following dependency. One idea could be to define an interface BonfireBurnCaps
 * and pass an object that implements such an interface to BonfireBurn every time.
 * That way we could have various implementations of the "autoconfigure" could be
 * possible and the GConf dep would remain in bonfire. */
#include <gconf/gconf-client.h>

#include "burn-basics.h"
#include "burn-caps.h"
#include "burn-imager.h"
#include "burn-recorder.h"
#include "burn-readcd.h"
#include "burn-transcode.h"
#include "burn-mkisofs-base.h"
#include "burn-mkisofs.h"
#include "burn-cdrdao.h"
#include "burn-cdrecord.h"
#include "burn-growisofs.h"
#include "burn-dvd-rw-format.h"

/* This object is intended to autoconfigure the burn-engine. It should provide
 * supported and safest default flags according to system config and arguments
 * given, and also creates the most appropriate recorder and imager objects */

/* FIXME: extension
	  gboolean cdrecord_02  => no on the fly
	  gboolean cdrecord_02_1 => no the fly for audio
	  gboolean dvdrecord
*/
	 

static void bonfire_burn_caps_class_init (BonfireBurnCapsClass *klass);
static void bonfire_burn_caps_init (BonfireBurnCaps *sp);
static void bonfire_burn_caps_finalize (GObject *object);

struct BonfireBurnCapsPrivate {
	/* if FALSE: we can't create *.cue files and can't copy on the fly */
	gboolean cdrdao_disabled;
};

#define GCONF_KEY_CDRDAO_DISABLED	"/apps/bonfire/config/cdrdao_disabled"

static GObjectClass *parent_class = NULL;
static BonfireBurnCaps *default_caps = NULL;

GType
bonfire_burn_caps_get_type ()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireBurnCapsClass),
			NULL,
			NULL,
			(GClassInitFunc)bonfire_burn_caps_class_init,
			NULL,
			NULL,
			sizeof (BonfireBurnCaps),
			0,
			(GInstanceInitFunc)bonfire_burn_caps_init,
		};

		type = g_type_register_static (G_TYPE_OBJECT,
					       "BonfireBurnCaps",
					       &our_info,
					       0);
	}

	return type;
}

static void
bonfire_burn_caps_class_init (BonfireBurnCapsClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);
	object_class->finalize = bonfire_burn_caps_finalize;
}

static void
bonfire_burn_caps_init (BonfireBurnCaps *obj)
{
	GConfClient *client;

	obj->priv = g_new0 (BonfireBurnCapsPrivate, 1);

	/* load our "configuration" */
	client = gconf_client_get_default ();
	obj->priv->cdrdao_disabled = gconf_client_get_bool (client,
							    GCONF_KEY_CDRDAO_DISABLED,
							    NULL);
	g_object_unref (client);
}

static void
bonfire_burn_caps_finalize (GObject *object)
{
	BonfireBurnCaps *cobj;

	cobj = BONFIRE_BURNCAPS(object);
	default_caps = NULL;
		
	g_free (cobj->priv);
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

BonfireBurnCaps *
bonfire_burn_caps_get_default ()
{
	if (!default_caps) 
		default_caps = BONFIRE_BURNCAPS (g_object_new (BONFIRE_TYPE_BURNCAPS, NULL));
	else
		g_object_ref (default_caps);

	return default_caps;
}

/* that function receives all errors returned by the object and 'learns' from 
 * these errors what are the safest defaults for a particular system. It should 
 * also offer fallbacks if an error occurs through a signal */
static BonfireBurnResult
bonfire_burn_caps_job_error_cb (BonfireJob *job,
				BonfireBurnError error,
				BonfireBurnCaps *caps)
{
	if (BONFIRE_IS_CDRDAO (job) && error == BONFIRE_BURN_ERROR_SCSI_IOCTL) {
		GError *error = NULL;
		GConfClient *client;

		/* This is for a bug in fedora 5 that prevents from sending
		 * SCSI commands as a normal user through cdrdao. There is a
		 * fallback fortunately with cdrecord and raw images but no 
		 * on_the_fly burning */
		caps->priv->cdrdao_disabled = 1;

		/* set it in GConf to remember that next time */
		client = gconf_client_get_default ();
		gconf_client_set_bool (client, GCONF_KEY_CDRDAO_DISABLED, TRUE, &error);
		if (error) {
			g_warning ("Can't write with GConf: %s\n", error->message);
			g_error_free (error);
		}
		g_object_unref (client);

		return BONFIRE_BURN_RETRY;
	}

	return BONFIRE_BURN_ERR;
}

/* sets the best (safest) appropriate flags given the information
 * and the system configuration we have */
BonfireBurnResult
bonfire_burn_caps_get_default_flags (BonfireBurnCaps *caps,
				     BonfireTrackSourceType track_type,
				     NautilusBurnDriveType drive_type,
				     NautilusBurnMediaType media_type,
				     BonfireBurnFlag *flags)
{
	BonfireBurnFlag default_flags = BONFIRE_BURN_FLAG_CHECK_SIZE |
					  BONFIRE_BURN_FLAG_NOGRACE |
					  BONFIRE_BURN_FLAG_EJECT;

	if (drive_type != NAUTILUS_BURN_DRIVE_TYPE_FILE) {
		default_flags |= BONFIRE_BURN_FLAG_BLANK_BEFORE_WRITE;
		default_flags |= BONFIRE_BURN_FLAG_BURNPROOF;

		if (track_type == BONFIRE_TRACK_SOURCE_GRAFTS
		||  track_type == BONFIRE_TRACK_SOURCE_DATA
		||  track_type == BONFIRE_TRACK_SOURCE_IMAGER)
			default_flags |= BONFIRE_BURN_FLAG_ON_THE_FLY;
	
		if (track_type == BONFIRE_TRACK_SOURCE_DISC) {
			/* in cd-to-cd the only way to use on the fly burning is
			 * to have a CD and cdrdao working */
			if (media_type <= NAUTILUS_BURN_MEDIA_TYPE_CDRW
			&&  !caps->priv->cdrdao_disabled)
				default_flags |= BONFIRE_BURN_FLAG_ON_THE_FLY;
		}

		/* NOTE: for audio burning ON_THE_FLY is possible
		 * (see below) but is no a default */
	}
	else
		default_flags |= BONFIRE_BURN_FLAG_DONT_CLEAN_OUTPUT;

	if (flags)
		*flags = default_flags;

	return BONFIRE_BURN_OK;
}

/* returns the flags that can be used given the information
 * that were passed and the system configuration 
 * track_type is the type of track that is going to be passed to the recorder */
BonfireBurnResult
bonfire_burn_caps_get_supported_flags (BonfireBurnCaps *caps,
				       BonfireTrackSourceType track_type,
				       NautilusBurnDriveType drive_type,
				       NautilusBurnMediaType media_type,
				       BonfireBurnFlag *flags)
{
	BonfireBurnFlag supported_flags = BONFIRE_BURN_FLAG_DONT_OVERWRITE|
					    BONFIRE_BURN_FLAG_CHECK_SIZE|
					    BONFIRE_BURN_FLAG_NOGRACE|
					    BONFIRE_BURN_FLAG_EJECT|
					    BONFIRE_BURN_FLAG_DEBUG; /* always supported */

	if (drive_type != NAUTILUS_BURN_DRIVE_TYPE_FILE) {
		supported_flags |= BONFIRE_BURN_FLAG_BLANK_BEFORE_WRITE;
		supported_flags |= BONFIRE_BURN_FLAG_BURNPROOF;

		/* we don't support this for DVD+-RW. Growisofs doesn't honour the option */
		if (media_type != NAUTILUS_BURN_MEDIA_TYPE_DVD_PLUS_RW)
			supported_flags |= BONFIRE_BURN_FLAG_DUMMY;

		if (track_type == BONFIRE_TRACK_SOURCE_DISC) {
			/* in cd-to-cd copy we can:
			 * - make an accurate copy (on the fly) burning with cdrdao
			 * - make an accurate copy with an image with readcd -clone (RAW)
			 * - make a copy of single session CD (on the fly) with readcd (ISO).
			 *   that's what we do with DVD for example.
			 * so if no cdrdao => no on the fly */
			if (media_type <= NAUTILUS_BURN_MEDIA_TYPE_CDRW
			&&  !caps->priv->cdrdao_disabled)
				supported_flags |= BONFIRE_BURN_FLAG_ON_THE_FLY;
		}
		else if (track_type == BONFIRE_TRACK_SOURCE_CUE
		      ||  track_type == BONFIRE_TRACK_SOURCE_RAW) {
			/* *.cue file and *.raw file only work with CDs */
			if (media_type <= NAUTILUS_BURN_MEDIA_TYPE_CDRW)
				return BONFIRE_BURN_NOT_SUPPORTED;

			/* NOTE: no need for ON_THE_FLY with _ISO or _ISO_JOLIET
			 * since image is already done */
		}
		else if (track_type == BONFIRE_TRACK_SOURCE_SONG
		      || track_type == BONFIRE_TRACK_SOURCE_AUDIO) {
			/* for audio burning our capabilities are limited to CDs */
			if (media_type > NAUTILUS_BURN_MEDIA_TYPE_CDRW)
				return BONFIRE_BURN_NOT_SUPPORTED;
			else
				supported_flags |= BONFIRE_BURN_FLAG_ON_THE_FLY;
		}
		else if (track_type == BONFIRE_TRACK_SOURCE_GRAFTS
		      ||  track_type == BONFIRE_TRACK_SOURCE_DATA)
			supported_flags |= BONFIRE_BURN_FLAG_ON_THE_FLY|
					   BONFIRE_BURN_FLAG_DONT_CLOSE|
					   BONFIRE_BURN_FLAG_APPEND|
					   BONFIRE_BURN_FLAG_MERGE;
		else if (track_type == BONFIRE_TRACK_SOURCE_IMAGER)
			supported_flags |= BONFIRE_BURN_FLAG_ON_THE_FLY;

		/* NOTE: with _ISO or _ISO_JOLIET no need for ON_THE_FLY since
		 * image is already done */
	}

	if (flags)
		*flags = supported_flags;

	return BONFIRE_BURN_OK;
}

BonfireBurnFlag
bonfire_burn_caps_check_flags_consistency (BonfireBurnCaps *caps,
					   BonfireTrackSourceType track_type,
					   NautilusBurnDriveType drive_type,
					   NautilusBurnMediaType media_type,
					   BonfireBurnFlag flags)
{
	BonfireBurnFlag retval;
	BonfireBurnFlag supported;

	/* we make sure first that all the flags given are supported */
	bonfire_burn_caps_get_supported_flags (caps,
					       track_type,
					       drive_type,
					       media_type,
					       &supported);
	retval = flags & supported;
	if (retval != flags)
		g_warning ("Some flags were not supported (%i => %i). Corrected\n",
			   flags,
			   retval);

	/* we check flags consistency 
	 * NOTE: should we return an error if they are not consistent? */
	if ((track_type != BONFIRE_TRACK_SOURCE_SONG
	&&   track_type != BONFIRE_TRACK_SOURCE_DATA
	&&   track_type != BONFIRE_TRACK_SOURCE_GRAFTS
	&&   track_type != BONFIRE_TRACK_SOURCE_DISC)
	||   drive_type == NAUTILUS_BURN_DRIVE_TYPE_FILE) {
		if (retval & BONFIRE_BURN_FLAG_MERGE) {
			g_warning ("Inconsistent flag: you can't use flag merge\n");
			retval &= ~BONFIRE_BURN_FLAG_MERGE;
		}
			
		if (retval & BONFIRE_BURN_FLAG_APPEND) {
			g_warning ("Inconsistent flags: you can't use flag append\n");
			retval &= ~BONFIRE_BURN_FLAG_APPEND;
		}

		if (retval & BONFIRE_BURN_FLAG_ON_THE_FLY) {
			g_warning ("Inconsistent flag: you can't use flag on_the_fly\n");
			retval &= ~BONFIRE_BURN_FLAG_ON_THE_FLY;
		}
	}

	if ((retval & (BONFIRE_BURN_FLAG_MERGE|BONFIRE_BURN_FLAG_APPEND)) != 0
	&&  (retval & BONFIRE_BURN_FLAG_BLANK_BEFORE_WRITE) != 0) {
		g_warning ("Inconsistent flag: you can't use flag blank_before_write\n");
		retval &= ~BONFIRE_BURN_FLAG_BLANK_BEFORE_WRITE;
	}

	if (drive_type == NAUTILUS_BURN_DRIVE_TYPE_FILE
	&&  (retval & BONFIRE_BURN_FLAG_DONT_CLEAN_OUTPUT) == 0) {
		g_warning ("Inconsistent flag: you can't use flag blank_before_write\n");
		retval |= BONFIRE_BURN_FLAG_DONT_CLEAN_OUTPUT;
	}

	/* since we use growisofs we have to check that */
	if ((track_type == BONFIRE_TRACK_SOURCE_GRAFTS
	||   track_type == BONFIRE_TRACK_SOURCE_DATA)
	&&  media_type > NAUTILUS_BURN_MEDIA_TYPE_CDRW
	&&  (flags & (BONFIRE_BURN_FLAG_MERGE|BONFIRE_BURN_FLAG_APPEND))
	&&  !(retval & BONFIRE_BURN_FLAG_ON_THE_FLY)) {
		g_warning ("Merging data on a DVD: adding on_the_fly flag\n");
		retval |= BONFIRE_BURN_FLAG_ON_THE_FLY;
	}

	return retval;
}

BonfireBurnResult
bonfire_burn_caps_create_recorder (BonfireBurnCaps *caps,
				   BonfireRecorder **recorder,
				   BonfireTrackSource *source,
				   NautilusBurnMediaType media_type,
				   GError **error)
{
	BonfireRecorder *obj = NULL;

	switch (source->type) {
	case BONFIRE_TRACK_SOURCE_IMAGER:
		if (BONFIRE_IS_READCD (source->contents.imager.obj)) {
			if (media_type > NAUTILUS_BURN_MEDIA_TYPE_CDRW)
				obj = BONFIRE_RECORDER (g_object_new (BONFIRE_TYPE_GROWISOFS, NULL));
			else
				obj = BONFIRE_RECORDER (g_object_new (BONFIRE_TYPE_CD_RECORD, NULL));
		}
		else if (BONFIRE_IS_TRANSCODE (source->contents.imager.obj)) {
			obj = BONFIRE_RECORDER (g_object_new (BONFIRE_TYPE_CD_RECORD, NULL));
		}
		else if (BONFIRE_IS_GROWISOFS (source->contents.imager.obj)) {
			obj = BONFIRE_RECORDER (source->contents.imager.obj);
			g_object_ref (obj);
		}
		else if (BONFIRE_IS_CDRDAO (source->contents.imager.obj)) {
			/* just to be sure and have no incoherence:
			 * better an error than a loop*/
			if (caps->priv->cdrdao_disabled)
				return BONFIRE_BURN_ERR;

			obj = BONFIRE_RECORDER (source->contents.imager.obj);
			g_object_ref (obj);
		}
		else if (BONFIRE_IS_MKISOFS (source->contents.imager.obj))
			obj = BONFIRE_RECORDER (g_object_new (BONFIRE_TYPE_CD_RECORD, NULL));
		else
			return BONFIRE_BURN_NOT_SUPPORTED;

		break;

	case BONFIRE_TRACK_SOURCE_AUDIO:
		if (media_type > NAUTILUS_BURN_MEDIA_TYPE_CDRW)
			return BONFIRE_BURN_NOT_SUPPORTED;

		obj = BONFIRE_RECORDER (g_object_new (BONFIRE_TYPE_CD_RECORD, NULL));
		break;

	case BONFIRE_TRACK_SOURCE_ISO:
	case BONFIRE_TRACK_SOURCE_ISO_JOLIET:
		if (media_type > NAUTILUS_BURN_MEDIA_TYPE_CDRW)
			obj = BONFIRE_RECORDER (g_object_new (BONFIRE_TYPE_GROWISOFS, NULL));
		else
			obj = BONFIRE_RECORDER (g_object_new (BONFIRE_TYPE_CD_RECORD, NULL));
		break;

	case BONFIRE_TRACK_SOURCE_CUE:
		if (media_type > NAUTILUS_BURN_MEDIA_TYPE_CDRW)
			return BONFIRE_BURN_NOT_SUPPORTED;

		/* here we have to solution either cdrdao or cdrecord but we'll
		 * use cdrdao only as a fallback except for on the fly burning
		 * see above in IMAGERS */
		if (!caps->priv->cdrdao_disabled)
			obj = BONFIRE_RECORDER (g_object_new (BONFIRE_TYPE_CDRDAO, NULL));
		else
			obj = BONFIRE_RECORDER (g_object_new (BONFIRE_TYPE_CD_RECORD, NULL));
		break;

	case BONFIRE_TRACK_SOURCE_RAW:
		if (media_type > NAUTILUS_BURN_MEDIA_TYPE_CDRW)
			return BONFIRE_BURN_NOT_SUPPORTED;

		obj = BONFIRE_RECORDER (g_object_new (BONFIRE_TYPE_CD_RECORD, NULL));
		break;

	default:
		return BONFIRE_BURN_NOT_SUPPORTED;
	}

	*recorder = obj;

	/* connect to the error signal to detect error and autoconfigure */
	g_signal_connect (obj,
			  "error",
			  G_CALLBACK (bonfire_burn_caps_job_error_cb),
			  caps);

	return BONFIRE_BURN_OK;
}

BonfireBurnResult
bonfire_burn_caps_create_recorder_for_blanking (BonfireBurnCaps *caps,
						BonfireRecorder **recorder,
						NautilusBurnMediaType type,
						GError **error)
{
	BonfireRecorder *obj;

	if (type > NAUTILUS_BURN_MEDIA_TYPE_CDRW) {
		if (type == NAUTILUS_BURN_MEDIA_TYPE_DVDRW || type == NAUTILUS_BURN_MEDIA_TYPE_DVD_PLUS_RW)
			obj = BONFIRE_RECORDER (g_object_new (BONFIRE_TYPE_GROWISOFS, NULL));
		else
			obj = BONFIRE_RECORDER (g_object_new (BONFIRE_TYPE_DVD_RW_FORMAT, NULL));
	}
	else if (type == NAUTILUS_BURN_MEDIA_TYPE_CDRW)
		obj = BONFIRE_RECORDER (g_object_new (BONFIRE_TYPE_CD_RECORD, NULL));
	else
		return BONFIRE_BURN_NOT_SUPPORTED;

	*recorder = obj;
	return BONFIRE_BURN_OK;
}

/* returns the available targets for the given settings and system configuration.
 * NOTE: the first type in the array is the default one */
BonfireBurnResult
bonfire_burn_caps_get_imager_available_targets (BonfireBurnCaps *caps,
						BonfireTrackSourceType **targets,
						BonfireTrackSourceType source_type,
						NautilusBurnMediaType src_media_type)
{
	BonfireTrackSourceType *retval;

	switch (source_type) {
	case BONFIRE_TRACK_SOURCE_SONG:
		retval = g_new0 (BonfireTrackSourceType, 3);
		retval [0] = BONFIRE_TRACK_SOURCE_AUDIO;
		retval [1] = BONFIRE_TRACK_SOURCE_INF;
		retval [2] = BONFIRE_TRACK_SOURCE_UNKNOWN;
		break;

	case BONFIRE_TRACK_SOURCE_GRAFTS:
	case BONFIRE_TRACK_SOURCE_DATA:
		retval = g_new0 (BonfireTrackSourceType, 3);
		retval [0] = BONFIRE_TRACK_SOURCE_ISO_JOLIET;
		retval [1] = BONFIRE_TRACK_SOURCE_ISO;
		retval [2] = BONFIRE_TRACK_SOURCE_UNKNOWN;
		break;

	case BONFIRE_TRACK_SOURCE_DISC:
		if (src_media_type <= NAUTILUS_BURN_MEDIA_TYPE_CDRW) {
			/* with CDs there are three possibilities:
			 * - if cdrdao is working => *.cue
			 * - readcd -clone => *.raw
			 * - simply readcd => *.iso 
			 */
			if (!caps->priv->cdrdao_disabled) {
				retval = g_new0 (BonfireTrackSourceType, 4);
				retval [0] = BONFIRE_TRACK_SOURCE_CUE;
				retval [1] = BONFIRE_TRACK_SOURCE_RAW;
				retval [2] = BONFIRE_TRACK_SOURCE_ISO;
				retval [3] = BONFIRE_TRACK_SOURCE_UNKNOWN;
			}
			else {
				retval = g_new0 (BonfireTrackSourceType, 3);
				retval [0] = BONFIRE_TRACK_SOURCE_RAW;
				retval [1] = BONFIRE_TRACK_SOURCE_ISO;
				retval [2] = BONFIRE_TRACK_SOURCE_UNKNOWN;
			}
		}
		else {
			/* with DVDs only one type of track is possible: *.iso */
			retval = g_new0 (BonfireTrackSourceType, 2);
			retval [0] = BONFIRE_TRACK_SOURCE_ISO;
			retval [1] = BONFIRE_TRACK_SOURCE_UNKNOWN;
		}
		break;

	default:
		return BONFIRE_BURN_NOT_SUPPORTED;
	}

	*targets = retval;
	return BONFIRE_BURN_OK;
}

/* returns what kind of BonfireTrackSourceType will be returned
 * by default for the given settings and system configuration */
BonfireTrackSourceType
bonfire_burn_caps_get_imager_default_target (BonfireBurnCaps *caps,
					     BonfireTrackSourceType source_type,
					     NautilusBurnMediaType src_media_type)
{
	switch (source_type) {
	case BONFIRE_TRACK_SOURCE_SONG:
		return BONFIRE_TRACK_SOURCE_AUDIO;

	case BONFIRE_TRACK_SOURCE_GRAFTS:
	case BONFIRE_TRACK_SOURCE_DATA:
		return BONFIRE_TRACK_SOURCE_ISO_JOLIET;

	case BONFIRE_TRACK_SOURCE_DISC:
		if (src_media_type <= NAUTILUS_BURN_MEDIA_TYPE_CDRW) {
			/* with CDs there are two possible default:
			 * - if cdrdao is working => copy on the fly
			 * - readcd -clone => image => cdrecord */
			if (!caps->priv->cdrdao_disabled)
				return BONFIRE_TRACK_SOURCE_CUE;
			else
				return BONFIRE_TRACK_SOURCE_RAW;
		}
		return BONFIRE_TRACK_SOURCE_ISO;

	default:
		break;
	}

	return BONFIRE_TRACK_SOURCE_UNKNOWN;
}

BonfireBurnResult
bonfire_burn_caps_create_imager (BonfireBurnCaps *caps,
				 BonfireImager **imager,
				 const BonfireTrackSource *source,
				 BonfireTrackSourceType target,
				 NautilusBurnMediaType src_media_type,
				 NautilusBurnMediaType dest_media_type,
				 GError **error)
{
	BonfireImager *obj = NULL;

	if (target == BONFIRE_TRACK_SOURCE_DEFAULT)
		target = bonfire_burn_caps_get_imager_default_target (caps,
								      source->type,
								      src_media_type);

	switch (source->type) {
	case BONFIRE_TRACK_SOURCE_SONG:
		/* works only with CDs */
		if (dest_media_type > NAUTILUS_BURN_MEDIA_TYPE_CDRW)
			return BONFIRE_BURN_NOT_SUPPORTED;

		/* we can only output one of these two types */
		if (target != BONFIRE_TRACK_SOURCE_INF
		&&  target != BONFIRE_TRACK_SOURCE_AUDIO)
			return BONFIRE_BURN_NOT_SUPPORTED;

		obj = BONFIRE_IMAGER (g_object_new (BONFIRE_TYPE_TRANSCODE, NULL));
		break;

	case BONFIRE_TRACK_SOURCE_GRAFTS:
		/* we can only output one of these two types */
		if (target != BONFIRE_TRACK_SOURCE_ISO_JOLIET
		&&  target != BONFIRE_TRACK_SOURCE_ISO)
			return BONFIRE_BURN_NOT_SUPPORTED;

		if (dest_media_type > NAUTILUS_BURN_MEDIA_TYPE_CDRW) 
			obj = BONFIRE_IMAGER (g_object_new (BONFIRE_TYPE_GROWISOFS, NULL));
		else
			obj = BONFIRE_IMAGER (g_object_new (BONFIRE_TYPE_MKISOFS, NULL));

		break;

	case BONFIRE_TRACK_SOURCE_DATA:
		/* we can only output one of these three types */
		if (target == BONFIRE_TRACK_SOURCE_GRAFTS)
			obj = BONFIRE_IMAGER (g_object_new (BONFIRE_TYPE_MKISOFS_BASE, NULL));
		else if (target == BONFIRE_TRACK_SOURCE_ISO_JOLIET
		      ||  target == BONFIRE_TRACK_SOURCE_ISO) {
			      if (dest_media_type > NAUTILUS_BURN_MEDIA_TYPE_CDRW) 
				      obj = BONFIRE_IMAGER (g_object_new (BONFIRE_TYPE_GROWISOFS, NULL));
			      else
				      obj = BONFIRE_IMAGER (g_object_new (BONFIRE_TYPE_MKISOFS, NULL));
		}
		else
			return BONFIRE_BURN_NOT_SUPPORTED;

		break;

	case BONFIRE_TRACK_SOURCE_DISC:
		if (src_media_type > NAUTILUS_BURN_MEDIA_TYPE_CDRW) {
			if (target != BONFIRE_TRACK_SOURCE_ISO)
				return BONFIRE_BURN_NOT_SUPPORTED;

			obj = BONFIRE_IMAGER (g_object_new (BONFIRE_TYPE_READCD, NULL));
		}
		else if (target == BONFIRE_TRACK_SOURCE_RAW) 
			obj = BONFIRE_IMAGER (g_object_new (BONFIRE_TYPE_READCD, NULL));
		else if (target == BONFIRE_TRACK_SOURCE_ISO)
			obj = BONFIRE_IMAGER (g_object_new (BONFIRE_TYPE_READCD, NULL));
		else if (target == BONFIRE_TRACK_SOURCE_CUE) {
			/* only cdrdao can create cues */
			obj = BONFIRE_IMAGER (g_object_new (BONFIRE_TYPE_CDRDAO, NULL));
		}
		else
			return BONFIRE_BURN_NOT_SUPPORTED;

		break;

	default:
		return BONFIRE_BURN_NOT_SUPPORTED;
	}

	*imager = obj;

	/* connect to the error signal to detect error and autoconfigure */
	g_signal_connect (obj,
			  "error",
			  G_CALLBACK (bonfire_burn_caps_job_error_cb),
			  caps);

	return BONFIRE_BURN_OK;
}

BonfireMediaType
bonfire_burn_caps_get_required_media_type (BonfireBurnCaps *caps,
					   BonfireTrackSourceType track_type)
{
	BonfireMediaType required_media;

	/* all the following type can only be burnt to a CD not a DVD */
	if (track_type == BONFIRE_TRACK_SOURCE_SONG
	||  track_type == BONFIRE_TRACK_SOURCE_AUDIO
	||  track_type == BONFIRE_TRACK_SOURCE_INF
	||  track_type == BONFIRE_TRACK_SOURCE_CUE
	||  track_type == BONFIRE_TRACK_SOURCE_RAW)
		required_media = BONFIRE_MEDIA_WRITABLE|
				 BONFIRE_MEDIA_TYPE_CD;
	else
		required_media = BONFIRE_MEDIA_WRITABLE;

	return required_media;
}
