/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
/***************************************************************************
 *            burn-common.c
 *
 *  Tue Feb 14 15:43:28 2006
 *  Copyright  2006  philippe
 *  <bonfire-app@wanadoo.fr>
 ****************************************************************************/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <errno.h>
#include <string.h>
#include <math.h>

#include <glib.h>
#include <glib/gi18n-lib.h>
#include <glib/gstdio.h>

#include <libgnomevfs/gnome-vfs.h>
#include <nautilus-burn-drive.h>

#include "burn-basics.h"
#include "burn-common.h"

long
bonfire_burn_common_compute_time_remaining (gint64 bytes, double bytes_per_sec)
{
	long secs;

	if (bytes_per_sec <= 1)
		return -1;

	secs = bytes / bytes_per_sec;

	return secs;	
}

gboolean
bonfire_burn_common_rm (const char *uri)
{
	GnomeVFSDirectoryHandle *handle;
	GnomeVFSFileInfo *info;
	char *file_uri, *name;

	/* NOTE : we don't follow uris as certain files could be linked */
	if (gnome_vfs_directory_open (&handle, uri, GNOME_VFS_FILE_INFO_DEFAULT) != GNOME_VFS_OK) {
		gnome_vfs_unlink (uri);
		return FALSE;
	}

	info = gnome_vfs_file_info_new ();
	while (gnome_vfs_directory_read_next (handle, info) == GNOME_VFS_OK) {
		if (*info->name == '.'
		&& (info->name[1] == 0
		|| (info->name[1] == '.' && info->name[2] == 0)))
			continue;

		name = gnome_vfs_escape_host_and_path_string (info->name);
		file_uri = g_strconcat (uri, "/", name, NULL);
		g_free (name);

		if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY)
			bonfire_burn_common_rm (file_uri);
		else if (gnome_vfs_unlink (file_uri) != GNOME_VFS_OK)
			g_warning ("Cannot remove file %s\n", file_uri);

		g_free (file_uri);
		gnome_vfs_file_info_clear (info);
	}
	gnome_vfs_file_info_unref (info);

	gnome_vfs_directory_close (handle);
	if (gnome_vfs_remove_directory (uri) != GNOME_VFS_OK) {
		g_warning ("Cannot remove directory %s\n", uri);
		return FALSE;
	}

	return TRUE;
}

BonfireBurnResult
bonfire_burn_common_check_output (char **output,
				  gboolean overwrite,
				  char **toc,
				  GError **error)
{
	if (!output)
		goto toc;

	/* takes care of the output file */
	if (!*output) {
		int fd;
		char *tmp;

		tmp = g_strdup_printf ("%s/bonfire_image_XXXXXX", g_get_tmp_dir ());
		fd = g_mkstemp (tmp);
		if (fd == -1) {
			g_free (tmp);
			g_set_error (error, 
				     BONFIRE_BURN_ERROR,
				     BONFIRE_BURN_ERROR_GENERAL,
				     _("a temporary file can't be created: %s"),
				     strerror (errno));
			return BONFIRE_BURN_ERR;
		}

		close (fd);
		g_remove (tmp);
		*output = tmp;
	}
	else if (g_file_test (*output, G_FILE_TEST_EXISTS)) {
		if (!overwrite) {
			g_set_error (error,
				     BONFIRE_BURN_ERROR,
				     BONFIRE_BURN_ERROR_GENERAL,
				     _("%s already exists"),
				     *output);
			return BONFIRE_BURN_ERR;
		}
		else if (!g_remove (*output)
		      &&  !bonfire_burn_common_rm (*output)) {
			g_set_error (error,
				     BONFIRE_BURN_ERROR,
				     BONFIRE_BURN_ERROR_GENERAL,
				     _("%s can't be removed"),
				     *output);
			return BONFIRE_BURN_ERR;
		}
	}

toc:
	if (!toc)
		return BONFIRE_BURN_OK;

	if ((*toc) == NULL)
		*toc = g_strdup_printf ("%s.toc", *output);

	if (g_file_test (*toc, G_FILE_TEST_EXISTS)) {
		if (!overwrite) {
			g_set_error (error,
				     BONFIRE_BURN_ERROR,
				     BONFIRE_BURN_ERROR_GENERAL,
				     _("%s already exists"),
				     *toc);
			return BONFIRE_BURN_ERR;
		}
		else if (!g_remove (*toc)
		      &&  !bonfire_burn_common_rm (*toc)) {
			g_set_error (error,
				     BONFIRE_BURN_ERROR,
				     BONFIRE_BURN_ERROR_GENERAL,
				     _("%s can't be removed"),
				     *toc);
			return BONFIRE_BURN_ERR;
		}			
	}

	return BONFIRE_BURN_OK;
}

gdouble
bonfire_burn_common_get_average_rate (GSList ** rates, gdouble rate)
{
	const unsigned int max_num = 16;
	const unsigned int scale = 10000;
	unsigned int num = 0;
	gdouble average;
	gint32 int_rate;
	GSList *l;

	if (rate * scale < G_MAXINT)
		int_rate = (gint32) ceil (scale * rate);
	else if (rate / scale < G_MAXINT)
		int_rate = (gint32) ceil (-1.0 * rate / scale);
	else
		return rate;
		
	*rates = g_slist_prepend (*rates, GINT_TO_POINTER (int_rate));

	average = 0;
	for (l = *rates; l; l = l->next) {
		gdouble r = (gdouble) GPOINTER_TO_INT (l->data);

		if (r < 0)
			r *= scale * -1.0;
		else
			r /= scale;

		average += r;
		num++;
		if (num == max_num && l->next)
			l = g_slist_delete_link (l, l->next);
	}

	average /= num;
	return average;
}

static gpointer
_eject_async (gpointer data)
{
	NautilusBurnDrive *drive = NAUTILUS_BURN_DRIVE (data);

	nautilus_burn_drive_eject (drive);
	nautilus_burn_drive_unref (drive);

	return NULL;
}

void
bonfire_burn_common_eject_async (NautilusBurnDrive *drive)
{
	GError *error = NULL;

	nautilus_burn_drive_ref (drive);
	g_thread_create (_eject_async, drive, FALSE, &error);
	if (error) {
		g_warning ("Could not create thread %s\n", error->message);
		g_error_free (error);

		nautilus_burn_drive_unref (drive);
		nautilus_burn_drive_eject (drive);
	}
}
