/***************************************************************************
 *            dvd-rw-format.c
 *
 *  sam fév  4 13:50:07 2006
 *  Copyright  2006  Rouquier Philippe
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>

#include <nautilus-burn-drive.h>

#include "burn-basics.h"
#include "burn-dvd-rw-format.h"
#include "burn-recorder.h"
#include "burn-process.h"

static void bonfire_dvd_rw_format_class_init (BonfireDvdRwFormatClass *klass);
static void bonfire_dvd_rw_format_init (BonfireDvdRwFormat *sp);
static void bonfire_dvd_rw_format_finalize (GObject *object);
static void bonfire_dvd_rw_format_iface_init (BonfireRecorderIFace *iface);

struct BonfireDvdRwFormatPrivate {
	NautilusBurnDrive *drive;

	int dummy:1;
	int blank_fast:1;
};

static BonfireBurnResult
bonfire_dvd_rw_format_blank (BonfireRecorder *recorder,
			     GError **error);
static BonfireBurnResult
bonfire_dvd_rw_format_set_drive (BonfireRecorder *recorder,
				 NautilusBurnDrive *drive,
				 GError **error);
static BonfireBurnResult
bonfire_dvd_rw_format_set_flags (BonfireRecorder *recorder,
				 BonfireRecorderFlag flags,
				 GError **error);

static BonfireBurnResult
bonfire_dvd_rw_format_set_argv (BonfireProcess *process,
				GPtrArray *argv,
				gboolean has_master,
				GError **error);
static BonfireBurnResult
bonfire_dvd_rw_format_read_stderr (BonfireProcess *process,
				   const char *line);

static GObjectClass *parent_class = NULL;

GType
bonfire_dvd_rw_format_get_type ()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireDvdRwFormatClass),
			NULL,
			NULL,
			(GClassInitFunc)bonfire_dvd_rw_format_class_init,
			NULL,
			NULL,
			sizeof (BonfireDvdRwFormat),
			0,
			(GInstanceInitFunc)bonfire_dvd_rw_format_init,
		};

		static const GInterfaceInfo recorder_info =
		{
			(GInterfaceInitFunc) bonfire_dvd_rw_format_iface_init,
			NULL,
			NULL
		};

		type = g_type_register_static(BONFIRE_TYPE_PROCESS,
					      "BonfireDvdRwFormat",
					      &our_info,
					      0);

		g_type_add_interface_static (type,
					     BONFIRE_TYPE_RECORDER,
					     &recorder_info);
	}

	return type;
}

static void
bonfire_dvd_rw_format_iface_init (BonfireRecorderIFace *iface)
{
	iface->set_drive = bonfire_dvd_rw_format_set_drive;
	iface->set_flags = bonfire_dvd_rw_format_set_flags;
	iface->blank = bonfire_dvd_rw_format_blank;
}

static void
bonfire_dvd_rw_format_class_init (BonfireDvdRwFormatClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	BonfireProcessClass *process_class = BONFIRE_PROCESS_CLASS (klass);

	parent_class = g_type_class_peek_parent(klass);
	object_class->finalize = bonfire_dvd_rw_format_finalize;

	process_class->set_argv = bonfire_dvd_rw_format_set_argv;
	process_class->stderr_func = bonfire_dvd_rw_format_read_stderr;
}

static void
bonfire_dvd_rw_format_init (BonfireDvdRwFormat *obj)
{
	obj->priv = g_new0 (BonfireDvdRwFormatPrivate, 1);
}

static void
bonfire_dvd_rw_format_finalize(GObject *object)
{
	BonfireDvdRwFormat *cobj;
	cobj = BONFIRE_DVD_RW_FORMAT (object);

	if (cobj->priv->drive) {
		nautilus_burn_drive_unref (cobj->priv->drive);
		cobj->priv->drive = NULL;
	}

	g_free(cobj->priv);
	G_OBJECT_CLASS(parent_class)->finalize(object);
}

BonfireDvdRwFormat *
bonfire_dvd_rw_format_new ()
{
	BonfireDvdRwFormat *obj;
	
	obj = BONFIRE_DVD_RW_FORMAT (g_object_new(BONFIRE_TYPE_DVD_RW_FORMAT, NULL));
	
	return obj;
}

static BonfireBurnResult
bonfire_dvd_rw_format_blank (BonfireRecorder *recorder,
			     GError **error)
{
	BonfireDvdRwFormat *dvdformat;
	NautilusBurnMediaType media;
	BonfireBurnResult result;

	dvdformat = BONFIRE_DVD_RW_FORMAT (recorder);
	media = nautilus_burn_drive_get_media_type (dvdformat->priv->drive);

	if (media <= NAUTILUS_BURN_MEDIA_TYPE_CDRW)
		return BONFIRE_BURN_NOT_SUPPORTED;

	/* There is no need to format RW+ in a fast way */
        if (media == NAUTILUS_BURN_MEDIA_TYPE_DVD_PLUS_RW && dvdformat->priv->blank_fast)
		return BONFIRE_BURN_OK;

	result = bonfire_job_run (BONFIRE_JOB (dvdformat), error);

	return result;
}

static BonfireBurnResult
bonfire_dvd_rw_format_set_drive (BonfireRecorder *recorder,
				 NautilusBurnDrive *drive,
				 GError **error)
{
	BonfireDvdRwFormat *dvdformat;

	dvdformat = BONFIRE_DVD_RW_FORMAT (recorder);

	if (dvdformat->priv->drive)
		nautilus_burn_drive_unref (dvdformat->priv->drive);

	nautilus_burn_drive_ref (drive);
	dvdformat->priv->drive = drive;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_dvd_rw_format_set_flags (BonfireRecorder *recorder,
				 BonfireRecorderFlag flags,
				 GError **error)
{
	BonfireDvdRwFormat *dvdformat;

	dvdformat = BONFIRE_DVD_RW_FORMAT (recorder);

	/* apparently there is no switch for BONFIRE_RECORDER_BLANK_FLAG_NOGRACE */
	dvdformat->priv->blank_fast = (flags & BONFIRE_RECORDER_FLAG_FAST_BLANK);
	dvdformat->priv->dummy = (flags & BONFIRE_RECORDER_FLAG_DUMMY);

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_dvd_rw_format_read_stderr (BonfireProcess *process, const char *line)
{
	BonfireDvdRwFormat *dvdformat;
	float percent;

	dvdformat = BONFIRE_DVD_RW_FORMAT (process);

	if ((sscanf (line, "* blanking %f%%,", &percent) == 1)
	||  (sscanf (line, "* formatting %f%%,", &percent) == 1)
	||  (sscanf (line, "* relocating lead-out %f%%,", &percent) == 1)) {

		bonfire_job_set_dangerous (BONFIRE_JOB (process), TRUE);

		if (percent > 1) {
			bonfire_job_action_changed (BONFIRE_JOB (dvdformat),
						    BONFIRE_BURN_ACTION_BLANKING,
						    FALSE);
			bonfire_job_progress_changed (BONFIRE_JOB (dvdformat),
						      percent,
						      -1);
		}

	}

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_dvd_rw_format_set_argv (BonfireProcess *process,
				GPtrArray *argv,
				gboolean has_master,
				GError **error)
{
	BonfireDvdRwFormat *dvdformat;
	NautilusBurnMediaType media;
	char *dev_str;

	if (has_master)
		return BONFIRE_BURN_NOT_SUPPORTED;

	dvdformat = BONFIRE_DVD_RW_FORMAT (process);
	bonfire_job_set_run_slave (BONFIRE_JOB (dvdformat), FALSE);

	g_ptr_array_add (argv, g_strdup ("dvd+rw-format"));

	/* undocumented option to show progress */
	g_ptr_array_add (argv, g_strdup ("-gui"));

	media = nautilus_burn_drive_get_media_type (dvdformat->priv->drive);
        if (media != NAUTILUS_BURN_MEDIA_TYPE_DVD_PLUS_RW) {
		char *blank_str;

		blank_str = g_strdup_printf ("-blank%s",
					     dvdformat->priv->blank_fast ? "" : "=full");
		g_ptr_array_add (argv, blank_str);
	}

	/* it seems that dvd-format prefers the device path not cdrecord_id */
	dev_str = g_strdup_printf ("%s", dvdformat->priv->drive->device);
	g_ptr_array_add (argv, dev_str);

	return BONFIRE_BURN_OK;
}

