/***************************************************************************
 *            growisofs.c
 *
 *  dim jan  15:8:51 6
 *  Copyright  6  Rouquier Philippe
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version  of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite , Boston, MA 111-17, USA.
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>

#include "burn-basics.h"
#include "burn-common.h"
#include "burn-caps.h"
#include "burn-mkisofs-base.h"
#include "burn-growisofs.h"
#include "burn-process.h"
#include "burn-recorder.h"
#include "burn-imager.h"

static void bonfire_growisofs_class_init (BonfireGrowisofsClass *klass);
static void bonfire_growisofs_init (BonfireGrowisofs *sp);
static void bonfire_growisofs_finalize (GObject *object);
static void bonfire_growisofs_iface_init_image (BonfireImagerIFace *iface);
static void bonfire_growisofs_iface_init_record (BonfireRecorderIFace *iface);

/* Imaging part */
static BonfireBurnResult
bonfire_growisofs_set_source (BonfireJob *job,
			      const BonfireTrackSource *source,
			      GError **error);
static BonfireBurnResult
bonfire_growisofs_set_output_type (BonfireImager *imager,
				   BonfireTrackSourceType type,
				   GError **error);
static BonfireBurnResult
bonfire_growisofs_set_append (BonfireImager *imager,
			      NautilusBurnDrive *drive,
			      gboolean merge,
			      GError **error);
static BonfireBurnResult
bonfire_growisofs_get_size (BonfireImager *imager,
			    gint64 *size,
			    gboolean sectors,
			    GError **error);

/* Process functions */
static BonfireBurnResult
bonfire_growisofs_read_stdout (BonfireProcess *process, 
			       const char *line);
static BonfireBurnResult
bonfire_growisofs_read_stderr (BonfireProcess *process,
			       const char *line);
static BonfireBurnResult
bonfire_growisofs_set_argv (BonfireProcess *process,
			    GPtrArray *array,
			    gboolean has_master,
			    GError **error);
static BonfireBurnResult
bonfire_growisofs_post (BonfireProcess *process,
			BonfireBurnResult retval);
			
/* Recording part */
static BonfireBurnResult
bonfire_growisofs_set_drive (BonfireRecorder *recorder,
			     NautilusBurnDrive *drive,
			     GError **error);
static BonfireBurnResult
bonfire_growisofs_set_flags (BonfireRecorder *recorder,
			     BonfireRecorderFlag flags,
			     GError **error);
static BonfireBurnResult
bonfire_growisofs_set_speed (BonfireJob *job,
			     gint64 speed);

static BonfireBurnResult
bonfire_growisofs_record (BonfireRecorder *recorder,
			  GError **error);
static BonfireBurnResult
bonfire_growisofs_blank (BonfireRecorder *recorder,
			 GError **error);

static BonfireBurnResult
bonfire_growisofs_get_status (BonfireRecorder *recorder,
			      int *speed,
			      int *fifo,
			      gint64 *bytes_written);
static BonfireBurnResult
bonfire_growisofs_get_action_string (BonfireJob *job,
				     BonfireBurnAction action,
				     char **string);

typedef enum {
	BONFIRE_GROWISOFS_ACTION_NONE,
	BONFIRE_GROWISOFS_ACTION_RECORD,
	BONFIRE_GROWISOFS_ACTION_BLANK,
	BONFIRE_GROWISOFS_ACTION_GET_SIZE
} BonfireGrowisofsAction;

struct BonfireGrowisofsPrivate {
	BonfireGrowisofsAction action;

	NautilusBurnDrive *drive;
	int speed;

	int cur_speed;
	int fifo;
	gint64 bytes_written;
	GTimer *timer;

	BonfireTrackSource *source;

	int sectors_num;

	int use_joliet:1;
	int fast_blank:1;
	int use_utf8:1;
	int append:1;
	int merge:1;
	int dummy:1;
	int multi:1;
	int dao:1;
};

static GObjectClass *parent_class = NULL;

GType
bonfire_growisofs_get_type()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireGrowisofsClass),
			NULL,
			NULL,
			(GClassInitFunc)bonfire_growisofs_class_init,
			NULL,
			NULL,
			sizeof (BonfireGrowisofs),
			0,
			(GInstanceInitFunc)bonfire_growisofs_init,
		};
		static const GInterfaceInfo imager_info =
		{
			(GInterfaceInitFunc) bonfire_growisofs_iface_init_image,
			NULL,
			NULL
		};
		static const GInterfaceInfo recorder_info =
		{
			(GInterfaceInitFunc) bonfire_growisofs_iface_init_record,
			NULL,
			NULL
		};

		type = g_type_register_static (BONFIRE_TYPE_PROCESS, 
					       "BonfireGrowisofs",
					       &our_info,
					       0);
		g_type_add_interface_static (type,
					     BONFIRE_TYPE_IMAGER,
					     &imager_info);
		g_type_add_interface_static (type,
					     BONFIRE_TYPE_RECORDER,
					     &recorder_info);
	}

	return type;
}

static void
bonfire_growisofs_class_init (BonfireGrowisofsClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	BonfireJobClass *job_class = BONFIRE_JOB_CLASS (klass);
	BonfireProcessClass *process_class = BONFIRE_PROCESS_CLASS (klass);

	parent_class = g_type_class_peek_parent(klass);
	object_class->finalize = bonfire_growisofs_finalize;

	job_class->get_action_string = bonfire_growisofs_get_action_string;
	job_class->set_source = bonfire_growisofs_set_source;
	job_class->set_rate = bonfire_growisofs_set_speed;

	process_class->stdout_func = bonfire_growisofs_read_stdout;
	process_class->stderr_func = bonfire_growisofs_read_stderr;
	process_class->set_argv = bonfire_growisofs_set_argv;
	process_class->post = bonfire_growisofs_post;
}

static void
bonfire_growisofs_iface_init_image (BonfireImagerIFace *iface)
{
	iface->set_output_type = bonfire_growisofs_set_output_type;
	iface->set_append = bonfire_growisofs_set_append;
	iface->get_size = bonfire_growisofs_get_size;
}

static void
bonfire_growisofs_iface_init_record (BonfireRecorderIFace *iface)
{
	iface->get_status = bonfire_growisofs_get_status;
	iface->set_drive = bonfire_growisofs_set_drive;
	iface->set_flags = bonfire_growisofs_set_flags;
	iface->record = bonfire_growisofs_record;
	iface->blank = bonfire_growisofs_blank;
}

static void
bonfire_growisofs_init(BonfireGrowisofs *obj)
{
	char *standard_error;
	gboolean res;

	obj->priv = g_new0 (BonfireGrowisofsPrivate, 1);

	/* this code comes from ncb_mkisofs_supports_utf8 */
	res = g_spawn_command_line_sync ("mkisofs -input-charset utf8", 
					 NULL,
					 &standard_error,
					 NULL, 
					 NULL);
	if (res && !g_strrstr (standard_error, "Unknown charset"))
		obj->priv->use_utf8 = TRUE;
	else
		obj->priv->use_utf8 = FALSE;

	g_free (standard_error);
}

static void
bonfire_growisofs_finalize (GObject *object)
{
	BonfireGrowisofs *cobj;
	cobj = BONFIRE_GROWISOFS(object);

	if (cobj->priv->source) {
		bonfire_track_source_free (cobj->priv->source);
		cobj->priv->source = NULL;
	}

	if (cobj->priv->drive) {
		nautilus_burn_drive_unref (cobj->priv->drive);
		cobj->priv->drive = NULL;
	}

	g_free (cobj->priv);
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

BonfireGrowisofs *
bonfire_growisofs_new ()
{
	BonfireGrowisofs *obj;
	
	obj = BONFIRE_GROWISOFS (g_object_new (BONFIRE_TYPE_GROWISOFS, NULL));
	
	return obj;
}

static BonfireBurnResult
bonfire_growisofs_set_output_type (BonfireImager *imager,
				   BonfireTrackSourceType type,
				   GError **error)
{
	BonfireGrowisofs *growisofs;

	growisofs = BONFIRE_GROWISOFS (imager);

	if (type == BONFIRE_TRACK_SOURCE_DEFAULT)
		type = BONFIRE_TRACK_SOURCE_ISO_JOLIET;
	else if (type != BONFIRE_TRACK_SOURCE_ISO
	      &&  type != BONFIRE_TRACK_SOURCE_ISO_JOLIET)
		return BONFIRE_BURN_NOT_SUPPORTED;

	if (type == BONFIRE_TRACK_SOURCE_ISO_JOLIET)
		growisofs->priv->use_joliet = 1;
	else 
		growisofs->priv->use_joliet = 0;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_growisofs_get_track (BonfireGrowisofs *growisofs,
			     const BonfireTrackSource *source,
			     GError **error)
{
	BonfireBurnResult result;
	BonfireBurnCaps *caps;
	BonfireImager *imager;

	/* we need to download all non local files first
	* and make a list of excluded and graft points */

	/* ask BurnCaps to create an object to get GRAFTS */
	caps = bonfire_burn_caps_get_default ();
	result = bonfire_burn_caps_create_imager (caps,
						  &imager,
						  source,
						  BONFIRE_TRACK_SOURCE_GRAFTS,
						  NAUTILUS_BURN_MEDIA_TYPE_UNKNOWN,
						  NAUTILUS_BURN_MEDIA_TYPE_UNKNOWN,
						  error);
	g_object_unref (caps);

	/* that way the slave will be unref at the same
	 * time as us or if we set another slave */
	bonfire_job_set_slave (BONFIRE_JOB (growisofs), BONFIRE_JOB (imager));
	g_object_unref (imager);

	result = bonfire_job_set_source (BONFIRE_JOB (imager), source, error);
	if (result != BONFIRE_BURN_OK)
		return result;

	result = bonfire_imager_set_output (imager,
					    NULL,
					    FALSE,
					    TRUE,
					    error);
	if (result != BONFIRE_BURN_OK)
		return result;

	result = bonfire_imager_set_output_type (imager,
						 BONFIRE_TRACK_SOURCE_GRAFTS,
						 error);
	if (result != BONFIRE_BURN_OK)
		return result;

	bonfire_job_set_relay_slave_signals (BONFIRE_JOB (growisofs), TRUE);
	result = bonfire_imager_get_track (imager,
					   &growisofs->priv->source,
					   error);
	bonfire_job_set_relay_slave_signals (BONFIRE_JOB (growisofs), FALSE);
	return result;
}

static BonfireBurnResult
bonfire_growisofs_set_source (BonfireJob *job,
			      const BonfireTrackSource *source,
			      GError **error)
{
	BonfireGrowisofs *growisofs;
	BonfireBurnResult result = BONFIRE_BURN_OK;

	growisofs = BONFIRE_GROWISOFS (job);

	/* we accept ourselves as source and in this case we don't change 
	 * anything: growisofs is both imager and recorder. In this case
	 * we don't delete the previous source */
	if (source->type == BONFIRE_TRACK_SOURCE_IMAGER
	&&  source->contents.imager.obj == BONFIRE_IMAGER (growisofs))
		return BONFIRE_BURN_OK;

	if (growisofs->priv->source) {
		bonfire_track_source_free (growisofs->priv->source);
		growisofs->priv->source = NULL;
	}
	growisofs->priv->sectors_num = 0;

	if (source->type != BONFIRE_TRACK_SOURCE_DATA
	&&  source->type != BONFIRE_TRACK_SOURCE_GRAFTS
	&&  source->type != BONFIRE_TRACK_SOURCE_ISO
	&&  source->type != BONFIRE_TRACK_SOURCE_ISO_JOLIET
	&&  source->type != BONFIRE_TRACK_SOURCE_IMAGER)
		return BONFIRE_BURN_NOT_SUPPORTED;

	if (source->type == BONFIRE_TRACK_SOURCE_DATA)
		result = bonfire_growisofs_get_track (growisofs, source, error);
	else
		growisofs->priv->source = bonfire_track_source_copy (source);

	return result;
}

static BonfireBurnResult
bonfire_growisofs_set_append (BonfireImager *imager,
			      NautilusBurnDrive *drive,
			      gboolean merge,
			      GError **error)
{
	BonfireGrowisofs *growisofs;

	growisofs = BONFIRE_GROWISOFS (imager);

	if (drive) {
		if (growisofs->priv->drive) {
			nautilus_burn_drive_unref (growisofs->priv->drive);
			growisofs->priv->drive = NULL;
		}
	
		nautilus_burn_drive_ref (drive);
		growisofs->priv->drive = drive;
	}

	/* growisofs doesn't give the choice it merges */
	growisofs->priv->append = 1;
	growisofs->priv->merge = 1;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_growisofs_get_size (BonfireImager *imager,
			    gint64 *size,
			    gboolean sectors,
			    GError **error)
{
	BonfireBurnResult result = BONFIRE_BURN_OK;
	BonfireGrowisofs *growisofs;

	growisofs = BONFIRE_GROWISOFS (imager);

	if (!growisofs->priv->source)
		return BONFIRE_BURN_NOT_READY;

	if (growisofs->priv->source->type != BONFIRE_TRACK_SOURCE_GRAFTS)
		return BONFIRE_BURN_NOT_SUPPORTED;

	if (!growisofs->priv->sectors_num) {
		if (bonfire_job_is_running (BONFIRE_JOB (imager)))
			return BONFIRE_BURN_RUNNING;

		growisofs->priv->action = BONFIRE_GROWISOFS_ACTION_GET_SIZE;
		result = bonfire_job_run (BONFIRE_JOB (growisofs), error);
		growisofs->priv->action = BONFIRE_GROWISOFS_ACTION_NONE;

		if (result != BONFIRE_BURN_OK)
			return result;
	}

	/* NOTE: the size in bytes doesn't mean anything since growisofs doesn't
	 * write to the disc the size in sectors is more relevant to check if it
	 * will fit on the disc */
	if (sectors)
		*size = growisofs->priv->sectors_num;
	else 
		*size = growisofs->priv->sectors_num * 2048;

	return result;
}

/* Recording part */
static BonfireBurnResult
bonfire_growisofs_record (BonfireRecorder *recorder,
			  GError **error)
{
	BonfireGrowisofs *growisofs;
	BonfireBurnResult result;

	growisofs = BONFIRE_GROWISOFS (recorder);

	if (!growisofs->priv->drive)
		return BONFIRE_BURN_NOT_READY;

	if (!growisofs->priv->source)
		return BONFIRE_BURN_NOT_READY;

	/* set as slave if track is an imager (on the fly burning) */
	if (growisofs->priv->source->type == BONFIRE_TRACK_SOURCE_IMAGER) {
		BonfireJob *slave;

		slave = BONFIRE_JOB (growisofs->priv->source->contents.imager.obj);
		bonfire_job_set_slave (BONFIRE_JOB (growisofs), slave);
		bonfire_job_set_relay_slave_signals (BONFIRE_JOB (growisofs), FALSE);
		bonfire_job_set_run_slave (BONFIRE_JOB (growisofs), TRUE);
	}
	else
		bonfire_job_set_run_slave (BONFIRE_JOB (growisofs), FALSE);

	growisofs->priv->action = BONFIRE_GROWISOFS_ACTION_RECORD;
	result = bonfire_job_run (BONFIRE_JOB (growisofs), error);
	growisofs->priv->action = BONFIRE_GROWISOFS_ACTION_NONE;

	return result;
}

static BonfireBurnResult
bonfire_growisofs_blank (BonfireRecorder *recorder,
			 GError **error)
{
	BonfireBurnResult result;
	BonfireGrowisofs *growisofs;
	NautilusBurnMediaType media;

	growisofs = BONFIRE_GROWISOFS (recorder);

	if (!growisofs->priv->drive)
		return BONFIRE_BURN_NOT_READY;

	media = nautilus_burn_drive_get_media_type (growisofs->priv->drive);
	if (media <= NAUTILUS_BURN_MEDIA_TYPE_CDRW);
		return BONFIRE_BURN_NOT_SUPPORTED;

	/* There is no need to format RW+ in a fast way */
        if (media == NAUTILUS_BURN_MEDIA_TYPE_DVD_PLUS_RW
	&&  growisofs->priv->fast_blank)
		return BONFIRE_BURN_OK;

	/* if we have a slave we don't want it to run */
	bonfire_job_set_run_slave (BONFIRE_JOB (recorder), FALSE);

	growisofs->priv->action = BONFIRE_GROWISOFS_ACTION_BLANK;
	result = bonfire_job_run (BONFIRE_JOB (recorder), error);
	growisofs->priv->action = BONFIRE_GROWISOFS_ACTION_NONE;

	return result;
}

static BonfireBurnResult
bonfire_growisofs_set_drive (BonfireRecorder *recorder,
			     NautilusBurnDrive *drive,
			     GError **error)
{
	BonfireGrowisofs *growisofs;

	growisofs = BONFIRE_GROWISOFS (recorder);

	if (growisofs->priv->drive) {
		nautilus_burn_drive_unref (growisofs->priv->drive);
		growisofs->priv->drive = NULL;
	}
	
	nautilus_burn_drive_ref (drive);
	growisofs->priv->drive = drive;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_growisofs_set_flags (BonfireRecorder *recorder,
			     BonfireRecorderFlag flags,
			     GError **error)
{
	BonfireGrowisofs *growisofs;

	growisofs = BONFIRE_GROWISOFS (recorder);

	growisofs->priv->dao = (flags & BONFIRE_RECORDER_FLAG_DAO) != 0;
	growisofs->priv->fast_blank = (flags & BONFIRE_RECORDER_FLAG_FAST_BLANK) != 0;
	growisofs->priv->dummy = (flags & BONFIRE_RECORDER_FLAG_DUMMY) != 0;
	growisofs->priv->multi = (flags & BONFIRE_RECORDER_FLAG_MULTI) != 0;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_growisofs_set_speed (BonfireJob *job,
			     gint64 speed)
{
	BonfireGrowisofs *growisofs;

	if (bonfire_job_is_running (job))
		return BONFIRE_BURN_RUNNING;

	growisofs = BONFIRE_GROWISOFS (job);
	growisofs->priv->speed = speed / DVD_SPEED;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_growisofs_get_status (BonfireRecorder *recorder,
			      int *speed,
			      int *fifo,
			      gint64 *bytes_written)
{
	BonfireGrowisofs *growisofs;

	growisofs = BONFIRE_GROWISOFS (recorder);

	if (growisofs->priv->action != BONFIRE_GROWISOFS_ACTION_RECORD)
		return BONFIRE_BURN_NOT_RUNNING;

	if (speed)
		*speed = growisofs->priv->cur_speed;

	if (fifo)
		*fifo = growisofs->priv->fifo;

	if (bytes_written)
		*bytes_written = growisofs->priv->bytes_written;

	return BONFIRE_BURN_OK;
}

/* Process start */
static BonfireBurnResult
bonfire_growisofs_read_stdout (BonfireProcess *process, const char *line)
{
	int perc_1, perc_2;
	int speed_1, speed_2;
	long long b_written, b_total;
	BonfireGrowisofs *growisofs;

	growisofs = BONFIRE_GROWISOFS (process);

	if (sscanf (line, "%10lld/%lld ( %2d.%1d%%) @%2d.%1dx, remaining %*d:%*d",
		    &b_written, &b_total, &perc_1, &perc_2, &speed_1, &speed_2) == 6) {
		double percent;
		double speed;
		long   secs;

		speed = (speed_1 + ((float) speed_2 / 10));
		growisofs->priv->cur_speed = (int) ceil (speed);
		growisofs->priv->bytes_written = b_written;

		bonfire_job_action_changed (BONFIRE_JOB (process),
					    BONFIRE_BURN_ACTION_WRITING,
					    FALSE);

		percent = (perc_1 + ((float) perc_2 / 10)) / 100;
		secs = bonfire_burn_common_compute_time_remaining (b_total - b_written, (double) speed * DVD_SPEED);

		bonfire_job_progress_changed (BONFIRE_JOB (process), 
					      percent * 0.98,
					      secs);
	}
	else if (strstr (line, "About to execute") || strstr (line, "Executing")) {
		if (growisofs->priv->action != BONFIRE_GROWISOFS_ACTION_GET_SIZE) {
			bonfire_job_action_changed (BONFIRE_JOB (process),
						    BONFIRE_BURN_ACTION_PREPARING,
						    FALSE);
			bonfire_job_set_dangerous (BONFIRE_JOB (process), TRUE);
		}
	}

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_growisofs_read_stderr (BonfireProcess *process, const char *line)
{
	int perc_1, perc_2;
	BonfireGrowisofs *growisofs;

	growisofs = BONFIRE_GROWISOFS (process);

	if (sscanf (line, " %2d.%1d%% done, estimate finish", &perc_1, &perc_2) == 2) {
		long secs = -1;
		gdouble percent;

		percent = (perc_1 + ((float) perc_2 / 10)) / 100;

		if (growisofs->priv->sectors_num) {
			gulong micro;
			gint64 b_total;
			gdouble elapsed;
			gint64 b_written;
			gdouble raw_speed;

			b_total = growisofs->priv->sectors_num * 2048;
			b_written = b_total * percent;

			elapsed = g_timer_elapsed (growisofs->priv->timer, &micro);
			elapsed += (gdouble) micro / 1000000;
			raw_speed = (gdouble) b_written / elapsed;

			growisofs->priv->bytes_written = b_written;
			growisofs->priv->cur_speed = (int) ceil (raw_speed / (gdouble) DVD_SPEED);
			secs = bonfire_burn_common_compute_time_remaining (b_total - b_written, raw_speed);
		}

		bonfire_job_action_changed (BONFIRE_JOB (process),
					    BONFIRE_BURN_ACTION_WRITING,
					    FALSE);
		bonfire_job_progress_changed (BONFIRE_JOB (process), 
					      percent * 0.98,
					      secs);
	}
	else if (strstr (line, "restarting DVD+RW format")) {
		bonfire_job_action_changed (BONFIRE_JOB (process),
					    BONFIRE_BURN_ACTION_BLANKING,
					    FALSE);
	}
	else if (strstr (line, "\"Current Write Speed\" is ")) {
		bonfire_job_action_changed (BONFIRE_JOB (process),
					    BONFIRE_BURN_ACTION_PREPARING,
					    FALSE);
	}
	else if (strstr (line, "Writing:   The File(s)")) {
		growisofs->priv->timer = g_timer_new ();
	}
	else if (strstr (line, "Total extents scheduled to be written = ")) {
		BonfireGrowisofs *growisofs;

		growisofs = BONFIRE_GROWISOFS (process);

		line += strlen ("Total extents scheduled to be written = ");
		growisofs->priv->sectors_num = atoi (line);
	}
	else if (strstr (line, "unsupported MMC profile")
	      || (strstr (line, "already carries isofs") && strstr (line, "FATAL:"))) {
		/* This is not supposed to happen since we checked for the cd
		   type before starting, but we try to handle it anyway, since mmc
		   profiling can fail. */
		bonfire_job_error (BONFIRE_JOB (process), 
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_MEDIA_NOT_WRITABLE,
						_("The disc is already burnt")));
	}
	else if (strstr (line, "unable to open") || strstr (line, "unable to stat")) {
		/* This fits the "open6" and "open"-like messages */
		bonfire_job_error (BONFIRE_JOB (process), 
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_BUSY_DRIVE,
						_("The recorder could not be accessed")));
	}
	else if (strstr (line, "not enough space available") != NULL) {
		bonfire_job_error (BONFIRE_JOB (process), 
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_GENERAL,
						_("Not enough space available on the disc")));
	}
	else if (strstr (line, "end of user area encountered on this track") != NULL) {
		bonfire_job_error (BONFIRE_JOB (process), 
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_GENERAL,
						_("The files selected did not fit on the CD")));
	}
	else if (strstr (line, "blocks are free") != NULL) {
		bonfire_job_error (BONFIRE_JOB (process), 
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_GENERAL,
						_("The files selected did not fit on the CD")));
	}
	else if (strstr (line, "flushing cache") != NULL) {
		bonfire_job_progress_changed (BONFIRE_JOB (process), 
					      0.98,
					      -1);
		bonfire_job_action_changed (BONFIRE_JOB (process),
					    BONFIRE_BURN_ACTION_FIXATING,
					    FALSE);
		bonfire_job_set_dangerous (BONFIRE_JOB (process), FALSE);
	}
	else if (strstr (line, ":-(") != NULL || strstr (line, "FATAL") != NULL) {
		bonfire_job_error (BONFIRE_JOB (process), 
				   g_error_new (BONFIRE_BURN_ERROR,
						BONFIRE_BURN_ERROR_GENERAL,
						_("Unhandled error, aborting")));
	}
	else if (strstr (line, "Incorrectly encoded string")) {
		bonfire_job_error (BONFIRE_JOB (process),
				   g_error_new_literal (BONFIRE_BURN_ERROR,
							BONFIRE_BURN_ERROR_JOLIET_TREE,
							_("Some files have invalid filenames")));
	}
	else if (strstr (line, "Joliet tree sort failed.")) {
		bonfire_job_error (BONFIRE_JOB (process), 
				   g_error_new_literal (BONFIRE_BURN_ERROR,
							BONFIRE_BURN_ERROR_JOLIET_TREE,
							_("the image can't be created")));
	}

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_growisofs_set_mkisofs_argv (BonfireGrowisofs *growisofs,
				    GPtrArray *argv,
				    GError **error)
{
	g_ptr_array_add (argv, g_strdup ("-r"));

	if (growisofs->priv->use_joliet)
		g_ptr_array_add (argv, g_strdup ("-J"));

	if (growisofs->priv->use_utf8) {
		g_ptr_array_add (argv, g_strdup ("-input-charset"));
		g_ptr_array_add (argv, g_strdup ("utf8"));
	}

	g_ptr_array_add (argv, g_strdup ("-graft-points"));
	g_ptr_array_add (argv, g_strdup ("-D"));	// This is dangerous the manual says but apparently it works well

	g_ptr_array_add (argv, g_strdup ("-path-list"));
	g_ptr_array_add (argv, g_strdup (growisofs->priv->source->contents.grafts.grafts_path));

	if (growisofs->priv->source->contents.grafts.excluded_path) {
		g_ptr_array_add (argv, g_strdup ("-exclude-list"));
		g_ptr_array_add (argv, g_strdup (growisofs->priv->source->contents.grafts.excluded_path));
	}

	if (growisofs->priv->action != BONFIRE_GROWISOFS_ACTION_GET_SIZE) {
		if (growisofs->priv->source->contents.grafts.label) {
			g_ptr_array_add (argv, g_strdup ("-V"));
			g_ptr_array_add (argv, g_strdup (growisofs->priv->source->contents.grafts.label));
		}

		g_ptr_array_add (argv, g_strdup ("-A"));
		g_ptr_array_add (argv, g_strdup_printf ("Bonfire-%i.%i.%i",
							BONFIRE_MAJOR_VERSION,
							BONFIRE_MINOR_VERSION,
							BONFIRE_SUB));
	
		g_ptr_array_add (argv, g_strdup ("-sysid"));
		g_ptr_array_add (argv, g_strdup ("LINUX"));
	
		/* FIXME! -sort is an interesting option allowing to decide where the 
		 * files are written on the disc and therefore to optimize later reading */
		/* FIXME: -hidden --hidden-list -hide-jolie -hide-joliet-list will allow to hide
		 * some files when we will display the contents of a disc we will want to merge */
		/* FIXME: support preparer publisher options */

		g_ptr_array_add (argv, g_strdup ("-v"));
	}
	else {
		/* we don't specify -q as there wouldn't be anything */
		g_ptr_array_add (argv, g_strdup ("-print-size"));
	}

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_growisofs_set_argv_record (BonfireGrowisofs *growisofs,
				   GPtrArray *argv,
				   GError **error)
{
	BonfireBurnResult result;

	if (!growisofs->priv->drive)
		return BONFIRE_BURN_NOT_READY;

	if (!growisofs->priv->source)
		return BONFIRE_BURN_NOT_READY;

	/* This seems to help to eject tray after burning (at least with mine) */
	g_ptr_array_add (argv, g_strdup ("-use-the-force-luke=notray"));

	if (growisofs->priv->dummy)
		g_ptr_array_add (argv, g_strdup ("-use-the-force-luke=dummy"));

	/* NOTE: dao is supported for DL DVD after 6.0 (think about that for BurnCaps) */
	if (growisofs->priv->dao)
		g_ptr_array_add (argv, g_strdup ("-use-the-force-luke=dao"));

	if (growisofs->priv->speed > 0)
		g_ptr_array_add (argv, g_strdup_printf ("-speed=%d", growisofs->priv->speed));

	/* dvd-compat closes the discs and could cause
	 * problems if multi session is required */
	if (!growisofs->priv->multi)
		g_ptr_array_add (argv, g_strdup ("-dvd-compat"));

	/* see if we're asked to merge some new data: in this case we MUST have
	 * a list of grafts. The image can't come through stdin or an already 
	 * made image */
	if (growisofs->priv->merge) {
		if (growisofs->priv->source->type != BONFIRE_TRACK_SOURCE_GRAFTS) {
			g_set_error (error, 
				     BONFIRE_BURN_ERROR,
				     BONFIRE_BURN_ERROR_GENERAL,
				     _("only graft points can be appended in multisession mode"));
			return BONFIRE_BURN_NOT_SUPPORTED;
		}

		g_ptr_array_add (argv, g_strdup ("-M"));
		if (growisofs->priv->drive->cdrecord_id) 
			g_ptr_array_add (argv, g_strdup (growisofs->priv->drive->cdrecord_id));
		else if (growisofs->priv->drive->device)
			g_ptr_array_add (argv, g_strdup (growisofs->priv->drive->device));
		else
			return BONFIRE_BURN_ERR;
		
		/* this can only happen if source->type == BONFIRE_TRACK_SOURCE_GRAFTS */
		if (growisofs->priv->action == BONFIRE_GROWISOFS_ACTION_GET_SIZE) {
			g_ptr_array_add (argv, g_strdup ("-dry-run"));
			bonfire_job_action_changed (BONFIRE_JOB (growisofs),
						    BONFIRE_BURN_ACTION_GETTING_SIZE,
						    FALSE);
		}

		result = bonfire_growisofs_set_mkisofs_argv (growisofs, 
							     argv,
							     error);
		if (result != BONFIRE_BURN_OK)
			return result;

		bonfire_job_set_run_slave (BONFIRE_JOB (growisofs), FALSE);
	}
	else {
		/* Weird, innit? We tell growisofs we have a tty so it ignores
		 * the fact that the DVD+ has an ISO fs already */
		if (!growisofs->priv->multi)
			g_ptr_array_add (argv, g_strdup ("-use-the-force-luke=tty"));

		if (growisofs->priv->source->type == BONFIRE_TRACK_SOURCE_IMAGER) {
			BonfireImager *imager;
			gint64 sectors;

			imager = growisofs->priv->source->contents.imager.obj;

			/* in this case this must be an iso : check that */
			result = bonfire_imager_set_output_type (imager,
								 BONFIRE_TRACK_SOURCE_ISO,
								 error);

			if (result == BONFIRE_BURN_NOT_SUPPORTED) {
				if (!error)
					g_set_error (error,
						     BONFIRE_BURN_ERROR,
						     BONFIRE_BURN_ERROR_GENERAL,
						     _("imager can't create iso9660 images"));
				return BONFIRE_BURN_ERR;
			}

			if (result != BONFIRE_BURN_OK) {
				if (!error)
					g_set_error (error,
						     BONFIRE_BURN_ERROR,
						     BONFIRE_BURN_ERROR_GENERAL,
						     _("imager doesn't seem to be ready"));
				return BONFIRE_BURN_ERR;
			}

			/* ask the size */
			result = bonfire_imager_get_size (imager, &sectors, TRUE, error);
			if (result != BONFIRE_BURN_OK) {
				if (!error)
					g_set_error (error,
						     BONFIRE_BURN_ERROR,
						     BONFIRE_BURN_ERROR_GENERAL,
						     _("imager doesn't seem to be ready"));
				return BONFIRE_BURN_ERR;
			}

			/* set the buffer. NOTE: apparently this needs to be a power of 2 */
			/* FIXME: is it right to mess with it ? 
			   g_ptr_array_add (argv, g_strdup_printf ("-use-the-force-luke=bufsize:%im", 32)); */

			/* NOTE: tracksize is in block number (2048 bytes) */
			g_ptr_array_add (argv, g_strdup_printf ("-use-the-force-luke=tracksize:%" G_GINT64_FORMAT, sectors));
			if (!g_file_test ("/proc/self/fd/0", G_FILE_TEST_EXISTS)) {
				g_set_error (error,
					     BONFIRE_BURN_ERROR,
					     BONFIRE_BURN_ERROR_GENERAL,
					     _("the file /proc/self/fd/0 is missing"));
				return BONFIRE_BURN_ERR;
			}

			/* FIXME: should we use DAO ? */
			g_ptr_array_add (argv, g_strdup ("-Z"));
			g_ptr_array_add (argv, g_strdup_printf ("%s=/proc/self/fd/0", growisofs->priv->drive->device));

			/* we set the imager as slave */
			bonfire_job_set_slave (BONFIRE_JOB (growisofs), BONFIRE_JOB (imager));
			bonfire_job_set_relay_slave_signals (BONFIRE_JOB (growisofs), FALSE);
			bonfire_job_set_run_slave (BONFIRE_JOB (growisofs), TRUE);
		}
		else if (growisofs->priv->source->type == BONFIRE_TRACK_SOURCE_ISO
		      ||  growisofs->priv->source->type == BONFIRE_TRACK_SOURCE_ISO_JOLIET) {
			/* FIXME: should we use DAO ? */
			g_ptr_array_add (argv, g_strdup ("-Z"));
			if (growisofs->priv->drive->cdrecord_id) 
				g_ptr_array_add (argv, g_strdup_printf ("%s=%s",
									growisofs->priv->drive->cdrecord_id,
									growisofs->priv->source->contents.iso.image));
			else if (growisofs->priv->drive->device)
				g_ptr_array_add (argv, g_strdup_printf ("%s=%s",
									growisofs->priv->drive->device,
									growisofs->priv->source->contents.iso.image));

			bonfire_job_set_run_slave (BONFIRE_JOB (growisofs), FALSE);
		}
		else if (growisofs->priv->source->type == BONFIRE_TRACK_SOURCE_GRAFTS) {
			g_ptr_array_add (argv, g_strdup ("-Z"));
			if (growisofs->priv->drive->cdrecord_id) 
				g_ptr_array_add (argv, g_strdup_printf ("%s",
									growisofs->priv->drive->cdrecord_id));
			else if (growisofs->priv->drive->device)
				g_ptr_array_add (argv, g_strdup_printf ("%s",
									growisofs->priv->drive->device));

			/* this can only happen if source->type == BONFIRE_TRACK_SOURCE_GRAFTS */
			if (growisofs->priv->action == BONFIRE_GROWISOFS_ACTION_GET_SIZE) {
				g_ptr_array_add (argv, g_strdup ("-dry-run"));
				bonfire_job_action_changed (BONFIRE_JOB (growisofs),
							    BONFIRE_BURN_ACTION_GETTING_SIZE,
							    FALSE);
			}

			result = bonfire_growisofs_set_mkisofs_argv (growisofs, 
								     argv,
								     error);
			if (result != BONFIRE_BURN_OK)
				return result;

			bonfire_job_set_run_slave (BONFIRE_JOB (growisofs), FALSE);
		}
		else
			return BONFIRE_BURN_NOT_SUPPORTED;
	}

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_growisofs_set_argv_blank (BonfireGrowisofs *growisofs,
				  GPtrArray *argv)
{
	if (!growisofs->priv->fast_blank) {
		g_ptr_array_add (argv, g_strdup ("-Z"));
		g_ptr_array_add (argv, g_strdup_printf ("%s=%s", 
							growisofs->priv->drive->device,
							"/dev/zero"));

		if (growisofs->priv->dummy)
			g_ptr_array_add (argv, g_strdup ("-use-the-force-luke=dummy"));
	}
	else
		bonfire_job_debug_message (BONFIRE_JOB (growisofs),
					   "Skipping fast blank for already formatted DVD+RW media\n");

	/* we never want any slave to be started */
	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_growisofs_set_argv (BonfireProcess *process,
			    GPtrArray *argv,
			    gboolean has_master,
			    GError **error)
{
	BonfireGrowisofs *growisofs;
	BonfireBurnResult result;

	growisofs = BONFIRE_GROWISOFS (process);

	if (has_master)
		return BONFIRE_BURN_NOT_SUPPORTED;

	g_ptr_array_add (argv, g_strdup ("growisofs"));

	if (growisofs->priv->action == BONFIRE_GROWISOFS_ACTION_RECORD)
		result = bonfire_growisofs_set_argv_record (growisofs,
							    argv,
							    error);
	else if (growisofs->priv->action == BONFIRE_GROWISOFS_ACTION_GET_SIZE)
		result = bonfire_growisofs_set_argv_record (growisofs,
							    argv,
							    error);
	else if (growisofs->priv->action == BONFIRE_GROWISOFS_ACTION_BLANK)
		result = bonfire_growisofs_set_argv_blank (growisofs, argv);
	else
		return BONFIRE_BURN_NOT_READY;

	return result;
}

static BonfireBurnResult
bonfire_growisofs_post (BonfireProcess *process, BonfireBurnResult retval)
{
	BonfireGrowisofs *growisofs;

	growisofs = BONFIRE_GROWISOFS (process);

	if (growisofs->priv->timer) {
		g_timer_destroy (growisofs->priv->timer);
		growisofs->priv->timer = NULL;
	}

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_growisofs_get_action_string (BonfireJob *job,
				     BonfireBurnAction action,
				     char **string)
{
	job = bonfire_job_get_slave (job);
	if (!job)
		return BONFIRE_BURN_NOT_SUPPORTED;

	return bonfire_job_get_action_string (job, action, string);
}
