/***************************************************************************
 *            imager.c
 *
 *  dim jan 22 17:32:17 2006
 *  Copyright  2006  Rouquier Philippe
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>

#include <nautilus-burn-drive.h>

#include "burn-basics.h"
#include "burn-imager.h"
#include "burn-job.h"

static void bonfire_imager_base_init (gpointer g_class);

GType
bonfire_imager_get_type ()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireImagerIFace),
			bonfire_imager_base_init,
			NULL,
			NULL,
		};

		type = g_type_register_static(G_TYPE_INTERFACE, 
					      "BonfireImager",
					      &our_info,
					      0);
	}

	return type;
}

static void
bonfire_imager_base_init (gpointer g_class)
{
        static gboolean initialized = FALSE;

        if (initialized)
		return;

        initialized = TRUE;
}

BonfireBurnResult
bonfire_imager_get_track (BonfireImager *imager,
			  BonfireTrackSource **track,
			  GError **error)
{
	BonfireImagerIFace *iface;

	g_return_val_if_fail (BONFIRE_IS_IMAGER (imager), BONFIRE_BURN_ERR);
	g_return_val_if_fail (track != NULL, BONFIRE_BURN_ERR);

	if (bonfire_job_is_running (BONFIRE_JOB (imager)))
		return BONFIRE_BURN_RUNNING;
	
	iface = BONFIRE_IMAGER_GET_IFACE (imager);
	if (!iface->get_track)
		return BONFIRE_BURN_NOT_SUPPORTED;

	bonfire_job_debug_message (BONFIRE_JOB (imager),
				   "imager (%s) get_track",
				   G_OBJECT_TYPE_NAME (imager));

	return (* iface->get_track) (imager,
				     track,
				     error);
}

BonfireBurnResult
bonfire_imager_set_output_type (BonfireImager *imager,
				BonfireTrackSourceType type,
				GError **error)
{
	BonfireImagerIFace *iface;

	g_return_val_if_fail (type != BONFIRE_TRACK_SOURCE_UNKNOWN, BONFIRE_BURN_ERR);
	g_return_val_if_fail (BONFIRE_IS_IMAGER (imager), BONFIRE_BURN_ERR);

	if (bonfire_job_is_running (BONFIRE_JOB (imager)))
		return BONFIRE_BURN_RUNNING;

	iface = BONFIRE_IMAGER_GET_IFACE (imager);
	if (!iface->set_output_type)
		return BONFIRE_BURN_NOT_SUPPORTED;

	bonfire_job_debug_message (BONFIRE_JOB (imager),
				   "imager (%s) set_output_type",
				   G_OBJECT_TYPE_NAME (imager));

	return (* iface->set_output_type) (imager,
					   type,
					   error);
}

BonfireBurnResult 
bonfire_imager_set_output (BonfireImager *imager,
			   const char *output,
			   gboolean overwrite,
			   gboolean clean,
			   GError **error)
{
	BonfireImagerIFace *iface;

	g_return_val_if_fail (BONFIRE_IS_IMAGER (imager), BONFIRE_BURN_ERR);
	if (bonfire_job_is_running (BONFIRE_JOB (imager)))
		return BONFIRE_BURN_RUNNING;

	iface = BONFIRE_IMAGER_GET_IFACE (imager);
	if (!iface->set_output)
		return BONFIRE_BURN_NOT_SUPPORTED;

	bonfire_job_debug_message (BONFIRE_JOB (imager),
				   "imager (%s) set_output",
				   G_OBJECT_TYPE_NAME (imager));

	return (* iface->set_output) (imager,
				       output,
				       overwrite,
				       clean,
				       error);
}

BonfireBurnResult 
bonfire_imager_set_append (BonfireImager *imager,
			   NautilusBurnDrive *drive,
			   gboolean merge,
			   GError **error)
{
	BonfireImagerIFace *iface;

	g_return_val_if_fail (BONFIRE_IS_IMAGER (imager), BONFIRE_BURN_ERR);
	if (bonfire_job_is_running (BONFIRE_JOB (imager)))
		return BONFIRE_BURN_RUNNING;

	iface = BONFIRE_IMAGER_GET_IFACE (imager);
	if (!iface->set_append)
		return BONFIRE_BURN_NOT_SUPPORTED;

	bonfire_job_debug_message (BONFIRE_JOB (imager),
				   "imager (%s) set_append",
				   G_OBJECT_TYPE_NAME (imager));

	return (* iface->set_append) (imager,
				      drive,
				      merge,
				      error);
}

BonfireBurnResult
bonfire_imager_get_size (BonfireImager *imager,
			 gint64 *size,
			 gboolean sectors,
			 GError **error)
{
	BonfireImagerIFace *iface;

	g_return_val_if_fail (BONFIRE_IS_IMAGER (imager), BONFIRE_BURN_ERR);
	g_return_val_if_fail (size != NULL, BONFIRE_BURN_ERR);

	iface = BONFIRE_IMAGER_GET_IFACE (imager);
	if (!iface->get_size)
		return BONFIRE_BURN_NOT_SUPPORTED;

	bonfire_job_debug_message (BONFIRE_JOB (imager),
				   "imager (%s) get_size",
				   G_OBJECT_TYPE_NAME (imager));

	return (* iface->get_size) (imager, size, sectors, error);
}

/* only available when obj is copying drive */
BonfireBurnResult
bonfire_imager_get_track_type (BonfireImager *imager, BonfireTrackSourceType *type)
{
	BonfireImagerIFace *iface;

	g_return_val_if_fail (BONFIRE_IS_IMAGER (imager), BONFIRE_BURN_ERR);
	g_return_val_if_fail (type != NULL, BONFIRE_BURN_ERR);

	iface = BONFIRE_IMAGER_GET_IFACE (imager);
	if (!iface->get_track_type)
		return BONFIRE_BURN_NOT_SUPPORTED;

	bonfire_job_debug_message (BONFIRE_JOB (imager),
				   "imager (%s) get_track_type",
				   G_OBJECT_TYPE_NAME (imager));

	return (* iface->get_track_type) (imager, type);
}

BonfireBurnResult
bonfire_imager_get_speed (BonfireImager *imager, int *speed)
{
	BonfireImagerIFace *iface;

	g_return_val_if_fail (BONFIRE_IS_IMAGER (imager), BONFIRE_BURN_ERR);

	if (!bonfire_job_is_running (BONFIRE_JOB (imager)))
		return BONFIRE_BURN_NOT_READY;

	iface = BONFIRE_IMAGER_GET_IFACE (imager);
	if (!iface->get_speed)
		return BONFIRE_BURN_NOT_SUPPORTED;

	bonfire_job_debug_message (BONFIRE_JOB (imager),
				   "imager get_speed (%s)",
				   G_OBJECT_TYPE_NAME (imager));

	return (* iface->get_speed) (imager, speed);
}

