/***************************************************************************
 *            job.c
 *
 *  dim jan 22 10:40:26 2006
 *  Copyright  2006  Rouquier Philippe
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif


#include <string.h>

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>

#include "burn-basics.h"
#include "burn-job.h"
#include "bonfire-marshal.h"
 
static void bonfire_job_class_init (BonfireJobClass *klass);
static void bonfire_job_init (BonfireJob *sp);
static void bonfire_job_finalize (GObject *object);


static BonfireBurnResult
bonfire_job_stop (BonfireJob *job,
		  BonfireBurnResult retval,
		  GError *error);

struct _BonfireJobTask {
	BonfireBurnResult retval;
	GMainLoop *loop;
	GError *error;
};
typedef struct _BonfireJobTask BonfireJobTask;

#define TASK_KEY "TASK_KEY"

struct BonfireJobPrivate {
	BonfireBurnAction action;

	BonfireJob *master;
	BonfireJob *slave;

	int relay_slave_signal:1;
	int run_slave:1;

	int is_running:1;
	int dangerous:1;
	int debug:1;
};

typedef enum {
	ERROR_SIGNAL,
	ACTION_CHANGED_SIGNAL,
	PROGRESS_CHANGED_SIGNAL,
	ANIMATION_CHANGED_SIGNAL,
	LAST_SIGNAL
} BonfireJobSignalType;

static guint bonfire_job_signals [LAST_SIGNAL] = { 0 };
static GObjectClass *parent_class = NULL;

GType
bonfire_job_get_type ()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireJobClass),
			NULL,
			NULL,
			(GClassInitFunc)bonfire_job_class_init,
			NULL,
			NULL,
			sizeof (BonfireJob),
			0,
			(GInstanceInitFunc)bonfire_job_init,
		};

		type = g_type_register_static(G_TYPE_OBJECT, 
			"BonfireJob", &our_info, 0);
	}

	return type;
}

static void
bonfire_job_class_init (BonfireJobClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent(klass);
	object_class->finalize = bonfire_job_finalize;
	
	bonfire_job_signals[PROGRESS_CHANGED_SIGNAL] =
	    g_signal_new ("progress_changed",
			  G_TYPE_FROM_CLASS (klass),
			  G_SIGNAL_RUN_LAST,
			  G_STRUCT_OFFSET (BonfireJobClass,
					   progress_changed),
			  NULL, NULL,
			  bonfire_marshal_VOID__DOUBLE_LONG,
			  G_TYPE_NONE, 2, G_TYPE_DOUBLE, G_TYPE_LONG);
	bonfire_job_signals[ACTION_CHANGED_SIGNAL] =
	    g_signal_new ("action_changed",
			  G_TYPE_FROM_CLASS (klass),
			  G_SIGNAL_RUN_LAST,
			  G_STRUCT_OFFSET (BonfireJobClass,
					   action_changed),
			  NULL, NULL,
			  g_cclosure_marshal_VOID__INT,
			  G_TYPE_NONE, 1, G_TYPE_INT);
	bonfire_job_signals[ANIMATION_CHANGED_SIGNAL] =
	    g_signal_new ("animation_changed",
			  G_TYPE_FROM_CLASS (klass),
			  G_SIGNAL_RUN_LAST,
			  G_STRUCT_OFFSET (BonfireJobClass,
					   animation_changed),
			  NULL, NULL,
			  g_cclosure_marshal_VOID__BOOLEAN,
			  G_TYPE_NONE, 1, G_TYPE_BOOLEAN);
	bonfire_job_signals[ERROR_SIGNAL] =
	    g_signal_new ("error",
			  G_TYPE_FROM_CLASS (klass),
			  G_SIGNAL_RUN_LAST,
			  G_STRUCT_OFFSET (BonfireJobClass,
					   error),
			  NULL, NULL,
			  bonfire_marshal_INT__INT,
			  G_TYPE_INT, 1, G_TYPE_INT);
}

static void
bonfire_job_init (BonfireJob *obj)
{
	obj->priv = g_new0 (BonfireJobPrivate, 1);
	obj->priv->relay_slave_signal = 1;
}

static void
bonfire_job_finalize (GObject *object)
{
	BonfireJob *cobj;
	BonfireJobTask *task;

	cobj = BONFIRE_JOB (object);

	task = g_object_get_data (G_OBJECT (cobj), TASK_KEY);
	if (task)
		bonfire_job_stop (cobj, BONFIRE_BURN_CANCEL, NULL);

	/* NOTE: it can't reach this function and have a
	 * master since the master holds a reference on it */
	if (cobj->priv->slave) {
		cobj->priv->slave->priv->master = NULL;
		g_object_unref (cobj->priv->slave);
		cobj->priv->slave = NULL;
	}

	g_free (cobj->priv);
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

void
bonfire_job_set_dangerous (BonfireJob *job, gboolean value)
{
	job->priv->dangerous = value;
}

void
bonfire_job_set_debug (BonfireJob *job, gboolean value)
{
	job->priv->debug = value;
}

static BonfireBurnResult
bonfire_job_loop (BonfireJob *job, GError **error)
{
	BonfireJobTask task = { BONFIRE_BURN_OK, NULL, NULL };

	/* start the loop 
	 * NOTE: loop is unreffed in bonfire_job_stop */
	bonfire_job_debug_message (job, "In loop");

	g_object_set_data (G_OBJECT (job), TASK_KEY, &task);
	task.loop = g_main_loop_new (NULL, FALSE);
	g_main_loop_run (task.loop);

	if (task.error) {
		g_propagate_error (error, task.error);
		task.error = NULL;
	}

	return task.retval;	
}

static BonfireBurnResult
bonfire_job_plug (BonfireJob *job,
		  BonfireJob *prev_job,
		  int *in_fd,
		  int *out_fd,
		  GError **error)
{
	BonfireBurnResult result;
	BonfireJobClass *klass;

	klass = BONFIRE_JOB_GET_CLASS (job);
	if (!klass->start) {
		bonfire_job_debug_message (job, "strange implementation, that job can't be started");
		return BONFIRE_BURN_NOT_SUPPORTED;
	}

	result = klass->start (job,
			       *in_fd,
			       out_fd,
			       error);

	if (result != BONFIRE_BURN_OK)
		return result;

	if (out_fd && *out_fd == -1) {
		bonfire_job_debug_message (job, "this job can't be plugged into another");
		return BONFIRE_BURN_NOT_SUPPORTED;
	}

	job->priv->is_running = 1;

	if (out_fd && *out_fd != -1) {
		*in_fd = *out_fd;
		*out_fd = -1;
	}

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_job_send_stop_signal (BonfireJob *job,
			      BonfireBurnResult retval)
{
	BonfireJob *iter;
	BonfireJobClass *klass;
	BonfireBurnResult result = retval;

	/* we stop all the first slave first and then go up the list */
	iter = job;
	while (iter->priv->slave && iter->priv->slave->priv->is_running)
		iter = iter->priv->slave;

	do {
		klass = BONFIRE_JOB_GET_CLASS (iter);

		bonfire_job_debug_message (iter,
					   "stopping %s",
					   G_OBJECT_TYPE_NAME (iter));

		if (klass->stop)
			result = klass->stop (iter, result);

		bonfire_job_debug_message (iter,
					   "%s stopped",
					   G_OBJECT_TYPE_NAME (iter));

		iter->priv->is_running = 0;
		iter = iter->priv->master;
	} while (iter && iter != job->priv->master);

	/* we don't want to lose the original result if it was not OK */
	if (retval != BONFIRE_BURN_OK)
		return retval;

	return result;
}

static BonfireBurnResult
bonfire_job_pre_init (BonfireJob *job,
		      gboolean has_master_running,
		      GError **error)
{
	BonfireJobClass *klass;
	BonfireBurnResult result = BONFIRE_BURN_OK;

	klass = BONFIRE_JOB_GET_CLASS (job);

	if (!BONFIRE_IS_JOB (job)) {
		g_warning ("object %p is not a job.", job);
		return BONFIRE_BURN_ERR;
	}

	if (g_object_get_data (G_OBJECT (job), TASK_KEY)) {
		bonfire_job_debug_message (job, "object is already in use.");
		return BONFIRE_BURN_RUNNING;
	}

	if (klass->start_init)
		result = klass->start_init (job,
					    has_master_running,
					    error);

	return result;
}

BonfireBurnResult
bonfire_job_run (BonfireJob *job, GError **error)
{
	BonfireBurnResult result = BONFIRE_BURN_OK;
	BonfireJob *prev_job = NULL;
	BonfireJob *iter;
	int out_fd = -1;
	int in_fd = -1;

	/* check the job or one of its master is not running */
	if (bonfire_job_is_running (job)) {
		bonfire_job_debug_message (job, "object or master or slave is already running");
		return BONFIRE_BURN_RUNNING;
	}

	/* we first init all jobs starting from the master down to the slave */
	iter = job;
	while (1) {
		result = bonfire_job_pre_init (iter, (iter != job), error);
		if (result != BONFIRE_BURN_OK)
			goto error;

		if (!iter->priv->run_slave)
			break;

		if (!iter->priv->slave)
			break;

		iter = iter->priv->slave;
	}	

	/* now start from the slave up to the master */
	prev_job = NULL;
	while (iter != job) {
		result = bonfire_job_plug (iter,
					   prev_job,
					   &in_fd,
					   &out_fd,
					   error);

		if (result != BONFIRE_BURN_OK)
			goto error;
	
		prev_job = iter;
		iter = iter->priv->master;
	}
	
	result = bonfire_job_plug (iter,
				   prev_job,
				   &in_fd,
				   NULL,
				   error);

	if (result != BONFIRE_BURN_OK)
		goto error;

	result = bonfire_job_loop (iter, error);
	return result;

error:
	bonfire_job_send_stop_signal (job, result);
	return result;
}

gboolean
bonfire_job_is_running (BonfireJob *job)
{
	g_return_val_if_fail (job != NULL, FALSE);

	if (job->priv->is_running)
		return TRUE;

	return FALSE;
}

static BonfireBurnResult
bonfire_job_stop (BonfireJob *job,
		  BonfireBurnResult retval,
		  GError *error)
{
	BonfireBurnResult result = BONFIRE_BURN_OK;
	BonfireJobTask *task = NULL;
	BonfireJob *job_owning_task;
	BonfireJob *iter;
	GMainLoop *loop;

	bonfire_job_debug_message (job, "job %s ask to stop", G_OBJECT_TYPE_NAME (job));

	if (job->priv->is_running) {
		task = g_object_get_data (G_OBJECT (job), TASK_KEY);

		job_owning_task = job;
	}
	else {
		/* maybe we signalled a master to stop and it isn't running
		 * while his slaves are: see if it has slaves running */
		iter = job->priv->slave;
		while (iter && !(task = g_object_get_data (G_OBJECT (iter), TASK_KEY)))
			iter = iter->priv->slave;

		job_owning_task = iter;
	}

	if (!task) {
		/* if we reached this point it means that the job itself nor its slaves
		 * is controlling the running */

		/* if this job hasn't got the running bit no need to see if its master's
		 * running for fear we might stop the master unwillingly */
		if (!job->priv->is_running) {
			bonfire_job_debug_message (job, "This job nor his slaves are running.");
			return BONFIRE_BURN_NOT_RUNNING;
		}

		/* search the master which is running */
		iter = job->priv->master;
		while (iter && !(task = g_object_get_data (G_OBJECT (iter), TASK_KEY)))
			iter = iter->priv->master;

		if (!task) {
			bonfire_job_debug_message (job, "This job nor his slaves are running. Strange it hasn't got any master running either.");
			return BONFIRE_BURN_NOT_RUNNING;
		}

		job_owning_task = iter;

		/* we discard all messages from slaves saying that all is good */
		if (retval == BONFIRE_BURN_OK) {
			if (task->loop) {
				bonfire_job_debug_message (job, "This job is not a leader.");
				return BONFIRE_BURN_ERR;
			}

			return BONFIRE_BURN_OK;
		}
	}

	/* means we still accept errors. This is meant for
	 * master to override the errors of their slaves */
	if (!task->loop) {
		if (error) {
			if (task->error)
				g_error_free (task->error);

			task->retval = retval;
			task->error = error;
		}

		return BONFIRE_BURN_OK;
	}

	loop = task->loop;
	task->loop = NULL;

	/* tell all the jobs we've been cancelled */
	result = bonfire_job_send_stop_signal (job_owning_task, retval);

	/* stop the loop */
	g_object_set_data (G_OBJECT (job), TASK_KEY, NULL);

	if (job_owning_task == job) {
		if (task->error)
			g_error_free (task->error);

		task->error = error;
		task->retval = result;
	}
	else if (!task->error) {
		task->error = error;
		task->retval = retval;
	}

	g_main_loop_quit (loop);
	g_main_loop_unref (loop);

	bonfire_job_debug_message (job, "getting out of loop");
	return result;
}

BonfireBurnResult
bonfire_job_cancel (BonfireJob *job, gboolean protect)
{
	g_return_val_if_fail (job != NULL, BONFIRE_BURN_ERR);

	if (protect && job->priv->dangerous)
		return BONFIRE_BURN_DANGEROUS;

	return bonfire_job_stop (job, BONFIRE_BURN_CANCEL, NULL);
}

/* used for implementation */
BonfireBurnResult
bonfire_job_finished (BonfireJob *job)
{
	return bonfire_job_stop (job, BONFIRE_BURN_OK, NULL);
}

BonfireBurnResult
bonfire_job_error (BonfireJob *job, GError *error)
{
	BonfireBurnResult result = BONFIRE_BURN_ERR;

	/* There was an error: signal it. That's mainly done
	 * for BonfireBurnCaps to override the result value */
	g_signal_emit (job,
		       bonfire_job_signals [ERROR_SIGNAL],
		       0,
		       error->code,
		       &result);

	return bonfire_job_stop (job, result, error);
}

static BonfireJob *
bonfire_job_get_emitter (BonfireJob *job)
{
	if (!job->priv->master)
		return job;

	while (job->priv->master)
		job = job->priv->master;

	if (job->priv->relay_slave_signal)
		return job;

	return NULL;
}

void
bonfire_job_progress_changed (BonfireJob *job,
			      gdouble progress,
			      long remaining_time)
{
	job = bonfire_job_get_emitter (job);
	if (!job)
		return;

	g_signal_emit (job,
		       bonfire_job_signals [PROGRESS_CHANGED_SIGNAL],
		       0,
		       progress,
		       remaining_time);
}

void
bonfire_job_action_changed (BonfireJob *job,
			    BonfireBurnAction action,
			    gboolean force)
{
	BonfireJob *emitter;

	/* NOTE: emitter must be the top master */
	emitter = bonfire_job_get_emitter (job);
	if (!emitter)
		return;

	if (!force && job->priv->action == action)
		return;

	job->priv->action = action;
	g_signal_emit (emitter,
		       bonfire_job_signals [ACTION_CHANGED_SIGNAL],
		       0,
		       action);
}

BonfireBurnAction
bonfire_job_get_current_action (BonfireJob *job)
{
	g_return_val_if_fail (BONFIRE_IS_JOB (job), BONFIRE_BURN_ACTION_NONE);

	return job->priv->action;
}

void
bonfire_job_animation_changed (BonfireJob *job, gboolean spinning)
{
	job = bonfire_job_get_emitter (job);
	if (!job)
		return;

	g_signal_emit (job,
		       bonfire_job_signals [ANIMATION_CHANGED_SIGNAL],
		       0,
		       spinning);
}

void
bonfire_job_set_run_slave (BonfireJob *job, gboolean run_slave)
{
	g_return_if_fail (BONFIRE_IS_JOB (job));
	job->priv->run_slave = (run_slave == TRUE);
}

void
bonfire_job_set_relay_slave_signals (BonfireJob *job, gboolean relay)
{
	g_return_if_fail (BONFIRE_IS_JOB (job));
	job->priv->relay_slave_signal = relay;
}

BonfireJob *
bonfire_job_get_slave (BonfireJob *master)
{
	g_return_val_if_fail (BONFIRE_IS_JOB (master), NULL);

	return master->priv->slave;
}

BonfireBurnResult
bonfire_job_set_slave (BonfireJob *master, BonfireJob *slave)
{
	g_return_val_if_fail (BONFIRE_IS_JOB (master), BONFIRE_BURN_ERR);
	g_return_val_if_fail (master != slave, BONFIRE_BURN_ERR);

	if (slave)
		g_return_val_if_fail (BONFIRE_IS_JOB (slave), BONFIRE_BURN_ERR);

	/* check if one of them is running */
	if (bonfire_job_is_running (master)
	|| (slave && bonfire_job_is_running (slave)))
		return BONFIRE_BURN_RUNNING;

	/* set */
	if (master->priv->slave) {
		master->priv->slave->priv->master = NULL;
		g_object_unref (master->priv->slave);
	}

	master->priv->slave = slave;

	if (!slave)
		return BONFIRE_BURN_OK;

	/* NOTE: the slave may already have a reference from a master,
	 * that's why we don't unref it to ref it just afterwards in 
	 * case its reference count reaches zero in between*/
	if (slave->priv->master)
		slave->priv->master->priv->slave = NULL;
	else
		g_object_ref (slave);

	slave->priv->master = master;
	return BONFIRE_BURN_OK;
}

BonfireBurnResult
bonfire_job_get_action_string (BonfireJob *job,
			       BonfireBurnAction action,
			       char **string)
{
	const char *tmp;
	BonfireJobClass *klass;
	BonfireBurnResult result;

	g_return_val_if_fail (BONFIRE_IS_JOB (job), BONFIRE_BURN_ERR);
	g_return_val_if_fail (string != NULL, BONFIRE_BURN_ERR);

	bonfire_job_debug_message (BONFIRE_JOB (job),
				   "job (%s) get_action_string",
				   G_OBJECT_TYPE_NAME (job));

	klass = BONFIRE_JOB_GET_CLASS (job);
	if (klass->get_action_string) {
		result = (* klass->get_action_string) (job,
						       action,
						       string);
		if (result == BONFIRE_BURN_OK)
			return result;
	}

	tmp = bonfire_burn_action_to_string (action);
	*string = g_strdup (tmp);

	return BONFIRE_BURN_OK;
}

BonfireBurnResult
bonfire_job_set_source (BonfireJob *job,
			const BonfireTrackSource *source,
			GError **error)
{
	BonfireJobClass *klass;

	g_return_val_if_fail (BONFIRE_IS_JOB (job), BONFIRE_BURN_ERR);

	bonfire_job_debug_message (job,
				   "job (%s) set_source",
				   G_OBJECT_TYPE_NAME (job));

	if (bonfire_job_is_running (job))
		return BONFIRE_BURN_RUNNING;

	klass = BONFIRE_JOB_GET_CLASS (job);
	if (!klass->set_source)
		return BONFIRE_BURN_NOT_SUPPORTED;

	return (* klass->set_source) (job,
				       source,
				       error);
}

BonfireBurnResult
bonfire_job_set_rate (BonfireJob *job, gint64 rate)
{
	BonfireJobClass *klass;
	BonfireBurnResult result;

	g_return_val_if_fail (BONFIRE_IS_JOB (job), BONFIRE_BURN_ERR);

	klass = BONFIRE_JOB_GET_CLASS (job);
	if (klass->set_rate)
		result = klass->set_rate (job, rate);
	else
		result = BONFIRE_BURN_NOT_SUPPORTED;

	return result;
}

/* used for debugging */
void
bonfire_job_debug_message (BonfireJob *job, const char *format, ...)
{
	va_list arg_list;

	g_return_if_fail (job != NULL);
	if (format == NULL)
		return;

	while (job->priv->master)
		job = job->priv->master;

	if (!job->priv->debug)
		return;

	va_start (arg_list, format);
	g_logv (NULL,
		G_LOG_LEVEL_DEBUG,
		format,
		arg_list);
	va_end (arg_list);
}
