/***************************************************************************
 *            mkisofs.c
 *
 *  dim jan 22 15:20:57 2006
 *  Copyright  2006  Rouquier Philippe
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <time.h>

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>
#include <glib/gstdio.h>

#include <nautilus-burn-drive.h>

#include "burn-basics.h"
#include "burn-caps.h"
#include "burn-common.h"
#include "burn-process.h"
#include "burn-mkisofs.h"
#include "burn-imager.h"
 
static void bonfire_mkisofs_class_init (BonfireMkisofsClass *klass);
static void bonfire_mkisofs_init (BonfireMkisofs *sp);
static void bonfire_mkisofs_finalize (GObject *object);
static void bonfire_mkisofs_iface_init_image (BonfireImagerIFace *iface);

static BonfireBurnResult
bonfire_mkisofs_read_stdout (BonfireProcess *process, const char *line);
static BonfireBurnResult
bonfire_mkisofs_read_stderr (BonfireProcess *process, const char *line);
static BonfireBurnResult
bonfire_mkisofs_set_argv (BonfireProcess *process,
			  GPtrArray *argv,
			  gboolean has_master,
			  GError **error);
static BonfireBurnResult
bonfire_mkisofs_post (BonfireProcess *process, BonfireBurnResult retval);

static BonfireBurnResult
bonfire_mkisofs_set_source (BonfireJob *job,
			    const BonfireTrackSource *source,
			    GError **error);
static BonfireBurnResult
bonfire_mkisofs_set_append (BonfireImager *imager,
			    NautilusBurnDrive *drive,
			    gboolean merge,
			    GError **error);
static BonfireBurnResult
bonfire_mkisofs_set_output (BonfireImager *imager,
			    const char *output,
			    gboolean overwrite,
			    gboolean clean,
			    GError **error);
static BonfireBurnResult
bonfire_mkisofs_set_output_type (BonfireImager *imager,
				 BonfireTrackSourceType type,
				 GError **error);
static BonfireBurnResult
bonfire_mkisofs_get_track (BonfireImager *imager,
			   BonfireTrackSource **track,
			   GError **error);
static BonfireBurnResult
bonfire_mkisofs_get_track_type (BonfireImager *imager,
				BonfireTrackSourceType *type);
static BonfireBurnResult
bonfire_mkisofs_get_size (BonfireImager *imager,
			  gint64 *size,
			  gboolean sectors,
			  GError **error);
static BonfireBurnResult
bonfire_mkisofs_get_action_string (BonfireJob *job,
				   BonfireBurnAction action,
				   char **string);

typedef enum {
	BONFIRE_MKISOFS_ACTION_NONE,
	BONFIRE_MKISOFS_ACTION_GET_SIZE,
	BONFIRE_MKISOFS_ACTION_GET_IMAGE
} BonfireMkisofsAction;

struct BonfireMkisofsPrivate {
	BonfireMkisofsAction action;
	BonfireTrackSourceType track_type;

	GTimer *timer;

	int sectors_num;

        NautilusBurnDrive *drive;
	char *output;

	BonfireTrackSource *source;

	int use_joliet:1;
	int overwrite:1;
	int use_utf8:1;
	int merge:1;
	int clean:1;

	int iso_ready:1;
	int iso_joliet_ready:1;
};

static GObjectClass *parent_class = NULL;

GType
bonfire_mkisofs_get_type ()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireMkisofsClass),
			NULL,
			NULL,
			(GClassInitFunc)bonfire_mkisofs_class_init,
			NULL,
			NULL,
			sizeof (BonfireMkisofs),
			0,
			(GInstanceInitFunc)bonfire_mkisofs_init,
		};
		static const GInterfaceInfo imager_info =
		{
			(GInterfaceInitFunc) bonfire_mkisofs_iface_init_image,
			NULL,
			NULL
		};
		type = g_type_register_static (BONFIRE_TYPE_PROCESS, 
					       "BonfireMkisofs",
					       &our_info,
					       0);
		g_type_add_interface_static (type,
					     BONFIRE_TYPE_IMAGER,
					     &imager_info);
	}

	return type;
}

static void
bonfire_mkisofs_class_init (BonfireMkisofsClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	BonfireJobClass *job_class = BONFIRE_JOB_CLASS (klass);
	BonfireProcessClass *process_class = BONFIRE_PROCESS_CLASS (klass);

	parent_class = g_type_class_peek_parent(klass);
	object_class->finalize = bonfire_mkisofs_finalize;

	job_class->get_action_string = bonfire_mkisofs_get_action_string;
	job_class->set_source = bonfire_mkisofs_set_source;

	process_class->stdout_func = bonfire_mkisofs_read_stdout;
	process_class->stderr_func = bonfire_mkisofs_read_stderr;
	process_class->set_argv = bonfire_mkisofs_set_argv;
	process_class->post = bonfire_mkisofs_post;
}

static void
bonfire_mkisofs_iface_init_image (BonfireImagerIFace *iface)
{
	iface->get_size = bonfire_mkisofs_get_size;
	iface->get_track = bonfire_mkisofs_get_track;
	iface->get_track_type = bonfire_mkisofs_get_track_type;
	iface->set_output = bonfire_mkisofs_set_output;
	iface->set_append = bonfire_mkisofs_set_append;
	iface->set_output_type = bonfire_mkisofs_set_output_type;
}

static void
bonfire_mkisofs_init (BonfireMkisofs *obj)
{
	char *standard_error;
	gboolean res;

	obj->priv = g_new0(BonfireMkisofsPrivate, 1);

	/* this code used to be ncb_mkisofs_supports_utf8 */
	res = g_spawn_command_line_sync ("mkisofs -input-charset utf8",
					 NULL,
					 &standard_error,
					 NULL,
					 NULL);

	if (res && !g_strrstr (standard_error, "Unknown charset"))
		obj->priv->use_utf8 = TRUE;
	else
		obj->priv->use_utf8 = FALSE;

	g_free (standard_error);
}

static void
bonfire_mkisofs_clear (BonfireMkisofs *mkisofs)
{
	if (mkisofs->priv->source) {
		bonfire_track_source_free (mkisofs->priv->source);
		mkisofs->priv->source = NULL;
	}

	if (mkisofs->priv->drive) {
		nautilus_burn_drive_unref (mkisofs->priv->drive);
		mkisofs->priv->drive = NULL;
	}

	if (mkisofs->priv->output) {
		if (mkisofs->priv->clean)
			g_remove (mkisofs->priv->output);

		g_free (mkisofs->priv->output),
		mkisofs->priv->output = NULL;
	}

	mkisofs->priv->use_utf8 = FALSE;
	mkisofs->priv->use_joliet = FALSE;
}
	
static void
bonfire_mkisofs_finalize (GObject *object)
{
	BonfireMkisofs *cobj;
	cobj = BONFIRE_MKISOFS(object);

	bonfire_mkisofs_clear (cobj);

	g_free(cobj->priv);
	G_OBJECT_CLASS(parent_class)->finalize(object);
}

BonfireMkisofs *
bonfire_mkisofs_new ()
{
	BonfireMkisofs *obj;
	
	obj = BONFIRE_MKISOFS(g_object_new(BONFIRE_TYPE_MKISOFS, NULL));
	return obj;
}

static BonfireBurnResult
bonfire_mkisofs_read_isosize (BonfireMkisofs *mkisofs, const char *line)
{
	/* mkisofs reports blocks of 2048 bytes */
	mkisofs->priv->sectors_num = atoi (line);
	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_mkisofs_read_stdout (BonfireProcess *process, const char *line)
{
	BonfireMkisofs *mkisofs;

	mkisofs = BONFIRE_MKISOFS (process);

	if (mkisofs->priv->action == BONFIRE_MKISOFS_ACTION_GET_SIZE)
		return bonfire_mkisofs_read_isosize (mkisofs, line);

	return TRUE;
}

static BonfireBurnResult
bonfire_mkisofs_read_stderr (BonfireProcess *process, const char *line)
{
	char fraction_str [7];

	if (strstr (line, "estimate finish")
	&&  sscanf (line, "%6c%% done, estimate finish", fraction_str) == 1) {
		double fraction;
		BonfireMkisofs *mkisofs;
	
		mkisofs = BONFIRE_MKISOFS (process);

		fraction = g_strtod (fraction_str, NULL);
		if (mkisofs->priv->timer) {
			long secs = -1;
			gdouble elapsed;

			elapsed = g_timer_elapsed (mkisofs->priv->timer, NULL);
			secs = 100.0 * elapsed / fraction - elapsed;

			bonfire_job_progress_changed (BONFIRE_JOB (process),
						      fraction / 100.0,
						      secs);
		}
		else
			mkisofs->priv->timer = g_timer_new ();
	}
	else if (!strncmp (line, "Total translation table size", 28)) {
		bonfire_job_progress_changed (BONFIRE_JOB (process), 1.0, -1);
	}
	else if (!strncmp (line, "Total extents scheduled to be written", 37)) {
		bonfire_job_progress_changed (BONFIRE_JOB (process), 1.0, -1);
	}
	else if (strstr (line, "Unable to sort directory")) {
		bonfire_job_error (BONFIRE_JOB (process), 
				   g_error_new_literal (BONFIRE_BURN_ERROR,
							BONFIRE_BURN_ERROR_GENERAL,
							_("the image can't be created")));
	}
	else if (strstr (line, "Joliet tree sort failed.")) {
		bonfire_job_error (BONFIRE_JOB (process), 
				   g_error_new_literal (BONFIRE_BURN_ERROR,
							BONFIRE_BURN_ERROR_JOLIET_TREE,
							_("the image can't be created")));
	}
	else if (strstr (line, "Use mkisofs -help")) {
		bonfire_job_error (BONFIRE_JOB (process), 
				   g_error_new_literal (BONFIRE_BURN_ERROR,
							BONFIRE_BURN_ERROR_JOLIET_TREE,
							_("this version of mkisofs doesn't seem to be supported")));
	}
/*	else if ((pos =  strstr (line,"mkisofs: Permission denied. "))) {
		int res = FALSE;
		gboolean isdir = FALSE;
		char *path = NULL;

		pos += strlen ("mkisofs: Permission denied. ");
		if (!strncmp (pos, "Unable to open directory ", 24)) {
			isdir = TRUE;

			pos += strlen ("Unable to open directory ");
			path = g_strdup (pos);
			path[strlen (path) - 1] = 0;
		}
		else if (!strncmp (pos, "File ", 5)) {
			char *end;

			isdir = FALSE;
			pos += strlen ("File ");
			end = strstr (pos, " is not readable - ignoring");
			if (end)
				path = g_strndup (pos, end - pos);
		}
		else
			return TRUE;

		res = bonfire_mkisofs_base_ask_unreadable_file (BONFIRE_MKISOFS_BASE (process),
								path,
								isdir);
		if (!res) {
			g_free (path);

			bonfire_job_progress_changed (BONFIRE_JOB (process), 1.0, -1);
			bonfire_job_cancel (BONFIRE_JOB (process), FALSE);
			return FALSE;
		}
	}*/
	else if (strstr (line, "Incorrectly encoded string")) {
		bonfire_job_error (BONFIRE_JOB (process),
				   g_error_new_literal (BONFIRE_BURN_ERROR,
							BONFIRE_BURN_ERROR_JOLIET_TREE,
							_("Some files have invalid filenames")));
	}
	else if (strstr (line, "Unknown charset")) {
		bonfire_job_error (BONFIRE_JOB (process),
				   g_error_new_literal (BONFIRE_BURN_ERROR,
							BONFIRE_BURN_ERROR_GENERAL,
							_("Unknown character encoding")));
	}
	else if (strstr (line, "No space left on device")) {
		bonfire_job_error (BONFIRE_JOB (process),
				   g_error_new_literal (BONFIRE_BURN_ERROR,
							BONFIRE_BURN_ERROR_DISC_SPACE,
							_("There is no space left on the device")));
	}
	else if (strstr (line, "Value too large for defined data type")) {
		/* TODO: get filename from error message */
		bonfire_job_error (BONFIRE_JOB (process),
				   g_error_new_literal (BONFIRE_BURN_ERROR,
							BONFIRE_BURN_ERROR_GENERAL,
							_("The file is too large for a CD")));
	}

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_mkisofs_set_append (BonfireImager *imager,
			    NautilusBurnDrive *drive,
			    gboolean merge,
			    GError **error)
{
	BonfireMkisofs *mkisofs;

	mkisofs = BONFIRE_MKISOFS (imager);

	if (mkisofs->priv->output
	&& (mkisofs->priv->iso_ready || mkisofs->priv->iso_joliet_ready)
	&&  mkisofs->priv->clean)
		g_remove (mkisofs->priv->output);

	mkisofs->priv->iso_ready = 0;
	mkisofs->priv->iso_joliet_ready = 0;

	if (mkisofs->priv->drive) {
		nautilus_burn_drive_unref (mkisofs->priv->drive);
		mkisofs->priv->drive = NULL;
	}

        nautilus_burn_drive_ref (drive);
	mkisofs->priv->drive = drive;
	mkisofs->priv->merge = merge;

	/* we'll get the start point later */
	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_mkisofs_set_output (BonfireImager *imager,
			    const char *output,
			    gboolean overwrite,
			    gboolean clean,
			    GError **error)
{
	BonfireMkisofs *mkisofs;

	mkisofs = BONFIRE_MKISOFS (imager);

	if (mkisofs->priv->output) {
		if ((mkisofs->priv->iso_ready || mkisofs->priv->iso_joliet_ready)
		&&  mkisofs->priv->clean)
			g_remove (mkisofs->priv->output);

		g_free (mkisofs->priv->output);
		mkisofs->priv->output = NULL;

		mkisofs->priv->iso_ready = 0;
		mkisofs->priv->iso_joliet_ready = 0;
	}

	if (output)
		mkisofs->priv->output = g_strdup (output);

	mkisofs->priv->overwrite = overwrite;
	mkisofs->priv->clean = clean;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_mkisofs_set_output_type (BonfireImager *imager,
				 BonfireTrackSourceType type,
				 GError **error)
{
	BonfireMkisofs *mkisofs;

	mkisofs = BONFIRE_MKISOFS (imager);

	if (type == BONFIRE_TRACK_SOURCE_DEFAULT)
		type = BONFIRE_TRACK_SOURCE_ISO_JOLIET;
	else if (type != BONFIRE_TRACK_SOURCE_ISO
	      &&  type != BONFIRE_TRACK_SOURCE_ISO_JOLIET)
		return BONFIRE_BURN_NOT_SUPPORTED;

	if (mkisofs->priv->track_type == type)
		return BONFIRE_BURN_OK;

	mkisofs->priv->track_type = type;
	if (mkisofs->priv->output) {
		if ((mkisofs->priv->iso_ready || mkisofs->priv->iso_joliet_ready)
		&&  mkisofs->priv->clean)
			g_remove (mkisofs->priv->output);

		mkisofs->priv->iso_ready = 0;
		mkisofs->priv->iso_joliet_ready = 0;
	}
	
	if (type == BONFIRE_TRACK_SOURCE_ISO_JOLIET)
		mkisofs->priv->use_joliet = 1;
	else 
		mkisofs->priv->use_joliet = 0;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_mkisofs_get_track_type (BonfireImager *imager,
				BonfireTrackSourceType *type)
{
	BonfireMkisofs *mkisofs;

	mkisofs = BONFIRE_MKISOFS (imager);

	if (mkisofs->priv->use_joliet)
		*type = BONFIRE_TRACK_SOURCE_ISO_JOLIET;
	else
		*type = BONFIRE_TRACK_SOURCE_ISO;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_mkisofs_get_grafts (BonfireMkisofs *mkisofs,
			    const BonfireTrackSource *source,
			    GError **error)
{
	BonfireBurnCaps *caps;
	BonfireBurnResult result;
	BonfireImager *imager = NULL;

	/* we need to download all non local files first
	* and make a list of excluded and graft points */

	/* ask BurnCaps to create an object to get GRAFTS */
	caps = bonfire_burn_caps_get_default ();
	result = bonfire_burn_caps_create_imager (caps,
						  &imager,
						  source,
						  BONFIRE_TRACK_SOURCE_GRAFTS,
						  NAUTILUS_BURN_MEDIA_TYPE_UNKNOWN,
						  NAUTILUS_BURN_MEDIA_TYPE_UNKNOWN,
						  error);
	g_object_unref (caps);

	/* that way the slave will be unref at the same
	 * time as us or if we set another slave */
	bonfire_job_set_slave (BONFIRE_JOB (mkisofs), BONFIRE_JOB (imager));
	g_object_unref (imager);

	result = bonfire_job_set_source (BONFIRE_JOB (imager),
					 source,
					 error);
	if (result != BONFIRE_BURN_OK)
		return result;

	result = bonfire_imager_set_output (imager,
					    NULL,
					    FALSE,
					    TRUE,
					    error);
	if (result != BONFIRE_BURN_OK)
		return result;

	result = bonfire_imager_set_output_type (imager,
						 BONFIRE_TRACK_SOURCE_GRAFTS,
						 error);
	if (result != BONFIRE_BURN_OK)
		return result;

	bonfire_job_set_relay_slave_signals (BONFIRE_JOB (mkisofs), TRUE);
	result = bonfire_imager_get_track (imager,
					   &mkisofs->priv->source,
					   error);
	bonfire_job_set_relay_slave_signals (BONFIRE_JOB (mkisofs), FALSE);
	return result;
}

static BonfireBurnResult
bonfire_mkisofs_set_source (BonfireJob *job,
			    const BonfireTrackSource *source,
			    GError **error)
{
	BonfireMkisofs *mkisofs;
	BonfireBurnResult result = BONFIRE_BURN_OK;

	mkisofs = BONFIRE_MKISOFS (job);

	if (mkisofs->priv->source) {
		bonfire_track_source_free (mkisofs->priv->source);
		mkisofs->priv->source = NULL;
	}
	mkisofs->priv->sectors_num = 0;

	if (mkisofs->priv->output
	&& (mkisofs->priv->iso_ready || mkisofs->priv->iso_joliet_ready)
	&&  mkisofs->priv->clean)
		g_remove (mkisofs->priv->output);

	mkisofs->priv->iso_ready = 0;
	mkisofs->priv->iso_joliet_ready = 0;

	if (source->type != BONFIRE_TRACK_SOURCE_GRAFTS
	&&  source->type != BONFIRE_TRACK_SOURCE_DATA)
		return BONFIRE_BURN_NOT_SUPPORTED;

	if (source->type == BONFIRE_TRACK_SOURCE_DATA)
		result = bonfire_mkisofs_get_grafts (mkisofs, source, error);
	else
		mkisofs->priv->source = bonfire_track_source_copy (source);

	return result;
}

static BonfireBurnResult
bonfire_mkisofs_set_argv_image (BonfireMkisofs *mkisofs,
				GPtrArray *argv,
				gboolean has_master,
				GError **error)
{
	BonfireBurnResult result;

	if (!mkisofs->priv->source || mkisofs->priv->source->type != BONFIRE_TRACK_SOURCE_GRAFTS)
		return BONFIRE_BURN_NOT_READY;

	/* set argv */
	g_ptr_array_add (argv, g_strdup ("-r"));

	if (mkisofs->priv->use_joliet)
		g_ptr_array_add (argv, g_strdup ("-J"));

	g_ptr_array_add (argv, g_strdup ("-graft-points"));
	g_ptr_array_add (argv, g_strdup ("-D"));	// This is dangerous the manual says but apparently it works well

	g_ptr_array_add (argv, g_strdup ("-path-list"));
	g_ptr_array_add (argv, g_strdup (mkisofs->priv->source->contents.grafts.grafts_path));

	if (mkisofs->priv->source->contents.grafts.excluded_path) {
		g_ptr_array_add (argv, g_strdup ("-exclude-list"));
		g_ptr_array_add (argv, g_strdup (mkisofs->priv->source->contents.grafts.excluded_path));
	}

	if (mkisofs->priv->action == BONFIRE_MKISOFS_ACTION_GET_SIZE) {
		g_ptr_array_add (argv, g_strdup ("-q"));
		g_ptr_array_add (argv, g_strdup ("-print-size"));
		bonfire_job_action_changed (BONFIRE_JOB (mkisofs),
					    BONFIRE_BURN_ACTION_GETTING_SIZE,
					    FALSE);
		return BONFIRE_BURN_OK;
	}

	if (mkisofs->priv->source->contents.grafts.label) {
		g_ptr_array_add (argv, g_strdup ("-V"));
		g_ptr_array_add (argv, g_strdup (mkisofs->priv->source->contents.grafts.label));
	}

	g_ptr_array_add (argv, g_strdup ("-A"));
	g_ptr_array_add (argv, g_strdup_printf ("Bonfire-%i.%i.%i",
						BONFIRE_MAJOR_VERSION,
						BONFIRE_MINOR_VERSION,
						BONFIRE_SUB));
	
	g_ptr_array_add (argv, g_strdup ("-sysid"));
	g_ptr_array_add (argv, g_strdup ("LINUX"));
	
	/* FIXME! -sort is an interesting option allowing to decide where the 
	* files are written on the disc and therefore to optimize later reading */
	/* FIXME: -hidden --hidden-list -hide-jolie -hide-joliet-list will allow to hide
	* some files when we will display the contents of a disc we will want to merge */
	/* FIXME: support preparer publisher options */

	if (!has_master) {
		/* see if an output was given otherwise create a temp one */
		result = bonfire_burn_common_check_output (&mkisofs->priv->output,
							   mkisofs->priv->overwrite,
							   NULL,
							   error);
		if (result != BONFIRE_BURN_OK)
			return result;

		g_ptr_array_add (argv, g_strdup ("-o"));
		g_ptr_array_add (argv, g_strdup (mkisofs->priv->output));
	}

	if (mkisofs->priv->drive) {
		char *startpoint = NULL;
		char *command = NULL;
		gboolean res;
		int tmp;

		if (mkisofs->priv->drive->cdrecord_id) 
			command = g_strdup_printf ("cdrecord -msinfo dev=%s",
						   mkisofs->priv->drive->cdrecord_id);
		else if (mkisofs->priv->drive->device)
			command = g_strdup_printf ("cdrecord -msinfo dev=%s",
						   mkisofs->priv->drive->device);
		 
		res = g_spawn_command_line_sync (command,
						 &startpoint,
						 NULL,
						 NULL,
						 NULL);
		g_free (command);

		if (!res && sscanf (startpoint, "%d,%d", &tmp, &tmp) != 2) {
			g_set_error (error,
				     BONFIRE_BURN_ERROR,
				     BONFIRE_BURN_ERROR_GENERAL,
				     _("failed to get the start point of the track. Make sure the media allow to add files (it is not closed)"));
			return BONFIRE_BURN_ERR;
		}

		g_ptr_array_add (argv, g_strdup ("-C"));
		g_ptr_array_add (argv, startpoint);

		if (mkisofs->priv->merge) {
		        char *dev_str = NULL;

			g_ptr_array_add (argv, g_strdup ("-M"));

			if (mkisofs->priv->drive->cdrecord_id) 
				dev_str = g_strdup (mkisofs->priv->drive->cdrecord_id);
			else if (mkisofs->priv->drive->device)
				dev_str = g_strdup (mkisofs->priv->drive->device);
			g_ptr_array_add (argv, dev_str);
		}
	}

	bonfire_job_action_changed (BONFIRE_JOB (mkisofs), 
				    BONFIRE_BURN_ACTION_CREATING_IMAGE,
				    FALSE);
	bonfire_job_progress_changed (BONFIRE_JOB (mkisofs),
				      0.0, -1);
	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_mkisofs_set_argv (BonfireProcess *process,
			  GPtrArray *argv,
			  gboolean has_master,
			  GError **error)
{
	BonfireMkisofs *mkisofs;
	BonfireBurnResult result;

	mkisofs = BONFIRE_MKISOFS (process);

	g_ptr_array_add (argv, g_strdup ("mkisofs"));

	if (mkisofs->priv->use_utf8) {
		g_ptr_array_add (argv, g_strdup ("-input-charset"));
		g_ptr_array_add (argv, g_strdup ("utf8"));
	}

	if (mkisofs->priv->action == BONFIRE_MKISOFS_ACTION_GET_SIZE)
		result = bonfire_mkisofs_set_argv_image (mkisofs, argv, has_master, error);
	else if (mkisofs->priv->action == BONFIRE_MKISOFS_ACTION_GET_IMAGE)
		result = bonfire_mkisofs_set_argv_image (mkisofs, argv, has_master, error);
	else if (has_master)
		result = bonfire_mkisofs_set_argv_image (mkisofs, argv, has_master, error);
	else
		return BONFIRE_BURN_NOT_READY;

	if (result != BONFIRE_BURN_OK)
		return result;

	/* if we have a slave (ex: BonfireMkisofsBase)
	 * we don't want it to be started */
	bonfire_job_set_run_slave (BONFIRE_JOB (mkisofs), FALSE);

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_mkisofs_post (BonfireProcess *process, BonfireBurnResult retval)
{
	BonfireMkisofs *mkisofs;

	mkisofs = BONFIRE_MKISOFS (process);

	if (mkisofs->priv->timer) {
		g_timer_destroy (mkisofs->priv->timer);
		mkisofs->priv->timer = NULL;
	}

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_mkisofs_get_track (BonfireImager *imager,
			   BonfireTrackSource **track,
			   GError **error)
{
	BonfireMkisofs *mkisofs;
	BonfireTrackSource *retval;

	mkisofs = BONFIRE_MKISOFS (imager);

	if (mkisofs->priv->track_type == BONFIRE_TRACK_SOURCE_UNKNOWN)
		return BONFIRE_BURN_NOT_READY;

	if ((!mkisofs->priv->iso_ready && !mkisofs->priv->use_joliet)
	||  (!mkisofs->priv->iso_joliet_ready && mkisofs->priv->use_joliet)){
		BonfireBurnResult result;

		mkisofs->priv->action = BONFIRE_MKISOFS_ACTION_GET_IMAGE;
		result = bonfire_job_run (BONFIRE_JOB (mkisofs), error);
		mkisofs->priv->action = BONFIRE_MKISOFS_ACTION_NONE;
	
		if (result != BONFIRE_BURN_OK)
			return result;

		if (mkisofs->priv->use_joliet)
			mkisofs->priv->iso_joliet_ready = 1;
		else
			mkisofs->priv->iso_ready = 1;
	}

	retval = g_new0 (BonfireTrackSource, 1);
	retval->type = mkisofs->priv->track_type;
	retval->contents.iso.image = g_strdup (mkisofs->priv->output);

	*track = retval;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_mkisofs_get_size (BonfireImager *imager,
			  gint64 *size,
			  gboolean sectors,
			  GError **error)
{
	BonfireMkisofs *mkisofs;
	BonfireBurnResult result = BONFIRE_BURN_OK;

	mkisofs = BONFIRE_MKISOFS (imager);

	if (!mkisofs->priv->source) {
		BonfireJob *slave;

		slave = bonfire_job_get_slave (BONFIRE_JOB (mkisofs));
		if (!slave)
			return BONFIRE_BURN_NOT_READY;

		return bonfire_imager_get_size (BONFIRE_IMAGER (slave), size, sectors, error);
	}

	if (!mkisofs->priv->sectors_num) {
		if (bonfire_job_is_running (BONFIRE_JOB (imager)))
			return BONFIRE_BURN_RUNNING;

		mkisofs->priv->action = BONFIRE_MKISOFS_ACTION_GET_SIZE;
		result = bonfire_job_run (BONFIRE_JOB (mkisofs), error);
		mkisofs->priv->action = BONFIRE_MKISOFS_ACTION_NONE;

		if (result != BONFIRE_BURN_OK)
			return result;
	}

	if (sectors)
		*size = mkisofs->priv->sectors_num;
	else
		*size = mkisofs->priv->sectors_num * 2048;

	return result;
}
static BonfireBurnResult
bonfire_mkisofs_get_action_string (BonfireJob *job,
				   BonfireBurnAction action,
				   char **string)
{
	job = bonfire_job_get_slave (job);
	if (!job)
		return BONFIRE_BURN_NOT_SUPPORTED;

	return bonfire_job_get_action_string (job, action, string);
}
