/***************************************************************************
 *            data-disc.c
 *
 *  dim nov 27 15:34:04 2005
 *  Copyright  2005  Rouquier Philippe
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <errno.h>

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <glib/gi18n-lib.h>
#include <glib/gstdio.h>

#include <gdk/gdkkeysyms.h>

#include <gtk/gtkvbox.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtktreeview.h>
#include <gtk/gtktreemodel.h>
#include <gtk/gtktreestore.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkcellrendererpixbuf.h>
#include <gtk/gtkcellrenderer.h>
#include <gtk/gtktreeselection.h>
#include <gtk/gtktreeviewcolumn.h>
#include <gtk/gtkentry.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkcontainer.h>
#include <gtk/gtkcombobox.h>
#include <gtk/gtkalignment.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkmessagedialog.h>
#include <gtk/gtktooltips.h>

#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-file-info.h>

#ifdef BUILD_INOTIFY
#include "inotify.h"
#include "inotify-syscalls.h"
#endif

#include "disc.h"
#include "async-job-manager.h"
#include "data-disc.h"
#include "filtered-window.h"
#include "utils.h"

typedef enum {
	STATUS_NO_DRAG,
	STATUS_DRAGGING,
	STATUS_DRAG_DROP
} BonfireDragStatus;

struct BonfireDataDiscPrivate {
	GtkWidget *tree;
	GtkTreeModel *model;
	GtkTreeModel *sort;
	GtkTooltips *tooltip;
	GtkWidget *filter_dialog;
	GtkWidget *filter_button;
	GtkWidget *notebook;

	GtkUIManager *manager;

	BonfireDragStatus drag_status;
	GtkTreePath *drag_source;
	int press_start_x;
	int press_start_y;
	gint scroll_timeout;
	gint expand_timeout;

	int activity_counter;

	double size;
	GSList *rescan;
	GSList *loading;

	GtkTreePath *selected_path;

#ifdef BUILD_INOTIFY
	int notify_id;
	GIOChannel *notify;
	GHashTable *monitored;
#endif

	BonfireAsyncJobManager *jobs;
	int file_type;
	int load_type;
	int expose_type;

	GHashTable *dirs;
	GHashTable *files;
	GHashTable *paths;
	GHashTable *grafts;
	GHashTable *unreadable;
	GHashTable *excluded;
	GHashTable *symlinks;

	GHashTable *path_refs;

	GMutex *references_lock;
	GHashTable *references;

	GMutex *restored_lock;
	GHashTable *restored;

	GSList *expose;
	gint expose_id;

	GSList *exposing;

	int editing:1;
	int is_loading:1;
	int reject_files:1;
};

typedef enum {
	ROW_BOGUS,
	ROW_NEW,
	ROW_NOT_EXPLORED,
	ROW_EXPLORING,
	ROW_EXPLORED,
	ROW_EXPANDED
} BonfireRowStatus;

enum {
	ICON_COL,
	NAME_COL,
	SIZE_COL,
	MIME_COL,
	DSIZE_COL,
	ROW_STATUS_COL,
	ISDIR_COL,
	MARKUP_COL,
	NB_COL
};


struct _BonfireLoadDirError {
	char *uri;
	BonfireFilterStatus status;
};
typedef struct _BonfireLoadDirError BonfireLoadDirError;

struct _BonfireDirectoryContentsData {
	char *uri;
	GSList *infos;
	GSList *errors;
	int cancel:1;
};
typedef struct _BonfireDirectoryContentsData BonfireDirectoryContentsData;

static void bonfire_data_disc_class_init (BonfireDataDiscClass *klass);
static void bonfire_data_disc_init (BonfireDataDisc *sp);
static void bonfire_data_disc_finalize (GObject *object);
static void bonfire_data_disc_iface_disc_init (BonfireDiscIface *iface);
static void bonfire_data_disc_get_property (GObject * object,
					    guint prop_id,
					    GValue * value,
					    GParamSpec * pspec);
static void bonfire_data_disc_set_property (GObject * object,
					    guint prop_id,
					    const GValue * value,
					    GParamSpec * spec);
#ifdef BUILD_INOTIFY
typedef union {
	int wd;
	GnomeVFSMonitorHandle *hvfs;
} BonfireMonitorHandle;

struct _BonfireFile {
	char *uri;
	double size;
	BonfireMonitorHandle handle;
};
typedef struct _BonfireFile BonfireFile;

static BonfireMonitorHandle
bonfire_data_disc_start_monitoring (BonfireDataDisc *disc,
				    BonfireFile *file);
static void
bonfire_data_disc_cancel_monitoring (BonfireDataDisc *disc,
				     BonfireFile *file);
static gboolean
bonfire_data_disc_inotify_monitor_cb (GIOChannel *channel,
				      GIOCondition condition,
				      BonfireDataDisc *disc);
#else
struct _BonfireFile {
	char *uri;
	double size;
};
typedef struct _BonfireFile BonfireFile;
#endif

static BonfireDiscResult
bonfire_data_disc_add_uri (BonfireDisc *disc, const char *uri);

static void
bonfire_data_disc_delete_selected (BonfireDisc *disc);

static void
bonfire_data_disc_clear (BonfireDisc *disc);
static void
bonfire_data_disc_reset (BonfireDisc *disc);

static BonfireDiscResult
bonfire_data_disc_load_track (BonfireDisc *disc,
			      BonfireDiscTrack *track);
static BonfireDiscResult
bonfire_data_disc_get_track (BonfireDisc *disc,
			     BonfireDiscTrack *track,
			     gboolean src_format);
static BonfireDiscResult
bonfire_data_disc_get_status (BonfireDisc *disc);

static gboolean
bonfire_data_disc_button_pressed_cb (GtkTreeView *tree,
				     GdkEventButton *event,
				     BonfireDataDisc *disc);
static gboolean
bonfire_data_disc_key_released_cb (GtkTreeView *tree,
				   GdkEventKey *event,
				   BonfireDataDisc *disc);

static void
bonfire_data_disc_name_edited_cb (GtkCellRendererText *cellrenderertext,
				  gchar *path_string,
				  gchar *text,
				  BonfireDataDisc *disc);
static void
bonfire_data_disc_name_editing_started_cb (GtkCellRenderer *renderer,
					   GtkCellEditable *editable,
					   gchar *path,
					   BonfireDataDisc *disc);
static void
bonfire_data_disc_name_editing_canceled_cb (GtkCellRenderer *renderer,
					    BonfireDataDisc *disc);

static gboolean
bonfire_data_disc_drag_motion_cb(GtkWidget *tree,
				 GdkDragContext *drag_context,
				 gint x,
				 gint y,
				 guint time,
				 BonfireDataDisc *disc);
void
bonfire_data_disc_drag_leave_cb (GtkWidget *tree,
				 GdkDragContext *drag_context,
				 guint time,
				 BonfireDataDisc *disc);
static gboolean
bonfire_data_disc_drag_drop_cb (GtkTreeView *tree,
				GdkDragContext *drag_context,
				gint x,
				gint y,
				guint time,
				BonfireDataDisc *disc);
static void
bonfire_data_disc_drag_data_received_cb (GtkTreeView *tree,
					 GdkDragContext *drag_context,
					 gint x,
					 gint y,
					 GtkSelectionData *selection_data,
					 guint info,
					 guint time,
					 BonfireDataDisc *disc);
static void
bonfire_data_disc_drag_begin_cb (GtkTreeView *tree,
				 GdkDragContext *drag_context,
				 BonfireDataDisc *disc);
static void
bonfire_data_disc_drag_get_cb (GtkWidget *tree,
                               GdkDragContext *context,
                               GtkSelectionData *selection_data,
                               guint info,
                               guint time,
			       BonfireDataDisc *disc);
static void
bonfire_data_disc_drag_end_cb (GtkWidget *tree,
			       GdkDragContext *drag_context,
			       BonfireDataDisc *disc);

static void
bonfire_data_disc_row_collapsed_cb (GtkTreeView *tree,
				     GtkTreeIter *sortparent,
				     GtkTreePath *sortpath,
				     BonfireDataDisc *disc);

static void
bonfire_data_disc_new_folder_clicked_cb (GtkButton *button,
					 BonfireDataDisc *disc);
static void
bonfire_data_disc_filtered_files_clicked_cb (GtkButton *button,
					     BonfireDataDisc *disc);

static void
bonfire_data_disc_open_activated_cb (GtkAction *action,
				     BonfireDataDisc *disc);
static void
bonfire_data_disc_rename_activated_cb (GtkAction *action,
				       BonfireDataDisc *disc);
static void
bonfire_data_disc_delete_activated_cb (GtkAction *action,
				       BonfireDataDisc *disc);
static void
bonfire_data_disc_paste_activated_cb (GtkAction *action,
				      BonfireDataDisc *disc);

static void
bonfire_data_disc_clean (BonfireDataDisc *disc);
static void
bonfire_data_disc_reset_real (BonfireDataDisc *disc);

static GSList *
bonfire_data_disc_uri_to_paths (BonfireDataDisc *disc,
				const char *uri);
static char *
bonfire_data_disc_graft_get (BonfireDataDisc *disc,
			     const char *path);
static void
bonfire_data_disc_graft_remove_all (BonfireDataDisc *disc,
				    const char *uri);
static void
bonfire_data_disc_graft_children_remove (BonfireDataDisc *disc,
					 GSList *paths);

static void
bonfire_data_disc_remove_uri (BonfireDataDisc *disc,
			      const char *uri);
static void
bonfire_data_disc_restore_excluded_children (BonfireDataDisc *disc,
					     BonfireFile *dir);
static void
bonfire_data_disc_replace_symlink_children (BonfireDataDisc *disc,
					    BonfireFile *dir,
					    GSList *grafts);
static void
bonfire_data_disc_exclude_uri (BonfireDataDisc *disc,
			       const char *path,
			       const char *uri);
static gboolean
bonfire_data_disc_is_excluded (BonfireDataDisc *disc,
			       const char *uri,
			       BonfireFile *top);

static void
bonfire_data_disc_load_dir_error (BonfireDataDisc *disc, GSList *errors);

static BonfireDiscResult
bonfire_data_disc_expose_path (BonfireDataDisc *disc,
			       const char *path);
static void
bonfire_data_disc_directory_priority (BonfireDataDisc *disc,
				      BonfireFile *file);
static BonfireDiscResult
bonfire_data_disc_directory_load (BonfireDataDisc *disc,
				  BonfireFile *dir,
				  gboolean append);
static BonfireFile *
bonfire_data_disc_directory_new (BonfireDataDisc *disc,
				 char *uri,
				 gboolean append);

static char *
bonfire_data_disc_get_selected_uri (BonfireDisc *disc);

static char *BONFIRE_CREATED_DIR = "created";

#define BONFIRE_ADD_TO_EXPOSE_QUEUE(disc, data)	\
	disc->priv->expose = g_slist_append (disc->priv->expose, data);	\
	if (!disc->priv->expose_id)	\
		disc->priv->expose_id = g_idle_add ((GSourceFunc) bonfire_data_disc_expose_path_real,	\
						    disc);

enum {
	TREE_MODEL_ROW = 150,
	TARGET_URIS_LIST,
};

static GtkTargetEntry ntables_cd[] = {
	{"GTK_TREE_MODEL_ROW", GTK_TARGET_SAME_WIDGET, TREE_MODEL_ROW},
	{"text/uri-list", 0, TARGET_URIS_LIST}
};
static guint nb_targets_cd = sizeof (ntables_cd) / sizeof (ntables_cd[0]);

static GtkTargetEntry ntables_source[] = {
	{"GTK_TREE_MODEL_ROW", GTK_TARGET_SAME_WIDGET, TREE_MODEL_ROW},
};

static guint nb_targets_source = sizeof (ntables_source) / sizeof (ntables_source[0]);

enum {
	PROP_NONE,
	PROP_REJECT_FILE,
};

static GObjectClass *parent_class = NULL;

static GtkActionEntry entries[] = {
	{"ContextualMenu", NULL, N_("Menu")},
	{"Open", GTK_STOCK_OPEN, N_("Open"), NULL, NULL,
	 G_CALLBACK (bonfire_data_disc_open_activated_cb)},
	{"Rename", NULL, N_("Rename"), NULL, NULL,
	 G_CALLBACK (bonfire_data_disc_rename_activated_cb)},
	{"Delete", GTK_STOCK_REMOVE, N_("Remove"), NULL, NULL,
	 G_CALLBACK (bonfire_data_disc_delete_activated_cb)},
	{"Paste", GTK_STOCK_PASTE, N_("Paste"), NULL, NULL,
	 G_CALLBACK (bonfire_data_disc_paste_activated_cb)},
};

static const char *menu_description = {
	"<ui>"
	"<popup action='ContextMenu'>"
		"<menuitem action='Open'/>"
		"<menuitem action='Delete'/>"
		"<menuitem action='Rename'/>"
		"<separator/>"
		"<menuitem action='Paste'/>"
	"</popup>"
	"</ui>"
};

GType
bonfire_data_disc_get_type ()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireDataDiscClass),
			NULL,
			NULL,
			(GClassInitFunc) bonfire_data_disc_class_init,
			NULL,
			NULL,
			sizeof (BonfireDataDisc),
			0,
			(GInstanceInitFunc) bonfire_data_disc_init,
		};

		static const GInterfaceInfo disc_info =
		{
			(GInterfaceInitFunc) bonfire_data_disc_iface_disc_init,
			NULL,
			NULL
		};

		type = g_type_register_static (GTK_TYPE_VBOX, 
					       "BonfireDataDisc",
					       &our_info, 0);

		g_type_add_interface_static (type,
					     BONFIRE_TYPE_DISC,
					     &disc_info);
	}

	return type;
}

static void
bonfire_data_disc_class_init (BonfireDataDiscClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent(klass);
	object_class->finalize = bonfire_data_disc_finalize;
	object_class->set_property = bonfire_data_disc_set_property;
	object_class->get_property = bonfire_data_disc_get_property;

	g_object_class_install_property (object_class,
					 PROP_REJECT_FILE,
					 g_param_spec_boolean
					 ("reject-file",
					  "Whether it accepts files",
					  "Whether it accepts files",
					  FALSE,
					  G_PARAM_READWRITE));
}

static void
bonfire_data_disc_iface_disc_init (BonfireDiscIface *iface)
{
	iface->add_uri = bonfire_data_disc_add_uri;
	iface->delete_selected = bonfire_data_disc_delete_selected;
	iface->clear = bonfire_data_disc_clear;
	iface->reset = bonfire_data_disc_reset;
	iface->get_track = bonfire_data_disc_get_track;
	iface->load_track = bonfire_data_disc_load_track;
	iface->get_status = bonfire_data_disc_get_status;
	iface->get_selected_uri = bonfire_data_disc_get_selected_uri;
}

static void bonfire_data_disc_get_property (GObject * object,
					    guint prop_id,
					    GValue * value,
					    GParamSpec * pspec)
{
	BonfireDataDisc *disc;

	disc = BONFIRE_DATA_DISC (object);

	switch (prop_id) {
	case PROP_REJECT_FILE:
		g_value_set_boolean (value, disc->priv->reject_files);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

static void bonfire_data_disc_set_property (GObject * object,
					    guint prop_id,
					    const GValue * value,
					    GParamSpec * pspec)
{
	BonfireDataDisc *disc;

	disc = BONFIRE_DATA_DISC (object);

	switch (prop_id) {
	case PROP_REJECT_FILE:
		disc->priv->reject_files = g_value_get_boolean (value);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

static int
bonfire_data_disc_sort_default (GtkTreeModel *model,
				GtkTreeIter *a,
				GtkTreeIter *b,
				BonfireDataDisc *disc)
{
	GtkTreePath *patha, *pathb;
	gboolean isdira, isdirb;
	int retval;

	gtk_tree_model_get (model, a,
			    ISDIR_COL, &isdira, -1);
	gtk_tree_model_get (model, b,
			    ISDIR_COL, &isdirb, -1);

	if (isdira && !isdirb)
		return -1;
	else if (!isdira && isdirb)
		return 1;

	patha = gtk_tree_model_get_path(model, a);
	pathb = gtk_tree_model_get_path(model, b);

	retval = gtk_tree_path_compare(patha, pathb);
	gtk_tree_path_free(patha);
	gtk_tree_path_free(pathb);

	return retval;
}

static int
bonfire_data_disc_sort_size (GtkTreeModel *model,
			     GtkTreeIter *a,
			     GtkTreeIter *b,
			     gpointer data)
{
	gboolean isdira, isdirb;
	double sizea, sizeb;

	gtk_tree_model_get (model, a,
			    ISDIR_COL, &isdira, -1);
	gtk_tree_model_get (model, b,
			    ISDIR_COL, &isdirb, -1);

	if (isdira && !isdirb)
		return -1;
	else if (!isdira && isdirb)
		return 1;

	if (isdira) {
		int nba, nbb;

		nba = gtk_tree_model_iter_n_children (model, a);
		nbb = gtk_tree_model_iter_n_children (model, b);
		return nbb - nba;
	}

	gtk_tree_model_get (model, a,
			    DSIZE_COL,
			    &sizea, -1);
	gtk_tree_model_get (model, b,
			    DSIZE_COL,
			    &sizeb, -1);
	return sizeb - sizea;
}

static int
bonfire_data_disc_sort_string (GtkTreeModel *model,
			       GtkTreeIter *a,
			       GtkTreeIter *b,
			       int column)
{
	gboolean isdira, isdirb;
	char *stringa, *stringb;
	GtkSortType order;
	int retval;

	gtk_tree_sortable_get_sort_column_id (GTK_TREE_SORTABLE (model),
					      NULL, &order);

	gtk_tree_model_get (model, a,
			    ISDIR_COL, &isdira, -1);
	gtk_tree_model_get (model, b,
			    ISDIR_COL, &isdirb, -1);

	if (isdira && !isdirb)
		return -1;
	else if (!isdira && isdirb)
		return 1;

	gtk_tree_model_get (model, a, column, &stringa, -1);
	gtk_tree_model_get (model, b, column, &stringb, -1);

	if(stringa && !stringb) {
		g_free(stringa);
		return -1;
	}
	else if(!stringa && stringb) {
		g_free(stringb);
		return 1;
	}
	else if(!stringa && !stringb)
		return 0;
	
	retval = strcmp (stringa, stringb);
	g_free (stringa);
	g_free (stringb);

	return retval;
}

static int
bonfire_data_disc_sort_display (GtkTreeModel *model,
				GtkTreeIter *a,
				GtkTreeIter *b,
				gpointer data)
{
	return bonfire_data_disc_sort_string(model, a, b, NAME_COL);
}

static int
bonfire_data_disc_sort_description (GtkTreeModel *model,
				    GtkTreeIter *a,
				    GtkTreeIter *b,
				    gpointer data)
{
	return bonfire_data_disc_sort_string(model, a, b, MIME_COL);
}

static void
bonfire_data_disc_build_context_menu (BonfireDataDisc *disc)
{
	GtkActionGroup *action_group;
	GError *error = NULL;

	action_group = gtk_action_group_new ("MenuAction");
	gtk_action_group_set_translation_domain (action_group, GETTEXT_PACKAGE);
	gtk_action_group_add_actions (action_group,
				      entries,
				      G_N_ELEMENTS (entries),
				      disc);

	disc->priv->manager = gtk_ui_manager_new ();
	gtk_ui_manager_insert_action_group (disc->priv->manager,
					    action_group,
					    0);

	if (!gtk_ui_manager_add_ui_from_string (disc->priv->manager,
						menu_description,
						-1,
						&error)) {
		g_message ("building menus failed: %s", error->message);
		g_error_free (error);
	}
}

static void
bonfire_data_disc_init (BonfireDataDisc *obj)
{
	GtkTreeViewColumn *column;
	GtkCellRenderer *renderer;
	GtkTreeModel *model;
	GtkWidget *scroll;
	GtkWidget *button;
	GtkWidget *hbox;

	obj->priv = g_new0 (BonfireDataDiscPrivate, 1);
	gtk_box_set_spacing (GTK_BOX (obj), 6);

	obj->priv->tooltip = gtk_tooltips_new ();

	/* the information displayed about how to use this tree */
	obj->priv->notebook = bonfire_utils_get_use_info_notebook ();
	gtk_box_pack_start (GTK_BOX (obj), obj->priv->notebook, TRUE, TRUE, 0);

	/* Tree */
	obj->priv->tree = gtk_tree_view_new ();
	g_signal_connect (G_OBJECT (obj->priv->tree),
			  "button-press-event",
			  G_CALLBACK (bonfire_data_disc_button_pressed_cb),
			  obj);
	g_signal_connect (G_OBJECT (obj->priv->tree),
			  "key-release-event",
			  G_CALLBACK (bonfire_data_disc_key_released_cb),
			  obj);

	gtk_tree_selection_set_mode (gtk_tree_view_get_selection
				     (GTK_TREE_VIEW (obj->priv->tree)),
				     GTK_SELECTION_MULTIPLE);
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (obj->priv->tree),
				      TRUE);

	model = (GtkTreeModel*) gtk_tree_store_new (NB_COL,
						    GDK_TYPE_PIXBUF,
						    G_TYPE_STRING,
						    G_TYPE_STRING,
						    G_TYPE_STRING,
						    G_TYPE_DOUBLE,
						    G_TYPE_INT,
						    G_TYPE_BOOLEAN,
						    PANGO_TYPE_STYLE);

	obj->priv->model = GTK_TREE_MODEL (model);

	model = gtk_tree_model_sort_new_with_model (model);
	g_object_unref (obj->priv->model);

	gtk_tree_view_set_model (GTK_TREE_VIEW (obj->priv->tree),
				 GTK_TREE_MODEL (model));
	obj->priv->sort = model;
	g_object_unref (G_OBJECT (model));

	column = gtk_tree_view_column_new ();

	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 128);

	renderer = gtk_cell_renderer_pixbuf_new ();
	gtk_tree_view_column_pack_start (column, renderer, FALSE);
	gtk_tree_view_column_add_attribute (column, renderer, "pixbuf",
					    ICON_COL);

	renderer = gtk_cell_renderer_text_new ();
	g_signal_connect (G_OBJECT (renderer), "edited",
			  G_CALLBACK (bonfire_data_disc_name_edited_cb), obj);
	g_signal_connect (G_OBJECT (renderer), "editing-started",
			  G_CALLBACK (bonfire_data_disc_name_editing_started_cb), obj);
	g_signal_connect (G_OBJECT (renderer), "editing-canceled",
			  G_CALLBACK (bonfire_data_disc_name_editing_canceled_cb), obj);

	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_add_attribute (column, renderer,
					    "text", NAME_COL);
	gtk_tree_view_column_add_attribute (column, renderer,
					    "style", MARKUP_COL);
	gtk_tree_view_column_add_attribute (column, renderer,
					    "editable", ROW_STATUS_COL);

	gtk_tree_view_column_set_title (column, _("Files"));
	gtk_tree_view_append_column (GTK_TREE_VIEW (obj->priv->tree),
				     column);
	gtk_tree_view_column_set_sort_column_id (column,
						 NAME_COL);
	gtk_tree_view_set_expander_column (GTK_TREE_VIEW (obj->priv->tree),
					   column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Size"),
							   renderer, "text",
							   SIZE_COL,
							   NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (obj->priv->tree),
				     column);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_sort_column_id (column,
						 SIZE_COL);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Description"),
							   renderer, "text",
							   MIME_COL,
							   NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (obj->priv->tree),
				     column);
	gtk_tree_view_column_set_sort_column_id (column,
						 MIME_COL);

	gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (obj->priv->sort),
					 NAME_COL,
					 bonfire_data_disc_sort_display,
					 NULL, NULL);
	gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (obj->priv->sort),
					 SIZE_COL,
					 bonfire_data_disc_sort_size,
					 NULL, NULL);
	gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (obj->priv->sort),
					 MIME_COL,
					 bonfire_data_disc_sort_description,
					 NULL, NULL);

	gtk_tree_sortable_set_default_sort_func (GTK_TREE_SORTABLE(obj->priv->sort),
						 (GtkTreeIterCompareFunc) bonfire_data_disc_sort_default,
						 obj, NULL);
	scroll = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scroll),
					     GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (scroll), obj->priv->tree);
	gtk_notebook_append_page (GTK_NOTEBOOK (obj->priv->notebook), scroll, NULL);

	/* dnd */
	gtk_tree_view_enable_model_drag_dest (GTK_TREE_VIEW
					      (obj->priv->tree),
					      ntables_cd, nb_targets_cd,
					      GDK_ACTION_COPY |
					      GDK_ACTION_MOVE);

	g_signal_connect (G_OBJECT (obj->priv->tree), "drag_motion",
			  G_CALLBACK (bonfire_data_disc_drag_motion_cb),
			  obj);
	g_signal_connect (G_OBJECT (obj->priv->tree), "drag_leave",
			  G_CALLBACK (bonfire_data_disc_drag_leave_cb),
			  obj);
	g_signal_connect (G_OBJECT (obj->priv->tree), "drag_drop",
			  G_CALLBACK (bonfire_data_disc_drag_drop_cb),
			  obj);
	g_signal_connect (G_OBJECT (obj->priv->tree), "drag_data_received",
			  G_CALLBACK (bonfire_data_disc_drag_data_received_cb),
			  obj);

	gtk_tree_view_enable_model_drag_source (GTK_TREE_VIEW (obj->priv->tree),
						GDK_BUTTON1_MASK,
						ntables_source,
						nb_targets_source,
						GDK_ACTION_COPY |
						GDK_ACTION_MOVE);
	g_signal_connect (G_OBJECT (obj->priv->tree), "drag_begin",
			  G_CALLBACK (bonfire_data_disc_drag_begin_cb),
			  obj);
	g_signal_connect (G_OBJECT (obj->priv->tree), "drag_data_get",
			  G_CALLBACK (bonfire_data_disc_drag_get_cb),
			  obj);
	g_signal_connect (G_OBJECT (obj->priv->tree), "drag_end",
			  G_CALLBACK (bonfire_data_disc_drag_end_cb),
			  obj);

	g_signal_connect (G_OBJECT (obj->priv->tree),
			  "row-expanded",
			  G_CALLBACK (bonfire_data_disc_row_collapsed_cb),
			  obj);

	bonfire_data_disc_build_context_menu (obj);

	/* new folder button */
	hbox = gtk_hbox_new (FALSE, 10);

	button = bonfire_utils_make_button (_("New folder"),
					    GTK_STOCK_DIRECTORY);
	g_signal_connect (G_OBJECT (button),
			  "clicked",
			  G_CALLBACK (bonfire_data_disc_new_folder_clicked_cb),
			  obj);
	gtk_tooltips_set_tip (obj->priv->tooltip,
			      button,
			      _("Create a new empty folder"),
			      NULL);
	gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);

	obj->priv->filter_button = gtk_button_new_with_label (_("Filtered files"));
	gtk_widget_set_sensitive (obj->priv->filter_button, FALSE);
	g_signal_connect (G_OBJECT (obj->priv->filter_button),
			  "clicked",
			  G_CALLBACK (bonfire_data_disc_filtered_files_clicked_cb),
			  obj);
	gtk_box_pack_start (GTK_BOX (hbox),
			    obj->priv->filter_button,
			    FALSE,
			    FALSE,
			    0);

	gtk_tooltips_set_tip (obj->priv->tooltip,
			      obj->priv->filter_button,
			      _("Some files were removed from the project. Clik here to see them."),
			      NULL);

	gtk_box_pack_start (GTK_BOX (obj), hbox, FALSE, FALSE, 0);

	/* useful things for directory exploration */
	obj->priv->dirs = g_hash_table_new (g_str_hash, g_str_equal);
	obj->priv->files = g_hash_table_new (g_str_hash, g_str_equal);
	obj->priv->grafts = g_hash_table_new (g_str_hash, g_str_equal);
	obj->priv->paths = g_hash_table_new (g_str_hash, g_str_equal);

	obj->priv->restored_lock = g_mutex_new ();
	obj->priv->references_lock = g_mutex_new ();


#ifdef BUILD_INOTIFY
	int fd;

	obj->priv->monitored = g_hash_table_new (g_direct_hash, g_direct_equal);

	/* start inotify monitoring backend */
	fd = inotify_init ();
	if (fd != -1) {
		obj->priv->notify = g_io_channel_unix_new (fd);
		g_io_channel_set_encoding (obj->priv->notify, NULL, NULL);
		g_io_channel_set_close_on_unref (obj->priv->notify, TRUE);
		obj->priv->notify_id = g_io_add_watch (obj->priv->notify,
						       G_IO_IN | G_IO_HUP | G_IO_PRI,
						       (GIOFunc) bonfire_data_disc_inotify_monitor_cb,
						       obj);
		g_io_channel_unref (obj->priv->notify);
	}
	else
		g_warning ("Failed to open inotify: %s\n",
			   strerror (errno));
#endif
}

static void
bonfire_data_disc_finalize (GObject *object)
{
	BonfireDataDisc *cobj;
	cobj = BONFIRE_DATA_DISC(object);

	bonfire_data_disc_clean (cobj);

	if (cobj->priv->jobs) {
		bonfire_async_job_manager_unregister_type (cobj->priv->jobs,
							   cobj->priv->file_type);
		bonfire_async_job_manager_unregister_type (cobj->priv->jobs,
							   cobj->priv->load_type);
		bonfire_async_job_manager_unregister_type (cobj->priv->jobs,
							   cobj->priv->expose_type);
		g_object_unref (cobj->priv->jobs);
		cobj->priv->jobs = NULL;
	}

#ifdef BUILD_INOTIFY
	if (cobj->priv->notify_id)
		g_source_remove (cobj->priv->notify_id);
	g_hash_table_destroy (cobj->priv->monitored);

#endif

	if (cobj->priv->scroll_timeout) {
		g_source_remove (cobj->priv->scroll_timeout);
		cobj->priv->scroll_timeout = 0;
	}

	if (cobj->priv->expand_timeout) {
		g_source_remove (cobj->priv->expand_timeout);
		cobj->priv->expand_timeout = 0;
	}

	g_mutex_free (cobj->priv->references_lock);
	g_mutex_free (cobj->priv->restored_lock);

	g_hash_table_destroy (cobj->priv->grafts);
	g_hash_table_destroy (cobj->priv->paths);
	g_hash_table_destroy (cobj->priv->dirs);
	g_hash_table_destroy (cobj->priv->files);

	if (cobj->priv->tooltip)
		gtk_object_sink (GTK_OBJECT (cobj->priv->tooltip));

	if (cobj->priv->path_refs)
		g_hash_table_destroy (cobj->priv->path_refs);

	g_object_unref (cobj->priv->manager);

	g_free (cobj->priv);
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

GtkWidget *
bonfire_data_disc_new ()
{
	BonfireDataDisc *obj;
	
	obj = BONFIRE_DATA_DISC (g_object_new (BONFIRE_TYPE_DATA_DISC, NULL));
	
	return GTK_WIDGET (obj);
}

/*************************** activity ******************************************/
static void
bonfire_data_disc_increase_activity_counter (BonfireDataDisc *disc)
{
	GdkCursor *cursor;

	if (disc->priv->activity_counter == 0 && GTK_WIDGET (disc)->window) {
		cursor = gdk_cursor_new (GDK_WATCH);
		gdk_window_set_cursor (GTK_WIDGET (disc)->window,
				       cursor);
		gdk_cursor_unref (cursor);
	}

	disc->priv->activity_counter++;
}

static void
bonfire_data_disc_decrease_activity_counter (BonfireDataDisc *disc)
{
	if (disc->priv->activity_counter == 1 && GTK_WIDGET (disc)->window)
		gdk_window_set_cursor (GTK_WIDGET (disc)->window, NULL);

	disc->priv->activity_counter--;
}

static BonfireDiscResult
bonfire_data_disc_get_status (BonfireDisc *disc)
{
	if (BONFIRE_DATA_DISC (disc)->priv->activity_counter)
		return BONFIRE_DISC_NOT_READY;

	return BONFIRE_DISC_OK;
}

/**************************** burn button **************************************/
static void
bonfire_data_disc_selection_changed (BonfireDataDisc *disc, gboolean notempty)
{
	bonfire_disc_contents_changed (BONFIRE_DISC (disc), notempty);
}

/*************************** GtkTreeView functions *****************************/
static void
bonfire_data_disc_name_exist_dialog (BonfireDataDisc *disc,
				     const char *name)
{
	GtkWidget *dialog;
	GtkWidget *toplevel;

	toplevel = gtk_widget_get_toplevel (GTK_WIDGET (disc));
	dialog = gtk_message_dialog_new (GTK_WINDOW (toplevel),
					 GTK_DIALOG_DESTROY_WITH_PARENT |
					 GTK_DIALOG_MODAL,
					 GTK_MESSAGE_WARNING,
					 GTK_BUTTONS_CLOSE,
					 _("\"%s\" already exists in the directory:"),
					 name);

	gtk_window_set_title (GTK_WINDOW (dialog), _("Already existing file"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
						  _("it won't be added."));

	gtk_widget_show_all (dialog);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

static gboolean
bonfire_data_disc_joliet_compatibility_dialog (BonfireDataDisc *disc,
					       const char *name)
{
	int answer;
	GtkWidget *dialog;
	GtkWidget *toplevel;

	toplevel = gtk_widget_get_toplevel (GTK_WIDGET (disc));
	dialog = gtk_message_dialog_new (GTK_WINDOW (toplevel),
					 GTK_DIALOG_DESTROY_WITH_PARENT |
					 GTK_DIALOG_MODAL,
					 GTK_MESSAGE_QUESTION,
					 GTK_BUTTONS_OK_CANCEL,
					 _("\"%s\" breaks windows compatilibility:"),
					 name);

	gtk_window_set_title (GTK_WINDOW (dialog), _("Windows compatibility"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
						  _("the name of this file breaks the joliet format."
						    " This format allows the disc to be used with Windows"
						    " operating system. \nDo you want to add this file there"
						    " and drop window compatibility?"));

	gtk_widget_show_all (dialog);
	answer = gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);

	return (answer == GTK_RESPONSE_OK);
}

static BonfireDiscResult
bonfire_data_disc_check_for_existence (BonfireDataDisc *disc,
				       const char *name,
				       GtkTreePath *treepath,
				       gboolean usedialog)
{
	char *row_name;
	GtkTreeIter iter;
	GtkTreeIter child;
	GtkTreeModel *model;
	gboolean check_joliet;

	model = disc->priv->model;
	if (!treepath
	||  gtk_tree_path_get_depth (treepath) < 1) {
		if (!gtk_tree_model_get_iter_first (model, &child))
			return BONFIRE_DISC_OK;
	}
	else {
		if (!gtk_tree_model_get_iter (model, &iter, treepath))
			return BONFIRE_DISC_OK;

		if (!gtk_tree_model_iter_children (model, &child, &iter))
			return BONFIRE_DISC_OK;
	}

	check_joliet = (strlen (name) > 64);

	do {
		gtk_tree_model_get (model, &child,
				    NAME_COL, &row_name, -1);

		if (!row_name)
			continue;

		if (!strcmp (name, row_name)) {
			
			if (usedialog)
				bonfire_data_disc_name_exist_dialog (disc, name);

			g_free (row_name);
			return BONFIRE_DISC_ERROR_ALREADY_IN_TREE;
		}
		else if (check_joliet
		     && !strncmp (name, row_name, 64)
		     && !bonfire_data_disc_joliet_compatibility_dialog (disc, name)) {
			g_free (row_name);
			return BONFIRE_DISC_ERROR_ALREADY_IN_TREE;
		}

		g_free (row_name);
	} while (gtk_tree_model_iter_next (model, &child));

	return BONFIRE_DISC_OK;
}

static void
bonfire_data_disc_tree_update_directory_real (BonfireDataDisc *disc,
					      GtkTreeIter *iter)
{
	char *nb_items_string;
	GtkTreeModel *model;
	int nb_items;

	model = disc->priv->model;

	nb_items = gtk_tree_model_iter_n_children (model, iter);
	if (nb_items == 0) {
		GtkTreeIter child;

		nb_items_string = g_strdup (_("empty"));
		gtk_tree_store_prepend (GTK_TREE_STORE (model), &child, iter);
		gtk_tree_store_set (GTK_TREE_STORE (model), &child,
				    NAME_COL, _("(empty)"),
				    MARKUP_COL, PANGO_STYLE_ITALIC,
				    ROW_STATUS_COL, ROW_BOGUS, -1);
	}
	else if (nb_items == 1) {
		int status;
		GtkTreeIter child;

		gtk_tree_model_iter_children (model, &child, iter);
		gtk_tree_model_get (model, &child,
				    ROW_STATUS_COL, &status, -1);

		if (status == ROW_BOGUS)
			nb_items_string = g_strdup (_("empty"));
		else
			nb_items_string = g_strdup (_("1 item"));
	}
	else {
		int status;
		GtkTreeIter child;

		gtk_tree_model_iter_children (model, &child, iter);
		gtk_tree_model_get (model, &child,
				    ROW_STATUS_COL, &status, -1);

		if (status == ROW_BOGUS) {
			gtk_tree_store_remove (GTK_TREE_STORE (model), &child);
			nb_items --;
		}

		if (nb_items == 1)
			nb_items_string = g_strdup (_("1 item"));
		else
			nb_items_string = g_strdup_printf (_("%i items"),
							   nb_items);
	}

	gtk_tree_store_set (GTK_TREE_STORE (disc->priv->model), iter,
			    SIZE_COL, nb_items_string,
			    -1);

	g_free (nb_items_string);
}

static void
bonfire_data_disc_tree_update_directory (BonfireDataDisc *disc,
					 const GtkTreePath *path)
{
	GtkTreeModel *model;
	GtkTreeIter iter;

	model = disc->priv->model;
	gtk_tree_model_get_iter (model, &iter, (GtkTreePath *) path);
	bonfire_data_disc_tree_update_directory_real (disc, &iter);
}

static void
bonfire_data_disc_tree_update_parent (BonfireDataDisc *disc,
				     const GtkTreePath *path)
{
	GtkTreePath *parent_path;

	parent_path = gtk_tree_path_copy (path);
	gtk_tree_path_up (parent_path);

	if (gtk_tree_path_get_depth (parent_path) < 1) {
		gtk_tree_path_free (parent_path);
		return;
	}

	bonfire_data_disc_tree_update_directory (disc, parent_path);
	gtk_tree_path_free (parent_path);
}

static gboolean
bonfire_data_disc_tree_path_to_disc_path (BonfireDataDisc *disc,
					  GtkTreePath *treepath,
					  char **discpath)
{
	int i;
	char *name;
	GString *path;
	GtkTreeIter row;
	GtkTreePath *iter;
	GtkTreeModel *model;
	gint *indices, depth;

	if (treepath == NULL
	||  gtk_tree_path_get_depth (treepath) < 1) {
		*discpath = g_strdup ("/");
		return TRUE;
	}

	model = disc->priv->model;
	depth = gtk_tree_path_get_depth (treepath);
	indices = gtk_tree_path_get_indices (treepath);

	path = g_string_new_len (NULL, 16);
	iter = gtk_tree_path_new ();

	for (i = 0; i < depth; i++) {
		gtk_tree_path_append_index (iter, indices[i]);
		if (!gtk_tree_model_get_iter (model, &row, iter)) {
			gtk_tree_path_free (iter);
			g_string_free (path, TRUE);
			return FALSE;
		}

		gtk_tree_model_get (model, &row,
				    NAME_COL, &name,
				    -1);

		g_string_append_c (path, '/');
		g_string_append (path, name);
		g_free (name);
	}

	gtk_tree_path_free (iter);
	g_string_set_size (path, path->len + 1);
	*discpath = g_string_free (path, FALSE);

	return TRUE;
}

static const char *
bonfire_data_disc_add_path_item_position (GtkTreeModel *model,
					  GtkTreeIter *row,
					  GtkTreePath *path,
					  const char *ptr)
{
	GtkTreeIter child;
	int position;
	char *next;
	char *name;
	int size;

	ptr++;
	next = strchr (ptr, '/');
	if (!next)
		size = strlen (ptr);
	else
		size = next - ptr;

	position = 0;
	do {
		gtk_tree_model_get (model, row,
				    NAME_COL, &name,
				    -1);

		if (name && strlen (name) == size && !strncmp (name, ptr, size)) {
			gtk_tree_path_append_index (path, position);
			g_free (name);
			position = -1;
			break;
		}

		position++;
		g_free (name);
	} while (gtk_tree_model_iter_next (model, row));

	if (position != -1)
		return ptr;

	ptr += size;
	if (*ptr == '\0')
		return ptr;

	if (!gtk_tree_model_iter_children (model, &child, row))
		return ptr;

	return bonfire_data_disc_add_path_item_position (model,
							  &child,
							  path,
							  ptr);
}

/* FIXME: this is very slow we need to come up with something else */
static gboolean
bonfire_data_disc_disc_path_to_tree_path (BonfireDataDisc *disc,
					  const char *path,
					  GtkTreePath **treepath,
					  const char **end)
{
	GtkTreeModel *model;
	GtkTreePath *retval;
	GtkTreeIter row;
	const char *ptr;

	g_return_val_if_fail (path != NULL, FALSE);
	if (!strcmp (path, "/")) {
		*treepath = NULL;
		return TRUE;
	}

	model = disc->priv->model;
	if (!gtk_tree_model_get_iter_first (model, &row))
		return FALSE;

	ptr = path;

	retval = gtk_tree_path_new ();
	ptr = bonfire_data_disc_add_path_item_position (model,
							 &row,
							 retval,
							 ptr);

	if (*ptr != '\0') {
		if (!end) {
			gtk_tree_path_free (retval);
			return FALSE;
		}

		*end = ptr;
		*treepath = retval;
		return FALSE;
	}

	*treepath = retval;
	return TRUE;
}

static void
bonfire_data_disc_tree_remove_path (BonfireDataDisc *disc,
				    const char *path)
{
	GtkTreePath *treepath;
	GtkTreeModel *model;
	GtkTreeIter iter;
	gboolean result;

	if (!path)
		return;

	result = bonfire_data_disc_disc_path_to_tree_path (disc,
							   path,
							   &treepath,
							   NULL);
	if (!result)
		return;

	model = disc->priv->model;

	result = gtk_tree_model_get_iter (model, &iter, treepath);
	if (!result) {
		gtk_tree_path_free (treepath);
		return;
	}

	gtk_tree_store_remove (GTK_TREE_STORE (model), &iter);

	bonfire_data_disc_tree_update_parent (disc, treepath);
	gtk_tree_path_free (treepath);
}

static void
bonfire_data_disc_remove_uri_from_tree (BonfireDataDisc *disc,
					const char *uri)
{
	GSList *paths;
	GSList *iter;
	char *path;

	/* remove all occurences from the tree */
	paths = bonfire_data_disc_uri_to_paths (disc, uri);
	for (iter = paths; iter; iter = iter->next) {
		path = iter->data;

		bonfire_data_disc_tree_remove_path (disc, path);
		g_free (path);
	}
	g_slist_free (paths);
}

static gboolean
bonfire_data_disc_tree_new_path (BonfireDataDisc *disc,
				 const char *path,
				 const GtkTreePath *parent_treepath,
				 GtkTreePath **child_treepath)
{
	GtkTreePath *treepath;
	GtkTreeModel *model;
	GtkTreeIter child;
	gboolean result;
	char *name;

	if (!parent_treepath) {
		char *parent;

		parent = g_path_get_dirname (path);
		result = bonfire_data_disc_disc_path_to_tree_path (disc,
								   parent,
								   &treepath,
								   NULL);
		g_free (parent);
		if (!result)
			return FALSE;
	}
	else
		treepath = (GtkTreePath *) parent_treepath;

	model = disc->priv->model;
	if (treepath) {
		GtkTreeIter iter;

		result = gtk_tree_model_get_iter (model, &iter, treepath);
		if (parent_treepath != treepath)
			gtk_tree_path_free (treepath);

		if (!result)
			return FALSE;

		gtk_tree_store_append (GTK_TREE_STORE (model), &child, &iter);
		bonfire_data_disc_tree_update_directory_real (disc, &iter);
	}
	else
		gtk_tree_store_append (GTK_TREE_STORE (model), &child, NULL);

	if (child_treepath)
		*child_treepath = gtk_tree_model_get_path (model, &child);

	name = g_path_get_basename (path);
	gtk_tree_store_set (GTK_TREE_STORE (model), &child,
			    NAME_COL, name,
			    ROW_STATUS_COL, ROW_NEW, -1);
	g_free (name);

	return TRUE;
}

static gboolean
bonfire_data_disc_tree_set_path_from_info (BonfireDataDisc *disc,
					   const char *path,
					   const GtkTreePath *treepath,
					   const GnomeVFSFileInfo *info)
{
	const char *description;
	GtkTreeModel *model;
	GtkTreeIter parent;
	GdkPixbuf *pixbuf;
	GtkTreeIter iter;
	gboolean result;
	gboolean isdir;
	double dsize;
	char *name;
	char *size;

	if (!path)
		return FALSE;

	model = disc->priv->model;

	if (!treepath) {
		GtkTreePath *tmp_treepath;

		result = bonfire_data_disc_disc_path_to_tree_path (disc,
								   path,
								   &tmp_treepath,
								   NULL);
	
		if (!result)
			return FALSE;

		result = gtk_tree_model_get_iter (model, &iter, tmp_treepath);
		gtk_tree_path_free (tmp_treepath);
	
		if (!result)
			return FALSE;
	}
	else {
		result = gtk_tree_model_get_iter (model, &iter, (GtkTreePath*) treepath);

		if (!result)
			return FALSE;
	}

	if (info->type != GNOME_VFS_FILE_TYPE_DIRECTORY) {
		size = bonfire_utils_get_size_string (info->size, TRUE, TRUE);
		isdir = FALSE;
		dsize = info->size;
	}
	else {
		dsize = 0.0;
		size = g_strdup (_("(loading ...)"));
		isdir = TRUE;
	}

	if (info->mime_type) {
		pixbuf = bonfire_utils_get_icon_for_mime (info->mime_type, 16);
		description = gnome_vfs_mime_get_description (info->mime_type);
	}
	else {
		pixbuf = NULL;
		description = NULL;
	}

	name = g_path_get_basename (path);
	gtk_tree_store_set (GTK_TREE_STORE (model), &iter,
			    ICON_COL, pixbuf,
			    NAME_COL, name,
			    DSIZE_COL, dsize,
			    SIZE_COL, size,
			    MIME_COL, description,
			    ISDIR_COL, isdir, -1);
	g_free (name);
	g_free (size);

	if (pixbuf)
		g_object_unref (pixbuf);

	if (info->type != GNOME_VFS_FILE_TYPE_DIRECTORY) {
		gtk_tree_store_set (GTK_TREE_STORE (model), &iter,
				    ROW_STATUS_COL, ROW_EXPANDED,
				    -1);
		return TRUE;
	}

	/* see if this directory should be explored */
	if (gtk_tree_model_iter_parent (model, &parent, &iter)) {
		int status;

		gtk_tree_model_get (model, &parent,
				    ROW_STATUS_COL, &status,
				    -1);

		if (status == ROW_EXPANDED) {
			bonfire_data_disc_expose_path (disc, path);
			gtk_tree_store_set (GTK_TREE_STORE (model), &iter,
					    ROW_STATUS_COL, ROW_EXPLORING,
					    -1);
		}
		else
			gtk_tree_store_set (GTK_TREE_STORE (model), &iter,
					    ROW_STATUS_COL, ROW_NOT_EXPLORED,
					    -1);
	}
	else {
		bonfire_data_disc_expose_path (disc, path);
		gtk_tree_store_set (GTK_TREE_STORE (model), &iter,
				    ROW_STATUS_COL, ROW_EXPLORING,
				    -1);
	}

	return TRUE;
}

static gboolean
bonfire_data_disc_tree_new_empty_folder_real (BonfireDataDisc *disc,
					      const char *path,
					      gint state)
{
	const char *description;
	GtkTreePath *treepath;
	GtkTreeModel *model;
	GdkPixbuf *pixbuf;
	GtkTreeIter child;
	gboolean result;
	char *parent;
	char *name;

	parent = g_path_get_dirname (path);
	result = bonfire_data_disc_disc_path_to_tree_path (disc,
							   parent,
							   &treepath,
							   NULL);
	g_free (parent);
	if (!result)
		return FALSE;

	model = disc->priv->model;
	if (treepath) {
		GtkTreeIter iter;

		result = gtk_tree_model_get_iter (model, &iter, treepath);
		gtk_tree_path_free (treepath);
		if (!result)
			return FALSE;

		gtk_tree_store_append (GTK_TREE_STORE (model), &child, &iter);
		bonfire_data_disc_tree_update_directory_real (disc, &iter);
	}
	else
		gtk_tree_store_append (GTK_TREE_STORE (model), &child, NULL);

	pixbuf = bonfire_utils_get_icon_for_mime ("x-directory/normal", 16);
	description = gnome_vfs_mime_get_description ("x-directory/normal");
	name = g_path_get_basename (path);

	gtk_tree_store_set (GTK_TREE_STORE (model), &child,
			    NAME_COL, name,
			    MIME_COL, description,
			    ISDIR_COL, TRUE,
			    ICON_COL, pixbuf,
			    ROW_STATUS_COL, state,
			    -1);
	g_object_unref (pixbuf);
	g_free (name);

	bonfire_data_disc_tree_update_directory_real (disc, &child);
	return TRUE;
}

static gboolean
bonfire_data_disc_tree_new_empty_folder (BonfireDataDisc *disc,
					 const char *path)
{
	return bonfire_data_disc_tree_new_empty_folder_real (disc, path, ROW_EXPLORED);
}

static gboolean
bonfire_data_disc_tree_new_loading_row (BonfireDataDisc *disc,
					const char *path)
{
	GtkTreePath *treepath;
	GtkTreeModel *model;
	GtkTreeIter child;
	gboolean result;
	char *parent;
	char *name;

	parent = g_path_get_dirname (path);
	result = bonfire_data_disc_disc_path_to_tree_path (disc,
							    parent,
							    &treepath,
							    NULL);
	g_free (parent);
	if (!result)
		return FALSE;

	model = disc->priv->model;
	if (treepath) {
		GtkTreeIter iter;

		result = gtk_tree_model_get_iter (model, &iter, treepath);
		gtk_tree_path_free (treepath);
		if (!result)
			return FALSE;

		gtk_tree_store_append (GTK_TREE_STORE (model), &child, &iter);
		bonfire_data_disc_tree_update_directory_real (disc, &iter);
	}
	else
		gtk_tree_store_append (GTK_TREE_STORE (model), &child, NULL);

	name = g_path_get_basename (path);
	gtk_tree_store_set (GTK_TREE_STORE (model), &child,
			    NAME_COL, name,
			    SIZE_COL, _("(loading ...)"),
			    MIME_COL, _("(loading ...)"),
			    ROW_STATUS_COL, ROW_NEW,
			    -1);
	g_free (name);
	return TRUE;
}

/******************************** references ***********************************/
typedef gint BonfireDataDiscReference;
static char *BONFIRE_INVALID_REFERENCE = "Invalid";

static BonfireDataDiscReference
bonfire_data_disc_reference_new (BonfireDataDisc *disc,
				 const char *path)
{
	static BonfireDataDiscReference counter = 1;
	BonfireDataDiscReference retval;

	if (!disc->priv->path_refs)
		disc->priv->path_refs = g_hash_table_new (g_direct_hash,
							  g_direct_equal);

	retval = counter;
	while (g_hash_table_lookup (disc->priv->path_refs, GINT_TO_POINTER (retval))) {
		retval ++;

		if (retval == G_MAXINT)
			retval = 1;

		if (retval == counter)
			return 0;
	}

	g_hash_table_insert (disc->priv->path_refs,
			     GINT_TO_POINTER (retval),
			     g_strdup (path));

	counter = retval + 1;
	if (counter == G_MAXINT)
		counter = 1;

	return retval;
}

static void
bonfire_data_disc_reference_free (BonfireDataDisc *disc,
				  BonfireDataDiscReference ref)
{
	char *value;

	if (!disc->priv->path_refs)
		return;

	value = g_hash_table_lookup (disc->priv->path_refs, GINT_TO_POINTER (ref));
	g_hash_table_remove (disc->priv->path_refs, GINT_TO_POINTER (ref));
	if (value != BONFIRE_INVALID_REFERENCE)
		g_free (value);

	if (!g_hash_table_size (disc->priv->path_refs)) {
		g_hash_table_destroy (disc->priv->path_refs);
		disc->priv->path_refs = NULL;
	}
}

static void
bonfire_data_disc_reference_free_list (BonfireDataDisc *disc,
				       GSList *list)
{
	BonfireDataDiscReference ref;

	for (; list; list = g_slist_remove (list, GINT_TO_POINTER (ref))) {
		ref = GPOINTER_TO_INT (list->data);
		bonfire_data_disc_reference_free (disc, ref);
	}
}

static char *
bonfire_data_disc_reference_get (BonfireDataDisc *disc,
				 BonfireDataDiscReference ref)
{
	gpointer value;

	if (!disc->priv->path_refs)
		return NULL;

	value = g_hash_table_lookup (disc->priv->path_refs, GINT_TO_POINTER (ref));
	if (!value
	||   value == BONFIRE_INVALID_REFERENCE)
		return NULL;

	return g_strdup (value);
}

static GSList *
bonfire_data_disc_reference_get_list (BonfireDataDisc *disc,
				      GSList *references,
				      gboolean free_refs)
{
	char *path;
	GSList *iter;
	GSList *paths = NULL;
	BonfireDataDiscReference ref;

	if (!disc->priv->path_refs)
		return NULL;

	for (iter = references; iter; iter = iter->next) {
		ref = GPOINTER_TO_INT (iter->data);

		path = bonfire_data_disc_reference_get (disc, ref);

		if (free_refs)
			bonfire_data_disc_reference_free (disc, ref);

		if (path)
			paths = g_slist_prepend (paths, path);
	}

	if (free_refs)
		g_slist_free (references);

	return paths;
}

struct _MakeReferencesListData {
	char *path;
	int size;
	GSList *list;
};
typedef struct _MakeReferencesListData MakeReferencesListData;

static void
_foreach_make_references_list_cb (BonfireDataDiscReference num,
				  char *path,
				  MakeReferencesListData *data)
{
	if (!strncmp (path, data->path, data->size)
	&& (*(path + data->size) == '/' || *(path + data->size) == '\0'))
		data->list = g_slist_prepend (data->list, GINT_TO_POINTER (num));
}

static void
bonfire_data_disc_move_references (BonfireDataDisc *disc,
				   const char *oldpath,
				   const char *newpath)
{
	MakeReferencesListData callback_data;
	BonfireDataDiscReference ref;
	char *newvalue;
	char *value;
	int size;

	if (!disc->priv->path_refs)
		return;

	size = strlen (oldpath);
	callback_data.path = (char*) oldpath;
	callback_data.size = size;
	callback_data.list = NULL;

	g_hash_table_foreach (disc->priv->path_refs,
			      (GHFunc) _foreach_make_references_list_cb,
			      &callback_data);

	for (; callback_data.list; callback_data.list = g_slist_remove (callback_data.list, GINT_TO_POINTER (ref))) {
		ref = GPOINTER_TO_INT (callback_data.list->data);

		value = g_hash_table_lookup (disc->priv->path_refs,
					     GINT_TO_POINTER (ref));
		newvalue = g_strconcat (newpath, value + size, NULL);
		g_hash_table_replace (disc->priv->path_refs,
				      GINT_TO_POINTER (ref),
				      newvalue);
		g_free (value);
	}
}

static void
bonfire_data_disc_remove_references (BonfireDataDisc *disc,
				     const char *path)
{
	MakeReferencesListData callback_data;
	BonfireDataDiscReference ref;
	char *value;

	if (!disc->priv->path_refs)
		return;

	callback_data.path = (char*) path;
	callback_data.size = strlen (path);
	callback_data.list = NULL;

	g_hash_table_foreach (disc->priv->path_refs,
			      (GHFunc) _foreach_make_references_list_cb,
			      &callback_data);

	for (; callback_data.list; callback_data.list = g_slist_remove (callback_data.list, GINT_TO_POINTER (ref))) {
		ref = GPOINTER_TO_INT (callback_data.list->data);

		value = g_hash_table_lookup (disc->priv->path_refs,
					     GINT_TO_POINTER (ref));
		g_hash_table_replace (disc->priv->path_refs,
				      GINT_TO_POINTER (ref),
				      BONFIRE_INVALID_REFERENCE);
		g_free (value);
	}
}

static void
bonfire_data_disc_reference_invalidate_all (BonfireDataDisc *disc)
{
	char *root = "/";

	if (!disc->priv->path_refs)
		return;

	bonfire_data_disc_remove_references (disc, root);
}

/*********************** get file info asynchronously **************************/
typedef void (*BonfireInfoAsyncResultFunc) (BonfireDataDisc *disc,
					         GSList *results,
					         gpointer user_data);

struct _GetFileInfoAsyncData {
	GSList *uris;
	GSList *results;
	GnomeVFSFileInfoOptions flags;
	BonfireInfoAsyncResultFunc callback_func;
	gpointer user_data;

	int cancel:1;
};
typedef struct _GetFileInfoAsyncData GetFileInfoAsyncData;

struct _BonfireInfoAsyncResult {
	GnomeVFSResult result;
	GnomeVFSFileInfo *info;
	char *uri;
};
typedef struct _BonfireInfoAsyncResult BonfireInfoAsyncResult;

static void
bonfire_data_disc_get_file_info_async_destroy (gpointer data)
{
	GetFileInfoAsyncData *callback_data = data;
	BonfireInfoAsyncResult *result;
	GSList *iter;

	g_slist_foreach (callback_data->uris, (GFunc) g_free, NULL);
	g_slist_free (callback_data->uris);

	for (iter = callback_data->results; iter; iter = iter->next) {
		result = iter->data;
		if (result->uri)
			g_free (result->uri);

		if (result->info) {
			gnome_vfs_file_info_clear (result->info);
			gnome_vfs_file_info_unref (result->info);
		}

		g_free (result);
	}
	g_slist_free (callback_data->results);
	g_free (callback_data);
}

static void
bonfire_data_disc_get_file_info_cancel (gpointer data)
{
	GetFileInfoAsyncData *callback_data = data;

	callback_data->cancel = 1;
}

static gboolean
bonfire_data_disc_get_file_info_async_results (GObject *obj, gpointer data)
{
	GetFileInfoAsyncData *callback_data = data;
	BonfireDataDisc *disc = BONFIRE_DATA_DISC (obj);

	callback_data->callback_func (disc,
				      callback_data->results,
				      callback_data->user_data);

	bonfire_data_disc_decrease_activity_counter (disc);
	return TRUE;
}

static gboolean
bonfire_data_disc_get_file_info_async (GObject *obj, gpointer data)
{
	char *uri;
	GSList *next;
	char *escaped_uri;
	GnomeVFSFileInfo *info;
	BonfireInfoAsyncResult *result;
	GetFileInfoAsyncData *callback_data = data;

	for (; callback_data->uris; callback_data->uris = next) {
		uri = callback_data->uris->data;

		if (callback_data->cancel)
			return FALSE;

		result = g_new0 (BonfireInfoAsyncResult, 1);
		info = gnome_vfs_file_info_new ();

		/* If we want to make sure a directory is not added twice we have to make sure
		 * that it doesn't have a symlink as parent otherwise "/home/Foo/Bar" with Foo
		 * as a symlink pointing to /tmp would be seen as a different file from /tmp/Bar 
		 * It would be much better if we could use the inode numbers provided by gnome_vfs
		 * unfortunately they are guint64 and can't be used in hash tables as keys.
		 * Therefore we check parents up to root to see if there are symlinks and if so
		 * we get a path without symlinks in it. This is done only for local file */
		result->uri = bonfire_utils_check_for_parent_symlink (uri);
		next = g_slist_remove (callback_data->uris, uri);
		g_free (uri);

		escaped_uri = gnome_vfs_escape_host_and_path_string (result->uri);
		result->result = gnome_vfs_get_file_info (escaped_uri,
							  info,
							  callback_data->flags);
		g_free (escaped_uri);

		callback_data->results = g_slist_prepend (callback_data->results,
							  result);

		if (result->result != GNOME_VFS_OK) {
			gnome_vfs_file_info_clear (info);
			gnome_vfs_file_info_unref (info);
			continue;
		}

		if (info->type == GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK) {
			gnome_vfs_file_info_clear (info);

			if (!bonfire_utils_get_symlink_target (result->uri,
							       info,
							       callback_data->flags)) {
				if (info->type != GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK) {
					result->result = GNOME_VFS_ERROR_LOOP;
					gnome_vfs_file_info_clear (info);
					gnome_vfs_file_info_unref (info);
				}
			}
		}

		result->info = info;
	}

	callback_data->uris = NULL;
	return TRUE;
}

static BonfireDiscResult 
bonfire_data_disc_get_info_async (BonfireDataDisc *disc,
				  GSList *uris,
				  GnomeVFSFileInfoOptions flags,
				  BonfireInfoAsyncResultFunc func,
				  gpointer user_data)
{
	char *uri;
	GetFileInfoAsyncData *callback_data;

	if (!disc->priv->jobs)
		disc->priv->jobs = bonfire_async_job_manager_get_default ();

	if (!disc->priv->file_type) {
		disc->priv->file_type = bonfire_async_job_manager_register_type (disc->priv->jobs,
										 G_OBJECT (disc),
										 bonfire_data_disc_get_file_info_async,
										 bonfire_data_disc_get_file_info_async_results,
										 bonfire_data_disc_get_file_info_async_destroy,
										 bonfire_data_disc_get_file_info_cancel);
	}

	callback_data = g_new0 (GetFileInfoAsyncData, 1);
	for (; uris; uris = uris->next) {
		uri = uris->data;
		callback_data->uris = g_slist_prepend (callback_data->uris,
						       g_strdup (uri));
	}

	/* NOTE: we don't reverse the list on purpose as when we get infos 
	 * the results will be prepended restoring the right order of things */
	callback_data->flags = flags;
	callback_data->callback_func = func;
	callback_data->user_data = user_data;

	/* NOTE : if an error occurs the callback_data will be freed by async_job_manager */
	if (!bonfire_async_job_manager_queue (disc->priv->jobs,
					      disc->priv->file_type,
					      callback_data))
		return BONFIRE_DISC_ERROR_THREAD;

	bonfire_data_disc_increase_activity_counter (disc);
	return BONFIRE_DISC_OK;
}

/**************************************** **************************************/
static gboolean
_foreach_remove_excluded_cb (char *key,
			     GSList *excluding,
			     BonfireDataDisc *disc)
{
	if (!g_hash_table_lookup (disc->priv->dirs, key)
	&&  !g_hash_table_lookup (disc->priv->files, key))
		g_free (key);

	g_slist_free (excluding);

	return TRUE;
}

static void
bonfire_data_disc_empty_excluded_hash (BonfireDataDisc *disc)
{
	if (!disc->priv->excluded)
		return;

	g_hash_table_foreach_remove (disc->priv->excluded,
				     (GHRFunc) _foreach_remove_excluded_cb,
				     disc);

	g_hash_table_destroy (disc->priv->excluded);
	disc->priv->excluded = NULL;
}

static gboolean
_foreach_remove_restored_cb (char *key,
			     BonfireFilterStatus status,
			     BonfireDataDisc *disc)
{
	if (!g_hash_table_lookup (disc->priv->dirs, key)
	&&  !g_hash_table_lookup (disc->priv->files, key))
		g_free (key);

	return TRUE;
}

static void
bonfire_data_disc_empty_restored_hash (BonfireDataDisc *disc)
{
	if (!disc->priv->restored)
		return;

	g_hash_table_foreach_remove (disc->priv->restored,
				     (GHRFunc) _foreach_remove_restored_cb,
				     disc);

	g_hash_table_destroy (disc->priv->restored);
	disc->priv->restored = NULL;
}

static gboolean
_foreach_remove_symlink_cb (char *symlink,
			    char *target,
			    BonfireDataDisc *disc)
{
/*
	if (!g_hash_table_lookup (disc->priv->dirs, target)
	&&  !g_hash_table_lookup (disc->priv->files, target))
		g_free (target);
*/
	return TRUE;
}

static void
bonfire_data_disc_empty_symlink_hash (BonfireDataDisc *disc)
{

	if (!disc->priv->symlinks)
		return;

	g_hash_table_foreach_remove (disc->priv->symlinks,
				     (GHRFunc) _foreach_remove_symlink_cb,
				     disc);

	g_hash_table_destroy (disc->priv->symlinks);
	disc->priv->symlinks = NULL;
}

static gboolean
_foreach_remove_grafts_cb (const char *uri,
			   GSList *grafts,
			   BonfireDataDisc *disc)
{
	g_slist_foreach (grafts, (GFunc) g_free, NULL);
	g_slist_free (grafts);
	return TRUE;
}

static void
_foreach_remove_created_dirs_cb (char *graft, 
				 const char *uri,
				 BonfireDataDisc *disc)
{
	if (uri == BONFIRE_CREATED_DIR)
		g_free (graft);
}

static void
bonfire_data_disc_empty_grafts_hash (BonfireDataDisc *disc)
{
	g_hash_table_foreach (disc->priv->paths,
			     (GHFunc) _foreach_remove_created_dirs_cb,
			      disc);
	g_hash_table_destroy (disc->priv->paths);
	disc->priv->paths = g_hash_table_new (g_str_hash, g_str_equal);

	g_hash_table_foreach_remove (disc->priv->grafts,
				    (GHRFunc) _foreach_remove_grafts_cb,
				     disc);
}

static gboolean
_foreach_remove_files_cb (const char *key,
			  BonfireFile *file,
			  gpointer data)
{
	g_free (file->uri);
	g_free (file);

	return TRUE;
}

static void
bonfire_data_disc_empty_files_hash (BonfireDataDisc *disc)
{
	g_hash_table_foreach_remove (disc->priv->files,
				     (GHRFunc) _foreach_remove_files_cb,
				     disc);
}

static void
bonfire_data_disc_empty_dirs_hash (BonfireDataDisc *disc)
{
	g_hash_table_foreach_remove (disc->priv->dirs,
				     (GHRFunc) _foreach_remove_files_cb,
				     disc);
}

#ifdef BUILD_INOTIFY
static gboolean
_foreach_remove_monitored_cb (const int wd,
			      const char *uri,
			      BonfireDataDisc *disc)
{
	int dev_fd;

	dev_fd = g_io_channel_unix_get_fd (disc->priv->notify);
	inotify_rm_watch (dev_fd, wd);

	return TRUE;
}

static void
bonfire_data_disc_empty_monitor_hash (BonfireDataDisc *disc)
{
	g_hash_table_foreach_remove (disc->priv->monitored,
				     (GHRFunc) _foreach_remove_monitored_cb,
				     disc);
}
#endif

static gboolean
_foreach_remove_loading_cb (const char *key,
			    GSList *loading,
			    BonfireDataDisc *disc)
{
	BonfireDataDiscReference ref;
	GSList *iter;

	for (iter = loading; iter; iter = iter->next) {
		ref = GPOINTER_TO_INT (iter->data);
		bonfire_data_disc_reference_free (disc, ref);
	}

	g_slist_free (loading);
	return TRUE;
}

static void
bonfire_data_disc_empty_loading_hash (BonfireDataDisc *disc)
{
	if (!disc->priv->references)
		return;

	/* we don't need a mutex here as all threads have been stopped */
	g_hash_table_foreach_remove (disc->priv->references,
				     (GHRFunc) _foreach_remove_loading_cb,
				     disc);
	g_hash_table_destroy (disc->priv->references);
	disc->priv->references = NULL;
}

static void
bonfire_data_disc_clean (BonfireDataDisc *disc)
{
	if (disc->priv->selected_path) {
		gtk_tree_path_free (disc->priv->selected_path);
		disc->priv->selected_path = NULL;
	}

	/* set all references to paths to be invalid :
	 * this comes first as many callback functions rely on references
	 * to know whether they should still do what needs to be done */
	bonfire_data_disc_reference_invalidate_all (disc);

	/* stop any job: do it here to prevent any further addition */
	if (disc->priv->jobs)
		bonfire_async_job_manager_cancel_by_object (disc->priv->jobs, 
							    G_OBJECT (disc));

	/* empty expose loading, rescan queues */
	if (disc->priv->expose_id) {
		g_source_remove (disc->priv->expose_id);
		disc->priv->expose_id = 0;
	}

	g_slist_free (disc->priv->expose);
	disc->priv->expose = NULL;

	if (disc->priv->loading) {
		g_slist_free (disc->priv->loading);
		disc->priv->loading = NULL;
	}

	if (disc->priv->rescan) {
		g_slist_free (disc->priv->rescan);
		disc->priv->rescan = NULL;
	}

	/* empty restored hash table */
	bonfire_data_disc_empty_restored_hash (disc);

	/* empty excluded hash IT SHOULD COME FIRST before all other hashes */
	bonfire_data_disc_empty_excluded_hash (disc);

	/* empty symlinks hash comes first as well */
	bonfire_data_disc_empty_symlink_hash (disc);

	/* empty grafts : it should come first as well */
	bonfire_data_disc_empty_grafts_hash (disc);

	/* empty unreadable */
	if (disc->priv->unreadable) {
		g_hash_table_destroy (disc->priv->unreadable);
		disc->priv->unreadable = NULL;
	}

	/* empty files hash table */
	bonfire_data_disc_empty_files_hash (disc);

	/* empty dirs hash table */
	bonfire_data_disc_empty_dirs_hash (disc);

#ifdef BUILD_INOTIFY
	/* empty monitor hash table */
	bonfire_data_disc_empty_monitor_hash (disc);
#endif

	/* empty loading hash table */
	bonfire_data_disc_empty_loading_hash (disc);
}

static void
bonfire_data_disc_reset_real (BonfireDataDisc *disc)
{
	bonfire_data_disc_clean (disc);

	if (GTK_WIDGET (disc->priv->filter_button) && disc->priv->unreadable)
		gtk_widget_set_sensitive (disc->priv->filter_button, FALSE);

	disc->priv->activity_counter = 1;
	bonfire_data_disc_decrease_activity_counter (disc);

	/* reset size */
	disc->priv->size = 0.0;
}

/******************************** utility functions ****************************/
inline static gboolean
bonfire_data_disc_is_readable (GnomeVFSFileInfo *info)
{
	if (!GNOME_VFS_FILE_INFO_LOCAL (info))
		return TRUE;

	if (getuid () == info->uid && (info->permissions & GNOME_VFS_PERM_USER_READ))
		return TRUE;
	else if (getgid () == info->gid
		 && (info->permissions & GNOME_VFS_PERM_GROUP_READ))
		return TRUE;
	else if (info->permissions & GNOME_VFS_PERM_OTHER_READ)
		return TRUE;

	return FALSE;
}


inline static void
bonfire_data_disc_add_rescan (BonfireDataDisc *disc,
			      BonfireFile *dir)
{
	/* have a look at the rescan list and see if we uri was already inserted */
	if (g_slist_find (disc->priv->rescan, dir)
	||  g_slist_find (disc->priv->loading, dir))
		return;

	disc->priv->rescan = g_slist_append (disc->priv->rescan, dir);
	bonfire_data_disc_directory_load (disc, dir, FALSE);
}

static void
bonfire_data_disc_size_changed (BonfireDataDisc *disc,
				double size)
{
	char *string;

	if (size == 0.0)
		return;

	disc->priv->size += size;
	if (disc->priv->size == 0)
		string = NULL;
	else
		string = bonfire_utils_get_size_string (disc->priv->size,
							TRUE,
							TRUE);

	/* if there are still uris waiting in the queue to be explored and
	 * CDContent says it's full what should we do ?
	 * the best solution is just to continue exploration but prevent
	 * any other manual addition to the selection.
	 * another solution would be to stop everything all together but then
	 * the user wouldn't necessarily know that some directories were not 
	 * added and even then he would certainly still have to remove some
	 * files. It's better for him making the choice of which files and 
	 * directories to remove */
	bonfire_disc_size_changed (BONFIRE_DISC (disc), disc->priv->size, string);
	g_free (string);
}

static gboolean
bonfire_data_disc_original_parent (BonfireDataDisc *disc,
				   const char *uri,
				   const char *path)
{
	int result;
	char *graft;
	char *path_uri;
	const char *graft_uri;

	graft = bonfire_data_disc_graft_get (disc, path);
	if (!graft)
		return FALSE;

	graft_uri = g_hash_table_lookup (disc->priv->paths, graft);
	path_uri = g_strconcat (graft_uri, path + strlen (graft), NULL);
	g_free (graft);

	result = strcmp (path_uri, uri);
	g_free (path_uri);
	return (result == 0);
}

/******************************* unreadable files ******************************/
static void
bonfire_data_disc_unreadable_new (BonfireDataDisc *disc,
				  char *uri,
				  BonfireFilterStatus status)
{
	char *parenturi;
	BonfireFile *parent;

	parenturi = g_path_get_dirname (uri);
	parent = g_hash_table_lookup (disc->priv->dirs, parenturi);
	g_free (parenturi);

	if (!parent)  {
		g_free (uri);
		return;
	}

	if (!disc->priv->unreadable) {
		disc->priv->unreadable = g_hash_table_new_full (g_str_hash,
							        g_str_equal,
							        g_free,
							        NULL);

		/* we can now signal the user that some files were removed */
		gtk_widget_set_sensitive (disc->priv->filter_button, TRUE);
		gtk_tooltips_enable (disc->priv->tooltip);
	}

	if (disc->priv->filter_dialog
	&&  !g_hash_table_lookup (disc->priv->unreadable, uri)) {
		bonfire_filtered_dialog_add (BONFIRE_FILTERED_DIALOG (disc->priv->filter_dialog),
					     uri,
					     FALSE,
					     status);
	}

	g_hash_table_replace (disc->priv->unreadable,
			      uri,
			      GINT_TO_POINTER(status));
}

static BonfireFilterStatus
bonfire_data_disc_unreadable_free (BonfireDataDisc *disc,
				   const char *uri)
{
	BonfireFilterStatus status;

	if (!disc->priv->unreadable)
		return 0;

	status = GPOINTER_TO_INT (g_hash_table_lookup (disc->priv->unreadable, uri));
	g_hash_table_remove (disc->priv->unreadable, uri);
	if (!g_hash_table_size (disc->priv->unreadable)) {
		if (!disc->priv->restored) {
			gtk_widget_set_sensitive (disc->priv->filter_button,
						  FALSE);
		}

		g_hash_table_destroy (disc->priv->unreadable);
		disc->priv->unreadable = NULL;
	}

	return status;
}

static void
bonfire_data_disc_restored_new (BonfireDataDisc *disc, 
				const char *uri, 
				BonfireFilterStatus status)
{
	BonfireFile *file;
	char *key;

	if (!disc->priv->restored) {
		disc->priv->restored = g_hash_table_new (g_str_hash, g_str_equal);
		gtk_widget_set_sensitive (disc->priv->filter_button, TRUE);
	}

	if (!status)
		status = bonfire_data_disc_unreadable_free (disc, uri);

	g_mutex_lock (disc->priv->restored_lock);

	if ((file = g_hash_table_lookup (disc->priv->files, uri))
	||  (file = g_hash_table_lookup (disc->priv->dirs, uri)))
		key = file->uri;
	else
		key = g_strdup (uri);

	g_hash_table_insert (disc->priv->restored,
			     key,
			     GINT_TO_POINTER (status));
	g_mutex_unlock (disc->priv->restored_lock);
}

static BonfireFilterStatus
bonfire_data_disc_restored_free (BonfireDataDisc *disc,
				       const char *uri)
{
	gpointer key = NULL;
	gpointer status;

	if (!disc->priv->restored)
		return 0;

	g_mutex_lock (disc->priv->restored_lock);
	g_hash_table_lookup_extended (disc->priv->restored,
				      uri,
				      &key,
				      &status);
	g_hash_table_remove (disc->priv->restored, uri);
	g_mutex_unlock (disc->priv->restored_lock);

	if (!g_hash_table_lookup (disc->priv->dirs, uri)
	&&  !g_hash_table_lookup (disc->priv->files, uri))
		g_free (key);

	if (!g_hash_table_size (disc->priv->restored)) {
		if (!disc->priv->unreadable) {
			gtk_widget_set_sensitive (disc->priv->filter_button,
						  FALSE);
		}
		g_hash_table_destroy (disc->priv->restored);
		disc->priv->restored = NULL;
	}

	return GPOINTER_TO_INT (status);
}

/****************************** filtered dialog ********************************/
static gboolean
bonfire_data_disc_unreadable_dialog (BonfireDataDisc *disc,
				     const char *uri,
				     GnomeVFSResult result,
				     gboolean isdir)
{
	char *name;
	char *message_disc;
	GtkWidget *toplevel;
	GtkWidget *dialog;
	guint answer;

	name = g_filename_display_basename (uri);

	if (!isdir)
		message_disc = g_strdup_printf (_("The file \"%s\" is unreadable:"), name);
	else
		message_disc = g_strdup_printf (_("The directory \"%s\" is unreadable:"), name);

	g_free (name);

	toplevel = gtk_widget_get_toplevel (GTK_WIDGET (disc));
	dialog = gtk_message_dialog_new (GTK_WINDOW (toplevel),
					 GTK_DIALOG_DESTROY_WITH_PARENT |
					 GTK_DIALOG_MODAL,
					 GTK_MESSAGE_ERROR,
					 GTK_BUTTONS_CLOSE,
					 message_disc);
	g_free (message_disc);

	gtk_window_set_title (GTK_WINDOW (dialog), _("Unreadable file"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
						  "%s.",
						  gnome_vfs_result_to_string (result));

	gtk_widget_show_all (dialog);
	answer = gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);

	return TRUE;
}
static gboolean
bonfire_data_disc_restore_unreadable (BonfireDataDisc *disc,
				      const char *uri,
				      GnomeVFSResult result,
				      GnomeVFSFileInfo *info,
				      GSList *references)
{
	GSList *paths;
	char *path;

	/* NOTE: it can only be unreadable here or perhaps a
	 * recursive symlink if it was broken symlinks and a
	 * target was added afterwards */
	if (result != GNOME_VFS_OK
	|| !bonfire_data_disc_is_readable (info)) {
		bonfire_data_disc_remove_uri (disc, uri);
		bonfire_data_disc_unreadable_new (disc,
						  g_strdup (uri),
						  BONFIRE_FILTER_UNREADABLE);
		return FALSE;
	}

	/* start restoring : see if we still want to update */
	if (bonfire_data_disc_is_excluded (disc, uri, NULL))
		return TRUE;

	/* NOTE : the file could be in dirs or files (for example if it's a 
	 * hidden file) if the user has explicitly grafted it in the tree */
	if (info->type != GNOME_VFS_FILE_TYPE_DIRECTORY
	&& !g_hash_table_lookup (disc->priv->files, uri)) {
		char *parent;
		BonfireFile *file;

		parent = g_path_get_dirname (uri);
		file = g_hash_table_lookup (disc->priv->dirs, parent);
		g_free (parent);

		file->size += info->size;
		bonfire_data_disc_size_changed (disc, info->size);
	}
	else if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY
	     && !g_hash_table_lookup (disc->priv->dirs, uri))
		bonfire_data_disc_directory_new (disc,
						 g_strdup (uri),
						 TRUE);

	/* now let's see the tree */
	paths = bonfire_data_disc_reference_get_list (disc, references, FALSE);
	for (; paths; paths = g_slist_remove (paths, path)) {
		path = paths->data;
		bonfire_data_disc_tree_set_path_from_info (disc, path, NULL, info);
		g_free (path);
	}

	return TRUE;
}

static void
bonfire_data_disc_restore_unreadable_cb (BonfireDataDisc *disc,
					 GSList *results,
					 GSList *referencess)
{
	BonfireInfoAsyncResult *result;
	GSList *references;
	GSList *unreadable;
	gboolean success;
	GSList *iter;

	unreadable = NULL;
	for (iter = referencess ; results && iter; results = results->next, iter = iter->next) {
		references = iter->data;
		result = results->data;

		success = bonfire_data_disc_restore_unreadable (disc,
								result->uri,
								result->result,
								result->info,
								references);

		bonfire_data_disc_reference_free_list (disc, references);

		if (!success)	/* keep for later to tell the user it didn't go on well */
			unreadable = g_slist_prepend (unreadable, result);
		else
			bonfire_data_disc_selection_changed (disc, TRUE);
	}
	g_slist_free (referencess);

	if (unreadable) {
		GtkWidget *dialog;
		GtkWidget *toplevel;

		/* tell the user about the files that are definitely unreadable */
		toplevel = gtk_widget_get_toplevel (GTK_WIDGET (disc));
		dialog = gtk_message_dialog_new (GTK_WINDOW (toplevel),
						 GTK_DIALOG_DESTROY_WITH_PARENT |
						 GTK_DIALOG_MODAL,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_CLOSE,
						 _("Some files couldn't be restored."));

		gtk_window_set_title (GTK_WINDOW (dialog), _("File restoration failure"));

		for (; unreadable; unreadable = g_slist_remove (unreadable, unreadable->data)) {
			result = unreadable->data;
			g_warning ("ERROR : file \"%s\" couldn't be restored : %s\n",
				   result->uri,
				   gnome_vfs_result_to_string (result->result));
		}

		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);
	}
}


static void
bonfire_data_disc_filtered_restore (BonfireDataDisc *disc,
				    GSList *restored)
{
	GSList *referencess;
	GSList *references;
	GSList *paths;
	GSList *iter;
	GSList *uris;
	char *uri;

	referencess = NULL;
	uris = NULL;
	for (; restored; restored = restored->next) {
		uri = restored->data;

		/* we filter out those that haven't changed */
		if (disc->priv->restored
		&&  g_hash_table_lookup (disc->priv->restored, uri))
			continue;

		references = NULL;
		paths = bonfire_data_disc_uri_to_paths (disc, uri);
		for (iter = paths; iter; iter = iter->next) {
			BonfireDataDiscReference ref;
			char *path_uri;
			char *path;

			path = iter->data;
			path_uri = g_hash_table_lookup (disc->priv->paths, path);
			if (path_uri) {
				char *graft;
				char *parent;
		
				/* see if it's not this uri that is grafted there */
				if (!strcmp (path_uri, uri))
					continue;

				parent = g_path_get_dirname (path);
				graft = bonfire_data_disc_graft_get (disc, parent);
				g_free (parent);
		
				bonfire_data_disc_exclude_uri (disc, graft, uri);
				g_free (graft);
				continue;
			}

			bonfire_data_disc_tree_new_loading_row (disc, path);
			ref = bonfire_data_disc_reference_new (disc, path);
			references = g_slist_prepend (references, GINT_TO_POINTER (ref));
		}
		g_slist_foreach (paths, (GFunc) g_free, NULL);
		g_slist_free (paths);

		if (references) {
			bonfire_data_disc_restored_new (disc, uri, 0);
			uris = g_slist_prepend (uris, uri);
			referencess = g_slist_prepend (referencess, references);
		}
	}

	if (uris) {
		BonfireDiscResult success;

		success = bonfire_data_disc_get_info_async (disc,
							    uris,
							    GNOME_VFS_FILE_INFO_GET_ACCESS_RIGHTS |
							    GNOME_VFS_FILE_INFO_GET_MIME_TYPE |
							    GNOME_VFS_FILE_INFO_FORCE_SLOW_MIME_TYPE,
							    (BonfireInfoAsyncResultFunc) bonfire_data_disc_restore_unreadable_cb,
							    referencess);
		g_slist_free (uris);

		if (success != BONFIRE_DISC_OK) {
			for (iter = referencess; iter; iter = iter->next) {
				references = iter->data;
				bonfire_data_disc_reference_free_list (disc, references);
			}
			g_slist_free (referencess);
		}
	}
}

static void
_foreach_add_restored (char *uri,
		       BonfireFilterStatus status,
		       BonfireFilteredDialog *dialog)
{
	if (status == BONFIRE_FILTER_UNKNOWN)
		return;

	bonfire_filtered_dialog_add (dialog, uri, TRUE, status);
}

static void
_foreach_add_unreadable (char *uri,
			 BonfireFilterStatus status,
			 BonfireFilteredDialog *dialog)
{
	bonfire_filtered_dialog_add (dialog, uri, FALSE, status);
}

static void
bonfire_data_disc_filtered_files_clicked_cb (GtkButton *button,
					     BonfireDataDisc *disc)
{
	GSList *restored;
	GSList *removed;

	disc->priv->filter_dialog = bonfire_filtered_dialog_new ();
	if (disc->priv->restored)
		g_hash_table_foreach (disc->priv->restored,
				      (GHFunc) _foreach_add_restored,
				      disc->priv->filter_dialog);

	if (disc->priv->unreadable)
		g_hash_table_foreach (disc->priv->unreadable,
				      (GHFunc) _foreach_add_unreadable,
				      disc->priv->filter_dialog);

	gtk_window_set_transient_for (GTK_WINDOW (disc->priv->filter_dialog), 
				      GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (disc))));
	gtk_window_set_modal (GTK_WINDOW (disc->priv->filter_dialog), TRUE);
	gtk_widget_show_all (disc->priv->filter_dialog);
	gtk_dialog_run (GTK_DIALOG (disc->priv->filter_dialog));

	/* lets get the list of uri that were restored */
	bonfire_filtered_dialog_get_status (BONFIRE_FILTERED_DIALOG (disc->priv->filter_dialog),
					    &restored,
					    &removed);
	gtk_widget_destroy (disc->priv->filter_dialog);
	disc->priv->filter_dialog = NULL;

	bonfire_data_disc_filtered_restore (disc, restored);
	g_slist_foreach (restored, (GFunc) g_free, NULL);
	g_slist_free (restored);

	for (; removed; removed = g_slist_remove (removed, removed->data)) {
		BonfireFilterStatus status;
		char *uri;

		uri = removed->data;

		if (disc->priv->unreadable
		&&  g_hash_table_lookup (disc->priv->unreadable, uri)) {
			g_free (uri);
			continue;
		}

		status = bonfire_data_disc_restored_free (disc, uri);
		bonfire_data_disc_remove_uri (disc, uri);
		bonfire_data_disc_unreadable_new (disc, uri, status);
		bonfire_data_disc_selection_changed (disc, TRUE);
	}
}

/**************************** file/dir freeing *********************************/
static void
bonfire_data_disc_file_object_free (BonfireDataDisc *disc,
					  BonfireFile *file)
{
	GSList *excluding;
	BonfireFilterStatus status;

	if (disc->priv->restored
	&&  (status = GPOINTER_TO_INT (g_hash_table_lookup (disc->priv->restored, file->uri))))
		g_hash_table_replace (disc->priv->restored,
				      g_strdup (file->uri),
				      GINT_TO_POINTER (status));

	/* see if any excluding key points to it */
	if (disc->priv->excluded
	&& (excluding = g_hash_table_lookup (disc->priv->excluded, file->uri)))
		g_hash_table_replace (disc->priv->excluded,
				      g_strdup (file->uri),
				      excluding);

#ifdef BUILD_INOTIFY
	bonfire_data_disc_cancel_monitoring (disc, file);
#endif

	/* now we can safely free the structure */
	g_free (file->uri);
	g_free (file);

}

static void
bonfire_data_disc_file_free (BonfireDataDisc *disc,
			     BonfireFile *file)
{
	if (!bonfire_data_disc_is_excluded (disc, file->uri, NULL)) {
		BonfireFile *parent;
		char *parent_uri;

		parent_uri = g_path_get_dirname (file->uri);
		parent = g_hash_table_lookup (disc->priv->dirs, parent_uri);
		g_free (parent_uri);

		parent->size += file->size;
	}
	else
		bonfire_data_disc_size_changed (disc, file->size * (-1.0));

	bonfire_data_disc_file_object_free (disc, file);
}

static void
bonfire_data_disc_file_remove_from_tree (BonfireDataDisc *disc,
					       BonfireFile *file)
{
	bonfire_data_disc_remove_uri_from_tree (disc, file->uri);

	/* remove all file graft points if any and free */
	bonfire_data_disc_graft_remove_all (disc, file->uri);
	g_hash_table_remove (disc->priv->files, file->uri);
	bonfire_data_disc_file_free (disc, file);
}

#ifdef BUILD_INOTIFY
static void
_foreach_find_file_to_monitor (char *uri,
			       BonfireFile *file,
			       BonfireDataDisc *disc)
{
	char *parent;

	if (file->handle.wd != -1
	||  strncmp (uri, "file:", 5))
		return;

	parent = g_path_get_dirname (uri);
	if (!g_hash_table_lookup (disc->priv->dirs, parent))
		bonfire_data_disc_start_monitoring (disc, file);

	g_free (parent);
}
#endif

static gboolean
_foreach_unreadable_remove (char *uri,
			    BonfireFilterStatus status,
			    BonfireDataDisc *disc)
{
	char *parent;

	parent = g_path_get_dirname (uri);
	if (!g_hash_table_lookup (disc->priv->dirs, parent)) {
		g_free (parent);
		return TRUE;
	}

	g_free (parent);
	return FALSE;
}

static gboolean
_foreach_remove_symlink_children_cb (char *symlink,
				     int value,
				     BonfireDataDisc *disc)
{
	char *parent;

	parent = g_path_get_dirname (symlink);
	if (!g_hash_table_lookup (disc->priv->dirs, parent)) {
		g_free (parent);
		return TRUE;
	}

	g_free (parent);
	return FALSE;
}

static gboolean
_foreach_restored_remove (char *uri,
			  BonfireFilterStatus status,
			  BonfireDataDisc *disc)
{
	char *parent;

	parent = g_path_get_dirname (uri);
	if (!g_hash_table_lookup (disc->priv->dirs, parent)) {
		if (!g_hash_table_lookup (disc->priv->files, uri)
		&&  !g_hash_table_lookup (disc->priv->dirs, uri))
			g_free (uri);

		g_free (parent);
		return TRUE;
	}

	g_free (parent);
	return FALSE;
}

static void
bonfire_data_disc_update_hashes (BonfireDataDisc *disc)
{
#ifdef BUILD_INOTIFY
	/* check for children in files hash table since they may need monitoring now */
	g_hash_table_foreach (disc->priv->files,
			      (GHFunc) _foreach_find_file_to_monitor,
			      disc);
#endif
	/* remove unreadable children */
	if (disc->priv->unreadable) {
		g_hash_table_foreach_remove (disc->priv->unreadable,
					     (GHRFunc) _foreach_unreadable_remove,
					     disc);

		if (!g_hash_table_size (disc->priv->unreadable)) {
			g_hash_table_destroy (disc->priv->unreadable);
			disc->priv->unreadable = NULL;
			if (!disc->priv->restored) {
				gtk_widget_set_sensitive (disc->priv->filter_button,
							  FALSE);
			}
		}
	}

	/* remove symlink children */
	if (disc->priv->symlinks) {
		g_hash_table_foreach_remove (disc->priv->symlinks,
					     (GHRFunc) _foreach_remove_symlink_children_cb,
					     disc);

		if (!g_hash_table_size (disc->priv->symlinks)) {
			g_hash_table_destroy (disc->priv->symlinks);
			disc->priv->symlinks = NULL;
		}
	}

	/* remove restored file children */
	if (disc->priv->restored) {
		g_mutex_lock (disc->priv->restored_lock);
		g_hash_table_foreach_remove (disc->priv->restored,
					     (GHRFunc) _foreach_restored_remove,
					     disc);

		if (!g_hash_table_size (disc->priv->restored)) {
			g_hash_table_destroy (disc->priv->restored);
			disc->priv->restored = NULL;
			if (!disc->priv->unreadable) {
				gtk_widget_set_sensitive (disc->priv->filter_button,
							  FALSE);
			}
		}
		g_mutex_unlock (disc->priv->restored_lock);
	}
}

static void
bonfire_data_disc_dir_free (BonfireDataDisc *disc,
			    BonfireFile *dir)
{
	GSList *references;

	/* remove potential references */
	if (disc->priv->references
	&& (references = g_hash_table_lookup (disc->priv->references, dir->uri))) {
		BonfireDataDiscReference ref;

		references = g_hash_table_lookup (disc->priv->references, dir->uri);
		g_mutex_lock (disc->priv->references_lock);
		g_hash_table_remove (disc->priv->references, dir->uri);
		if (!g_hash_table_size (disc->priv->references)) {
			g_hash_table_destroy (disc->priv->references);
			disc->priv->references = NULL;
		}
		g_mutex_unlock (disc->priv->references_lock);

		for (; references; references = g_slist_remove (references, GINT_TO_POINTER (ref))) {
			ref = GPOINTER_TO_INT (references->data);
			bonfire_data_disc_reference_free (disc, ref);
		}
	}

	/* remove it from the rescan list */
	disc->priv->rescan = g_slist_remove (disc->priv->rescan, dir);

	/* it could be in the waiting list remove it */
	disc->priv->loading = g_slist_remove (disc->priv->loading, dir);

	/* update the size */
	bonfire_data_disc_size_changed (disc, dir->size * (-1.0));
	bonfire_data_disc_file_object_free (disc, dir);
}


struct _BonfireRemoveChildrenData {
	int size;
	BonfireFile *dir;
	GSList *dirs;
	GSList *files;
	char *graft;
	BonfireDataDisc *disc;
};
typedef struct _BonfireRemoveChildrenData BonfireRemoveChildrenData;

static gboolean
_foreach_remove_children_dirs_cb (const char *uri,
				  BonfireFile *dir,
				  BonfireRemoveChildrenData *data)
{
	if (!strncmp (dir->uri, data->dir->uri, data->size)
	&& *(dir->uri + data->size) == '/') {
		/* make sure that this children is not grafted somewhere else
		 * this can't be under dir since we removed all children grafted
		 * points so just check that it hasn't graft points neither any
		 * parent of this directory up to data->uri */
		if (!bonfire_data_disc_is_excluded (data->disc,
						    dir->uri,
						    data->dir))
			return FALSE;

		bonfire_data_disc_dir_free (data->disc, dir);
		return TRUE;
	}

	return FALSE;
}

static void
bonfire_data_disc_remove_dir_and_children (BonfireDataDisc *disc,
					   BonfireFile *dir)
{
	BonfireRemoveChildrenData callback_data;

	/* we remove all children from dirs hash table */
	callback_data.dir = dir;
	callback_data.size = strlen (dir->uri);
	callback_data.disc = disc;
	g_hash_table_foreach_remove (disc->priv->dirs,
				     (GHRFunc) _foreach_remove_children_dirs_cb,
				     &callback_data);

	/* remove the directory itself from the hash table */
	g_hash_table_remove (disc->priv->dirs, dir->uri);
	bonfire_data_disc_dir_free (disc, dir);
	bonfire_data_disc_update_hashes (disc);
}

static void
bonfire_data_disc_directory_remove_from_tree (BonfireDataDisc *disc,
					      BonfireFile *dir)
{
	GSList *paths;
	GSList *iter;
	char *path;

	/* we need to remove all occurence of file in the tree */
	paths = bonfire_data_disc_uri_to_paths (disc, dir->uri);

	/* remove all graft points for this dir 
	 * NOTE: order is important otherwise the above function would not 
	 * add the graft points path.
	 * NOTE: we don't care for children since we will have a notification */
	bonfire_data_disc_graft_children_remove (disc, paths);
	bonfire_data_disc_graft_remove_all (disc, dir->uri);

	for (iter = paths; iter; iter = iter->next) {
		path = iter->data;

		bonfire_data_disc_tree_remove_path (disc, path);
		g_free (path);
	}

	g_slist_free (paths);
	bonfire_data_disc_remove_dir_and_children (disc, dir);
}

static void
bonfire_data_disc_remove_children_async_cb (BonfireDataDisc *disc,
						  GSList *results,
						  char *uri_dir)
{
	BonfireInfoAsyncResult *result;
	BonfireFile *dir;
	GnomeVFSFileInfo *info;
	char *parent;
	char *uri;

	if (!g_hash_table_lookup (disc->priv->dirs, uri_dir)) {
		g_free (uri_dir);
		return;
	}

	for (; results; results = results->next) {
		result = results->data;

		/* NOTE: we don't care if it is still excluded or not:
		 * if it's no longer excluded it means that one of his parents
		 * has been added again and so there is a function that is going
		 * to add or has added its size already */
		uri = result->uri;

		parent = g_path_get_dirname (uri);
		dir = g_hash_table_lookup (disc->priv->dirs, uri);
		g_free (parent);

		if (!dir)
			continue;

		info = result->info;
		if (result->result == GNOME_VFS_ERROR_NOT_FOUND) {
			bonfire_data_disc_add_rescan (disc, dir);
			continue;
		}

		/* There shouldn't be any symlink so no need to check for loop */
		if (result->result != GNOME_VFS_OK) {
			/* we don't remove it from excluded in case it appears again */
			bonfire_data_disc_add_rescan (disc, dir);
			bonfire_data_disc_unreadable_new (disc,
							  g_strdup (uri),
							  BONFIRE_FILTER_UNREADABLE);
			continue;
		}

		/* we update the parent directory */
		dir->size -= info->size;
		bonfire_data_disc_size_changed (disc, info->size * (-1.0));
	}

	g_free (uri_dir);
}

static void
_foreach_remove_children_files_cb (char *uri,
				   GSList *excluding,
				   BonfireRemoveChildrenData *data)
{
	BonfireFile *dir;
	int excluding_num;
	int grafts_num;
	char *parent;
	GSList *list;

	if (data->disc->priv->unreadable
	&&  g_hash_table_lookup (data->disc->priv->unreadable, uri))
		return;

	/* we only want the children */
	if (strncmp (uri, data->dir->uri, data->size)
	||  *(uri + data->size) != '/')
		return;

	/* we don't want those with graft points (they can't be excluded
	 * and won't be even if we remove one of their parent graft point */
	if (g_hash_table_lookup (data->disc->priv->grafts, uri))
		return;

	/* we make a list of the graft points that exclude the file
	 * or one of its parents : for this file to be strictly excluded,
	 * the list must contains all parent graft points but the one we
	 * are going to remove */
	if (g_slist_find (excluding, data->graft))
		return;

	excluding_num = g_slist_length (excluding);
	grafts_num = 0;

	parent = g_path_get_basename (uri);
	dir = g_hash_table_lookup (data->disc->priv->dirs, parent);
	g_free (parent);

	while (dir) {
		list = g_hash_table_lookup (data->disc->priv->excluded, dir->uri);
		if (list) {
			if (g_slist_find (list, data->graft))
				return;

			excluding_num += g_slist_length (list);
		}

		/* NOTE: data->graft should have been previously removed from grafts */
		list = g_hash_table_lookup (data->disc->priv->grafts, dir->uri);
		if (list) {
			grafts_num += g_slist_length (list);
			if (grafts_num > excluding_num)
				return;
		}

		parent = g_path_get_dirname (dir->uri);
		dir = g_hash_table_lookup (data->disc->priv->dirs, parent);
		g_free (parent);
	}

	/* NOTE: it can't be uris in files as they couldn't 
	 * be excluded anyway since they have graft points */
	if (g_hash_table_lookup (data->disc->priv->dirs, uri))
		data->dirs = g_slist_prepend (data->dirs, g_strdup (uri));
	else 
		data->files = g_slist_prepend (data->files, g_strdup (uri));
}

static void
bonfire_data_disc_remove_children (BonfireDataDisc *disc,
				   BonfireFile *dir,
				   const char *graft)
{
	BonfireRemoveChildrenData callback_data;
	BonfireDiscResult result;
	BonfireFile *file;
	GSList *iter;
	char *uri;

	if (!disc->priv->excluded)
		return;

	/* we remove all children from dirs hash table */
	callback_data.dir = dir;
	callback_data.size = strlen (dir->uri);
	callback_data.graft = (char *) graft;
	callback_data.disc = disc;
	callback_data.dirs = NULL;
	callback_data.files = NULL;

	g_hash_table_foreach (disc->priv->excluded,
			      (GHFunc) _foreach_remove_children_files_cb,
			      &callback_data);

	for (iter = callback_data.dirs; iter; iter = g_slist_remove (iter, uri)) {
		uri = iter->data;

		/* make sure it still exists (it could have been destroyed 
		 * with a parent in bonfire_data_disc_remove_dir_and_children */
		if ((file = g_hash_table_lookup (disc->priv->dirs, uri))) {
			g_hash_table_remove (disc->priv->dirs, file->uri);
			bonfire_data_disc_dir_free (disc, file);
		}
	
		g_free (uri);
	}
	bonfire_data_disc_update_hashes (disc);

	if (!callback_data.files)
		return;

	uri = g_strdup (dir->uri);
	result = bonfire_data_disc_get_info_async (disc,
						   callback_data.files,
						   0,
						   (BonfireInfoAsyncResultFunc) bonfire_data_disc_remove_children_async_cb,
						   g_strdup (dir->uri));
	g_slist_foreach (callback_data.files, (GFunc) g_free, NULL);
	g_slist_free (callback_data.files);

	if (result != BONFIRE_DISC_OK)
		g_free (uri);
}

static void
bonfire_data_disc_remove_uri (BonfireDataDisc *disc,
			      const char *uri)
{
	BonfireFile *file;

	if ((file = g_hash_table_lookup (disc->priv->dirs, uri))) {
		bonfire_data_disc_directory_remove_from_tree (disc, file);
	}
	else if ((file = g_hash_table_lookup (disc->priv->files, uri))) {
		bonfire_data_disc_file_remove_from_tree (disc, file);
	}
	else if (disc->priv->symlinks
	     &&  g_hash_table_lookup (disc->priv->symlinks, uri)) {
		g_hash_table_remove (disc->priv->symlinks, uri);
	
		if (!g_hash_table_size (disc->priv->symlinks)) {
			g_hash_table_destroy (disc->priv->symlinks);
			disc->priv->symlinks = NULL;
		}
	}
	else if (!bonfire_data_disc_is_excluded (disc, uri, NULL)) {
		BonfireFile *parent;
		char *parenturi;

		/* NOTE: excluded files are already removed */
		parenturi = g_path_get_dirname (uri);
		parent = g_hash_table_lookup (disc->priv->dirs, parenturi);
		g_free (parenturi);
		if (!parent)
			return;

		bonfire_data_disc_remove_uri_from_tree (disc, uri);
		bonfire_data_disc_add_rescan (disc, parent);
	}
}

/******************************** graft points *********************************/
static const char *
bonfire_data_disc_graft_get_real (BonfireDataDisc *disc,
				  const char *path)
{
	char *tmp;
	char *parent;
	gpointer key = NULL;

	if (g_hash_table_lookup_extended (disc->priv->paths,
					  path,
					  &key,
					  NULL))
		return key;

	parent = g_path_get_dirname (path);
	while (parent [1] != '\0'
	&& !g_hash_table_lookup_extended (disc->priv->paths,
					  parent,
					  &key,
					  NULL)) {
	
		tmp = parent;
		parent = g_path_get_dirname (parent);
		g_free (tmp);
	}
	g_free (parent);

	return key;
}

static char *
bonfire_data_disc_graft_get (BonfireDataDisc *disc,
			     const char *path)
{
	return g_strdup (bonfire_data_disc_graft_get_real (disc, path));
}

static gboolean
bonfire_data_disc_graft_new (BonfireDataDisc *disc,
			     const char *uri,
			     const char *graft)
{
	char *realgraft;
	GSList *grafts;
	char *realuri;

	if (g_hash_table_lookup (disc->priv->paths, graft))
		return FALSE;

	realgraft = g_strdup (graft);

	if (uri) {
		BonfireFile *file;

		if (!(file = g_hash_table_lookup (disc->priv->dirs, uri)))
			file = g_hash_table_lookup (disc->priv->files, uri);

		realuri = file->uri;
	}
	else
		realuri = BONFIRE_CREATED_DIR;

	g_hash_table_insert (disc->priv->paths,
			     realgraft,
			     realuri);

	if (realuri == BONFIRE_CREATED_DIR)
		return TRUE;

	grafts = g_hash_table_lookup (disc->priv->grafts, realuri);
	grafts = g_slist_prepend (grafts, realgraft);
	g_hash_table_insert (disc->priv->grafts,
			     realuri,
			     grafts);
	
	return TRUE;
}

static GSList *
bonfire_data_disc_graft_new_list (BonfireDataDisc *disc,
				  const char *uri,
				  GSList *grafts)
{
	char *graft;
	GSList *next;
	GSList *iter;

	for (iter = grafts; iter; iter = next) {
		graft = iter->data;
		next = iter->next;

		if (!bonfire_data_disc_graft_new (disc, uri, graft)) {
			grafts = g_slist_remove (grafts, graft);
			g_free (graft);
		}	
	}

	return grafts;
}

struct _BonfireRemoveGraftPointersData {
	char *graft;
	GSList *list;
};
typedef struct _BonfireRemoveGraftPointersData
    BonfireRemoveGraftPointersData;

static void
_foreach_remove_graft_pointers_cb (char *key,
				   GSList *excluding,
				   BonfireRemoveGraftPointersData *data)
{
	/* we can't change anything in this function
	 * we simply make a list of keys with this graft */
	if (g_slist_find (excluding, data->graft))
		data->list = g_slist_prepend (data->list, key);
}

static void
bonfire_data_disc_graft_clean_excluded (BonfireDataDisc *disc,
					      const char *graft)
{
	char *uri;
	GSList *iter;
	BonfireRemoveGraftPointersData callback_data;

	if (!disc->priv->excluded)
		return;

	/* we need to remove any pointer to the graft point from the excluded hash */
	callback_data.graft = (char *) graft;
	callback_data.list = NULL;
	g_hash_table_foreach (disc->priv->excluded,
			      (GHFunc) _foreach_remove_graft_pointers_cb,
			      &callback_data);

	for (iter = callback_data.list; iter; iter = iter->next) {
		GSList *excluding;

		uri = iter->data;
		excluding = g_hash_table_lookup (disc->priv->excluded,
						 uri);
		excluding = g_slist_remove (excluding, graft);

		if (!excluding) {
			g_hash_table_remove (disc->priv->excluded,
					     uri);

			if (!g_hash_table_lookup (disc->priv->dirs, uri)
			&&  !g_hash_table_lookup (disc->priv->files, uri))
				g_free (uri);
		}
		else
			g_hash_table_insert (disc->priv->excluded,
					     uri, excluding);
	}

	if (!g_hash_table_size (disc->priv->excluded)) {
		g_hash_table_destroy (disc->priv->excluded);
		disc->priv->excluded = NULL;
	}

	g_slist_free (callback_data.list);
}

static gboolean
bonfire_data_disc_graft_remove (BonfireDataDisc *disc,
				const char *path)
{
	BonfireFile *file;
	gpointer oldgraft = NULL;
	gpointer uri = NULL;
	GSList *grafts;

	if (!g_hash_table_lookup_extended (disc->priv->paths,
					   path,
					   &oldgraft,
					   &uri))
		return FALSE;

	if (uri == BONFIRE_CREATED_DIR)
		goto end;

	grafts = g_hash_table_lookup (disc->priv->grafts, uri);
	grafts = g_slist_remove (grafts, oldgraft);

	if (grafts) {
		g_hash_table_insert (disc->priv->grafts, uri, grafts);
		if ((file = g_hash_table_lookup (disc->priv->dirs, uri))) {
			bonfire_data_disc_remove_children (disc,
							   file,
							   oldgraft);
			bonfire_data_disc_graft_clean_excluded (disc, oldgraft);
		}

		goto end;
	}

	g_hash_table_remove (disc->priv->grafts, uri);
	file = g_hash_table_lookup (disc->priv->files, uri);
	if (file) {
		g_hash_table_remove (disc->priv->files, file->uri);
		bonfire_data_disc_file_free (disc, file);
		goto end;
	}

	file = g_hash_table_lookup (disc->priv->dirs, uri);
	if (bonfire_data_disc_is_excluded (disc, uri, NULL)) {
		bonfire_data_disc_graft_clean_excluded (disc, oldgraft);
		bonfire_data_disc_remove_dir_and_children (disc, file);
	}
	else {
		bonfire_data_disc_remove_children (disc, file, oldgraft);
		bonfire_data_disc_graft_clean_excluded (disc, oldgraft);
	}

end:

	g_hash_table_remove (disc->priv->paths, oldgraft);
	g_free (oldgraft);

	return TRUE;
}

static void
bonfire_data_disc_graft_remove_all (BonfireDataDisc *disc,
				    const char *uri)
{
	GSList *grafts;
	char *graft;

	grafts = g_hash_table_lookup (disc->priv->grafts, uri);
	if (!grafts)
		return;

	g_hash_table_remove (disc->priv->grafts, uri);
	for (; grafts; grafts = g_slist_remove (grafts, graft)) {
		graft = grafts->data;

		bonfire_data_disc_graft_clean_excluded (disc, graft);
		g_hash_table_remove (disc->priv->paths, graft);
		g_free (graft);
	}
}

struct _BonfireMoveGraftChildData {
	int size;
	char *uri;
	char *oldgraft;
	char *newgraft;
	BonfireDataDisc *disc;
};
typedef struct _BonfireMoveGraftChildData BonfireMoveGraftChildData;

static void
_foreach_graft_changed_cb (char *key,
			   GSList *excluding,
			   BonfireMoveGraftChildData *data)
{
	/* the old graft can only be once in excluding */
	for (; excluding; excluding = excluding->next) {
		if (excluding->data == data->oldgraft) {
			excluding->data = data->newgraft;
			return;
		}
	}
}

static void
_foreach_move_children_graft_cb (char *uri,
				 GSList *grafts,
				 BonfireMoveGraftChildData *data)
{
	char *graft;

	for (; grafts; grafts = grafts->next) {
		graft = grafts->data;

		if (!strncmp (graft, data->oldgraft, data->size)
		&&  *(graft + data->size) == '/') {
			char *newgraft;

			newgraft = g_strconcat (data->newgraft,
						graft + data->size,
						NULL);

			/* be careful with excluded hash */
			if (data->disc->priv->excluded) {
				BonfireMoveGraftChildData callback_data;

				callback_data.oldgraft = graft;
				callback_data.newgraft = newgraft;
				g_hash_table_foreach (data->disc->priv->excluded,
						      (GHFunc) _foreach_graft_changed_cb,
						      &callback_data);
			}

			grafts->data = newgraft;
			g_hash_table_remove (data->disc->priv->paths, graft);
			g_hash_table_insert (data->disc->priv->paths,
					     newgraft,
					     uri);
			g_free (graft);
		}
	}
}

static void
bonfire_data_disc_graft_children_move (BonfireDataDisc *disc,
				       const char *oldpath,
				       const char *newpath)
{
	BonfireMoveGraftChildData callback_data;

	callback_data.disc = disc;
	callback_data.size = strlen (oldpath);
	callback_data.newgraft = (char *) newpath;
	callback_data.oldgraft = (char *) oldpath;

	g_hash_table_foreach (disc->priv->grafts,
			      (GHFunc) _foreach_move_children_graft_cb,
			      &callback_data);
}

struct _BonfireRemoveGraftedData {
	GSList *paths;
	BonfireDataDisc *disc;
};
typedef struct _BonfireRemoveGraftedData BonfireRemoveGraftedData;

static gboolean
_foreach_unreference_grafted_cb (char *graft,
				 char *uri,
				 BonfireRemoveGraftedData *data)
{
	GSList *iter;
	char *path;
	int size;

	if (graft == BONFIRE_CREATED_DIR)
		return FALSE;

	for (iter = data->paths; iter; iter = iter->next) {
		path = iter->data;
		size = strlen (path);

		if (!strncmp (path, graft, size)
		&&  *(graft + size) == '/') {
			BonfireFile *file;
			GSList *grafts;

			if (uri == BONFIRE_CREATED_DIR) {
				g_free (graft);
				return TRUE;
			}

			/* NOTE: the order is important here for 
			 * bonfire_data_disc_remove_children */
			grafts = g_hash_table_lookup (data->disc->priv->grafts, uri);
			grafts = g_slist_remove (grafts, graft);
			if (grafts) {
				g_hash_table_insert (data->disc->priv->grafts,
						     uri,
						     grafts);

				if ((file = g_hash_table_lookup (data->disc->priv->dirs, uri))) {
					bonfire_data_disc_remove_children (data->disc, file, graft);
					bonfire_data_disc_graft_clean_excluded (data->disc, graft);
				}

				g_free (graft);
				return TRUE;
			}
			g_hash_table_remove (data->disc->priv->grafts, uri);

			if ((file = g_hash_table_lookup (data->disc->priv->dirs, uri))) {
				if (bonfire_data_disc_is_excluded (data->disc, file->uri, NULL)) {
					bonfire_data_disc_graft_clean_excluded (data->disc, graft);
					bonfire_data_disc_remove_dir_and_children (data->disc, file);
				}
				else { 
					bonfire_data_disc_remove_children (data->disc, file, graft);
					bonfire_data_disc_graft_clean_excluded (data->disc, graft);
				}
			}
			else if ((file = g_hash_table_lookup (data->disc->priv->files, uri))) {
				g_hash_table_remove (data->disc->priv->files, uri);
				bonfire_data_disc_file_free (data->disc, file);
			}

			g_free (graft);
			return TRUE;
		}
	}

	return FALSE;
}

static void
bonfire_data_disc_graft_children_remove (BonfireDataDisc *disc,
					 GSList *paths)
{
	BonfireRemoveGraftedData callback_data;

	callback_data.disc = disc;
	callback_data.paths = paths;

	/* we remove all dirs /files which were grafted inside and which don't have any
	 * more reference. There is no need to see if there are grafted inside the
	 * grafted since they are necessarily the children of path */
	g_hash_table_foreach_remove (disc->priv->paths,
				     (GHRFunc) _foreach_unreference_grafted_cb,
				     &callback_data);
}

static void
bonfire_data_disc_graft_changed (BonfireDataDisc *disc,
				       const char *oldpath,
				       const char *newpath)
{
	gpointer oldgraft = NULL;
	gpointer newgraft = NULL;
	gpointer uri = NULL;

	newgraft = g_strdup (newpath);
	g_hash_table_lookup_extended (disc->priv->paths,
				      oldpath,
				      &oldgraft,
				      &uri);

	g_hash_table_remove (disc->priv->paths, oldgraft);
	g_hash_table_insert (disc->priv->paths, newgraft, uri);

	if (uri != BONFIRE_CREATED_DIR) {
		GSList *graft_node;
		GSList *grafts;

		grafts = g_hash_table_lookup (disc->priv->grafts, uri);
		graft_node = g_slist_find (grafts, oldgraft);
		graft_node->data = newgraft;
	}

	if (disc->priv->excluded
	&&  g_hash_table_lookup (disc->priv->dirs, uri)) {
		BonfireMoveGraftChildData callback_data;

		callback_data.oldgraft = oldgraft;
		callback_data.newgraft = newgraft;

		g_hash_table_foreach (disc->priv->excluded,
				      (GHFunc) _foreach_graft_changed_cb,
				      &callback_data);
	}

	g_free (oldgraft);
}

static void
_foreach_transfer_excluded_cb (const char *uri,
			       GSList *excluding,
			       BonfireMoveGraftChildData *data)
{
	/* only the children of data->uri are interesting */
	if (strncmp (uri, data->uri, data->size)
	||  *(uri + data->size) != '/')
		return;

	for (; excluding; excluding = excluding->next) {
		if (excluding->data == data->oldgraft) {
			/* there can only be one */
			excluding->data = data->newgraft;
			return;
		}
	}
}

static void
bonfire_data_disc_graft_transfer_excluded (BonfireDataDisc *disc,
						 const char *oldpath,
						 const char *newpath)
{
	BonfireMoveGraftChildData callback_data;
	gpointer newgraft = NULL;
	gpointer oldgraft = NULL;
	gpointer uri = NULL;

	/* NOTE : there shouldn't be nothing else but children here
	 * since we can't exclude anything else but children */

	if (!disc->priv->excluded)
		return;

	if (!g_hash_table_lookup_extended (disc->priv->paths,
					   oldpath,
					   &oldgraft,
					   NULL))
		return;

	if (!g_hash_table_lookup_extended (disc->priv->paths,
					   newpath,
					   &newgraft,
					   &uri))
		return;

	callback_data.uri = uri;
	callback_data.size = strlen (uri);
	callback_data.oldgraft = oldgraft;
	callback_data.newgraft = newgraft;

	g_hash_table_foreach (disc->priv->excluded,
			      (GHFunc) _foreach_transfer_excluded_cb,
			      &callback_data);
}

/********************* convert path to uri and vice versa **********************/
static GSList *
bonfire_data_disc_uri_to_paths (BonfireDataDisc *disc,
				const char *uri)
{
	char *tmp;
	char *path;
	char *graft;
	char *parent;
	GSList *list;
	GSList *grafts;
	GSList *excluding;
	GSList *paths = NULL;

	excluding = NULL;
	parent = g_strdup (uri);
	while (parent [1] != '\0') {
		if (disc->priv->excluded) {
			list = g_hash_table_lookup (disc->priv->excluded, parent);
			list = g_slist_copy (list);
			excluding = g_slist_concat (excluding, list);
		}

		grafts = g_hash_table_lookup (disc->priv->grafts, parent);
		for (; grafts; grafts = grafts->next) {
			graft = grafts->data;

			if (g_slist_find (excluding, graft))
				continue;

			path = g_strconcat (graft,
					    uri + strlen (parent),
					    NULL);

			paths = g_slist_append (paths, path);
		}

		tmp = parent;
		parent = g_path_get_dirname (tmp);
		g_free (tmp);
	}
	g_free (parent);
	g_slist_free (excluding);

	return paths;
}

static char *
bonfire_data_disc_path_to_uri (BonfireDataDisc *disc,
			       const char *path)
{
	const char *graft;
	char *graft_uri;
	char *retval;

	graft = bonfire_data_disc_graft_get_real (disc, path);
	if (!graft)
		return NULL;

	graft_uri = g_hash_table_lookup (disc->priv->paths, graft);

	if (graft_uri == BONFIRE_CREATED_DIR)
		return NULL;

	retval = g_strconcat (graft_uri, path + strlen (graft), NULL);
	return retval;
}

/********************************** new folder *********************************/
static void
bonfire_data_disc_new_folder_clicked_cb (GtkButton *button,
					 BonfireDataDisc *disc)
{
	GtkTreeSelection *selection;
	BonfireDiscResult success;
	GtkTreePath *treepath;
	GtkTreeModel *model;
	GtkTreeModel *sort;
	GtkTreeIter iter;
	GList *list;
	char *path;
	char *name;
	int nb;

	if (disc->priv->is_loading)
		return;

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (disc->priv->tree));
	sort = disc->priv->sort;
	model = disc->priv->model;
	list = gtk_tree_selection_get_selected_rows (selection, &sort);

	if (g_list_length (list) > 1) {
		g_list_foreach (list, (GFunc) gtk_tree_path_free, NULL);
		g_list_free (list);
		treepath = NULL;
	}
	else if (!list) {
		treepath = NULL;
	}
	else {
		int explored;
		gboolean isdir;
		GtkTreePath *tmp;

		treepath = list->data;
		g_list_free (list);

		tmp = gtk_tree_model_sort_convert_path_to_child_path (GTK_TREE_MODEL_SORT(sort),
								      treepath);
		gtk_tree_path_free (treepath);
		treepath = tmp;

		gtk_tree_model_get_iter (model, &iter, treepath);
		gtk_tree_model_get (model, &iter,
				    ISDIR_COL, &isdir,
				    ROW_STATUS_COL, &explored,
				    -1);

		if (!isdir 
		||  explored < ROW_EXPLORED) {
			gtk_tree_path_up (treepath);

			if (gtk_tree_path_get_depth (treepath) < 1) {
				gtk_tree_path_free (treepath);
				treepath = NULL;
			}
		}
	}

	name = g_strdup_printf (_("New folder"));
	nb = 1;

      newname:
	success = bonfire_data_disc_check_for_existence (disc,
							 name,
							 treepath,
							 FALSE);
	if (success == BONFIRE_DISC_ERROR_ALREADY_IN_TREE) {
		g_free (name);
		name = g_strdup_printf (_("New folder %i"), nb);
		nb++;
		goto newname;
	}

	if (treepath) {
		char *parent;

		bonfire_data_disc_tree_path_to_disc_path (disc,
							  treepath,
							  &parent);

		path = g_strconcat (parent, "/", name, NULL);
		gtk_tree_path_free (treepath);
		g_free (parent);
	}
	else
		path = g_strconcat ("/", name, NULL);

	g_free (name);

	bonfire_data_disc_graft_new (disc, NULL, path);

	/* just to make sure that tree is not hidden behind info */
	gtk_notebook_set_current_page (GTK_NOTEBOOK (BONFIRE_DATA_DISC (disc)->priv->notebook), 1);

	bonfire_data_disc_tree_new_empty_folder (disc, path);
	g_free (path);

	bonfire_data_disc_selection_changed (disc, TRUE);
}

/************************************ files excluded ***************************/
/* this is done in order:
 * - to minimize memory usage as much as possible ??? if we've got just one excluded ....
 * - to help building of lists of excluded files
 * downside: we need to be careful when we remove a file from the dirs 
 * hash or from the file hash and check that it is not in the excluded hash */
static void
bonfire_data_disc_exclude_uri (BonfireDataDisc *disc,
			       const char *path,
			       const char *uri)
{
	gpointer key = NULL;
	gpointer graft = NULL;
	gpointer excluding = NULL;
	BonfireFile *file;

	if (!g_hash_table_lookup_extended (disc->priv->paths,
					   path,
					   &graft,
					   NULL))
		return;

	if (!disc->priv->excluded) {
		/* we create the hash table on demand */
		disc->priv->excluded = g_hash_table_new (g_str_hash, g_str_equal);
	}
	else if (g_hash_table_lookup_extended (disc->priv->excluded,
					       uri,
					       &key,
					       &excluding)) {
		excluding = g_slist_prepend (excluding, graft);
		g_hash_table_insert (disc->priv->excluded,
				     key,
				     excluding);
		return;
	}

	excluding = g_slist_prepend (NULL, graft);

	if ((file = g_hash_table_lookup (disc->priv->dirs, uri)))
		g_hash_table_insert (disc->priv->excluded,
				     file->uri,
				     excluding);
	else if ((file = g_hash_table_lookup (disc->priv->files, uri)))
		g_hash_table_insert (disc->priv->excluded,
				     file->uri,
				     excluding);
	else
		g_hash_table_insert (disc->priv->excluded,
				     g_strdup (uri),
				     excluding);
}

static void
bonfire_data_disc_restore_uri (BonfireDataDisc *disc,
				     const char *path,
				     const char *uri)
{
	gpointer excluding = NULL;
	gpointer graft = NULL;
	gpointer key = NULL;

	if (!disc->priv->excluded)
		return;

	if (!g_hash_table_lookup_extended (disc->priv->paths,
					   path,
					   &graft,
					   NULL))
		return;

	if (!g_hash_table_lookup_extended (disc->priv->excluded,
					   uri, 
					   &key,
					   &excluding))
		return;

	excluding = g_slist_remove (excluding, graft);
	if (excluding) {
		g_hash_table_insert (disc->priv->excluded,
				     key,
				     excluding);
		return;
	}

	if (!g_hash_table_lookup (disc->priv->dirs, key)
	&&  !g_hash_table_lookup (disc->priv->files, key))
		g_free (key);

	g_hash_table_remove (disc->priv->excluded, uri);
	if (g_hash_table_size (disc->priv->excluded))
		return;

	g_hash_table_destroy (disc->priv->excluded);
	disc->priv->excluded = NULL;
}

static gboolean
bonfire_data_disc_is_excluded (BonfireDataDisc *disc,
			       const char *uri,
			       BonfireFile *top)
{
	BonfireFile *dir;
	int excluding_num;
	int grafts_num;
	char *parent;
	GSList *list;

	/* to be strictly excluded a files mustn't have graft points */
	if (g_hash_table_lookup (disc->priv->grafts, uri))
		return FALSE;

	parent = g_path_get_dirname (uri);
	dir = g_hash_table_lookup (disc->priv->dirs, parent);
	g_free (parent);

	/* to be strictly excluded it mustn't have parent dir nor graft points */
	if (!dir)
		return TRUE;

	/* we make a list of all exclusions up to the top existing directory */
	if (disc->priv->excluded) {
		list = g_hash_table_lookup (disc->priv->excluded, uri);
		excluding_num = g_slist_length (list);
	}
	else
		excluding_num = 0;

	grafts_num = 0;
	while (dir && dir != top) {
		if (disc->priv->excluded) {
			list = g_hash_table_lookup (disc->priv->excluded, dir->uri);
			excluding_num += g_slist_length (list);
		}

		if ((list = g_hash_table_lookup (disc->priv->grafts, dir->uri))) {
			grafts_num += g_slist_length (list);
			if (grafts_num > excluding_num) 
				return FALSE;
		}

		parent = g_path_get_dirname (dir->uri);
		dir = g_hash_table_lookup (disc->priv->dirs, parent);
		g_free (parent);
	}

	grafts_num -= excluding_num;
	if (grafts_num > 0)
		return FALSE;

	return TRUE;
}

/************************************** expose row *****************************/
#define EXPOSE_EXCLUDE_FILE(data, uri, path)	\
		(disc->priv->excluded	&& \
		(g_slist_find (g_hash_table_lookup (disc->priv->excluded, uri),	\
			       bonfire_data_disc_graft_get_real (data, path)) != NULL))

struct _MakeGraftedFilesList  {
	char *path;
	GSList *grafts;
};
typedef struct _MakeGraftedFilesList MakeGraftedFilesList;

static void
bonfire_data_disc_expose_grafted_cb (BonfireDataDisc *disc,
				     GSList *results,
				     GSList *refs)
{
	BonfireDataDiscReference ref;
	BonfireInfoAsyncResult *result;
	GnomeVFSFileInfo *info;
	char *path;

	for (; results && refs; results = results->next, refs = refs->next) {
		result = results->data;
		ref = GPOINTER_TO_INT (refs->data);
		info = result->info;

		path = bonfire_data_disc_reference_get (disc, ref);
		bonfire_data_disc_reference_free (disc, ref);
		if (!path)
			continue;

		/* NOTE: we don't need to check if path still corresponds to 
		 * a graft point since the only way for a graft point to be
		 * removed is if notification announced a removal or if the
		 * user explicitly removed the graft in the tree. either way
		 * the references hash table will be updated */

		if (result->result != GNOME_VFS_OK) {
			bonfire_data_disc_remove_uri (disc, result->uri);
			if (result->result != GNOME_VFS_ERROR_NOT_FOUND)
				bonfire_data_disc_unreadable_new (disc,
								  g_strdup (result->uri),
								  BONFIRE_FILTER_UNREADABLE);
			g_free (path);
			continue;
		}

		bonfire_data_disc_tree_set_path_from_info (disc,
							   path,
							   NULL,
							   info);
		g_free (path);
	}

	g_slist_free (refs);
}

static void
bonfire_data_disc_expose_grafted (BonfireDataDisc *disc,
				  GSList *grafts)
{
	char *uri;
	const char *path;
	GSList *uris = NULL;
	GSList *paths = NULL;
	GSList *created = NULL;
	BonfireDataDiscReference ref;

	for (; grafts; grafts = grafts->next) {
		path = grafts->data;

		uri = g_hash_table_lookup (disc->priv->paths, path);
		if (uri == BONFIRE_CREATED_DIR) {
			created = g_slist_prepend (created, (char *) path);
			continue;
		}
		else if (!uri)
			continue;

		bonfire_data_disc_tree_new_loading_row (disc, path);
		uris = g_slist_prepend (uris, uri);
		ref = bonfire_data_disc_reference_new (disc, path);
		paths = g_slist_prepend (paths, GINT_TO_POINTER (ref));
	}

	if (uris) {
		gboolean success;

		success = bonfire_data_disc_get_info_async (disc,
							    uris,
							    GNOME_VFS_FILE_INFO_GET_MIME_TYPE|
							    GNOME_VFS_FILE_INFO_FORCE_SLOW_MIME_TYPE,
							    (BonfireInfoAsyncResultFunc) bonfire_data_disc_expose_grafted_cb,
							    paths);
		g_slist_free (uris);

		if (success != BONFIRE_DISC_OK) {
			for (; paths; paths = g_slist_remove (paths, GINT_TO_POINTER (ref))) {
				ref = GPOINTER_TO_INT (paths->data);
				bonfire_data_disc_reference_free (disc, ref);
			}
		}
	}

	for (; created; created = g_slist_remove (created, path)) {
		path = created->data;
		bonfire_data_disc_tree_new_empty_folder (disc, path);
	}
}

static void
bonfire_data_disc_dir_contents_cancel (gpointer data)
{
	BonfireDirectoryContentsData *callback_data = data;

	callback_data->cancel = 1;
}

static void
_free_dir_contents_error (BonfireLoadDirError *error, gpointer null_data)
{
	g_free (error->uri);
	g_free (error);
}

static void
bonfire_data_disc_dir_contents_destroy (gpointer data)
{
	BonfireDirectoryContentsData *callback_data = data;

	g_slist_foreach (callback_data->infos, (GFunc) gnome_vfs_file_info_clear, NULL);
	g_slist_foreach (callback_data->infos, (GFunc) gnome_vfs_file_info_unref, NULL);
	g_slist_free (callback_data->infos);

	g_slist_foreach (callback_data->errors, (GFunc) _free_dir_contents_error, NULL);
	g_slist_free (callback_data->errors);

	g_free (callback_data->uri);
	g_free (callback_data);
}

static gboolean
bonfire_data_disc_expose_path_real (BonfireDataDisc *disc)
{
	char *uri;
	char *path;
	GSList *iter;
	GSList *paths;
	GSList *infos;
	GSList *references;
	GtkTreePath *treepath;
	GnomeVFSFileInfo *info;
	GSList *treepaths = NULL;
	BonfireDirectoryContentsData *data;

next:
	if (!disc->priv->expose) {
		disc->priv->expose_id = 0;
		return FALSE;
	}

	data = disc->priv->expose->data;
	disc->priv->expose = g_slist_remove (disc->priv->expose, data);

	/* convert all references to paths ignoring the invalid ones */
	if (disc->priv->references
	&& (references = g_hash_table_lookup (disc->priv->references, data->uri))) {
		paths = bonfire_data_disc_reference_get_list (disc,
							      references,
							      TRUE);
	}
	else
		paths = NULL;

	if (!paths) {
		bonfire_data_disc_decrease_activity_counter (disc);
		bonfire_data_disc_dir_contents_destroy (data);
		goto next;
	}

	g_mutex_lock (disc->priv->references_lock);
	g_hash_table_remove (disc->priv->references, data->uri);
	if (!g_hash_table_size (disc->priv->references)) {
		g_hash_table_destroy (disc->priv->references);
		disc->priv->references = NULL;
	}
	g_mutex_unlock (disc->priv->references_lock);

	/* for every path we look for the corresponding tree paths in treeview */
	for (iter = paths; iter; iter = iter->next) {
		char *path;

		path = iter->data;

		if (bonfire_data_disc_disc_path_to_tree_path (disc,
							      path,
							      &treepath,
							      NULL)) {
			if (!treepath)
				gtk_tree_path_new_from_indices (-1);

			treepaths = g_slist_prepend (treepaths, treepath);
		}
	}

	for (infos = data->infos; infos; infos = g_slist_remove (infos, info)) {
		GSList *treepath_iter;
		GSList *path_iter;

		info = infos->data;
		uri = g_strconcat (data->uri, "/", info->name, NULL);

		/* see if this file is not in unreadable. it could happen for files we
		 * couldn't read.
		 * but in the latter case if we can read them again inotify will tell
		 * us and we'll add them later so for the moment ignore all unreadable */
		if (disc->priv->unreadable
		&&  g_hash_table_lookup (disc->priv->unreadable, uri)) {
			gnome_vfs_file_info_clear(info);
			gnome_vfs_file_info_unref(info);
			g_free (uri);
			continue;
		}

		/* NOTE: that can't be a symlink as they are handled
		 * separately (or rather their targets) */
		path_iter = paths;
		treepath_iter = treepaths;
		for (; path_iter; path_iter = path_iter->next, treepath_iter = treepath_iter->next) {
			char *parent;
			GtkTreePath *tmp_treepath = NULL;

			/* make sure this file wasn't excluded */
			parent = path_iter->data;
			path = g_strconcat (parent, "/", info->name, NULL);

			if (EXPOSE_EXCLUDE_FILE (disc, uri, path)) {
				g_free (path);
				continue;
			}

			treepath = treepath_iter->data;
			bonfire_data_disc_tree_new_path (disc,
							 path,
							 treepath,
							 &tmp_treepath);
			bonfire_data_disc_tree_set_path_from_info (disc,
								   path,
								   tmp_treepath,
								   info);

			gtk_tree_path_free (tmp_treepath);
			g_free (path);
		}

		gnome_vfs_file_info_clear(info);
		gnome_vfs_file_info_unref(info);
		g_free (uri);
	}

	/* free disc */
	g_free (data->uri);
	g_free (data);

	/* free tree paths */
	g_slist_foreach (paths, (GFunc) g_free, NULL);
	g_slist_free (paths);
	paths = NULL;

	for (; treepaths; treepaths = g_slist_remove (treepaths, treepaths->data)) {
		GtkTreePath *treepath;
		GtkTreeModel *model;
		GtkTreeIter iter;

		treepath = treepaths->data;
		model = disc->priv->model;
		if (gtk_tree_model_get_iter (model, &iter, treepath)) {
			gtk_tree_store_set (GTK_TREE_STORE (model), &iter,
					    ROW_STATUS_COL, ROW_EXPLORED,
					    -1);

			bonfire_data_disc_tree_update_directory_real (disc, &iter);
		}

		gtk_tree_path_free (treepath);
	}
	
	bonfire_data_disc_decrease_activity_counter (disc);
	return TRUE;
}

/* this used to be done async */
static gboolean
bonfire_data_disc_expose_result (GObject *object, gpointer data)
{
	BonfireDirectoryContentsData *callback_data = data;
	BonfireDataDisc *disc = BONFIRE_DATA_DISC (object);

	if (callback_data->errors) {
		bonfire_data_disc_load_dir_error (disc, callback_data->errors);
		return TRUE;
	}

	BONFIRE_ADD_TO_EXPOSE_QUEUE (disc, callback_data);
	return FALSE;
}

static gboolean
bonfire_data_disc_expose_thread (GObject *object, gpointer data)
{
	BonfireDirectoryContentsData *callback_data = data;
	GnomeVFSDirectoryHandle *handle;
	GnomeVFSFileInfo *info;
	GnomeVFSResult result;
	GSList *infos = NULL;
	char *escaped_uri;

	handle = NULL;
	escaped_uri = gnome_vfs_escape_host_and_path_string (callback_data->uri);
	result = gnome_vfs_directory_open (&handle,
					   escaped_uri,
					   GNOME_VFS_FILE_INFO_GET_MIME_TYPE |
					   GNOME_VFS_FILE_INFO_FORCE_SLOW_MIME_TYPE);
	g_free (escaped_uri);

	if(result != GNOME_VFS_OK) {
		BonfireLoadDirError *error;

		error = g_new0 (BonfireLoadDirError, 1);
		error->uri = callback_data->uri;
		callback_data->uri = NULL;

		error->status = BONFIRE_FILTER_UNREADABLE;
		callback_data->errors = g_slist_prepend (callback_data->errors, error);

		if (handle) {
			gnome_vfs_directory_close (handle);
			handle = NULL;
		}

		g_warning ("Cannot open dir %s : %s\n",
			   error->uri,
			   gnome_vfs_result_to_string(result));

		return TRUE;
	}

	infos = NULL;
	info = gnome_vfs_file_info_new();

	while (!callback_data->cancel) {
		result = gnome_vfs_directory_read_next (handle, info);

		if (result != GNOME_VFS_OK)
			break;

		if (*info->name == '.' && (info->name[1] == 0
		||  (info->name[1] == '.' && info->name[2] == 0))) {
			gnome_vfs_file_info_clear(info);
			continue;
		}

		/* symlinks are exposed through expose_grafted */
		if (info->type == GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK) {
			gnome_vfs_file_info_clear (info);
			continue;
		}

		infos = g_slist_prepend (infos, info);
		info = gnome_vfs_file_info_new ();
	}

	gnome_vfs_directory_close (handle);
	gnome_vfs_file_info_unref (info);
	callback_data->infos = infos;

	return TRUE;
}

static BonfireDiscResult
bonfire_data_disc_expose_insert_path_real (BonfireDataDisc *disc,
					   const char *uri)
{
	BonfireDirectoryContentsData *callback_data;

	if (!disc->priv->jobs)
		bonfire_async_job_manager_get_default ();

	if (!disc->priv->expose_type) {
		disc->priv->expose_type = bonfire_async_job_manager_register_type (disc->priv->jobs,
										   G_OBJECT (disc),
										   bonfire_data_disc_expose_thread,
										   bonfire_data_disc_expose_result,
										   bonfire_data_disc_dir_contents_destroy,
										   bonfire_data_disc_dir_contents_cancel);
	}

	callback_data = g_new0 (BonfireDirectoryContentsData, 1);
	callback_data->uri = g_strdup (uri);

	if (!bonfire_async_job_manager_queue (disc->priv->jobs,
					      disc->priv->expose_type,
					      callback_data))
		return BONFIRE_DISC_ERROR_THREAD;

	bonfire_data_disc_increase_activity_counter (disc);
	return BONFIRE_DISC_OK;
}

static void
_foreach_make_grafted_files_list_cb (char *path,
				     const char *uri,
				     MakeGraftedFilesList *data)
{
	char *parent;

	parent = g_path_get_dirname (path);
	if (!strcmp (parent, data->path))
		data->grafts = g_slist_prepend (data->grafts, path);
	g_free (parent);
}

static BonfireDiscResult
bonfire_data_disc_expose_path (BonfireDataDisc *disc,
			       const char *path)
{
	BonfireDataDiscReference ref;
	MakeGraftedFilesList callback_data;
	BonfireFile *dir;
	GSList *references;
	char *uri;

	/* have a look at paths hash to see if any grafts should be exposed under this path */
	callback_data.grafts = NULL;
	callback_data.path = (char *) path;
	g_hash_table_foreach (disc->priv->paths,
			      (GHFunc) _foreach_make_grafted_files_list_cb,
			      &callback_data);

	bonfire_data_disc_expose_grafted (disc, callback_data.grafts);
	g_slist_free (callback_data.grafts);

	uri = bonfire_data_disc_path_to_uri (disc, path);
	if (!uri) {
		if (!g_hash_table_lookup (disc->priv->paths, path))
			return BONFIRE_DISC_NOT_IN_TREE;

		return BONFIRE_DISC_OK;
	}

	if (!(dir = g_hash_table_lookup(disc->priv->dirs, uri))) {
		g_free (uri);
		return BONFIRE_DISC_NOT_IN_TREE;
	}

	/* add a reference */
	ref = bonfire_data_disc_reference_new (disc, path);
	if (!disc->priv->references) {
		references = g_slist_prepend (NULL, GINT_TO_POINTER (ref));

		g_mutex_lock (disc->priv->references_lock);
		disc->priv->references = g_hash_table_new (g_str_hash,
							   g_str_equal);
		g_hash_table_insert (disc->priv->references,
				     dir->uri,
				     references);
		g_mutex_unlock (disc->priv->references_lock);
	}
	else {
		references = g_hash_table_lookup (disc->priv->references, uri);
		references = g_slist_prepend (references, GINT_TO_POINTER (ref));
		g_mutex_lock (disc->priv->references_lock);
		g_hash_table_insert (disc->priv->references, dir->uri, references);
		g_mutex_unlock (disc->priv->references_lock);
	}

	/* if this directory is waiting vfs exploration, then when 
	 * the exploration is finished, its disc will be exposed */
	if (g_slist_find (disc->priv->loading, dir))
		bonfire_data_disc_directory_priority (disc, dir);
	else
		bonfire_data_disc_expose_insert_path_real (disc, uri);

	g_free (uri);
	return BONFIRE_DISC_OK;
}

static void
bonfire_data_disc_row_collapsed_cb (GtkTreeView *tree,
				    GtkTreeIter *sortparent,
				    GtkTreePath *sortpath,
				    BonfireDataDisc *disc)
{
	char *parent_discpath;
	GtkTreeModel *model;
	GtkTreeModel *sort;
	GtkTreeIter parent;
	GtkTreeIter child;
	GtkTreePath *path;
	gboolean isdir;
	char *discpath;
	int explored;
	char *name;

	model = disc->priv->model;
	sort = disc->priv->sort;

	gtk_tree_model_sort_convert_iter_to_child_iter (GTK_TREE_MODEL_SORT (sort),
							&parent,
							sortparent);

	gtk_tree_store_set (GTK_TREE_STORE (model), &parent,
			    ROW_STATUS_COL, ROW_EXPANDED,
			    -1);

	if (gtk_tree_model_iter_children (model, &child, &parent) == FALSE)
		return;

	path = gtk_tree_model_get_path (model, &parent);
	bonfire_data_disc_tree_path_to_disc_path (disc,
						  path,
						  &parent_discpath);
	gtk_tree_path_free (path);

	do {
		gtk_tree_model_get (model, &child,
				    ISDIR_COL, &isdir,
				    ROW_STATUS_COL, &explored,
				    NAME_COL, &name, -1);

		if (explored != ROW_NOT_EXPLORED || !isdir) {
			g_free (name);
			continue;
		}

		discpath = g_strconcat (parent_discpath, "/", name, NULL);
		g_free (name);

		bonfire_data_disc_expose_path (disc, discpath);
		gtk_tree_store_set (GTK_TREE_STORE (model), &child,
				    ROW_STATUS_COL, ROW_EXPLORING,
				    -1);

		g_free (discpath);
	} while (gtk_tree_model_iter_next (model, &child));

	g_free(parent_discpath);
}
/************************** files, directories handling ************************/
static double
bonfire_data_disc_file_info (BonfireDataDisc *disc,
			     const char *uri,
			     GnomeVFSFileInfo *info)
{
	double retval = 0.0;
	BonfireFile *file;

	/* deal with symlinks */
	if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
		/* same as below for directories */
		if(!(file = g_hash_table_lookup (disc->priv->dirs, uri)))
			file = bonfire_data_disc_directory_new (disc,
								g_strdup (uri),
								TRUE);
	}
	else if (!g_hash_table_lookup (disc->priv->files, uri))
		retval = info->size;

	return retval;
} 

static void
bonfire_data_disc_obj_new (BonfireDataDisc *disc,
			   BonfireFile *file)
{
	gpointer value = NULL;
	gpointer key = NULL;

	if (disc->priv->excluded
	&&  g_hash_table_lookup_extended (disc->priv->excluded,
					  file->uri,
					  &key,
					  &value)) {
		g_hash_table_replace (disc->priv->excluded, file->uri, value);
		g_free (key);
	}

	if (disc->priv->restored
	&&  g_hash_table_lookup_extended (disc->priv->restored,
					  file->uri,
					  &key,
					  &value)) {
		g_mutex_lock (disc->priv->restored_lock);
		g_hash_table_replace (disc->priv->restored, file->uri, value);
		g_mutex_unlock (disc->priv->restored_lock);
		g_free (key);
	}
}

static BonfireFile *
bonfire_data_disc_file_new (BonfireDataDisc *disc,
			    const char *uri,
			    double size)
{
	BonfireFile *parent;
	BonfireFile *file;
	char *parent_uri;

	file = g_new0 (BonfireFile, 1);
	file->uri = g_strdup (uri);
	file->size = size;

#ifdef BUILD_INOTIFY
	file->handle.wd = -1;
#endif
	/* see if it needs monitoring */
	parent_uri = g_path_get_dirname (uri);
	parent = g_hash_table_lookup (disc->priv->dirs, parent_uri);
	g_free (parent_uri);

	if (!parent) {
#ifdef BUILD_INOTIFY
		bonfire_data_disc_start_monitoring (disc, file);
#endif
		bonfire_data_disc_size_changed (disc, size);
	}
	else if (bonfire_data_disc_is_excluded (disc, file->uri, NULL))
		bonfire_data_disc_size_changed (disc, size);
	else
		parent->size -= size;

	/* because of above we only insert it at the end */
	g_hash_table_insert(disc->priv->files, file->uri, file);

	bonfire_data_disc_obj_new (disc, file);
	return file;
}

struct _BonfireSymlinkChildrenData {
	BonfireDataDisc *disc;
	int size;
	char *uri;
	GSList *list;
};
typedef struct _BonfireSymlinkChildrenData BonfireSymlinkChildrenData;

static void
_foreach_replace_symlink_children_cb (char *symlink,
				      int value,
				      BonfireSymlinkChildrenData *data)
{
	/* symlink must be a child of uri 
	 * NOTE: can't be uri itself since we found it in dirs hash */
	if (strncmp (data->uri, symlink, data->size)
	|| (*(symlink + data->size) != '/' && *(symlink + data->size) != '\0'))
		return;

	data->list = g_slist_prepend (data->list, symlink);
}

static GSList *
bonfire_data_disc_symlink_get_uri_children (BonfireDataDisc *disc,
					    const char *uri)
{
	BonfireSymlinkChildrenData callback_data;

	if (!disc->priv->symlinks)
		return NULL;

	callback_data.disc = disc;
	callback_data.uri = (char *) uri;
	callback_data.size = strlen (uri);
	callback_data.list = NULL;

	g_hash_table_foreach (disc->priv->symlinks,
			      (GHFunc) _foreach_replace_symlink_children_cb,
			      &callback_data);

	return callback_data.list;
}

static gboolean
bonfire_data_disc_symlink_is_recursive (BonfireDataDisc *disc,
					const char *uri,
					const char *target)
{
	int size;
	char *symlink;
	gboolean result;
	GSList *symlinks;

	/* 1. we get a list of all the symlinks under the target
	 * 2. for each of their targets we check:
	 * - the target doesn't point to symlink or one of his parents
	 * - for each target we start back at 1
	 * 3. we stop if :
	 * - the target is a file not a directory
	 * - the target doesn't have symlinks children
	 * NOTE: if the target hasn't all of its subdirectories explored
	 * or if it's not explored itself it doesn't matter since when it
	 * will be explored for each of its symlinks we will do the same
	 * and if a symlink is recursive then we'll notice it then */

	symlinks = bonfire_data_disc_symlink_get_uri_children (disc, target);
	if (!symlinks)
		return FALSE;

	for (; symlinks; symlinks = g_slist_remove (symlinks, symlink)) {
		symlink = symlinks->data;

		target = g_hash_table_lookup (disc->priv->symlinks, symlink);

		if (!g_hash_table_lookup (disc->priv->dirs, target))
			continue;

		size = strlen (target);
		if (!strncmp (uri, target, size) &&  *(uri + size) == '/')
			goto recursive;

		result = bonfire_data_disc_symlink_is_recursive (disc,
								 uri,
								 target);
		if (result == TRUE)
			goto recursive;
	}

	return FALSE;

recursive:
	g_slist_free (symlinks);
	return TRUE;
}

static GSList *
bonfire_data_disc_symlink_new (BonfireDataDisc *disc,
			       const char *uri,
			       GnomeVFSFileInfo *info,
			       GSList *paths)
{
	BonfireFile *file = NULL;
	GSList *next;
	GSList *iter;
	char *path;

	/* we don't want paths overlapping already grafted paths.
	 * This might happen when we are loading a project or when
	 * we are notified of the creation of a new file */
	for (iter = paths; iter; iter = next) {
		next = iter->next;
		path = iter->data;

		if (g_hash_table_lookup (disc->priv->paths, path)) {
			paths = g_slist_remove (paths, path);
			g_free (path);
		}
	}

	if (!paths)
		goto end;

	/* make sure the target was explored or is about to be explored  */
	if ((file = g_hash_table_lookup (disc->priv->dirs, info->symlink_name))) {
		if (!g_slist_find (disc->priv->loading, file)) {
			bonfire_data_disc_restore_excluded_children (disc, file);
			bonfire_data_disc_replace_symlink_children (disc, file, paths);
		}
	}
	else if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
		file = bonfire_data_disc_directory_new (disc,
							g_strdup (info->symlink_name),
							FALSE);
	}
	else if (!(file = g_hash_table_lookup (disc->priv->files, info->symlink_name))) 
		file = bonfire_data_disc_file_new (disc,
						   info->symlink_name,
						   info->size);


end :
	if (!disc->priv->symlinks)
		disc->priv->symlinks = g_hash_table_new_full (g_str_hash,
							      g_str_equal,
							      (GDestroyNotify) g_free,
							      (GDestroyNotify) g_free);
	
	if (!g_hash_table_lookup (disc->priv->symlinks, uri))
		g_hash_table_insert (disc->priv->symlinks,
				     g_strdup (uri),
				     g_strdup (info->symlink_name));

	/* add graft points to the target */
	for (iter = paths; iter; iter = iter->next) {
		path = iter->data;
		bonfire_data_disc_graft_new (disc,
					     file->uri,
					     path);
	}

	return paths;
}

/* NOTE: it has a parent so if it is strictly excluded, it MUST have excluding */
#define EXPLORE_EXCLUDED_FILE(disc, uri)	\
	(!disc->priv->excluded	\
	 || !g_hash_table_lookup (disc->priv->excluded, uri) \
	 || !bonfire_data_disc_is_excluded (disc, uri, NULL))

static void
bonfire_data_disc_symlink_list_new (BonfireDataDisc *disc,
				    BonfireDirectoryContentsData *content,
				    const char *parent,
				    GSList *symlinks)
{
	GSList *iter;
	GSList *paths;
	char *current;
	GSList *grafts = NULL;
	GnomeVFSFileInfo *info;

	for (iter = symlinks; iter; iter = iter->next) {
		info = iter->data;
		content->infos = g_slist_remove (content->infos, info);

		if (disc->priv->unreadable
		&&  g_hash_table_lookup (disc->priv->unreadable, info->symlink_name))
			continue;

		current = g_strconcat (parent, "/", info->name, NULL);
		if (disc->priv->symlinks
		&&  g_hash_table_lookup (disc->priv->symlinks, current)) {
			g_free (current);
			continue;
		}

		if (info->type == GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK
		|| !bonfire_data_disc_is_readable (info)) {
			bonfire_data_disc_unreadable_new (disc,
							  current,
							  BONFIRE_FILTER_BROKEN_SYM);
			continue;
		}

		if (bonfire_data_disc_symlink_is_recursive (disc,
							    current,
							    info->symlink_name)) {
			bonfire_data_disc_unreadable_new (disc,
							  current,
							  BONFIRE_FILTER_RECURSIVE_SYM);
			continue;
		}

		paths = bonfire_data_disc_uri_to_paths (disc, current);
		paths = bonfire_data_disc_symlink_new (disc,
						       current,
						       info,
						       paths);
		g_free (current);

		grafts = g_slist_concat (grafts, paths);
	}

	if (grafts) {
		if (disc->priv->references
		&&  g_hash_table_lookup (disc->priv->references, parent))
			bonfire_data_disc_expose_grafted (disc, grafts);

		g_slist_foreach (grafts, (GFunc) g_free, NULL);
		g_slist_free (grafts);
	}

	g_slist_foreach (symlinks, (GFunc) gnome_vfs_file_info_clear, NULL);
	g_slist_foreach (symlinks, (GFunc) gnome_vfs_file_info_unref, NULL);
	g_slist_free (symlinks);
}

static gboolean
bonfire_data_disc_load_result (GObject *object, gpointer data)
{
	BonfireDirectoryContentsData *callback_data = data;
	BonfireDataDisc *disc = BONFIRE_DATA_DISC (object);
	GSList *symlinks = NULL;
	GnomeVFSFileInfo *info;
	double dir_size = 0.0;
	BonfireFile *dir;
	double diffsize;
	char *current;
	GSList *next;
	GSList *iter;

	bonfire_data_disc_decrease_activity_counter (disc);

	/* process errors */
	if (callback_data->errors) {
		bonfire_data_disc_load_dir_error (disc, callback_data->errors);

		/* we don't want the errors to be processed twice */
		g_slist_foreach (callback_data->errors,
				 (GFunc) _free_dir_contents_error,
				 NULL);
		g_slist_free (callback_data->errors);
		callback_data->errors = NULL;

		/* the following means that we couldn't open the directory */
		if (!callback_data->uri)
			return TRUE;
	}

	dir = g_hash_table_lookup (disc->priv->dirs, callback_data->uri);
	if (!dir)
		return TRUE;

	disc->priv->loading = g_slist_remove (disc->priv->loading, dir);
	disc->priv->rescan = g_slist_remove (disc->priv->rescan, dir);
	for (iter = callback_data->infos; iter; iter = next) {
		info = iter->data;
		next = iter->next;
		
		if(GNOME_VFS_FILE_INFO_SYMLINK (info)) {
			symlinks = g_slist_prepend (symlinks, info);
			continue;
		}

		current = g_strconcat (dir->uri, "/", info->name, NULL);
		if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
			if (!g_hash_table_lookup (disc->priv->dirs, current)
			&&  EXPLORE_EXCLUDED_FILE (disc, current)) {
				bonfire_data_disc_directory_new (disc,
								 g_strdup (current),
								 TRUE);
			}
		}
		else if (!g_hash_table_lookup (disc->priv->files, current)
		     &&  EXPLORE_EXCLUDED_FILE (disc, current)) {
			dir_size += info->size;
		}

		g_free (current);
	}

	diffsize = dir_size - dir->size;
	if (diffsize) {
		dir->size += diffsize;
		bonfire_data_disc_size_changed (disc, diffsize);
	}

	/* we need to check that they are not symlinks */
	bonfire_data_disc_symlink_list_new (disc,
					    callback_data,
					    dir->uri,
					    symlinks);

	if (disc->priv->references
	&&  g_hash_table_lookup (disc->priv->references, dir->uri)) {
		if (!callback_data->infos) {
			/* empty directory to be exposed */
			bonfire_data_disc_increase_activity_counter (disc);
			BONFIRE_ADD_TO_EXPOSE_QUEUE (disc, callback_data);
			return FALSE;
		}

		/* make sure that when we explored we asked for mime types :
		 * a directory could be explored and then its parent expanded
		 * which adds references but unfortunately we won't have any
		 * mime types */
		info = callback_data->infos->data;
		if (info->mime_type) {
			/* there are references put that in the queue for it to be exposed */
			bonfire_data_disc_increase_activity_counter (disc);
			BONFIRE_ADD_TO_EXPOSE_QUEUE (disc, callback_data);
			return FALSE;
		}

		bonfire_data_disc_expose_insert_path_real (disc, callback_data->uri);
	}

	return TRUE;
}

static void
bonfire_data_disc_load_dir_error (BonfireDataDisc *disc, GSList *errors)
{
	BonfireLoadDirError *error;
	BonfireFile *file;
	char *parent;

	for (; errors; errors = errors->next) {
		error = errors->data;

		/* remove it wherever it is */
		if (disc->priv->unreadable
		&&  g_hash_table_lookup (disc->priv->unreadable, error->uri))
			continue;

		if ((file = g_hash_table_lookup (disc->priv->dirs, error->uri)))
			bonfire_data_disc_directory_remove_from_tree (disc, file);
		else if ((file = g_hash_table_lookup (disc->priv->files, error->uri)))
			bonfire_data_disc_file_remove_from_tree (disc, file);
		else if (disc->priv->symlinks
		     &&  g_hash_table_lookup (disc->priv->symlinks, error->uri)) {
			g_hash_table_remove (disc->priv->symlinks, error->uri);
		
			if (!g_hash_table_size (disc->priv->symlinks)) {
				g_hash_table_destroy (disc->priv->symlinks);
				disc->priv->symlinks = NULL;
			}
		}
		else if (!bonfire_data_disc_is_excluded (disc, error->uri, NULL))
			bonfire_data_disc_remove_uri_from_tree (disc, error->uri);

		/* insert it in the unreadable hash table if need be */
		parent = g_path_get_dirname (error->uri);
		if (g_hash_table_lookup (disc->priv->dirs, parent)
		&&  error->status != BONFIRE_FILTER_UNKNOWN) {
			bonfire_data_disc_unreadable_new (disc,
							  error->uri,
							  error->status);
			error->uri = NULL;
		}

		g_free (parent);
	}
}

gboolean
_foreach_clear_joliet_hash_cb (char *key,
			       gpointer value,
			       gpointer null_pointer)
{
	if (value == GINT_TO_POINTER (1))
		g_free (key);

	return TRUE;
}

struct _BonfireCheckRestoredData {
	BonfireFilterStatus status;
	BonfireDataDisc *disc;
	char *uri;
};
typedef struct _BonfireCheckRestoredData BonfireCheckRestoredData;

static gboolean
_check_for_restored_modify_state (BonfireCheckRestoredData *data)
{
	g_hash_table_insert (data->disc->priv->restored,
			     data->uri,
			     GINT_TO_POINTER (data->status));

	if (!data->disc->priv->filter_dialog)
		goto end;

	bonfire_filtered_dialog_add (BONFIRE_FILTERED_DIALOG (data->disc->priv->filter_dialog),
				     data->uri,
				     TRUE,
				     data->status);

end:
	g_free (data->uri);
	g_free (data);
	return FALSE;
}

static gboolean
_check_for_restored (BonfireDataDisc *disc,
		     const char *uri,
		     BonfireFilterStatus status) {
	BonfireFilterStatus current_status;
	gboolean retval;

	retval = FALSE;
	g_mutex_lock (disc->priv->restored_lock);
	if (!disc->priv->restored)
		goto end;

	current_status = GPOINTER_TO_INT (g_hash_table_lookup (disc->priv->restored, uri));
	if (!current_status)
		goto end;

	if (status == BONFIRE_FILTER_UNREADABLE
	||  status == BONFIRE_FILTER_RECURSIVE_SYM) {
		g_mutex_unlock (disc->priv->restored_lock);
		bonfire_data_disc_restored_free (disc, uri);		
		goto end;
	}

	if (current_status == BONFIRE_FILTER_UNKNOWN) {
		BonfireCheckRestoredData *data;

		data = g_new0 (BonfireCheckRestoredData, 1);
		data->disc = disc;
		data->uri = g_strdup (uri);
		data->status = status;

		g_idle_add ((GSourceFunc) _check_for_restored_modify_state,
			    data);
	}
	retval = TRUE;


end:
	g_mutex_unlock (disc->priv->restored_lock);
	return retval;
}

static gboolean
bonfire_data_disc_load_thread (GObject *object, gpointer data)
{
	BonfireDirectoryContentsData *callback_data = data;
	BonfireDataDisc *disc = BONFIRE_DATA_DISC (object);
	BonfireLoadDirError *error = NULL;
	GnomeVFSDirectoryHandle *handle;
	GHashTable *joliet_hash = NULL;
	GnomeVFSFileInfoOptions flags;
	GnomeVFSFileInfo *info;
	GnomeVFSResult result;
	char *escaped_uri;
	GSList *infos;
	char *current;

	flags = GNOME_VFS_FILE_INFO_GET_ACCESS_RIGHTS;

	g_mutex_lock (disc->priv->references_lock);
	if (disc->priv->references
	&&  g_hash_table_lookup (disc->priv->references, callback_data->uri))
		flags |= GNOME_VFS_FILE_INFO_GET_MIME_TYPE|
			 GNOME_VFS_FILE_INFO_FORCE_SLOW_MIME_TYPE;
	g_mutex_unlock (disc->priv->references_lock);

	handle = NULL;
	escaped_uri = gnome_vfs_escape_host_and_path_string (callback_data->uri);
	result = gnome_vfs_directory_open (&handle,
					   escaped_uri,
					   flags);
	g_free (escaped_uri);

	if (result != GNOME_VFS_OK) {
		error = g_new0 (BonfireLoadDirError, 1);
		error->uri = callback_data->uri;
		callback_data->uri = NULL;

		if (result != GNOME_VFS_ERROR_NOT_FOUND) {
			_check_for_restored (disc,
					     error->uri,
					     BONFIRE_FILTER_UNREADABLE);

			error->status = BONFIRE_FILTER_UNREADABLE;
		}
		else
			error->status = BONFIRE_FILTER_UNKNOWN;

		callback_data->errors = g_slist_prepend (callback_data->errors, error);

		if (handle) {
			gnome_vfs_directory_close (handle);
			handle = NULL;
		}

		g_warning ("Can't open directory %s : %s\n",
			   error->uri,
			   gnome_vfs_result_to_string (result));
		return TRUE;
	}

	infos = NULL;
	info = gnome_vfs_file_info_new();
	while (gnome_vfs_directory_read_next (handle, info) == GNOME_VFS_OK) {
		if (*info->name == '.' && (info->name[1] == 0
		||  (info->name[1] == '.' && info->name[2] == 0))) {
			gnome_vfs_file_info_clear(info);
			continue;
		}

		current = g_strconcat (callback_data->uri, "/", info->name, NULL);

		if (!bonfire_data_disc_is_readable (info)) {
			_check_for_restored (disc,
					     current,
					     BONFIRE_FILTER_UNREADABLE);

			error = g_new0 (BonfireLoadDirError, 1);
 			error->uri = current;
 			error->status = BONFIRE_FILTER_UNREADABLE;
			callback_data->errors = g_slist_prepend (callback_data->errors, error);

 			gnome_vfs_file_info_clear (info);
			continue;
		}

		if (info->type == GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK) {
			gnome_vfs_file_info_clear (info);

			if(!bonfire_utils_get_symlink_target (current, info, flags)) {
				BonfireFilterStatus status;

				if (info->symlink_name)
					status = BONFIRE_FILTER_RECURSIVE_SYM;
				else
					status = BONFIRE_FILTER_BROKEN_SYM;

				if (!_check_for_restored (disc, current, status)) {
					error = g_new0 (BonfireLoadDirError, 1);
					error->uri = current;
					error->status = status;
					callback_data->errors = g_slist_prepend (callback_data->errors, error);

					gnome_vfs_file_info_clear (info);
					continue;
				}
			}
		}

		/* a new hidden file ? */
		if (*info->name == '.'
		&& !_check_for_restored (disc, current, BONFIRE_FILTER_HIDDEN)) {
 			error = g_new0 (BonfireLoadDirError, 1);
 			error->uri = current;
 			error->status = BONFIRE_FILTER_HIDDEN;
			callback_data->errors = g_slist_prepend (callback_data->errors, error);
 
 			gnome_vfs_file_info_clear (info);
			continue;
		}

		/* see if the name of the file is more than 64 bytes long
		 * since microsoft Joliet would then truncate it. if there
		 * is in the same directory a file which has go the same
		 * first 64 bytes then the user will have to drop the 
		 * Joliet compatibility. if two files happen to have the 
		 * same first 64 bytes we filter them out */
		if (strlen (info->name) >= 64) {
			GnomeVFSFileInfo *twin;

			if (!joliet_hash) {
				joliet_hash = g_hash_table_new (bonfire_utils_str_hash_64,
								bonfire_utils_str_equal_64);
			}

			if ((twin = g_hash_table_lookup (joliet_hash, info->name))) {

				if (twin != GINT_TO_POINTER (1)) {
					char *twin_uri;

					twin_uri = g_strconcat (callback_data->uri, "/", twin->name, NULL);

					if (!_check_for_restored (disc,
								  twin_uri,
								  BONFIRE_FILTER_JOLIET_INCOMPAT)) {
						error = g_new0 (BonfireLoadDirError, 1);
						error->uri = twin_uri;
						error->status = BONFIRE_FILTER_JOLIET_INCOMPAT;
						callback_data->errors = g_slist_prepend (callback_data->errors, error);

						infos = g_slist_remove (infos, twin);
						g_hash_table_insert (joliet_hash,
								     twin->name,
								     GINT_TO_POINTER (1));
	
						twin->name = NULL;
						gnome_vfs_file_info_clear (twin);
						gnome_vfs_file_info_unref (twin);
					}
					else
						g_free (twin_uri);
				}

				if (!_check_for_restored (disc,
							  current,
							  BONFIRE_FILTER_JOLIET_INCOMPAT)) {
					error = g_new0 (BonfireLoadDirError, 1);
					error->uri = current;
					error->status = BONFIRE_FILTER_JOLIET_INCOMPAT;
					callback_data->errors = g_slist_prepend (callback_data->errors, error);
					
					gnome_vfs_file_info_clear (info);
					continue;
				}
			}
			else {
				_check_for_restored (disc,
						     current,
						     BONFIRE_FILTER_JOLIET_INCOMPAT);
	
				g_hash_table_insert (joliet_hash,
						     info->name,
						     info);
			}
		}

		g_free (current);
		infos = g_slist_prepend (infos, info);
		info = gnome_vfs_file_info_new ();
	}

	gnome_vfs_file_info_unref (info);
	gnome_vfs_directory_close (handle);
	callback_data->infos = infos;

	if (joliet_hash) {
		if (g_hash_table_size (joliet_hash))
			g_hash_table_foreach_remove (joliet_hash,
						     (GHRFunc) _foreach_clear_joliet_hash_cb,
						     NULL);
		g_hash_table_destroy (joliet_hash);
	}

	return TRUE;
}

static BonfireDiscResult
bonfire_data_disc_directory_load (BonfireDataDisc *disc,
				  BonfireFile *dir,
				  gboolean append)
{
	BonfireDirectoryContentsData *callback_data;

	/* start exploration */
	disc->priv->loading = g_slist_prepend (disc->priv->loading, dir);

	if (!disc->priv->jobs)
		disc->priv->jobs = bonfire_async_job_manager_get_default ();

	if (!disc->priv->load_type) {
		disc->priv->load_type = bonfire_async_job_manager_register_type (disc->priv->jobs,
										 G_OBJECT (disc),
										 bonfire_data_disc_load_thread,
										 bonfire_data_disc_load_result,
										 bonfire_data_disc_dir_contents_destroy,
										 bonfire_data_disc_dir_contents_cancel);
	}

	callback_data = g_new0 (BonfireDirectoryContentsData, 1);
	callback_data->uri = g_strdup (dir->uri);

	if (!bonfire_async_job_manager_queue (disc->priv->jobs,
					      disc->priv->load_type,
					      callback_data))
		return BONFIRE_DISC_ERROR_THREAD;

	bonfire_data_disc_increase_activity_counter (disc);
	return BONFIRE_DISC_OK;
}

static BonfireFile *
bonfire_data_disc_directory_new (BonfireDataDisc *disc,
				 char *uri,
				 gboolean append)
{
	BonfireFile *dir;

	dir = g_new0 (BonfireFile, 1);
	dir->uri = uri;
	dir->size = 0.0;

#ifdef BUILD_INOTIFY
	dir->handle.wd = -1;
	bonfire_data_disc_start_monitoring (disc, dir);
#endif

	g_hash_table_insert (disc->priv->dirs, dir->uri, dir);
	bonfire_data_disc_obj_new (disc, dir);
	bonfire_data_disc_directory_load (disc, dir, append);
	return dir;
}

static gboolean
bonfire_data_disc_directory_priority_cb (gpointer data, gpointer user_data)
{
	BonfireDirectoryContentsData *callback_data = data;
	BonfireFile *dir = user_data;

	if (!strcmp (dir->uri, callback_data->uri))
		return TRUE;

	return FALSE;
}

static void
bonfire_data_disc_directory_priority (BonfireDataDisc *disc,
				      BonfireFile *dir)
{
	bonfire_async_job_manager_find_urgent_job (disc->priv->jobs,
						   disc->priv->load_type,
						   bonfire_data_disc_directory_priority_cb,
						   dir);
}

/******************************* Row removal ***********************************/
static void
bonfire_data_disc_remove_row_in_dirs_hash (BonfireDataDisc *disc,
					   BonfireFile *dir,
					   const char *path)
{
	GSList *grafts;

	/* remove all the children graft point of this path */
	grafts = g_slist_append (NULL, (gpointer) path);
	bonfire_data_disc_graft_children_remove (disc, grafts);
	g_slist_free (grafts);

	/* remove graft point if path == graft point */
	if (!bonfire_data_disc_graft_remove (disc, path)) {
		char *graft;

		/* otherwise we exclude dir */
		graft = bonfire_data_disc_graft_get (disc, path);
		bonfire_data_disc_exclude_uri (disc, graft, dir->uri);
		g_free (graft);

		if (bonfire_data_disc_is_excluded (disc, dir->uri, NULL))
			bonfire_data_disc_remove_dir_and_children (disc, dir);
		return;
	}
}

static void
bonfire_data_disc_remove_row_in_files_hash (BonfireDataDisc *disc,
					    BonfireFile *file,
					    const char *path)
{
	/* see if path == graft point. If so, remove it */
	if (!bonfire_data_disc_graft_remove (disc, path)) {
		char *graft;

		/* the path was not of the graft points of the file so 
		 * it has a parent graft point, find it and exclude it */
		graft = bonfire_data_disc_graft_get (disc, path);
		bonfire_data_disc_exclude_uri (disc, graft, file->uri);
		g_free (graft);
	}
}

static void
bonfire_data_disc_delete_row_cb (BonfireDataDisc *disc,
				 GSList *results,
				 gpointer null_data)
{
	BonfireInfoAsyncResult *result;
	BonfireFile *parent;
	GnomeVFSFileInfo *info;
	char *parenturi;
	char *uri;

	for (; results; results = results->next) {
		result = results->data;

		uri = result->uri;
		info = result->info;

		parenturi = g_path_get_dirname (uri);
		parent = g_hash_table_lookup (disc->priv->dirs, parenturi);
		g_free (parenturi);
		if (!parent)
			continue;

		if (result->result == GNOME_VFS_ERROR_NOT_FOUND) {
			bonfire_data_disc_add_rescan (disc, parent);
			continue;
		}

		if (result->result == GNOME_VFS_ERROR_LOOP) {
			bonfire_data_disc_unreadable_new (disc,
							  g_strdup (uri),
							  BONFIRE_FILTER_RECURSIVE_SYM);
			bonfire_data_disc_add_rescan (disc, parent);
			continue;
		}

		if (result->result != GNOME_VFS_OK
		||  !bonfire_data_disc_is_readable (info)) {
			bonfire_data_disc_unreadable_new (disc,
							  g_strdup (uri),
							  BONFIRE_FILTER_UNREADABLE);
			bonfire_data_disc_add_rescan (disc, parent);
			continue;
		}
		bonfire_data_disc_size_changed (disc, info->size * (-1.0));
		parent->size -= info->size;
	}
}

static void
bonfire_data_disc_remove_path (BonfireDataDisc *disc,
			       const char *path)
{
	BonfireFile *file;
	char *uri;

	bonfire_data_disc_remove_references (disc, path);

	/* uri can be NULL if uri is an empty directory
	 * added or if the row hasn't been loaded yet */
	uri = bonfire_data_disc_path_to_uri (disc, path);
	if (!uri) {
		gpointer value = NULL;

		g_hash_table_lookup_extended (disc->priv->paths,
					      path,
					      &value,
					      NULL);

		if (value) {
			GSList *paths;

			paths = g_slist_append (NULL, (char *) value);
			bonfire_data_disc_graft_children_remove (disc, paths);
			g_slist_free (paths);

			g_hash_table_remove (disc->priv->paths, value);
			g_free (value);
		}
		return;
	}

	/* see if this file is not already in the hash tables */
	if ((file = g_hash_table_lookup (disc->priv->dirs, uri))) {
		bonfire_data_disc_remove_row_in_dirs_hash (disc,
							   file,
							   path);
	} 
	else if ((file = g_hash_table_lookup (disc->priv->files, uri))) {
		bonfire_data_disc_remove_row_in_files_hash (disc,
							    file,
							    path);
	}
	else {
		char *graft;

		/* exclude it from parent */
		graft = bonfire_data_disc_graft_get (disc, path);
		bonfire_data_disc_exclude_uri (disc, graft, uri);
		g_free (graft);

		/* if it is excluded in all parent graft points exclude it
		 * and update the selection and the parent size */
		if (bonfire_data_disc_is_excluded (disc, uri, NULL)) {
			GSList *uris;

			uris = g_slist_prepend (NULL, (char*) uri);
			bonfire_data_disc_get_info_async (disc,
							  uris, 
							  0,
							  bonfire_data_disc_delete_row_cb,
							  NULL);
			g_slist_free (uris);
		}
	}
	g_free (uri);
}

static void
bonfire_data_disc_delete_selected (BonfireDisc *disc)
{
	GtkTreeSelection *selection;
	GtkTreePath *realpath;
	GtkTreePath *treepath;
	BonfireDataDisc *data;
	GtkTreeModel *model;
	GtkTreeModel *sort;
	GList *list, *iter;
	GtkTreeIter row;
	char *discpath;

	data = BONFIRE_DATA_DISC (disc);
	if (data->priv->is_loading)
		return;

	model = data->priv->model;
	sort = data->priv->sort;
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (data->priv->tree));

	/* we must start by the end for the treepaths to point to valid rows */
	list = gtk_tree_selection_get_selected_rows (selection, &sort);
	list = g_list_reverse (list);

	for (iter = list; iter; iter = iter->next) {
		treepath = iter->data;

		realpath = gtk_tree_model_sort_convert_path_to_child_path (GTK_TREE_MODEL_SORT (sort),
									   treepath);
		gtk_tree_path_free (treepath);

		bonfire_data_disc_tree_path_to_disc_path (data,
							  realpath,
							  &discpath);

		if (gtk_tree_model_get_iter (model, &row, realpath)) {
			GtkTreeIter parent;
			int status;

			gtk_tree_model_get (model, &row,
 					    ROW_STATUS_COL, &status,
 					    -1);
  
 			if (status != ROW_BOGUS) {
 				gboolean is_valid;
 
 				is_valid = gtk_tree_model_iter_parent (model, &parent, &row);
  				gtk_tree_store_remove (GTK_TREE_STORE (model), &row);
 	
 				if (is_valid)
 					bonfire_data_disc_tree_update_directory_real (data, &parent);
 
  				if (status != ROW_NEW)
  					bonfire_data_disc_remove_path (data, discpath);
  			}
  		}

 		gtk_tree_path_free (realpath);
		g_free (discpath);
	}

	g_list_free (list);

	if (g_hash_table_size (data->priv->paths) == 0)
		bonfire_data_disc_selection_changed (data, FALSE);
	else
		bonfire_data_disc_selection_changed (data, TRUE);
}

static void
bonfire_data_disc_clear (BonfireDisc *disc)
{
	BonfireDataDisc *data;

	data = BONFIRE_DATA_DISC (disc);
	if (data->priv->is_loading)
		return;

	gtk_tree_store_clear (GTK_TREE_STORE (data->priv->model));
	bonfire_data_disc_reset_real (data);

	bonfire_disc_size_changed (disc, 0.0, NULL);
	bonfire_data_disc_selection_changed (data, FALSE);
}

static void
bonfire_data_disc_reset (BonfireDisc *disc) {
	BonfireDataDisc *data;

	data = BONFIRE_DATA_DISC (disc);
	if (data->priv->is_loading)
		return;

	gtk_tree_store_clear (GTK_TREE_STORE (data->priv->model));
	bonfire_data_disc_reset_real (data);

	bonfire_disc_size_changed (disc, 0.0, NULL);
	gtk_notebook_set_current_page (GTK_NOTEBOOK (BONFIRE_DATA_DISC (disc)->priv->notebook), 0);
	bonfire_data_disc_selection_changed (data, FALSE);
}

/*************************************** new row *******************************/
struct _BonfireRestoreChildrenData {
	int size;
	char *uri;
	GSList *list;
	BonfireDataDisc *disc;
};
typedef struct _BonfireRestoreChildrenData BonfireRestoreChildrenData;

static void
bonfire_data_disc_restore_excluded_children_cb (BonfireDataDisc *disc,
						GSList *results,
						char *dir_uri)
{
	int size;
	char *uri;
	GnomeVFSFileInfo *info;
	BonfireInfoAsyncResult *result;

	if (!g_hash_table_lookup (disc->priv->dirs, dir_uri)) {
		g_free (dir_uri);
		return;
	}

	for (;results ; results = results->next) {
		result = results->data;
		uri = result->uri;

		/* see if it has not been excluded in between */
		if (bonfire_data_disc_is_excluded (disc, uri, NULL))
			continue;

		/* as an excluded file it is not in the hashes
		 * and its size is not taken into account */
		if (result->result == GNOME_VFS_ERROR_NOT_FOUND) {
			bonfire_data_disc_remove_uri_from_tree (disc, uri);
			continue;
		}

		info = result->info;
		if (result->result == GNOME_VFS_ERROR_LOOP) {
			bonfire_data_disc_remove_uri_from_tree (disc, uri);
			bonfire_data_disc_unreadable_new (disc,
							  g_strdup (uri),
							  BONFIRE_FILTER_RECURSIVE_SYM);
			continue;
		}

		if (result->result != GNOME_VFS_OK
		||  !bonfire_data_disc_is_readable (info)) {
			bonfire_data_disc_remove_uri_from_tree (disc, uri);
			bonfire_data_disc_unreadable_new (disc,
							  g_strdup (uri),
							  BONFIRE_FILTER_UNREADABLE);
			continue;
		}

		size = bonfire_data_disc_file_info (disc, uri, info);
		if (size) {
			char *parent;
			BonfireFile *dir;
	
			parent = g_path_get_dirname (uri);
			dir = g_hash_table_lookup (disc->priv->dirs, parent);
			g_free (parent);
	
			dir->size += size;
			bonfire_data_disc_size_changed (disc, size);
		}
	}

	g_free (dir_uri);
}

static void
_foreach_restore_strictly_excluded_children_cb (char *key,
						GSList *grafts,
						BonfireRestoreChildrenData *data)
{
	char *parent;

	/* keep only the children of data->uri */
	if (strncmp (data->uri, key, data->size)
	|| *(key + data->size) != '/')
		return;

	/* keep only those that are stricly excluded by all parent graft points */
	if (!bonfire_data_disc_is_excluded (data->disc, key, NULL))
		return;

	/* keep only those who have a parent. As for the others when the parent
	 * will be restored later (it has to be a directory), at this point 
	 * they won't be excluded any more and will be "naturally restored" */
	parent = g_path_get_dirname (key);
	if (g_hash_table_lookup (data->disc->priv->dirs, parent))
		data->list = g_slist_prepend (data->list, key);
	g_free (parent);
}

static void
bonfire_data_disc_restore_excluded_children (BonfireDataDisc *disc,
					     BonfireFile *dir)
{
	BonfireRestoreChildrenData callback_data;
	BonfireDiscResult result;
	char *dir_uri;

	if (!disc->priv->excluded)
		return;

	callback_data.disc = disc;
	callback_data.uri = dir->uri;
	callback_data.size = strlen (dir->uri);
	callback_data.list = NULL;

	/* find all children excluded by all parent graft points and restore them */
	g_hash_table_foreach (disc->priv->excluded,
			      (GHFunc) _foreach_restore_strictly_excluded_children_cb,
			      &callback_data);

	if (!callback_data.list)
		return;

	dir_uri = g_strdup (dir->uri);
	result = bonfire_data_disc_get_info_async (disc,
						   callback_data.list,
						   GNOME_VFS_FILE_INFO_GET_ACCESS_RIGHTS,
						   (BonfireInfoAsyncResultFunc) bonfire_data_disc_restore_excluded_children_cb,
						   dir_uri);
	g_slist_free (callback_data.list);

	if (result != BONFIRE_DISC_OK)
		g_free (dir_uri);
}

struct _ReplaceSymlinkChildrenData {
	GSList *grafts;
	char *parent;
};
typedef struct _ReplaceSymlinkChildrenData ReplaceSymlinkChildrenData;

static GSList *
bonfire_data_disc_get_target_grafts (BonfireDataDisc *disc,
				     const char *sym_parent,
				     GSList *grafts,
				     const char *symlink)
{
	int size;
	char *path;
	char *graft;
	GSList *newgrafts = NULL;

	size = strlen (sym_parent);

	/* we add the graft point */
	for (; grafts; grafts = grafts->next) {
		graft = grafts->data;
		path = g_strconcat (graft, symlink + size, NULL);
		newgrafts = g_slist_append (newgrafts, path);
	}

	return newgrafts;
}

static void
bonfire_data_disc_replace_symlink_children_cb (BonfireDataDisc *disc,
					       GSList *results,
					       ReplaceSymlinkChildrenData *callback_data)
{
	BonfireInfoAsyncResult *result;
	BonfireFile *file;
	GnomeVFSFileInfo *info;
	GSList *grafts;
	GSList *paths;
	char *symlink;
	char *target;
	char *parent;

	grafts = bonfire_data_disc_reference_get_list (disc,
						       callback_data->grafts,
						       TRUE);

	if (!grafts || !g_hash_table_lookup (disc->priv->dirs, callback_data->parent))
		goto cleanup;

	parent = callback_data->parent;

	for (; results; results = results->next) {
		result = results->data;

		info = result->info;
		symlink = result->uri;
		target = info->symlink_name;

		if (result->result != GNOME_VFS_OK
		||  !bonfire_data_disc_is_readable (info))
			continue;

		/* if target is in unreadable remove it */
		if (disc->priv->unreadable
		&&  g_hash_table_lookup (disc->priv->unreadable, target))
			bonfire_data_disc_unreadable_free (disc, target);

		paths = bonfire_data_disc_get_target_grafts (disc,
							     parent,
							     grafts,
							     symlink);

		if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
			if (!(file = g_hash_table_lookup (disc->priv->dirs, target)))
				file = bonfire_data_disc_directory_new (disc,
								        g_strdup (target),
									FALSE);

			paths = bonfire_data_disc_graft_new_list (disc,
								  target,
								  paths);

			bonfire_data_disc_restore_excluded_children (disc,
								     file);
			bonfire_data_disc_replace_symlink_children (disc,
								    file,
								    paths);

		}
		else {
			if (!g_hash_table_lookup (disc->priv->files, target))
				file = bonfire_data_disc_file_new (disc,
								   target,
								   info->size);

			paths = bonfire_data_disc_graft_new_list (disc,
								  target,
								  paths);
		}

		g_slist_foreach (paths, (GFunc) g_free, NULL);
		g_slist_free (paths);
	}

cleanup:

	g_free (callback_data->parent);
	g_free (callback_data);

	g_slist_foreach (grafts, (GFunc) g_free, NULL);
	g_slist_free (grafts);
}

static void
bonfire_data_disc_replace_symlink_children (BonfireDataDisc *disc,
					    BonfireFile *dir,
					    GSList *grafts)
{
	ReplaceSymlinkChildrenData *async_data;
	BonfireDataDiscReference ref;
	BonfireDiscResult result;
	GSList *list;

	list = bonfire_data_disc_symlink_get_uri_children (disc, dir->uri);

	if (!list)
		return;

	async_data = g_new0 (ReplaceSymlinkChildrenData, 1);
	async_data->parent = g_strdup (dir->uri);

	for (; grafts; grafts = grafts->next) {
		char *graft;

		graft = grafts->data;
		ref = bonfire_data_disc_reference_new (disc, graft);
		async_data->grafts = g_slist_prepend (async_data->grafts,
						      GINT_TO_POINTER (ref));
	}

	result = bonfire_data_disc_get_info_async (disc,
						   list,
						   GNOME_VFS_FILE_INFO_GET_ACCESS_RIGHTS,
						   (BonfireInfoAsyncResultFunc) bonfire_data_disc_replace_symlink_children_cb,
						   async_data);
	g_slist_free (list);

	if (result != BONFIRE_DISC_OK) {
		for (; async_data->grafts; async_data->grafts = g_slist_remove (async_data->grafts, GINT_TO_POINTER (ref))) {
			ref = GPOINTER_TO_INT (async_data->grafts->data);
			bonfire_data_disc_reference_free (disc, ref);
		}
		g_free (async_data->parent);
		g_free (async_data);
	}
}

struct _BonfireGraftInfoData {
	BonfireDataDiscReference ref;
	GSList *excluded;
};
typedef struct _BonfireGraftInfoData BonfireGraftInfoData;

#define FREE_GRAFT_INFO(disc, graft_info) \
	{ \
		g_slist_foreach (graft_info->excluded, (GFunc) g_free, NULL); \
		g_slist_free (graft_info->excluded); \
		bonfire_data_disc_reference_free (disc, graft_info->ref); \
		g_free (graft_info); \
	}

static char *
bonfire_data_disc_new_file (BonfireDataDisc *disc,
			    const char *uri,
			    const char *path,
			    const GnomeVFSFileInfo *info)
{
	char *graft;

	if (bonfire_data_disc_original_parent (disc, uri, path)) {
		if (bonfire_data_disc_is_excluded (disc, uri, NULL)) {
			BonfireFile *parent;
			char *parent_uri;
	
			parent_uri = g_path_get_dirname (uri);
			parent = g_hash_table_lookup (disc->priv->dirs, parent_uri);
			parent->size += info->size;
			bonfire_data_disc_size_changed (disc, info->size);
			g_free (parent_uri);
		}

		graft = bonfire_data_disc_graft_get (disc, path);
		bonfire_data_disc_restore_uri (disc,
					       graft,
					       uri);
	}
	else {
		bonfire_data_disc_file_new (disc,
					    uri,
					    info->size);

		bonfire_data_disc_graft_new (disc,
					     uri,
					     path);
		graft = g_strdup (path);
	}

	return graft;
}

static char *
bonfire_data_disc_new_row_added (BonfireDataDisc *disc,
				 const char *uri,
				 const char *path)
{
	char *graft = NULL;

	/* create and add a graft point if need be, that is if the file wasn't
	 * added to a directory which is its parent in the file system */
	if (bonfire_data_disc_original_parent (disc, uri, path)) {
		graft = bonfire_data_disc_graft_get (disc, path);
		bonfire_data_disc_restore_uri (disc,
					       graft,
					       uri);
	}
	else {
		bonfire_data_disc_graft_new (disc,
					     uri,
					     path);
		graft = g_strdup (path);
	}

	return graft;
}

static BonfireDiscResult
bonfire_data_disc_new_row_real (BonfireDataDisc *disc,
				BonfireInfoAsyncResult *result,
				const char *path,
				GSList *excluded)
{
	BonfireFilterStatus status;
	GnomeVFSFileInfo *info;
	char *excluded_uri;
	BonfireFile *file;
	GSList *iter;
	char *graft;
	char *uri;

	info = result->info;
	if (result->result != GNOME_VFS_OK || !bonfire_data_disc_is_readable (info)) {
		bonfire_data_disc_tree_remove_path (disc, path);
		bonfire_data_disc_unreadable_dialog (disc,
						     result->uri,
						     result->result,
						     FALSE);
		return BONFIRE_DISC_ERROR_UNREADABLE;
	}

	uri = g_strdup (result->uri);

	if (GNOME_VFS_FILE_INFO_SYMLINK (info)) {
		if (info->type == GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK) {
			bonfire_data_disc_tree_remove_path (disc, path);
			bonfire_data_disc_unreadable_dialog (disc,
							     uri,
							     GNOME_VFS_ERROR_TOO_MANY_LINKS,
							     FALSE);
			return BONFIRE_DISC_BROKEN_SYMLINK;
		}

		g_free (uri);
		uri = g_strdup (info->symlink_name);
	}

	if (info->type != GNOME_VFS_FILE_TYPE_DIRECTORY) {
		if (!g_hash_table_lookup (disc->priv->files, uri)) {
			graft = bonfire_data_disc_new_file (disc,
							    uri,
							    path,
							    info);
		}
		else
			graft = bonfire_data_disc_new_row_added (disc,
								 uri,
								 path);
	}
	else if ((file = g_hash_table_lookup (disc->priv->dirs, uri))
	     &&  !g_slist_find (disc->priv->loading, file)) {
		GSList *paths;

		/* the problem here is that despite the fact this directory was explored
		 * one or various subdirectories could have been removed because excluded */
		bonfire_data_disc_restore_excluded_children (disc, file);

		paths = g_slist_prepend (NULL, (char*) path);
		bonfire_data_disc_replace_symlink_children (disc, file, paths);
		g_slist_free (paths);

		graft = bonfire_data_disc_new_row_added (disc,
							 uri,
							 path);
	}
	else if ((file = g_hash_table_lookup (disc->priv->dirs, uri))) {
		bonfire_data_disc_directory_priority (disc, file);
		graft = bonfire_data_disc_new_row_added (disc,
							 uri,
							 path);
	}
	else {
		bonfire_data_disc_directory_new (disc,
						 g_strdup (uri),
						 FALSE);
		graft = bonfire_data_disc_new_row_added (disc,
							 uri,
							 path);
	}

	for (iter = excluded; iter; iter = iter->next) {
		excluded_uri = iter->data;
		bonfire_data_disc_exclude_uri (disc, graft, excluded_uri);
	}
	g_free (graft);

	/* very unlikely case */
	if (disc->priv->unreadable
	&& (status = GPOINTER_TO_INT (g_hash_table_lookup (disc->priv->unreadable, uri)))
	&&  status == BONFIRE_FILTER_UNREADABLE) {
		/* remove the file from unreadable */
		bonfire_data_disc_unreadable_free (disc, uri);

		/* make it appear wherever it is needed */
	}
	else
		bonfire_data_disc_tree_set_path_from_info (disc, path, NULL, info);

	g_free (uri);
	return BONFIRE_DISC_OK;
}

static void
bonfire_data_disc_new_row_cb (BonfireDataDisc *disc,
			      GSList *results,
			      GSList *graft_infos)
{
	char *path;
	GSList *iter;
	BonfireDiscResult success;
	BonfireGraftInfoData *graft_info;

	for (iter = graft_infos; results && iter; iter = iter->next, results = results->next) {
		graft_info = iter->data;

		/* we check wether the row still exists and if everything went well */
		path = bonfire_data_disc_reference_get (disc, graft_info->ref);
		if (!path) {
			FREE_GRAFT_INFO (disc, graft_info);
			continue;
		}

		success = bonfire_data_disc_new_row_real (disc,
							  results->data,
							  path,
							  graft_info->excluded);

		if (success == BONFIRE_DISC_OK)
			bonfire_data_disc_selection_changed (disc, TRUE);

		g_free (path);
		FREE_GRAFT_INFO (disc, graft_info);
	}
	g_slist_free (graft_infos);
}

static BonfireDiscResult
bonfire_data_disc_add_uri_real (BonfireDataDisc *disc,
				const char *uri,
				GtkTreePath *treeparent)
{
	BonfireGraftInfoData *graft_info;
	BonfireDiscResult success;
	GtkTreeModel *model;
	GSList *graft_infos;
	GtkTreeIter iter;
	GSList *uris;
	char *name;
	char *path;

	g_return_val_if_fail (uri != NULL, BONFIRE_DISC_ERROR_UNKNOWN);

	if (disc->priv->reject_files
	||  disc->priv->is_loading)
		return BONFIRE_DISC_NOT_READY;

	/* We make sure there isn't the same file in the directory */
	name = g_path_get_basename (uri);
	success = bonfire_data_disc_check_for_existence (disc,
							 name,
							 treeparent,
							 TRUE);

	if (success != BONFIRE_DISC_OK) {
		g_free (name);
		return success;
	}

	/* create the path */
	if (treeparent
	&&  gtk_tree_path_get_depth (treeparent) > 0) {
		char *parent;

		bonfire_data_disc_tree_path_to_disc_path (disc,
							  treeparent,
							  &parent);
		path = g_strconcat (parent, "/", name, NULL);
		g_free (parent);
	}
	else
		path = g_strconcat ("/", name, NULL);

	uris = g_slist_prepend (NULL, (char *) uri);
	graft_info = g_new0 (BonfireGraftInfoData, 1);
	graft_info->ref = bonfire_data_disc_reference_new (disc, path);
	graft_info->excluded = NULL;
	graft_infos = g_slist_prepend (NULL, graft_info);
	g_free (path);

	success = bonfire_data_disc_get_info_async (disc,
						    uris,
						    GNOME_VFS_FILE_INFO_GET_ACCESS_RIGHTS|
						    GNOME_VFS_FILE_INFO_GET_MIME_TYPE|
						    GNOME_VFS_FILE_INFO_FORCE_SLOW_MIME_TYPE,
						    (BonfireInfoAsyncResultFunc) bonfire_data_disc_new_row_cb,
						    graft_infos);
	g_slist_free (uris);

	if (success != BONFIRE_DISC_OK) {
		g_slist_foreach (graft_info->excluded, (GFunc) g_free, NULL);
		g_slist_free (graft_info->excluded);
		g_free (name);

		bonfire_data_disc_reference_free (disc, graft_info->ref);
		return success;
	}

	/* make it appear in the tree */
	model = disc->priv->model;
	if (treeparent
	&&  gtk_tree_path_get_depth (treeparent) > 0) {
		GtkTreeIter parent;

		gtk_tree_model_get_iter (model, &parent, treeparent);
		gtk_tree_store_append (GTK_TREE_STORE (model), &iter, &parent);
		bonfire_data_disc_tree_update_directory_real (disc, &parent);
	}
	else
		gtk_tree_store_append (GTK_TREE_STORE (model), &iter, NULL);

	gtk_tree_store_set (GTK_TREE_STORE (model), &iter,
			    NAME_COL, name,
			    SIZE_COL, _("(loading ...)"),
			    MIME_COL, _("(loading ...)"),
			    ROW_STATUS_COL, ROW_NEW,
			    -1);

	g_free (name);

	return BONFIRE_DISC_OK;
}

static BonfireDiscResult
bonfire_data_disc_add_uri (BonfireDisc *disc, const char *uri)
{
	char *unescaped_uri;
	BonfireDiscResult success;

	if (BONFIRE_DATA_DISC (disc)->priv->is_loading)
		return BONFIRE_DISC_LOADING;

	gtk_notebook_set_current_page (GTK_NOTEBOOK (BONFIRE_DATA_DISC (disc)->priv->notebook), 1);

	unescaped_uri = gnome_vfs_unescape_string_for_display (uri);
	success = bonfire_data_disc_add_uri_real (BONFIRE_DATA_DISC (disc),
						  unescaped_uri,
						  NULL);
	g_free (unescaped_uri);
	return success;
}

/********************************* export internal tracks *********************/
struct _MakeExcludedListData {
	GSList *list;
	char *graft;
};
typedef struct _MakeExcludedListData MakeExcludedListData;
struct _MakeListData {
	GSList *list;
	BonfireDataDisc *disc;
};
typedef struct _MakeListData MakeListData;

static void
_foreach_unreadable_make_list_cb (const char *uri,
				  BonfireFilterStatus status,
				  MakeListData *data)
{
	data->list = g_slist_prepend (data->list, g_strdup (uri));
}

static void
_foreach_symlink_make_list_cb (const char *symlink,
			       int value,
			       MakeListData *data)
{
	data->list = g_slist_prepend (data->list, g_strdup (symlink));
}

static void
_foreach_restored_make_list_cb (const char *restored,
			        BonfireFilterStatus status,
			        MakeListData *data)
{
	data->list = g_slist_prepend (data->list, g_strdup (restored));
}

static void
_foreach_excluded_make_list_cb (const char *uri,
				GSList *grafts,
				MakeExcludedListData *data)
{
	for (; grafts; grafts = grafts->next) {
		if (data->graft == grafts->data) {
			data->list = g_slist_prepend (data->list, g_strdup (uri));
			return;
		}
	}
}

static void
_foreach_grafts_make_list_cb (char *path,
			      const char *uri,
			      MakeListData *data)
{
	MakeExcludedListData callback_data;
	BonfireGraftPt *graft;

	graft = g_new0 (BonfireGraftPt, 1);
	graft->uri = uri != BONFIRE_CREATED_DIR ? g_strdup (uri) : NULL;
	graft->path = g_strdup (path);

	if (uri
	&&  data->disc->priv->excluded
	&&  g_hash_table_lookup (data->disc->priv->dirs, uri)) {
		callback_data.list = NULL;
		callback_data.graft = path;

		g_hash_table_foreach (data->disc->priv->excluded,
				      (GHFunc) _foreach_excluded_make_list_cb,
				      &callback_data);
		graft->excluded = callback_data.list;
	}

	data->list = g_slist_prepend (data->list, graft);
}

static BonfireDiscResult
bonfire_data_disc_get_track (BonfireDisc *disc,
			     BonfireDiscTrack *track,
			     gboolean src_format)
{
	GSList *grafts;
	GSList *restored;
	GSList *unreadable;
	BonfireDataDisc *data;
	MakeListData callback_data;

	data = BONFIRE_DATA_DISC (disc);
	if (!g_hash_table_size (data->priv->paths))
		return BONFIRE_DISC_ERROR_EMPTY_SELECTION;

	callback_data.disc = data;
	callback_data.list = NULL;

	if (data->priv->unreadable)
		g_hash_table_foreach (data->priv->unreadable,
				      (GHFunc) _foreach_unreadable_make_list_cb,
				      &callback_data);

	if (data->priv->symlinks)
		g_hash_table_foreach (data->priv->symlinks,
				      (GHFunc) _foreach_symlink_make_list_cb,
				      &callback_data);
	unreadable = callback_data.list;

	callback_data.list = NULL;
	if (data->priv->restored)
		g_hash_table_foreach (data->priv->restored,
				      (GHFunc) _foreach_restored_make_list_cb,
				      &callback_data);
	restored = callback_data.list;

	callback_data.list = NULL;
	g_hash_table_foreach (data->priv->paths,
			      (GHFunc) _foreach_grafts_make_list_cb,
			      &callback_data);
	grafts = callback_data.list;

	if (src_format)
		track->type = BONFIRE_DISC_TRACK_SOURCE;
	else
		track->type = BONFIRE_DISC_TRACK_DATA;

	if (src_format) {
		BonfireTrackSource *src;

		if (!restored && !grafts)
			return BONFIRE_DISC_ERROR_EMPTY_SELECTION;

		if (!track->contents.src) {
			track->contents.src = g_new0 (BonfireTrackSource, 1);
			track->contents.src->type = BONFIRE_TRACK_SOURCE_DATA;
		}

		src = track->contents.src;
		src->contents.data.grafts = grafts;
		src->contents.data.excluded = unreadable;

		/* we don't need to add the restored files since they are normal
		 * files that are not in excluded list */
		g_slist_foreach (restored, (GFunc) g_free, NULL);
		g_slist_free (restored);
	}
	else {
		track->contents.data.grafts = grafts;
		track->contents.data.unreadable = unreadable;
		track->contents.data.restored = restored;
	}

	return BONFIRE_DISC_OK;
}

/******************************** load track ***********************************/
static void
bonfire_data_disc_load_error_dialog (BonfireDataDisc *disc)
{
	GtkWidget *dialog;
	GtkWidget *toplevel;

	toplevel = gtk_widget_get_toplevel (GTK_WIDGET (disc));
	dialog = gtk_message_dialog_new (GTK_WINDOW (toplevel),
					 GTK_DIALOG_DESTROY_WITH_PARENT |
					 GTK_DIALOG_MODAL,
					 GTK_MESSAGE_WARNING,
					 GTK_BUTTONS_CLOSE,
					 _("Project couldn't be loaded:"));

	gtk_window_set_title (GTK_WINDOW (dialog), _("Project loading failure"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
						  _("A thread couldn't be created"));

	gtk_widget_show_all (dialog);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

enum {
	BONFIRE_GRAFT_CHECK_OK,
	BONFIRE_GRAFT_CHECK_PARENT_FILE,
	BONFIRE_GRAFT_CHECK_PARENT_UNREADABLE,
	BONFIRE_GRAFT_CHECK_PARENT_NOT_FOUND,
	BONFIRE_GRAFT_CHECK_FILE_WITH_SAME_NAME, 
	BONFIRE_GRAFT_CHECK_DIR_WITH_SAME_NAME
};

struct _BonfireCheckGraftResultData {
	BonfireDataDiscReference ref;
	char *parent;
	char *path;
	int status;
};
typedef struct _BonfireCheckGraftResultData BonfireCheckGraftResultData;

struct _BonfireLoadStep3Data {
	BonfireDataDisc *disc;
	GSList *grafts;
};
typedef struct _BonfireLoadStep3Data BonfireLoadStep3Data;

static void
_free_graft_check_result (BonfireDataDisc *disc,
			  BonfireCheckGraftResultData *graft)
{
	bonfire_data_disc_reference_free (disc, graft->ref);
	g_free (graft->parent);
	g_free (graft->path);
	g_free (graft);
}

static gboolean
bonfire_data_disc_load_track_step_3 (BonfireLoadStep3Data *callback_data)
{
	BonfireCheckGraftResultData *graft;
	const char *graft_path;
	BonfireDataDisc *disc;
	char *graft_uri;
	char *last_path;
	char *ref_path;
	GSList *iter;
	char *parent;

	disc = callback_data->disc;
	for (iter = callback_data->grafts; iter; iter = iter->next) {
		graft = iter->data;

		if (graft->status == BONFIRE_GRAFT_CHECK_OK) {
			_free_graft_check_result (disc, graft);
			continue;
		}

		/* check that we still care about it */
		ref_path = bonfire_data_disc_reference_get (disc, graft->ref);
		if (!ref_path) {
			_free_graft_check_result (disc, graft);
			continue;
		}

		if (strcmp (ref_path, graft->path)) {
			_free_graft_check_result (disc, graft);
			g_free (ref_path);
			continue;
		}
		g_free (ref_path);

		parent = g_path_get_dirname (graft->path);

		/* NOTE: graft_path has to exist since we checked the reference */
		graft_path = bonfire_data_disc_graft_get_real (disc, parent);
		if (graft->status != BONFIRE_GRAFT_CHECK_PARENT_NOT_FOUND) {
			GSList *excluding = NULL;

			/* see that it isn't already excluded if not do it */
			if (disc->priv->excluded)
				excluding = g_hash_table_lookup (disc->priv->excluded,
								 graft->parent);

			if (excluding
			&& !g_slist_find (excluding, graft_path)) {
				bonfire_data_disc_exclude_uri (disc,
							       graft_path,
							       graft->parent);
			}
		}

		if (graft->status == BONFIRE_GRAFT_CHECK_FILE_WITH_SAME_NAME
		||  graft->status == BONFIRE_GRAFT_CHECK_DIR_WITH_SAME_NAME) {
			_free_graft_check_result (disc, graft);
			g_free (parent);
			continue;
		}

		/* we need to create all the directories until last */
		/* NOTE : graft_uri can't be a created dir as we checked that before */
		graft_uri = g_hash_table_lookup (disc->priv->paths, graft_path);
		last_path = g_strconcat (graft_path,
					 graft->parent + strlen (graft_uri),
					 NULL);

		while (strcmp (parent, last_path) && strcmp (parent, "/")) {
			char *tmp;
			bonfire_data_disc_graft_new (disc,
						     NULL,
						     parent);

			tmp = parent;
			parent = g_path_get_dirname (parent);
			g_free (tmp);
		}

		/* NOTE: the last directory exists or was excluded */
		_free_graft_check_result (disc, graft);
		g_free (last_path);
		g_free (parent);
	}
	g_slist_free (callback_data->grafts);
	g_free (callback_data);

	return FALSE;
}

static gpointer
bonfire_data_disc_graft_check_async (BonfireLoadStep3Data *data)
{
	BonfireCheckGraftResultData *graft;
	GnomeVFSFileInfo *info;
	GnomeVFSResult result;
	char *escaped_uri;
	GSList *iter;
	char *uri;
	char *tmp;

	info = gnome_vfs_file_info_new ();
	for (iter = data->grafts; iter; iter = iter->next) {
		graft = iter->data;

		/* check a file with the same name doesn't exist */
		escaped_uri = gnome_vfs_escape_host_and_path_string (graft->parent);
		result = gnome_vfs_get_file_info (escaped_uri, info, 0);
		g_free (escaped_uri);

		if (result != GNOME_VFS_ERROR_NOT_FOUND) {
			if (info->type == GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK) {
				/* we ignore this path since when the symlink is met
				 * we'll check its path doesn't overlap a graft point
				 * and if so, nothing will happen */
				graft->status = BONFIRE_GRAFT_CHECK_OK;
			}
			else if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY)
				graft->status = BONFIRE_GRAFT_CHECK_DIR_WITH_SAME_NAME;
			else
				graft->status = BONFIRE_GRAFT_CHECK_FILE_WITH_SAME_NAME;

			gnome_vfs_file_info_clear (info);
			continue;
		}

		/* now we check that we have an existing directory as parent on
		 * the disc */
		uri = g_path_get_dirname (graft->parent);
		g_free (graft->parent);

		gnome_vfs_file_info_clear (info);
		escaped_uri = gnome_vfs_escape_host_and_path_string (uri);
		result = gnome_vfs_get_file_info (escaped_uri, info, 0);		
		g_free (escaped_uri);

		if (result == GNOME_VFS_OK && info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
			graft->status = BONFIRE_GRAFT_CHECK_OK;
			gnome_vfs_file_info_clear (info);
			graft->parent = uri;
			continue;
		}

		while (1) {
			if (result == GNOME_VFS_ERROR_NOT_FOUND) {
				tmp = uri;
				uri = g_path_get_dirname (uri);
				g_free (tmp);
			}
			else if (info->type == GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK) {
				/* symlink: do as if it didn't exist since it will
				 * be replaced when met during exploration at this
				 * point we'll check if the paths of the symlink
				 * doesn't overlap a graft point and if so nothing
				 * will happen */
				graft->status = BONFIRE_GRAFT_CHECK_PARENT_NOT_FOUND;

				tmp = uri;
				uri = g_path_get_dirname (uri);
				g_free (tmp);
			}
			else if (result != GNOME_VFS_OK) {
				graft->status = BONFIRE_GRAFT_CHECK_PARENT_UNREADABLE;
				break;
			}
			else if (info->type != GNOME_VFS_FILE_TYPE_DIRECTORY) {
				graft->status = BONFIRE_GRAFT_CHECK_PARENT_FILE;
				break;
			}
			else {
				graft->status = BONFIRE_GRAFT_CHECK_PARENT_NOT_FOUND;
				break;
			}
	
			gnome_vfs_file_info_clear (info);
			escaped_uri = gnome_vfs_escape_host_and_path_string (uri);
			result = gnome_vfs_get_file_info (escaped_uri, info, 0);
			g_free (escaped_uri);
		}
		graft->parent = uri;
		gnome_vfs_file_info_clear (info);
	}

	gnome_vfs_file_info_unref (info);
	g_idle_add ((GSourceFunc) bonfire_data_disc_load_track_step_3,
		    data);

	return NULL;
}

static void
bonfire_data_disc_path_create (BonfireDataDisc *disc,
			       const char *path)
{
	char *tmp;
	char *tmp_path;

	tmp_path = g_path_get_dirname (path);
	while (!strcmp (tmp_path, "/")) {
		bonfire_data_disc_graft_new (disc, NULL, tmp_path);
	
		tmp = tmp_path;
		tmp_path = g_path_get_dirname (tmp);
		if (!strcmp (tmp_path, "/")) {
			bonfire_data_disc_tree_new_empty_folder (disc, tmp);
			bonfire_data_disc_expose_path (disc, tmp);
		}
		g_free (tmp);
	}

	g_free (tmp_path);
}

/* This function checks that a graft point consistency:
 * First, when graft points are added as children of another graft points
 * we need to make sure that a child file of the top graft points on the 
 * file system doesn't have the same name as the child graft point.
 * If so, we exclude child file
 * A second problem might be that one of the parent directory doesn't exist
 * on the file system. we'll have therefore to add empty directories */
static void
bonfire_data_disc_graft_check (BonfireDataDisc *disc,
			       GSList *paths)
{
	char *path;
	char *parent;
	GSList *iter;
	GError *error = NULL;
	GSList *grafts = NULL;
	const char *graft_uri;
	const char *graft_path;
	BonfireCheckGraftResultData *graft;
	BonfireLoadStep3Data *callback_data;

	for (iter = paths; iter; iter = iter->next) {
		path = iter->data;

		parent = g_path_get_dirname (path);
		graft_path = bonfire_data_disc_graft_get_real (disc, parent);
		g_free (parent);

		if (!graft_path) {
			/* no parent (maybe it was unreadable) so we need to 
			 * create empty directories but we don't need to check
			 * if it overlaps anything */
			bonfire_data_disc_path_create (disc, path);
			continue;
		}

		graft_uri = g_hash_table_lookup (disc->priv->paths, graft_path);
		if (!graft_uri || graft_uri == BONFIRE_CREATED_DIR)
			continue;

		graft = g_new0 (BonfireCheckGraftResultData, 1);
		graft->path = g_strdup (path);
		graft->ref = bonfire_data_disc_reference_new (disc, path);
		graft->parent = g_strconcat (graft_uri,
					     path + strlen (graft_path),
					     NULL);
		grafts = g_slist_prepend (grafts, graft);
	}

	if (!grafts)
		return;

	callback_data = g_new0 (BonfireLoadStep3Data, 1);
	callback_data->disc = disc;
	callback_data->grafts = grafts;

	g_thread_create ((GThreadFunc) bonfire_data_disc_graft_check_async,
			 callback_data,
			 FALSE,
			 &error);

	if (error) {
		g_warning ("ERROR creating thread : %s\n", error->message);
		g_error_free (error);

		/* it failed so cancel everything */
		bonfire_data_disc_reset_real (disc);

		/* warn the user */
		bonfire_data_disc_load_error_dialog (disc);

		for (; callback_data->grafts; callback_data->grafts = g_slist_remove (callback_data->grafts, graft)) {
			graft = callback_data->grafts->data;
			_free_graft_check_result (disc, graft);
		}
		g_free (callback_data);
	}
}

static void
bonfire_data_disc_load_step_2 (BonfireDataDisc *disc,
			       GSList *results,
			       GSList *grafts)
{
	GSList *iter;
	GSList *paths = NULL;
	BonfireGraftPt *graft;
	BonfireDiscResult success;
	BonfireInfoAsyncResult *result;

	/* whenever a graft point is valid add it to the hash */
	for (iter = grafts; iter; iter = iter->next) {
		graft = iter->data;

		if (!graft->uri) {
			char *parent;

			/* these are created directories no need to check results */
			bonfire_data_disc_graft_new (disc,
						     NULL,
						     graft->path);

			/* see if the parent was added to the tree at the root
			 * of the disc or if it's itself at the root of the disc.
			 * if so, show it in the tree */
			parent = g_path_get_dirname (graft->path);

			if (!strcmp (parent, "/")
			&&  bonfire_data_disc_tree_new_empty_folder (disc, graft->path))
				/* we can expose its contents right away (won't be explored) */
				bonfire_data_disc_expose_path (disc, graft->path);
			else if (g_hash_table_lookup (disc->priv->paths, parent)) {
				char *tmp;

				tmp = parent;
				parent = g_path_get_dirname (tmp);
				g_free (tmp);

				if (!strcmp (parent, "/"))
					bonfire_data_disc_tree_new_empty_folder_real (disc,
										      graft->path,
										      ROW_NOT_EXPLORED); 
			}
			g_free (parent);

			success = BONFIRE_DISC_OK;
		}
		else {
			char *parent;

			/* see if the parent was added to the tree at the root
			 * of the disc or if it's itself at the root of the disc.
			 * if so, show it in the tree */
			parent = g_path_get_dirname (graft->path);
			if (g_hash_table_lookup (disc->priv->paths, parent)) {
				char *tmp;

				tmp = parent;
				parent = g_path_get_dirname (parent);
				g_free (tmp);

				if (!strcmp (parent, "/"))
					bonfire_data_disc_tree_new_path (disc,
									 graft->path,
									 NULL,
									 NULL);
			}
			else if (!strcmp (parent, "/")) {
				bonfire_data_disc_tree_new_path (disc,
								 graft->path,
								 NULL,
								 NULL);
			}
			g_free (parent);

			if (results) {
				result = results->data;
				/* the following function will create a graft point */
				success = bonfire_data_disc_new_row_real (disc,
									  result,
									  graft->path,
									  graft->excluded);
				results = results->next;
			}
			else {
				/* we shouldn't reach this since there should be
				 * as many results as there are graft files */

				/* something went wrong cancel operations */
				g_slist_foreach (iter, (GFunc) bonfire_graft_point_free, NULL);
				g_slist_free (grafts);

				g_slist_foreach (paths, (GFunc) g_free, NULL);
				g_slist_free (paths);

				/* warn the user */
				bonfire_data_disc_load_error_dialog (disc);
				bonfire_data_disc_reset_real (disc);
				return;
			}
		}

		if (success == BONFIRE_DISC_OK) {
			char *parent;

			/* This is for additional checks (see above function) */
			parent = g_path_get_dirname (graft->path);
			if (strcmp (parent, "/")) 
				paths = g_slist_prepend (paths, g_strdup (graft->path));

			g_free (parent);
		}

		bonfire_graft_point_free (graft);
	}
	g_slist_free (grafts);

	if (paths) {
		bonfire_data_disc_graft_check (disc, paths);
		g_slist_foreach (paths, (GFunc) g_free, NULL);
		g_slist_free (paths);
	}

	disc->priv->is_loading = FALSE;
	bonfire_data_disc_selection_changed (disc, (g_hash_table_size (disc->priv->paths) > 0));
}

/* we now check if the graft points are still valid files. */
static void
bonfire_data_disc_load_step_1 (BonfireDataDisc *disc,
			       GSList *results,
			       GSList *grafts)
{
	BonfireInfoAsyncResult *result;
	BonfireDiscResult success;
	BonfireGraftPt *graft;
	GSList *iter;
	GSList *next;
	GSList *uris;

	/* see if restored file are still valid. If so, add them to restored hash */
	for (; results; results = results->next) {
		result = results->data;

		if (result->result == GNOME_VFS_ERROR_NOT_FOUND)
			continue;

		bonfire_data_disc_restored_new (disc,
						result->uri,
						BONFIRE_FILTER_UNKNOWN);
	}

	uris = NULL;
	for (iter = grafts; iter; iter = next) {
		graft = iter->data;
		next = iter->next;

		if (graft->uri) {
			/* NOTE: it might happen that the same uri will be put
			 * several times in the list but it doesn't matter as
			 * gnome-vfs caches the results so it won't really hurt */
			uris = g_slist_prepend (uris, graft->uri);
		}
	}

	uris = g_slist_reverse (uris);
	success = bonfire_data_disc_get_info_async (disc,
						    uris,
						    GNOME_VFS_FILE_INFO_GET_ACCESS_RIGHTS |
						    GNOME_VFS_FILE_INFO_GET_MIME_TYPE |
						    GNOME_VFS_FILE_INFO_FORCE_SLOW_MIME_TYPE,
						    (BonfireInfoAsyncResultFunc) bonfire_data_disc_load_step_2,
						    grafts);
	g_slist_free (uris);

	if (success != BONFIRE_DISC_OK) {
		/* something went wrong cancel operations */
		g_slist_foreach (grafts, (GFunc) bonfire_graft_point_free, NULL);
		g_slist_free (grafts);

		/* warn the user */
		bonfire_data_disc_load_error_dialog (disc);
		bonfire_data_disc_reset_real (disc);
	}
}

/* first, we make list and copy graft points and restored files:
 * we'll then check first the existence and the state of restored 
 * files. we must add them to the list before the graft points since
 * we don't want to see them added to the unreadable list when we'll
 * explore the graft points */
static BonfireDiscResult
bonfire_data_disc_load_track (BonfireDisc *disc,
			      BonfireDiscTrack *track)
{
	char *uri;
	GSList *iter;
	GSList *uris = NULL;
	GSList *grafts = NULL;
	BonfireGraftPt *graft;
	BonfireDiscResult success;

	g_return_val_if_fail (track->type == BONFIRE_DISC_TRACK_DATA, BONFIRE_DISC_OK);

	gtk_notebook_set_current_page (GTK_NOTEBOOK (BONFIRE_DATA_DISC (disc)->priv->notebook), 1);

	if (track->contents.data.grafts == NULL)
		return BONFIRE_DISC_ERROR_EMPTY_SELECTION;

	/* we don't really need to add the unreadable files since 
	 * the important thing is those that must be restored.
	 * that's the same for the symlinks both types of files
	 * will be added as exploration of graft points goes on */
	for (iter = track->contents.data.grafts; iter; iter = iter->next) {
		graft = iter->data;
		grafts = g_slist_prepend (grafts,
					  bonfire_graft_point_copy (graft));
	}

	grafts = g_slist_reverse (grafts);

	/* add restored : we must make sure that they still exist 
	 * before doing the exploration of graft points so that 
	 * they won't be added to unreadable list */
	for (iter = track->contents.data.restored; iter; iter = iter->next) {
		uri = iter->data;
		uris = g_slist_prepend (uris, uri);
	}

	success = bonfire_data_disc_get_info_async (BONFIRE_DATA_DISC (disc),
						    uris,
						    GNOME_VFS_FILE_INFO_GET_ACCESS_RIGHTS,
						    (BonfireInfoAsyncResultFunc) bonfire_data_disc_load_step_1,
						    grafts);
	g_slist_free (uris);

	if (success) {
		g_slist_foreach (grafts,
				(GFunc) bonfire_graft_point_free,
				 NULL);

		g_slist_free (grafts);
		return success;
	}

	BONFIRE_DATA_DISC (disc)->priv->is_loading = TRUE;
	return BONFIRE_DISC_LOADING;
}

/******************************* row moving ************************************/
static BonfireDiscResult
bonfire_data_disc_restore_row (BonfireDataDisc *disc,
			       const char *uri,
			       const char *oldpath,
			       const char *newpath)
{
	BonfireFile *file;
	char *newgraft;
	char *oldgraft;

	/* the file is no longer excluded since it came back to the right place */
	newgraft = bonfire_data_disc_graft_get (disc, newpath);
	bonfire_data_disc_restore_uri (disc, newgraft, uri);

	/* NOTE: we don't know for sure that oldpath is a graft:
	 * indeed it could be that the same directory was grafted
	 * twice and one of his subdirectories is moved between
	 * the two graft points */
	oldgraft = bonfire_data_disc_graft_get (disc, oldpath);

	/* now we need to find the old graft point and remove it */
	if ((file = g_hash_table_lookup (disc->priv->dirs, uri))) {
		/* we move all children graft point as well */
		bonfire_data_disc_graft_children_move (disc,
						       oldpath,
						       newpath);

		/* no need for mutex here since it doesn't change */
		if (!g_slist_find (disc->priv->loading, file->uri))
			bonfire_data_disc_graft_transfer_excluded (disc,
								   oldgraft,
								   newgraft);
	}
	else if (!g_hash_table_lookup (disc->priv->files, uri)) {
		g_free (newgraft);
		g_free (oldgraft);
		g_error ("ERROR: This file (%s) must have a graft point.\n",
			 file->uri);
		/* ERROR : this file must have a graft point since it was moved 
		 * back to place. Now the graft points are either in loading,
		 * dirs, files */
	}

	/* find the old graft points if it exists
	 * otherwise exclude it (see NOTE above) */
	/* NOTE : in case of directories no need to update 
	 * the excluded hash (excluded were moved) */
	if (strcmp (oldgraft, oldpath))
		bonfire_data_disc_exclude_uri (disc, oldgraft, uri);
	else
		bonfire_data_disc_graft_remove (disc, oldgraft);

	g_free (newgraft);
	g_free (oldgraft);

	return BONFIRE_DISC_OK;
}

static void
bonfire_data_disc_move_row_in_dirs_hash (BonfireDataDisc *disc,
					 BonfireFile *dir,
					 const char *oldpath,
					 const char *newpath)
{
	char *oldgraft;

	/* move all children graft points */
	bonfire_data_disc_graft_children_move (disc,
					       oldpath,
					       newpath);

	/* see if the dir the user is moving was already grafted at oldpath */
	if (g_hash_table_lookup (disc->priv->paths, oldpath)) {
		bonfire_data_disc_graft_changed (disc,
						 oldpath,
						 newpath);
		return;
	}

	/* we exclude it from his previous graft point */
	oldgraft = bonfire_data_disc_graft_get (disc, oldpath);
	bonfire_data_disc_exclude_uri (disc, oldgraft, dir->uri);

	/* apparently the old path did not correspond to 
	 * a graft point so we make a new graft point */
	bonfire_data_disc_graft_new (disc,
				     dir->uri,
				     newpath);

	/* now since it became a graft point we must remove
	 * from the old parent the excluded that were pointing
	 * to children of uri and add them for the new graft
	 * point we created. NOTE : that's only for directories
	 * which are not loading */
	/* no need for mutex hash doesn't change */
	if (!g_slist_find (disc->priv->loading, dir->uri))
		bonfire_data_disc_graft_transfer_excluded (disc,
							   oldgraft,
							   newpath);

	g_free (oldgraft);
}

static void
bonfire_data_disc_move_row_in_files_hash (BonfireDataDisc *disc,
					  BonfireFile *file,
					  const char *oldpath,
					  const char *newpath)
{
	char *oldgraft;

	/* see if the old path was already grafted */
	if (g_hash_table_lookup (disc->priv->paths, oldpath)) {
		bonfire_data_disc_graft_changed (disc,
						       oldpath,
						       newpath);
		return;
	}

	/* we exclude it from his previous graft point */
	oldgraft = bonfire_data_disc_graft_get (disc, oldpath);
	bonfire_data_disc_exclude_uri (disc, oldgraft, file->uri);
	g_free (oldgraft);

	/* apparently the old path did not correspond to a graft point so we
	 * make a new graft point since a moved file becomes a graft point 
	 * we exclude it from his previous graft point as well */
	bonfire_data_disc_graft_new (disc, file->uri, newpath);
}

struct _MoveRowSimpleFileData {
	char *newpath;
	char *oldpath;
};
typedef struct _MoveRowSimpleFileData MoveRowSimpleFileData;

static void
bonfire_data_disc_move_row_simple_file_cb (BonfireDataDisc *disc,
					   GSList *results,
					   MoveRowSimpleFileData *callback_data)
{
	BonfireInfoAsyncResult *result;
	BonfireFile *file;
	GnomeVFSFileInfo *info;
	char *parenturi;
	char *graft;
	char *uri;

	for (; results; results = results->next) {
		result = results->data;

		uri = result->uri;
		info = result->info;

		/* see if the parent still exists and we are still valid */
		parenturi = g_path_get_dirname (uri);
		file = g_hash_table_lookup (disc->priv->dirs, parenturi);
		g_free (parenturi);

		if (!file)
			continue;

		if (result->result == GNOME_VFS_ERROR_NOT_FOUND) {
			bonfire_data_disc_remove_uri_from_tree (disc, uri);
			bonfire_data_disc_add_rescan (disc, file);
			continue;
		}
		if (result->result == GNOME_VFS_ERROR_LOOP) {
			bonfire_data_disc_remove_uri_from_tree (disc, uri);
			bonfire_data_disc_add_rescan (disc, file);
			bonfire_data_disc_unreadable_new (disc,
								g_strdup (uri),
								BONFIRE_FILTER_RECURSIVE_SYM);
		}
	
		if (result->result != GNOME_VFS_OK
		||  !bonfire_data_disc_is_readable (info)) {
			bonfire_data_disc_remove_uri_from_tree (disc, uri);
			bonfire_data_disc_add_rescan (disc, file);
			bonfire_data_disc_unreadable_new (disc,
								g_strdup (uri),
								BONFIRE_FILTER_UNREADABLE);
			continue;
		}
		
		/* it's a simple file. Make a file structure and insert
		* it in files hash and finally exclude it from its parent */
		bonfire_data_disc_file_new (disc,
						  uri,
						  info->size);
		bonfire_data_disc_graft_new (disc,
						   uri,
						   callback_data->newpath);
	
		graft = bonfire_data_disc_graft_get (disc, callback_data->oldpath);
		bonfire_data_disc_exclude_uri (disc, graft, uri);
		g_free (graft);
	}

	g_free (callback_data->newpath);
	g_free (callback_data->oldpath);
	g_free (callback_data);
}

static BonfireDiscResult
bonfire_data_disc_move_row_simple_file (BonfireDataDisc *disc,
					const char *uri,
					const char *oldpath,
					const char *newpath)
{
	GSList *uris;
	BonfireDiscResult result;
	MoveRowSimpleFileData *callback_data;

	callback_data = g_new0 (MoveRowSimpleFileData, 1);
	callback_data->newpath = g_strdup (newpath);
	callback_data->oldpath = g_strdup (oldpath);

	uris = g_slist_prepend (NULL, (char *) uri);
	result = bonfire_data_disc_get_info_async (disc,
						   uris,
						   GNOME_VFS_FILE_INFO_GET_ACCESS_RIGHTS,
						   (BonfireInfoAsyncResultFunc) bonfire_data_disc_move_row_simple_file_cb,
						   callback_data);
	g_slist_free (uris);

	if (result != BONFIRE_DISC_OK) {
		g_free (callback_data->newpath);
		g_free (callback_data->oldpath);
		g_free (callback_data);
		return result;
	}

	return BONFIRE_DISC_OK;
}

static BonfireDiscResult
bonfire_data_disc_move_row (BonfireDataDisc *disc,
			    const char *oldpath,
			    const char *newpath)
{
	BonfireDiscResult result;
	BonfireFile *file;
	char *uri;

	/* update all path references */
	bonfire_data_disc_move_references (disc, oldpath, newpath);

	/* uri can be NULL if it is a new created directory */
	uri = bonfire_data_disc_path_to_uri (disc, oldpath);
	if (!uri) {
		gpointer value = NULL;

		bonfire_data_disc_graft_children_move (disc,
						       oldpath,
						       newpath);

		g_hash_table_lookup_extended (disc->priv->paths,
					      oldpath,
					      &value,
					      NULL);
		g_hash_table_remove (disc->priv->paths, oldpath);
		g_free (value);

		g_hash_table_insert (disc->priv->paths,
				     g_strdup (newpath),
				     BONFIRE_CREATED_DIR);
		return BONFIRE_DISC_OK;
	}

	/* the file has been moved to what would be its original place in the 
	 * file system, in other words, the disc tree hierarchy matches the file
	 * system hierarchy as well as the names which are similar. so we don't 
	 * need a graft point any more and drop the exclusion */
	result = BONFIRE_DISC_OK;
	if (bonfire_data_disc_original_parent (disc, uri, newpath))
		result = bonfire_data_disc_restore_row (disc,
							uri,
							oldpath,
							newpath);
	else if ((file = g_hash_table_lookup (disc->priv->dirs, uri)))
		bonfire_data_disc_move_row_in_dirs_hash (disc,
							 file,
							 oldpath,
							 newpath);
	else if ((file = g_hash_table_lookup (disc->priv->files, uri)))
		bonfire_data_disc_move_row_in_files_hash (disc,
							  file,
							  oldpath,
							  newpath);
	else	/* this one could fail */
		result = bonfire_data_disc_move_row_simple_file (disc,
								 uri,
								 oldpath,
								 newpath);

	if (result == BONFIRE_DISC_OK)
		bonfire_data_disc_selection_changed (disc, TRUE);

	g_free (uri);
	return result;
}

/************************************** DND ************************************/
static GtkTreePath *
bonfire_data_disc_get_dest_path (BonfireDataDisc *disc,
				 gint x,
				 gint y)
{
	GtkTreeViewDropPosition pos;
	GtkTreePath *realpath;
	GtkTreeModel *sort;
	GtkTreePath *path;

	sort = disc->priv->sort;

	gtk_tree_view_get_dest_row_at_pos (GTK_TREE_VIEW(disc->priv->tree), x, y, &path, &pos);
	if (path) {
		gboolean isdir;

		if (pos == GTK_TREE_VIEW_DROP_INTO_OR_AFTER
		||  pos == GTK_TREE_VIEW_DROP_INTO_OR_BEFORE) {
			GtkTreeIter iter;

			/* the parent is the row we're dropping into 
			 * we make sure that the parent is a directory
			 * otherwise put it before or after */
			gtk_tree_model_get_iter (sort, &iter, path);
			gtk_tree_model_get (sort, &iter,
					    ISDIR_COL,
					    &isdir, -1);
		}
		else
			isdir = FALSE;

		if (!isdir) {
			if (pos == GTK_TREE_VIEW_DROP_AFTER
			||  pos == GTK_TREE_VIEW_DROP_INTO_OR_AFTER)
				gtk_tree_path_next (path);
		}
		else {
			GtkTreeIter parent;

			gtk_tree_model_get_iter (sort, &parent, path);
			pos = gtk_tree_model_iter_n_children (sort, &parent);
			gtk_tree_path_append_index (path, pos);
		}
	}
	else
		path = gtk_tree_path_new_from_indices
		    (gtk_tree_model_iter_n_children (sort, NULL), -1);

	realpath = gtk_tree_model_sort_convert_path_to_child_path(GTK_TREE_MODEL_SORT(disc->priv->sort),
								path);

	/* realpath can be NULL if the row has been dropped into an empty directory */
	if(!realpath) {
		GtkTreePath *path_parent;

		path_parent = gtk_tree_path_copy(path);
		gtk_tree_path_up(path_parent);

		if(gtk_tree_path_get_depth(path_parent)) {
			GtkTreeIter iter;

			realpath = gtk_tree_model_sort_convert_path_to_child_path(GTK_TREE_MODEL_SORT(sort),
										path_parent);
			gtk_tree_model_get_iter(sort, &iter, path_parent);
			gtk_tree_path_append_index(realpath, gtk_tree_model_iter_n_children(sort, &iter));
		}
		else
			realpath = gtk_tree_path_new_from_indices(gtk_tree_model_iter_n_children(sort, NULL),
								-1);

		gtk_tree_path_free(path_parent);
	}
	gtk_tree_path_free(path);

	return realpath;
}

static char*
bonfire_data_disc_new_disc_path (BonfireDataDisc *disc,
				 const char *display,
				 GtkTreePath *dest)
{
	GtkTreePath *parent;
	char *newparentpath;
	char *newpath;

	parent = gtk_tree_path_copy (dest);
	gtk_tree_path_up (parent);
	bonfire_data_disc_tree_path_to_disc_path (disc,
						  parent,
						  &newparentpath);
	gtk_tree_path_free (parent);

	if (!strcmp (newparentpath, "/"))
		newpath = g_strconcat ("/", display, NULL);
	else
		newpath = g_strconcat (newparentpath, "/",
				       display, NULL);
	g_free (newparentpath);

	return newpath;
}

static gboolean
bonfire_data_disc_native_data_received (BonfireDataDisc *disc,
					GtkSelectionData *selection_data,
					gint x,
					gint y)
{
	GtkTreeRowReference *destref = NULL;
	GtkTreeRowReference *srcref = NULL;
	BonfireDiscResult result;
	GtkTreeModel *model;
	GtkTreePath *path;
	GtkTreePath *dest;
	GtkTreePath *src;
	GtkTreeIter row;
	char *oldpath;
	char *newpath;
	char *name;

	model = disc->priv->model;

	/* check again if move is possible */
	dest = bonfire_data_disc_get_dest_path (disc, x, y);
	if (gtk_tree_drag_dest_row_drop_possible (GTK_TREE_DRAG_DEST (model),
						  dest,
						  selection_data) == FALSE) {
		gtk_tree_path_free (dest);
		return FALSE;
	}

	gtk_tree_get_row_drag_data (selection_data,
				    &model,
				    &src);

	/* move it in the backend */
	gtk_tree_model_get_iter (model, &row, src);
	gtk_tree_model_get (model, &row, NAME_COL, &name, -1);
	newpath = bonfire_data_disc_new_disc_path (disc, name, dest);
	g_free (name);

	bonfire_data_disc_tree_path_to_disc_path (disc, src, &oldpath);
	result = bonfire_data_disc_move_row (disc,
					     oldpath,
					     newpath);

	if (result != BONFIRE_DISC_OK)
		goto end;

	/* keep some necessary references for later */
	srcref = gtk_tree_row_reference_new (model, src);
	if (gtk_tree_path_get_depth (dest) > 1) {
		int nb_children;

		/* we can only put a reference on the parent
		 * since the child doesn't exist yet */
		gtk_tree_path_up (dest);
		destref = gtk_tree_row_reference_new (model, dest);

		gtk_tree_model_get_iter (model, &row, dest);
		nb_children = gtk_tree_model_iter_n_children (model, &row);
		gtk_tree_path_append_index (dest, nb_children);
	}
	else
		destref = NULL;

	/* move it */
	if (!gtk_tree_drag_dest_drag_data_received (GTK_TREE_DRAG_DEST (model),
						    dest,
						    selection_data)) {
		bonfire_data_disc_move_row (disc,
					    newpath,
					    oldpath);
		goto end;
	}

	/* update parent directories */
	path = gtk_tree_row_reference_get_path (srcref);
	gtk_tree_drag_source_drag_data_delete (GTK_TREE_DRAG_SOURCE (model),
					       path);
	bonfire_data_disc_tree_update_parent (disc, path);
	gtk_tree_path_free (path);

	if (destref
	&& (path = gtk_tree_row_reference_get_path (destref))) {
		bonfire_data_disc_tree_update_directory (disc, path);
		gtk_tree_path_free (path);
	}

end:

	if (srcref)
		gtk_tree_row_reference_free (srcref);
	if (destref)
		gtk_tree_row_reference_free (destref);

	gtk_tree_path_free (dest);
	gtk_tree_path_free (src);
	g_free (oldpath);
	g_free (newpath);

	return TRUE;
}

static GdkDragAction 
bonfire_data_disc_drag_data_received_dragging (BonfireDataDisc *disc,
					       GtkSelectionData *selection_data)
{
	char *name;
	GtkTreeIter iter;
	GtkTreePath *dest;
	GtkTreeModel *sort;
	GtkTreeModel *model;
	GtkTreePath *sort_dest;
	GtkTreePath *src_parent;
	GtkTreePath *dest_parent;
	BonfireDiscResult result;
	GtkTreeViewDropPosition pos;
	GdkDragAction action = GDK_ACTION_MOVE;

	if (!disc->priv->drag_source)
		return GDK_ACTION_DEFAULT;

	model = disc->priv->model;
	sort = disc->priv->sort;

	src_parent = NULL;
	dest_parent = NULL;

	gtk_tree_view_get_drag_dest_row (GTK_TREE_VIEW (disc->priv->tree),
					 &sort_dest,
					 &pos);

	if (!sort_dest) {
		pos = GTK_TREE_VIEW_DROP_AFTER;
		dest = gtk_tree_path_new_from_indices (gtk_tree_model_iter_n_children (model, NULL) - 1, -1);
	}
	else {
		dest = gtk_tree_model_sort_convert_path_to_child_path (GTK_TREE_MODEL_SORT (sort),
								       sort_dest);
		gtk_tree_path_free (sort_dest);
	}

	/* if we drop into make sure it is a directory */
	if (gtk_tree_model_get_iter (model, &iter, dest)
	&& (pos == GTK_TREE_VIEW_DROP_INTO_OR_AFTER
	||  pos == GTK_TREE_VIEW_DROP_INTO_OR_BEFORE)) {
		gboolean isdir;
		int explored;

		gtk_tree_model_get (model, &iter,
				    ISDIR_COL, &isdir,
				    ROW_STATUS_COL, &explored, -1);

		if (!isdir) {
			if (GTK_TREE_VIEW_DROP_INTO_OR_AFTER)
				pos = GTK_TREE_VIEW_DROP_AFTER;
			else if (GTK_TREE_VIEW_DROP_INTO_OR_BEFORE)
				pos = GTK_TREE_VIEW_DROP_BEFORE;
		}
		else if (explored < ROW_EXPLORED) {
			/* we prevent any addition to a row not yet explored
			 * as we could have two files with the same name */
			action = GDK_ACTION_DEFAULT;
			goto end;
		}
	}

	if (pos == GTK_TREE_VIEW_DROP_AFTER
	||  pos == GTK_TREE_VIEW_DROP_BEFORE) {
		dest_parent = gtk_tree_path_copy (dest);
		gtk_tree_path_up (dest_parent);
	}
	else
		dest_parent = gtk_tree_path_copy (dest);

	src_parent = gtk_tree_path_copy (disc->priv->drag_source);
	gtk_tree_path_up (src_parent);

	/* check that we are actually changing the directory */
	if(!gtk_tree_path_get_depth (dest_parent)
	&& !gtk_tree_path_get_depth (src_parent)) {
		action = GDK_ACTION_DEFAULT;
		goto end;
	}

	if (gtk_tree_path_get_depth (dest_parent)
	&&  gtk_tree_path_get_depth (src_parent)
	&& !gtk_tree_path_compare (src_parent, dest_parent)) {
		action = GDK_ACTION_DEFAULT;
		goto end;
	}

	/* make sure that a row doesn't exist with the same name */
	gtk_tree_model_get_iter (model, &iter, disc->priv->drag_source);
	gtk_tree_model_get (model, &iter,
			    NAME_COL, &name,
			    -1);

	result = bonfire_data_disc_check_for_existence (disc,
							name,
							dest_parent,
							FALSE);
	g_free (name);

	if (result != BONFIRE_DISC_OK) {
		action = GDK_ACTION_DEFAULT;
		goto end;
	}

	if (!gtk_tree_drag_dest_row_drop_possible (GTK_TREE_DRAG_DEST (model),
						   dest,
						   selection_data))
		action = GDK_ACTION_DEFAULT;

end:
	gtk_tree_path_free (dest);
	gtk_tree_path_free (src_parent);
	gtk_tree_path_free (dest_parent);

	return action;
}

static void
bonfire_data_disc_drag_data_received_cb (GtkTreeView *tree,
					 GdkDragContext *drag_context,
					 gint x,
					 gint y,
					 GtkSelectionData *selection_data,
					 guint info,
					 guint time,
					 BonfireDataDisc *disc)
{
	gboolean result = FALSE;

	if (disc->priv->drag_status == STATUS_DRAGGING) {
		GdkDragAction action;

		if (!disc->priv->is_loading
		&&  !disc->priv->reject_files)
			action = bonfire_data_disc_drag_data_received_dragging (disc,
										selection_data);
		else
			action = GDK_ACTION_DEFAULT;

		gdk_drag_status (drag_context, action, time);
		if(action == GDK_ACTION_DEFAULT)
			gtk_tree_view_set_drag_dest_row (tree,
							 NULL,
							 GTK_TREE_VIEW_DROP_BEFORE);

		g_signal_stop_emission_by_name (tree, "drag-data-received");
		return;
	}

	if (disc->priv->scroll_timeout) {
		g_source_remove (disc->priv->scroll_timeout);
		disc->priv->scroll_timeout = 0;
	}

	if (disc->priv->expand_timeout) {
		g_source_remove (disc->priv->expand_timeout);
		disc->priv->expand_timeout = 0;
	}

	if (selection_data->length <= 0
	||  selection_data->format != 8) {
		gtk_drag_finish (drag_context, FALSE, FALSE, time);
		disc->priv->drag_status = STATUS_NO_DRAG;
		g_signal_stop_emission_by_name (tree, "drag-data-received");

		return;
	}

	/* we get URIS */
	if (info == TARGET_URIS_LIST) {
		gboolean func_results;
		char **uri, **uris;
		GtkTreePath *dest;

		uris = gtk_selection_data_get_uris (selection_data);
		dest = bonfire_data_disc_get_dest_path (disc, x, y);
		gtk_tree_path_up (dest);

		for (uri = uris; *uri != NULL; uri++) {
			char *unescaped_uri;

			unescaped_uri = gnome_vfs_unescape_string_for_display (*uri);
			func_results = bonfire_data_disc_add_uri_real (disc,
								       unescaped_uri,
								       dest);
			result = (result ? TRUE : func_results);
			g_free (unescaped_uri);
		}

		gtk_tree_path_free (dest);
		g_strfreev (uris);
	} 
	else if (info == TREE_MODEL_ROW)
		result = bonfire_data_disc_native_data_received (disc,
								 selection_data,
								 x,
								 y);

	gtk_drag_finish (drag_context,
			 result,
			 (drag_context->action == GDK_ACTION_MOVE),
			 time);

	g_signal_stop_emission_by_name (tree, "drag-data-received");
	disc->priv->drag_status = STATUS_NO_DRAG;
}

static void
bonfire_data_disc_drag_begin_cb (GtkTreeView *tree,
				 GdkDragContext *drag_context,
				 BonfireDataDisc *disc)
{
	GtkTreePath *sort_src;
	GtkTreeModel *model;
	GtkTreeModel *sort;
	GdkPixmap *row_pix;
	GtkTreePath *src;
	GtkTreeIter iter;
	gint cell_y;
	int status;

	disc->priv->drag_status = STATUS_DRAGGING;
	g_signal_stop_emission_by_name (tree, "drag-begin");

	/* Put the icon */
	gtk_tree_view_get_path_at_pos (tree,
				       disc->priv->press_start_x,
				       disc->priv->press_start_y,
				       &sort_src,
				       NULL,
				       NULL,
				       &cell_y);
	
	g_return_if_fail (sort_src != NULL);

	sort = disc->priv->sort;
	src = gtk_tree_model_sort_convert_path_to_child_path (GTK_TREE_MODEL_SORT(sort),
							      sort_src);

	model = disc->priv->model;
	gtk_tree_model_get_iter (model, &iter, src);
	gtk_tree_model_get (model, &iter, ROW_STATUS_COL, &status, -1);
	if (status == ROW_BOGUS) {
		disc->priv->drag_source = NULL;
		gtk_tree_path_free (sort_src);
		gtk_tree_path_free (src);
		return;
	}

	disc->priv->drag_source = src;

	row_pix = gtk_tree_view_create_row_drag_icon (tree, sort_src);
	gtk_drag_set_icon_pixmap (drag_context,
				  gdk_drawable_get_colormap (row_pix),
				  row_pix,
				  NULL,
				  /* the + 1 is for the black border in the icon */
				  disc->priv->press_start_x + 1,
				  cell_y + 1);
	
	gtk_tree_path_free(sort_src);
	g_object_unref (row_pix);
}

static gboolean
bonfire_data_disc_drag_drop_cb (GtkTreeView *tree,
				GdkDragContext *drag_context,
				gint x,
				gint y,
				guint time,
				BonfireDataDisc *disc)
{
	GdkAtom target = GDK_NONE;

	if (disc->priv->scroll_timeout) {
		g_source_remove (disc->priv->scroll_timeout);
		disc->priv->scroll_timeout = 0;
	}

	if (disc->priv->expand_timeout) {
		g_source_remove (disc->priv->expand_timeout);
		disc->priv->expand_timeout = 0;
	}

	g_signal_stop_emission_by_name (tree, "drag-drop");
	disc->priv->drag_status = STATUS_DRAG_DROP;

	target = gtk_drag_dest_find_target (GTK_WIDGET(tree),
					    drag_context,
					    gtk_drag_dest_get_target_list(GTK_WIDGET(tree)));

	if (target != GDK_NONE) {
		gtk_drag_get_data (GTK_WIDGET(tree),
				   drag_context,
				   target,
				   time);
		return TRUE;
	}

	return FALSE;
}

/* in the following functions there are quick and dirty cut'n pastes from gtktreeview.c shame on me */
static GtkTreeViewDropPosition
bonfire_data_disc_set_dest_row (BonfireDataDisc *disc,
				gint x,
				gint y)
{
	GtkTreeViewDropPosition pos;
	GtkTreePath *old_dest = NULL;
	GtkTreePath *sort_dest;

	if (!gtk_tree_view_get_dest_row_at_pos (GTK_TREE_VIEW(disc->priv->tree),
						x,
						y,
						&sort_dest,
						&pos)) {
		gint n_children;
		GtkTreeModel *sort;

		sort = disc->priv->sort;
		n_children = gtk_tree_model_iter_n_children (sort, NULL);
		if (n_children) {
			pos = GTK_TREE_VIEW_DROP_AFTER;
			sort_dest = gtk_tree_path_new_from_indices (n_children - 1, -1);
		}
		else {
			pos = GTK_TREE_VIEW_DROP_BEFORE;
			sort_dest = gtk_tree_path_new_from_indices (0, -1);
		}
	}

	gtk_tree_view_get_drag_dest_row (GTK_TREE_VIEW (disc->priv->tree),
					 &old_dest,
					 NULL);

	if (old_dest
	&&  sort_dest
	&&  (gtk_tree_path_compare (old_dest, sort_dest) != 0
	||  !(pos == GTK_TREE_VIEW_DROP_INTO_OR_AFTER 
	||    pos == GTK_TREE_VIEW_DROP_INTO_OR_BEFORE))
	&&  disc->priv->expand_timeout) {
		g_source_remove (disc->priv->expand_timeout);
		disc->priv->expand_timeout = 0;
	}

	gtk_tree_view_set_drag_dest_row (GTK_TREE_VIEW(disc->priv->tree),
					 sort_dest,
					 pos);
	gtk_tree_path_free (sort_dest);

	if (old_dest)
		gtk_tree_path_free (old_dest);
	return pos;
}

static gboolean
bonfire_data_disc_scroll_timeout_cb (BonfireDataDisc *data)
{
	int y;
	double value;
	int scroll_area;
	GdkWindow *window;
	GdkRectangle area;
	GtkAdjustment *adjustment;

	window = gtk_tree_view_get_bin_window (GTK_TREE_VIEW (data->priv->tree));
	gdk_window_get_pointer (window, NULL, &y, NULL);
	gtk_tree_view_get_visible_rect (GTK_TREE_VIEW (data->priv->tree), &area);

	/* height */
	scroll_area = area.height / 6;
	value = y - scroll_area;
	if (value >= 0) {
		value = y - (area.height - scroll_area);
		if (value <= 0)
			return TRUE;
	}

	g_object_get (data->priv->tree, "vadjustment", &adjustment, NULL);
	value = CLAMP (adjustment->value + value,
		       0.0,
		       adjustment->upper - adjustment->page_size);
	gtk_adjustment_set_value (adjustment, value);

	return TRUE;
}

static gboolean
bonfire_data_disc_expand_timeout_cb (BonfireDataDisc *disc)
{
	gboolean result;
	GtkTreePath *dest;
	GtkTreeViewDropPosition pos;

	gtk_tree_view_get_drag_dest_row (GTK_TREE_VIEW (disc->priv->tree),
					 &dest,
					 &pos);

	/* we don't need to check if it's a directory because:
	   - a file wouldn't have children anyway
	   - we check while motion if it's a directory and if not remove the INTO from pos */
	if (dest
	&&  (pos == GTK_TREE_VIEW_DROP_INTO_OR_AFTER || pos == GTK_TREE_VIEW_DROP_INTO_OR_BEFORE)) {
		gtk_tree_view_expand_row (GTK_TREE_VIEW (disc->priv->tree), dest, FALSE);
		disc->priv->expand_timeout = 0;
	
		gtk_tree_path_free (dest);
	}
	else {
		if (dest)
			gtk_tree_path_free (dest);
	
		result = TRUE;
	}

	return result;
}

static gboolean
bonfire_data_disc_drag_motion_cb (GtkWidget *tree,
				  GdkDragContext *drag_context,
				  gint x,
				  gint y,
				  guint time,
				  BonfireDataDisc *disc)
{
	GdkAtom target;

	if (disc->priv->is_loading || disc->priv->reject_files) {
		g_signal_stop_emission_by_name (tree, "drag-motion");

		gdk_drag_status (drag_context, GDK_ACTION_DEFAULT, time);
		gtk_tree_view_set_drag_dest_row (GTK_TREE_VIEW (tree),
						 NULL,
						 GTK_TREE_VIEW_DROP_BEFORE);
		return FALSE;
	}

	target = gtk_drag_dest_find_target (tree,
					    drag_context,
					    gtk_drag_dest_get_target_list(tree));

	if (target == gdk_atom_intern ("GTK_TREE_MODEL_ROW", FALSE)) {
		GtkTreeViewDropPosition pos;

		pos = bonfire_data_disc_set_dest_row (disc, x, y);

		/* since we mess with the model we have to re-implement the following two */
		if (!disc->priv->expand_timeout
		&&  (pos == GTK_TREE_VIEW_DROP_INTO_OR_AFTER || pos == GTK_TREE_VIEW_DROP_INTO_OR_BEFORE)) {
			disc->priv->expand_timeout = g_timeout_add (500,
								    (GSourceFunc) bonfire_data_disc_expand_timeout_cb,
								    disc);
		}
		else if (disc->priv->scroll_timeout == 0) {
			disc->priv->scroll_timeout = g_timeout_add (150,
								    (GSourceFunc) bonfire_data_disc_scroll_timeout_cb,
								    disc);
		}
		gtk_drag_get_data (tree,
				   drag_context,
				   target,
				   time);
		g_signal_stop_emission_by_name (tree, "drag-motion");
		return TRUE;
	}

	return FALSE;
}

static void
bonfire_data_disc_drag_get_cb (GtkWidget *tree,
                               GdkDragContext *context,
                               GtkSelectionData *selection_data,
                               guint info,
                               guint time,
			       BonfireDataDisc *disc)
{
	g_signal_stop_emission_by_name (tree, "drag-data-get");

	/* that could go into begin since we only accept GTK_TREE_MODEL_ROW as our source target */
	if (selection_data->target == gdk_atom_intern ("GTK_TREE_MODEL_ROW", FALSE)
	&&  disc->priv->drag_source) {
		GtkTreeModel *model;

		model = disc->priv->model;
		gtk_tree_set_row_drag_data (selection_data,
					    model,
					    disc->priv->drag_source);
	}
}

static void
bonfire_data_disc_drag_end_cb (GtkWidget *tree,
			       GdkDragContext *drag_context,
			       BonfireDataDisc *disc)
{
	g_signal_stop_emission_by_name (tree, "drag-end");

	gtk_tree_view_set_drag_dest_row (GTK_TREE_VIEW (tree),
					 NULL,
					 GTK_TREE_VIEW_DROP_BEFORE);

	if (disc->priv->scroll_timeout) {
		g_source_remove (disc->priv->scroll_timeout);
		disc->priv->scroll_timeout = 0;
	}

	if (disc->priv->expand_timeout) {
		g_source_remove (disc->priv->expand_timeout);
		disc->priv->expand_timeout = 0;
	}

	gtk_tree_path_free(disc->priv->drag_source);
	disc->priv->drag_source = NULL;
}

void
bonfire_data_disc_drag_leave_cb (GtkWidget *tree,
				 GdkDragContext *drag_context,
				 guint time,
				 BonfireDataDisc *disc)
{
	if (disc->priv->scroll_timeout) {
		g_signal_stop_emission_by_name (tree, "drag-leave");

		g_source_remove (disc->priv->scroll_timeout);
		disc->priv->scroll_timeout = 0;
	}
	if (disc->priv->expand_timeout) {
		g_signal_stop_emission_by_name (tree, "drag-leave");

		g_source_remove (disc->priv->expand_timeout);
		disc->priv->expand_timeout = 0;
	}
}

/**************************** MENUS ********************************************/
static void
bonfire_data_disc_open_file (BonfireDataDisc *disc, GList *list)
{
	char *uri;
	char *path;
	int status;
	GList *item;
	GSList *uris;
	GtkTreeIter iter;
	GtkTreeModel *sort;
	GtkTreePath *realpath;
	GtkTreePath *treepath;

	sort = disc->priv->sort;

	uris = NULL;
	for (item = list; item; item = item->next) {
		treepath = item->data;

		gtk_tree_model_get_iter (sort, &iter, treepath);
		gtk_tree_model_get (sort, &iter, ROW_STATUS_COL, &status, -1);
		if (status == ROW_BOGUS) {
			gtk_tree_path_free (treepath);
			continue;
		}

		realpath = gtk_tree_model_sort_convert_path_to_child_path (GTK_TREE_MODEL_SORT (sort),
									   treepath);

		bonfire_data_disc_tree_path_to_disc_path (disc, realpath, &path);
		gtk_tree_path_free (realpath);

		uri = bonfire_data_disc_path_to_uri (disc, path);
		g_free (path);
		if (uri)
			uris = g_slist_prepend (uris, uri);

	}

	if (!uris)
		return;

	bonfire_utils_launch_app (GTK_WIDGET (disc), uris);
	g_slist_foreach (uris, (GFunc) g_free, NULL);
	g_slist_free (uris);
}

static void
bonfire_data_disc_open_activated_cb (GtkAction *action,
				     BonfireDataDisc *disc)
{
	GList *list;
	GtkTreeModel *sort;
	GtkTreeSelection *selection;

	sort = disc->priv->sort;
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (disc->priv->tree));
	list = gtk_tree_selection_get_selected_rows (selection, &sort);
	bonfire_data_disc_open_file (disc, list);

	g_list_foreach (list, (GFunc) gtk_tree_path_free, NULL);
	g_list_free (list);
}

static void
bonfire_data_disc_rename_activated_cb (GtkAction *action,
				       BonfireDataDisc *disc)
{
	GtkTreeSelection *selection;
	GtkTreeViewColumn *column;
	GtkTreePath *treepath;
	GtkTreeModel *model;
	GtkTreeIter iter;
	GList *list;
	int status;

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (disc->priv->tree));
	model = disc->priv->sort;

	list = gtk_tree_selection_get_selected_rows (selection, &model);
	for (; list; list = g_list_remove (list, treepath)) {
		treepath = list->data;

		gtk_tree_model_get_iter (model, &iter, treepath);
		gtk_tree_model_get (model, &iter,
				    ROW_STATUS_COL, &status,
				    -1);

		if (status == ROW_BOGUS) {
			gtk_tree_path_free (treepath);
			continue;
		}

		gtk_widget_grab_focus (disc->priv->tree);
		column = gtk_tree_view_get_column (GTK_TREE_VIEW (disc->priv->tree),
						   0);
		gtk_tree_view_set_cursor (GTK_TREE_VIEW (disc->priv->tree),
					  treepath,
					  column,
					  TRUE);

		gtk_tree_path_free (treepath);
	}
}

static void
bonfire_data_disc_delete_activated_cb (GtkAction *action,
				       BonfireDataDisc *disc)
{
	bonfire_data_disc_delete_selected (BONFIRE_DISC (disc));
}

struct _BonfireClipData {
	BonfireDataDisc *disc;
	GtkTreeRowReference *reference;
};
typedef struct _BonfireClipData BonfireClipData;

static void
bonfire_data_disc_clipboard_text_cb (GtkClipboard *clipboard,
				     const char *text,
				     BonfireClipData *data)
{
	GtkTreePath *treepath = NULL;
	GtkTreeModel *model = NULL;
	GtkTreePath *parent = NULL;
	GtkTreeIter row;
	char **array;
	char **item;
	char *uri;

	model = data->disc->priv->sort;
	if (data->reference) {
		parent = gtk_tree_row_reference_get_path (data->reference);
		gtk_tree_model_get_iter (model, &row, parent);
	}

	array = g_strsplit_set (text, "\n\r", 0);
	item = array;
	while (*item) {
		if (**item != '\0') {
			char *escaped_uri;

			if (parent) {
				treepath = gtk_tree_path_copy (parent);
				gtk_tree_path_append_index (treepath,
							    gtk_tree_model_iter_n_children
							    (model, &row));
			}

			escaped_uri = gnome_vfs_make_uri_canonical (*item);
			uri = gnome_vfs_unescape_string_for_display (escaped_uri);
			g_free (escaped_uri);
			bonfire_data_disc_add_uri_real (data->disc,
							uri,
							treepath);
			g_free (uri);
			if (treepath)
				gtk_tree_path_free (treepath);
		}

		item++;
	}

	if (parent)
		gtk_tree_path_free (parent);
	g_strfreev (array);

	if (data->reference)
		gtk_tree_row_reference_free (data->reference);

	g_free (data);
}

static void
bonfire_data_disc_clipboard_targets_cb (GtkClipboard *clipboard,
					 GdkAtom *atoms,
					 gint n_atoms,
					 BonfireClipData *data)
{
	GdkAtom *iter;
	char *target;

	iter = atoms;
	while (n_atoms) {
		target = gdk_atom_name (*iter);

		if (!strcmp (target, "x-special/gnome-copied-files")
		    || !strcmp (target, "UTF8_STRING")) {
			gtk_clipboard_request_text (clipboard,
						    (GtkClipboardTextReceivedFunc)
						    bonfire_data_disc_clipboard_text_cb,
						    data);
			g_free (target);
			return;
		}

		g_free (target);
		iter++;
		n_atoms--;
	}

	if (data->reference)
		gtk_tree_row_reference_free (data->reference);
	g_free (data);
}

static void
bonfire_data_disc_paste_activated_cb (GtkAction *action,
				      BonfireDataDisc *disc)
{
	BonfireClipData *data;
	GtkTreeSelection *selection;
	GtkClipboard *clipboard;
	GtkTreeModel *model;
	GList *list;

	data = g_new0 (BonfireClipData, 1);
	data->disc = disc;

	/* we must keep a reference to the row selected */
	model = disc->priv->sort;
	selection =  gtk_tree_view_get_selection (GTK_TREE_VIEW (disc->priv->tree));
	list = gtk_tree_selection_get_selected_rows (selection, &model);
	if (list) {
		GtkTreePath *treepath;
		GtkTreeIter row;
		gboolean isdir;

		treepath = list->data;
		g_list_free (list);

		/* see if it a dir or a file */
		gtk_tree_model_get_iter (model, &row, treepath);
		gtk_tree_model_get (model, &row,
				    ISDIR_COL, &isdir,
				    -1);

		if (isdir 
		|| (gtk_tree_path_up (treepath)
		&&  gtk_tree_path_get_depth (treepath) > 0))
			data->reference = gtk_tree_row_reference_new (model, treepath);

		gtk_tree_path_free (treepath);
	}

	clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
	gtk_clipboard_request_targets (clipboard,
				       (GtkClipboardTargetsReceivedFunc)
				       bonfire_data_disc_clipboard_targets_cb,
				       data);
}

static gboolean
bonfire_data_disc_button_pressed_cb (GtkTreeView *tree,
				     GdkEventButton *event,
				     BonfireDataDisc *disc)
{
	if (disc->priv->is_loading)
		return FALSE;

	if (event->button == 3) {
		GtkTreeSelection *selection;
		GtkTreePath *treepath;
		GtkTreeModel *model;
		int nb_selected = 0;
		GList *list;

		selection = gtk_tree_view_get_selection (tree);
		model = disc->priv->sort;
		list = gtk_tree_selection_get_selected_rows (selection,
							     &model);

		for (; list; list = g_list_remove (list, treepath)) {
			GtkTreeIter iter;
			int status;

			treepath = list->data;

			gtk_tree_model_get_iter (model, &iter, treepath);
			gtk_tree_model_get (model, &iter,
					    ROW_STATUS_COL, &status, -1);

			if (status != ROW_BOGUS)
				nb_selected ++;

			gtk_tree_path_free (treepath);
		}

		bonfire_utils_show_menu (nb_selected, disc->priv->manager, event);
		return TRUE;
	}
	else if (event->button == 1) {
		GList *list;
		gboolean result;
		GtkTreePath *treepath = NULL;

		disc->priv->press_start_x = event->x;
		disc->priv->press_start_y = event->y;

		result = gtk_tree_view_get_path_at_pos (GTK_TREE_VIEW (disc->priv->tree),
							event->x,
							event->y,
							&treepath,
							NULL,
							NULL,
							NULL);
		if (!result || !treepath)
			return FALSE;

		if (disc->priv->selected_path)
			gtk_tree_path_free (disc->priv->selected_path);

		disc->priv->selected_path = treepath;
		bonfire_disc_selection_changed (BONFIRE_DISC (disc));

		list = g_list_prepend (NULL, treepath);

		if (event->type == GDK_2BUTTON_PRESS)
			bonfire_data_disc_open_file (disc, list);

		g_list_free (list);
	}

	return FALSE;
}

static gboolean
bonfire_data_disc_key_released_cb (GtkTreeView *tree,
				   GdkEventKey *event,
				   BonfireDataDisc *disc)
{
	if (disc->priv->is_loading)
		return FALSE;

	if (disc->priv->editing)
		return FALSE;

	if (event->keyval == GDK_KP_Delete || event->keyval == GDK_Delete) {
		bonfire_data_disc_delete_selected (BONFIRE_DISC (disc));
	}

	return FALSE;
}

/*********************************** CELL EDITING ******************************/
static void
bonfire_data_disc_name_editing_started_cb (GtkCellRenderer *renderer,
					   GtkCellEditable *editable,
					   gchar *path,
					   BonfireDataDisc *disc)
{
	disc->priv->editing = 1;
}

static void
bonfire_data_disc_name_editing_canceled_cb (GtkCellRenderer *renderer,
					    BonfireDataDisc *disc)
{
	disc->priv->editing = 0;
}

static void
bonfire_data_disc_name_edited_cb (GtkCellRendererText *cellrenderertext,
				  gchar *path_string,
				  gchar *text,
				  BonfireDataDisc *disc)
{
	GtkTreePath *realpath;
	GtkTreeModel *model;
	GtkTreeModel *sort;
	GtkTreePath *path;
	GtkTreeIter row;
	char *oldpath;
	char *newpath;
	char *parent;
	char *name;

	disc->priv->editing = 0;

	sort = disc->priv->sort;
	path = gtk_tree_path_new_from_string (path_string);
	realpath = gtk_tree_model_sort_convert_path_to_child_path (GTK_TREE_MODEL_SORT(sort),
								   path);
	gtk_tree_path_free (path);

	model = disc->priv->model;
	gtk_tree_model_get_iter (model, &row, realpath);
	gtk_tree_model_get (model, &row, NAME_COL, &name, -1);

	/* make sure it actually changed */
	if (!strcmp (name, text))
		goto end;

	/* make sure there isn't the same name in the directory */
	gtk_tree_path_up (realpath);
	if (bonfire_data_disc_check_for_existence (disc, text, realpath, TRUE))
		goto end;

	bonfire_data_disc_tree_path_to_disc_path (disc, realpath, &parent);
	if (strcmp (parent, "/"))
		oldpath = g_strconcat (parent, "/", name, NULL);
	else
		oldpath = g_strconcat (parent, name, NULL);

	if (strcmp (parent, "/"))
		newpath = g_strconcat (parent, "/", text, NULL);
	else
		newpath = g_strconcat (parent, text, NULL);

	bonfire_data_disc_move_row (disc, oldpath, newpath);

	gtk_tree_store_set (GTK_TREE_STORE (model), &row,
			    NAME_COL, text, -1);

	g_free (parent);
	g_free (oldpath);
	g_free (newpath);

	bonfire_data_disc_selection_changed (disc, TRUE);

end:
	g_free (name);
	gtk_tree_path_free (realpath);
}

/*******************************            ************************************/
static char *
bonfire_data_disc_get_selected_uri (BonfireDisc *disc)
{
	char *uri;
	char *path;
	GtkTreePath *realpath;
	BonfireDataDisc *data;

	data = BONFIRE_DATA_DISC (disc);

	if (!data->priv->selected_path)
		return NULL;

	realpath = gtk_tree_model_sort_convert_path_to_child_path (GTK_TREE_MODEL_SORT (data->priv->sort),
								   data->priv->selected_path);
	bonfire_data_disc_tree_path_to_disc_path (data, realpath, &path);
	gtk_tree_path_free (realpath);

	uri = bonfire_data_disc_path_to_uri (data, path);
	g_free (path);

	return uri;
}

/******************************* monitoring ************************************/
#ifdef BUILD_INOTIFY

static void
bonfire_data_disc_notify_create_file_cb (BonfireDataDisc *disc,
					 GSList *results,
					 gpointer null_data)
{
	char *uri;
	char *path;
	GSList *iter;
	GSList *paths;
	GnomeVFSFileInfo *info;
	BonfireInfoAsyncResult *result;

	result = results->data;

	if (disc->priv->unreadable
	&&  g_hash_table_lookup (disc->priv->unreadable, result->uri)) {
		bonfire_data_disc_remove_uri (disc, result->uri);
		return;
	}

	if (result->result == GNOME_VFS_ERROR_NOT_FOUND) {
		bonfire_data_disc_remove_uri (disc, result->uri);
		return;
	}

	uri = g_strdup (result->uri);
	info = result->info;
	if (info->type == GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK) {
		bonfire_data_disc_remove_uri (disc, result->uri);
		bonfire_data_disc_unreadable_new (disc,
						  uri,
						  BONFIRE_FILTER_BROKEN_SYM);
		return;
	}

	if (result->result == GNOME_VFS_ERROR_LOOP) {
		bonfire_data_disc_remove_uri (disc, result->uri);
		bonfire_data_disc_unreadable_new (disc,
						  uri,
						  BONFIRE_FILTER_RECURSIVE_SYM);
		return;
	}

	if (result->result != GNOME_VFS_OK
	||  !bonfire_data_disc_is_readable (info)) {
		bonfire_data_disc_remove_uri (disc, result->uri);
		bonfire_data_disc_unreadable_new (disc,
						  uri,
						  BONFIRE_FILTER_UNREADABLE);
		return;	
	}

	/* make sure we still care about this change */
	if (bonfire_data_disc_is_excluded (disc, uri, NULL)) {
		g_free (uri);
		return;
	}

	/* get the paths where this file might appear */
	paths = bonfire_data_disc_uri_to_paths (disc, uri);
	if (GNOME_VFS_FILE_INFO_SYMLINK (info)) {
		if (bonfire_data_disc_symlink_is_recursive (disc,
							    uri,
							    info->symlink_name)) {
			bonfire_data_disc_unreadable_new (disc,
							  g_strdup (uri),
							  BONFIRE_FILTER_RECURSIVE_SYM);
			goto end;
		}

		paths = bonfire_data_disc_symlink_new (disc,
						       uri,
						       info,
						       paths);
	}
	else {
		GSList *next;

		/* we check a path doesn't overlap an already grafted file */
		for (iter = paths; iter; iter = next) {
			path = iter->data;
			next = iter->next;

			if (g_hash_table_lookup (disc->priv->paths, path)) {
				char *graft;
				char *parent;
	
				parent = g_path_get_dirname (path);
				graft = bonfire_data_disc_graft_get (disc, parent);
				g_free (parent);
	
				bonfire_data_disc_exclude_uri (disc, graft, uri);
				g_free (graft);

				paths = g_slist_remove (paths, path);
				g_free (path);
				continue;
			}
		}

		if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY
		     && !g_hash_table_lookup (disc->priv->dirs, uri)) {
			bonfire_data_disc_directory_new (disc,
							 g_strdup (uri),
							 FALSE);
		}
		else if (info->type != GNOME_VFS_FILE_TYPE_DIRECTORY
		     && !g_hash_table_lookup (disc->priv->files, uri)) {
			char *parent;
			BonfireFile *file;
	
			parent = g_path_get_dirname (uri);
			file = g_hash_table_lookup (disc->priv->dirs, parent);
			g_free (parent);
	
			file->size += info->size;
			bonfire_data_disc_size_changed (disc, info->size);
		}
	}

	/* make it appear in the tree */
	for (iter = paths; iter; iter = iter->next) {
		path = iter->data;
		bonfire_data_disc_tree_new_path (disc, path, NULL, NULL);
		bonfire_data_disc_tree_set_path_from_info (disc, path, NULL, info);
	}

	g_free (uri);

end:
	g_slist_foreach (paths, (GFunc) g_free, NULL);
	g_slist_free (paths);
}

static void
bonfire_data_disc_notify_create_file_real (BonfireDataDisc *disc,
					   const char *uri)
{
	GSList *uris = NULL;
	BonfireDiscResult result;

	uris = g_slist_prepend (NULL, (char *) uri);
	result = bonfire_data_disc_get_info_async (disc,
						   uris,
						   GNOME_VFS_FILE_INFO_GET_ACCESS_RIGHTS |
						   GNOME_VFS_FILE_INFO_GET_MIME_TYPE |
						   GNOME_VFS_FILE_INFO_FORCE_SLOW_MIME_TYPE,
						   bonfire_data_disc_notify_create_file_cb,
						   NULL);
	g_slist_free (uris);

	if (result != BONFIRE_DISC_OK)
		bonfire_data_disc_unreadable_new (disc,
						  g_strdup (uri),
						  BONFIRE_FILTER_UNREADABLE);
}

struct _BonfireCheckJolietData {
	BonfireDataDisc *disc;
	char *uri;
	gboolean result;
};
typedef struct _BonfireCheckJolietData BonfireCheckJolietData;

static gboolean
bonfire_data_disc_notify_check_joliet_compat_result (BonfireCheckJolietData *data)
{
	if (!data->result) {
		bonfire_data_disc_unreadable_new (data->disc,
						  data->uri,
						  BONFIRE_FILTER_JOLIET_INCOMPAT);
		g_free (data);
		return FALSE;
	}

	bonfire_data_disc_notify_create_file_real (data->disc, data->uri);
	g_free (data->uri);
	g_free (data);
	return FALSE;
}

static gpointer
bonfire_data_disc_notify_check_joliet_compat_cb (BonfireCheckJolietData *data)
{
	GnomeVFSDirectoryHandle *handle;
	GnomeVFSFileInfo *info;
	GnomeVFSResult success;
	char *parent;
	char *name;
	char *tmp;

	tmp = g_path_get_dirname (data->uri);
	parent = gnome_vfs_escape_host_and_path_string (tmp);
	g_free (tmp);

	success = gnome_vfs_directory_open (&handle, parent, 0);
	g_free (parent);

	if (success != GNOME_VFS_OK)
		return NULL;

	name = g_path_get_basename (data->uri);
	info = gnome_vfs_file_info_new ();
	while (gnome_vfs_directory_read_next (handle, info) == GNOME_VFS_OK) {
		if (!strncmp (info->name, name, 64)) {
			gnome_vfs_directory_close (handle);
			goto end;
		}

		gnome_vfs_file_info_clear (info);
	}
	g_free (name);
	gnome_vfs_file_info_unref (info);
	gnome_vfs_directory_close (handle);

	data->result = TRUE;
end:
	g_idle_add ((GSourceFunc) bonfire_data_disc_notify_check_joliet_compat_result,
		    data);

	return NULL;
}

static void
bonfire_data_disc_notify_check_joliet_compat (BonfireDataDisc *disc,
					      const char *uri)
{
	GError *error = NULL;
	BonfireCheckJolietData *data;

	data = g_new0 (BonfireCheckJolietData, 1);
	data->uri = g_strdup (uri);
	data->disc = disc;
	data->result = FALSE;

	g_thread_create ((GThreadFunc) bonfire_data_disc_notify_check_joliet_compat_cb,
			 data,
			 TRUE,
			 &error);

	if (error) {
		bonfire_data_disc_unreadable_new (disc,
						  data->uri,
						  BONFIRE_FILTER_UNREADABLE);

		g_warning ("ERROR creating thread checkin joliet compatibility : %s\n",
			   error->message);
		g_error_free (error);
		g_free (data);
	}
}

static void
bonfire_data_disc_notify_create_file (BonfireDataDisc *disc,
				      BonfireFile *parent,
				      const char *uri)
{
	char *name;

	if (g_slist_find (disc->priv->loading, parent))
		return;

	/* check that this file is not hidden */
	name = g_path_get_basename (uri);
	if (name [0] == '.') {
		bonfire_data_disc_unreadable_new (disc,
						  g_strdup (uri),
						  BONFIRE_FILTER_HIDDEN);
		g_free (name);
		return;
	}

	/* do we have to check for joliet compatibility ? */
	if (strlen (name) > 64)
		bonfire_data_disc_notify_check_joliet_compat (disc, uri);
	else
		bonfire_data_disc_notify_create_file_real (disc, uri);
	g_free (name);
}

static void
bonfire_data_disc_notify_attributes_changed_cb (BonfireDataDisc *disc,
						GSList *results,
						gpointer null_data)
{
	BonfireInfoAsyncResult *result;
	BonfireFilterStatus status;
	GnomeVFSFileInfo *info;
	char *uri;

	for (; results; results = results->next) {
		result = results->data;

		uri = result->uri;
		info = result->info;
		if (result->result == GNOME_VFS_OK
		&&  bonfire_data_disc_is_readable (info)) {
			if (disc->priv->unreadable
			&& (status = GPOINTER_TO_INT (g_hash_table_lookup (disc->priv->unreadable, uri)))
			&&  status == BONFIRE_FILTER_UNREADABLE) {
				GSList *tmp;

				bonfire_data_disc_unreadable_free (disc, uri);
				tmp = g_slist_prepend (NULL, result);
				bonfire_data_disc_notify_create_file_cb (disc,
									 tmp,
									 NULL);
				g_slist_free (tmp);
			}		
			continue;
		}

		/* the file couldn't be a symlink anyway don't check for loop */
		bonfire_data_disc_remove_uri (disc, uri);
		if (result->result != GNOME_VFS_ERROR_NOT_FOUND)
			bonfire_data_disc_unreadable_new (disc,
							  g_strdup (uri),
							  BONFIRE_FILTER_UNREADABLE);
	}
}

static void
bonfire_data_disc_notify_attributes_changed (BonfireDataDisc *disc,
					     const char *uri)
{
	GSList *uris;

	uris = g_slist_prepend (NULL, (char *) uri);
	bonfire_data_disc_get_info_async (disc,
						uris,
						GNOME_VFS_FILE_INFO_GET_ACCESS_RIGHTS,
						bonfire_data_disc_notify_attributes_changed_cb,
						NULL);
	g_slist_free (uris);
}

static void
bonfire_data_disc_notify_modify_file_cb (BonfireDataDisc *disc,
					 GSList *results,
					 gpointer null_data)
{
	BonfireInfoAsyncResult *result;
	BonfireFile *file;
	GnomeVFSFileInfo *info;
	GSList *paths;
	char *uri;

	for (; results; results = results->next) {
		result = results->data;

		uri = result->uri;
		if (result->result == GNOME_VFS_ERROR_NOT_FOUND) {
			bonfire_data_disc_remove_uri (disc, uri);
			continue;
		}

		info = result->info;
		/* the file couldn't be a symlink so no need to check for loop */
		if (result->result != GNOME_VFS_OK
		||  !bonfire_data_disc_is_readable (info)) {
			bonfire_data_disc_remove_uri (disc, uri);
			bonfire_data_disc_unreadable_new (disc,
							  g_strdup (uri),
							  result->result);
			continue;
		}

		/* see if this file has already been looked up */
		if ((file = g_hash_table_lookup (disc->priv->files, uri))) {
			if (info->size != file->size) {
				bonfire_data_disc_size_changed (disc,
								      info->size - file->size);
				file->size = info->size;
			}
		}
		else {
			char *parent;

			parent = g_path_get_dirname (uri);
			file = g_hash_table_lookup (disc->priv->dirs, parent);
			g_free (parent);

			if (file) {
				/* its parent exists and must be rescanned */
				bonfire_data_disc_add_rescan (disc, file);
			}
			else {
				/* the parent doesn't exist any more so nothing happens */
				continue;
			}
		}

		/* search for all the paths it could appear at */
		paths = bonfire_data_disc_uri_to_paths (disc, uri);
		for (; paths; paths = paths->next) {
			char *path;

			path = paths->data;
			bonfire_data_disc_tree_set_path_from_info (disc,
								   path,
								   NULL,
								   info);
			g_free (path);
		}
		g_slist_free (paths);
	}
}

static void
bonfire_data_disc_notify_modify_file (BonfireDataDisc *disc,
				      const char *uri)
{
	GSList *uris;

	uris = g_slist_prepend (NULL, (char *) uri);
	bonfire_data_disc_get_info_async (disc,
						uris,
						GNOME_VFS_FILE_INFO_GET_ACCESS_RIGHTS |
						GNOME_VFS_FILE_INFO_GET_MIME_TYPE |
						GNOME_VFS_FILE_INFO_FORCE_SLOW_MIME_TYPE,
						bonfire_data_disc_notify_modify_file_cb,
						NULL);
	g_slist_free (uris);
}

static gboolean
bonfire_data_disc_inotify_monitor_cb (GIOChannel *channel,
				      GIOCondition condition,
				      BonfireDataDisc *disc)
{
	BonfireFile *monitored;
	struct inotify_event event;
	GError *err = NULL;
	GIOStatus status;
	guint size;
	char *name;

	while (condition & G_IO_IN) {
		monitored = NULL;

		status = g_io_channel_read_chars (channel, (char *) &event,
					     sizeof (struct inotify_event),
					     &size, &err);
		if (status == G_IO_STATUS_EOF)
			return TRUE;

		if (event.len) {
			name = g_new (char, event.len + 1);

			name[event.len] = '\0';

			status = g_io_channel_read_chars (channel, name,
							  event.len, &size,
							  &err);
			if (status != G_IO_STATUS_NORMAL) {
				g_warning ("Error reading inotify: %s\n",
					   err ? "Unknown error" : err->
					   message);
				g_error_free (err);
				return TRUE;
			}
		}
		else
			name = NULL;

		/* look for ignored signal usually following deletion */
		if (event.mask & IN_IGNORED) {
			g_hash_table_remove (disc->priv->monitored,
					     GINT_TO_POINTER (event.wd));
			if (name)
				g_free (name);
			condition = g_io_channel_get_buffer_condition (channel);
			continue;
		}

		/* FIXME: see if we can use IN_ISDIR */
		monitored = g_hash_table_lookup (disc->priv->monitored,
						 GINT_TO_POINTER (event.wd));

		if (!monitored) {
			g_warning ("Unknown (or already deleted) monitored directory = > ignored \n");
			if (name)
				g_free (name);
			condition = g_io_channel_get_buffer_condition (channel);
			continue;
		}
		else if (event.mask & IN_UNMOUNT) {
			bonfire_data_disc_directory_remove_from_tree (disc, monitored);
		}
		/* see if monitored is a monitored file it hasn't got parents */
		else if (g_hash_table_lookup (disc->priv->files, monitored->uri)) {
			if (event.mask & (IN_DELETE_SELF|IN_DELETE|IN_MOVED_FROM|IN_MOVE_SELF))
				bonfire_data_disc_file_remove_from_tree (disc, monitored);
			else if (event.mask & IN_ATTRIB)
				bonfire_data_disc_notify_attributes_changed (disc,
									     monitored->uri);
			else if (event.mask & IN_MODIFY)
				bonfire_data_disc_notify_modify_file (disc, monitored->uri);

			/* NOTE: IN_CREATE & IN_MOVED_TO are impossible here */
			/* what if the name changed (file was moved) */
		}
		else if (!name) {
			char *parent;
			BonfireFile *file;

			/* this is just for events happening for one of the top directories */
			parent = g_path_get_dirname (monitored->uri);
			file = g_hash_table_lookup (disc->priv->dirs, parent);
			g_free (parent);

			if (file) {
				/* do nothing it's not a top directory */
			}
			else if (event.mask & (IN_DELETE_SELF | IN_MOVE_SELF))
				bonfire_data_disc_directory_remove_from_tree (disc, monitored);
			else if (event.mask & IN_ATTRIB)
				bonfire_data_disc_notify_attributes_changed (disc,
										   monitored->uri);
			else if (event.mask & IN_MODIFY)
				bonfire_data_disc_add_rescan (disc, monitored);
		}
		else if (g_slist_find (disc->priv->rescan, monitored)) {
			/* no need to heed modifications inside a 
			 * directory that is going to be rescanned */
			g_free (name);
		}
		else {
			BonfireFile *file;
			char *uri;

			uri = g_strconcat (monitored->uri, "/", name, NULL);
			g_free (name);
			name = NULL;

			if (event.mask & IN_ATTRIB)
				bonfire_data_disc_notify_attributes_changed (disc, uri);
			/* check if this is a directory already explored */
			else if ((file = g_hash_table_lookup (disc->priv->dirs, uri))) {
				if (event.mask & (IN_DELETE | IN_MOVED_FROM))
					bonfire_data_disc_directory_remove_from_tree (disc,
										      file);
				else if (event.mask & IN_MODIFY)	/* very unlikely */
					bonfire_data_disc_add_rescan (disc,
								      file);
				/* NOTE: IN_CREATE & IN_MOVED_TO are impossible here */
			}
			else if ((file = g_hash_table_lookup (disc->priv->files, uri))) {
				if (event.mask & (IN_DELETE | IN_MOVED_FROM))
					bonfire_data_disc_file_remove_from_tree (disc,
									         file);
				else if (event.mask & IN_MODIFY)
					bonfire_data_disc_notify_modify_file (disc,
									      file->uri);

				/* NOTE: IN_CREATE & IN_MOVED_TO are impossible here */
			}
			else if (disc->priv->unreadable
			     &&  g_hash_table_lookup (disc->priv->unreadable, uri)) {
				if (event.mask & (IN_DELETE | IN_MOVED_FROM))
					bonfire_data_disc_unreadable_free (disc, uri);
				/* NOTE: IN_CREATE & IN_MOVED_TO are impossible here */
			}
			else if (disc->priv->symlinks
			     &&  g_hash_table_lookup (disc->priv->symlinks, uri)) {
				/* NOTE : no need to remove target from files or dirs hash tables */
				if (event.mask & (IN_DELETE|IN_MOVED_FROM))
					bonfire_data_disc_remove_uri (disc, uri);
				/* NOTE: IN_CREATE & IN_MOVED_TO are impossible here */
			}
			else if (bonfire_data_disc_is_excluded (disc, uri, NULL)) {
				/* file was excluded and so not taken into account anyway
				 * => no need for rescanning. we don't care if it is deleted
				 * NOTE: we don't delete them from excluded list in case
				 * they come up again i.e in case they are moved back */

				/* NOTE: IN_CREATE & IN_MOVED_TO are impossible here */
			}
			else if (event.mask & (IN_CREATE | IN_MOVED_TO))
				bonfire_data_disc_notify_create_file (disc,
								      monitored,
								      uri);
			else if (event.mask & (IN_DELETE | IN_MOVED_FROM)) {
				bonfire_data_disc_remove_uri_from_tree (disc, uri);
				bonfire_data_disc_add_rescan (disc, monitored);
			}
			else if (event.mask & IN_MODIFY)
				bonfire_data_disc_notify_modify_file (disc,
								      uri);

			g_free (uri);
		} 

		condition = g_io_channel_get_buffer_condition (channel);
	}

	return TRUE;
}

static int warned = 0;

static void
bonfire_data_disc_cancel_monitoring (BonfireDataDisc *disc,
				     BonfireFile *file)
{
	if (disc->priv->notify && !strncmp (file->uri, "file://", 7)
	    && file->handle.wd != -1) {
		int dev_fd;

		dev_fd = g_io_channel_unix_get_fd (disc->priv->notify);
		inotify_rm_watch (dev_fd, file->handle.wd);
		g_hash_table_remove (disc->priv->monitored,
				     GINT_TO_POINTER (file->handle.wd));
		file->handle.wd = -1;
		warned = 0;
	}
}

static BonfireMonitorHandle
bonfire_data_disc_start_monitoring (BonfireDataDisc *disc,
				    BonfireFile *file)
{
	if (disc->priv->notify && !strncmp (file->uri, "file://", 7)) {
		char *escaped_uri;
		int dev_fd;
		char *path;
		__u32 mask;

		escaped_uri = gnome_vfs_escape_host_and_path_string (file->uri);
		path = gnome_vfs_get_local_path_from_uri (escaped_uri);
		g_free (escaped_uri);

		dev_fd = g_io_channel_unix_get_fd (disc->priv->notify);
		mask = IN_MODIFY
		    | IN_ATTRIB
		    | IN_MOVED_FROM
		    | IN_MOVED_TO
		    | IN_CREATE
		    | IN_DELETE 
		    | IN_DELETE_SELF
		    | IN_MOVE_SELF;

		file->handle.wd = inotify_add_watch (dev_fd, path, mask);
		if (file->handle.wd != -1)
			g_hash_table_insert (disc->priv->monitored,
					     GINT_TO_POINTER (file->handle.
							      wd), file);
		else if (!warned) {
			g_warning
			    ("ERROR creating watch for local file %s : %s\n",
			     path, strerror (errno));
			warned = 1;
		}

		g_free (path);
	}

	return file->handle;
}

#endif
