/***************************************************************************
 *            disc.c
 *
 *  dim nov 27 14:58:13 2005
 *  Copyright  2005  Rouquier Philippe
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <glib.h>
#include <glib-object.h>

#include "bonfire-marshal.h"
#include "disc.h"
 
static void bonfire_disc_base_init (gpointer g_class);

typedef enum {
	SELECTION_CHANGED_SIGNAL,
	CONTENTS_CHANGED_SIGNAL,
	SIZE_CHANGED_SIGNAL,
	LAST_SIGNAL
} BonfireDiscSignalType;

static guint bonfire_disc_signals [LAST_SIGNAL] = { 0 };

GType
bonfire_disc_get_type()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireDiscIface),
			bonfire_disc_base_init,
			NULL,
			NULL,
			NULL,
			NULL,
			0,
			0,
			NULL
		};

		type = g_type_register_static(G_TYPE_INTERFACE, 
					      "BonfireDisc",
					      &our_info,
					      0);

		g_type_interface_add_prerequisite (type, G_TYPE_OBJECT);
	}

	return type;
}

static void
bonfire_disc_base_init (gpointer g_class)
{
	static gboolean initialized = FALSE;

	if (initialized)
		return;

	bonfire_disc_signals [CONTENTS_CHANGED_SIGNAL] =
	    g_signal_new ("contents_changed",
			  BONFIRE_TYPE_DISC,
			  G_SIGNAL_RUN_LAST|G_SIGNAL_NO_RECURSE,
			  G_STRUCT_OFFSET (BonfireDiscIface, contents_changed),
			  NULL,
			  NULL,
			  g_cclosure_marshal_VOID__INT,
			  G_TYPE_NONE,
			  1,
			  G_TYPE_INT);
	bonfire_disc_signals [SELECTION_CHANGED_SIGNAL] =
	    g_signal_new ("selection_changed",
			  BONFIRE_TYPE_DISC,
			  G_SIGNAL_RUN_LAST|G_SIGNAL_NO_RECURSE,
			  G_STRUCT_OFFSET (BonfireDiscIface, selection_changed),
			  NULL,
			  NULL,
			  g_cclosure_marshal_VOID__VOID,
			  G_TYPE_NONE,
			  0);
	bonfire_disc_signals [SIZE_CHANGED_SIGNAL] =
	    g_signal_new ("size_changed",
			  BONFIRE_TYPE_DISC,
			  G_SIGNAL_RUN_LAST|G_SIGNAL_NO_RECURSE,
			  G_STRUCT_OFFSET (BonfireDiscIface, size_changed),
			  NULL,
			  NULL,
			  bonfire_marshal_VOID__DOUBLE_STRING,
			  G_TYPE_NONE,
			  2,
			  G_TYPE_DOUBLE,
			  G_TYPE_STRING);
	initialized = TRUE;
}

BonfireDiscResult
bonfire_disc_add_uri (BonfireDisc *disc,
		      const char *uri)
{
	BonfireDiscIface *iface;

	g_return_val_if_fail (BONFIRE_IS_DISC (disc), BONFIRE_DISC_ERROR_UNKNOWN);
	g_return_val_if_fail (uri != NULL, BONFIRE_DISC_ERROR_UNKNOWN);
	
	iface = BONFIRE_DISC_GET_IFACE (disc);
	if (iface->add_uri)
		return (* iface->add_uri) (disc, uri);

	return BONFIRE_DISC_ERROR_UNKNOWN;
}

void
bonfire_disc_delete_selected (BonfireDisc *disc)
{
	BonfireDiscIface *iface;

	g_return_if_fail (BONFIRE_IS_DISC (disc));
	
	iface = BONFIRE_DISC_GET_IFACE (disc);
	if (iface->delete_selected)
		(* iface->delete_selected) (disc);
}

void
bonfire_disc_clear (BonfireDisc *disc)
{
	BonfireDiscIface *iface;

	g_return_if_fail (BONFIRE_IS_DISC (disc));
	
	iface = BONFIRE_DISC_GET_IFACE (disc);
	if (iface->clear)
		(* iface->clear) (disc);
}

void
bonfire_disc_reset (BonfireDisc *disc)
{
	BonfireDiscIface *iface;

	g_return_if_fail (BONFIRE_IS_DISC (disc));
	
	iface = BONFIRE_DISC_GET_IFACE (disc);
	if (iface->reset)
		(* iface->reset) (disc);
}

BonfireDiscResult
bonfire_disc_get_status (BonfireDisc *disc)
{
	BonfireDiscIface *iface;

	g_return_val_if_fail (BONFIRE_IS_DISC (disc), BONFIRE_DISC_ERROR_UNKNOWN);
	
	iface = BONFIRE_DISC_GET_IFACE (disc);
	if (iface->get_status)
		return (* iface->get_status) (disc);

	return BONFIRE_DISC_ERROR_UNKNOWN;
}

BonfireDiscResult
bonfire_disc_get_track (BonfireDisc *disc,
			BonfireDiscTrack *track,
			gboolean src_format)
{
	BonfireDiscIface *iface;

	g_return_val_if_fail (BONFIRE_IS_DISC (disc), BONFIRE_DISC_ERROR_UNKNOWN);
	g_return_val_if_fail (track != NULL, BONFIRE_DISC_ERROR_UNKNOWN);
	
	iface = BONFIRE_DISC_GET_IFACE (disc);
	if (iface->get_track)
		return (* iface->get_track) (disc, track, src_format);

	return BONFIRE_DISC_ERROR_UNKNOWN;
}

BonfireDiscResult
bonfire_disc_load_track (BonfireDisc *disc,
			 BonfireDiscTrack *track)
{
	BonfireDiscIface *iface;

	g_return_val_if_fail (BONFIRE_IS_DISC (disc), BONFIRE_DISC_ERROR_UNKNOWN);
	g_return_val_if_fail (track != NULL, BONFIRE_DISC_ERROR_UNKNOWN);
	
	iface = BONFIRE_DISC_GET_IFACE (disc);
	if (iface->load_track)
		return (* iface->load_track) (disc, track);

	return BONFIRE_DISC_ERROR_UNKNOWN;
}

char *
bonfire_disc_get_selected_uri (BonfireDisc *disc)
{
	BonfireDiscIface *iface;

	g_return_val_if_fail (BONFIRE_IS_DISC (disc), NULL);
	iface = BONFIRE_DISC_GET_IFACE (disc);
	if (iface->get_selected_uri)
		return (* iface->get_selected_uri) (disc);

	return NULL;
}

void
bonfire_disc_selection_changed (BonfireDisc *disc)
{
	g_return_if_fail (BONFIRE_IS_DISC (disc));
	g_signal_emit (disc,
		       bonfire_disc_signals [SELECTION_CHANGED_SIGNAL],
		       0);
}

void
bonfire_disc_contents_changed (BonfireDisc *disc, int nb_files)
{
	g_return_if_fail (BONFIRE_IS_DISC (disc));
	g_signal_emit (disc,
		       bonfire_disc_signals [CONTENTS_CHANGED_SIGNAL],
		       0,
		       nb_files);
}

void
bonfire_disc_size_changed (BonfireDisc *disc,
			   double size,
			   const char *string)
{
	g_return_if_fail (BONFIRE_IS_DISC (disc));

	g_signal_emit (disc,
		       bonfire_disc_signals [SIZE_CHANGED_SIGNAL],
		       0,
		       size,
		       string);

}

/************************************ ******************************************/
void
bonfire_track_clear (BonfireDiscTrack *track)
{
	if (!track)
		return;

	if (track->type == BONFIRE_DISC_TRACK_AUDIO) {
		g_slist_foreach (track->contents.uris, (GFunc) g_free, NULL);
		g_slist_free (track->contents.uris);
	}
	else if (track->type == BONFIRE_DISC_TRACK_DATA) {
		g_slist_foreach (track->contents.data.grafts, (GFunc) bonfire_graft_point_free, NULL);
		g_slist_free (track->contents.data.grafts);
		g_slist_foreach (track->contents.data.restored, (GFunc) g_free, NULL);
		g_slist_free (track->contents.data.restored);
		g_slist_foreach (track->contents.data.unreadable, (GFunc) g_free, NULL);
		g_slist_free (track->contents.data.unreadable);
		g_free (track->contents.data.label);
	}
	else if (track->type == BONFIRE_DISC_TRACK_SOURCE)
		bonfire_track_source_free (track->contents.src);
}

void
bonfire_track_free (BonfireDiscTrack *track)
{
	bonfire_track_clear (track);
	g_free (track);
}
