/***************************************************************************
 *            filtered-window.c
 *
 *  dim oct 30 12:25:50 2005
 *  Copyright  2005  Rouquier Philippe
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <glib/gi18n-lib.h>
#include <glib/gstdio.h>

#include <gtk/gtkdialog.h>
#include <gtk/gtklabel.h>
#include <gtk/gtktreeview.h>
#include <gtk/gtktreemodel.h>
#include <gtk/gtkliststore.h>
#include <gtk/gtktreeviewcolumn.h>
#include <gtk/gtkcellrenderer.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkcellrendererpixbuf.h>
#include <gtk/gtkcellrenderertoggle.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkbox.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkstock.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkmessagedialog.h>

#include <libgnomevfs/gnome-vfs.h>

#include "filtered-window.h"
#include "utils.h"


static void bonfire_filtered_dialog_class_init(BonfireFilteredDialogClass *klass);
static void bonfire_filtered_dialog_init(BonfireFilteredDialog *sp);
static void bonfire_filtered_dialog_finalize(GObject *object);

struct BonfireFilteredDialogPrivate {
	GtkWidget *tree;
	GtkWidget *restore_hidden;
	GtkWidget *restore_broken;
	GtkWidget *restore_joliet;

	GHashTable *joliet_hash;

	int broken_state:1;
	int hidden_state:1;
	int joliet_state:1;

	int disable_joliet_warn:1;
};

enum  {
	STOCK_ID_COL,
	URI_COL,
	TYPE_COL,
	STATUS_COL,
	ACTIVABLE_COL,
	INCLUDED_COL,
	NB_COL,
};

typedef enum {
	REMOVED_SIGNAL,
	RESTORED_SIGNAL,
	LAST_SIGNAL
} BonfireFilteredDialogSignalType;

static guint bonfire_filtered_dialog_signals[LAST_SIGNAL] = { 0 };
static GObjectClass *parent_class = NULL;

static void
bonfire_filtered_dialog_item_toggled_cb (GtkCellRendererToggle *toggle,
					 const gchar *path,
					 BonfireFilteredDialog *dialog);
static void
bonfire_filtered_dialog_restore_hidden_cb (GtkButton *button,
					   BonfireFilteredDialog *dialog);
static void
bonfire_filtered_dialog_restore_broken_symlink_cb (GtkButton *button,
						   BonfireFilteredDialog *dialog);
static void
bonfire_filtered_dialog_restore_joliet_incompat_cb (GtkButton *button,
						    BonfireFilteredDialog *dialog);
static void
bonfire_filtered_dialog_row_activated_cb (GtkTreeView *tree,
                                          GtkTreePath *path,
                                          GtkTreeViewColumn *column,
                                          BonfireFilteredDialog *dialog);

GType
bonfire_filtered_dialog_get_type ()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireFilteredDialogClass),
			NULL,
			NULL,
			(GClassInitFunc)bonfire_filtered_dialog_class_init,
			NULL,
			NULL,
			sizeof (BonfireFilteredDialog),
			0,
			(GInstanceInitFunc)bonfire_filtered_dialog_init,
		};

		type = g_type_register_static(GTK_TYPE_DIALOG, 
			"BonfireFilteredDialog", &our_info, 0);
	}

	return type;
}

static void
bonfire_filtered_dialog_class_init (BonfireFilteredDialogClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent(klass);
	object_class->finalize = bonfire_filtered_dialog_finalize;
	
	
	bonfire_filtered_dialog_signals[REMOVED_SIGNAL] =
	    g_signal_new ("removed",
			  G_OBJECT_CLASS_TYPE (object_class),
			  G_SIGNAL_ACTION | G_SIGNAL_RUN_FIRST,
			  G_STRUCT_OFFSET (BonfireFilteredDialogClass, removed),
			  NULL, NULL,
			  g_cclosure_marshal_VOID__STRING, G_TYPE_NONE, 1,
			  G_TYPE_STRING);

	bonfire_filtered_dialog_signals[RESTORED_SIGNAL] =
	    g_signal_new ("restored",
			  G_OBJECT_CLASS_TYPE (object_class),
			  G_SIGNAL_ACTION | G_SIGNAL_RUN_FIRST,
			  G_STRUCT_OFFSET (BonfireFilteredDialogClass, restored),
			  NULL, NULL,
			  g_cclosure_marshal_VOID__STRING, G_TYPE_NONE, 1,
			  G_TYPE_STRING);

}

static void
bonfire_filtered_dialog_init(BonfireFilteredDialog *obj)
{
	GtkWidget *box;
	GtkWidget *label;
	GtkWidget *scroll;
	GtkListStore *model;
	GtkTreeViewColumn *column;
	GtkCellRenderer *renderer;

	obj->priv = g_new0 (BonfireFilteredDialogPrivate, 1);
	gtk_window_set_title (GTK_WINDOW (obj), _("Removed files"));
	gtk_container_set_border_width (GTK_CONTAINER (GTK_BOX (GTK_DIALOG (obj)->vbox)), 16);
	gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (obj)->vbox), 10);

	label = gtk_label_new (_("<span weight=\"bold\" size=\"larger\">The following files were removed automatically from the project.</span>"));
	g_object_set (G_OBJECT (label), "use-markup", TRUE, NULL);
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (obj)->vbox),
			    label,
			    FALSE,
			    TRUE,
			    0);

	label = gtk_label_new (_("Select the files you want to restore:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (obj)->vbox),
			    label,
			    FALSE,
			    TRUE,
			    0);

	model = gtk_list_store_new (NB_COL,
				    G_TYPE_STRING,
				    G_TYPE_STRING,
				    G_TYPE_STRING,
				    G_TYPE_INT,
				    G_TYPE_BOOLEAN,
				    G_TYPE_BOOLEAN);

	obj->priv->tree = gtk_tree_view_new_with_model (GTK_TREE_MODEL (model));
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (obj->priv->tree), TRUE);
	g_object_unref (model);

	g_signal_connect (G_OBJECT (obj->priv->tree), "row-activated",
			  G_CALLBACK (bonfire_filtered_dialog_row_activated_cb), obj);

	column = gtk_tree_view_column_new ();
	renderer = gtk_cell_renderer_pixbuf_new ();
	gtk_tree_view_column_pack_start (column, renderer, FALSE);
	gtk_tree_view_column_add_attribute (column, renderer,
					    "stock-id", STOCK_ID_COL);

	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_column_pack_end (column, renderer, TRUE);
	gtk_tree_view_column_add_attribute (column, renderer,
					    "text", URI_COL);
	gtk_tree_view_column_set_title (column, _("Files"));
	gtk_tree_view_append_column (GTK_TREE_VIEW (obj->priv->tree), column);
	gtk_tree_view_column_set_sort_column_id (column, URI_COL);
	gtk_tree_view_column_set_clickable (column, TRUE);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_min_width (column, 450);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Type"), renderer,
							   "text", TYPE_COL, NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (obj->priv->tree), column);
	gtk_tree_view_column_set_sort_column_id (column, TYPE_COL);
	gtk_tree_view_column_set_clickable (column, TRUE);

	renderer = gtk_cell_renderer_toggle_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Status"), renderer,
							   "inconsistent", ACTIVABLE_COL,
							   "active", INCLUDED_COL, NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (obj->priv->tree), column);
	gtk_tree_view_column_set_sort_column_id (column, INCLUDED_COL);
	gtk_tree_view_column_set_clickable (column, TRUE);

	g_signal_connect (G_OBJECT (renderer), "toggled",
			  G_CALLBACK (bonfire_filtered_dialog_item_toggled_cb), obj);

	scroll = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scroll),
					     GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (scroll), obj->priv->tree);

	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (obj)->vbox),
			    scroll,
			    TRUE,
			    TRUE,
			    0);

	box = gtk_hbox_new (FALSE, 8);
	obj->priv->restore_hidden = gtk_button_new_with_label (_("Restore hidden files"));
	gtk_widget_set_sensitive (obj->priv->restore_hidden, FALSE);
	g_signal_connect (G_OBJECT (obj->priv->restore_hidden),
			  "clicked",
			  G_CALLBACK (bonfire_filtered_dialog_restore_hidden_cb),
			  obj);
	gtk_box_pack_start (GTK_BOX (box), obj->priv->restore_hidden, FALSE, FALSE, 0);

	obj->priv->restore_broken = gtk_button_new_with_label (_("Restore broken symlink"));
	gtk_widget_set_sensitive (obj->priv->restore_broken, FALSE);
	g_signal_connect (G_OBJECT (obj->priv->restore_broken),
			  "clicked",
			  G_CALLBACK (bonfire_filtered_dialog_restore_broken_symlink_cb),
			  obj);
	gtk_box_pack_start (GTK_BOX (box), obj->priv->restore_broken, FALSE, FALSE, 0);

	obj->priv->restore_joliet = gtk_button_new_with_label (_("Restore joliet incompatible files"));
	gtk_widget_set_sensitive (obj->priv->restore_joliet, FALSE);
	g_signal_connect (G_OBJECT (obj->priv->restore_joliet),
			  "clicked",
			  G_CALLBACK (bonfire_filtered_dialog_restore_joliet_incompat_cb),
			  obj);
	gtk_box_pack_start (GTK_BOX (box), obj->priv->restore_joliet, FALSE, FALSE, 0);

	gtk_box_pack_end (GTK_BOX (GTK_DIALOG (obj)->vbox),
			  box,
			  FALSE,
			  FALSE,
			  0);

	gtk_dialog_add_button (GTK_DIALOG (obj),
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
	gtk_dialog_add_button (GTK_DIALOG (obj),
				GTK_STOCK_OK, GTK_RESPONSE_OK);

	gtk_dialog_set_has_separator (GTK_DIALOG (obj), TRUE);
}

static void
bonfire_filtered_dialog_finalize (GObject *object)
{
	BonfireFilteredDialog *cobj;
	cobj = BONFIRE_FILTERED_DIALOG(object);
	
	if (cobj->priv->joliet_hash)
		g_hash_table_destroy (cobj->priv->joliet_hash);

	g_free(cobj->priv);
	G_OBJECT_CLASS(parent_class)->finalize(object);
}

GtkWidget *
bonfire_filtered_dialog_new ()
{
	BonfireFilteredDialog *obj;
	
	obj = BONFIRE_FILTERED_DIALOG (g_object_new (BONFIRE_TYPE_FILTERED_DIALOG, NULL));
	gtk_window_set_default_size (GTK_WINDOW (obj), 640, 500);
	return GTK_WIDGET (obj);
}

void
bonfire_filtered_dialog_add (BonfireFilteredDialog *dialog,
			     const char *uri,
			     gboolean restored,
			     BonfireFilterStatus status)
{
	char *labels [] = { N_("hidden file"),
			    N_("unreadable file"),
			    N_("broken symlink"),
			    N_("recursive symlink"),
			    N_("Joliet standard incompatible") };
	const char *stock_id;
	GtkTreeModel *model;
	GtkTreeIter iter;
	char *type;

	type = labels [ status - 1 ];
	if (status == BONFIRE_FILTER_UNREADABLE)
		stock_id = GTK_STOCK_CANCEL;
	else
		stock_id = NULL;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (dialog->priv->tree));
	gtk_list_store_append (GTK_LIST_STORE (model), &iter);
	gtk_list_store_set (GTK_LIST_STORE (model), &iter,
			    STOCK_ID_COL, stock_id,
			    URI_COL, uri,
			    TYPE_COL, _(type),
			    STATUS_COL, status,
			    INCLUDED_COL, restored,
			    ACTIVABLE_COL, (status == BONFIRE_FILTER_UNREADABLE || status == BONFIRE_FILTER_RECURSIVE_SYM),
			    -1);

	if (status == BONFIRE_FILTER_HIDDEN)
		gtk_widget_set_sensitive (dialog->priv->restore_hidden, TRUE);

	if (status == BONFIRE_FILTER_BROKEN_SYM)
		gtk_widget_set_sensitive (dialog->priv->restore_broken, TRUE);

	if (status == BONFIRE_FILTER_JOLIET_INCOMPAT)
		gtk_widget_set_sensitive (dialog->priv->restore_joliet, TRUE);
}

void
bonfire_filtered_dialog_get_status (BonfireFilteredDialog *dialog,
				    GSList **restored,
				    GSList **removed)
{
	BonfireFilterStatus status;
	GSList *retval_restored;
	GSList *retval_removed;
	GtkTreeModel *model;
	gboolean included;
	GtkTreeIter iter;
	char *uri;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (dialog->priv->tree));
	if (!gtk_tree_model_get_iter_first (model, &iter))
		return;

	retval_restored = NULL;
	retval_removed = NULL;
	do {
		gtk_tree_model_get (model, &iter,
				    URI_COL, &uri,
				    STATUS_COL, &status,
				    INCLUDED_COL, &included, -1);

		if (status == BONFIRE_FILTER_UNREADABLE
		||  status == BONFIRE_FILTER_RECURSIVE_SYM) {
			g_free (uri);
			continue;
		}

		if (included)
			retval_restored = g_slist_prepend (retval_restored, uri);
		else
			retval_removed = g_slist_prepend (retval_removed, uri);
	} while (gtk_tree_model_iter_next (model, &iter));

	*restored = retval_restored;
	*removed = retval_removed;
}

static gboolean
bonfire_filtered_dialog_joliet_warning_dialog (BonfireFilteredDialog *dialog)
{
	GtkWidget *warn_dialog;
	int answer;

	if (dialog->priv->disable_joliet_warn)
		return TRUE;

	warn_dialog = gtk_message_dialog_new (GTK_WINDOW (dialog),
					      GTK_DIALOG_DESTROY_WITH_PARENT |
					      GTK_DIALOG_MODAL,
					      GTK_MESSAGE_WARNING,
					      GTK_BUTTONS_OK_CANCEL,
					      _("Microsoft compatibility will be disabled:"));

	gtk_window_set_title (GTK_WINDOW (warn_dialog), _("Miscrosoft compatibility"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (warn_dialog),
						  _("Do you want to continue?"));
	answer = gtk_dialog_run (GTK_DIALOG (warn_dialog));
	gtk_widget_destroy (warn_dialog);
	if (answer != GTK_RESPONSE_OK)
		return FALSE;

	dialog->priv->disable_joliet_warn = 1;
	return TRUE;
}

static gboolean
_foreach_restored_uri_cb (char *uris,
			  int n_uris,
			  BonfireFilteredDialog *dialog)
{
	if (n_uris > 1) {
		dialog->priv->disable_joliet_warn = 1;
		return TRUE;
	}

	dialog->priv->disable_joliet_warn = 0;
	return FALSE;
}

static void
bonfire_filtered_dialog_joliet_item_excluded (BonfireFilteredDialog *dialog,
					      const char *uri)
{
	int n_uris;
	char *parent;

	parent = g_path_get_dirname (uri);
	n_uris = GPOINTER_TO_INT (g_hash_table_lookup (dialog->priv->joliet_hash, parent));
	n_uris --;
	if (!n_uris) {
		g_hash_table_remove (dialog->priv->joliet_hash,
				     parent);
		g_free (parent);
	}
	else 
		g_hash_table_replace (dialog->priv->joliet_hash,
				      parent,
				      GINT_TO_POINTER (n_uris));

	g_hash_table_find (dialog->priv->joliet_hash,
			   (GHRFunc) _foreach_restored_uri_cb,
			   dialog);
}

static gboolean
bonfire_filtered_dialog_joliet_item_restored (BonfireFilteredDialog *dialog,
					      const char *uri)
{
	int n_uris = 0;
	char *parent;

	parent = g_path_get_dirname (uri);

	if (!dialog->priv->joliet_hash) {
		dialog->priv->joliet_hash = g_hash_table_new_full (g_str_hash,
								   g_str_equal,
								   g_free,
								   NULL);
	}
	else 
		n_uris = GPOINTER_TO_INT (g_hash_table_lookup (dialog->priv->joliet_hash, parent));

	if (n_uris && !bonfire_filtered_dialog_joliet_warning_dialog (dialog)) {
		g_free (parent);
		return TRUE;
	}

	n_uris ++;
	g_hash_table_replace (dialog->priv->joliet_hash,
			      parent,
			      GINT_TO_POINTER (n_uris));

	return FALSE;
}

static void
bonfire_filtered_dialog_item_state_changed (BonfireFilteredDialog *dialog,
				            const GtkTreePath *path)
{
	BonfireFilterStatus status;
	GtkTreeModel *model;
	GtkTreeIter iter;
	gboolean active;
	char *uri;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (dialog->priv->tree));
	gtk_tree_model_get_iter (model, &iter, (GtkTreePath*) path);

	gtk_tree_model_get (model, &iter,
			    URI_COL, &uri,
			    STATUS_COL, &status,
			    INCLUDED_COL, &active, -1);

	if (active) { /* (RE) EXCLUDE */
		if (status == BONFIRE_FILTER_JOLIET_INCOMPAT)
			bonfire_filtered_dialog_joliet_item_excluded (dialog, uri);

		gtk_list_store_set (GTK_LIST_STORE (model), &iter,
				    INCLUDED_COL, FALSE, -1);
	}
	else { /* RESTORE */
		if (status == BONFIRE_FILTER_JOLIET_INCOMPAT
		&&  bonfire_filtered_dialog_joliet_item_restored (dialog, uri)) {
			g_free (uri);
			return;
		}

		gtk_list_store_set (GTK_LIST_STORE (model), &iter,
				    INCLUDED_COL, TRUE, -1);
	}

	g_free (uri);
}

static void
bonfire_filtered_dialog_item_toggled_cb (GtkCellRendererToggle *toggle,
					 const gchar *path,
					 BonfireFilteredDialog *dialog)
{
	GtkTreePath *treepath;

	treepath = gtk_tree_path_new_from_string (path);
	bonfire_filtered_dialog_item_state_changed (dialog, treepath);
	gtk_tree_path_free (treepath);
}

static void
bonfire_filtered_dialog_restore_all (BonfireFilteredDialog *dialog,
				     BonfireFilterStatus status)
{
	GtkTreeIter iter;
	GtkTreeModel *model;
	BonfireFilterStatus row_status;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (dialog->priv->tree));
	if (!gtk_tree_model_get_iter_first (model, &iter))
		return;

	do {
		gtk_tree_model_get (model, &iter,
				    STATUS_COL, &row_status, -1);

		if (status == row_status) {
			if (status == BONFIRE_FILTER_JOLIET_INCOMPAT) {
				char *uri;

				gtk_tree_model_get (model, &iter,
						    URI_COL, &uri, -1);
				bonfire_filtered_dialog_joliet_item_restored (dialog,
									      uri);
				g_free (uri);
			}

			gtk_list_store_set (GTK_LIST_STORE (model), &iter,
					    INCLUDED_COL, TRUE, -1);
		}

	} while (gtk_tree_model_iter_next (model, &iter));
}

static void
bonfire_filtered_dialog_exclude_all (BonfireFilteredDialog *dialog,
				     BonfireFilterStatus status)
{
	GtkTreeIter iter;
	GtkTreeModel *model;
	BonfireFilterStatus row_status;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (dialog->priv->tree));
	if (!gtk_tree_model_get_iter_first (model, &iter))
		return;

	do {
		gtk_tree_model_get (model, &iter,
				    STATUS_COL, &row_status, -1);

		if (status == row_status)
			gtk_list_store_set (GTK_LIST_STORE (model), &iter,
					    INCLUDED_COL, FALSE, -1);

	} while (gtk_tree_model_iter_next (model, &iter));
}

static void
bonfire_filtered_dialog_restore_hidden_cb (GtkButton *button,
					   BonfireFilteredDialog *dialog)
{
	if (!dialog->priv->hidden_state) {
		gtk_button_set_label (button, _("Exclude hidden files"));
		bonfire_filtered_dialog_restore_all (dialog, BONFIRE_FILTER_HIDDEN);
		dialog->priv->hidden_state = 1;
	}
	else {
		gtk_button_set_label (button, _("Restore hidden files"));
		bonfire_filtered_dialog_exclude_all (dialog, BONFIRE_FILTER_HIDDEN);
		dialog->priv->hidden_state = 0;
	}
}

static void
bonfire_filtered_dialog_restore_broken_symlink_cb (GtkButton *button,
						   BonfireFilteredDialog *dialog)
{
	if (!dialog->priv->broken_state) {
		gtk_button_set_label (button, _("Exclude broken symlinks"));
		bonfire_filtered_dialog_restore_all (dialog, BONFIRE_FILTER_BROKEN_SYM);
		dialog->priv->broken_state = 1;
	}
	else {
		gtk_button_set_label (button, _("Restore broken symlinks"));
		bonfire_filtered_dialog_exclude_all (dialog, BONFIRE_FILTER_BROKEN_SYM);
		dialog->priv->broken_state = 0;
	}
}

static void
bonfire_filtered_dialog_restore_joliet_incompat_cb (GtkButton *button,
						    BonfireFilteredDialog *dialog)
{
	if (!dialog->priv->joliet_state) {
		if (!bonfire_filtered_dialog_joliet_warning_dialog (dialog))
			return;

		gtk_button_set_label (button, _("Exclude joliet incompatible files"));
		bonfire_filtered_dialog_restore_all (dialog, BONFIRE_FILTER_JOLIET_INCOMPAT);
		dialog->priv->joliet_state = 1;
		
	}
	else {
		gtk_button_set_label (button, _("Restore joliet incompatible files"));
		bonfire_filtered_dialog_exclude_all (dialog, BONFIRE_FILTER_JOLIET_INCOMPAT);
		dialog->priv->joliet_state = 0;
		dialog->priv->disable_joliet_warn = 0;
		g_hash_table_destroy (dialog->priv->joliet_hash);
		dialog->priv->joliet_hash = NULL;
	}
}

static void
bonfire_filtered_dialog_row_activated_cb (GtkTreeView *tree,
                                          GtkTreePath *path,
                                          GtkTreeViewColumn *column,
                                          BonfireFilteredDialog *dialog)
{
	bonfire_filtered_dialog_item_state_changed (dialog, path);
}
