/* extract_base.c
 *
 * Purpose: perform a few sanity checks then install the base system. 
 * The base system can be installed from the network, 
 * CD, or any mounted filesystem.
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <sys/mount.h>
#include "dbootstrap.h"
#include "net-fetch.h"
#include "lang.h"
#include <syslog.h> 
#include "util.h"
#include <assert.h>
#include <sys/types.h>
#include <signal.h>
#include <sys/wait.h>
#include <unistd.h>
#include <newt.h>

char *base_filenames[] = { "base12", "base14", NULL };

int extract_from_file (const char *fil, const char *descr)
{
  int status;

  sprintf(prtbuf,_("Extracting %s from %s..."),descr,fil);
  pleaseWaitBox(prtbuf);
  chdir("/target");
  sprintf(prtbuf,"tar -xf %s",fil);
  status=execlog(prtbuf, LOG_INFO);
  chdir("/");
  boxPopWindow();
  if (status) {
    sprintf(prtbuf,_("There was a problem extracting the %s from %s"),descr,fil);
    problemBox(prtbuf,_("File error!"));
    return 1;
  }
  return 0;
}

static int
install_base_floppies (const char *device, const char *filename, const char *descr)
{
  int          status;
  struct stat  statbuf;

  status = getFloppies (_("The base deb packages are being copied from the %s floppy drive.\n"
                          "\n"
                          "Please insert base disk %d."),
		        device, filename, base_filenames, _("the base series"));

  if (status == 0) {
    status = extract_from_file (filename, descr);
  }
  else if (status != DLG_CANCEL) {
    vaproblemBox(prtbuf, _("Floppy Error"),
                 _("There was a problem reading the %s from %s"), descr, device); 
  }

  if (NAME_ISREG (filename, &statbuf)) {
    unlink (filename);
  }

  return status;
}

/* TODO
 * test proxy support
 * pick default hostname more intelligently, can we figure out where we are?
 */ 


/*
   execute 'debootstrap'
   
   debootstrap is a shell script that grabs base system packages and manually
   installs them onto the larget filesystem.  Our interaction with debootstrap
   is rather peculiar.  debootstrap sends us messages through fd 3.
   debootstrap's stdout and stderr are verbose messages from dpkg that most
   people don't want to see.  

   argv - path to debootstrap as well as arguments to pass it

   return 0 on success, != 0 on failure
*/
static int
exec_debootstrap(char **argv){

    int to_db[2], from_db[2]; /* 0=read, 1=write */
    char *ptr;
    FILE *ifp, *ofp;
    char buf[256];
    pid_t pid;
    int status, rv;

    pipe(to_db);
    pipe(from_db);
    
    if ((pid = fork()) == 0 )
    {
	close(to_db[1]);
	close(from_db[0]);
	
	if ( (dup2(to_db[0], 0) == -1) ||
		(dup2(from_db[1], 3) == -1))
	    perror("dup2");

	if (freopen("/dev/tty4", "w", stderr)== NULL)
	    /* okay, so we'll log it */
	    if (freopen("/target/tmp/debootstrap.log", "w", stderr)== NULL)
		perror("freopen");

	dup2(2,1);

   setenv("PERL_BADLANG", "0", 1);
	if (execv(argv[0], argv) != 0)
	    perror("execv");
	return -1;
    }
    else if (pid == -1)
    {
	perror("fork");
	return -1;
    }

    close(to_db[0]);
    close(from_db[1]);

    if ((ifp = fdopen(from_db[0], "r")) == NULL ) {
	perror("fdopen");
	return -1;
    }
    
    if ((ofp = fdopen(to_db[1], "w")) == NULL ) {
	perror("fdopen");
	return -1;
    }
  
    if (serialConsole >= 0
#if #cpu(s390) /* s390 does not support virtual consoles */
	|| 1
#endif
       ) {
	boxProgress(PROGRESS_INIT, _("Installing Base System"));
    } else {
	boxProgress(PROGRESS_INIT, _("Installing Base System, please wait..."));
    }
    boxProgress(PROGRESS_SETTEXT, _("Calling debootstrap"));

    while ((ptr = fgets (buf, sizeof (buf), ifp)))
      {
	  switch (ptr[0])
	    {
	    case 'W':
		{
		    boxProgress (PROGRESS_PAUSE, NULL);
		    problemBox (ptr + 3, _("Warning"));
		    boxProgress (PROGRESS_RESUME, NULL);
		    break;
		}
	    case 'E':
		{
		    boxProgress (PROGRESS_CLOSE, NULL);
		    problemBox (ptr + 3, _("Error"));
		    return -1;
		}
	    case '?':
		{
		    boxProgress (PROGRESS_PAUSE, NULL);
		    /* ?: bool Please just tell me: yes, or no? */
		    if (strncmp (ptr + 3, "bool", strlen ("bool")))
		      {
			  if ((yesNoBox (ptr + 8, _("Question")) == -1))
			      fwrite ("no", sizeof (char), strlen ("no"),
				      ofp);
			  else
			      fwrite ("yes", sizeof (char), strlen ("yes"),
				      ofp);
		      }
		    boxProgress (PROGRESS_RESUME, NULL);
		    break;
		}
	    case 'P':
		{
		    boxProgress (PROGRESS_UPDATE, ptr + 3);
		    break;
		}
	    case 'I':
		{
		    if (!strncmp (ptr, "I: Retrieving ", 14)
			|| !strncmp (ptr, "I: Extracting ", 14)
			|| !strncmp (ptr, "I: Validating ", 14)) {
			/* Retrieving or Extracting; assume file name follows.                           
			 * Force filename on to following line, to stop the 
			 * text jumping around so much.  */
			ptr[13] = '\n';
			boxProgress (PROGRESS_SETTEXT, ptr + 3);
		    }
		    else if (!strncmp (ptr, "I: Installing ", 14)
			     || !strncmp (ptr, "I: Unpacking ", 13)
			     || !strncmp (ptr, "I: Configuring ", 15)
			     || !strncmp (ptr, "I: Base system installed", 24)) {
			/* This is for basic progress during the
                         * unpacking and actual installation of the
                         * base system, no long filenames thus no need
                         * for the newline intrusion.  */
			boxProgress(PROGRESS_SETTEXT, ptr + 3);
		    }
		    break;
		}
	    default:
		{
		    /* Unknown, display the whole thing. */
		    boxProgress (PROGRESS_SETTEXT, ptr);
		    break;
		}
	    }
      }
  

    boxProgress(PROGRESS_CLOSE, NULL);

    if (waitpid(pid, &status, 0) != -1 && (WIFEXITED(status) != 0 ))
    {
	rv = WEXITSTATUS(status);
    	if (rv != 0 ) {
	    snprintf(buf, sizeof(buf), 
                _("debootstrap exited with an error (return value %d)"), 
	        rv);
	    problemBox(buf, _("Base Installation Error"));
	    return rv;
	}
	return 0;
    }
    else {
	kill(SIGKILL, pid);
	problemBox(_("debootstrap exited abnormally"), _("Base Installation Error"));
	return 1;
    }

    return 0;
}

#define BASESUITE "woody" /* FIXME: change to stable for release */

int 
debootstrap_extract_base (void) {
  int rs=0;
  char *argv[8];
  struct stat statbuf;
  char *source;
  char filename [512];
  int status;

  /* Sanity Check */
  if (NAME_ISREG ("/target/etc/debian_version", &statbuf)) {
    if (yesNoBox(_("It looks like either you have already installed the base system once or there is a Debian system installed already. This installer routine is not intended to overwrite an existing system. It's safer to start from scratch, i.e. umount the partition, create a new filesystem upon it and come back to this step.  Installing the base system over an existing system will cause existing data to be overwritten. Do you want to continue?"), _("Warning")) == DLG_NO)
      return DLG_CANCEL;
  } else {
    if (NAME_ISREG ("/target/sbin/init", &statbuf)) {
      if (yesNoBox(_("It looks like you are trying to install Debian GNU/Linux over an existing Linux system. This installer routine is not intended to overwrite an existing system. It's safer to start from scratch, i.e. umount the partition, create a new filesystem upon it and come back to this step.  Installing the base system over an existing system will result in existing data being overwritten. Do you want to continue?"), _("Warning")) == DLG_NO)
      return DLG_CANCEL;
    }
  }

  disqtype=debootstrap;
  if ( choose_medium() ) return 1;
  
  DEBUGMSG("Extracting base, Archive_Dir: %s", Archive_Dir);
  if (!strcmp (Archive_Dir, "netfetch"))
    {
	nf_initialize();
	free (nf_state.path);
	nf_state.path=strdup("debian");
	if( nf_select_server() != 0 )
	    return 1;

	snprintf(prtbuf, sizeof prtbuf, "%s://%s:%d/%s", 
		nf_state.method,
		nf_state.server.hostname, 
		nf_state.server.port,
		nf_state.path); 
	source = strdup(prtbuf);

	/* just test to see if this source gives us a Release file */
	sprintf(prtbuf,"wget -q %s/dists/woody/Release", source);
	if ( execlog(prtbuf, LOG_INFO) ) 
	    {
	    /* Any response from wget -q means there was an error. Check vc3. 
	     * 2 possible errors are
	     * wget: $hostname: Unknown host
	     * wget: server returned error 404: HTTP:/1.1 404 Not Found
	     */
		unlink ("Release"); /* in case it did something */
		sprintf(prtbuf,_("The server was unavailable or contained no Release file"));
		problemBox(prtbuf,_("Release Check Failed"));
		return 1;
	    }
	unlink ("Release");
    }
    /*
     * in case we're installing from a floppy device ...
     */
  else if (! strncmp(Archive_Dir,"/dev/fd",7)
#ifdef SCSI_FLOPPY
	   || ! strncmp(Archive_Dir,"/dev/sfd",8)
#endif
	   ) {
    strcpy(filename, "/target/" BASEDEBS);
    status = install_base_floppies(Archive_Dir, filename, _("Base debs"));
    eject_floppy(Archive_Dir);
    if (status) {
	ERRMSG("extracting base debs from floppy device %s failed with %d", Archive_Dir, status);
	return 1;
    }
    source = strdup("null:");
  }
  /* Are we trying to install from basedebs.tar? */
  else if (
	   ({
	       snprintf(prtbuf, sizeof(prtbuf), "%s/%s",
			Archive_Dir, BASEDEBS);
	       NAME_ISREG(prtbuf, &statbuf);
	   })) {
      char *filename = strdup(prtbuf);
      status = extract_from_file (filename, _("Base debs"));
      if (status) {
	  ERRMSG("extracting %s failed with %d", filename, status);
	  free(filename);
	  return 1;
      }
      free(filename);
      source = strdup("null:");
  } else {
      /* archive is on the filesystem */
      snprintf(prtbuf, sizeof prtbuf, "file:%s", Archive_Dir); 
      source = strdup(prtbuf);
  }
  argv[0] = "/usr/sbin/debootstrap";
  argv[1] = "--boot-floppies";
  argv[2] = "--arch";
  argv[3] = ARCHNAME;
  argv[4] = "woody";
  argv[5] = "/target";
  argv[6] = source;
  argv[7] = NULL;
  
  INFOMSG("running '%s %s %s %s %s %s %s'",
          argv[0], argv[1], argv[2], argv[3], argv[4], argv[5], argv[6]);

  rs = exec_debootstrap(argv);

  if (rs) {
      unmount_dir("/instmnt"); /* release the disk */
      return -1;
  }

    
  if (!strcmp (Archive_Dir, "netfetch")){
      write_userconfig("DEBIAN_MIRROR_HOSTNAME", nf_state.server.hostname);
      write_userconfig("DEBIAN_MIRROR_METHOD", nf_state.method);
      write_userconfig("DEBIAN_MIRROR_PATH", nf_state.path);
      snprintf(prtbuf, sizeof prtbuf, "%d", nf_state.server.port);
      write_userconfig("DEBIAN_MIRROR_PORT", prtbuf);
  }

  write_userconfig ("SUITE", BASESUITE);

  free (source);
 
  if (NAME_ISREG("/target/etc/inittab",&statbuf)) {
      rename("/target/etc/inittab","/target/etc/inittab.real");
  }
  sprintf(prtbuf,"cp %s %s","/etc/inittab.install","/target/etc/inittab"); 
  execlog(prtbuf, LOG_INFO);

  /* ioctl.save may contain incorrect settings for the serial console.  */
  if (serialConsole >= 0 && NAME_ISREG("/target/etc/ioctl.save",&statbuf))
    unlink("/target/etc/ioctl.save");

  configure_base () ;
  
  sync();
  return 0;
  
}


#ifdef _TESTING_
int
main(){
    LOAD_TRMFILE ("test.trm");
    InstallationRootDevice="/dev/ram";
    get_kver();
    boxInit();
    debootstrap_extract_base();
    return 0;
}
#endif

/*
  Local Variables:
  c-basic-offset: 4
  End:
*/
