//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Job/JobPropertiesWidget.cpp
//! @brief     Implements class JobPropertiesWidget
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Job/JobPropertiesWidget.h"
#include "GUI/Model/Job/JobItem.h"
#include "GUI/View/Job/JobPropertiesTableModel.h"
#include "GUI/View/Tool/mainwindow_constants.h"
#include <QHeaderView>
#include <QTextEdit>
#include <QTreeView>
#include <QVBoxLayout>

namespace {

enum ETabId { JOB_PROPERTIES, JOB_MESSAGES };
}

//==================================================================================================
// JobPropertiesWidget
//==================================================================================================

//--------------------------------------------------------------------------------------------------
// public member functions
//--------------------------------------------------------------------------------------------------

JobPropertiesWidget::JobPropertiesWidget(QWidget* parent, Qt::WindowFlags f)
    : QWidget(parent, f)
    , m_jobItem(nullptr)
{
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::MinimumExpanding);
    setWindowTitle(GUI::Constants::JobPropertiesWidgetName);

    auto* layout = new QVBoxLayout(this);
    layout->setContentsMargins(0, 0, 0, 0);
    layout->setSpacing(0);
    layout->setContentsMargins(0, 0, 0, 0);

    m_tabWidget = new QTabWidget(this);
    m_tabWidget->setTabPosition(QTabWidget::South);
    layout->addWidget(m_tabWidget);

    m_propertiesView = new QTreeView(this);
    m_propertiesView->setRootIsDecorated(false);
    m_propertiesView->setAlternatingRowColors(true);
    m_tabWidget->insertTab(JOB_PROPERTIES, m_propertiesView, "Job Properties");
    m_propertiesModel = new JobPropertiesTableModel(this);
    m_propertiesView->setModel(m_propertiesModel);
    m_propertiesView->header()->setSectionResizeMode(QHeaderView::ResizeToContents);

    m_commentsEditor = new QTextEdit(this);
    m_tabWidget->insertTab(JOB_MESSAGES, m_commentsEditor, "Messages");

    connect(m_commentsEditor, &QTextEdit::textChanged, this,
            &JobPropertiesWidget::onCommentsEdited);

    setMinimumWidth(10);
    setMinimumHeight(190);
    applySettings();
}

JobPropertiesWidget::~JobPropertiesWidget()
{
    if (m_jobItem)
        disconnect(m_jobItem, nullptr, this, nullptr);
    saveSettings();
}

void JobPropertiesWidget::setJobItem(JobItem* jobItem)
{
    if (m_jobItem)
        disconnect(m_jobItem, nullptr, this, nullptr);
    m_jobItem = jobItem;
    m_propertiesModel->setJobItem(m_jobItem);
    if (m_jobItem) {
        if (m_jobItem->isFailed()) {
            m_tabWidget->tabBar()->setTabTextColor(JOB_MESSAGES, Qt::red);
            m_tabWidget->setCurrentIndex(JOB_MESSAGES);
        } else
            m_tabWidget->tabBar()->setTabTextColor(JOB_MESSAGES, Qt::black);
        m_commentsEditor->setText(m_jobItem->comments());

        notifyJobPropertyChange();
    } else
        m_commentsEditor->clear();
}

//--------------------------------------------------------------------------------------------------
// private member functions
//--------------------------------------------------------------------------------------------------

void JobPropertiesWidget::notifyJobPropertyChange()
{
    connect(m_jobItem, &JobItem::jobCommentsChanged, this, [this](const QString&) {
        if (m_jobItem && m_jobItem->comments() != m_commentsEditor->toPlainText()) {
            m_commentsEditor->blockSignals(true);
            m_commentsEditor->setPlainText(m_jobItem->comments());
            m_commentsEditor->blockSignals(false);
        }
    });
}

void JobPropertiesWidget::applySettings()
{
    QSettings settings;
    if (settings.childGroups().contains(GUI::Constants::S_JOB_PROPERTIES_WIDGET)) {
        settings.beginGroup(GUI::Constants::S_JOB_PROPERTIES_WIDGET);
        m_tabWidget->setCurrentIndex(
            settings.value(GUI::Constants::S_JOB_PROPERTIES_WIDGET_CURRENT_TAB).toInt());
        settings.endGroup();
    }
}

void JobPropertiesWidget::saveSettings()
{
    QSettings settings;
    settings.beginGroup(GUI::Constants::S_JOB_PROPERTIES_WIDGET);
    settings.setValue(GUI::Constants::S_JOB_PROPERTIES_WIDGET_CURRENT_TAB,
                      m_tabWidget->currentIndex());
    settings.endGroup();
    settings.sync();
}

//--------------------------------------------------------------------------------------------------
// private slots
//--------------------------------------------------------------------------------------------------

void JobPropertiesWidget::onCommentsEdited()
{
    if (m_jobItem) {
        m_jobItem->blockSignals(true);
        m_jobItem->setComments(m_commentsEditor->toPlainText());
        m_jobItem->blockSignals(false);
    }
}
