//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Sample/SampleEditorController.h
//! @brief     Defines class SampleEditorController.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_VIEW_SAMPLE_SAMPLEEDITORCONTROLLER_H
#define BORNAGAIN_GUI_VIEW_SAMPLE_SAMPLEEDITORCONTROLLER_H

#include "GUI/Model/Sample/FormfactorCatalog.h"
#include "GUI/Model/Sample/ParticleCatalog.h"
#include <QObject>

class CompoundItem;
class CoreAndShellForm;
class DoubleProperty;
class ISelectionForm;
class InterferenceForm;
class InterferenceItem;
class Item3D;
class ItemWithMaterial;
class ItemWithParticles;
class LatticeTypeSelectionForm;
class LayerForm;
class LayerItem;
class MaterialsSet;
class MesocrystalForm;
class ParticleLayoutItem;
class SampleForm;
class SampleItem;

//! Class to modify a sample from the layer oriented sample editor.
//!
//! Use this class to modify the sample model. It takes care of notifications.
//! It operates on one SampleItem.
//!
//! Originally, it also supported undo/redo (never completed, withdrawn in release 22).

class SampleEditorController : public QObject {
    Q_OBJECT
public:
    SampleEditorController(SampleItem* multi);

    //! Set the current form.
    //!
    //! The form can change e.g. when a different sample gets the current one
    //! in the layer editor. Also nullptr is allowed.
    void setSampleForm(SampleForm* view);

    //! The item on which this controller operates.
    SampleItem* sampleItem() const { return m_sample_item; }

    //! The materials of the current document
    MaterialsSet* materialModel() const;

    void addLayerItem(LayerItem* before);
    QColor findColor(size_t atIndex);
    void onLayerAdded(LayerItem* layer);
    void duplicateLayerItem(const LayerItem* layer);
    void removeLayerItem(LayerItem* layer);

    void onLayoutAdded(LayerForm* layerForm, ParticleLayoutItem* layout);
    void addLayoutItem(LayerForm* layerForm);
    void duplicateLayoutItem(LayerForm* layerForm, ParticleLayoutItem* layout);
    void removeLayoutItem(LayerForm* layerForm, ParticleLayoutItem* layout);

    void onParticleLayoutAdded(ParticleLayoutItem* layout, ItemWithParticles* newItem);
    void addParticleLayoutItem(ParticleLayoutItem* layout, ParticleCatalog::Type type);
    void addParticleLayoutItem(ParticleLayoutItem* layout, FormfactorCatalog::Type type);
    ParticleLayoutItem* parentLayoutItem(ItemWithParticles* item);

    void onParticleCompoundAdded(CompoundItem* composition, ItemWithParticles* newItem);
    void addCompoundItem(CompoundItem* composition, ParticleCatalog::Type type);
    void addCompoundItem(CompoundItem* composition, FormfactorCatalog::Type type);
    CompoundItem* parentCompoundItem(ItemWithParticles* item);

    void duplicateItemWithParticles(ItemWithParticles* item);
    void removeParticle(ItemWithParticles* item);

    void setCoreFormfactor(CoreAndShellForm* widget, FormfactorCatalog::Type type);
    void setShellFormfactor(CoreAndShellForm* widget, FormfactorCatalog::Type type);

    void setMesocrystalBasis(MesocrystalForm* widget, ParticleCatalog::Type type);
    void setMesocrystalBasis(MesocrystalForm* widget, FormfactorCatalog::Type type);
    void selectInterference(InterferenceForm* widget, int newIndex);
    void setIntegrateOverXi(LatticeTypeSelectionForm* widget, bool value);

    void setDouble(double value, DoubleProperty& d);

    void selectMaterial(ItemWithMaterial* item, const QString& newMaterialIdentifier);
    void setMaterialValue(ItemWithMaterial* item, double value, DoubleProperty& d);

    //! Set an interference function's value which affects the total particle density of the
    //! containing particle layout.
    //!
    //! Some values in interference settings affect the total density in the particle layout which
    //! contains this interference function. Call this method to provide all the related updating
    //! (data & UI).
    void setDensityRelatedValue(InterferenceItem* interferenceItem, double value,
                                DoubleProperty& d);

    void onStoppedToMoveLayer(QWidget* widgetToMove, QWidget* moveAboveThisWidget);

signals:
    void requestViewInRealspace(Item3D* item);
    void aboutToRemoveItem(Item3D* item);
    void modified();

private:
    ItemWithParticles* createAndInitItem(FormfactorCatalog::Type formFactorType) const;
    ItemWithParticles* createAndInitItem(ParticleCatalog::Type itemType) const;

    SampleItem* m_sample_item;
    SampleForm* m_sample_form;
};

#endif // BORNAGAIN_GUI_VIEW_SAMPLE_SAMPLEEDITORCONTROLLER_H
