#!/usr/bin/env python3
"""
Plot SLD and magnetization profiles for a 
multilayer sample with rough interfaces.
"""

import bornagain as ba
from bornagain import angstrom, ba_plot as bp, R3
import numpy as np
import matplotlib.pyplot as plt


def get_sample():
    # materials
    vacuum = ba.MaterialBySLD("Vacuum", 0, 0)
    material_ti = ba.MaterialBySLD("Ti", -1.9493e-06, 0)
    B_ni = R3(-5e7, 0, 0)
    material_ni = ba.MaterialBySLD("Ni", 9.4245e-06, 0, B_ni)
    B_substrate = R3(1e8, 0, 0)
    material_substrate = ba.MaterialBySLD("SiSubstrate", 2.0704e-06, 0, 
                                          B_substrate)

    # layers
    ambient_layer = ba.Layer(vacuum)
    ti_layer = ba.Layer(material_ti, 30*angstrom)
    ni_layer = ba.Layer(material_ni, 70*angstrom)
    substrate_layer = ba.Layer(material_substrate)

    # sample
    sample = ba.MultiLayer()
    sample.addLayer(ambient_layer)
    roughness_model = ba.TanhRoughness()
    roughness = ba.LayerRoughness(5*angstrom, 0.5, 10*angstrom, roughness_model)
    for _ in range(4):
        sample.addLayerWithTopRoughness(ti_layer, roughness)
        sample.addLayerWithTopRoughness(ni_layer, roughness)
    sample.addLayer(substrate_layer)

    return sample


if __name__ == '__main__':
    sample = get_sample()    
    n_points = 400
    z_min, z_max = ba.defaultMaterialProfileLimits(sample)    
    z_points = ba.generateZValues(n_points, z_min, z_max)

    sld = ba.materialProfileSLD(sample, n_points, z_min, z_max)
    mag_X = ba.magnetizationProfile(sample, "X", n_points, z_min, z_max)    
    
    plt.subplot(2, 1, 1)
    plt.plot(z_points, np.real(sld))
    plt.title("Re(SLD)")

    plt.subplot(2, 1, 2)
    plt.plot(z_points, mag_X)
    plt.title("X magnetization")

    plt.tight_layout()

    bp.show_or_export()
