(**************************************************************************)
(*                                                                        *)
(*  Copyright (C) 2012 Johannes 'josch' Schauer <j.schauer@email.de>      *)
(*                                                                        *)
(*  This library is free software: you can redistribute it and/or modify  *)
(*  it under the terms of the GNU Lesser General Public License as        *)
(*  published by the Free Software Foundation, either version 3 of the    *)
(*  License, or (at your option) any later version.  A special linking    *)
(*  exception to the GNU Lesser General Public License applies to this    *)
(*  library, see the COPYING file for more information.                   *)
(**************************************************************************)

open ExtLib
open Common
open Algo

include Util.Logging(struct let label = __FILE__ end) ;;

module Boilerplate = BoilerplateNoRpm

module Options = struct
  open OptParse

  let description = ("Annotate buildgraph.xml and srcgraph.xml with strong dependency information")
  let usage = "%prog buildgraph.xml srcgraph.xml Packages Sources"

  let options = OptParser.make ~description ~usage
  include BootstrapCommon.MakeOptions(struct let options = options end)

  let noindep = StdOpt.store_false ()

  open OptParser

  let prog_group = add_group options "Program specific options" in
  add options ~group:prog_group ~long_name:"keep-indep" ~help:"Do not drop Build-Depends-Indep dependencies" noindep;

  include Boilerplate.InputOptions;;
  let default = List.filter (fun e -> not (List.mem e ["checkonly"; "latest";"outfile"])) Boilerplate.InputOptions.default_options in
  Boilerplate.InputOptions.add_options ~default options ;;

  include Boilerplate.DistribOptions;;
  let default = List.filter (fun e -> not (List.mem e ["deb-ignore-essential"; "inputtype"])) Boilerplate.DistribOptions.default_options in
  Boilerplate.DistribOptions.add_options ~default options ;;
end

let main () =
  let posargs = OptParse.OptParser.parse_argv Options.options in
  Boilerplate.enable_debug (OptParse.Opt.get Options.verbose);
  Util.Debug.disable "Depsolver_int";
  Boilerplate.all_quiet (OptParse.Opt.get Options.quiet);

  let noindep = OptParse.Opt.get Options.noindep in
  let options = Options.set_deb_options () in
  let buildarch = Option.get options.Debian.Debcudf.native in
  let hostarch = match options.Debian.Debcudf.host with None -> "" | Some s -> s in
  let foreignarchs = options.Debian.Debcudf.foreign in

  let bgf, sgf, posargs = match posargs with
   | bgf::sgf::l -> bgf,sgf,l
   | _ -> fatal "you must provide buildgraph, srcgraph, Packages and Sources"
  in

  let (binlist, (fgsrclist,bgsrclist), _) = BootstrapCommon.parse_packages ~noindep Options.parse_cmdline buildarch hostarch foreignarchs posargs in

  let tables = Debian.Debcudf.init_tables (fgsrclist@bgsrclist@binlist) in
  let fgsl = List.map (Debian.Debcudf.tocudf ~options tables) fgsrclist in
  let bgsl = List.map (Debian.Debcudf.tocudf ~options tables) bgsrclist in
  let pkglist = List.map (Debian.Debcudf.tocudf ~options tables) binlist in
  let universe = Cudf.load_universe (BootstrapCommon.unique [pkglist;fgsl;bgsl]) in

  let ic = open_in bgf in
  let bg = BuildGraph.from_ic universe ic in
  close_in ic;

  let ic = open_in sgf in
  let sg = SrcGraph.from_ic universe ic in
  close_in ic;

  let module BE = BuildGraphExtras.Make(struct let univ = universe end) in
  let module SE = SrcGraphExtras.Make(struct let univ = universe end) in

  let sdg = Strongdeps.strongdeps_univ universe in
  let is_strong pkg dep = (CudfAdd.compare pkg dep = 0) || (BootstrapCommon.pkg_list_mem (Defaultgraphs.PackageGraph.succ_list sdg pkg) dep) in
  let bg = BE.annotate is_strong bg in
  let sg = SE.annotate is_strong sg in

  let module BGP = BuildGraph.Printer(struct let univ = universe end) in
  let module SGP = SrcGraph.Printer(struct let univ = universe end) in

  let oc = open_out bgf in
  BGP.print (Format.formatter_of_out_channel oc) bg;
  close_out oc;
  
  let oc = open_out sgf in
  SGP.print (Format.formatter_of_out_channel oc) sg;
  close_out oc;
;;

main ();;
