/***************************************************************************
 *            brasero-layout.c
 *
 *  mer mai 24 15:14:42 2006
 *  Copyright  2006  Rouquier Philippe
 *  brasero-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */



#include <string.h>

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <glib/gi18n-lib.h>

#include <gtk/gtkuimanager.h>
#include <gtk/gtktoggleaction.h>
#include <gtk/gtkradioaction.h>
#include <gtk/gtkaction.h>

#include <gtk/gtkstock.h>

#include <gtk/gtkbox.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkvpaned.h>
#include <gtk/gtkimage.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtkhseparator.h>

#include <gconf/gconf-client.h>

#include "brasero-layout.h"


static void brasero_layout_class_init (BraseroLayoutClass *klass);
static void brasero_layout_init (BraseroLayout *sp);
static void brasero_layout_finalize (GObject *object);

static void
brasero_layout_preview_toggled_cb (GtkToggleAction *action,
				   BraseroLayout *layout);
static void
brasero_layout_radio_toggled_cb (GtkRadioAction *action,
				 GtkRadioAction *current,
				 BraseroLayout *layout);

static void
brasero_layout_show (GtkWidget *widget);
static void
brasero_layout_hide (GtkWidget *widget);

typedef struct {
	gchar *id;
	GtkWidget *widget;
	BraseroLayoutType types;
} BraseroLayoutItem;

struct BraseroLayoutPrivate {
	GtkActionGroup *action_group;

	gint accel;

	BraseroLayoutType type;
	GSList *items;

	GConfClient *client;
	gint radio_notify;
	gint preview_notify;

	GtkWidget *notebook;
	GtkWidget *main_box;
	GtkWidget *preview_pane;
};


static GObjectClass *parent_class = NULL;

#define BRASERO_LAYOUT_PROJECT_ID	"Project"
#define BRASERO_LAYOUT_PROJECT_NAME	N_("Project")
#define BRASERO_LAYOUT_PROJECT_ICON	GTK_STOCK_CDROM /* it depends on project type */
#define BRASERO_LAYOUT_PREVIEW_ID	"Viewer"
#define BRASERO_LAYOUT_PREVIEW_NAME	N_("Preview")
#define BRASERO_LAYOUT_PREVIEW_MENU	N_("Preview")
#define BRASERO_LAYOUT_PREVIEW_TOOLTIP	N_("Display video, audio and image preview")
#define BRASERO_LAYOUT_PREVIEW_ICON	GTK_STOCK_FILE

/* GCONF keys */
#define BRASERO_KEY_DISPLAY_DIR		"/apps/brasero/display/"
#define BRASERO_KEY_SHOW_PREVIEW	BRASERO_KEY_DISPLAY_DIR "preview"
#define BRASERO_KEY_LAYOUT_AUDIO	BRASERO_KEY_DISPLAY_DIR "audio_pane"
#define BRASERO_KEY_LAYOUT_DATA		BRASERO_KEY_DISPLAY_DIR "data_pane"

GType
brasero_layout_get_type ()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BraseroLayoutClass),
			NULL,
			NULL,
			(GClassInitFunc)brasero_layout_class_init,
			NULL,
			NULL,
			sizeof (BraseroLayout),
			0,
			(GInstanceInitFunc)brasero_layout_init,
		};

#ifdef BUILD_GDL
		type = g_type_register_static (GDL_TYPE_DOCK, 
					       "BraseroLayout",
					       &our_info,
					       0);
#else
		type = g_type_register_static (GTK_TYPE_HPANED, 
					       "BraseroLayout",
					       &our_info,
					       0);
#endif
	}

	return type;
}

static void
brasero_layout_class_init (BraseroLayoutClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GtkWidgetClass *gtk_widget_class = GTK_WIDGET_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);
	object_class->finalize = brasero_layout_finalize;

	gtk_widget_class->hide = brasero_layout_hide;
	gtk_widget_class->show = brasero_layout_show;
}

static void
brasero_layout_init (BraseroLayout *obj)
{
	obj->priv = g_new0 (BraseroLayoutPrivate, 1);

	obj->priv->action_group = gtk_action_group_new ("BraseroLayoutActions");
	gtk_action_group_set_translation_domain (obj->priv->action_group, GETTEXT_PACKAGE);

	/* init GConf */
	obj->priv->client = gconf_client_get_default ();

	/* set up containers */
	obj->priv->main_box = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (obj->priv->main_box), 6);
	gtk_widget_show (obj->priv->main_box);
	gtk_paned_pack2 (GTK_PANED (obj), obj->priv->main_box, TRUE, FALSE);

	obj->priv->notebook = gtk_notebook_new ();
	gtk_widget_show (obj->priv->notebook);
	gtk_notebook_set_show_border (GTK_NOTEBOOK (obj->priv->notebook), FALSE);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (obj->priv->notebook), FALSE);
	gtk_box_pack_start (GTK_BOX (obj->priv->main_box),
			    obj->priv->notebook,
			    TRUE,
			    TRUE,
			    0);
}

static void
brasero_layout_finalize (GObject *object)
{
	BraseroLayout *cobj;
	GSList *iter;

	cobj = BRASERO_LAYOUT(object);

	for (iter = cobj->priv->items; iter; iter = iter->next) {
		BraseroLayoutItem *item;

		item = iter->data;
		g_free (item->id);
		g_free (item);
	}

	g_slist_free (cobj->priv->items);
	cobj->priv->items = NULL;

	/* close GConf */
	gconf_client_notify_remove (cobj->priv->client, cobj->priv->preview_notify);
	gconf_client_notify_remove (cobj->priv->client, cobj->priv->radio_notify);
	g_object_unref (cobj->priv->client);

	g_free(cobj->priv);
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

GtkWidget *
brasero_layout_new ()
{
	BraseroLayout *obj;
	
	obj = BRASERO_LAYOUT (g_object_new (BRASERO_TYPE_LAYOUT, NULL));

	return GTK_WIDGET (obj);
}

static void
brasero_layout_show (GtkWidget *widget)
{
	BraseroLayout *layout;

	layout = BRASERO_LAYOUT (widget);

	gtk_action_group_set_visible (layout->priv->action_group, TRUE);
	gtk_action_group_set_sensitive (layout->priv->action_group, TRUE);
	gtk_widget_set_sensitive (widget, TRUE);

	if (GTK_WIDGET_CLASS (parent_class)->show)
		GTK_WIDGET_CLASS (parent_class)->show (widget);
}

static void
brasero_layout_hide (GtkWidget *widget)
{
	BraseroLayout *layout;

	layout = BRASERO_LAYOUT (widget);

	gtk_action_group_set_visible (layout->priv->action_group, FALSE);
	gtk_action_group_set_sensitive (layout->priv->action_group, FALSE);
	gtk_widget_set_sensitive (widget, FALSE);

	if (GTK_WIDGET_CLASS (parent_class)->hide)
		GTK_WIDGET_CLASS (parent_class)->hide (widget);
}

void
brasero_layout_add_project (BraseroLayout *layout,
			    GtkWidget *project)
{
	gtk_container_set_border_width (GTK_CONTAINER (project), 6);
	gtk_paned_pack1 (GTK_PANED (layout), project, TRUE, FALSE);
}

static GtkWidget *
_make_pane (GtkWidget *widget,
	    const char *icon,
	    const char *text,
	    gboolean fill)
{
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *image;

	vbox = gtk_vbox_new (FALSE, 6);
	gtk_widget_show (vbox);

	gtk_box_pack_end (GTK_BOX (vbox), widget, fill, fill, 0);

	hbox = gtk_hbox_new (FALSE, 6);
	gtk_widget_show (hbox);
	gtk_box_pack_end (GTK_BOX (vbox), hbox, FALSE, FALSE, 4);

	image = gtk_image_new_from_stock (icon, GTK_ICON_SIZE_LARGE_TOOLBAR);
	gtk_widget_show (image);
	gtk_misc_set_alignment (GTK_MISC (image), 0.0, 0.0);
	gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, TRUE, 0);

	label = gtk_label_new (text);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);

	return vbox;
}

static void
brasero_layout_preview_toggled_cb (GtkToggleAction *action, BraseroLayout *layout)
{
	gboolean active;

	active = gtk_toggle_action_get_active (action);
	if (active)
		gtk_widget_show (layout->priv->preview_pane);
	else
		gtk_widget_hide (layout->priv->preview_pane);
	
	/* we set the correct value in GConf */
	gconf_client_set_bool (layout->priv->client,
			       BRASERO_KEY_SHOW_PREVIEW,
			       active,
			       NULL);
}

static void
brasero_layout_preview_changed_cb (GConfClient *client,
				   guint cxn,
				   GConfEntry *entry,
				   gpointer data)
{
	BraseroLayout *layout;
	GtkAction *action;
	GConfValue *value;
	gboolean active;

	layout = BRASERO_LAYOUT (data);

	value = gconf_entry_get_value (entry);
	if (value->type != GCONF_VALUE_BOOL)
		return;

	active = gconf_value_get_bool (value);
	action = gtk_action_group_get_action (layout->priv->action_group,
					      BRASERO_LAYOUT_PREVIEW_ID);
	gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action), active);

 	if (active)
		gtk_widget_show (layout->priv->preview_pane);
	else
		gtk_widget_hide (layout->priv->preview_pane);
}

void
brasero_layout_add_preview (BraseroLayout *layout,
			    GtkWidget *preview)
{
	gboolean active;
	GtkAction *action;
	gchar *accelerator;
	GError *error = NULL;
	GtkToggleActionEntry entry;

	layout->priv->preview_pane = preview;
	gtk_box_pack_end (GTK_BOX (layout->priv->main_box),
			  layout->priv->preview_pane,
			  FALSE,
			  FALSE,
			  0);

	/* add menu entry in display */
	accelerator = g_strdup ("F11");

	entry.name = BRASERO_LAYOUT_PREVIEW_ID;
	entry.stock_id = BRASERO_LAYOUT_PREVIEW_ICON;
	entry.label = BRASERO_LAYOUT_PREVIEW_MENU;
	entry.accelerator = accelerator;
	entry.tooltip = BRASERO_LAYOUT_PREVIEW_TOOLTIP;
	entry.is_active = FALSE;
	entry.callback = G_CALLBACK (brasero_layout_preview_toggled_cb);
	gtk_action_group_add_toggle_actions (layout->priv->action_group,
					     &entry,
					     1,
					     layout);
	g_free (accelerator);

	/* initializes the display */
	active = gconf_client_get_bool (layout->priv->client,
					BRASERO_KEY_SHOW_PREVIEW,
					&error);
	if (error) {
		g_warning ("Can't access GConf key %s. This is probably harmless (first launch of brasero).\n", error->message);
		g_error_free (error);
		error = NULL;
	}

	action = gtk_action_group_get_action (layout->priv->action_group, BRASERO_LAYOUT_PREVIEW_ID);
	gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action), active);

	if (active)
		gtk_widget_show (layout->priv->preview_pane);
	else
		gtk_widget_hide (layout->priv->preview_pane);

	if (!layout->priv->preview_notify)
		layout->priv->preview_notify = gconf_client_notify_add (layout->priv->client,
									BRASERO_KEY_SHOW_PREVIEW,
									brasero_layout_preview_changed_cb,
									layout,
									NULL,
									&error);
	if (error) {
		g_warning ("Could set notify for GConf key %s.\n", error->message);
		g_error_free (error);
		error = NULL;
	}
}

/**************************** for the source panes *****************************/
static void
brasero_layout_pane_changed (BraseroLayout *layout, const char *id)
{
	GSList *iter;
	BraseroLayoutItem *item;

	for (iter = layout->priv->items; iter; iter = iter->next) {
		item = iter->data;
		if (!strcmp (id, item->id))
			gtk_widget_show (item->widget);
		else
			gtk_widget_hide (item->widget);
	}
}

static void
brasero_layout_radio_toggled_cb (GtkRadioAction *action,
				 GtkRadioAction *current,
				 BraseroLayout *layout)
{
	GError *error = NULL;
	const char *id;

	id = gtk_action_get_name (GTK_ACTION (current));
	brasero_layout_pane_changed (layout, id);

	/* update gconf value */
	if (layout->priv->type == BRASERO_LAYOUT_AUDIO)
		gconf_client_set_string (layout->priv->client,
					 BRASERO_KEY_LAYOUT_AUDIO,
					 id,
					 &error);
	else if (layout->priv->type == BRASERO_LAYOUT_DATA)
		gconf_client_set_string (layout->priv->client,
					 BRASERO_KEY_LAYOUT_DATA,
					 id,
					 &error);

	if (error) {
		g_warning ("Can't set GConf key %s. \n", error->message);
		g_error_free (error);
		error = NULL;
	}
}

static void
brasero_layout_displayed_item_changed_cb (GConfClient *client,
					  guint cxn,
					  GConfEntry *entry,
					  gpointer data)
{
	BraseroLayout *layout;
	GConfValue *value;
	const char *id;

	layout = BRASERO_LAYOUT (data);

	value = gconf_entry_get_value (entry);
	if (value->type != GCONF_VALUE_STRING)
		return;

	id = gconf_value_get_string (value);
	brasero_layout_pane_changed (layout, id);
}

void
brasero_layout_add_source (BraseroLayout *layout,
			   GtkWidget *source,
			   const char *id,
			   const char *name,
			   const char *menu,
			   const char *tooltip,
			   const char *icon,
			   BraseroLayoutType types)
{
	GtkWidget *pane;
	char *accelerator;
	BraseroLayoutItem *item;
	GtkRadioActionEntry entries;

	pane = _make_pane (source, icon, name, TRUE);
	gtk_notebook_append_page (GTK_NOTEBOOK (layout->priv->notebook),
				  pane,
				  NULL);

	/* add menu radio entry in display */
	accelerator = g_strdup_printf ("F%i", (layout->priv->accel ++) + 7);
	entries.name = id;
	entries.stock_id = icon;
	entries.label = menu;
	entries.accelerator = accelerator;
	entries.tooltip = tooltip;
	entries.value = 0;

	gtk_action_group_add_radio_actions (layout->priv->action_group,
					    &entries,
					    1,
					    0,
					    G_CALLBACK (brasero_layout_radio_toggled_cb),
					    layout);
	g_free (accelerator);
					    
	/* add it to the items list */
	item = g_new0 (BraseroLayoutItem, 1);
	item->id = g_strdup (id);
	item->widget = pane;
	item->types = types;

	layout->priv->items = g_slist_append (layout->priv->items, item);
}

void
brasero_layout_load (BraseroLayout *layout, BraseroLayoutType type)
{
	gboolean right_pane_visible = FALSE;
	BraseroLayoutItem *first = NULL;
	gchar *layout_id = NULL;
	GError *error = NULL;
	GSList *iter;

	/* remove GCONF notification if any */
	if (layout->priv->radio_notify)
		gconf_client_notify_remove (layout->priv->client,
					    layout->priv->radio_notify);

	if (type == BRASERO_LAYOUT_NONE) {
		gtk_widget_hide (GTK_WIDGET (layout));
		return;
	}
	else
		gtk_widget_show (GTK_WIDGET (layout));

	/* takes care of other panes */
	if (type == BRASERO_LAYOUT_AUDIO)
		layout_id = gconf_client_get_string (layout->priv->client,
						     BRASERO_KEY_LAYOUT_AUDIO,
						     &error);
	else if (type == BRASERO_LAYOUT_DATA)
		layout_id = gconf_client_get_string (layout->priv->client,
						     BRASERO_KEY_LAYOUT_DATA,
						     &error);

	if (error) {
		g_warning ("Can't access GConf key %s. This is probably harmless (first launch of brasero).\n", error->message);
		g_error_free (error);
		error = NULL;
	}

	/* add new notify for the new */
	if (type == BRASERO_LAYOUT_AUDIO)
		layout->priv->radio_notify = gconf_client_notify_add (layout->priv->client,
								      BRASERO_KEY_LAYOUT_AUDIO,
								      brasero_layout_displayed_item_changed_cb,
								      layout,
								      NULL,
								      &error);
	else if (type == BRASERO_LAYOUT_DATA)
		layout->priv->radio_notify = gconf_client_notify_add (layout->priv->client,
								      BRASERO_KEY_LAYOUT_DATA,
								      brasero_layout_displayed_item_changed_cb,
								      layout,
								      NULL,
								      &error);

	if (error) {
		g_warning ("Could not set notify for GConf key %s.\n", error->message);
		g_error_free (error);
		error = NULL;
	}

	layout->priv->type = type;
	for (iter = layout->priv->items; iter; iter = iter->next) {
		GtkAction *action;
		BraseroLayoutItem *item = NULL;

		item = iter->data;
		action = gtk_action_group_get_action (layout->priv->action_group, item->id);
		if (!(item->types & type)) {
			gtk_widget_hide (item->widget);
			gtk_action_set_visible (action, FALSE);
			continue;
		}

		if (!first)
			first = item;

		gtk_action_set_visible (action, TRUE);
		if (layout_id && !strcmp (layout_id, item->id)) {
			/* this is it! we found the pane to display */
			gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action), TRUE);
			gtk_widget_show (item->widget);
			right_pane_visible = TRUE;
		}
		else {
			gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action), FALSE);
			gtk_widget_hide (item->widget);
		}
	}

	if (!right_pane_visible) {
		GtkAction *action;

		/* There is no default pane since that must be a first 
		 * start. Therefore we activate the first pane in list */
		action = gtk_action_group_get_action (layout->priv->action_group, first->id);
		gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action), TRUE);
		gtk_widget_show_all (first->widget);
	}

	g_free (layout_id);
}

void
brasero_layout_register_menu (BraseroLayout *layout,
			      GtkUIManager *manager)
{
	GSList *iter;
	GtkAction *action;
	GSList *group = NULL;
	GString *description;
	GError *error = NULL;
	BraseroLayoutItem *item;

	gtk_ui_manager_insert_action_group (manager,
					    layout->priv->action_group,
					    0);

	/* build the description of the menu */
	description = g_string_new ("<ui>"
				    "<menubar name='menubar' >"
				    "<menu action='EditMenu'>"
				    "</menu>"
				    "<menu action='ViewMenu'>"
				    "<placeholder name='ViewPlaceholder'/>");
		
	for (iter = layout->priv->items; iter; iter = iter->next) {
		item = iter->data;
		g_string_append_printf (description,
					"<menuitem action='%s'/>",
					item->id);

		/* we also set all the radio buttons to belong to the same group */
		action = gtk_action_group_get_action (layout->priv->action_group, item->id);
		gtk_radio_action_set_group (GTK_RADIO_ACTION (action), group);
		group = gtk_radio_action_get_group (GTK_RADIO_ACTION (action));
	}

	g_string_append_printf (description, "<separator/>");
	g_string_append_printf (description,
				"<menuitem action='%s'/>",
				BRASERO_LAYOUT_PREVIEW_ID);

	g_string_append (description,
			 "<separator/>"
			 "</menu>"
			 "</menubar>"
			 "</ui>");

	if (!gtk_ui_manager_add_ui_from_string (manager, description->str, -1, &error)) {
		g_message ("building menus failed: %s", error->message);
		g_error_free (error);
	}

	g_string_free (description, TRUE);
}
