
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nickycopeland@hotmail.com> 1996,2008
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * This file was taken out of the engine and moved to the audio library where
 * it correctly lies however it incorporates a few flags from bristol.h header
 * file that should really be removed to keep them independent. That is for
 * later study.
 */

/*#define DEBUG */

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <string.h>

#ifdef _BRISTOL_JACK
#include "bristol.h"

#include "bristoljack.h"

extern int dupfd;

static jackDev jackdev;
static float *outbuf, *inbuf;

static int fsize = sizeof(jack_default_audio_sample_t);

static void
jack_shutdown(void *jackdev)
{
	((jackDev *) jackdev)->audiomain->atReq = BRISTOL_REQSTOP;
}

static int
audioShim(jack_nframes_t nframes, void *jd)
{
	jackDev *jackdev = (jackDev *) jd;
	register float *outL, *outR, *toutbuf = outbuf, gain;
	register int i = nframes;

	/*
	 * We may need to consider jack changing its nframes on the fly. Whilst
	 * decreasing frames is not an issue, increasing them could be painful.
	 *
	 * I think I would prefer to reap all synths rather than code such an
	 * operation, or alternatively add in a shim to do some reframing.
	 * #warning need to consider jackd changes to nframes?
	 */
	if (jackdev->jack_in[0])
		jackdev->inbuf[0] = jack_port_get_buffer(jackdev->jack_in[0], nframes);

	/*
	 * For now we are not going to be too picky about inbuf.
	 *
	 * The buffers are however mono, which is a bit of a bummer as now I have
	 * to do the interleaving and deinterleaving. This could be changed but
	 * it would have to apply to the native audio drivers as well.
	 *
	 * This doAudioOps is the same one used for the last few years by bristol
	 * as its own dispatch routine. However, there are additional features
	 * possible with Jack such that each synth could requisition its own 
	 * ports dynamically, one or two depending on the synth stereo status.
	 *
	 * That should be an option, probably, but either way, if it were done
	 * then we would need to rework this dispatcher. That is not a bad thing
	 * since we could have the audioEngine write directly to the jack buffer
	 * rather than use the stereo interleaved outbuf.
	 * 
	 * The reworked dispatcher should be placed in here, it is currently in
	 * audioEngine.c
	 */
	doAudioOps(jackdev->audiomain, outbuf, inbuf);

	/*
	 * The bristol audio does this with zero suppression. Hm. Saved data is
	 * the output from the algo, it does not have outgain applied.
	 */
	if (dupfd > 0)
		write(dupfd, outbuf, jackdev->audiomain->segmentsize * 2);

	/*
	 * We have an issue with sample formats. These are normalised to 0/1 
	 * whilst we have been working on full range values. No problem.
	 */
	if ((gain = jackdev->audiomain->outgain) < 1)
		gain = 1.0f;
	gain /= 32768.0;

	outL = (float *) jack_port_get_buffer(jackdev->jack_out[0], nframes);
	outR = (float *) jack_port_get_buffer(jackdev->jack_out[1], nframes);

	/*
	 * Deinterleave our output through to the jack buffers.
	 */
	for (; i > 0; i--)
	{
		*outL++ = *toutbuf++ * gain;
		*outR++ = *toutbuf++ * gain;
	}

	return(0);
}

static void
bristolIntJackClose()
{
	printf("closedown on interrupt\n");

	jack_deactivate(jackdev.handle);

	jack_port_unregister(jackdev.handle, jackdev.jack_out[0]);
	jack_port_unregister(jackdev.handle, jackdev.jack_out[1]);
	jack_port_unregister(jackdev.handle, jackdev.jack_in[0]);

	jack_client_close(jackdev.handle);

	_exit(0);
}

static int
bristolJackClose(jackDev *jackdev)
{
	printf("unregistering jack interface\n");

	jack_deactivate(jackdev->handle);

	jack_port_unregister(jackdev->handle, jackdev->jack_out[0]);
	jack_port_unregister(jackdev->handle, jackdev->jack_out[1]);
	jack_port_unregister(jackdev->handle, jackdev->jack_in[0]);

	jack_client_close(jackdev->handle);

	jackdev->audiomain->flags |= BRISTOL_REQSTOP;

	_exit(-1);
}

/*
 * This builds us our port list for connections. Per default we will connect
 * to the first capture port and the first two playback ports, but that 
 * should also be optional. We should have some fixed starting ports though,
 * even if we are going to pass parameters, since when each synth starts 
 * registering its own set of ports then they have to be linked somewhere 
 * by default.
 */
static void
jackPortStats(jackDev *jackdev, int dir)
{
	int i;

	free(jackdev->ports);

	if ((jackdev->ports = jack_get_ports(jackdev->handle, NULL, NULL,
		JackPortIsPhysical|dir)) == NULL)
	{
		printf("Empty jack_get_ports()\n");
		return;
	}

	for (i = 0; jackdev->ports[i] != NULL; i++)
		printf("Found port %s\n", jackdev->ports[i]);
}

static char defreg[8] = "bristol";
static char *regname = defreg;

static int
bristolJackOpen(jackDev *jackdev, audioMain *audiomain,
JackProcessCallback shim)
{
	int waitc = 10, sr;

	if (audiomain->audiodev != NULL)
		regname = audiomain->audiodev;

	printf("registering jack interface: %s\n", regname);

	if ((jackdev->handle = jack_client_new(regname)) == 0)
	{
		char process_indexed[32];

		/*
		 * This had to be changed as the process ID is variable between 
		 * invocations. The port number is not, it has to be provided to build
		 * a connection and is unique.
		 */
		snprintf(process_indexed, 32, "%s_%i", regname, audiomain->port);

		if ((jackdev->handle = jack_client_new(process_indexed)) == 0)
		{
			printf("Cannot connect to jack\n");
			audiomain->atStatus = BRISTOL_REQSTOP;
			return(-1);
		}
	}

	signal(SIGINT, bristolIntJackClose);
	signal(SIGTERM, bristolIntJackClose);
	signal(SIGHUP, bristolIntJackClose);
	signal(SIGQUIT, bristolIntJackClose);
	/*
	 * This is (was) perhaps not wise....
	 * Thanks to Hagen.
	signal(SIGSEGV, bristolIntJackClose);
	 */

	/* Samplerate mismatches should be reported however they are not critical */
	if (audiomain->samplerate != (sr = jack_get_sample_rate(jackdev->handle)))
		printf("\nJack samplerate mismatch: %i (startBristol -jack -rate %i)\n",
			audiomain->samplerate, sr);

	/*
	 * This value can change, and we should register a callback for such an
	 * event. The values are used internally and are already configured for
	 * some parts of the system, hence, for now, if this does not match we
	 * are going to exit as this is critical.
	 */
	if (audiomain->samplecount != (sr = jack_get_buffer_size(jackdev->handle)))
	{
		printf("\nJack period count mismatch: try `startBristol -jack -count %i`)\n",
			sr);
		printf("\nYou need to ensure that bristol uses the same period size\n");
		printf("Bristol is exiting ungracefully, ^C or ^Z will help\n");

		/*
		 * This did call bristolJackClose(jackdev), however as we have not
		 * yet done much of the registration then the following is correct:
		 */
		jack_client_close(jackdev->handle);

		audiomain->atStatus = BRISTOL_REQSTOP;
		return(-1);
	}

	audiomain->samplecount = sr;
	audiomain->segmentsize = audiomain->samplecount * sizeof(float);

	/*
	 * We double this value since I am going to be using interleaved samples
	 * for some buffering. I should change this, just let the audio library
	 * do interleave if it is writing to physical devices.
	 */
	outbuf = (float *) bristolmalloc(audiomain->samplecount * fsize * 2);
	inbuf = (float *) bristolmalloc(audiomain->samplecount * fsize * 2);

	audiomain->atStatus = BRISTOL_OK;
	audiomain->flags |= BRISTOL_AUDIOWAIT;

	initAudioThread(audiomain);

	jackdev->audiomain = audiomain;

	jack_set_process_callback(jackdev->handle, shim, (void *) jackdev);

	jack_on_shutdown(jackdev->handle, jack_shutdown, (void *) jackdev);

	jackdev->jack_out[0] = jack_port_register(jackdev->handle,
		"out_left", JACK_DEFAULT_AUDIO_TYPE, JackPortIsOutput, 0);
	jackdev->jack_out[1] = jack_port_register(jackdev->handle,
		"out_right", JACK_DEFAULT_AUDIO_TYPE, JackPortIsOutput, 0);

	/*
	 * We only use the input for a couple of the synths (Mini/Explorer) and
	 * even that had limited use. It was also employed historically for sync
	 * - read/modify/write sequences. We could drop this registration, but it
	 * can stay in for now, it will be used in the future.
	 */
	jackdev->jack_in[0] = jack_port_register(jackdev->handle,
		"in", JACK_DEFAULT_AUDIO_TYPE, JackPortIsInput, 0);

	/*
	 * Our engine is going have to wait for the GUI to inform us what synth
	 * we are going to be running. It is wiser not to request callbacks until
	 * at least the first synth is active. If the priorities are correct this
	 * should not be an issue, but 'dropping' a synth costs a few cycles in 
	 * the midi thread.
	 */
	while (audiomain->flags & BRISTOL_AUDIOWAIT)
	{
		sleep(1);
		if (--waitc <= 0)
		{
			printf("Did not receive request from GUI, exiting.\n");
			/*
			 * We should really unregister here.....
			 */
			return(-1);
		}
	}

	if (jack_activate(jackdev->handle) != 0)
	{
		printf("Cannot activate jack\n");
		audiomain->atStatus = BRISTOL_REQSTOP;
		return(-1);
	}

	jackPortStats(jackdev, JackPortIsInput);

	if ((audiomain->flags & BRISTOL_AUTO_CONN) && (jackdev->ports != NULL))
	{
		int jport = 0;

		for (jport = 0; jackdev->ports[jport] != NULL; jport++)
		{
			/*
			 * I don't like this code but I cannot find a way to ask jack if
			 * this port is audio or midi. Without this scan then it is possible
			 * that the engine will default a connection from audio to midi.
			 *
			 * It is okay here to attempt the connection but continue through
			 * the list in the event of failure but that is sloppy. All we
			 * actually want is to find the first two available ports to return
			 * audio to and one to receive from the daemon.
			 */
			if (strstr(jackdev->ports[jport], "midi") != 0)
			{
				printf("Skipping Jack Conn: %s\n", jackdev->ports[jport]);
				continue;
			}

			if (jack_connect(jackdev->handle,
				jack_port_name(jackdev->jack_out[0]),
					jackdev->ports[jport]) != 0)
			{
				printf("Bristol Defaulted Conn: %s to %s failed\n",
					jack_port_name(jackdev->jack_out[0]),
					jackdev->ports[jport]);
				continue;
			} else
				printf("Bristol Defaulted Conn: %s to %s\n",
					jack_port_name(jackdev->jack_out[0]),
					jackdev->ports[jport]);

			for (jport++; jackdev->ports[jport] != 0; jport++)
			{
				if (strstr(jackdev->ports[jport], "midi") != 0)
				{
					printf("Skipping Jack Conn: %s\n", jackdev->ports[jport]);
					continue;
				}

				if (jack_connect(jackdev->handle,
					jack_port_name(jackdev->jack_out[1]),
						jackdev->ports[jport]) != 0)
				{
					printf("Bristol Defaulted Conn: %s to %s failed\n",
						jack_port_name(jackdev->jack_out[1]),
						jackdev->ports[jport]);
					continue;
				} else
					printf("Bristol Defaulted Conn: %s to %s\n",
						jack_port_name(jackdev->jack_out[1]),
						jackdev->ports[jport]);
				break;
			}

			break;
		}
	}

	/*
	 * We now need to connect the ports together. For now this will be fixed
	 * but actually needs to be a set of options.
	 */
	jackPortStats(jackdev, JackPortIsOutput);

	if ((audiomain->flags & BRISTOL_AUTO_CONN) && (jackdev->ports != NULL))
	{
		int jport = 0;

		for (jport = 0; jackdev->ports[jport] != 0; jport++)
		{
			if (strstr(jackdev->ports[jport], "midi") != 0)
			{
				printf("Skipping Jack Conn: %s\n", jackdev->ports[jport]);
				continue;
			}

			if (jack_connect(jackdev->handle, jackdev->ports[jport],
				jack_port_name(jackdev->jack_in[0])) != 0)
			{
				/*
				 * We don't actually need to fail any of these, it just means
				 * we have no defaults.
				 */
				printf("Bristol Defaulted Conn: %s to %s failed\n",
					jackdev->ports[jport], jack_port_name(jackdev->jack_in[0]));
				continue;
			} else
				printf("Bristol Defaulted Conn: %s to %s\n",
					jackdev->ports[jport], jack_port_name(jackdev->jack_in[0]));

			break;
		}
	}

	return(0);
}

/*
 * This was just to test dynamic registrations whilst the jack link is active
 */
static void
jackPortBounce(jackDev *jackdev)
{
	/*
	 * Test dynamic disconnect and reconnects. Eventually each synth will
	 * potentially register its own set of ports. Want to do this with a
	 * reasonably large number of iterations. Correctly speaking we should
	 * probably take a copy of the jack_in port, null the jackdev entry
	 * and then unregister it.
	 */
	jack_disconnect(jackdev->handle,
		jackdev->ports[0], jack_port_name(jackdev->jack_in[0]));

	if (jack_port_unregister(jackdev->handle, jackdev->jack_in[0]) == 0)
	{
		printf("Unregistered port\n");
		jackdev->jack_in[0] = 0;

		sleep(5);

		if ((jackdev->jack_in[0] = jack_port_register(jackdev->handle,
			"in", JACK_DEFAULT_AUDIO_TYPE, JackPortIsInput, 0)) != NULL)
			printf("Registered port\n");
	} else
		sleep(5);

	jack_connect(jackdev->handle,
		jackdev->ports[0], jack_port_name(jackdev->jack_in[0]));
}

/*
 * This should go out as the first release with Jack. After that the interface
 * will change - at the moment Jack subsumes Bristol, and this is the wrong
 * way around. The audiomain structure is buried inside the jack structure,
 * but I would prefer the contrary. In addition, with it the contrary then
 * it would be easier to integrate alternative distribution drivers (DSSI).
 */
int
bristolJackInterface(audioMain *audiomain)
{
	if (bristolJackOpen(&jackdev, audiomain, audioShim) != 0)
		return(-1);

	while (audiomain->atReq != BRISTOL_REQSTOP)
		sleep(1);

	bristolJackClose(&jackdev);

	return(0);
}
#endif /* _BRISTOL_JACK */

