
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nickycopeland@hotmail.com> 1996,2009
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>

#include <pthread.h>

#include "brighton.h"
#include "brightonMini.h"
#include "bristolmidi.h"
#include "brightonhelp.h"

#include "brightonKeyboards.h"

extern brightonApp miniApp;
extern brightonApp prophetApp;
extern brightonApp hammondApp;
extern brightonApp junoApp;
extern brightonApp dxApp;
extern brightonApp explorerApp;
extern brightonApp mixApp;
extern brightonApp hammondB3App;
extern brightonApp voxApp;
extern brightonApp rhodesApp;
extern brightonApp rhodesBassApp;
extern brightonApp pro10App;
extern brightonApp prophet52App;
extern brightonApp obxApp;
extern brightonApp obxaApp;
extern brightonApp polyApp;
extern brightonApp poly6App;
extern brightonApp axxeApp;
extern brightonApp odysseyApp;
extern brightonApp memMoogApp;
extern brightonApp arp2600App;
extern brightonApp sAksApp;
extern brightonApp ms20App;
extern brightonApp solinaApp;
extern brightonApp roadrunnerApp;
extern brightonApp granularApp;
extern brightonApp realisticApp;
extern brightonApp voxM2App;
extern brightonApp jupiterApp;
extern brightonApp bitoneApp;
extern brightonApp bit99App;
extern brightonApp bit100App;
extern brightonApp masterApp;
extern brightonApp cs80App;
extern brightonApp pro1App;
extern brightonApp voyagerApp;
extern brightonApp sonic6App;
extern brightonApp trilogyApp;
extern brightonApp stratusApp;
extern brightonApp poly800App;
extern brightonApp bme700App;
extern brightonApp bmApp;
extern brightonApp sidApp;
extern brightonApp sid2App;

#define BRIGHTON_BIT99APP (BRISTOL_SYNTHCOUNT + 1)

char *bristolhome = NULL;

guimain global;

brightonApp *synthesisers[BRISTOL_SYNTHCOUNT];

void *eventMgr();

int opacity = 60;
int gs = 1;
int nar = 0;
int quality = 6;
float scale = 1.0;
float antialias = 0.0;
int aliastype = 0;
int library = 1;
int dcTimeout = 500000;
int sampleRate = 44100;
int sampleCount = 256;
int mwt = 50000;
int activeSense = 2000;
int activeSensePeriod = 15000;

static char *BRISTOL_DEF_HOST = "localhost";

extern int brightonMidiInput(bristolMidiMsg *);

/*
 * Some of the emulations really benefit from a bit of detune, this is the
 * value they will use unless another has been requested.
 */
#define NON_DEF_DETUNE 100

/*
 * Need to make this multithreaded?
 */
int
main(int argc, char **argv)
{
	int argCount = 1, midiHandle = -1, cFD = -1, midiFD = -1, i, j, asc = 0;
	pthread_t thread;
	bristolMidiMsg msg;

	signal(SIGINT, cleanupBristol);
	signal(SIGPIPE, cleanupBristolQuietly);
	signal(SIGHUP, cleanupBristol);

	global.home = getenv("BRISTOL");

	global.synths = (guiSynth *) brightonmalloc(sizeof(guiSynth));

	global.voices = BRISTOL_VOICECOUNT;
	global.synths->voices = BRISTOL_VOICECOUNT;
	global.synths->pwd = 2;
	global.synths->gain = 32;
	global.synths->synthtype = BRISTOL_HAMMONDB3;
	global.synths->velocity = 510;  /* linear tracking */
	global.synths->glide = 5;  /* maximum glide delay - up to 30s */
	global.synths->detune = 40;
	global.synths->lwf = 0;
	global.synths->mbi = 0;
	global.synths->notepref = 0;
	global.synths->notetrig = 0;
	global.port = -1;
	global.host = BRISTOL_DEF_HOST;

	synthesisers[BRISTOL_MINI] = &miniApp;
	synthesisers[BRISTOL_PROPHET] = &prophetApp;
	synthesisers[BRISTOL_PRO52] = &prophet52App;
	synthesisers[BRISTOL_HAMMOND] = &hammondB3App;
	synthesisers[BRISTOL_JUNO] = &junoApp;
	synthesisers[BRISTOL_DX] = &dxApp;
	synthesisers[BRISTOL_EXPLORER] = &explorerApp;
	synthesisers[BRISTOL_HAMMONDB3] = &hammondB3App;
	synthesisers[BRISTOL_VOX] = &voxApp;
	synthesisers[BRISTOL_RHODES] = &rhodesApp;
	synthesisers[BRISTOL_RHODES_BASS] = &rhodesBassApp;
	synthesisers[BRISTOL_PROPHET10] = &pro10App;
	synthesisers[BRISTOL_MIXER] = &mixApp;
	synthesisers[BRISTOL_OBX] = &obxApp;
	synthesisers[BRISTOL_OBXA] = &obxaApp;
	synthesisers[BRISTOL_POLY] = &polyApp;
	synthesisers[BRISTOL_POLY6] = &poly6App;
	synthesisers[BRISTOL_AXXE] = &axxeApp;
	synthesisers[BRISTOL_ODYSSEY] = &odysseyApp;
	synthesisers[BRISTOL_MEMMOOG] = &memMoogApp;
	synthesisers[BRISTOL_2600] = &arp2600App;
	synthesisers[BRISTOL_SAKS] = &sAksApp;
	synthesisers[BRISTOL_MS20] = &ms20App;
	synthesisers[BRISTOL_SOLINA] = &solinaApp;
	synthesisers[BRISTOL_ROADRUNNER] = &roadrunnerApp;
	synthesisers[BRISTOL_GRANULAR] = &granularApp;
	synthesisers[BRISTOL_REALISTIC] = &realisticApp;
	synthesisers[BRISTOL_VOXM2] = &voxM2App;
	synthesisers[BRISTOL_JUPITER8] = &jupiterApp;
	synthesisers[BRISTOL_BIT_ONE] = &bitoneApp;
	synthesisers[BRISTOL_MASTER] = &masterApp;
	synthesisers[BRISTOL_CS80] = &cs80App;
	synthesisers[BRISTOL_PRO1] = &pro1App;
	synthesisers[BRISTOL_VOYAGER] = &voyagerApp;
	synthesisers[BRISTOL_SONIC6] = &sonic6App;
	synthesisers[BRISTOL_TRILOGY] = &trilogyApp;
	synthesisers[BRISTOL_STRATUS] = &stratusApp;
	synthesisers[BRISTOL_POLY800] = &poly800App;
	synthesisers[BRISTOL_BME700] = &bme700App;
	synthesisers[BRISTOL_BASSMAKER] = &bmApp;
	synthesisers[BRISTOL_SID_M1] = &sidApp;
	synthesisers[BRISTOL_SID_M2] = &sid2App;

	global.synths->location = 0;

	if ((argc > 1) && (strcmp(argv[1], "-summary") == 0))
	{
		printf("%s", summarytext);
		exit(0);
	}


	if (argc < 2) {
		printf("You probably prefer to use the startBristol script.\n");
		exit(1);
	}

	printf("bristol version %s\n", VERSION);

	/*
	 * close our standard input, create a pipe which will never be used.
	 */
	while (argc > argCount)
	{
		if ((strcmp(argv[argCount], "--help") == 0)
			|| (strcmp(argv[argCount], "-help") == 0)
			|| (strcmp(argv[argCount], "-h") == 0)
			|| (strcmp(argv[argCount], "--h") == 0))
		{
#ifdef BRISTOL_BUILD
			if (BRISTOL_BUILD == 0)
				printf("bristol %i.%i.%i: ",
					BRISTOL_MAJOR, BRISTOL_MINOR, BRISTOL_PATCH);
			else
				printf("bristol %i.%i.%i-%i: ",
					BRISTOL_MAJOR, BRISTOL_MINOR, BRISTOL_PATCH, BRISTOL_BUILD);
#else
				printf("bristol %s\n", VERSION);
#endif
			printf("%s", helptext);
			exit(0);
		}

		if ((strcmp(argv[argCount], "-V") == 0)
			&& (strlen(argv[argCount]) == 2))
		{
#ifdef BRISTOL_BUILD
			if (BRISTOL_BUILD == 0)
				printf("bristol version %i.%i.%i\n",
					BRISTOL_MAJOR, BRISTOL_MINOR, BRISTOL_PATCH);
			else
				printf("bristol version %i.%i.%i-%i\n",
					BRISTOL_MAJOR, BRISTOL_MINOR, BRISTOL_PATCH, BRISTOL_BUILD);
#endif
			exit(0);
		}

		if (strcmp(argv[argCount], "-libtest") == 0)
			global.libtest = 1 - global.libtest;

		if (strcmp(argv[argCount], "-pixmap") == 0)
			library = 0;

		if (((strcmp(argv[argCount], "-emulate") == 0)
			||(strcmp(argv[argCount], "-synth") == 0))
			&& (argCount < argc))
		{
			int i;

			for (i = 0 ; i < BRISTOL_SYNTHCOUNT; i++)
			{
				if (synthesisers[i] == NULL)
					continue;

				if (strcmp(synthesisers[i]->name, argv[argCount + 1]) == 0)
				{
					printf("found %s at %i\n", synthesisers[i]->name, i);
					global.synths->synthtype = i;
					break;
				}
			}

			if (i == BRISTOL_SYNTHCOUNT) {
				printf("Could not find synth named \"%s\"\n",
					argv[argCount + 1]);
				exit(-1);
			} else {
				/*
				 * We found the emulation, now we should put in the default
				 * from its application definition.
				 */
				printf("emulation defaults:\n");
				printf("	-voices  %i\n", synthesisers[i]->emulate.voices);
				printf("	-detune: %i\n", synthesisers[i]->emulate.detune);
				printf("	-gain:   %i\n",	synthesisers[i]->emulate.gain);
				printf("	-pwd:    %i\n",	synthesisers[i]->emulate.pwd);
				printf("	-glide:  %i\n",	synthesisers[i]->emulate.glide);
				printf("	-curve:  %i\n",	synthesisers[i]->emulate.velocity);

				global.synths->voices = synthesisers[i]->emulate.voices;
				global.synths->detune = synthesisers[i]->emulate.detune;
				global.synths->gain = synthesisers[i]->emulate.gain * 128;
				global.synths->pwd = synthesisers[i]->emulate.pwd;
				global.synths->glide = synthesisers[i]->emulate.glide;
				global.synths->velocity = synthesisers[i]->emulate.velocity;
				if (synthesisers[i]->emulate.opacity != 0)
					printf("    -opacity:%i\n",
						opacity = synthesisers[i]->emulate.opacity);
			}
			argCount++;
			continue;
		}

		if ((strcmp(argv[argCount], "-quality") == 0) && (argCount < argc))
		{
			if ((quality = atoi(argv[argCount + 1])) < 2)
				quality = 2;
			else if (quality > 8)
				quality = 8;
			argCount++;
			continue;
		}

		if ((strcmp(argv[argCount], "-aliastype") == 0) && (argCount < argc))
		{
			/*
			 * I want this to take a text value and if not recognised then
			 * an integer. Text will be 'texture' or 'all'.
			 */
			if (strcmp(argv[argCount + 1], "texture") == 0) {
				aliastype = 1;
				argCount++;
				if (antialias == 0.0)
					antialias = 0.3;
				continue;
			} else if (strcmp(argv[argCount + 1], "pre") == 0) {
				aliastype = 5;
				argCount++;
				if (antialias == 0.0)
					antialias = 0.3;
				continue;
			} else if (strcmp(argv[argCount + 1], "all") == 0) {
				aliastype = 3;
				argCount++;
				if (antialias == 0.0)
					antialias = 0.3;
				continue;
			} else if ((aliastype = atoi(argv[argCount + 1])) < 0) {
				aliastype = 0;
				continue;
			} else if (aliastype > 4) {
				aliastype = 0;
				continue;
			}
			argCount++;
			continue;
		}

		if ((strcmp(argv[argCount], "-antialias") == 0) && (argCount < argc))
		{
			if ((antialias = (float) atoi(argv[argCount + 1])) < 0)
				antialias = 0.0;
			else if (antialias > 99)
				antialias = 0.01;
			else
				antialias = 1.0 - antialias / 100.0;

			if (aliastype == 0)
				aliastype = 1;

			argCount++;
			continue;
		}

		if (strcmp(argv[argCount], "-rate") == 0)
		{
			if ((sampleRate = atoi(argv[argCount + 1])) <= 0)
				sampleRate = 44100;

			argCount++;
			continue;
		}

		if (strcmp(argv[argCount], "-count") == 0)
		{
			if ((sampleCount = atoi(argv[argCount + 1])) <= 0)
				sampleCount = 256;

			argCount++;

			continue;
		}

		if (((strcmp(argv[argCount], "-activesense") == 0)
			|| (strcmp(argv[argCount], "-as") == 0))
			&& (argCount < argc))
		{
			activeSense = atoi(argv[argCount + 1]);

			if ((activeSense != 0) && (activeSense < 50))
				activeSense = 50;

			/*
			 * This is now set to the default timer to pass to the engine
			 * It is defaulted to three times the update rate. We should do
			 * a quick check to ensure this is larger than the sample period
			 * count by a little bit but we can only do that later.
			 */
			if (activeSense > 5000)
				activeSense = 5000;

			if ((activeSensePeriod = activeSense * 3) > 16000)
				activeSensePeriod = 16000;

printf("as %i, asp %i\n", activeSense, activeSensePeriod);
			argCount++;
		}

		if ((strcmp(argv[argCount], "-ast") == 0)
			&& (argCount < argc))
		{
			if ((activeSensePeriod = atoi(argv[argCount + 1])) < 50)
				activeSensePeriod = 50;

			if (activeSensePeriod < activeSense * 2)
				activeSensePeriod = activeSense * 3;

			if (activeSensePeriod > 16000)
				activeSensePeriod = 16000;

			/*
			 * This is now set to the default timer to pass to the engine
			 * It is defaulted to three times the update rate.
			activeSensePeriod = activeSensePeriod * sampleRate / 1000;
			 */

printf("as %i, asp %i\n", activeSense, activeSensePeriod);
			argCount++;
		}

		if ((strcmp(argv[argCount], "-mct") == 0)
			&& (argCount < argc))
		{
			if ((mwt = atoi(argv[argCount + 1]) * 1000) < 0)
				mwt = 50000;
			argCount++;
		}

		if ((strcmp(argv[argCount], "-dct") == 0)
			&& (argCount < argc))
		{
			if ((dcTimeout = atoi(argv[argCount + 1]) * 1000) < 10000)
				dcTimeout = 250000;

			argCount++;
		}

		if ((strcmp(argv[argCount], "-opacity") == 0) && (argCount < argc))
		{
			if ((opacity = atoi(argv[argCount + 1])) < 20)
				opacity = 20;
			else if (opacity > 100)
				opacity = 100;
			argCount++;
			continue;
		}

		if ((strcmp(argv[argCount], "-scale") == 0) && (argCount < argc))
		{
			if ((strcmp(argv[argCount + 1], "fs") == 0)
				|| (strcmp(argv[argCount + 1], "fullscreen") == 0))
			{
				aliastype = 5;
				scale = 100.0;
				argCount++;
				continue;
			}
			if ((scale = atof(argv[argCount + 1])) > 0)
			{
				if ((scale > 1.1) || (scale < 0.9))
					aliastype = 5;
				else
					aliastype = 0;
				argCount++;
				continue;
			}
			scale = 1.0;
		}

		if ((strcmp(argv[argCount], "-ar") == 0) ||
			(strcmp(argv[argCount], "-nar") == 0) ||
			(strcmp(argv[argCount], "-aspect") == 0))
			nar = 1;

		if (((strcmp(argv[argCount], "-gs") == 0) ||
			(strcmp(argv[argCount], "-grayscale") == 0) ||
			(strcmp(argv[argCount], "-greyscale") == 0)) && (argCount < argc))
		{
			gs = atoi(argv[argCount + 1]);
			argCount++;
		}

		if ((strcmp(argv[argCount], "-detune") == 0) && (argCount < argc))
		{
			if (argCount < argc)
				global.synths->detune = atoi(argv[argCount + 1]);
			argCount++;
		}

		if ((strcmp(argv[argCount], "-gain") == 0) && (argCount < argc))
		{
			if (argCount < argc)
				global.synths->gain = atoi(argv[argCount + 1]) * 128;
			argCount++;
		}

		if ((strcmp(argv[argCount], "-pwd") == 0) && (argCount < argc))
		{
			if (argCount < argc)
				global.synths->pwd = atoi(argv[argCount + 1]);
			argCount++;
		}
/* Need to add in PitchWheel Depth (done), Find and Coarse tuning*/

		if ((strcmp(argv[argCount], "-load") == 0) && (argCount < argc))
		{
			if (argCount < argc)
				global.synths->location = atoi(argv[argCount + 1]);
			if (global.synths->location > 999)
			{
				global.synths->mbi = (global.synths->location / 1000) * 1000;
				global.synths->location = global.synths->location % 1000;
			}

			argCount++;
		}

		/* Master Bank Index */
		if ((strcmp(argv[argCount], "-mbi") == 0) && (argCount < argc))
		{
			int mbi;

			if (argCount < argc)
				mbi = atoi(argv[argCount + 1]) * 1000;

			if ((global.synths->mbi != 0) && (global.synths->mbi != mbi))
				printf("Master bank index and loaded memory conflict (%i/%i)\n",
					mbi/1000, global.synths->mbi/1000);

			global.synths->mbi = mbi;

			argCount++;
		}

		if ((strcmp(argv[argCount], "-glide") == 0) && (argCount < argc))
		{
			if (argCount < argc)
				global.synths->glide = atoi(argv[argCount + 1]);
			if (global.synths->glide <= 0)
				global.synths->glide = 30;
			if (global.synths->glide > 30)
				global.synths->glide = 30;
			argCount++;
		}

		if ((strcmp(argv[argCount], "-voices") == 0) && (argCount < argc))
		{
			if (argCount < argc)
				global.synths->voices = atoi(argv[argCount + 1]);
			argCount++;
		}

		if (strcmp(argv[argCount], "-mono") == 0)
			global.synths->voices = 1;

		if (strcmp(argv[argCount], "-mididbg2") == 0)
			global.synths->flags |= REQ_MIDI_DEBUG2;
		if ((strcmp(argv[argCount], "-mididbg") == 0)
			|| (strcmp(argv[argCount], "-mididbg1") == 0))
			global.synths->flags |= REQ_MIDI_DEBUG;

		if ((strcmp(argv[argCount], "-midinrp") == 0)
			|| (strcmp(argv[argCount], "-nrp") == 0))
			global.synths->flags |= MIDI_NRP;

		if ((strcmp(argv[argCount], "-channel") == 0) && (argCount < argc))
		{
			if (argCount < argc)
				global.synths->midichannel = atoi(argv[argCount + 1]) - 1;
			argCount++;
		}

		if (((strcmp(argv[argCount], "-velocity") == 0)
			|| (strcmp(argv[argCount], "-curve") == 0)
			|| (strcmp(argv[argCount], "-mvc") == 0))
			&& (argCount < argc))
		{
			if (argCount < argc)
				global.synths->velocity = atoi(argv[argCount + 1]);
			argCount++;
		}

		if (strcmp(argv[argCount], "-tracking") == 0)
		{
			global.synths->flags |= NO_KEYTRACK;
			argCount++;
		}

		if (strcmp(argv[argCount], "-engine") == 0)
			global.flags |= BRIGHTON_NOENGINE;

		if (strcmp(argv[argCount], "-port") == 0)
		{
			if (argCount < argc)
				global.port = atoi(argv[++argCount]);
		}

		if (strcmp(argv[argCount], "-host") == 0)
			global.host = argv[++argCount];

		if (strcmp(argv[argCount], "-lwf") == 0)
			global.synths->lwf = 1;

		if (strcmp(argv[argCount], "-lnp") == 0)
			global.synths->notepref = 1;
		if (strcmp(argv[argCount], "-hnp") == 0)
			global.synths->notepref = 2;
		if (strcmp(argv[argCount], "-nnp") == 0)
			global.synths->notepref = 0;
		if (strcmp(argv[argCount], "-retrig") == 0)
			global.synths->notetrig = 1;
		if (strcmp(argv[argCount], "-lvel") == 0)
			global.synths->legatovelocity = 1;

		/*
		 * And finally all the different synths
		 */
		if ((strcmp(argv[argCount], "-mini") == 0)
			||(strcmp(argv[argCount], "-minimoog") == 0))
			global.synths->synthtype = BRISTOL_MINI;

		if (strcmp(argv[argCount], "-hammond") == 0)
			global.synths->synthtype = BRISTOL_HAMMONDB3;

		if (strcmp(argv[argCount], "-prophet10") == 0)
			global.synths->synthtype = BRISTOL_PROPHET10;

		if ((strcmp(argv[argCount], "-pro1") == 0)
			|| (strcmp(argv[argCount], "-proone") == 0)
			|| (strcmp(argv[argCount], "-prophet1") == 0))
		{
			global.synths->synthtype = BRISTOL_PRO1;
			opacity = 60;
		}

		if ((strcmp(argv[argCount], "-pro52") == 0)
			|| (strcmp(argv[argCount], "-prophet52") == 0))
			global.synths->synthtype = BRISTOL_PRO52;

		if ((strcmp(argv[argCount], "-prophet") == 0)
			|| (strcmp(argv[argCount], "-sequential") == 0)
			|| (strcmp(argv[argCount], "-sequentialcircuits") == 0)
			|| (strcmp(argv[argCount], "-prophet5") == 0))
			global.synths->synthtype = BRISTOL_PROPHET;

		if (strcmp(argv[argCount], "-prophet10") == 0)
			global.synths->synthtype = BRISTOL_PROPHET10;

		if (strcmp(argv[argCount], "-pro5") == 0)
			global.synths->synthtype = BRISTOL_PROPHET;

		if (strcmp(argv[argCount], "-pro10") == 0)
		{
			if (global.synths->detune == 0)
				global.synths->detune = NON_DEF_DETUNE;
			global.synths->synthtype = BRISTOL_PROPHET10;
		}

		if (strcmp(argv[argCount], "-dx") == 0)
			global.synths->synthtype = BRISTOL_DX;

		if (strcmp(argv[argCount], "-juno") == 0)
			global.synths->synthtype = BRISTOL_JUNO;

		if (strcmp(argv[argCount], "-cs80") == 0)
		{
			global.synths->synthtype = BRISTOL_CS80;
			global.libtest = 1;
		}

		if ((strcmp(argv[argCount], "-jupiter8") == 0) ||
			(strcmp(argv[argCount], "-uranus") == 0) ||
			(strcmp(argv[argCount], "-jupiter") == 0))
		{
			if (global.synths->voices == BRISTOL_VOICECOUNT)
				global.synths->voices = 8; /* Two 4 voice Jupiters */
			if (global.synths->detune == 0)
				global.synths->detune = NON_DEF_DETUNE;
			global.synths->synthtype = BRISTOL_JUPITER8;
			opacity = 40;
		}

		if (strcmp(argv[argCount], "-sampler") == 0)
		{
			global.synths->synthtype = BRISTOL_SAMPLER;
			global.libtest = 1;
		}

		if (strcmp(argv[argCount], "-bristol") == 0)
			global.synths->synthtype = BRISTOL_BRISTOL;

		if (strcmp(argv[argCount], "-voyager") == 0)
			global.synths->synthtype = BRISTOL_EXPLORER;

		if (strcmp(argv[argCount], "-sonic6") == 0)
			global.synths->synthtype = BRISTOL_SONIC6;

		if (strcmp(argv[argCount], "-explorer") == 0)
			global.synths->synthtype = BRISTOL_EXPLORER;
		if (strcmp(argv[argCount], "-voyager") == 0)
			global.synths->synthtype = BRISTOL_VOYAGER;

		if (strcmp(argv[argCount], "-mixer") == 0)
		{
			global.libtest = 1;
			global.synths->synthtype = BRISTOL_MIXER;
		}

		if (strcmp(argv[argCount], "-rhodesbass") == 0)
			global.synths->synthtype = BRISTOL_RHODES_BASS;
		else if (strcmp(argv[argCount], "-rhodes") == 0)
			global.synths->synthtype = BRISTOL_RHODES;

		if (strcmp(argv[argCount], "-vox") == 0)
			global.synths->synthtype = BRISTOL_VOX;

		if ((strcmp(argv[argCount], "-voxm2") == 0)
			|| (strcmp(argv[argCount], "-vox300") == 0))
			global.synths->synthtype = BRISTOL_VOXM2;

		if (strcmp(argv[argCount], "-ddd") == 0)
			global.synths->synthtype = BRISTOL_DDD;

		if (strcmp(argv[argCount], "-b3") == 0)
			global.synths->synthtype = BRISTOL_HAMMONDB3;

		if (strcmp(argv[argCount], "-obxa") == 0)
		{
			if (global.synths->detune == 0)
				global.synths->detune = NON_DEF_DETUNE;
			global.synths->synthtype = BRISTOL_OBXA;
			if (opacity == 40)
				opacity = 60;
		} else if (strcmp(argv[argCount], "-obx") == 0) {
			if (global.synths->detune == 0)
				global.synths->detune = NON_DEF_DETUNE;
			global.synths->synthtype = BRISTOL_OBX;
		}

		if (strcmp(argv[argCount], "-moog") == 0)
			global.synths->synthtype = BRISTOL_MINI;

		if (strcmp(argv[argCount], "-arp") == 0)
			global.synths->synthtype = BRISTOL_2600;

		if (strcmp(argv[argCount], "-roland") == 0)
			global.synths->synthtype = BRISTOL_JUNO;

		if (strcmp(argv[argCount], "-oberheim") == 0)
			global.synths->synthtype = BRISTOL_OBXA;

		if ((strcmp(argv[argCount], "-polysix") == 0)
			|| (strcmp(argv[argCount], "-korg") == 0)
			|| (strcmp(argv[argCount], "-poly6") == 0)
			|| (strcmp(argv[argCount], "-poly") == 0))
			global.synths->synthtype = BRISTOL_POLY6;

		if (strcmp(argv[argCount], "-monopoly") == 0)
			global.synths->synthtype = BRISTOL_POLY;

		if (strcmp(argv[argCount], "-poly800") == 0)
		{
			global.synths->synthtype = BRISTOL_POLY800;
			global.synths->voices = 8;
		}

		if (strcmp(argv[argCount], "-ms20") == 0)
		{
			global.synths->synthtype = BRISTOL_MS20;
			global.libtest = 1;
		}

		if (strcmp(argv[argCount], "-bme700") == 0)
		{
			global.synths->synthtype = BRISTOL_BME700;
			global.synths->voices = 8;
		}

		if ((strcmp(argv[argCount], "-bassmaker") == 0)
			|| (strcmp(argv[argCount], "-bm") == 0))
		{
			global.synths->synthtype = BRISTOL_BASSMAKER;
			global.synths->voices = 1;
		}

		if ((strcmp(argv[argCount], "-sid") == 0)
			|| (strcmp(argv[argCount], "-sidney") == 0))
		{
			global.synths->synthtype = BRISTOL_SID_M1;
			global.synths->voices = 1;
			global.synths->notepref = 2;
			global.synths->notetrig = 1;
			antialias = 0.5;
			aliastype = 1;
			opacity = 50;
		}

		if ((strcmp(argv[argCount], "-sid2") == 0)
			|| (strcmp(argv[argCount], "-melbourne") == 0)
			|| (strcmp(argv[argCount], "-canberra") == 0)
			|| (strcmp(argv[argCount], "-perth") == 0)
			|| (strcmp(argv[argCount], "-resid") == 0)
			|| (strcmp(argv[argCount], "-asid") == 0)
			|| (strcmp(argv[argCount], "-acid") == 0))
		{
			global.synths->synthtype = BRISTOL_SID_M2;
			global.synths->voices = 1;
			global.synths->notepref = 2;
			global.synths->notetrig = 1;
			antialias = 0.5;
			aliastype = 1;
			opacity = 50;
		}

		if (strcmp(argv[argCount], "-arp2600") == 0)
			global.synths->synthtype = BRISTOL_2600;
		if (strcmp(argv[argCount], "-2600") == 0)
			global.synths->synthtype = BRISTOL_2600;

		if (strcmp(argv[argCount], "-axxe") == 0)
			global.synths->synthtype = BRISTOL_AXXE;

		if (strcmp(argv[argCount], "-odyssey") == 0)
			global.synths->synthtype = BRISTOL_ODYSSEY;

		if (strcmp(argv[argCount], "-solina") == 0)
			global.synths->synthtype = BRISTOL_SOLINA;

		if (strcmp(argv[argCount], "-roadrunner") == 0)
			global.synths->synthtype = BRISTOL_ROADRUNNER;

		if ((strcmp(argv[argCount], "-memory") == 0)
			|| (strcmp(argv[argCount], "-memmoog") == 0)
			|| (strcmp(argv[argCount], "-memorymoog") == 0))
			global.synths->synthtype = BRISTOL_MEMMOOG;

		if ((strcmp(argv[argCount], "-mas") == 0)
			|| (strcmp(argv[argCount], "-ems") == 0)
			|| (strcmp(argv[argCount], "-aks") == 0))
		{
			global.synths->synthtype = BRISTOL_SAKS;
			global.libtest = 1;
		}

		if ((strcmp(argv[argCount], "-granular") == 0)
			|| (strcmp(argv[argCount], "-quantum") == 0))
		{
			global.synths->synthtype = BRISTOL_GRANULAR;
			global.libtest = 1;
		}

		if ((strcmp(argv[argCount], "-realistic") == 0)
			|| (strcmp(argv[argCount], "-mg1") == 0))
			global.synths->synthtype = BRISTOL_REALISTIC;

#define BIT1_ACTIVE 200
		/*
		 * Also add some hacks for a bit-99, same algorithm but different
		 * GUI template
		 */
		if ((strcmp(argv[argCount], "-bitone") == 0)
			|| (strcmp(argv[argCount], "-crumar") == 0)
			|| (strcmp(argv[argCount], "-bit01") == 0)
			|| (strcmp(argv[argCount], "-bit1") == 0))
		{
			int i;

			if (global.synths->detune == 0)
				global.synths->detune = NON_DEF_DETUNE;
			global.synths->synthtype = BRISTOL_BIT_ONE;

			for (i = 68; i < 77; i++)
				bit99App.resources[0].devlocn[i].flags |= BRIGHTON_WITHDRAWN;

			if (opacity == 40)
				opacity = 60;

			bit99App.resources[0].devlocn[48].flags |= BRIGHTON_WITHDRAWN;
			/*
			 * Hide the stereo button
			 */
			bit99App.resources[0].devlocn[BIT1_ACTIVE + 2].flags
				|= BRIGHTON_WITHDRAWN;
			bit99App.resources[0].devlocn[BIT1_ACTIVE + 2].x = 50;
			bit99App.resources[0].devlocn[BIT1_ACTIVE + 2].y = 50;
			/* And add the unison button */
			bit99App.resources[0].devlocn[BIT1_ACTIVE + 38].flags
				&= ~BRIGHTON_WITHDRAWN;
		}

		/*
		 * The bit-99 hack, same algorithm but different GUI template
		 */
		if (strcmp(argv[argCount], "-bit99") == 0)
		{
			synthesisers[BRISTOL_BIT_ONE] = &bit99App;
			global.synths->synthtype = BRISTOL_BIT_ONE;
			if (global.synths->detune == 0)
				global.synths->detune = NON_DEF_DETUNE;

			if (opacity == 40)
				opacity = 60;
			/*
			 * Enable a few extra parameters.
			bit99App.resources[0].devlocn[77].flags &= ~BRIGHTON_WITHDRAWN;
			bit99App.resources[0].devlocn[78].flags &= ~BRIGHTON_WITHDRAWN;
			bit99App.resources[0].devlocn[79].flags &= ~BRIGHTON_WITHDRAWN;
			bit99App.resources[0].devlocn[80].flags &= ~BRIGHTON_WITHDRAWN;
			 */
			bit99App.resources[0].devlocn[12].flags &= ~BRIGHTON_WITHDRAWN;
			bit99App.resources[0].devlocn[BIT1_ACTIVE - 1].flags
				&= ~BRIGHTON_WITHDRAWN;
			/* Show the stereo button */
			bit99App.resources[0].devlocn[BIT1_ACTIVE + 2].flags
				&= ~BRIGHTON_WITHDRAWN;
			/* And remove the unison button */
			bit99App.resources[0].devlocn[BIT1_ACTIVE + 38].flags
				|= BRIGHTON_WITHDRAWN;
		}

		/*
		 * The bit-99 black hack, same algorithm but different GUI template
		 */
		if ((strcmp(argv[argCount], "-bit99m2") == 0)
			|| (strcmp(argv[argCount], "-bit100") == 0))
		{
			synthesisers[BRISTOL_BIT_ONE] = &bit100App;
			global.synths->synthtype = BRISTOL_BIT_ONE;
			if (global.synths->detune == 0)
				global.synths->detune = NON_DEF_DETUNE;

			if (opacity == 40)
				opacity = 60;
			/*
			 * Enable a few extra parameters.
			 */
			bit100App.resources[0].devlocn[88].flags &= ~BRIGHTON_WITHDRAWN;
//			bit100App.resources[0].devlocn[74].flags |= BRIGHTON_WITHDRAWN;
//			bit100App.resources[0].devlocn[75].flags |= BRIGHTON_WITHDRAWN;
//			bit100App.resources[0].devlocn[76].flags |= BRIGHTON_WITHDRAWN;
			bit100App.resources[0].devlocn[78].flags &= ~BRIGHTON_WITHDRAWN;
			bit100App.resources[0].devlocn[79].flags &= ~BRIGHTON_WITHDRAWN;
			bit100App.resources[0].devlocn[80].flags &= ~BRIGHTON_WITHDRAWN;
			bit100App.resources[0].devlocn[12].flags &= ~BRIGHTON_WITHDRAWN;
			bit100App.resources[0].devlocn[199].flags &= ~BRIGHTON_WITHDRAWN;
			/* Show the stereo button */
			bit100App.resources[0].devlocn[BIT1_ACTIVE + 2].flags
				&= ~BRIGHTON_WITHDRAWN;
			/* And remove the unison button */
			bit100App.resources[0].devlocn[BIT1_ACTIVE + 38].flags
				|= BRIGHTON_WITHDRAWN;
			bit100App.resources[0].devlocn[BIT1_ACTIVE - 5].flags
				|= BRIGHTON_CHECKBUTTON;
		}

		if (strcmp(argv[argCount], "-trilogy") == 0)
		{
			global.synths->synthtype = BRISTOL_TRILOGY;
			global.synths->detune = 150;
			opacity = 60;
		}

		if (strcmp(argv[argCount], "-stratus") == 0)
		{
			global.synths->synthtype = BRISTOL_STRATUS;
			global.synths->detune = 150;
			opacity = 60;
		}

		argCount++;
	}

	/*
	 * Hm, this is a joke really. The Tandy, Realistic, whatever is an old
	 * synth, and this mimics random slider changes. Should not really do it
	 * but its fun. The original used to lose its white caps.
	 * Once the gui is created its more work to change the image.
	 */
	if (global.synths->synthtype == BRISTOL_REALISTIC)
	{
		int cont, i;
		struct timeval now;

		gettimeofday(&now, NULL);

		srand(now.tv_usec);

		for (i = 0; i < 6; i++)
		{
			cont = rand() & 0x01f;
			if (synthesisers[BRISTOL_REALISTIC]->resources[0].devlocn[cont].device
				== 1)
			{
				synthesisers[BRISTOL_REALISTIC]->resources[0].devlocn[cont].image
					= "bitmaps/knobs/knob8.xpm";
			}
		}
	}

	if (synthesisers[global.synths->synthtype] == 0)
		exit(0);

	global.synths->resources = synthesisers[global.synths->synthtype];
	synthesisers[global.synths->synthtype]->width *= scale;
	synthesisers[global.synths->synthtype]->height *= scale;

	/*
	 * win is actually set by the configuration routines, but we use it 
	 * here anyway. The configuration options are a structure, they were a
	 * list of parameters however that became unwieldy.
	 */
	global.synths->win =
		brightonInterface(synthesisers[global.synths->synthtype],
			quality, library, aliastype, antialias, gs);

	global.synths->win->dcTimeout = dcTimeout;

	if (nar)
		global.synths->win->flags |= BRIGHTON_NO_ASPECT;

	/*
	 * Finally go and let the event manager handle our interface. Going to 
	 * create a separate GUI thread, which will allow us to handle things like
	 * MIDI events from the engine, timed operations, etc, from there.
	 */
	if (pthread_create(&thread, NULL, eventMgr, NULL) != 0)
		printf("Could not create GUI thread\n");
	if (pthread_detach(thread) != 0)
		printf("Could not detach GUI thread\n");

	brightonOpacity(global.synths->win, ((float) opacity) / 100.0f);

	/*
	 * These should be synth specific?
	 */
	for (i = 0; i < 128; i++)
	{
		global.synths->win->midimap[i] = i;

		for (j = 0; j < 128; j++)
			global.synths->win->valuemap[i][j] = j;
	}

	printf("user r %i/%i, e %i/%i\n", getuid(), getgid(), geteuid(), getegid());

	/*
	 * Here we should open an ALSA SEQ interface. We want to get an index back
	 * and pass that to the event handler for selection. This should not really
	 * be a SEQ interface, it should depend on the flags given to the GUI.
	 */
	if (global.libtest != 1)
	{
		if ((midiHandle = bristolMidiOpen("brighton",
			BRISTOL_CONN_SEQ|BRISTOL_DUPLEX,
			-1, BRISTOL_REQ_NSX, brightonMidiInput, &global)) < 0)
			printf("Error opening midi device %s\n", "0.0");

		midiFD = bristolGetMidiFD(midiHandle);
//		cFD = bristolMidiDupFD(global.controlfd, BRISTOL_REQ_NSX,
//			 brightonMidiInput, &global);
//		cFD = bristolGetMidiFD(cFD);

		printf("opened GUI midi handles: %i, %i\n", midiFD, cFD);
	}

	brightonReadConfiguration(global.synths->win,
		synthesisers[global.synths->synthtype],
		global.synths->midichannel,
		synthesisers[global.synths->synthtype]->name);

	while (1) {
		i = brightonEventMgr();

		/*
		 * This will now become a select on the ALSA SEQ socket looking for
		 * MIDI events. Not certain how they will be linked into the GUI at
		 * the moment. For now this is just a sleep until the ALSA SEQ interface
		 * registration code has been finalised.
		 *
		 * What we will be looking for are events on a MIDI channel, we will
		 * look for that MIDI channel in our window lists. We want to respond
		 * to key events and reflect that in the GUI optionally, but not send
		 * the key events since the engine will handle all those. We also want
		 * to look for controller values and have some configurable method to
		 * link those to our controls. Now this linkage will be done via the
		 * GUI, preferably, with the <Control><Middle Mouse><MIDI CC # motion>.
		 * Since the GUI handles this then we can dispatch events to another
		 * module that does the linkage. Need to be able to save and retrieve
		 * configurations - <Control><Middle> will go to this module, and all
		 * MIDI controller events as well, and it will make the linkage and
		 * dispatch the events.
		 *
		 * We should try and have a vkeydb type X event keymap.
		 *
		 * With certain intense events the activeSensing can fail, most notably
		 * with window resizing. Since 0.30.8 only a single window configure
		 * event is handled in any one pass of the event list to reduce this
		 * effect.
		 *
		 * This would perform better if X would give me its socket descriptors
		 * for a select operation. We should look at some point in putting
		 * this into a separate thread and using a semaphore since the 'busy'
		 * waiting is ugly.
		 */
//		bristolMidiDevRead(cFD, &msg);
		bristolMidiDevRead(midiFD, &msg);

		if (i == 0)
			usleep(mwt);

		/*
		 * We should have some 'tack' in here where we call a routine in the
		 * library that will execute any timed events that have been requested,
		 * this will cover things like flashing lights, VU metering. It will
		 * also be used to cover the midi sequencer.
		 *
		 * We should also attempt to recover lost time in graphical processing
		 * by changing mwt into a target sleep period by getting the current
		 * time and looking at the ms delta.
		 */
		brightonFastTimer(0, 0, 0, BRIGHTON_FT_CLOCK, mwt / 1000);

		if ((activeSense > 0) && ((asc -= mwt / 1000) < 0))
		{
			asc = activeSense;

			/*
			 * Hm, this is wrong, we should scan the whole synths list but for
			 * now we actually need to send one for each emulation, dual manual
			 * and all that.
			 * The check on send status is not really necessary, we will 
			 * probably end up seeing the SIGPIPE handler have us exit if the
			 * socket closes at the remote end.
			 */
			if (bristolMidiSendMsg(global.controlfd, global.synths->sid,
				BRISTOL_ACTIVE_SENSE, 0, activeSensePeriod) != 0)
			{
				printf("Active sense transmit failure\n");
				exit(1);
			}
			if ((global.synths->sid2 > 0) &&
				(bristolMidiSendMsg(global.controlfd, global.synths->sid2,
					BRISTOL_ACTIVE_SENSE, 0, activeSensePeriod) != 0))
			{
				printf("Active sense transmit failure\n");
				exit(1);
			}

			/*
			 * We are going to piggyback the slow event timer onto this code.
			 * It means the slow events will be related to the activeSensing
			 * but if anybody has issues due to changing AS or its timers we
			 * can review it at that time.
			 */
			brightonSlowTimer(0, 0, BRIGHTON_ST_CLOCK);
		}
	}
}

extern int vuInterval;

void *
eventMgr()
{
	while (1)
	{
		/*
		 * Do whatever we want. Will turn into a wait routine on the MIDI
		 * channel. This could be merged with the UI heartbeat code.
		 */
		if (vuInterval != 0)
		{
			usleep(vuInterval);
			if (global.synths->flags & OPERATIONAL)
				doAlarm();
		}
		else
			sleep(1);
	}
}

void
cleanout(void *id)
{
	if (id)
		brightonRemoveInterface(id);
	cleanupBristol();
	exit(4);
}

void
clearout(int result)
{
	exit(result);
}

