
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>

#include "brightoninternals.h"

int
brightonFindColor(brightonPalette *palette, int ncolors,
unsigned short r, unsigned short g, unsigned short b, float match)
{
	register int i;
	register unsigned short rmin, rmax, gmin, gmax, bmin, bmax;
	float lesser = match , greater = 1 / match;

	//printf("find %i, %i, %i %f %i\n", r, g, b, match, ncolors);

	rmin = (lesser  * ((float) r));
	if ((greater  * ((float) r)) > 65535)
		rmax = 65535;
	else
		rmax = (greater * ((float) r));

	gmin = (lesser  * ((float) g));
	if ((greater  * ((float) g)) > 65535)
		gmax = 65535;
	else
		gmax = (greater * ((float) g));

	bmin = (lesser  * ((float) b));
	if ((greater  * ((float) b)) > 65535)
		bmax = 65535;
	else
		bmax = (greater * ((float) b));

	if (lesser > greater)
		lesser = greater = 1.0;

	for (i = 0; i < ncolors; i++)
	{
		if (palette[i].flags & BRIGHTON_INACTIVE_COLOR)
			continue;

		if ((palette[i].red >= rmin) &&
			(palette[i].red <= rmax) &&
			(palette[i].green >= gmin) &&
			(palette[i].green <= gmax) &&
			(palette[i].blue >= bmin) &&
			(palette[i].blue <= bmax))
		{
//printf("reusing color\n");
			return(i);
		}
	}
	return(-1);
}

int
brightonFindFreeColor(brightonPalette *palette, int ncolors)
{
	int i;

	for (i = 0; i < ncolors; i++)
	{
		if (palette[i].flags & BRIGHTON_INACTIVE_COLOR)
		{
//printf("using %i colors\n", i);
			return(i);
		}
	}
	return(-1);
}

brightonFreeGC(brightonWindow *bwin, int index)
{
	if (--bwin->display->palette[index].uses == 0)
	{
		BFreeColor(bwin->display, &bwin->display->palette[index]);
	}
}

/*
 * The primary use of GCs is for color selection, and then also primarily in the
 * forgreound. We are going to request colors as R/G/B tuples, as yet not with
 * any structure management although this may happen. We will consider the use
 * of hash table lookup with best fit, limiting the total number of colours
 * that brighton can reserve, and keep stats on lookup performance for different
 * hashing functions.
 *
 * To minimise color requirements it is preferable for multiple synths to be
 * children, ie, share the same contexts.
 */
int
brightonGetGC(brightonWindow *bwin,
unsigned short r, unsigned short g, unsigned short b)
{
	int pindex, free = -1;

	//printf("brightonGetGC(%x, %x, %x)\n", r, g, b);

	/*
	 * See if we can find this color
	 */
	if ((pindex = brightonFindColor(bwin->display->palette, bwin->cmap_size,
		r, g, b, 0.995)) >= 0)
	{
		bwin->display->palette[pindex].uses++;
//printf("found match %i, uses %i\n", pindex, bwin->display->palette[pindex].uses);
		return(pindex);
	}

	/*
	 * If we have no free colors, then palette[0] is the default
	 */
	if ((pindex =
		brightonFindFreeColor(bwin->display->palette, bwin->cmap_size)) < 0)
		return(0);

	bwin->display->palette[pindex].red =  r;
	bwin->display->palette[pindex].green = g;
	bwin->display->palette[pindex].blue = b;
	bwin->display->palette[pindex].uses++;

	bwin->display->palette[pindex].flags &= ~BRIGHTON_INACTIVE_COLOR;
	bwin->display->palette[pindex].uses++;

	return(pindex);
}

/*
 * With named colors we need to allocate them in advance, we cannot leave this
 * up to the BRendering process.
 */
int
brightonGetGCByName(brightonWindow *bwin, char *name)
{
	int pindex, free = -1;

	//printf("brightonGetGCByName(%s)\n", name);

	/*
	 * If we have no free colors, then palette[0] is the default
	 */
	if ((pindex = brightonFindFreeColor(bwin->display->palette, bwin->cmap_size)) < 0)
		return(0);

	bwin->display->palette[pindex].uses++;

	BAllocColorByName(bwin->display, &bwin->display->palette[pindex], name);

	bwin->display->palette[pindex].flags &= ~BRIGHTON_INACTIVE_COLOR;
	bwin->display->palette[pindex].uses++;

	return(pindex);
}

/*
 * Make sure we have a suitable visual, then try and get a shitload of colors.
 * We should consider looking for any existing visuals of an acceptable type?
 */
brightonPalette *
brightonInitColormap(brightonWindow *bwin, int ncolors)
{
	//printf("brightonInitColormap(%i)\n", ncolors);

	if (bwin->display->palette == NULL)
	{
		int i;

		bwin->display->palette = (brightonPalette *)
			brightonmalloc(ncolors * sizeof(brightonPalette));

		for (i = 0; i < ncolors; i++)
		{
			bwin->display->palette[i].flags |= BRIGHTON_INACTIVE_COLOR;
			bwin->display->palette[i].pixel = -1;
		}
	}

	BInitColorMap(bwin->display, ncolors);
}

brightonSprintColor(brightonWindow *bwin, char *cstring, int pixel)
{
	int i;

	sprintf(cstring, "#%02x%02x%02x",
		bwin->display->palette[pixel].red >> 8,
		bwin->display->palette[pixel].green >> 8,
		bwin->display->palette[pixel].blue >> 8);
}

