/******************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BsC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BsC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "ViewWindow.h"
#include "InfoField.h"
#include "Attr.h"
#include "Viewer.h"
#include "Editor.h"
#include "Searcher.h"
#include "Busy.h"
#include "Shared.h"
#include "FtpServersList.h"
#include "Config.h"
#include "Tree.h"
#include "Limits.h"
#include "PleaseWait.h"
#include "DeleteQuestion.h"
#include <qcombobox.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qmessagebox.h>
#include <qdir.h>
#include <qinputdialog.h>
#include <qlistbox.h>
#include <qftp.h>
#include <qlineedit.h>
#include <qstringlist.h>
#include <unistd.h>
#include <qmessagebox.h>
#include <qapplication.h>
using namespace std;

/*------- local constants:
-------------------------------------------------------------------*/
const int     ViewWindow::OVER_STRETCH               = 100;
const int     ViewWindow::SPACING_WIDTH              = 2;
const int     ViewWindow::MARGIN_WIDTH               = 2;
const int     ViewWindow::MAX_ITEMS_ON_LIST          = 5;
//...................................................................
const QString ViewWindow::CONFIG_KEY_PATH            = "Path";
const QString ViewWindow::CONFIG_KEY_FILE            = "File";
const QString ViewWindow::CONFIG_KEY_COMBO           = "Combo";
const QString ViewWindow::KDE_EXECUTE                = "kfmclient exec ";
//...................................................................
const QString ViewWindow::DIR_INFO_LABEL             = QT_TR_NOOP( "Dirs:" );
const QString ViewWindow::FILE_INFO_LABEL            = QT_TR_NOOP( "Files:" );
const QString ViewWindow::SELECTION_INFO_LABEL       = QT_TR_NOOP( "Selected:" );
const QString ViewWindow::MSGBOX_DEL_QUEST_CAPTION   = QT_TR_NOOP( "Removing file(s)/directory(s)" );
const QString ViewWindow::MSGBOX_DEL_QUEST_MSG       = QT_TR_NOOP( "Are you sure to delete file(s)/directory(s)?" );
const QString ViewWindow::MSGBOX_EMPTY_QUEST_CAPTION = QT_TR_NOOP( "Empying directory" );
const QString ViewWindow::MSGBOX_EMPTY_QUEST_MSG     = QT_TR_NOOP( "Are you sure to empty directory:\n%1" );
const QString ViewWindow::INDLG_F6_CAPTION           = QT_TR_NOOP( "Rename a file/directory" );
const QString ViewWindow::INDLG_F6_MSG               = QT_TR_NOOP( "Current name: %1\nEnter new name:" );
const QString ViewWindow::MSGBOX_F6_CAPTION          = QT_TR_NOOP( "Error on renaming" );
const QString ViewWindow::MSGBOX_F6_MSG              = QT_TR_NOOP( "Can't rename\nfrom: %1\nto: %2" );
const QString ViewWindow::INDLG_F7_CAPTION           = QT_TR_NOOP( "Create subdirectory" );
const QString ViewWindow::INDLG_F7_MSG               = QT_TR_NOOP( "Enter subdirectory name:" );
const QString ViewWindow::MSGBOX_F7_CAPTION          = QT_TR_NOOP( "Error on create a subdirectory" );
const QString ViewWindow::MSGBOX_F7_MSG              = QT_TR_NOOP( "Can't create subdirectory: %1" );
const QString ViewWindow::CANT_SAVE_FILE             = QT_TR_NOOP( "You can't save this file.\nUse a viewer please." );
const QString ViewWindow::CANT_READ_FILE             = QT_TR_NOOP( "You can't read this file.\nProblems with permission acces." );
const QString ViewWindow::BINARY_FILE                = QT_TR_NOOP( "This is a binary file. Can't edit." );
const QString ViewWindow::TOUCH_CAPTION              = QT_TR_NOOP( "Create new empty file" );
const QString ViewWindow::TOUCH_PROMPT               = QT_TR_NOOP( "Enter a new file name:" );
const QString ViewWindow::TOUCH_FILE_EXISTS          = QT_TR_NOOP( "File with this name already exists." );
const QString ViewWindow::TOUCH_CANT_CREATE          = QT_TR_NOOP( "Can't create the file." );
const QString ViewWindow::FTP_MSG_CAPTION            = QT_TR_NOOP( "FTP connection" );
const QString ViewWindow::MD5Caption                 = QT_TR_NOOP( "MD5 check sum" );
const QString ViewWindow::MD5Msg                     = QT_TR_NOOP( "Check sum of %1\nResult: " );
const QString ViewWindow::MD5Ok                      = QT_TR_NOOP( "is valid" );
const QString ViewWindow::MD5Error                   = QT_TR_NOOP( "failure" );
const QString ViewWindow::TarExt                     = "tar";
const QString ViewWindow::GzipExt                    = "gz";
const QString ViewWindow::Bzip2Ext                   = "bz2";
const QString ViewWindow::ZipExt                     = "zip";
const QString ViewWindow::DeletingCaption            = QT_TR_NOOP( "Deleting file(s)/directory(s)" );
const QString ViewWindow::PleaseWaitMsg              = QT_TR_NOOP( "Please wait..." );


/*------- local static variable:
-------------------------------------------------------------------*/
int    ViewWindow::s_pack_idx = 0;


//*******************************************************************
// ViewWindow
//*******************************************************************
ViewWindow::ViewWindow( QWidget* const in_parent )
: QFrame             ( in_parent )
, d_path             ( new QComboBox( this ))
, d_fstab            ( new QComboBox( this ))
, d_table            ( new ViewTable( this ))
, d_status           ( new QLineEdit( this ))
, d_ftp              ( new QFtp( this ))
, d_dir_info_lbl     ( 0 )
, d_file_info_lbl    ( 0 )
, d_sel_info_lbl     ( 0 )
, d_dir_counter      ( 0 )
, d_file_counter     ( 0 )
, d_tmp_packed_path  ( Shared::EmptyStr )
, d_src_packed_path  ( Shared::EmptyStr )
, d_compress_type    ( NO_PACKED )
, d_is_ftp_dir       ( FALSE )
, d_is_ftp_root      ( FALSE )
, d_prv_dir          ( Shared::EmptyStr )
, d_prv_item         ( Shared::EmptyStr )
, d_max_items_on_list( MAX_ITEMS_ON_LIST )
, d_ftp_cmd          ( FTP_CMD_NONE )
, d_pwd_cmd_id       ( -1 )
, d_loop             ( FALSE )
, d_devname          ( "" )
{
	d_path->setEditable( TRUE );
	d_path->setFocusPolicy( ClickFocus );
	d_path->setAutoCompletion( TRUE );
	d_path->setInsertionPolicy( QComboBox::NoInsertion );

	d_fstab->setFocusPolicy( ClickFocus );
	d_fstab->setInsertionPolicy( QComboBox::NoInsertion );
	if( Limits::instance()->apply() ) d_fstab->hide();

	d_status->setReadOnly( TRUE );
	d_status->hide();
	d_status->setFocusPolicy( NoFocus );
	
	QVBoxLayout* const layout = new QVBoxLayout( this );
	layout->setSpacing( SPACING_WIDTH );
	layout->setMargin( MARGIN_WIDTH );
			
	QHBoxLayout* const path_layout = new QHBoxLayout;
	path_layout->addWidget( d_path );
	path_layout->addWidget( d_fstab );
	path_layout->setStretchFactor( d_path, 5 );
	path_layout->setStretchFactor( d_fstab, 2 );
	layout->addLayout( path_layout );

	layout->addWidget( d_table );
	layout->addWidget( d_status );
			
	layout->addWidget( make_info_area() );
	layout->setStretchFactor( d_table, OVER_STRETCH );

	connect( d_table, SIGNAL( remove            ()                                     ),
				this   , SLOT  ( F8                ()                                     ));
	connect( d_table, SIGNAL( next_dir          ()                                     ),
				this   , SLOT  ( inc_dir_counter   ()                                     ));
	connect( d_table, SIGNAL( next_file         ()                                     ),
				this   , SLOT  ( inc_file_counter  ()                                     ));
	connect( d_table, SIGNAL( selections_update ( int )                                ),
				this   , SLOT  ( selections_changed( int )                                ));
	connect( d_table, SIGNAL( doubleClicked     ( QListViewItem*, const QPoint&, int ) ),
				this   , SLOT  ( execute           ( QListViewItem*, const QPoint&, int ) ));
	connect( d_table, SIGNAL( cd_home           ()                                     ),
				this   , SLOT  ( cd_home           ()                                     ));
	connect( d_table, SIGNAL( cd_up             ()                                     ),
				this   , SLOT  ( cd_up             ()                                     ));
	connect( d_table, SIGNAL( cd_root           ()                                     ),
				this   , SLOT  ( cd_root           ()                                     ));
	connect( d_path->lineEdit() , SIGNAL( returnPressed()                              ),
				this   , SLOT  ( return_pressed    ()                                     ));
	connect( d_path , SIGNAL( activated         ( int )                                ),
				this   , SLOT  ( activated         ( int )                                ));
	connect( d_fstab, SIGNAL( activated         ( int )                                ),
	         this   , SLOT  ( fstab             ( int )                                ));
	connect( d_ftp  , SIGNAL( done              ( bool )                               ),
				this   , SLOT  ( ftp_done          ( bool )                               ));
	connect( d_ftp  , SIGNAL( stateChanged      ( int )                                ),
				this   , SLOT  ( ftp_state_changed ( int )                                ));
	connect( d_ftp  , SIGNAL( listInfo          ( const QUrlInfo& )                    ),
				this   , SLOT  ( ftp_list_info     ( const QUrlInfo& )                    ));
	connect( d_ftp  , SIGNAL( commandFinished   ( int, bool )                          ),
				this   , SLOT  ( ftp_cmd_finish    ( int, bool )                          ));
	connect( d_ftp  , SIGNAL( rawCommandReply   ( int, const QString& )                ),
				this   , SLOT  ( ftp_cmd_reply     ( int, const QString& )                ));
	connect( &d_timer,SIGNAL( timeout           ()                                     ),
				this   , SLOT  ( timeout           ()                                     ));
	connect( this   , SIGNAL( selection_changed ()                                     ),
				d_table, SLOT  ( selection_mode_changed ()                             ));
	
	read_fstab();
	looks_refresh();
}
// end of ViewWindow

//*******************************************************************
// ~ViewWindow                                            DESTRUCTOR
//*******************************************************************
ViewWindow::~ViewWindow()
{
	if( d_ftp ) {
		if( QFtp::Connected == d_ftp->state() ) {
			d_ftp->close();
			while( d_ftp->state() != QFtp::Unconnected );
		}
		delete d_ftp;
	}
}
// end of ~ViewWindow

//*******************************************************************
// looks_refresh                                              PUBLIC
//*******************************************************************
void ViewWindow::looks_refresh()
{
	if( d_is_ftp_dir ) {
		d_table->setFont( Config::instance()->ftp_font() );
		d_path->setFont( Config::instance()->ftp_font() );
		d_fstab->setFont( Config::instance()->ftp_font() );
		d_status->setFont( Config::instance()->ftp_font() );
		d_table->setPaletteBackgroundColor( Config::instance()->ftp_bkg_color() );
		d_status->setPaletteBackgroundColor( Config::instance()->ftp_bkg_color() );
	}
	else {
		d_table->setFont( Config::instance()->lfs_font() );
		d_path->setFont( Config::instance()->lfs_font() );
		d_fstab->setFont( Config::instance()->lfs_font() );
		d_status->setFont( Config::instance()->lfs_font() );
		d_table->setPaletteBackgroundColor( Config::instance()->lfs_bkg_color() );
		d_status->setPaletteBackgroundColor( Config::instance()->lfs_bkg_color() );
	}
	d_table->adjust();
}
// end of looks_refresh

//*******************************************************************
// lang_changed                                               PUBLIC
//*******************************************************************
void ViewWindow::lang_changed()
{
	d_dir_info_lbl->setText( tr(DIR_INFO_LABEL) );
	d_file_info_lbl->setText( tr(FILE_INFO_LABEL) );
	d_sel_info_lbl->setText( tr(SELECTION_INFO_LABEL) );
	d_table->retranslate_strings();
}
// end of lang_changed

//*******************************************************************
// keyPressEvent                                   PRIVATE inherited
//*******************************************************************
void ViewWindow::keyPressEvent( QKeyEvent* e )
{
	bool accepted = FALSE;

	if( Qt::ControlButton == e->state() ) {
		switch( e->key() ) {
			case Qt::Key_R:
				e->accept();
				accepted = TRUE;
				cd_refresh();
				break;
			case Qt::Key_K:
				e->accept();
				accepted = TRUE;
				ctrl_K();
				break;
			case Qt::Key_F7:
				e->accept();
				accepted = TRUE;
				start_searcher();
				break;
			case Qt::Key_Return:
				e->accept();
				accepted = TRUE;
				ctrl_return();
				break;
			case Qt::Key_L:
				e->accept();
				accepted = TRUE;
				ctrl_L();
				break;
		}
	}
	else {
		switch( e->key() ) {
			case Qt::Key_Escape:
                d_loop = FALSE;
				break;
			case Qt::Key_F3:
				e->accept();
				accepted = TRUE;
				F3();
				break;
			case Qt::Key_F4:
				e->accept();
				accepted = TRUE;
				F4();
				break;
			case Qt::Key_F6:
				e->accept();
				accepted = TRUE;
				F6();
				break;
			case Qt::Key_F7:
				e->accept();
				accepted = TRUE;
				F7();
				break;
			case Qt::Key_F8:
				e->accept();
				accepted = TRUE;
				F8();
				break;
			case Qt::Key_Return:
				e->accept();
				accepted = TRUE;
				execute( d_table->currentItem() );
				break;
		}
	}
	
	if( FALSE == accepted ) {
		QFrame::keyPressEvent( e );
	}
}
// end of keyPressEvent

//*******************************************************************
// ctrl_return                                               PRIVATE
//-------------------------------------------------------------------
// Nacisniecie kombinacji klawiszy Ctrl+Return dotyczy tylko i
// wylacznie katalogow. Jesli aktualna wskazywana pozycja jest
// katalog to przechodzimy do niego, ale w NOWEJ ZAKLADCE.
//*******************************************************************
void ViewWindow::ctrl_return()
{
	if( FALSE == d_is_ftp_dir ) {
		const ViewTableItem* const item = dynamic_cast<ViewTableItem*>( d_table->currentItem() );
		if( item ) {
			if( item->is_dir() ) {
				if( item->is_executable() && item->is_readable() ) {
					emit open_new_tab( item->path() );
				}
			}
		}
	}
}
// end of ctrl_return

//*******************************************************************
// ctrl_K                                                    PRIVATE
//-------------------------------------------------------------------
// Nacisniecie kombinacji Ctrl+K powoduje uruchomienie terminala.
//*******************************************************************
void ViewWindow::ctrl_K()
{
	const QString dir_marker = "$dir";
	
	QString call = Config::instance()->terminal_call();
	if( call.isEmpty() ) call = Shared::KonsoleCall;
	if( call.find( dir_marker ) != -1 ) call.replace( dir_marker, get_dir() );
	
	const int retval = system( call + " &" );
	if( -1 == retval ) {
	}
}
// end of ctrl_K

//*******************************************************************
// ctrl_L                                                    PRIVATE
//-------------------------------------------------------------------
// Nacisniecie kombinacji klawiszy Ctrl+L powoduje przejscie do
// pola edycyjnego z aktualna sciezka, znajdujacego sie nad lista
// plikow. Daje to mozliwosc uzytkownikowi wpisanie/zmodyfikowanie
// sciezki, do ktorej chce przejsc.
//*******************************************************************
void ViewWindow::ctrl_L()
{
	QLineEdit* const edit = d_path->lineEdit();
	edit->selectAll();
	edit->setFocus();
}
// end of ctrl_L

//*******************************************************************
// disp_this_location                                         PUBLIC
//*******************************************************************
void ViewWindow::disp_this_location( const QString& in_dname, const QString& in_fname )
{
	bool is_ok = FALSE;
	
	if( FALSE == in_dname.isEmpty() ) {
		const QFileInfo fi( in_dname );
		is_ok = ( fi.isDir() && fi.isReadable() && fi.isExecutable() );
		if( is_ok ) is_ok = Limits::instance()->accepted( in_dname );
	}
	
	if( is_ok ) {
		d_dir.setPath( in_dname );
		read_dir();
		if( FALSE == in_fname.isEmpty() ) {
			d_table->select_by_name( in_fname );
		}
	}
	else {
		d_dir.setPath( QDir::homeDirPath() );
		emit tab_update( d_dir.path() );
		read_dir();
	}
}
// end of disp_this_location

//*******************************************************************
// get_dir                                                    PUBLIC
//*******************************************************************
QString ViewWindow::get_dir()
{
	return ( d_is_ftp_dir ) ? d_fsi.dir() : d_dir.absPath();
}
// end of get_dir

//*******************************************************************
// get_current_fname                                          PUBLIC
//*******************************************************************
QString ViewWindow::get_current_fname()
{
	QString retval = Shared::EmptyStr;
	d_table->get_current_fname( retval );
	return retval;
}
// end of get_path

//*******************************************************************
// get_combobox_item
//*******************************************************************
QStringList ViewWindow::get_combox_items()
{
	QStringList item_list;
	
	for( int i = 0; i < d_path->count(); ++i ) {
		item_list.append( d_path->text( i ) );
	}
	return item_list;	
}
// end of get_combobox_item

//*******************************************************************
// get_combobox_item
//*******************************************************************
void ViewWindow::set_combox_items( const QStringList& in_data )
{
	d_path->insertStringList( in_data );
}
// end of get_combobox_item

//****************************************************************
// make_info_area                                         PRIVATE
//****************************************************************
QWidget* ViewWindow::make_info_area()
{
	QFrame* const panel = new QFrame( this );
	QHBoxLayout* const layout = new QHBoxLayout( panel );
	layout->setSpacing( SPACING_WIDTH );
	layout->setMargin( MARGIN_WIDTH );

	layout->addWidget( d_dir_info_lbl = new QLabel( tr(DIR_INFO_LABEL), panel ));
	layout->addWidget( d_dir_number = new InfoField( panel ));
	d_dir_number->setFocusPolicy( NoFocus );

	layout->addWidget( d_file_info_lbl = new QLabel( tr(FILE_INFO_LABEL), panel ));
	layout->addWidget( d_file_number = new InfoField( panel ));
	d_file_number->setFocusPolicy( NoFocus );			

	layout->addWidget( d_sel_info_lbl = new QLabel( tr(SELECTION_INFO_LABEL), panel ));
	layout->addWidget( d_selection_number = new InfoField( panel ));
	d_selection_number->setFocusPolicy( NoFocus );

	return panel;
}
// end of make_info_area

//*******************************************************************
// update_view                                          PRIVATE slot
//*******************************************************************
void ViewWindow::update_view( const SELECTION_CHANGE in_selection_change )
{
	QString current_item_name;
	d_table->get_current_fname( current_item_name );

	if( REMOVE_SELECTION == in_selection_change ) {
		d_table->clearSelection();
	}

	if( d_is_ftp_dir ) {
		ftp_list( FALSE );
		d_table->set_fname_on_start( current_item_name );
	}
	else {
		refresh_dir();
		d_table->select_by_name( current_item_name );
	}
}
// end of update_view

//*******************************************************************
// return_pressed                                       PRIVATE slot
//-------------------------------------------------------------------
// Reakcja na nacisniecie klawisza 'return' w polu edycyjnym 'd_path'.
// Aktualnie wprowadzona sciezka wprowadzana jest na poczatek listy
// (jesli byla juz gdzies na liscie zostanie z poprzedniej pozycji
// usunieta). Poniewaz przejmuje to zdarzenie zadbamy o wszystko
// sami.
//*******************************************************************
void ViewWindow::return_pressed()
{
	const QString path = d_path->currentText();
	if( path != d_dir.absPath() ) {
		if( Limits::instance()->accepted( path ) ) { // pytanie o zgode administratora
			// sprawdz czy mozna przejsc do katalogu (ograniczenia)
			const QFileInfo fi( path );
			if( fi.isDir() && fi.isReadable() && fi.isExecutable() ) {
				if( d_compress_type != NO_PACKED ) {
					close_packed();
				}
			
				// Przechodzimy do wprowadzonego katalogu
				// i wyswietlamy jego zawartosc.
				d_dir.setPath( path );
				refresh_dir();
			
				// Jesli wpisana sciezka juz jest na liscie
				// to ja usuwamy, bo ja wpiszemy za chwile
				// na poczatek listy.
				for( int i = 0; i < d_path->count(); ++i ) {
					if( path == d_path->text( i ) ) {
						d_path->removeItem( i );
						break;
					}
				}
			
				// Wstawiamy wprowadzona sciezke na poczatek
				// listy.
				d_path->insertItem( path, 0 );
			
				// Usuwamy z listy pozycje, ktore przekracza
				// narzucony limit pozycji na liscie.
				if( d_path->count() > d_max_items_on_list ) {
					d_path->listBox()->removeItem( d_max_items_on_list );
				}

				// Fokus przekazujemy do tabeli zawierajacej
				// juz zawartosc katalogu.
				d_table->setFocus();
			}
		}
	}
}
// end of return_pressed

//*******************************************************************
// activated                                            PRIVATE slot
//*******************************************************************
void ViewWindow::activated( int in_index )
{
	const QString path = d_path->text( in_index );
	d_path->removeItem( in_index );
	d_path->insertItem( path, 0 );
	d_path->setCurrentText( path );
	
	d_dir.setPath( path );
	refresh_dir();
	d_table->setFocus();
}
// end of activated

//*******************************************************************
// fstab                                                PRIVATE slot
//-------------------------------------------------------------------
// Uzytkownik wybrak jedna z pozycji odczytanych z pliku fstab.
//*******************************************************************
void ViewWindow::fstab( int in_index )
{
	bool must_by_mounted = FALSE;
	
	if( in_index ) {
		const QString devname = d_fstab->text( in_index );
		bool mounted = is_mounted( devname );
		if( FALSE == mounted ) {
			must_by_mounted = TRUE;
			mounted = ( 0 == system( "mount " + devname ));
		}
		if( mounted ) {
			if( FALSE == d_devname.isEmpty() ) {
				const int retval = system( "umount " + d_devname );
				if( -1 == retval ) {
				}
				d_devname = "";
			}
			if( must_by_mounted ) {
				d_devname = devname;
			}
			d_dir.setPath( devname );
			refresh_dir();
		}
	}
	d_table->setFocus();
}
// end of fstab

//*******************************************************************
// touch                                                     PRIVATE
//-------------------------------------------------------------------
// Utworzenie nowego pustego pliku.
//*******************************************************************
void ViewWindow::touch()
{
	if( FALSE == d_is_ftp_dir ) {
		bool ok;
		const QString new_name = QInputDialog::getText(	tr(TOUCH_CAPTION),
											tr(TOUCH_PROMPT),
											QLineEdit::Normal,
											QString::null,
											&ok,
											this );

		if( ok && ( FALSE == new_name.isEmpty())) {
			const QString fpath = get_dir() + Shared::DirSep + new_name;
			const QFileInfo fi( fpath );
			if( fi.exists() ) {
				QMessageBox::warning( this, tr(TOUCH_CAPTION), tr(TOUCH_FILE_EXISTS) );
			}
			else if( -1 == system( Shared::TouchPrgName + Shared::Spc + fpath )) {
				QMessageBox::warning( this, tr(TOUCH_CAPTION), tr(TOUCH_CANT_CREATE) );
				ok = FALSE;
			}
		
			if( ok ) {
				refresh_dir();
				d_table->select_by_name( new_name );
			}
		}
	}
}
// end of touch

//*******************************************************************
// cd_up                                                PRIVATE slot
//*******************************************************************
void ViewWindow::cd_up()
{
	if( !d_dir.isRoot() ) {
		QDir tmp = d_dir;
		tmp.cdUp();
		if( Limits::instance()->accepted( tmp.absPath() ) ) { // pytanie o zgode administratora
			const QString prv_dir = d_dir.dirName();
			d_dir.cdUp();
			refresh_dir();
		
			if( prv_dir != QString::null ) {
				d_table->select_by_name( prv_dir );
			}
		}
	}
}
// end of cd_up

//*******************************************************************
// cd_home                                              PRIVATE slot
//*******************************************************************
void ViewWindow::cd_home()
{
	const QString home_dir = d_dir.homeDirPath();
	
	if( Limits::instance()->accepted( home_dir ) ) { // pytanie o zgode administratora
		d_dir.cd( home_dir );
		refresh_dir();
	}
}
// end of cd_home

//*******************************************************************
// cd_root                                              PRIVATE slot
//*******************************************************************
void ViewWindow::cd_root()
{
	const QString root_dir = Shared::DirSep;
	if( Shared::is_dir_ok( root_dir ) ) {
		if( Limits::instance()->accepted( root_dir ) ) {	// pytanie o zgode administratora
			d_dir.cd( root_dir );
			refresh_dir();
		}
	}
}
// end of cd_home

//*******************************************************************
// cd_refresh                                           PRIVATE slot
//*******************************************************************
void ViewWindow::cd_refresh()
{
	refresh_dir();
}
// end of cd_refresh

//*******************************************************************
// execute                                              PRIVATE slot
//*******************************************************************
void ViewWindow::execute( QListViewItem* in_item, const QPoint&, int )
{
	execute( in_item );
}
void ViewWindow::execute( QListViewItem* in_item )
{
	ViewTableItem* const item = dynamic_cast<ViewTableItem*>( in_item );
	
	if( item ) {
		QString fname, fext;
		d_table->get_fname_fext( item, fname, fext );
		if( !fext.isEmpty() ) fname += ( Shared::ThisDir + fext );
	
		// ------- CD UP ------- 
		if( Shared::ParentDir == fname ) {
			// Jezeli jestesmy wewnatrz archiwum i aktualnym katalogiem
			// jest katalog poczatkowy archiwum, to znaczy ze nalezy
			// wrocic do katalogu, ktory byl w uzyciu przed wejsciem do
			// archiwum.
			if(( d_compress_type != NO_PACKED ) && ( d_dir.absPath() == d_tmp_packed_path )) {
				close_packed();
				d_dir.cd( d_prv_dir );
				refresh_dir();
			}
			else if( d_is_ftp_dir ) {
				ftp_cd( Shared::ParentDir, FALSE );
			}
			// Jezeli jestesmy w normalnym katalogu lub w ktoryms z
			// podkatalogow archiwum wystarczy przejsc do katalogu
			// powyzej.
			else {
				cd_up();
			}
		}
		// ------- CD ------- 
		else {
			// WYBRANO KATALOG
			if( item->is_dir() ) {
				if( d_is_ftp_dir ) {
					ftp_cd( item->path(), FALSE );
				}
				else {
					// wejscie do normalnego katalogu na dysku
					if( item->is_executable() && item->is_readable() ) {
						if( Limits::instance()->accepted( item->path() ) ) {	// pytanie o zgode administratora
							d_dir.cd( item->path() );
							refresh_dir();
						}
					}
					else {
						// brak dostepu do katalogu
						QMessageBox::warning( this,
								Shared::ProgramName,
								tr(Shared::NotReadableDir).arg( fname ));
					}
				}
			}
			// WYBRANO PLIK
			else {
				     if( TarExt   == fext ) d_compress_type = TAR;
				else if( GzipExt  == fext ) d_compress_type = GZIP;
				else if( Bzip2Ext == fext ) d_compress_type = BZIP2;
				else if( ZipExt   == fext ) d_compress_type = ZIP;
				else                        d_compress_type = NO_PACKED;

				if( NO_PACKED == d_compress_type ) {
					const int retval = system( KDE_EXECUTE + item->path() );
					if( -1 == retval ) {
					}
				}
				else {
					open_packed( item->path(), d_compress_type );
				}
			}
		}
	}
}
// end of slot_execute

//###################################################################
//#                                                                 #
//#                         P A C K I N G                           #
//#                                                                 #
//###################################################################

//*******************************************************************
// open_packed                                               PRIVATE
//-------------------------------------------------------------------
// Otwarcie spakowane pliku jako pseudo-katalogu.
// Tak naprawde plik jest rozpakowany w specjalnym, ukrytym
// katalogu 'd_tmp_packed_path'. Kazde rozpakowanie odbywa sie w
// UNIKALNYM katalogu o nazwie okreslonej przez 's_pack_idx'.
//*******************************************************************
void ViewWindow::open_packed( const QString& in_fpath, const CompressType in_type )
{
	// Generujemy nazwe katalog do rozpakowania.
	const QString user_home_dir = Shared::get_home_dir();
	const QString sub_dirs      = Shared::get_pack_dir() + Shared::DirSep + QString::number( ++s_pack_idx );
	d_tmp_packed_path           = user_home_dir + sub_dirs;

	Shared::remove_path_and_content( d_tmp_packed_path, TRUE );
	
	// Tworzymy katalog do rozpakowania.
	if( Shared::create_path( user_home_dir, sub_dirs ) ) {
		QString cmd = "";
		
		// TAR, GZIP, BZIP2
		if( in_type != ZIP )	{
			if( TAR == in_type ) {
				cmd = "tar -xf " + in_fpath + " -C " + d_tmp_packed_path;
			}
			else if( GZIP == in_type ) {
				cmd = "tar -xzf " + in_fpath + " -C " + d_tmp_packed_path;
			}
			else if( BZIP2 == in_type ) {
				cmd = "tar -xjf " + in_fpath + " -C " + d_tmp_packed_path;
			}
		}
		// ZIP
		else {
			cmd = "unzip " + in_fpath + " -d " + d_tmp_packed_path ;
		}
		cmd += " > /dev/null";

		Busy::set_busy( TRUE );
		const int retval = system( cmd );
		Busy::set_busy( FALSE );
		
		if( 0 == retval ) {
			d_prv_dir = d_dir.absPath();		// katalog przed rozpakowaniem
			d_src_packed_path = in_fpath;		// sciezka rozpakowanego pliku
			d_dir.cd( d_tmp_packed_path );
			refresh_dir();
		}
	}
}
// end of open_packed

//*******************************************************************
// close_packed
//*******************************************************************
void ViewWindow::close_packed()
{
	QString cmd = "cd " + d_tmp_packed_path + "; ";

	     if( TAR   == d_compress_type ) cmd += "tar -cmf ";
	else if( GZIP  == d_compress_type ) cmd += "tar -cmzf ";
	else if( BZIP2 == d_compress_type ) cmd += "tar -cmjf ";
	else if( ZIP   == d_compress_type ) cmd += "zip -r ";

	cmd += ( d_src_packed_path + " * > /dev/null" );

	Busy::set_busy( TRUE );
	Shared::remove_file( d_src_packed_path, TRUE );
	const int retval = system( cmd );
	if( -1 == retval ) {
	}
	Shared::remove_path_and_content( d_tmp_packed_path, TRUE );
	Busy::set_busy( FALSE );
		
	d_compress_type = NO_PACKED;
	--s_pack_idx;
	d_src_packed_path = "";
}
// end of close_packed

//###################################################################
//#                                                                 #
//#                 R E A D   D I R E C T O R Y                     #
//#                                                                 #
//###################################################################

//*******************************************************************
// refresh_dir                                               PRIVATE
//*******************************************************************
void ViewWindow::refresh_dir()
{
	if( d_is_ftp_dir ) {
		d_dir.absPath();
		QString current_fname;
		d_table->get_current_fname( current_fname );
		d_table->set_fname_on_start( current_fname );
		ftp_list( FALSE );
	}
	else {
		d_dir.refresh();
		read_dir();
	}
}
// end of refresh_dir

//*******************************************************************
// update_start                                              PRIVATE
//-------------------------------------------------------------------
// Przygotowanie tablica do wyswitlenia nowej zawartosci.
//*******************************************************************
void ViewWindow::update_start()
{
	if( FALSE == d_devname.isEmpty() ) {
		if( d_dir.absPath() != d_devname ) {
			const int retval = system( "umount " + d_devname );
			if( -1 == retval ) {
			}
			d_devname = "";
		}
	}

	Busy::set_busy( TRUE );
	d_table->reset();
	d_dir_counter = 0;
	d_file_counter = 0;
}
// end of update_start

//*******************************************************************
// read_dir                                                   PRIVATE
//*******************************************************************
void ViewWindow::read_dir()
{
	update_start();
	
	const QFileInfoList* const fil = d_dir.entryInfoList( Shared::get_file_filter(), QDir::Unsorted  );
	if( fil ) {
		const bool is_root = d_dir.isRoot();
		QFileInfoListIterator it( *fil );
		while( it.current() ) {
			d_table->add_new_item( *it.current(), is_root );
			++it;
		}
	}

	update_finish();
}
// end of read_dir

//*******************************************************************
// update_finish                                             PRIVATE
//*******************************************************************
void ViewWindow::update_finish()
{
	/*
	if( TRUE == d_is_packed_dir ) {
		QFileInfo fi( d_src_packed_path );
		d_path->setCurrentText( Shared::TarPrgName + Shared::Colon + Shared::Spc + fi.fileName() + Shared::get_subpath( d_tmp_packed_path, d_dir.absPath() ));
	}
	else 
	*/
	if( d_is_ftp_dir ) {
		if( FALSE == d_is_ftp_root ) d_table->check_parent_mark();
		d_path->setCurrentText( d_fsi.dir() );
	}
	else {
		d_path->setCurrentText( d_dir.path() );
		emit tab_update( d_dir.path() );
		if( d_dir.path() != d_fstab->currentText() ) {
			d_fstab->setCurrentItem( 0 );
		}
	}

	d_dir_number->setText( QString::number( d_dir_counter ));
	d_file_number->setText( QString::number( d_file_counter ));
	Busy::set_busy( FALSE );
	d_table->select_first_item();
	d_table->adjust();
}
// end of update_finish

//*******************************************************************
// filter                                               PRIVATE slot
//*******************************************************************
void ViewWindow::filter( bool in_flag )
{
	Shared::toggle_file_filter( in_flag );
	refresh_dir();
}
// end of filter


//###################################################################
//#                                                                 #
//#                    OBSLUGA LICZNIKOW                            #
//#                                                                 #
//###################################################################


//*******************************************************************
// selections_changed                                   PTIVATE slot
//*******************************************************************
void ViewWindow::selections_changed( int in_number )
{
	d_selection_number->setText( QString::number( in_number ) );
}
// end of selections_changed

//*******************************************************************
// inc_dir_counter                                      PRIVATE slot
//*******************************************************************
void ViewWindow::inc_dir_counter()
{
	++d_dir_counter;
}
// end of inc_dir_counter

//*******************************************************************
// inc_file_counter                                     PRIVATE slot
//*******************************************************************
void ViewWindow::inc_file_counter()
{
	++d_file_counter;
}
// end of inc_dir_counter


//###################################################################
//#                                                                 #
//#                          SEARCHER                               #
//#                                                                 #
//###################################################################


//*******************************************************************
// start_searcher                                            PRIVATE
//*******************************************************************
void ViewWindow::start_searcher()
{
	if( FALSE == d_is_ftp_dir ) {
		Searcher* srch = new Searcher( this );
		if( srch ) {
			if( QDialog::Accepted == srch->exec() ) {
				const QFileInfo fi( srch->get_goto_data() );
				d_dir.cd( fi.dirPath( TRUE ) );
				refresh_dir();
				d_table->select_by_name( fi.baseName( TRUE ), fi.extension( FALSE ) );
			}
			delete srch;
			srch = 0;
		}
	}
}
// end of start_searcher

//*******************************************************************
// read_fstab                                                PRIVATE
//*******************************************************************
void ViewWindow::read_fstab()
{
	QFile file( "/etc/fstab" );
	d_fstab->insertItem( "fstab" );
	
	if( file.open( IO_ReadOnly ) ) {
		QString txt;
		while( FALSE == file.atEnd() ) {
			if( file.readLine( txt, 1024 ) != -1 ) {
				if( Shared::remove_white_chars( txt ) ) {
					if( txt.startsWith( "/dev/" ) ) {
						if( -1 == txt.find( "swap" ) ) {
							const QStringList items = QStringList::split( Shared::Spc, txt );
							d_fstab->insertItem( items[1] );
						}
					}
				}
			}
		}
		file.close();
	}
	d_fstab->setCurrentItem( 0 );
}
// end of read_fstab

//*******************************************************************
// is_mounted                                                PRIVATE
//*******************************************************************
bool ViewWindow::is_mounted( const QString& in_devname )
{
	bool retval = FALSE;

	QFile file( "/etc/mtab" );

	if( file.open( IO_ReadOnly ) ) {
		QString txt;
		while( FALSE == file.atEnd() ) {
			if( file.readLine( txt, 1024 ) != -1 ) {
				if( txt.find( in_devname ) != -1 ) {
					retval = TRUE;
					break;
				}
			}
		}
	}
	
	return retval;
}
// end of is_mounted


//###################################################################
//#                                                                 #
//#                       OBSLUGA ZAZNACZEN                         #
//#                                                                 #
//###################################################################


//*******************************************************************
// slot_select                                          PRIVATE slot
//*******************************************************************
void ViewWindow::slot_select()
{
	d_table->select_plus();
}
// end of slot_select

//*******************************************************************
// slot_unselect                                        PRIVATE slot
//*******************************************************************
void ViewWindow::slot_unselect()
{
	d_table->select_minus();
}
// end of slot_unselect

//*******************************************************************
// slot_revers                                          PRIVATE slot
//*******************************************************************
void ViewWindow::slot_revers()
{
	d_table->invertSelection();
}
// end of slot_revers

//*******************************************************************
// break_work                                           PRIVATE slot
//-------------------------------------------------------------------
// Slot ktory podlaczamy pod sygnal 'break_work' wysylany z
// dialogu 'PleaseWait'.
// Tutaj obslugujemy rzadanie uzytkownika aby przerwac
// trwajacy proces.
// Trwajacy proces powinien obserwowac globalnie dostepna zmienna
// 'd_break' z obiektu 'Shared'.
//*******************************************************************
void ViewWindow::break_work()
{
    Shared::d_break = TRUE;
    Shared::idle();
}
// end of break_work

//###################################################################
//#                                                                 #
//#                OBSLUGA KLAWISZY FUNKCYJNYCH                     #
//#                                                                 #
//###################################################################


//*******************************************************************
// F2 - Access Permission                               PRIVATE slot
//*******************************************************************
void ViewWindow::F2()
{
	if( d_is_ftp_dir ) {
		// to do
	}
	else {
		const ViewTable::SelectedItems& items = d_table->selections();
		if ( FALSE == items.empty() ) {
			Attr* attr = new Attr( this, get_dir(), items );
			if( attr ) {
				if( QDialog::Accepted == attr->exec() ) {
					update_view( ViewWindow::STAY_SELECTION );
				}
				delete attr;
				attr = 0;
			}
		}
	}
}
// end of F2

//*******************************************************************
// F3 - Viewer                                               PRIVATE
//*******************************************************************
void ViewWindow::F3()
{
	const ViewTableItem* const item = d_table->current_item();
	
	if( item ) {
		if( FALSE == ( item->is_dir() || item->is_parent_dir() )) {
			if( d_is_ftp_dir ) {
				// to do
			}
			else {
				Viewer* view = new Viewer( this, item );
				if( view ) {
					view->exec();
					delete view;
					view = 0;
				}
			}
		}
	}
}
// end of F3

//*******************************************************************
// F4 - Edit                                                 PRIVATE
//-------------------------------------------------------------------
// Edycja aktualnego pliku.
// UWAGA: nie bedziemy edytowac pliku binarnego.
//*******************************************************************
void ViewWindow::F4()
{
	const ViewTableItem* const item = d_table->current_item();
	
	if( item ) {
		if( FALSE == ( item->is_dir() || item->is_parent_dir() )) {
			if( d_is_ftp_dir ) {
				// to do
			}
			else {
				if( item->is_readable() ) {
					if( item->is_writable() ) {
						if( FALSE == Shared::is_binary_file( item->path() )) {
							Editor* edt = new Editor( this, item->path() );
							if( edt ) {
								edt->exec();
								delete edt;
								edt = 0;
							}
						}
						else {
							QMessageBox::critical( this, Shared::ProgramName, tr(BINARY_FILE) );
						}
					}
					else {
						QMessageBox::critical( this, Shared::ProgramName, tr(CANT_SAVE_FILE) );
					}
				}
				else {
					QMessageBox::critical( this, Shared::ProgramName, tr(CANT_READ_FILE) );
				}
			}
		}
	}
}
// end of F4

//*******************************************************************
// F6 - Rename                                          PRIVATE slot
//*******************************************************************
void ViewWindow::F6()
{
	const ViewTableItem* const item = d_table->current_item();

	if( item ) {
		if( FALSE == item->is_parent_dir() ) {
			bool ok;
			const QString current_name = item->name();
			const QString new_name = QInputDialog::getText(	tr(INDLG_F6_CAPTION),
												tr(INDLG_F6_MSG).arg( current_name ),
												QLineEdit::Normal,
												current_name, &ok,
												this );
			if( ok && ( FALSE == new_name.isEmpty())) {
				if( new_name != current_name ) {
					( d_is_ftp_dir )	? F6_ftp( current_name, new_name )
											: F6_lfs( current_name, new_name );
				}
			}
		}
	}
}
void ViewWindow::F6_ftp( const QString& in_current_name, const QString& in_new_name )
{
	d_ftp_cmd = FTP_CMD_RENAME;
	update_start();
	d_table->set_fname_on_start( in_new_name );
	d_ftp->rename( in_current_name, in_new_name );
}
void ViewWindow::F6_lfs( const QString& in_current_name, const QString& in_new_name )
{
	if( d_dir.rename( in_current_name, in_new_name, FALSE )) {
		refresh_dir();
		d_table->select_by_name( in_new_name );
	}
	else {
		QMessageBox::critical( this,
			tr(MSGBOX_F6_CAPTION),
			tr(MSGBOX_F6_MSG).arg( in_current_name ).arg( in_new_name ),
			QMessageBox::Ok,
			QMessageBox::NoButton );
	}
}
// end of F6

//*******************************************************************
// F7 - MkDir                                           PRIVATE slot
//*******************************************************************
void ViewWindow::F7()
{
	bool ok;
	const QString dname = QInputDialog::getText(
									tr(INDLG_F7_CAPTION), tr(INDLG_F7_MSG),
									QLineEdit::Normal, QString::null,
									&ok, this );
	
	if( ok && ( FALSE == dname.isEmpty()) ) {
		( d_is_ftp_dir ) ? F7_ftp( dname ) : F7_lfs( dname );	
	}
}
void ViewWindow::F7_ftp( const QString& in_new_dir_name )
{
	d_ftp_cmd = FTP_CMD_MKDIR;
	update_start();
	d_table->set_fname_on_start( in_new_dir_name );
	d_ftp->mkdir( in_new_dir_name );
}
void ViewWindow::F7_lfs( const QString& in_new_dir_name )
{
	if( d_dir.mkdir( in_new_dir_name, FALSE ) ) {
		refresh_dir();
		d_table->select_dir_by_name( in_new_dir_name );
	}
	else {
		QMessageBox::critical( this,
			tr(MSGBOX_F7_CAPTION),
			tr(MSGBOX_F7_MSG).arg( in_new_dir_name ),
			QMessageBox::Ok,
			QMessageBox::NoButton );
	}
}
// end of F7

//*******************************************************************
// F8 - Delete                                          PRIVATE slot
//*******************************************************************
void ViewWindow::F8()
{
    const ViewTable::SelectedItems& items = selections();

    if( FALSE == items.empty() ) {	
        DeleteQuestion* quest = new DeleteQuestion( this, items, FALSE == d_is_ftp_dir );
        if( quest ) {
            bool wipe = FALSE;
            bool ok = FALSE;
            if( QDialog::Accepted == quest->exec()) {
                wipe = quest->wipe();
                ok = TRUE;
            }
            delete quest;
            quest = 0;
            if( ok ) {
                PleaseWait* pw = new PleaseWait( this );
                connect( pw, SIGNAL( break_work() ), this, SLOT( break_work() ));
                pw->show();
                pw->caption( tr(DeletingCaption) );
                pw->message( tr(PleaseWaitMsg) );
                ( d_is_ftp_dir ) ? F8_ftp( items ) : F8_lfs( items, wipe, pw );
                disconnect( pw, SIGNAL( break_work() ), this, SLOT( break_work() ));
                delete pw;
            }
        }
	}
}
void ViewWindow::F8_ftp( const ViewTable::SelectedItems& in_items )
{
	// jeden plik na jeden raz
	const QString fname = in_items[0]->name();
	
	d_ftp_cmd = FTP_CMD_REMOVE;
	d_table->set_fname_on_start( fname ); // na wypadek gdyby nie wyszlo
	update_start();
	in_items[0]->is_dir() ? d_ftp->rmdir( fname ) : d_ftp->remove( fname );
}
void ViewWindow::F8_lfs( const ViewTable::SelectedItems& in_items, const bool in_wipe, QDialog* const in_dialog )
{
    Shared::idle();
	ViewTableItem* const item = in_items.back();
	ViewTableItem* next_selection = dynamic_cast<ViewTableItem*>( item->itemBelow() );
	if( 0 == next_selection ) {
		next_selection = dynamic_cast<ViewTableItem*>( item->itemAbove() );
	}
	const QString next_current_item = next_selection->name();

    Shared::d_break = FALSE;
	//....................................
    ViewTable::SelectedItems::const_iterator it = in_items.begin();
    while(( FALSE == Shared::d_break ) && ( it != in_items.end() )) {
        if( (*it)->is_dir() ) {
            if( FALSE == Shared::remove_path_and_content( (*it)->path(), FALSE, in_wipe, in_dialog )) {
                break;
            }
        }
        else {
            if( FALSE == Shared::remove_file( (*it)->path(), FALSE, in_wipe, in_dialog )) {
                break;
            }
        }
        Shared::idle();
		++it;
	}
	//....................................
	
	refresh_dir();
	d_table->select_by_name( next_current_item );
}
// end of F8

//*******************************************************************
// empty                                                PRIVATE slot
//*******************************************************************
void ViewWindow::empty()
{
	const ViewTableItem* const item = d_table->current_item();
	
	if( item ) {
		if( item->is_dir() ) {
			if( FALSE == item->is_parent_dir() ) {
				// FTP
				if( d_is_ftp_dir ) {
					// to do
				}
				else {
					const int retval =  QMessageBox::question( this,
													tr(MSGBOX_EMPTY_QUEST_CAPTION),
													tr(MSGBOX_EMPTY_QUEST_MSG).arg( item->path() ),
													Shared::YesBtnLabel, Shared::NoBtnLabel );
					if ( 0 == retval ) {
						d_loop = TRUE;
						Busy::set_busy( TRUE );
						QDir dir( item->path() );
						const QFileInfoList* const fil = dir.entryInfoList();
						if( fil && d_loop ) {
							QFileInfoListIterator it( *fil );
							while( it.current() && d_loop ) {
								if( Shared::is_regular_file( (*it)->fileName() )) {
									if( (*it)->isDir() ) Shared::remove_path_and_content( (*it)->absFilePath() );
									else                 Shared::remove_file( (*it)->absFilePath() );
								}
								++it;
							}
						}
						Busy::set_busy( FALSE );
						d_loop = FALSE;
					}
				}
			}
		}
	}
}
// end of empty

//*******************************************************************
// disp_dir_size                                        PRIVATE slot
//*******************************************************************
void ViewWindow::disp_dir_size( ViewTableItem* in_item )
{
	QDir dir( in_item->path() );
	
	d_loop = TRUE;
	Busy::set_busy( TRUE );
	const Q_ULLONG size = read_dir_size( dir.absPath() );
	Busy::set_busy( FALSE );
	
	if( d_loop ) {
		d_loop = FALSE;
		d_table->set_dir_size( in_item, size );
	}
}
// end of disp_dir_size

//*******************************************************************
// read_dir_size                                             PRIVATE
//*******************************************************************
Q_ULLONG ViewWindow::read_dir_size( const QString& in_dname )
{
	Q_ULLONG retval = 0;
	QDir dir( in_dname );
	
	Shared::idle();
	if( d_loop && dir.isReadable() ) {
		const QFileInfoList* const fil = dir.entryInfoList();
		if( fil ) {
			QFileInfoListIterator it( *fil );
			while( d_loop && it.current() ) {
				Shared::idle();
				if( (*it)->fileName() != Shared::ThisDir ) {
					if( (*it)->fileName() != Shared::ParentDir ) {
						if( (*it)->isDir() ) retval += read_dir_size( (*it)->absFilePath() );
						else                 retval += (*it)->size();
					}
				}
				++it;
			}
		}
	}
	return retval;
}
// end of read_dir_size


//###################################################################
//#                                                                 #
//#                            F T P                                #
//#                                                                 #
//###################################################################


//*******************************************************************
// slot_ftp_start                                       PRIVATE slot
//*******************************************************************
void ViewWindow::slot_ftp_start()
{
	int retval = QDialog::Rejected;

	d_fsi.clear();
	FtpServersList* dialog = new FtpServersList( this );
	if( dialog ) {
		retval = dialog->exec();
		if( QDialog::Rejected == retval ) {
			emit ftp_rejected();
		}
		else {
			d_fsi = dialog->get_fsi();
		}
		delete dialog;
		dialog = 0;
	}
	
	if( QDialog::Accepted == retval ) {
		d_table->get_current_fname( d_prv_item );
		d_prv_dir = get_dir();
		d_path->setEnabled( FALSE );
		d_fstab->setEnabled( FALSE );
		d_is_ftp_dir = TRUE;
		looks_refresh();
		
		d_status->show();
		ftp_connect();
	}
}
// end of slot_ftp_start

//*******************************************************************
// ftp_connect                                               PRIVATE
//*******************************************************************
void ViewWindow::ftp_connect()
{
	update_start();
	d_ftp_cmd = FTP_CMD_LOGIN;
	d_ftp->connectToHost( d_fsi.addr(), d_fsi.port() );
	d_ftp->login( d_fsi.user(), d_fsi.pass() );
}
// end of ftp_connect

//*******************************************************************
// ftp_cd                                                    PRIVATE
//*******************************************************************
void ViewWindow::ftp_cd( const QString& in_path, const bool in_cascade )
{
	if( !in_cascade )	update_start();
	d_ftp_cmd = FTP_CMD_CD;
	d_ftp->cd( in_path );
}
// end of ftp_cd

//*******************************************************************
// ftp_pwd                                                   PRIVATE
//*******************************************************************
void ViewWindow::ftp_pwd( const bool in_cascade )
{
	if( !in_cascade )	update_start();
	d_ftp_cmd = FTP_CMD_PWD;
	d_pwd_cmd_id = d_ftp->rawCommand( "PWD" );
}
// end of ftp_pwd

//*******************************************************************
// ftp_list                                                  PRIVATE
//*******************************************************************
void ViewWindow::ftp_list( const bool in_cascade )
{
	if( !in_cascade )	update_start();
	d_ftp_cmd = FTP_CMD_LIST;
	d_ftp->list();
}
// end of ftp_list

//*******************************************************************
// slot_ftp_end                                         PRIVATE slot
//*******************************************************************
void ViewWindow::slot_ftp_end()
{
	if( d_is_ftp_dir ) {
		d_ftp_cmd = FTP_CMD_CLOSE;
		d_ftp->close();

		d_status->hide();
		d_is_ftp_dir = FALSE;
		d_path->setEnabled( TRUE );
		d_fstab->setEnabled( TRUE );
		
		looks_refresh();
		d_dir.cd( d_prv_dir );
		refresh_dir();
		d_table->select_by_name( d_prv_item );
	}
}
// end of slot_ftp_end

//*******************************************************************
// ftp_list_info                                        PRIVATE slot
//*******************************************************************
void ViewWindow::ftp_list_info( const QUrlInfo& in_info )
{
	d_table->add_new_item( in_info, d_is_ftp_root );
}
// end of ftp_list_info

//*******************************************************************
// ftp_done                                             PRIVATE slot
//*******************************************************************
void ViewWindow::ftp_done( bool in_error )
{
	if( in_error ) {
		if( d_ftp_cmd != FTP_CMD_CLOSE ) {
			QMessageBox::critical( this, tr(FTP_MSG_CAPTION), d_ftp->errorString() );
			switch( d_ftp_cmd ) {
				case FTP_CMD_RENAME:
				case FTP_CMD_MKDIR:
				case FTP_CMD_REMOVE:
					ftp_pwd();
					break;
				default:
					emit ftp_rejected();
			}
		}
	}
	else {
		switch( d_ftp_cmd ) {
			case FTP_CMD_LOGIN:
				ftp_cd( d_fsi.dir() );
				break;
			case FTP_CMD_CLOSE:
				emit ftp_rejected();
				break;
			case FTP_CMD_CD:
				ftp_pwd();
				break;
			case FTP_CMD_PWD:
				if( d_timer.isActive() ) d_ftp_cmd = FTP_CMD_NONE;
				else                     ftp_list();
				break;
			case FTP_CMD_RENAME:
			case FTP_CMD_MKDIR:
			case FTP_CMD_REMOVE:
				ftp_list();
				break;
			default:
				update_finish();
				d_table->select_item_after_start();
				d_ftp_cmd = FTP_CMD_NONE;
		}
	}
}
// end of ftp_done

//*******************************************************************
// ftp_state_changed                                    PRIVATE slot
//*******************************************************************
void ViewWindow::ftp_state_changed( int in_state )
{
	switch( in_state ) {
		case QFtp::Unconnected:
			d_status->setText( tr(Shared::FTP_Unconnected) );
			break;
		case QFtp::HostLookup:
			d_status->setText( tr(Shared::FTP_HostLookup) );
			break;
		case QFtp::Connecting:
			d_status->setText( tr(Shared::FTP_Connecting) );
			break;
		case QFtp::Connected:
			d_status->setText( tr(Shared::FTP_Connected) );
			break;
		case QFtp::LoggedIn:
			d_status->setText( tr(Shared::FTP_LoggedIn) );
			break;
		case QFtp::Closing:
			d_status->setText( tr(Shared::FTP_Closing) );
			break;
		default:
			d_status->setText( tr(Shared::FTP_Unknown) );
	}
}
// end of ftp_state_changed

//*******************************************************************
// ftp_cmd_finish                                       PRIVATE slot
//*******************************************************************
void ViewWindow::ftp_cmd_finish( int in_id, bool )
{
	if( in_id == d_pwd_cmd_id ) {
		d_pwd_cmd_id = -1;
	}
}
// end of ftp_list_info

//*******************************************************************
// ftp_cmd_reply                                        PRIVATE slot
//*******************************************************************
void ViewWindow::ftp_cmd_reply( int, const QString& in_detail )
{
	if( FALSE == d_timer.isActive() ) {
		const char DELIMITER = '\"';
	
		if( FTP_CMD_PWD == d_ftp_cmd ) {
			const int idx_start = in_detail.find( DELIMITER );
			if( idx_start != -1 ) {
				const int idx_end = in_detail.find( DELIMITER, idx_start + 1 );
				if( idx_end != -1 ) {
					d_fsi.set_dir( in_detail.mid( idx_start + 1, idx_end - idx_start - 1 ));
					d_is_ftp_root = ( d_fsi.dir() == Shared::DirSep );
				}
			}
		}
	}
}
// end of ftp_cmd_reply

//*******************************************************************
// timoeut                                              PRIVATE slot
//*******************************************************************
void ViewWindow::timeout()
{
	ftp_pwd();
}
// end of timeout

//*******************************************************************
// tree                                                 PRIVATE slot
//*******************************************************************
void ViewWindow::tree()
{
	Tree* tree = new Tree( this, get_dir() );
	if( tree ) {
		QString selected_dir = QString::null;
		if( QDialog::Accepted == tree->exec() ) {
			selected_dir = tree->get_dir();
		}
		delete tree;
		tree = 0;
		
		if( selected_dir ) {
			d_dir.cd( selected_dir );
			refresh_dir();
		}
	}
}
// end of tree

//*******************************************************************
// md5_create                                           PRIVATE slot
//*******************************************************************
void ViewWindow::md5_create()
{
	if( d_is_ftp_dir ) {
		// to do
	}
	else {
		const ViewTable::SelectedItems& items = d_table->selections();
		if ( FALSE == items.empty() ) {
			ViewTable::SelectedItems::const_iterator it = items.begin();
			while( it != items.end() ) {
				const ViewTableItem* const item = *it;
				if( item ) {
					if( FALSE == item->is_dir() ) {
						const QString cmd = "md5sum " + item->path() + " > " + item->path() + ".md5sum";
						const int retval = system( cmd );
						if( -1 == retval ) {
						}
					}
				}
				++it;
			}
		}
		refresh_dir();
		d_table->select_by_name( items[0]->name()+".md5sum" );
	}
}
// end of md5_create

//*******************************************************************
// md5_check                                            PRIVATE slot
//*******************************************************************
void ViewWindow::md5_check()
{
	if( d_is_ftp_dir ) {
		// to do
	}
	else {
		const ViewTable::SelectedItems& items = d_table->selections();
		const QString cmd = "md5sum -c " + items[0]->path();
		const int retval = system( cmd );
		

		QString fname = items[0]->name();
		const int idx = fname.findRev( ".md5" );
		if( idx != -1 ) fname.truncate( idx );
		else fname = "???";
		
		QString msg = tr(MD5Msg).arg( fname );
		msg += ( 0 == retval ) ? tr(MD5Ok) : tr(MD5Error);
		QMessageBox::information( 0, tr(MD5Caption), msg );
	}
}
// end of md5_check
