package buoy.widget;

import buoy.internal.*;
import buoy.xml.*;
import buoy.xml.delegate.*;
import java.awt.*;
import java.util.*;
import javax.swing.JPanel;

/**
 * ExplicitContainer is a WidgetContainer which allows the sizes and positions of its children to be
 * set explicitly.
 * <p>
 * In addition to the event types generated by all Widgets, ExplicitContainers generate the following
 * event types:
 * <ul>
 * <li>{@link buoy.event.RepaintEvent RepaintEvent}</li>
 * </ul>
 *
 * @author Peter Eastman
 */

public class ExplicitContainer extends WidgetContainer
{
  private ArrayList children;
  private ArrayList childBounds;
  private Dimension requiredSize;
  
  static
  {
    WidgetEncoder.setPersistenceDelegate(ExplicitContainer.class, new IndexedContainerDelegate(new String [] {"getChild", "getChildBounds"}));
  }

  /**
   * Create a new ExplicitContainer.
   */
  
  public ExplicitContainer()
  {
    component = new WidgetContainerPanel(this);
    children = new ArrayList();
    childBounds = new ArrayList();
    requiredSize = new Dimension(0, 0);
  }
  
  /**
   * Get the number of children in this container.
   */
  
  public int getChildCount()
  {
    return children.size();
  }
  
  /**
   * Get the i'th child of this container.
   */
  
  public Widget getChild(int i)
  {
    return (Widget) children.get(i);
  }
  
  /**
   * Get a Collection containing all child Widgets of this container.
   */
  
  public Collection getChildren()
  {
    return new ArrayList(children);
  }
  
  /**
   * Layout the child Widgets.  This may be invoked whenever something has changed (the size of this
   * WidgetContainer, the preferred size of one of its children, etc.) that causes the layout to no
   * longer be correct.  If a child is itself a WidgetContainer, its layoutChildren() method will be
   * called in turn.
   */
  
  public void layoutChildren()
  {
    for (int i = 0; i < children.size(); i++)
    {
      Widget child = (Widget) children.get(i);
      Rectangle bounds = (Rectangle) childBounds.get(i);
      child.getComponent().setBounds(bounds);
      if (child instanceof WidgetContainer)
        ((WidgetContainer) child).layoutChildren();
    }
  }
  
  /**
   * Add a Widget to this container.
   *
   * @param widget     the Widget to add
   * @param bounds     the location and size at which the Widget should appear
   */
  
  public void add(Widget widget, Rectangle bounds)
  {
    if (widget.getParent() != null)
      widget.getParent().remove(widget);
    children.add(widget);
    childBounds.add(bounds.clone());
    ((JPanel) component).add(widget.component);
    widget.component.setBounds(bounds);
    setAsParent(widget);
    requiredSize.width = Math.max(requiredSize.width, bounds.x+bounds.width);
    requiredSize.height = Math.max(requiredSize.height, bounds.y+bounds.height);
    invalidateSize();
  }

  /**
   * Get the index of a particular Widget.
   *
   * @param widget      the Widget to locate
   * @return the index of the Widget within this container
   */
  
  public int getChildIndex(Widget widget)
  {
    return children.indexOf(widget);
  }
  
  /**
   * Get the position of a child Widget within this container.
   *
   * @param index     the index of the Widget for which to get the location
   * @return the bounding rectangle to be used for the Widget
   */
  
  public Rectangle getChildBounds(int index)
  {
    return (Rectangle) ((Rectangle) childBounds.get(index)).clone();
  }
  
  /**
   * Set the position of a child Widget within this container.
   *
   * @param index      the index of the Widget to move
   * @param bounds     the location and size at which the Widget should appear
   */
  
  public void setChildBounds(int index, Rectangle bounds)
  {
    childBounds.set(index, bounds.clone());
    findRequiredSize();
  }
  
  /**
   * Get the position of a child Widget within this container.
   *
   * @param widget     the Widget for which to get the location
   * @return the bounding rectangle to be used for the Widget, or null if it is not a child
   * of this container
   */
  
  public Rectangle getChildBounds(Widget widget)
  {
    int index = children.indexOf(widget);
    if (index > -1)
      return (Rectangle) ((Rectangle) childBounds.get(index)).clone();
    return null;
  }
  
  /**
   * Set the position of a child Widget within this container.
   *
   * @param widget     the Widget to move
   * @param bounds     the location and size at which the Widget should appear
   */
  
  public void setChildBounds(Widget widget, Rectangle bounds)
  {
    int index = children.indexOf(widget);
    if (index > -1)
    {
      childBounds.set(index, bounds.clone());
      findRequiredSize();
    }
  }
  
  /**
   * Remove a child Widget from this container.
   *
   * @param widget     the Widget to remove
   */
  
  public void remove(Widget widget)
  {
    int index = children.indexOf(widget);
    if (index > -1)
    {
      ((JPanel) component).remove(widget.component);
      children.remove(index);
      childBounds.remove(index);
      removeAsParent(widget);
      findRequiredSize();
    }
  }
  
  /**
   * Remove all child Widgets from this container.
   */
  
  public void removeAll()
  {
    ((JPanel) component).removeAll();
    for (int i = 0; i < children.size(); i++)
      removeAsParent((Widget) children.get(i));
    children.clear();
    childBounds.clear();
    requiredSize.width = requiredSize.height = 0;
    invalidateSize();
  }
  
  /**
   * Calculate how large this Widget needs to be to hold all of its children.
   */
  
  private void findRequiredSize()
  {
    requiredSize.width = requiredSize.height = 0;
    for (int i = childBounds.size()-1; i >= 0; i--)
    {
      Rectangle r = (Rectangle) childBounds.get(i);
      requiredSize.width = Math.max(requiredSize.width, r.x+r.width);
      requiredSize.height = Math.max(requiredSize.height, r.y+r.height);
    }
    invalidateSize();
  }
  
  /**
   * Get the smallest size at which this Widget can reasonably be drawn.  When a WidgetContainer lays out
   * its contents, it will attempt never to make this Widget smaller than its minimum size.
   */
  
  public Dimension getMinimumSize()
  {
    return requiredSize;
  }

  /**
   * Get the preferred size at which this Widget will look best.  When a WidgetContainer lays out
   * its contents, it will attempt to make this Widget as close as possible to its preferred size.
   */
  
  public Dimension getPreferredSize()
  {
    return requiredSize;
  }
}