package buoy.widget;

import buoy.event.*;
import java.awt.*;
import javax.swing.*;
import javax.swing.event.*;

/**
 * A BColorChooser is a Widget that allows the user to select a color.  It has tabs which provide
 * various ways of choosing the color, such as selecting one from a palette or specifying values
 * for hue, saturation, and brightness.
 * <p>
 * BColorChooser can be used in two different ways.  First, it can be added to a container like
 * any other Widget.  This is useful when you want it to appear inside of a window along with
 * other Widgets.
 * <p>
 * Most often, however, BColorChooser is used in a modal dialog as a self contained user interface
 * element.  To use it this way, you simply instantiate a BColorChooser, set any properties, and then
 * call {@link buoy.widget.BColorChooser#showDialog showDialog()} to display it.  <code>showDialog()</code> will
 * automatically create a dialog, add the BColorChooser, display it, and block until the user
 * dismisses the dialog.  You can reuse a single BColorChooser by repeatedly calling <code>showDialog()</code>.
 * <p>
 * In addition to the event types generated by all Widgets, BColorChoosers generate the following event types:
 * <ul>
 * <li>{@link buoy.event.ValueChangedEvent ValueChangedEvent}</li>
 * </ul>
 *
 * @author Peter Eastman
 */

public class BColorChooser extends Widget
{
  private String dlgTitle;
  
  /**
   * Create a new BColorChooser whose initial color is white.
   */
  
  public BColorChooser()
  {
    this(Color.WHITE, "");
  }
  
  /**
   * Create a new BColorChooser
   *
   * @param color     the color which is initially selected
   * @param title     the title to display on the dialog
   */
  
  public BColorChooser(Color color, String title)
  {
    component = createComponent();
    setColor(color);
    setTitle(title);
    getComponent().getSelectionModel().addChangeListener(new ChangeListener() {
      public void stateChanged(ChangeEvent ev)
      {
        dispatchEvent(new ValueChangedEvent(BColorChooser.this));
      }
    });
  }

  /**
   * Create the JColorChooser which serves as this Widget's Component.  This method is protected so that
   * subclasses can override it.
   */
  
  protected JColorChooser createComponent()
  {
    return new JColorChooser();
  }

  public JColorChooser getComponent()
  {
    return (JColorChooser) component;
  }

  /**
   * Get the title displayed on the dialog.
   */
  
  public String getTitle()
  {
    return dlgTitle;
  }
  
  /**
   * Set the title displayed on the dialog.
   */
  
  public void setTitle(String title)
  {
    dlgTitle = title;
  }
  
  /**
   * Get the color selected in the BColorChooser.
   */
  
  public Color getColor()
  {
    return getComponent().getColor();
  }
  
  /**
   * Set the color selected in the BColorChooser.
   */
  
  public void setColor(Color color)
  {
    getComponent().setColor(color);
  }
  
  /**
   * Show a dialog containing this BColorChooser and block until the user closes it.  After this method
   * returns, you can call <code>getColor()</code> to determine what color was selected.
   *
   * @param parent    the dialog's parent Widget (usually a WindowWidget).  This may be null.
   * @return true if the user clicked the OK button, false if they clicked the cancel button
   */
  
  public boolean showDialog(Widget parent)
  {
    Color color = JColorChooser.showDialog(parent.getComponent(), dlgTitle, getColor());
    if (color == null)
      return false;
    setColor(color);
    return true;
  }
}
