# Copyright (C) 2006 Canonical Ltd
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""A plugin for converting from CVS to bzr using a cvsps changeset listing.

This program directly accesses the RCS files using 'co' to avoid the overhead
of cvs.
"""

from bzrlib import (
    branch,
    commands,
    errors,
    option,
    osutils,
    )


class cmd_cvsps_import(commands.Command):
    """Import from CVS into bzr.

    Convert the MODULE into a target at OUTPUT, extracting the data from
    CVSROOT.

    OUTPUT          Information for the conversion will be stored here
      bzr           Contains the converted bzr branches.
        MODULE      This will be a shared repository
          branches  All branches will be collected here
            HEAD    The CVS HEAD branch
          tags      Any CVS tags will be generated here
      staging       Stores the cvsps-import working files.
        MODULE.dump If not supplying --cvsps-dump, one will be created here
        MODULE.map  A mapping from cvsps revisions => bzr revisions
		MODULE.fmap A mapping from cvs path => bzr file_id to allow cross
					branches to use the same file id.

    When resuming, the default is to check that each revision id is present in
    the branch. This is a little bit slow, because it has to open up every
    branch. Passing --no-verify will change this to just assume they have been
    properly converted.
    """

    takes_args = ['cvsroot', 'module', 'output']
    takes_options = [option.Option('encoding', type=str,
                                   help='Set the encoding for cvs author names'
                                        ' and commit messages.'),
                     option.Option('cvsps-dump', type=unicode, argname='PATH',
                                   help='Path to an existing cvsps dump.'
                                        ' This will be used rather than'
                                        ' generating a new one.'),
                     option.Option('only-branches', type=str, argname='REGEX',
                                   help='Only convert branches matching the'
                                        ' specific regex.'),
                     option.Option('verify',
                                   help='Verify that processed versions exist'
                                        ' in the branch they claim to be in.'),
                     option.Option('use-cvs',
                                   help='Use cvs to extract texts.'),
                     option.Option('use-rcs',
                                   help='Use rcs to extract texts. (default)'),
                     option.Option('tags-as-branches',
                                   help='Import CVS tags as Bazaar branches'),
                    ]

    def run(self, cvsroot=None, module=None, output=None, cvsps_dump=None,
            encoding=None, verify=True, use_cvs=False, use_rcs=False,
            only_branches=None, tags_as_branches=False):
        from cvsps import importer

        if cvsroot.startswith(':pserver:') or cvsroot.startswith(':ext:'):
            raise errors.BzrCommandError('cvsps-import can only operate on a'
                     ' local copy of the cvs repository. :pserver: and :ext:'
                     ' protocols are not supported (yet)')

        if not use_cvs and not use_rcs:
            # Default is to use rcs, since it is slightly faster.
            use_cvs = False
        if tags_as_branches:
            tag_style = 'branch'
        else:
            tag_style = 'tag'
        importer = importer.Importer(cvsroot, module, output,
                                     cvsps_dump=cvsps_dump,
                                     encoding=encoding,
                                     verify=verify,
                                     use_cvs_for_text=use_cvs,
                                     only_branches=only_branches,
                                     tag_style=tag_style)
        importer.process()


class cmd_cvsps_upgrade_conversion(commands.Command):
    """Convert the old layout of branches to the new one.

    Originally, cvsps conversions were done with:

      out/
        staging/
            MODULE.dump
            MODULE.map
        bzr_repo/
            .bzr/repository
            MODULE.BRANCH1/
                .bzr/
                    branch/
                    checkout/
            MODULE.BRANCH2/
                .bzr/
                    branch/
                    checkout/

    Basically, a globally shared repository, and each branch had its own
    working tree. For large conversions, this uses a lot of disk space. So
    instead we switched to:

      out/
        staging/
          MODULE.dump
          MODULE.map
		  MODULE.fmap
        bzr/
          MODULE/
            .bzr/repository
            branches/
              BRANCH1/
                .bzr/branch
              BRANCH2/
                .bzr/branch

    And the working tree is kept only in memory.
    """

    takes_args = ['output']

    def run(self, output):
        from cvsps.upgrade_conversion import ConversionUpgrader

        output = osutils.abspath(output)

        upgrader = ConversionUpgrader(output)
        upgrader.upgrade()


commands.register_command(cmd_cvsps_import)
commands.register_command(cmd_cvsps_upgrade_conversion)


def test_suite():
    from bzrlib import tests
    import test_cvsps_import
    import test_cvsps_parser
    import test_lru_cache

    suite = tests.TestSuite()
    loader = tests.TestLoader()
    suite.addTests(loader.loadTestsFromModule(test_cvsps_import))
    suite.addTests(loader.loadTestsFromModule(test_cvsps_parser))
    suite.addTests(loader.loadTestsFromModule(test_lru_cache))
    return suite
