/*--------------------------------------------------------------------*/
/*--- Callgrind                                                    ---*/
/*---                                                       data.c ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Callgrind, a Valgrind tool for call tracing.

   Copyright (C) 2002-2004, Josef Weidendorfer (Josef.Weidendorfer@gmx.de)

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

#include "global.h"

/*------------------------------------------------------------*/
/*--- Tracing Data Allocations                             ---*/
/*------------------------------------------------------------*/

typedef struct _SubEntry   SubEntry;
typedef struct _DataType   DataType;
typedef struct _DataObject DataObject;
typedef struct _TypeList   TypeList;
typedef struct _ObjectList ObjectList;

struct _SubEntry {
  DataType* type;
  UInt offset;
  UInt size;    /* Array if greater than type->size */
};

/* How a type was deduced:
 * - CLASS    : by trapped a C++ constructor call, has class name
 * - MALLOCED : allocated chunk of memory. name is call context
 * - STATIC   : static variable, with type from debug info
 * - STACK    : local variable, with type (call context, debug info)
 */
#define TYPE_UNKNOWN   0
#define TYPE_CLASS     1  
#define TYPE_MALLOCED  2
#define TYPE_STATIC    3
#define TYPE_STACK     4

struct _DataType {
  UInt number;
  
  ULong* counters; /* allocated lazy */

  UInt size;
  UInt subentries, capacity;
  SubEntry e[1];
};

typedef struct _TablePos TablePos;
struct _TablePos {
  UInt addr;
  UInt pos1, pos2, pos3;
  UInt *table2, *table3;
};

struct _DataObject {
  DataType* type;
  Char* name;
  UInt addr, size;

  ULong* cost; /* allocated lazy */
};

/* 2 lower bits of each table entry encode the type.
 * The pointers itself have to be 4-byte aligned.
 */
#define ENTRY_TYPE_MASK 3
#define ENTRY_INVALID   0
#define ENTRY_OBJECT    1
#define ENTRY_SUBTABLE  2

/* 3-level page table for mapping address->data structure
 * For now, only 32-bit addresses, split 12:10:10
 */
#define TABLE_LEVEL1_SIZE  4096
#define TABLE_LEVEL1_BITS  12
#define TABLE_LEVEL1_SHIFT 20

#define TABLE_LEVEL2_SIZE  1024
#define TABLE_LEVEL2_BITS  10
#define TABLE_LEVEL2_SHIFT 10

#define TABLE_LEVEL3_SIZE 1024


struct _TypeList {
  int entries, capacity;
  DataType* t[1];
};

struct _ObjectList {
  int entries, capacity;
  DataObject* o[1];
};

#define INITIAL_OBJECTLIST_SIZE 1024
#define INITIAL_TYPELIST_SIZE 1024


UInt  data_table[TABLE_LEVEL1_SIZE];
TypeList*   types = 0;
ObjectList* objects = 0;

void init_data()
{
  Int i;

  for(i=0;i<TABLE_LEVEL1_SIZE;i++)
    data_table[i] = 0;

  types = (TypeList*) VG_(malloc)(sizeof(TypeList) +
				  INITIAL_TYPELIST_SIZE *
				  sizeof(DataType*));
  types->capacity = INITIAL_TYPELIST_SIZE;
  types->entries = 0;
  for(i=0;i<INITIAL_TYPELIST_SIZE;i++)
    types->t[i] = 0;

  objects = (ObjectList*) VG_(malloc)(sizeof(ObjectList) +
				      INITIAL_OBJECTLIST_SIZE *
				      sizeof(DataObject*));
  objects->capacity = INITIAL_OBJECTLIST_SIZE;
  objects->entries = 0;
  for(i=0;i<INITIAL_OBJECTLIST_SIZE;i++)
    objects->o[i] = 0;
}

static
DataType* get_type(const char* s)
{
  return 0;
}

static
UInt* new_l2table()
{
  Int i;
  UInt* t = (UInt*) VG_(malloc)(sizeof(UInt)*TABLE_LEVEL2_SIZE);
  CT_ASSERT(( ((UInt)t)&3) == 0);

  for(i=0;i<TABLE_LEVEL2_SIZE;i++)
    t[i] = ENTRY_INVALID;
  
  return t;
}

static
UInt* new_l3table()
{
  Int i;
  UInt* t = (UInt*) VG_(malloc)(sizeof(UInt)*TABLE_LEVEL3_SIZE);
  CT_ASSERT( (((UInt)t)&3) == 0);

  for(i=0;i<TABLE_LEVEL3_SIZE;i++)
    t[i] = ENTRY_INVALID;
  
  return t;
}

static
DataObject* new_object(char* name, UInt addr, UInt size)
{
  DataObject* o = (DataObject*) VG_(malloc)(sizeof(DataObject));

  o->type  = 0;
  o->name  = VG_(strdup)(name);
  o->addr  = addr;
  o->size  = size;
  o->cost  = 0;

  return o;
}

static UInt tablePosAddr(UInt pos1, UInt pos2, UInt pos3)
{
  return (pos1 << TABLE_LEVEL1_SHIFT) |
    (pos2 << TABLE_LEVEL2_SHIFT) | pos3;
}
    

/* Set a table start position from an address
 * with minimum depth. Doesn't set subtables.
 *
 * Returns depth.
 */
static int minTableStartPos(UInt addr, TablePos* tp)
{
  int d;

  tp->addr = addr;
  tp->pos1 = addr >> TABLE_LEVEL1_SHIFT;
  tp->pos2 = (addr >> TABLE_LEVEL2_SHIFT) & (TABLE_LEVEL2_SIZE-1);
  tp->pos3 = addr & (TABLE_LEVEL3_SIZE-1);
  
  d =
    (tp->pos3 > 0) ? 3 :
    (tp->pos2 > 0) ? 2 : 
    (tp->pos1 > 0) ? 1 : 0;

  CT_DEBUG(5," minTableStartPos(addr %p): pos %d/%d/%d, depth %d\n",
	   addr, tp->pos1, tp->pos2, tp->pos3, d);

  return d;
}

static int minTableEndPos(UInt addr, TablePos* tp)
{
  int d;

  tp->addr = addr;
  tp->pos1 = addr >> TABLE_LEVEL1_SHIFT;
  tp->pos2 = (addr >> TABLE_LEVEL2_SHIFT) & (TABLE_LEVEL2_SIZE-1);
  tp->pos3 = addr & (TABLE_LEVEL3_SIZE-1);

  d =
    (tp->pos3 < TABLE_LEVEL3_SIZE-1) ? 3 :
    (tp->pos2 < TABLE_LEVEL2_SIZE-1) ? 2 : 
    (tp->pos1 < TABLE_LEVEL1_SIZE-1) ? 1 : 0;

  CT_DEBUG(5," minTableEndPos(addr %p): pos %d/%d/%d, depth %d\n",
	   addr, tp->pos1, tp->pos2, tp->pos3, d);

  return d;
}

/* Attaches subtables to a table position
 * up to the requested depth, creating subtables as
 * needed.
 * Returns the depth where attaching was possible.
 */
static int attachSubtables(int depth, TablePos* tp)
{
  UInt entry1, entry2, *table2, *table3;

  CT_DEBUG(5," attachSubtables(depth %d, pos %d/%d/%d)\n",
	   depth, tp->pos1, tp->pos2, tp->pos3);

  entry1 = data_table[tp->pos1];
  if ((entry1 & ENTRY_TYPE_MASK) == ENTRY_OBJECT) {
    tp->table2 = tp->table3 = 0;

    CT_DEBUG(5,"  => 0 [Object at L1]\n");
    return 0;
  }
  if ((entry1 & ENTRY_TYPE_MASK) == ENTRY_INVALID) {
    if (depth <= 0) {
      tp->table2 = tp->table3 = 0;

      CT_DEBUG(5,"  => 0\n");
      return 0;
    }
    table2 = new_l2table();
    entry1 = (UInt)table2 | ENTRY_SUBTABLE;
    data_table[tp->pos1] = entry1;

    CT_DEBUG(5,"   new L2 subtable at %d: %p\n",
	     tp->pos1, table2, entry1);
  }
  else {
    table2 = (UInt*)(entry1 & ~3);
    CT_DEBUG(5,"   L2 subtable at %d: %p\n", tp->pos1, table2);
  }
  tp->table2 = table2;

  entry2 = table2[tp->pos2];
  if ((entry2 & ENTRY_TYPE_MASK) == ENTRY_OBJECT) {
    tp->table3 = 0;

    CT_DEBUG(5,"  => 1 [Object at L2]\n");
    return 1;
  }
  if ((entry2 & ENTRY_TYPE_MASK) == ENTRY_INVALID) {
    if (depth == 1) {
      tp->table3 = 0;

      CT_DEBUG(5,"  => 1\n");
      return 1;
    }
    table3 = new_l3table();
    entry2 = (UInt)table3 | ENTRY_SUBTABLE;
    table2[tp->pos2] = entry2;
    CT_DEBUG(5,"   new L3 subtable at %d: %p\n",
	     tp->pos2, table3, entry2);
  }
  else {
    table3 = (UInt*)(entry2 & ~3);
    CT_DEBUG(5,"   L3 subtable at %d: %p\n", tp->pos2, table3);
  }
  tp->table3 = table3;

  return depth;
}

static
void compactTable2(UInt* table, int idx)
{ 
  int i;
  UInt* subtable;
  Bool isEmpty = True;
 
  if ((table[idx] & ENTRY_TYPE_MASK) != ENTRY_SUBTABLE) return;
  subtable = (UInt*)(table[idx] & ~3);
  for(i=0;i<TABLE_LEVEL3_SIZE;i++) {
    if ((subtable[i] & ENTRY_TYPE_MASK) != ENTRY_INVALID)
      isEmpty = False;
  }
  if (!isEmpty) return;

  table[idx] = ENTRY_INVALID;
  VG_(free)(subtable);

  CT_DEBUG(5,"   compactTable2(./%d): deleted subtable\n", idx);
}

static
void compactTable1(UInt* table, int idx)
{ 
  int i;
  UInt* subtable;
  Bool isEmpty = True;
 
  if ((table[idx] & ENTRY_TYPE_MASK) != ENTRY_SUBTABLE) return;
  subtable = (UInt*)(table[idx] & ~3);
  for(i=0;i<TABLE_LEVEL2_SIZE;i++) {
    compactTable2(subtable,i);
    if ((subtable[i] & ENTRY_TYPE_MASK) != ENTRY_INVALID)
      isEmpty = False;
  }
  if (!isEmpty) return;

  table[idx] = ENTRY_INVALID;
  VG_(free)(subtable);

  CT_DEBUG(5,"  compactTable1(%d): deleted subtable\n", idx);
}
    

static
Bool hasOnlyEntry(UInt entry, TablePos* start, TablePos* end)
{
  int i;
  int start_pos1, end_pos1;
  int start_pos2, end_pos2;

  CT_DEBUG(5," hasOnlyEntry(%p, %d/%d/%d - %d/%d/%d)\n",
	   entry,
	   start->pos1, start->pos2, start->pos3,
	   end->pos1, end->pos2, end->pos3);

  if (start->pos1 == end->pos1) {
    if (start->pos2 == end->pos2) {
      CT_DEBUG(5,"  Checking (%d/%d/%d-%d)\n",
	       start->pos1, start->pos2, start->pos3, end->pos3);
      for(i = start->pos3; i<=end->pos3; i++)
	if (start->table3[i] != entry) {
	  CT_DEBUG(5,"  => No (at %d)\n", i);
	  return False;
	}
    }
    else {
      start_pos2 = start->pos2, end_pos2 = end->pos2;
      if (start->table3) {
	CT_DEBUG(5,"  Checking (%d/%d/%d-%d)\n",
		 start->pos1, start->pos2, start->pos3, TABLE_LEVEL3_SIZE-1);
	start_pos2++;
	for(i = start->pos3; i<TABLE_LEVEL3_SIZE; i++)
	  if (start->table3[i] != entry) {
	    CT_DEBUG(5,"  => No (at %d)\n", i);
	    return False;
	  }
      }
      if (end->table3) {
	CT_DEBUG(5,"  Checking (%d/%d/%d-%d)\n",
		 start->pos1, end->pos2, 0, end->pos3);
	end_pos2--;
	for(i = 0; i<= end->pos3; i++)
	  if (end->table3[i] != entry) {
	    CT_DEBUG(5,"  => No (at %d)\n", i);
	    return False;
	  }
      }
      if (start_pos2 <= end_pos2)
	CT_DEBUG(5,"  Checking (%d/%d-%d)\n", start->pos1, start_pos2, end_pos2);
      for(i = start_pos2; i<=end_pos2; i++) {
	compactTable2(start->table2,i);
	if (start->table2[i] != entry) {
	  CT_DEBUG(5,"  => No (at %d)\n", i);
	  return False;
	}
      }
    }
    CT_DEBUG(5,"  => Yes\n");
    return True;
  }

  start_pos1 = start->pos1, end_pos1 = end->pos1;
  if (start->table2) {
    start_pos1++;
    start_pos2 = start->pos2;
    if (start->table3) {
      CT_DEBUG(5,"  Checking (%d/%d/%d-%d)\n", start->pos1, start->pos2, start->pos3,
	       TABLE_LEVEL3_SIZE-1);
      start_pos2++;      
      for(i = start->pos3; i<TABLE_LEVEL3_SIZE; i++)
	if (start->table3[i] != entry) {
	  CT_DEBUG(5,"  => No (at %d)\n", i);
	  return False;
	}
    }
    CT_DEBUG(5,"  Checking (%d/%d-%d)\n", start->pos1, start_pos2,
	     TABLE_LEVEL2_SIZE-1);
    for(i = start_pos2; i<TABLE_LEVEL2_SIZE; i++) {
      compactTable2(start->table2,i);
      if (start->table2[i] != entry) {
	CT_DEBUG(5,"  => No (at %d)\n", i);
	return False;
      }
    }
  }
  if (end->table2) {
    end_pos1++;
    end_pos2 = end->pos2;
    if (end->table3) {
      CT_DEBUG(5,"  Checking (%d/%d/%d-%d)\n", end->pos1, end->pos2, 0, end->pos3);
      end_pos2--;
      for(i = 0; i<= end->pos3; i++)
	if (end->table3[i] != entry) {
	  CT_DEBUG(5,"  => No (at %d)\n", i);
	  return False;
	}
    }
    CT_DEBUG(5,"  Checking (%d/%d-%d)\n", end->pos1, 0, end_pos2);
    for(i = 0; i<=end_pos2; i++) {
      compactTable2(end->table2,i);
      if (end->table2[i] != entry) {
	CT_DEBUG(5,"  => No (at %d)\n", i);
	return False;
      }
    }
  }
  if (start_pos1 <= end_pos1)
    CT_DEBUG(5,"  Checking (%d-%d)\n", start_pos1, end_pos1);
  for(i = start_pos1; i<=end_pos1; i++) {
    compactTable1(data_table,i);
    if (data_table[i] != entry) {
      CT_DEBUG(5,"  => No (at %d)\n", i);
      return False;
    }
  }
  CT_DEBUG(5,"  => Yes\n");
  return True;
}

static
void setEntry(UInt entry, TablePos* start, TablePos* end)
{
  int i;
  int start_pos1, end_pos1;
  int start_pos2, end_pos2;

  CT_DEBUG(5," setEntry(%p, %d/%d/%d - %d/%d/%d)\n",
	   entry,
	   start->pos1, start->pos2, start->pos3,
	   end->pos1, end->pos2, end->pos3);

  if (start->pos1 == end->pos1) {
    if (start->pos2 == end->pos2) {
      for(i = start->pos3; i<=end->pos3; i++)
	start->table3[i] = entry;
    }
    else {
      start_pos2 = start->pos2, end_pos2 = end->pos2;
      if (start->table3) {
	start_pos2++;
	for(i = start->pos3; i<TABLE_LEVEL3_SIZE; i++)
	  start->table3[i] = entry;
      }
      if (end->table3) {
	end_pos2--;
	for(i = 0; i<= end->pos3; i++)
	  end->table3[i] = entry;
      }
      for(i = start_pos2; i<=end_pos2; i++)
	start->table2[i] = entry;
    }
    return;
  }

  start_pos1 = start->pos1, end_pos1 = end->pos1;
  if (start->table2) {
    start_pos1++;
    start_pos2 = start->pos2;
    if (start->table3) {
      start_pos2++;      
      for(i = start->pos3; i<TABLE_LEVEL3_SIZE; i++)
	start->table3[i] = entry;
    }
    for(i = start_pos2; i<TABLE_LEVEL2_SIZE; i++)
      start->table2[i] = entry;
  }
  if (end->table2) {
    end_pos1++;
    end_pos2 = end->pos2;
    if (end->table3) {
      end_pos2--;
      for(i = 0; i<= end->pos3; i++)
	end->table3[i] = entry;
    }
    for(i = 0; i<=end_pos2; i++)
      end->table2[i] = entry;
  }
  for(i = start_pos1; i<=end_pos1; i++)
    data_table[i] = entry;
}

/* Sets start and end according to addr and size */
Bool setRange(UInt addr, UInt size, TablePos* start, TablePos* end)
{
  int startDepth, endDepth, d;

  CT_DEBUG(5," setRange(addr %p, size %u)\n", addr, size);

  startDepth = minTableStartPos(addr, start);
  endDepth   = minTableEndPos(addr+size-1, end);

  /* if position path goes over same subtables, depths have to
   * be adjusted */
  if (start->pos1 == end->pos1) {
    if (start->pos2 == end->pos2) {
      if (startDepth<3) startDepth=3;
      if (endDepth<3) endDepth=3;
    }
    else {
      if (startDepth<2) startDepth=2;
      if (endDepth<2) endDepth=2;
    }
  }
  
  CT_DEBUG(5,"  > adjusted: depth %d - %d\n", startDepth, endDepth);

  /* attach subtables, creating if needed.
   * return 0 if conflicts exist */
  d = attachSubtables(startDepth, start);
  if (d<startDepth) return False;
  d = attachSubtables(endDepth, end);
  if (d<endDepth) return False;

  return True;
}


static
void printTable()
{
  int pos1, pos2, pos3;
  UInt entry1, entry2, entry3, *table2, *table3;
  DataObject* o;
  UInt last1, last2, last3;

  CT_DEBUG(3," printTable:\n");

  last1 = 0;
  for(pos1=0; pos1<TABLE_LEVEL1_SIZE; pos1++) {
    entry1 = data_table[pos1];

    if ((entry1 == last1) &&
	(entry1 & ENTRY_TYPE_MASK) != ENTRY_SUBTABLE) continue;
    if (last1 != 0)
      CT_DEBUG(3,"  ... - (%d)\n", pos1-1);
    last1 = entry1;

    if ((entry1 & ENTRY_TYPE_MASK) == ENTRY_OBJECT) {
      o = (DataObject*)(entry1 & ~3);
      CT_DEBUG(3,"  at L1 (%d): (%p/%u) %s\n",
	       pos1, o->addr, o->size, o->name); 
      continue;
    }
    if ((entry1 & ENTRY_TYPE_MASK) == ENTRY_INVALID) {
      CT_DEBUG(3,"  at L1 (%d): None\n", pos1);
      continue;
    }
    last1 = 0;

    table2 = (UInt*)(entry1 & ~3);
    CT_DEBUG(3,"  subtable (%p) at L1 %d (%p-%p)\n",
	     table2, pos1,
	     tablePosAddr(pos1,0,0),
	     tablePosAddr(pos1,TABLE_LEVEL2_SIZE-1, TABLE_LEVEL3_SIZE-1));

    last2 = 0;
    for(pos2=0; pos2<TABLE_LEVEL2_SIZE; pos2++) {
      entry2 = table2[pos2];

      if ((entry2 == last2) &&
	  (entry2 & ENTRY_TYPE_MASK) != ENTRY_SUBTABLE) continue;
      if (last2 != 0)
	CT_DEBUG(3,"   ... - (%d/%d)\n", pos1, pos2-1);
      last2 = entry2;

      if ((entry2 & ENTRY_TYPE_MASK) == ENTRY_OBJECT) {
	o = (DataObject*)(entry2 & ~3);
	CT_DEBUG(3,"   at L2 (%d/%d): (%p/%u) %s\n",
		 pos1, pos2, o->addr, o->size, o->name); 
	continue;
      }
      if ((entry2 & ENTRY_TYPE_MASK) == ENTRY_INVALID) {
	CT_DEBUG(3,"   at L2 (%d/%d): None\n", pos1, pos2);
	continue;
      }
      last2 = 0;

      table3 = (UInt*)(entry2 & ~3);
      CT_DEBUG(3,"   subtable (%p) at L2 %d/%d (%p-%p)\n", 
	       table3, pos1, pos2,
	       tablePosAddr(pos1,pos2,0),
	       tablePosAddr(pos1,pos2, TABLE_LEVEL3_SIZE-1));

      last3 = 0;
      for(pos3=0; pos3<TABLE_LEVEL3_SIZE; pos3++) {
	entry3 = table3[pos3];

	if (entry3 == last3) continue;
	if (last3 != 0)
	  CT_DEBUG(3,"    ... - (%d/%d/%d)\n", pos1, pos2, pos3-1);
	last3 = entry3;

	if ((entry3 & ENTRY_TYPE_MASK) == ENTRY_OBJECT) {
	  o = (DataObject*)(entry3 & ~3);
	  CT_DEBUG(3,"    at L3 (%d/%d/%d): (%p/%u) %s\n",
		   pos1, pos2, pos3, o->addr, o->size, o->name); 
	}
	else
	  CT_DEBUG(3,"    at L3 (%d/%d/%d): None\n", pos1, pos2, pos3);
      }
    }
  }
}

/* Insert an object into the data table
 */
static
DataObject* add_object(Char* name, UInt addr, UInt size)
{
  DataObject* o;
  TablePos start, end;

  CT_DEBUG(5," add_object(%s, addr %p, size %u)\n", name, addr, size);
  if (!setRange(addr,size,&start,&end)) return 0;

  /* check for free range */
  if (!hasOnlyEntry(ENTRY_INVALID, &start, &end)) return 0;

  o = new_object(name, addr, size);
  setEntry( ((UInt)o) | ENTRY_OBJECT, &start, &end);

  return o;
}

static
void remove_object(DataObject* o)
{
  TablePos start, end;

  CT_DEBUG(5," remove_object(%s, addr %p, size %u)\n", 
	   o->name, o->addr, o->size);
  if (!setRange(o->addr,o->size, &start, &end)) return;
  if (!hasOnlyEntry(((UInt)o) | ENTRY_OBJECT, &start, &end)) return;
  setEntry( ENTRY_INVALID, &start, &end);
}

DataObject* get_object(UInt addr)
{
  int pos1, pos2, pos3;
  UInt entry, *table2, *table3;
  DataObject* o;

  pos1 = addr >> TABLE_LEVEL1_SHIFT;
  entry = data_table[pos1];
  if ((entry & ENTRY_TYPE_MASK) == ENTRY_OBJECT) {
    o = (DataObject*)(entry & ~3);
    CT_DEBUG(5," get_object(%p): at L1 (%d)\n  => (%p/%d) %s\n",
	     addr, pos1, o->addr, o->size, o->name); 
    return o;
  }
  if ((entry & ENTRY_TYPE_MASK) == ENTRY_INVALID) {
    CT_DEBUG(5," get_object(%p): None at L1 (%d)\n", addr, pos1); 
    return 0;
  }
  table2 = (UInt*)(entry & ~3);
  
  pos2 = (addr >> TABLE_LEVEL2_SHIFT) & (TABLE_LEVEL2_SIZE-1);
  entry = table2[pos2];
  if ((entry & ENTRY_TYPE_MASK) == ENTRY_OBJECT) {
    o = (DataObject*)(entry & ~3);
    CT_DEBUG(5," get_object(%p): at L2 (%d/%d)\n  => (%p/%d) %s\n",
	     addr, pos1, pos2, o->addr, o->size, o->name); 
    return o;
  }
  if ((entry & ENTRY_TYPE_MASK) == ENTRY_INVALID) {
    CT_DEBUG(5," get_object(%p): None at L2 (%d/%d)\n", addr, pos1, pos2); 
    return 0;
  }
  table3 = (UInt*)(entry & ~3);

  pos3 = addr & (TABLE_LEVEL3_SIZE-1);
  entry = table3[pos3];
  if ((entry & ENTRY_TYPE_MASK) == ENTRY_OBJECT) {
    o = (DataObject*)(entry & ~3);
    CT_DEBUG(5," get_object(%p): at L3 (%d/%d/%d)\n  => (%p/%u) %s\n",
	     addr, pos1, pos2, pos3, o->addr, o->size, o->name); 
    return o;
  }
  CT_DEBUG(5," get_object(%p): None at L3 (%d/%d/%d)\n",
	   addr, pos1, pos2, pos3); 
  return 0;
}


/* Need to get the size and subelements first time
 * from debug info.
 */
void SK_(handle_constructor)(UInt addr, Char* fullname)
{
  VG_(message)(Vg_DebugMsg, "Constructor %s for %x",
	       fullname, addr);
}

void SK_(handle_destructor)(UInt addr, Char* fullname)
{
  VG_(message)(Vg_DebugMsg, "Destructor %s for %x",
	       fullname, addr);
}

Char obj_name[BUF_LEN];
int allocated = 0, freed = 0;

void SK_(handle_malloc)(UInt addr, UInt size)
{
  fn_node** pfn = SK_(current_fn_stack).top;
  int opos = 0, d=0;

  CT_DEBUG(3," Allocated (%x-%x, size %d)\n", addr, addr+size-1, size);
  while(*pfn) {
    CT_DEBUG(3,"  from %s\n", (*pfn)->name);
    if ((opos<BUF_LEN) && (++d<4))
      opos += VG_(sprintf)(obj_name+opos, "'%s", (*pfn)->name);
    pfn--;
  }
  add_object(obj_name+1, addr, size);

  if (((++allocated)%5)==0) printTable();
}

/* returns size of freed allocation */
Int SK_(handle_free)(UInt addr)
{
  fn_node** pfn = SK_(current_fn_stack).top-1;
  DataObject* o;
  Int res = 0;

  CT_DEBUG(3," Free %x\n", addr);
  while(*pfn) {
    CT_DEBUG(3,"  from %s\n", (*pfn)->name);
    pfn--;
  }

  o = get_object(addr);  
  if (o && (o->addr == addr)) {
    remove_object(o);
    res = o->size;
    VG_(free)(o);

    if (((++freed)%5)==0) printTable();
  }
  else
    CT_DEBUG(3,"  nothing found to free.\n");

  return res;
}

ULong* getcost(Addr addr, Context* cxt)
{
  return 0;
}
