/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/


// pml/lml
#include <pml/Atom.h>
#include <pml/StructuralComponent.h>
#include <pml/Cell.h>
#include <pml/StructuralComponentProperties.h>
#include <pml/PhysicalModel.h>
#include <pml/Cell.h>
#include <pml/Atom.h>
#include <pml/StructuralComponent.h>
#include <pml/MultiComponent.h>


#include "CellDC.h"
#include "CellDCProperties.h"
#include "AtomDC.h"
#include "PMManagerDC.h"

// camiTK
#include <InteractiveViewer.h>
#include <Geometry.h>

// Qt
#include <QPixmap>

// vtk includes
#include <vtkUnstructuredGrid.h>
#include <vtkPolyData.h>
#include <vtkPoints.h>
#include <vtkTransformFilter.h>
#include <vtkDoubleArray.h>
#include <vtkSmartPointer.h>



// -------------------- default constructor --------------------
CellDC::CellDC(camitk::Component *parent, PMManagerDC * pmManagerDC, Cell *theCell) : StructuralComponentDC(parent, pmManagerDC, theCell, true), myCell(theCell) {
  // call to the SCDC constructor with a third parameter to delay the initialization of the representation, i.e. the build of the Geometry

  // create a proper name
  QString name = dynamic_cast<Structure *>(theCell)->getName().c_str();

  if (name.isEmpty()) {
    // no name -> label is the cell nr
    name = QString::number(theCell->getIndex());
  }

  camitk::Component::setName(name);

  // associate the DC with the cell in the map
  myPMManagerDC->addCellDCPair(std::ComponentDCPair(myComponent, this));

  myProp = NULL;
}

// -------------------- destructor --------------------
CellDC::~CellDC() {
  delete myProp;
  myProp = NULL;
}

// -------------------- getPropertyObject --------------------
QObject * CellDC::getPropertyObject() {
  if (!myProp) {
    myProp = new CellDCProperties(this);
  }

  return myProp;
}

// -------------------- setSelected -------------------
void CellDC::setSelected(const bool selected, const bool recursive) {
  // if selected for the 1st time: create geometry
  if (!myGeometry) {
    // create it
    initRepresentation();
  }
  
  if (selected) {
    // add it in the InteractiveViewer
    setVisibility(InteractiveViewer::get3DViewer(), true);
    // add surfacic representation
    RenderingModes rm = getRenderingModes();
    camitk::Component::setRenderingModes(rm | InterfaceGeometry::Surface);
  }
  else {
    // remove the representation
    setVisibility(InteractiveViewer::get3DViewer(), false);
  }

  camitk::Component::setSelected(selected,false);
}

// -------------------- initRepresentation -------------------
void CellDC::initRepresentation() {
  // create the vector and map
  updateAtoms();

  // create the points for the cell
  vtkSmartPointer<vtkPoints> thePoints = vtkSmartPointer<vtkPoints>::New();
  vtkIdType *vtkPointIndex = new vtkIdType[childrenComponent.size()];
  
  // NOTE : the index of each atom in the vtk structure is in fact
  // its order number in the atoms structure (not its actual index)
  double pos[3];
  unsigned int i = 0;
  foreach(camitk::Component *dc, childrenComponent) {
    dynamic_cast<AtomDC *>(dc)->getAtom()->getPosition(pos);
    thePoints->InsertPoint(i, pos[0], pos[1], pos[2]);
    vtkPointIndex[i] = i;
    i++;
  }

  // instanciate a new one

  if (dynamic_cast<Cell *>(myComponent)->getType() != StructureProperties::POLY_LINE) {
    vtkSmartPointer<vtkUnstructuredGrid> uGrid = vtkSmartPointer<vtkUnstructuredGrid>::New();
    uGrid->Allocate(1);

    int cellType;

    switch (dynamic_cast<Cell *>(myComponent)->getType()){
      case StructureProperties::TETRAHEDRON:
        cellType = VTK_TETRA;
        break;
      case StructureProperties::HEXAHEDRON:
        cellType = VTK_HEXAHEDRON;
        break;
      case StructureProperties::WEDGE:
        cellType = VTK_WEDGE;
        break;
      case StructureProperties::PYRAMID:
        cellType = VTK_PYRAMID;
        break;
      case StructureProperties::POLY_VERTEX:
        cellType = VTK_POLY_VERTEX;
        break;
      case StructureProperties::TRIANGLE:
        cellType = VTK_TRIANGLE;
        break;
      case StructureProperties::QUAD:
        cellType = VTK_QUAD;
        break;
      case StructureProperties::LINE:
	cellType = VTK_LINE;
        break;
      default:
        cellType = VTK_VERTEX;
        break;
    }

    uGrid->InsertNextCell(cellType, (vtkIdType) childrenComponent.size(), vtkPointIndex);

    uGrid->SetPoints(thePoints);
    uGrid->Update();
    delete [] vtkPointIndex;

    // create the new geometry
    myGeometry = new Geometry(this->getName(), uGrid, myPMManagerDC->toDCRenderingMode(getSC()->getMode()));
  }
  else {
    // polyline -> tubes (and for tubes it has to be a polydata not a ugrid
    vtkSmartPointer<vtkPolyData> polyD = vtkSmartPointer<vtkPolyData>::New();

    polyD->Allocate(1);
    polyD->InsertNextCell(VTK_POLY_LINE, (vtkIdType) childrenComponent.size(), vtkPointIndex);
    polyD->SetPoints(thePoints);
    polyD->Update();
    delete [] vtkPointIndex;

    // create the new geometry
    myGeometry = new Geometry(this->getName(), polyD, myPMManagerDC->toDCRenderingMode(getSC()->getMode()));
  }

  // if the cell has no specically defined color, ask the parent structural component
  StructuralComponentProperties::Color oColor;

  oColor = getSC()->getStructuralComponentPropertiesColor();

  double oRGBA[4];

  // check the first sc color (if exist)
  if (oColor == StructuralComponentProperties::DEFAULT && myCell->getNumberOfStructuralComponents() > 0) {
    // take the first structural component that I am composing
    StructuralComponent *sc = myCell->getStructuralComponent(0);
    oColor = sc->getStructuralComponentPropertiesColor();
    // if the sc has a special color, use it as well

    if (oColor != StructuralComponentProperties::DEFAULT) {
      // get the color...
      sc->getColor(&oRGBA[0], &oRGBA[1], &oRGBA[2], &oRGBA[3]);
      //... but not the alpha channel (by default = 255)
      //oRGBA[3] = 255;
      myGeometry->setActorColor(InterfaceGeometry::Surface, oRGBA);
      // if the sc is no specifically defined color, do nothing (use the default Geometry color)
    }
  }
  else
    // if the cell has a specifically defined color, use it
    myGeometry->setActorColor(InterfaceGeometry::Surface, getSC()->getColor());
}

// -------------------- setName --------------------
void CellDC::setName(const QString & n) {
  Cell * myCell = dynamic_cast<Cell *>(myComponent);
  dynamic_cast<Structure *>(myCell)->setName(n.toStdString());
  camitk::Component::setName(n);
}

//------------------------ getPixmap ---------------------
#include "cell_20x20.xpm"
QPixmap * CellDC::myPixmap = NULL;
QPixmap CellDC::getIcon() {
  if (!myPixmap) {
    myPixmap = new QPixmap(cell_20x20);
  }

  return (*myPixmap);
}

//-------------------- createPointData -------------------
void CellDC::createPointData() {
  if (!pointData) {
    // create a new vtkDataArray
    pointData = vtkSmartPointer<vtkDoubleArray>::New();
    pointData->SetNumberOfTuples(childrenComponent.size());
    setPointData(pointData);

    // give the atom DCs the address of the corresponding point data
    int i = 0;
    foreach(camitk::Component *dc, childrenComponent) {
      dynamic_cast<AtomDC *>(dc)->addPointData(this, pointData->GetPointer(i));
      i++;
    }
  }
}

//-------------------- destroyPointData -------------------
void CellDC::destroyPointData() {
  if (pointData) {
    // delete the pointData
    pointData = NULL;
    setPointData(NULL);

    // clear the AtomDC's pointers
    foreach(camitk::Component *dc, childrenComponent) {
      dynamic_cast<AtomDC *>(dc)->clearPointData();
    }
  }
}




